# The MIT License (MIT)
#
# Copyright (c) 2018 Niklas Rosenstein
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

import functools


class dualmethod:
  """
  A combination of #classmethod() and instance methods. Methods decorated
  with this class can be called from the class and instance level. The first
  parameter will then refer to the class or the instance, respectively.
  """

  def __init__(self, func):
    self.func = func
    functools.update_wrapper(self, func)

  def __get__(self, instance, owner):
    assert owner is not None
    if instance is not None:
      return functools.partial(self.func, owner, instance)
    return functools.partial(self.func, owner)


def getter(name, key=None):
  """
  Creates a read-only property for the attribute name *name*. If a *key*
  function is provided, it can be used to post-process the value of the
  attribute.
  """

  if not key:
    key = lambda x: x

  def wrapper(self):
    return key(getattr(self, name))
  wrapper.__name__ = wrapper.__qualname__ = name
  return property(wrapper)
