#
# Copyright The NOMAD Authors.
#
# This file is part of NOMAD.
# See https://nomad-lab.eu for further info.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
import numpy as np            # pylint: disable=unused-import
import typing                 # pylint: disable=unused-import
from nomad.metainfo import (  # pylint: disable=unused-import
    MSection, MCategory, Category, Package, Quantity, Section, SubSection, SectionProxy,
    Reference
)

from nomad.datamodel.metainfo import simulation


m_package = Package()


class x_cp2k_section_input_ATOM_AE_BASIS_BASIS(MSection):
    '''
    Section used to specify a general basis set for QM calculations.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_AE_BASIS_BASIS_DEFAULT_KEYWORD = Quantity(
        type=str,
        shape=[],
        description='''
        CP2K Basis Set Standard Format
        ''')


class x_cp2k_section_input_ATOM_AE_BASIS(MSection):
    '''
    Section of basis set information for all-electron calculations.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_AE_BASIS_BASIS_SET_FILE_NAME = Quantity(
        type=str,
        shape=[],
        description='''
        Name of the basis set file, may include a path
        ''')

    x_cp2k_input_ATOM_AE_BASIS_BASIS_SET = Quantity(
        type=str,
        shape=[],
        description='''
        The contracted Gaussian basis set
        ''')

    x_cp2k_input_ATOM_AE_BASIS_BASIS_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Basis set type
        ''')

    x_cp2k_input_ATOM_AE_BASIS_D_EXPONENTS = Quantity(
        type=str,
        shape=[],
        description='''
        Exponents for d functions
        ''')

    x_cp2k_input_ATOM_AE_BASIS_D_QUANTUM_NUMBERS = Quantity(
        type=str,
        shape=[],
        description='''
        Main quantum numbers for d functions
        ''')

    x_cp2k_input_ATOM_AE_BASIS_EPS_EIGENVALUE = Quantity(
        type=str,
        shape=[],
        description='''
        Cutoff of overlap matrix eigenvalues included into basis
        ''')

    x_cp2k_input_ATOM_AE_BASIS_F_EXPONENTS = Quantity(
        type=str,
        shape=[],
        description='''
        Exponents for f functions
        ''')

    x_cp2k_input_ATOM_AE_BASIS_F_QUANTUM_NUMBERS = Quantity(
        type=str,
        shape=[],
        description='''
        Main quantum numbers for f functions
        ''')

    x_cp2k_input_ATOM_AE_BASIS_GEO_START_VALUE = Quantity(
        type=str,
        shape=[],
        description='''
        Geometrical basis: starting value a in a*C^k
        ''')

    x_cp2k_input_ATOM_AE_BASIS_GEOMETRICAL_FACTOR = Quantity(
        type=str,
        shape=[],
        description='''
        Geometrical basis: factor C in a*C^k
        ''')

    x_cp2k_input_ATOM_AE_BASIS_GRID_POINTS = Quantity(
        type=str,
        shape=[],
        description='''
        Number of radial grid points
        ''')

    x_cp2k_input_ATOM_AE_BASIS_NUM_GTO = Quantity(
        type=str,
        shape=[],
        description='''
        Number of Gaussian type functions for s, p, d, ...
        ''')

    x_cp2k_input_ATOM_AE_BASIS_NUM_SLATER = Quantity(
        type=str,
        shape=[],
        description='''
        Number of Slater type functions for s, p, d, ...
        ''')

    x_cp2k_input_ATOM_AE_BASIS_P_EXPONENTS = Quantity(
        type=str,
        shape=[],
        description='''
        Exponents for p functions
        ''')

    x_cp2k_input_ATOM_AE_BASIS_P_QUANTUM_NUMBERS = Quantity(
        type=str,
        shape=[],
        description='''
        Main quantum numbers for p functions
        ''')

    x_cp2k_input_ATOM_AE_BASIS_QUADRATURE = Quantity(
        type=str,
        shape=[],
        description='''
        Algorithm to construct the atomic radial grids
        ''')

    x_cp2k_input_ATOM_AE_BASIS_S_EXPONENTS = Quantity(
        type=str,
        shape=[],
        description='''
        Exponents for s functions
        ''')

    x_cp2k_input_ATOM_AE_BASIS_S_QUANTUM_NUMBERS = Quantity(
        type=str,
        shape=[],
        description='''
        Main quantum numbers for s functions
        ''')

    x_cp2k_input_ATOM_AE_BASIS_START_INDEX = Quantity(
        type=str,
        shape=[],
        description='''
        Starting index for Geometrical Basis sets
        ''')

    x_cp2k_section_input_ATOM_AE_BASIS_BASIS = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_AE_BASIS_BASIS'),
        repeats=True)


class x_cp2k_section_input_ATOM_METHOD_EXTERNAL_VXC(MSection):
    '''
    Section used to specify exernal VXC Potentials.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_EXTERNAL_VXC_FILE_VXC = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the filename containing the external vxc
        ''')

    x_cp2k_input_ATOM_METHOD_EXTERNAL_VXC_GRID_TOL = Quantity(
        type=str,
        shape=[],
        description='''
        Tolerance in the equivalence of read-grid in ZMP method
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_ADIABATIC_RESCALING(MSection):
    '''
    Parameters for self interation corrected hybrid functionals
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_ADIABATIC_RESCALING_FUNCTIONAL_MODEL = Quantity(
        type=str,
        shape=[],
        description='''
        Which model for the coupling constant integration should be used.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_ADIABATIC_RESCALING_FUNCTIONAL_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Which Hybrid functional should be used. (Has to be consistent with the definitions
        in XC and HF).
        ''')

    x_cp2k_input_ATOM_METHOD_XC_ADIABATIC_RESCALING_LAMBDA = Quantity(
        type=str,
        shape=[],
        description='''
        The point to be used along the adiabatic curve (0 &#60; λ &#60; 1)
        ''')

    x_cp2k_input_ATOM_METHOD_XC_ADIABATIC_RESCALING_OMEGA = Quantity(
        type=str,
        shape=[],
        description='''
        Long-range parameter
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_HF_HF_INFO(MSection):
    '''
    Controls the printing basic info about hf method
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_HF_HF_INFO_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_HF_HF_INFO_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_ATOM_METHOD_XC_HF_HF_INFO_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_HF_HF_INFO_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_ATOM_METHOD_XC_HF_HF_INFO_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_HF_HFX_RI(MSection):
    '''
    All parameters needed in a HFX RI calculation
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_HF_HFX_RI_EPS_OPTIMIZATION = Quantity(
        type=str,
        shape=[],
        description='''
        Accuracy of iterative RI fit
        ''')

    x_cp2k_input_ATOM_METHOD_XC_HF_HFX_RI_EPS_SCREENING = Quantity(
        type=str,
        shape=[],
        description='''
        Accuracy of geminal integral evaluation
        ''')

    x_cp2k_input_ATOM_METHOD_XC_HF_HFX_RI_MAX_ITER = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum number of iteration in RI fit
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_HF_INTERACTION_POTENTIAL(MSection):
    '''
    Sets up interaction potential if requested
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_HF_INTERACTION_POTENTIAL_CUTOFF_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Determines cutoff radius for the truncated 1/r potential. Only valid when doing
        truncated calculation
        ''')

    x_cp2k_input_ATOM_METHOD_XC_HF_INTERACTION_POTENTIAL_OMEGA = Quantity(
        type=str,
        shape=[],
        description='''
        Parameter for short/longrange interaction
        ''')

    x_cp2k_input_ATOM_METHOD_XC_HF_INTERACTION_POTENTIAL_POTENTIAL_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Which interaction potential should be used (Coulomb, longrange or shortrange).
        ''')

    x_cp2k_input_ATOM_METHOD_XC_HF_INTERACTION_POTENTIAL_SCALE_COULOMB = Quantity(
        type=str,
        shape=[],
        description='''
        Scales Hartree-Fock contribution arising from a coulomb potential. Only valid when
        doing a mixed potential calculation
        ''')

    x_cp2k_input_ATOM_METHOD_XC_HF_INTERACTION_POTENTIAL_SCALE_GAUSSIAN = Quantity(
        type=str,
        shape=[],
        description='''
        Scales Hartree-Fock contribution arising from a gaussian potential. Only valid
        when doing a mixed potential calculation
        ''')

    x_cp2k_input_ATOM_METHOD_XC_HF_INTERACTION_POTENTIAL_SCALE_LONGRANGE = Quantity(
        type=str,
        shape=[],
        description='''
        Scales Hartree-Fock contribution arising from a longrange potential. Only valid
        when doing a mixed potential calculation
        ''')

    x_cp2k_input_ATOM_METHOD_XC_HF_INTERACTION_POTENTIAL_T_C_G_DATA = Quantity(
        type=str,
        shape=[],
        description='''
        Location of the file t_c_g.dat that contains the data for the evaluation of the
        truncated gamma function
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_HF_LOAD_BALANCE(MSection):
    '''
    Parameters influencing the load balancing of the HF
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_HF_LOAD_BALANCE_BLOCK_SIZE = Quantity(
        type=str,
        shape=[],
        description='''
        Determines the blocking used for the atomic quartet loops. A proper choice can
        speedup the calculation. The default (-1) is automatic.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_HF_LOAD_BALANCE_NBINS = Quantity(
        type=str,
        shape=[],
        description='''
        Number of bins per process used to group atom quartets.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_HF_LOAD_BALANCE_RANDOMIZE = Quantity(
        type=str,
        shape=[],
        description='''
        This flag controls the randomization of the bin assignment to processes. For
        highly ordered input structures with a bad load balance, setting this flag to TRUE
        might improve.
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_HF_MEMORY(MSection):
    '''
    Sets up memory parameters for the storage of the ERI's if requested
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_HF_MEMORY_EPS_STORAGE_SCALING = Quantity(
        type=str,
        shape=[],
        description='''
        Scaling factor to scale eps_schwarz. Storage threshold for compression will be
        EPS_SCHWARZ*EPS_STORAGE_SCALING.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_HF_MEMORY_MAX_DISK_SPACE = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the maximum amount of disk space [MB] used to store precomputed compressed
        four-center integrals. If 0, nothing is stored to disk
        ''')

    x_cp2k_input_ATOM_METHOD_XC_HF_MEMORY_MAX_MEMORY = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the maximum amount of memory [MB] to be consumed by the full HFX module.
        All temporary buffers and helper arrays are subtracted from this number. What
        remains will be used for storage of integrals. NOTE: This number is assumed to
        represent the memory available to one MPI process. When running a threaded
        version, cp2k automatically takes care of distributing the memory among all the
        threads within a process.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_HF_MEMORY_STORAGE_LOCATION = Quantity(
        type=str,
        shape=[],
        description='''
        Loaction where ERI's are stored if MAX_DISK_SPACE /=0 Expects a path to a
        directory.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_HF_MEMORY_TREAT_FORCES_IN_CORE = Quantity(
        type=str,
        shape=[],
        description='''
        Determines whether the derivative ERI's should be stored to RAM or not. Only
        meaningful when performing Ehrenfest MD. Memory usage is defined via MAX_MEMORY,
        i.e. the memory is shared wit the energy ERI's.
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_HF_PERIODIC(MSection):
    '''
    Sets up periodic boundary condition parameters if requested
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_HF_PERIODIC_NUMBER_OF_SHELLS = Quantity(
        type=str,
        shape=[],
        description='''
        Number of shells taken into account for periodicity. By default, cp2k tries to
        automatically evaluate this number. This algorithm might be to conservative,
        resulting in some overhead. You can try to adjust this number in order to make a
        calculation cheaper.
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_HF_SCREENING(MSection):
    '''
    Sets up screening parameters if requested
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_HF_SCREENING_EPS_SCHWARZ_FORCES = Quantity(
        type=str,
        shape=[],
        description='''
        Screens the near field part of the electronic repulsion integrals using the
        Schwarz inequality for the given threshold. This will be approximately the
        accuracy of the forces,  and should normally be similar to EPS_SCF
        ''')

    x_cp2k_input_ATOM_METHOD_XC_HF_SCREENING_EPS_SCHWARZ = Quantity(
        type=str,
        shape=[],
        description='''
        Screens the near field part of the electronic repulsion integrals using the
        Schwarz inequality for the given threshold.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_HF_SCREENING_P_SCREEN_CORRECTION_FACTOR = Quantity(
        type=str,
        shape=[],
        description='''
        Recalculates integrals on the fly if the actual density matrix is larger by a
        given factor than the initial one. If the factor is set to 0.0_dp, this feature is
        disabled.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_HF_SCREENING_SCREEN_ON_INITIAL_P = Quantity(
        type=str,
        shape=[],
        description='''
        Screen on an initial density matrix. For the first MD step this matrix must be
        provided by a Restart File.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_HF_SCREENING_SCREEN_P_FORCES = Quantity(
        type=str,
        shape=[],
        description='''
        Screens the electronic repulsion integrals for the forces using the density
        matrix. This results in a significant speedup for large systems, but might require
        a somewhat tigher EPS_SCHWARZ_FORCES.
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_HF(MSection):
    '''
    Sets up the Hartree-Fock parameters if requested
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_HF_FRACTION = Quantity(
        type=str,
        shape=[],
        description='''
        The fraction of Hartree-Fock to add to the total energy. 1.0 implies standard
        Hartree-Fock if used with XC_FUNCTIONAL NONE. NOTE: In a mixed potential
        calculation this should be set to 1.0, otherwise all parts are multiplied with
        this factor.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_HF_PW_HFX_BLOCKSIZE = Quantity(
        type=str,
        shape=[],
        description='''
        Improve the performance of pw_hfx at the cost of some additional memory by storing
        the realspace representation of PW_HFX_BLOCKSIZE states.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_HF_PW_HFX = Quantity(
        type=str,
        shape=[],
        description='''
        Compute the Hartree-Fock energy also in the plane wave basis.The value is ignored,
        and intended for debugging only.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_HF_TREAT_LSD_IN_CORE = Quantity(
        type=str,
        shape=[],
        description='''
        Determines how spin denisities are taken into account. If true, the beta spin
        density is included via a second in core call. If false, alpha and beta spins are
        done in one shot
        ''')

    x_cp2k_section_input_ATOM_METHOD_XC_HF_HF_INFO = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_HF_HF_INFO'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_HF_HFX_RI = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_HF_HFX_RI'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_HF_INTERACTION_POTENTIAL = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_HF_INTERACTION_POTENTIAL'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_HF_LOAD_BALANCE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_HF_LOAD_BALANCE'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_HF_MEMORY = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_HF_MEMORY'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_HF_PERIODIC = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_HF_PERIODIC'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_HF_SCREENING = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_HF_SCREENING'),
        repeats=True)


class x_cp2k_section_input_ATOM_METHOD_XC_VDW_POTENTIAL_NON_LOCAL(MSection):
    '''
    Information on the non local dispersion functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_VDW_POTENTIAL_NON_LOCAL_CUTOFF = Quantity(
        type=str,
        shape=[],
        description='''
        The cutoff of the FFT grid used in the calculation of the nonlocal vdW functional
        [Ry].
        ''')

    x_cp2k_input_ATOM_METHOD_XC_VDW_POTENTIAL_NON_LOCAL_KERNEL_FILE_NAME = Quantity(
        type=str,
        shape=[],
        description='''
        Name of the kernel data file, may include a path.vdW_kernel_table.dat is for DRSLL
        and LMKLL andrVV10_kernel_table.dat is for rVV10.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_VDW_POTENTIAL_NON_LOCAL_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Parameters b and C of the rVV10 functional
        ''')

    x_cp2k_input_ATOM_METHOD_XC_VDW_POTENTIAL_NON_LOCAL_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Type of functional (the corresponding kernel data file should be selected).Allows
        for common forms such as vdW-DF, vdW-DF2, optB88-vdW, rVV10.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_VDW_POTENTIAL_NON_LOCAL_VERBOSE_OUTPUT = Quantity(
        type=str,
        shape=[],
        description='''
        Extensive output for non local functionals
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_VDW_POTENTIAL_PAIR_POTENTIAL_PRINT_DFTD(MSection):
    '''
    Controls the printing of some info about DFTD contributions
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_VDW_POTENTIAL_PAIR_POTENTIAL_PRINT_DFTD_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_VDW_POTENTIAL_PAIR_POTENTIAL_PRINT_DFTD_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_ATOM_METHOD_XC_VDW_POTENTIAL_PAIR_POTENTIAL_PRINT_DFTD_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_VDW_POTENTIAL_PAIR_POTENTIAL_PRINT_DFTD_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_ATOM_METHOD_XC_VDW_POTENTIAL_PAIR_POTENTIAL_PRINT_DFTD_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_VDW_POTENTIAL_PAIR_POTENTIAL(MSection):
    '''
    Information on the pair potential to calculate dispersion
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_VDW_POTENTIAL_PAIR_POTENTIAL_ATOM_COORDINATION_NUMBERS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the coordination number of a set of atoms for the C9 term in DFT-D3.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_VDW_POTENTIAL_PAIR_POTENTIAL_ATOMPARM = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies parameters for atom types (in atomic units). If not provided default
        parameters are used (DFT-D2).
        ''')

    x_cp2k_input_ATOM_METHOD_XC_VDW_POTENTIAL_PAIR_POTENTIAL_CALCULATE_C9_TERM = Quantity(
        type=str,
        shape=[],
        description='''
        Calculate C9 terms in DFT-D3 model
        ''')

    x_cp2k_input_ATOM_METHOD_XC_VDW_POTENTIAL_PAIR_POTENTIAL_D3_SCALING = Quantity(
        type=str,
        shape=[],
        description='''
        XC Functional dependent scaling parameters (s6,sr6,s8) for the DFT-D3 method, if
        set to zero CP2K attempts to guess the xc functional from REFERENCE_FUNCTIONAL and
        sets the associated scaling parameter.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_VDW_POTENTIAL_PAIR_POTENTIAL_D3BJ_SCALING = Quantity(
        type=str,
        shape=[],
        description='''
        XC Functional dependent scaling parameters (s6,a1,s8,a2) for the DFT-D3(BJ)
        method, if set to zero CP2K attempts to guess the xc functional from
        REFERENCE_FUNCTIONAL and sets the associated scaling parameter.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_VDW_POTENTIAL_PAIR_POTENTIAL_EPS_CN = Quantity(
        type=str,
        shape=[],
        description='''
        Cutoff value for coordination number function (DFT-D3 method)
        ''')

    x_cp2k_input_ATOM_METHOD_XC_VDW_POTENTIAL_PAIR_POTENTIAL_EXP_PRE = Quantity(
        type=str,
        shape=[],
        description='''
        Prefactor in exponential damping factor (DFT-D2 potential)
        ''')

    x_cp2k_input_ATOM_METHOD_XC_VDW_POTENTIAL_PAIR_POTENTIAL_KIND_COORDINATION_NUMBERS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the coordination number for a kind for the C9 term in DFT-D3.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_VDW_POTENTIAL_PAIR_POTENTIAL_LONG_RANGE_CORRECTION = Quantity(
        type=str,
        shape=[],
        description='''
        Calculate a long range correction to the DFT-D3 model
        ''')

    x_cp2k_input_ATOM_METHOD_XC_VDW_POTENTIAL_PAIR_POTENTIAL_PARAMETER_FILE_NAME = Quantity(
        type=str,
        shape=[],
        description='''
        Name of the parameter file, may include a path
        ''')

    x_cp2k_input_ATOM_METHOD_XC_VDW_POTENTIAL_PAIR_POTENTIAL_R_CUTOFF = Quantity(
        type=str,
        shape=[],
        description='''
        Range of potential. The cutoff will be 2 times this value
        ''')

    x_cp2k_input_ATOM_METHOD_XC_VDW_POTENTIAL_PAIR_POTENTIAL_REFERENCE_C9_TERM = Quantity(
        type=str,
        shape=[],
        description='''
        Calculate C9 terms in DFT-D3 model using reference coordination numbers
        ''')

    x_cp2k_input_ATOM_METHOD_XC_VDW_POTENTIAL_PAIR_POTENTIAL_REFERENCE_FUNCTIONAL = Quantity(
        type=str,
        shape=[],
        description='''
        Use parameters for this specific density functional. For available D3 and D3(BJ)
        parameters see: http://www.thch.uni-bonn.de/tc/downloads/DFT-D3/functionals.html,
        http://www.thch.uni-bonn.de/tc/downloads/DFT-D3/functionalsbj.html
        ''')

    x_cp2k_input_ATOM_METHOD_XC_VDW_POTENTIAL_PAIR_POTENTIAL_SCALING = Quantity(
        type=str,
        shape=[],
        description='''
        XC Functional dependent scaling parameter, if set to zero CP2K attempts to guess
        the xc functional that is in use and sets the associated scaling parameter.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_VDW_POTENTIAL_PAIR_POTENTIAL_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Type of potential
        ''')

    x_cp2k_input_ATOM_METHOD_XC_VDW_POTENTIAL_PAIR_POTENTIAL_VERBOSE_OUTPUT = Quantity(
        type=str,
        shape=[],
        description='''
        Extensive output for the DFT-D2 and DFT-D3 models
        ''')

    x_cp2k_section_input_ATOM_METHOD_XC_VDW_POTENTIAL_PAIR_POTENTIAL_PRINT_DFTD = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_VDW_POTENTIAL_PAIR_POTENTIAL_PRINT_DFTD'),
        repeats=True)


class x_cp2k_section_input_ATOM_METHOD_XC_VDW_POTENTIAL(MSection):
    '''
    This section combines all possible additional dispersion corrections to the normal XC
    functionals. This can be more functionals or simple empirical pair potentials.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_VDW_POTENTIAL_POTENTIAL_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Type of dispersion/vdW functional or potential to use
        ''')

    x_cp2k_section_input_ATOM_METHOD_XC_VDW_POTENTIAL_NON_LOCAL = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_VDW_POTENTIAL_NON_LOCAL'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_VDW_POTENTIAL_PAIR_POTENTIAL = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_VDW_POTENTIAL_PAIR_POTENTIAL'),
        repeats=True)


class x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_CPHF(MSection):
    '''
    Parameters influencing the solution of the Z-vector equations in MP2 gradients
    calculations.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_CPHF_EPS_CONV = Quantity(
        type=str,
        shape=[],
        description='''
        Convergence threshold for the solution of the Z-vector equations. The Z-vector
        equations have the form of a linear system of equations Ax=b, convergence is
        achieved when |Ax-b|<=EPS_CONV.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_CPHF_MAX_ITER = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum number of iterations allowed for the solution of the Z-vector equations.
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_DIRECT_CANONICAL(MSection):
    '''
    Parameters influencing the direct canonical method
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_DIRECT_CANONICAL_BIG_SEND = Quantity(
        type=str,
        shape=[],
        description='''
        Send big messages between processes (useful for >48 processors).
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_INTERACTION_POTENTIAL(MSection):
    '''
    Parameters the interaction potential in computing the biel integrals
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_INTERACTION_POTENTIAL_POTENTIAL_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Which interaction potential should be used (Coulomb, TShPSC operator).
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_INTERACTION_POTENTIAL_TRUNCATION_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Determines truncation radius for the truncated TShPSC potential. Only valid when
        doing truncated calculation
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_INTERACTION_POTENTIAL_TSHPSC_DATA = Quantity(
        type=str,
        shape=[],
        description='''
        Location of the file TShPSC.dat that contains the data for the evaluation of the
        TShPSC G0
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_MP2_INFO(MSection):
    '''
    Controls the printing basic info about MP2 method
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_MP2_INFO_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_MP2_INFO_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_MP2_INFO_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_MP2_INFO_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_MP2_INFO_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_OPT_RI_BASIS(MSection):
    '''
    Parameters influencing the optimization of the RI MP2 basis. Only exponents of non-
    contracted auxiliary basis can be optimized. An initial RI auxiliary basis has to be
    specified.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_OPT_RI_BASIS_BASIS_SIZE = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the size of the auxiliary basis set automatically generated as initial
        guess. This will be effective only if RI_AUX_BASIS_SET in the KIND section and
        NUM_FUNC are not specified.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_OPT_RI_BASIS_DELTA_I_REL = Quantity(
        type=str,
        shape=[],
        description='''
        Target accuracy in the relative deviation of the amplitudes calculated with and
        without RI approximation, (more details in Chem.Phys.Lett.294(1998)143).
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_OPT_RI_BASIS_DELTA_RI = Quantity(
        type=str,
        shape=[],
        description='''
        Target accuracy in the absolute difference between the RI-MP2 and the exact MP2
        energy, DRI=ABS(E_MP2-E_RI-MP2).
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_OPT_RI_BASIS_EPS_DERIV = Quantity(
        type=str,
        shape=[],
        description='''
        The derivatives of the MP2 energy with respect to the exponents of the basis are
        calculated numerically. The change in the exponent a_i employed for the numerical
        evaluation is defined as h_i=EPS_DERIV*a_i.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_OPT_RI_BASIS_MAX_ITER = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the maximum number of steps in the RI basis optimization.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_OPT_RI_BASIS_NUM_FUNC = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the number of function, for each angular momentum (s, p, d ...),
        employed in the automatically generated initial guess. This will be effective only
        if RI_AUX_BASIS_SET in the KIND section is not specified.
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_RI_LAPLACE(MSection):
    '''
    Parameters influencing the RI-SOS-MP2-Laplace method
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_RI_LAPLACE_QUADRATURE_POINTS = Quantity(
        type=str,
        shape=[],
        description='''
        Number of quadrature points for the numerical integration in the RI-SOS-
        MP2-Laplace method.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_RI_LAPLACE_SIZE_INTEG_GROUP = Quantity(
        type=str,
        shape=[],
        description='''
        Group size for the integration in the Laplace method, that is the number of
        processes involved in the computation of each integration point. SIZE_INTEG_GROUP
        has to be a multiple of GROUP_SIZE in the WF_CORRELATION section. The default (-1)
        is automatic.
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_RI_MP2(MSection):
    '''
    Parameters influencing the RI MP2 method
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_RI_MP2_BLOCK_SIZE = Quantity(
        type=str,
        shape=[],
        description='''
        Determines the blocking used for communication in RI-MP2. Larger BLOCK_SIZE
        reduces communication but requires more memory. The default (-1) is automatic.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_RI_MP2_EPS_CANONICAL = Quantity(
        type=str,
        shape=[],
        description='''
        Threshold for discriminate if a given ij pairs of the unrelaxed MP2 density matrix
        has to be calculated with a canonical reformulation based on the occupied
        eigenvalues differences.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_RI_MP2_FREE_HFX_BUFFER = Quantity(
        type=str,
        shape=[],
        description='''
        Free the buffer containing the 4 center integrals used in the Hartree-Fock
        exchange calculation. This will be effective only for gradients calculations,
        since for the energy only case, the buffers are released by default. (Right now
        debugging only).
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_HF_INFO(MSection):
    '''
    Controls the printing basic info about hf method
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_HF_INFO_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_HF_INFO_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_HF_INFO_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_HF_INFO_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_HF_INFO_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_HFX_RI(MSection):
    '''
    All parameters needed in a HFX RI calculation
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_HFX_RI_EPS_OPTIMIZATION = Quantity(
        type=str,
        shape=[],
        description='''
        Accuracy of iterative RI fit
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_HFX_RI_EPS_SCREENING = Quantity(
        type=str,
        shape=[],
        description='''
        Accuracy of geminal integral evaluation
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_HFX_RI_MAX_ITER = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum number of iteration in RI fit
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_INTERACTION_POTENTIAL(MSection):
    '''
    Sets up interaction potential if requested
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_INTERACTION_POTENTIAL_CUTOFF_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Determines cutoff radius for the truncated 1/r potential. Only valid when doing
        truncated calculation
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_INTERACTION_POTENTIAL_OMEGA = Quantity(
        type=str,
        shape=[],
        description='''
        Parameter for short/longrange interaction
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_INTERACTION_POTENTIAL_POTENTIAL_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Which interaction potential should be used (Coulomb, longrange or shortrange).
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_INTERACTION_POTENTIAL_SCALE_COULOMB = Quantity(
        type=str,
        shape=[],
        description='''
        Scales Hartree-Fock contribution arising from a coulomb potential. Only valid when
        doing a mixed potential calculation
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_INTERACTION_POTENTIAL_SCALE_GAUSSIAN = Quantity(
        type=str,
        shape=[],
        description='''
        Scales Hartree-Fock contribution arising from a gaussian potential. Only valid
        when doing a mixed potential calculation
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_INTERACTION_POTENTIAL_SCALE_LONGRANGE = Quantity(
        type=str,
        shape=[],
        description='''
        Scales Hartree-Fock contribution arising from a longrange potential. Only valid
        when doing a mixed potential calculation
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_INTERACTION_POTENTIAL_T_C_G_DATA = Quantity(
        type=str,
        shape=[],
        description='''
        Location of the file t_c_g.dat that contains the data for the evaluation of the
        truncated gamma function
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_LOAD_BALANCE(MSection):
    '''
    Parameters influencing the load balancing of the HF
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_LOAD_BALANCE_BLOCK_SIZE = Quantity(
        type=str,
        shape=[],
        description='''
        Determines the blocking used for the atomic quartet loops. A proper choice can
        speedup the calculation. The default (-1) is automatic.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_LOAD_BALANCE_NBINS = Quantity(
        type=str,
        shape=[],
        description='''
        Number of bins per process used to group atom quartets.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_LOAD_BALANCE_RANDOMIZE = Quantity(
        type=str,
        shape=[],
        description='''
        This flag controls the randomization of the bin assignment to processes. For
        highly ordered input structures with a bad load balance, setting this flag to TRUE
        might improve.
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_MEMORY(MSection):
    '''
    Sets up memory parameters for the storage of the ERI's if requested
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_MEMORY_EPS_STORAGE_SCALING = Quantity(
        type=str,
        shape=[],
        description='''
        Scaling factor to scale eps_schwarz. Storage threshold for compression will be
        EPS_SCHWARZ*EPS_STORAGE_SCALING.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_MEMORY_MAX_DISK_SPACE = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the maximum amount of disk space [MB] used to store precomputed compressed
        four-center integrals. If 0, nothing is stored to disk
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_MEMORY_MAX_MEMORY = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the maximum amount of memory [MB] to be consumed by the full HFX module.
        All temporary buffers and helper arrays are subtracted from this number. What
        remains will be used for storage of integrals. NOTE: This number is assumed to
        represent the memory available to one MPI process. When running a threaded
        version, cp2k automatically takes care of distributing the memory among all the
        threads within a process.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_MEMORY_STORAGE_LOCATION = Quantity(
        type=str,
        shape=[],
        description='''
        Loaction where ERI's are stored if MAX_DISK_SPACE /=0 Expects a path to a
        directory.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_MEMORY_TREAT_FORCES_IN_CORE = Quantity(
        type=str,
        shape=[],
        description='''
        Determines whether the derivative ERI's should be stored to RAM or not. Only
        meaningful when performing Ehrenfest MD. Memory usage is defined via MAX_MEMORY,
        i.e. the memory is shared wit the energy ERI's.
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_PERIODIC(MSection):
    '''
    Sets up periodic boundary condition parameters if requested
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_PERIODIC_NUMBER_OF_SHELLS = Quantity(
        type=str,
        shape=[],
        description='''
        Number of shells taken into account for periodicity. By default, cp2k tries to
        automatically evaluate this number. This algorithm might be to conservative,
        resulting in some overhead. You can try to adjust this number in order to make a
        calculation cheaper.
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_SCREENING(MSection):
    '''
    Sets up screening parameters if requested
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_SCREENING_EPS_SCHWARZ_FORCES = Quantity(
        type=str,
        shape=[],
        description='''
        Screens the near field part of the electronic repulsion integrals using the
        Schwarz inequality for the given threshold. This will be approximately the
        accuracy of the forces,  and should normally be similar to EPS_SCF
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_SCREENING_EPS_SCHWARZ = Quantity(
        type=str,
        shape=[],
        description='''
        Screens the near field part of the electronic repulsion integrals using the
        Schwarz inequality for the given threshold.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_SCREENING_P_SCREEN_CORRECTION_FACTOR = Quantity(
        type=str,
        shape=[],
        description='''
        Recalculates integrals on the fly if the actual density matrix is larger by a
        given factor than the initial one. If the factor is set to 0.0_dp, this feature is
        disabled.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_SCREENING_SCREEN_ON_INITIAL_P = Quantity(
        type=str,
        shape=[],
        description='''
        Screen on an initial density matrix. For the first MD step this matrix must be
        provided by a Restart File.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_SCREENING_SCREEN_P_FORCES = Quantity(
        type=str,
        shape=[],
        description='''
        Screens the electronic repulsion integrals for the forces using the density
        matrix. This results in a significant speedup for large systems, but might require
        a somewhat tigher EPS_SCHWARZ_FORCES.
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF(MSection):
    '''
    Sets up the Hartree-Fock parameters if requested
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_FRACTION = Quantity(
        type=str,
        shape=[],
        description='''
        The fraction of Hartree-Fock to add to the total energy. 1.0 implies standard
        Hartree-Fock if used with XC_FUNCTIONAL NONE. NOTE: In a mixed potential
        calculation this should be set to 1.0, otherwise all parts are multiplied with
        this factor.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_PW_HFX_BLOCKSIZE = Quantity(
        type=str,
        shape=[],
        description='''
        Improve the performance of pw_hfx at the cost of some additional memory by storing
        the realspace representation of PW_HFX_BLOCKSIZE states.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_PW_HFX = Quantity(
        type=str,
        shape=[],
        description='''
        Compute the Hartree-Fock energy also in the plane wave basis.The value is ignored,
        and intended for debugging only.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_TREAT_LSD_IN_CORE = Quantity(
        type=str,
        shape=[],
        description='''
        Determines how spin denisities are taken into account. If true, the beta spin
        density is included via a second in core call. If false, alpha and beta spins are
        done in one shot
        ''')

    x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_HF_INFO = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_HF_INFO'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_HFX_RI = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_HFX_RI'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_INTERACTION_POTENTIAL = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_INTERACTION_POTENTIAL'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_LOAD_BALANCE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_LOAD_BALANCE'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_MEMORY = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_MEMORY'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_PERIODIC = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_PERIODIC'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_SCREENING = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF_SCREENING'),
        repeats=True)


class x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA(MSection):
    '''
    Parameters influencing the RI RPA method
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_MINIMAX_QUADRATURE = Quantity(
        type=str,
        shape=[],
        description='''
        Use the Minimax quadrature scheme for performing the numerical integration.
        Maximum number of quadrature point limited to 20.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_MM_STYLE = Quantity(
        type=str,
        shape=[],
        description='''
        Matrix multiplication style for the Q matrix.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_QUADRATURE_POINTS = Quantity(
        type=str,
        shape=[],
        description='''
        Number of quadrature points for the numerical integration in the RI-RPA method.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_SIZE_FREQ_INTEG_GROUP = Quantity(
        type=str,
        shape=[],
        description='''
        Group size for frequency integration, that is the number of processes involved in
        the computation of each integration point. SIZE_FREQ_INTEG_GROUP has to be a
        multiple of GROUP_SIZE in the WF_CORRELATION section. The default (-1) is
        automatic.
        ''')

    x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA_HF'),
        repeats=True)


class x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_WFC_GPW(MSection):
    '''
    Parameters for the GPW approach in Wavefunction-based Correlation methods
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_WFC_GPW_CUTOFF = Quantity(
        type=str,
        shape=[],
        description='''
        The cutoff of the finest grid level in the MP2 gpw integration.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_WFC_GPW_EPS_FILTER = Quantity(
        type=str,
        shape=[],
        description='''
        Determines a threshold for the DBCSR based multiply (usually 10 times smaller than
        EPS_GRID).
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_WFC_GPW_EPS_GRID = Quantity(
        type=str,
        shape=[],
        description='''
        Determines a threshold for the GPW based integration
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_WFC_GPW_PRINT_LEVEL = Quantity(
        type=str,
        shape=[],
        description='''
        How much output is written by the individual groups.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_WFC_GPW_REL_CUTOFF = Quantity(
        type=str,
        shape=[],
        description='''
        Determines the grid at which a Gaussian is mapped.
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION(MSection):
    '''
    Sets up the Wavefunction-based Correlation parameters if requested
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_CALC_COND_NUM = Quantity(
        type=str,
        shape=[],
        description='''
        Calculate the condition number of the (P|Q) matrix for the RI methods.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_COL_BLOCK = Quantity(
        type=str,
        shape=[],
        description='''
        Size of the column block used in the SCALAPACK block cyclic data
        distribution.Default is (COL_BLOCK=-1) is automatic. A proper choice can speedup
        the parallel matrix multiplication in the case of RI-RPA and RI-SOS-MP2-Laplace.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_GROUP_SIZE = Quantity(
        type=str,
        shape=[],
        description='''
        Group size used in the computation of the integrals. Default is to use all
        processors (GROUP_SIZE=-1).A smaller group size (for example the node size), might
        a better choice if the actual MP2 time is large compared to integral computation
        time. This is usually the case if the total number of processors is not too large.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_MEMORY = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum allowed total memory usage during MP2 methods [Mb].
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_METHOD = Quantity(
        type=str,
        shape=[],
        description='''
        Which method should be used to compute the MP2 energy
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_ROW_BLOCK = Quantity(
        type=str,
        shape=[],
        description='''
        Size of the row block used in the SCALAPACK block cyclic data distribution.Default
        is (ROW_BLOCK=-1) is automatic. A proper choice can speedup the parallel matrix
        multiplication in the case of RI-RPA and RI-SOS-MP2-Laplace.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_SCALE_S = Quantity(
        type=str,
        shape=[],
        description='''
        Scaling factor of the singlet energy component (opposite spin, OS).
        ''')

    x_cp2k_input_ATOM_METHOD_XC_WF_CORRELATION_SCALE_T = Quantity(
        type=str,
        shape=[],
        description='''
        Scaling factor of the triplet energy component (same spin, SS).
        ''')

    x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_CPHF = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_CPHF'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_DIRECT_CANONICAL = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_DIRECT_CANONICAL'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_INTERACTION_POTENTIAL = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_INTERACTION_POTENTIAL'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_MP2_INFO = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_MP2_INFO'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_OPT_RI_BASIS = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_OPT_RI_BASIS'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_RI_LAPLACE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_RI_LAPLACE'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_RI_MP2 = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_RI_MP2'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_RI_RPA'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_WFC_GPW = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION_WFC_GPW'),
        repeats=True)


class x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_BECKE88_LR_ADIABATIC(MSection):
    '''
    Uses the Becke 88 longrange exchange functional in an adiabatic fashion
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_BECKE88_LR_ADIABATIC_LAMBDA = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the parameter of the adiabatic curve
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_BECKE88_LR_ADIABATIC_OMEGA = Quantity(
        type=str,
        shape=[],
        description='''
        Potential parameter in erf(omega*r)/r
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_BECKE88_LR_ADIABATIC_SCALE_X = Quantity(
        type=str,
        shape=[],
        description='''
        scales the exchange part of the functional
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_BECKE88_LR_ADIABATIC_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_BECKE88_LR(MSection):
    '''
    Uses the Becke 88 longrange exchange functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_BECKE88_LR_OMEGA = Quantity(
        type=str,
        shape=[],
        description='''
        Potential parameter in erf(omega*r)/r
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_BECKE88_LR_SCALE_X = Quantity(
        type=str,
        shape=[],
        description='''
        scales the exchange part of the functional
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_BECKE88_LR_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_BECKE88(MSection):
    '''
    Uses the Becke 88 exchange functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_BECKE88_SCALE_X = Quantity(
        type=str,
        shape=[],
        description='''
        scales the exchange part of the functional
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_BECKE88_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_BECKE97(MSection):
    '''
    Uses the Becke 97 exchange correlation functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_BECKE97_PARAMETRIZATION = Quantity(
        type=str,
        shape=[],
        description='''
        switches between the B97 and Grimme parametrization
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_BECKE97_SCALE_C = Quantity(
        type=str,
        shape=[],
        description='''
        scales the correlation part of the functional
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_BECKE97_SCALE_X = Quantity(
        type=str,
        shape=[],
        description='''
        scales the exchange part of the functional, if -1 the default for the given
        parametrization is used
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_BECKE97_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_BECKE_ROUSSEL(MSection):
    '''
    Becke Roussel exchange hole model. Can be usedas long range correction with a
    truncated coulomb potential
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_BECKE_ROUSSEL_CUTOFF_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the cutoff radius for the truncation. If put to zero, the standard full
        range potential will be used
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_BECKE_ROUSSEL_GAMMA = Quantity(
        type=str,
        shape=[],
        description='''
        Parameter in the exchange hole. Usually this is put to 1.0 or 0.8
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_BECKE_ROUSSEL_SCALE_X = Quantity(
        type=str,
        shape=[],
        description='''
        scales the exchange part of the functional
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_BECKE_ROUSSEL_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_BEEF(MSection):
    '''
    Uses the BEEFvdW exchange functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_BEEF_SCALE_X = Quantity(
        type=str,
        shape=[],
        description='''
        scales the exchange part of the functional
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_BEEF_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_CS1(MSection):
    '''
    Uses the CS1 functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_CS1_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_GV09(MSection):
    '''
    Combination of three different exchange hole models
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_GV09_CUTOFF_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines cutoff for lower integration boundary
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_GV09_GAMMA = Quantity(
        type=str,
        shape=[],
        description='''
        Parameter for Becke Roussel hole
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_GV09_SCALE_X = Quantity(
        type=str,
        shape=[],
        description='''
        scales the exchange part of the functional
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_GV09_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_HCTH(MSection):
    '''
    Uses the HCTH class of functionals
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_HCTH_PARAMETER_SET = Quantity(
        type=str,
        shape=[],
        description='''
        Which version of the parameters should be used
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_HCTH_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_KE_GGA(MSection):
    '''
    Uses one of the KE_GGA functionals (optimized versions of some of these functionals
    might be available outside this section). These functionals are needed for the
    computation of the kinetic energy in the Kim-Gordon method.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_KE_GGA_FUNCTIONAL = Quantity(
        type=str,
        shape=[],
        description='''
        Which one of the KE_GGA functionals should be used
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_KE_GGA_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_KE_LIBXC(MSection):
    '''
    To be used for KG runs. Uses kinetic energy functionals from LIBXC, see also
    http://www.tddft.org/programs/octopus/wiki/index.php/Libxc
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_KE_LIBXC_FUNCTIONAL = Quantity(
        type=str,
        shape=[],
        description='''
        names of the functionals, see also
        http://www.tddft.org/programs/octopus/wiki/index.php/Libxc:manual .The precise
        list of available functionals depends on the version of libxc interfaced
        (currently 2.0.1).
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_KE_LIBXC_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        parameters of the functionals
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_KE_LIBXC_SCALE = Quantity(
        type=str,
        shape=[],
        description='''
        scaling factors of the functionals
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_KE_LIBXC_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_LDA_HOLE_T_C_LR(MSection):
    '''
    LDA exchange hole model in truncated coulomb potential
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_LDA_HOLE_T_C_LR_CUTOFF_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines cutoff for lower integration boundary
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_LDA_HOLE_T_C_LR_SCALE_X = Quantity(
        type=str,
        shape=[],
        description='''
        scales the exchange part of the functional
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_LDA_HOLE_T_C_LR_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_LIBXC(MSection):
    '''
    Uses functionals from LIBXC, see also
    http://www.tddft.org/programs/octopus/wiki/index.php/Libxc
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_LIBXC_FUNCTIONAL = Quantity(
        type=str,
        shape=[],
        description='''
        names of the functionals, see also
        http://www.tddft.org/programs/octopus/wiki/index.php/Libxc:manual .The precise
        list of available functionals depends on the version of libxc interfaced
        (currently 2.0.1).
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_LIBXC_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        parameters of the functionals
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_LIBXC_SCALE = Quantity(
        type=str,
        shape=[],
        description='''
        scaling factors of the functionals
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_LIBXC_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_LYP_ADIABATIC(MSection):
    '''
    Uses the LYP correlation functional in an adiabatic fashion
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_LYP_ADIABATIC_LAMBDA = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the parameter of the adiabatic curve.
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_LYP_ADIABATIC_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_LYP(MSection):
    '''
    Uses the LYP functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_LYP_SCALE_C = Quantity(
        type=str,
        shape=[],
        description='''
        scales the correlation part of the functional
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_LYP_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_OPTX(MSection):
    '''
    Uses the OPTX functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_OPTX_SCALE_X = Quantity(
        type=str,
        shape=[],
        description='''
        scales the exchange part of the functional
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_OPTX_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_P86C(MSection):
    '''
    Uses the P86C functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_P86C_SCALE_C = Quantity(
        type=str,
        shape=[],
        description='''
        scales the correlation part of the functional
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_P86C_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_PADE(MSection):
    '''
    Uses the PADE functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_PADE_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_PBE_HOLE_T_C_LR(MSection):
    '''
    PBE exchange hole model in trucanted coulomb potential
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_PBE_HOLE_T_C_LR_CUTOFF_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines cutoff for lower integration boundary
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_PBE_HOLE_T_C_LR_SCALE_X = Quantity(
        type=str,
        shape=[],
        description='''
        scales the exchange part of the functional
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_PBE_HOLE_T_C_LR_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_PBE(MSection):
    '''
    Uses the PBE functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_PBE_PARAMETRIZATION = Quantity(
        type=str,
        shape=[],
        description='''
        switches between the different parametrizations of the functional
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_PBE_SCALE_C = Quantity(
        type=str,
        shape=[],
        description='''
        scales the correlation part of the functional
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_PBE_SCALE_X = Quantity(
        type=str,
        shape=[],
        description='''
        scales the exchange part of the functional
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_PBE_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_PW92(MSection):
    '''
    Uses the PerdewWang correlation functional.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_PW92_PARAMETRIZATION = Quantity(
        type=str,
        shape=[],
        description='''
        Which one of parametrizations should be used
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_PW92_SCALE = Quantity(
        type=str,
        shape=[],
        description='''
        Scaling of the energy functional
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_PW92_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_PZ81(MSection):
    '''
    Uses the PZ functional.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_PZ81_PARAMETRIZATION = Quantity(
        type=str,
        shape=[],
        description='''
        Which one of parametrizations should be used
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_PZ81_SCALE_C = Quantity(
        type=str,
        shape=[],
        description='''
        scales the correlation part of the functional
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_PZ81_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_TFW(MSection):
    '''
    Uses the TFW functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_TFW_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_TF(MSection):
    '''
    Uses the TF functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_TF_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_TPSS(MSection):
    '''
    Uses the TPSS functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_TPSS_SCALE_C = Quantity(
        type=str,
        shape=[],
        description='''
        scales the correlation part of the functional
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_TPSS_SCALE_X = Quantity(
        type=str,
        shape=[],
        description='''
        scales the exchange part of the functional
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_TPSS_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_VWN(MSection):
    '''
    Uses the VWN functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_VWN_FUNCTIONAL_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Which version of the VWN functional should be used
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_VWN_SCALE_C = Quantity(
        type=str,
        shape=[],
        description='''
        scales the correlation part of the functional
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_VWN_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_XALPHA(MSection):
    '''
    Uses the XALPHA (SLATER) functional.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_XALPHA_SCALE_X = Quantity(
        type=str,
        shape=[],
        description='''
        scales the exchange part of the functional
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_XALPHA_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_XALPHA_XA = Quantity(
        type=str,
        shape=[],
        description='''
        Value of the xa parameter (this does not change the exponent, just the mixing)
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_XGGA(MSection):
    '''
    Uses one of the XGGA functionals (optimized versions of some of these functionals
    might be available outside this section).
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_XGGA_FUNCTIONAL = Quantity(
        type=str,
        shape=[],
        description='''
        Which one of the XGGA functionals should be used
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_XGGA_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_XWPBE(MSection):
    '''
    Uses the short range PBE functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_XWPBE_OMEGA = Quantity(
        type=str,
        shape=[],
        description='''
        screening parameter
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_XWPBE_SCALE_X0 = Quantity(
        type=str,
        shape=[],
        description='''
        scales the exchange part of the original hole PBE-functional
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_XWPBE_SCALE_X = Quantity(
        type=str,
        shape=[],
        description='''
        scales the exchange part of the functional
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_XWPBE_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL(MSection):
    '''
    The xc functional to use
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_XC_FUNCTIONAL_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Shortcut for the most common functional combinations.
        ''')

    x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_BECKE88_LR_ADIABATIC = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_BECKE88_LR_ADIABATIC'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_BECKE88_LR = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_BECKE88_LR'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_BECKE88 = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_BECKE88'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_BECKE97 = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_BECKE97'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_BECKE_ROUSSEL = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_BECKE_ROUSSEL'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_BEEF = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_BEEF'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_CS1 = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_CS1'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_GV09 = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_GV09'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_HCTH = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_HCTH'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_KE_GGA = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_KE_GGA'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_KE_LIBXC = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_KE_LIBXC'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_LDA_HOLE_T_C_LR = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_LDA_HOLE_T_C_LR'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_LIBXC = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_LIBXC'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_LYP_ADIABATIC = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_LYP_ADIABATIC'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_LYP = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_LYP'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_OPTX = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_OPTX'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_P86C = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_P86C'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_PADE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_PADE'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_PBE_HOLE_T_C_LR = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_PBE_HOLE_T_C_LR'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_PBE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_PBE'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_PW92 = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_PW92'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_PZ81 = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_PZ81'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_TFW = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_TFW'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_TF = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_TF'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_TPSS = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_TPSS'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_VWN = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_VWN'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_XALPHA = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_XALPHA'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_XGGA = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_XGGA'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_XWPBE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL_XWPBE'),
        repeats=True)


class x_cp2k_section_input_ATOM_METHOD_XC_XC_GRID(MSection):
    '''
    The xc parameters used when calculating the xc on the grid
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_XC_GRID_USE_FINER_GRID = Quantity(
        type=str,
        shape=[],
        description='''
        Uses a finer grid only to calculate the xc
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_GRID_XC_DERIV = Quantity(
        type=str,
        shape=[],
        description='''
        The method used to compute the derivatives
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_GRID_XC_SMOOTH_RHO = Quantity(
        type=str,
        shape=[],
        description='''
        The density smoothing used for the xc calculation
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_XC_POTENTIAL_SAOP(MSection):
    '''
    Uses the SAOP potential
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_XC_POTENTIAL_SAOP_ALPHA = Quantity(
        type=str,
        shape=[],
        description='''
        Value of the alpha parameter (default = 1.19).
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_POTENTIAL_SAOP_BETA = Quantity(
        type=str,
        shape=[],
        description='''
        Value of the beta parameter (default = 0.01).
        ''')

    x_cp2k_input_ATOM_METHOD_XC_XC_POTENTIAL_SAOP_K_RHO = Quantity(
        type=str,
        shape=[],
        description='''
        Value of the K_rho parameter (default = 0.42).
        ''')


class x_cp2k_section_input_ATOM_METHOD_XC_XC_POTENTIAL(MSection):
    '''
    The xc potential to use (CAREFUL: xc potential here refers to potentials that are not
    derived from an xc functional, but rather are modelled directly. Therefore there is no
    consistent xc energy available. To still get an energy expression, see ENERGY below
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_XC_POTENTIAL_ENERGY = Quantity(
        type=str,
        shape=[],
        description='''
        How to determine the total energy.
        ''')

    x_cp2k_section_input_ATOM_METHOD_XC_XC_POTENTIAL_SAOP = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_XC_POTENTIAL_SAOP'),
        repeats=True)


class x_cp2k_section_input_ATOM_METHOD_XC(MSection):
    '''
    parameters needed calculate the xc potential
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_XC_DENSITY_CUTOFF = Quantity(
        type=str,
        shape=[],
        description='''
        The cutoff on the density used by the xc calculation
        ''')

    x_cp2k_input_ATOM_METHOD_XC_DENSITY_SMOOTH_CUTOFF_RANGE = Quantity(
        type=str,
        shape=[],
        description='''
        Parameter for the smoothing procedure inxc calculation
        ''')

    x_cp2k_input_ATOM_METHOD_XC_FUNCTIONAL_ROUTINE = Quantity(
        type=str,
        shape=[],
        description='''
        Select the code for xc calculation
        ''')

    x_cp2k_input_ATOM_METHOD_XC_GRADIENT_CUTOFF = Quantity(
        type=str,
        shape=[],
        description='''
        The cutoff on the gradient of the density used by the xc calculation
        ''')

    x_cp2k_input_ATOM_METHOD_XC_TAU_CUTOFF = Quantity(
        type=str,
        shape=[],
        description='''
        The cutoff on tau used by the xc calculation
        ''')

    x_cp2k_section_input_ATOM_METHOD_XC_ADIABATIC_RESCALING = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_ADIABATIC_RESCALING'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_HF = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_HF'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_VDW_POTENTIAL = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_VDW_POTENTIAL'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_WF_CORRELATION'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_XC_FUNCTIONAL'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_XC_GRID = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_XC_GRID'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC_XC_POTENTIAL = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC_XC_POTENTIAL'),
        repeats=True)


class x_cp2k_section_input_ATOM_METHOD_ZMP_RESTART(MSection):
    '''
    Section used to specify the restart option in the ZMPprocedure, and the file that must
    be read.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_ZMP_RESTART_FILE_RESTART = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the filename containing the restart file density
        ''')


class x_cp2k_section_input_ATOM_METHOD_ZMP(MSection):
    '''
    Section used to specify ZMP Potentials.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_ZMP_DM = Quantity(
        type=str,
        shape=[],
        description='''
        read external density from density matrix
        ''')

    x_cp2k_input_ATOM_METHOD_ZMP_FILE_DENSITY = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the filename containing the target density
        ''')

    x_cp2k_input_ATOM_METHOD_ZMP_GRID_TOL = Quantity(
        type=str,
        shape=[],
        description='''
        Tolerance in the equivalence of read-grid in ZMP method
        ''')

    x_cp2k_input_ATOM_METHOD_ZMP_LAMBDA = Quantity(
        type=str,
        shape=[],
        description='''
        Parameter used for the constraint in ZMP method
        ''')

    x_cp2k_section_input_ATOM_METHOD_ZMP_RESTART = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_ZMP_RESTART'),
        repeats=True)


class x_cp2k_section_input_ATOM_METHOD(MSection):
    '''
    Section of information on method to use.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_METHOD_METHOD_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Type of electronic structure method to be used
        ''')

    x_cp2k_input_ATOM_METHOD_RELATIVISTIC = Quantity(
        type=str,
        shape=[],
        description='''
        Type of scalar relativistic method to be used
        ''')

    x_cp2k_section_input_ATOM_METHOD_EXTERNAL_VXC = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_EXTERNAL_VXC'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_XC = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_XC'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD_ZMP = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD_ZMP'),
        repeats=True)


class x_cp2k_section_input_ATOM_OPTIMIZATION(MSection):
    '''
    Section of information on optimization thresholds and algorithms.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_OPTIMIZATION_DAMPING = Quantity(
        type=str,
        shape=[],
        description='''
        Damping parameter for extrapolation method
        ''')

    x_cp2k_input_ATOM_OPTIMIZATION_EPS_DIIS = Quantity(
        type=str,
        shape=[],
        description='''
        Starting DIIS method at convergence to EPS_DIIS
        ''')

    x_cp2k_input_ATOM_OPTIMIZATION_EPS_SCF = Quantity(
        type=str,
        shape=[],
        description='''
        Convergence criterion for SCF
        ''')

    x_cp2k_input_ATOM_OPTIMIZATION_MAX_ITER = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum number of iterations for optimization
        ''')

    x_cp2k_input_ATOM_OPTIMIZATION_N_DIIS = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum number of DIIS vectors
        ''')


class x_cp2k_section_input_ATOM_POTENTIAL_GTH_POTENTIAL(MSection):
    '''
    Section used to specify Potentials.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_POTENTIAL_GTH_POTENTIAL_DEFAULT_KEYWORD = Quantity(
        type=str,
        shape=[],
        description='''
        CP2K Pseudo Potential Standard Format (GTH, ALL or KG)
        ''')


class x_cp2k_section_input_ATOM_POTENTIAL(MSection):
    '''
    Section of information on potential.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_POTENTIAL_CONFINEMENT = Quantity(
        type=str,
        shape=[],
        description='''
        Definition of confinement potential
        ''')

    x_cp2k_input_ATOM_POTENTIAL_POTENTIAL_FILE_NAME = Quantity(
        type=str,
        shape=[],
        description='''
        Name of the pseudo potential file, may include a path
        ''')

    x_cp2k_input_ATOM_POTENTIAL_POTENTIAL_NAME = Quantity(
        type=str,
        shape=[],
        description='''
        The name of the pseudopotential for the defined kind.
        ''')

    x_cp2k_input_ATOM_POTENTIAL_PSEUDO_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Pseudopotential type
        ''')

    x_cp2k_section_input_ATOM_POTENTIAL_GTH_POTENTIAL = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_POTENTIAL_GTH_POTENTIAL'),
        repeats=True)


class x_cp2k_section_input_ATOM_POWELL(MSection):
    '''
    Section defines basic parameters for Powell optimization
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_POWELL_ACCURACY = Quantity(
        type=str,
        shape=[],
        description='''
        Final accuracy requested in optimization (RHOEND)
        ''')

    x_cp2k_input_ATOM_POWELL_MAX_FUN = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum number of function evaluations
        ''')

    x_cp2k_input_ATOM_POWELL_RCOV_MULTIPLICATION = Quantity(
        type=str,
        shape=[],
        description='''
        Multiply Rcov integration limit for charge conservation
        ''')

    x_cp2k_input_ATOM_POWELL_STEP_SIZE = Quantity(
        type=str,
        shape=[],
        description='''
        Initial step size for search algorithm (RHOBEG)
        ''')

    x_cp2k_input_ATOM_POWELL_TARGET_POT_SEMICORE = Quantity(
        type=str,
        shape=[],
        description='''
        Target accuracy for semicore state eigenvalues in pseudopotential optimization
        ''')

    x_cp2k_input_ATOM_POWELL_TARGET_POT_VALENCE = Quantity(
        type=str,
        shape=[],
        description='''
        Target accuracy for valence state eigenvalues in pseudopotential optimization
        ''')

    x_cp2k_input_ATOM_POWELL_TARGET_POT_VIRTUAL = Quantity(
        type=str,
        shape=[],
        description='''
        Target accuracy for virtual state eigenvalues in pseudopotential optimization
        ''')

    x_cp2k_input_ATOM_POWELL_WEIGHT_ELECTRON_CONFIGURATION = Quantity(
        type=str,
        shape=[],
        description='''
        Weight for different electronic states in optimization
        ''')

    x_cp2k_input_ATOM_POWELL_WEIGHT_METHOD = Quantity(
        type=str,
        shape=[],
        description='''
        Weight for different methods in optimization
        ''')

    x_cp2k_input_ATOM_POWELL_WEIGHT_POT_NODE = Quantity(
        type=str,
        shape=[],
        description='''
        Weight for node mismatch in pseudopotential optimization
        ''')

    x_cp2k_input_ATOM_POWELL_WEIGHT_POT_SEMICORE = Quantity(
        type=str,
        shape=[],
        description='''
        Weight for semi core states in pseudopotential optimization
        ''')

    x_cp2k_input_ATOM_POWELL_WEIGHT_POT_VALENCE = Quantity(
        type=str,
        shape=[],
        description='''
        Weight for valence states in pseudopotential optimization
        ''')

    x_cp2k_input_ATOM_POWELL_WEIGHT_POT_VIRTUAL = Quantity(
        type=str,
        shape=[],
        description='''
        Weight for virtual states in pseudopotential optimization
        ''')

    x_cp2k_input_ATOM_POWELL_WEIGHT_PSIR0 = Quantity(
        type=str,
        shape=[],
        description='''
        Weight for the wavefunctions at r=0 (only occupied states)
        ''')


class x_cp2k_section_input_ATOM_PP_BASIS_BASIS(MSection):
    '''
    Section used to specify a general basis set for QM calculations.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_PP_BASIS_BASIS_DEFAULT_KEYWORD = Quantity(
        type=str,
        shape=[],
        description='''
        CP2K Basis Set Standard Format
        ''')


class x_cp2k_section_input_ATOM_PP_BASIS(MSection):
    '''
    Section of basis set information for pseudopotential calculations.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_PP_BASIS_BASIS_SET_FILE_NAME = Quantity(
        type=str,
        shape=[],
        description='''
        Name of the basis set file, may include a path
        ''')

    x_cp2k_input_ATOM_PP_BASIS_BASIS_SET = Quantity(
        type=str,
        shape=[],
        description='''
        The contracted Gaussian basis set
        ''')

    x_cp2k_input_ATOM_PP_BASIS_BASIS_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Basis set type
        ''')

    x_cp2k_input_ATOM_PP_BASIS_D_EXPONENTS = Quantity(
        type=str,
        shape=[],
        description='''
        Exponents for d functions
        ''')

    x_cp2k_input_ATOM_PP_BASIS_D_QUANTUM_NUMBERS = Quantity(
        type=str,
        shape=[],
        description='''
        Main quantum numbers for d functions
        ''')

    x_cp2k_input_ATOM_PP_BASIS_EPS_EIGENVALUE = Quantity(
        type=str,
        shape=[],
        description='''
        Cutoff of overlap matrix eigenvalues included into basis
        ''')

    x_cp2k_input_ATOM_PP_BASIS_F_EXPONENTS = Quantity(
        type=str,
        shape=[],
        description='''
        Exponents for f functions
        ''')

    x_cp2k_input_ATOM_PP_BASIS_F_QUANTUM_NUMBERS = Quantity(
        type=str,
        shape=[],
        description='''
        Main quantum numbers for f functions
        ''')

    x_cp2k_input_ATOM_PP_BASIS_GEO_START_VALUE = Quantity(
        type=str,
        shape=[],
        description='''
        Geometrical basis: starting value a in a*C^k
        ''')

    x_cp2k_input_ATOM_PP_BASIS_GEOMETRICAL_FACTOR = Quantity(
        type=str,
        shape=[],
        description='''
        Geometrical basis: factor C in a*C^k
        ''')

    x_cp2k_input_ATOM_PP_BASIS_GRID_POINTS = Quantity(
        type=str,
        shape=[],
        description='''
        Number of radial grid points
        ''')

    x_cp2k_input_ATOM_PP_BASIS_NUM_GTO = Quantity(
        type=str,
        shape=[],
        description='''
        Number of Gaussian type functions for s, p, d, ...
        ''')

    x_cp2k_input_ATOM_PP_BASIS_NUM_SLATER = Quantity(
        type=str,
        shape=[],
        description='''
        Number of Slater type functions for s, p, d, ...
        ''')

    x_cp2k_input_ATOM_PP_BASIS_P_EXPONENTS = Quantity(
        type=str,
        shape=[],
        description='''
        Exponents for p functions
        ''')

    x_cp2k_input_ATOM_PP_BASIS_P_QUANTUM_NUMBERS = Quantity(
        type=str,
        shape=[],
        description='''
        Main quantum numbers for p functions
        ''')

    x_cp2k_input_ATOM_PP_BASIS_QUADRATURE = Quantity(
        type=str,
        shape=[],
        description='''
        Algorithm to construct the atomic radial grids
        ''')

    x_cp2k_input_ATOM_PP_BASIS_S_EXPONENTS = Quantity(
        type=str,
        shape=[],
        description='''
        Exponents for s functions
        ''')

    x_cp2k_input_ATOM_PP_BASIS_S_QUANTUM_NUMBERS = Quantity(
        type=str,
        shape=[],
        description='''
        Main quantum numbers for s functions
        ''')

    x_cp2k_input_ATOM_PP_BASIS_START_INDEX = Quantity(
        type=str,
        shape=[],
        description='''
        Starting index for Geometrical Basis sets
        ''')

    x_cp2k_section_input_ATOM_PP_BASIS_BASIS = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_PP_BASIS_BASIS'),
        repeats=True)


class x_cp2k_section_input_ATOM(MSection):
    '''
    Section handling input for atomic calculations.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_ATOM_ATOMIC_NUMBER = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the atomic number
        ''')

    x_cp2k_input_ATOM_CALCULATE_STATES = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the number of states calculated per l value
        ''')

    x_cp2k_input_ATOM_CORE = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the core electrons for a pseudopotential
        ''')

    x_cp2k_input_ATOM_COULOMB_INTEGRALS = Quantity(
        type=str,
        shape=[],
        description='''
        Method to calculate Coulomb integrals
        ''')

    x_cp2k_input_ATOM_ELECTRON_CONFIGURATION = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the electron configuration. Optional the multiplicity (m) and a core
        state [XX] can be declared
        ''')

    x_cp2k_input_ATOM_ELEMENT = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the element to be calculated
        ''')

    x_cp2k_input_ATOM_EXCHANGE_INTEGRALS = Quantity(
        type=str,
        shape=[],
        description='''
        Method to calculate Exchange integrals
        ''')

    x_cp2k_input_ATOM_MAX_ANGULAR_MOMENTUM = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the largest angular momentum calculated [0-3]
        ''')

    x_cp2k_input_ATOM_RUN_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Type of run that you want to perform
        [ENERGY,BASIS_OPTIMIZATION,PSEUDOPOTENTIAL_OPTIMIZATION,,...]
        ''')

    x_cp2k_section_input_ATOM_AE_BASIS = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_AE_BASIS'),
        repeats=True)

    x_cp2k_section_input_ATOM_METHOD = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_METHOD'),
        repeats=True)

    x_cp2k_section_input_ATOM_OPTIMIZATION = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_OPTIMIZATION'),
        repeats=True)

    x_cp2k_section_input_ATOM_POTENTIAL = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_POTENTIAL'),
        repeats=True)

    x_cp2k_section_input_ATOM_POWELL = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_POWELL'),
        repeats=True)

    x_cp2k_section_input_ATOM_PP_BASIS = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_ATOM_PP_BASIS'),
        repeats=True)


class x_cp2k_section_input_DEBUG_PROGRAM_RUN_INFO(MSection):
    '''
    Controls the printing of the DEBUG specific output
    '''

    m_def = Section(validate=False)

    x_cp2k_input_DEBUG_PROGRAM_RUN_INFO_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_DEBUG_PROGRAM_RUN_INFO_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_DEBUG_PROGRAM_RUN_INFO_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_DEBUG_PROGRAM_RUN_INFO_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_DEBUG_PROGRAM_RUN_INFO_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_DEBUG(MSection):
    '''
    Section to setup parameters for debug runs.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_DEBUG_DEBUG_FORCES = Quantity(
        type=str,
        shape=[],
        description='''
        Activates the debugging of the atomic forces
        ''')

    x_cp2k_input_DEBUG_DEBUG_STRESS_TENSOR = Quantity(
        type=str,
        shape=[],
        description='''
        Activates the debugging of the stress tensor
        ''')

    x_cp2k_input_DEBUG_DX = Quantity(
        type=str,
        shape=[],
        description='''
        Increment for the calculation of the numerical derivatives
        ''')

    x_cp2k_input_DEBUG_EPS_NO_ERROR_CHECK = Quantity(
        type=str,
        shape=[],
        description='''
        The mismatch between the numerical and the analytical value is not checked for
        analytical values smaller than this threshold value
        ''')

    x_cp2k_input_DEBUG_STOP_ON_MISMATCH = Quantity(
        type=str,
        shape=[],
        description='''
        Stop the debug run when a mismatch between the numerical and the analytical value
        is detected
        ''')

    x_cp2k_section_input_DEBUG_PROGRAM_RUN_INFO = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_DEBUG_PROGRAM_RUN_INFO'),
        repeats=True)


class x_cp2k_section_input_EXT_RESTART(MSection):
    '''
    Section for external restart, specifies an external input file where to take
    positions,...
    '''

    m_def = Section(validate=False)

    x_cp2k_input_EXT_RESTART_BINARY_RESTART_FILE_NAME = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the name of an additional restart file from which selected input
        sections are read in binary format (see SPLIT_RESTART_FILE).
        ''')

    x_cp2k_input_EXT_RESTART_CUSTOM_PATH = Quantity(
        type=str,
        shape=[],
        description='''
        Restarts the given path from the EXTERNAL file. Allows a major flexibility for
        restarts.
        ''')

    x_cp2k_input_EXT_RESTART_RESTART_AVERAGES = Quantity(
        type=str,
        shape=[],
        description='''
        Restarts information for AVERAGES.
        ''')

    x_cp2k_input_EXT_RESTART_RESTART_BAND = Quantity(
        type=str,
        shape=[],
        description='''
        Restarts positions and velocities of the Band.
        ''')

    x_cp2k_input_EXT_RESTART_RESTART_BAROSTAT_THERMOSTAT = Quantity(
        type=str,
        shape=[],
        description='''
        Restarts the barostat thermostat from the external file
        ''')

    x_cp2k_input_EXT_RESTART_RESTART_BAROSTAT = Quantity(
        type=str,
        shape=[],
        description='''
        Restarts the barostat from the external file
        ''')

    x_cp2k_input_EXT_RESTART_RESTART_BSSE = Quantity(
        type=str,
        shape=[],
        description='''
        Restarts information for BSSE calculations.
        ''')

    x_cp2k_input_EXT_RESTART_RESTART_CELL = Quantity(
        type=str,
        shape=[],
        description='''
        Restarts the cell (and cell_ref) from the EXTERNAL file
        ''')

    x_cp2k_input_EXT_RESTART_RESTART_CONSTRAINT = Quantity(
        type=str,
        shape=[],
        description='''
        Restarts constraint section. It's necessary when doing restraint calculation to
        have a perfect energy conservation. For constraints only it's use is optional.
        ''')

    x_cp2k_input_EXT_RESTART_RESTART_CORE_POS = Quantity(
        type=str,
        shape=[],
        description='''
        Takes the positions of the cores from the external file (only if shell-model)
        ''')

    x_cp2k_input_EXT_RESTART_RESTART_CORE_VELOCITY = Quantity(
        type=str,
        shape=[],
        description='''
        Takes the velocities of the shells from the external file (only if shell-model)
        ''')

    x_cp2k_input_EXT_RESTART_RESTART_COUNTERS = Quantity(
        type=str,
        shape=[],
        description='''
        Restarts the counters in MD schemes
        ''')

    x_cp2k_input_EXT_RESTART_RESTART_DEFAULT = Quantity(
        type=str,
        shape=[],
        description='''
        This keyword controls the default value for all possible  restartable keywords,
        unless explicitly defined. For example setting this keyword to .FALSE. does not
        restart any quantity. If, at the  same time, one keyword is set to .TRUE. only
        that quantity will be restarted.
        ''')

    x_cp2k_input_EXT_RESTART_RESTART_DIMER = Quantity(
        type=str,
        shape=[],
        description='''
        Restarts information for DIMER geometry optimizations.
        ''')

    x_cp2k_input_EXT_RESTART_RESTART_FILE_NAME = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the name of restart file (or any other input file) to be read. Only
        fields relevant to a restart will be used (unless switched off with the keywords
        in this section)
        ''')

    x_cp2k_input_EXT_RESTART_RESTART_HELIUM_DENSITIES = Quantity(
        type=str,
        shape=[],
        description='''
        Restarts helium density distributions from PINT%HELIUM%RHO.
        ''')

    x_cp2k_input_EXT_RESTART_RESTART_HELIUM_FORCE = Quantity(
        type=str,
        shape=[],
        description='''
        Restart helium forces exerted on the solute from PINT%HELIUM%FORCE.
        ''')

    x_cp2k_input_EXT_RESTART_RESTART_HELIUM_PERMUTATION = Quantity(
        type=str,
        shape=[],
        description='''
        Restart helium permutation state from PINT%HELIUM%PERM.
        ''')

    x_cp2k_input_EXT_RESTART_RESTART_HELIUM_POS = Quantity(
        type=str,
        shape=[],
        description='''
        Restart helium positions from PINT%HELIUM%COORD.
        ''')

    x_cp2k_input_EXT_RESTART_RESTART_HELIUM_RNG = Quantity(
        type=str,
        shape=[],
        description='''
        Restarts helium random number generators from PINT%HELIUM%RNG_STATE.
        ''')

    x_cp2k_input_EXT_RESTART_RESTART_METADYNAMICS = Quantity(
        type=str,
        shape=[],
        description='''
        Restarts hills from a previous metadynamics run from the EXTERNAL file
        ''')

    x_cp2k_input_EXT_RESTART_RESTART_OPTIMIZE_INPUT_VARIABLES = Quantity(
        type=str,
        shape=[],
        description='''
        Restart with the optimize input variables
        ''')

    x_cp2k_input_EXT_RESTART_RESTART_PINT_GLE = Quantity(
        type=str,
        shape=[],
        description='''
        Restart GLE thermostat for beads from PINT%GLE.
        ''')

    x_cp2k_input_EXT_RESTART_RESTART_PINT_NOSE = Quantity(
        type=str,
        shape=[],
        description='''
        Restart Nose thermostat for beads from PINT%NOSE.
        ''')

    x_cp2k_input_EXT_RESTART_RESTART_PINT_POS = Quantity(
        type=str,
        shape=[],
        description='''
        Restart bead positions from PINT%BEADS%COORD.
        ''')

    x_cp2k_input_EXT_RESTART_RESTART_PINT_VEL = Quantity(
        type=str,
        shape=[],
        description='''
        Restart bead velocities from PINT%BEADS%VELOCITY.
        ''')

    x_cp2k_input_EXT_RESTART_RESTART_POS = Quantity(
        type=str,
        shape=[],
        description='''
        Takes the positions from the external file
        ''')

    x_cp2k_input_EXT_RESTART_RESTART_QMMM = Quantity(
        type=str,
        shape=[],
        description='''
        Restarts the following specific QMMM info: translation vectors.
        ''')

    x_cp2k_input_EXT_RESTART_RESTART_RANDOMG = Quantity(
        type=str,
        shape=[],
        description='''
        Restarts the random number generator from the external file
        ''')

    x_cp2k_input_EXT_RESTART_RESTART_RTP = Quantity(
        type=str,
        shape=[],
        description='''
        Restarts information for REAL TIME PROPAGATION and EHRENFEST DYNAMICS.
        ''')

    x_cp2k_input_EXT_RESTART_RESTART_SHELL_POS = Quantity(
        type=str,
        shape=[],
        description='''
        Takes the positions of the shells from the external file (only if shell-model)
        ''')

    x_cp2k_input_EXT_RESTART_RESTART_SHELL_THERMOSTAT = Quantity(
        type=str,
        shape=[],
        description='''
        Restarts the shell thermostat from the external file
        ''')

    x_cp2k_input_EXT_RESTART_RESTART_SHELL_VELOCITY = Quantity(
        type=str,
        shape=[],
        description='''
        Takes the velocities of the shells from the external file (only if shell-model)
        ''')

    x_cp2k_input_EXT_RESTART_RESTART_THERMOSTAT = Quantity(
        type=str,
        shape=[],
        description='''
        Restarts the nose thermostats of the particles from the EXTERNAL file
        ''')

    x_cp2k_input_EXT_RESTART_RESTART_VEL = Quantity(
        type=str,
        shape=[],
        description='''
        Takes the velocities from the external file
        ''')

    x_cp2k_input_EXT_RESTART_RESTART_WALKERS = Quantity(
        type=str,
        shape=[],
        description='''
        Restarts walkers informations from a previous metadynamics run from the EXTERNAL
        file
        ''')


class x_cp2k_section_input_FARMING_JOB(MSection):
    '''
    description of the jobs to be executed
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FARMING_JOB_DEPENDENCIES = Quantity(
        type=str,
        shape=[],
        description='''
        specifies a list of JOB_IDs on which the current job depends. The current job will
        not be executed before all the dependencies have finished. The keyword requires a
        MASTER_SLAVE farming run. Beyond the default case, some special cases might arise:
        1) circular dependencies will lead to a deadlock. 2) This keyword is not
        compatible with CYCLE. 3) MAX_JOBS_PER_GROUP is ignored (though only a total of
        MAX_JOBS_PER_GROUP*NGROUPS jobs will be executed) 4) dependencies on jobs that
        will not be executed (due to RESTART or MAX_JOBS_PER_GROUP) are ignored.
        Additionally, note that, on some file systems,  output (restart) files might not
        be immediately available on all compute nodes,potentially resulting in unexpected
        failures.
        ''')

    x_cp2k_input_FARMING_JOB_DIRECTORY = Quantity(
        type=str,
        shape=[],
        description='''
        the directory in which the job should be executed
        ''')

    x_cp2k_input_FARMING_JOB_INPUT_FILE_NAME = Quantity(
        type=str,
        shape=[],
        description='''
        the filename of the input file
        ''')

    x_cp2k_input_FARMING_JOB_JOB_ID = Quantity(
        type=str,
        shape=[],
        description='''
        An ID used to indentify a job in DEPENDENCIES. JOB_IDs do not need to be unique,
        dependencies will be on all jobs with a given ID. If no JOB_ID is given, the index
        of the &JOB section in the input file will be used.
        ''')

    x_cp2k_input_FARMING_JOB_OUTPUT_FILE_NAME = Quantity(
        type=str,
        shape=[],
        description='''
        the filename of the output file, if not specified will use the project name in the
        &GLOBAL section.
        ''')


class x_cp2k_section_input_FARMING_PROGRAM_RUN_INFO(MSection):
    '''
    controls the printing of FARMING specific output
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FARMING_PROGRAM_RUN_INFO_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FARMING_PROGRAM_RUN_INFO_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FARMING_PROGRAM_RUN_INFO_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FARMING_PROGRAM_RUN_INFO_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FARMING_PROGRAM_RUN_INFO_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_FARMING_RESTART(MSection):
    '''
    controls the printing of the restart for FARMING.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FARMING_RESTART_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FARMING_RESTART_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FARMING_RESTART_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FARMING_RESTART_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FARMING_RESTART_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_FARMING(MSection):
    '''
    Describes a farming job, in which multiple inputs are executed.

    The RUN_TYPE in the global section has to be set to NONE for FARMING.

    The different groups are executed in parallel. The jobs inside the same groups in
    series.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FARMING_CYCLE = Quantity(
        type=str,
        shape=[],
        description='''
        If farming should process all jobs in a cyclic way, stopping only if
        MAX_JOBS_PER_GROUP is exceeded.
        ''')

    x_cp2k_input_FARMING_DO_RESTART = Quantity(
        type=str,
        shape=[],
        description='''
        Restart a farming job (and should pick up where the previous left off)
        ''')

    x_cp2k_input_FARMING_GROUP_PARTITION = Quantity(
        type=str,
        shape=[],
        description='''
        gives the exact number of processors for each group.
        ''')

    x_cp2k_input_FARMING_GROUP_SIZE = Quantity(
        type=str,
        shape=[],
        description='''
        Gives the preferred size of a working group, groups will always be equal or larger
        than this size.
        ''')

    x_cp2k_input_FARMING_MASTER_SLAVE = Quantity(
        type=str,
        shape=[],
        description='''
        If a master-slave setup should be employed, in which one process is used to
        distribute the tasks. This is most useful to load-balance if not all jobs have the
        same length, and a lot of CPUs/groups are availabe.
        ''')

    x_cp2k_input_FARMING_MAX_JOBS_PER_GROUP = Quantity(
        type=str,
        shape=[],
        description='''
        maximum number of jobs executed per group
        ''')

    x_cp2k_input_FARMING_NGROUPS = Quantity(
        type=str,
        shape=[],
        description='''
        Gives the preferred number of working groups.
        ''')

    x_cp2k_input_FARMING_RESTART_FILE_NAME = Quantity(
        type=str,
        shape=[],
        description='''
        Name of the restart file to use for restarting a FARMING run. If not specified the
        name is determined from PROJECT name.
        ''')

    x_cp2k_input_FARMING_WAIT_TIME = Quantity(
        type=str,
        shape=[],
        description='''
        Time to wait [s] for a new task if no task is currently available, make this zero
        if no clock is available
        ''')

    x_cp2k_section_input_FARMING_JOB = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FARMING_JOB'),
        repeats=True)

    x_cp2k_section_input_FARMING_PROGRAM_RUN_INFO = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FARMING_PROGRAM_RUN_INFO'),
        repeats=True)

    x_cp2k_section_input_FARMING_RESTART = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FARMING_RESTART'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_BSSE_CONFIGURATION(MSection):
    '''
    Specify additional parameters for the combinatorial configurations.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_BSSE_CONFIGURATION_CHARGE = Quantity(
        type=str,
        shape=[],
        description='''
        The total charge for each fragment.
        ''')

    x_cp2k_input_FORCE_EVAL_BSSE_CONFIGURATION_GLB_CONF = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the global configuration using 1 or 0.
        ''')

    x_cp2k_input_FORCE_EVAL_BSSE_CONFIGURATION_MULTIPLICITY = Quantity(
        type=str,
        shape=[],
        description='''
        Specify for each fragment the multiplicity. Two times the total spin plus one.
        Specify 3 for a triplet, 4 for a quartet,and so on. Default is 1 (singlet) for an
        even number and 2 (doublet) for an odd number of electrons.
        ''')

    x_cp2k_input_FORCE_EVAL_BSSE_CONFIGURATION_SUB_CONF = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the subconfiguration using 1 or 0 belonging to the global configuration.
        ''')


class x_cp2k_section_input_FORCE_EVAL_BSSE_FRAGMENT_ENERGIES(MSection):
    '''
    This section contains the energies of the fragments already computed. It is useful as
    a summary and specifically for restarting BSSE runs.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_BSSE_FRAGMENT_ENERGIES_DEFAULT_KEYWORD = Quantity(
        type=str,
        shape=[],
        description='''
        The energy computed for each fragment
        ''')


class x_cp2k_section_input_FORCE_EVAL_BSSE_FRAGMENT(MSection):
    '''
    Specify the atom number belonging to this fragment.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_BSSE_FRAGMENT_LIST = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies a list of atoms.
        ''')


class x_cp2k_section_input_FORCE_EVAL_BSSE(MSection):
    '''
    This section is used to set up the BSSE calculation. It also requires that for each
    atomic kind X a kind X_ghost is present, with the GHOST keyword specified, in addition
    to the other required fields.
    '''

    m_def = Section(validate=False)

    x_cp2k_section_input_FORCE_EVAL_BSSE_CONFIGURATION = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_BSSE_CONFIGURATION'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_BSSE_FRAGMENT_ENERGIES = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_BSSE_FRAGMENT_ENERGIES'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_BSSE_FRAGMENT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_BSSE_FRAGMENT'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT_ALMO_SCF_ALMO_OPTIMIZER_DIIS(MSection):
    '''
    Controls the iterative DIIS-accelerated optimization of block-diagonal ALMOs.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_ALMO_SCF_ALMO_OPTIMIZER_DIIS_EPS_ERROR = Quantity(
        type=str,
        shape=[],
        description='''
        Target value of the MAX norm of the error
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_ALMO_SCF_ALMO_OPTIMIZER_DIIS_MAX_ITER = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum number of iterations
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_ALMO_SCF_ALMO_OPTIMIZER_DIIS_N_DIIS = Quantity(
        type=str,
        shape=[],
        description='''
        Number of error vectors to be used in the DIIS optimization procedure
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_ALMO_SCF_ALMO_OPTIMIZER_PCG(MSection):
    '''
    Controls the PCG optimization of block-diagonal ALMOs.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_ALMO_SCF_ALMO_OPTIMIZER_PCG_CONJUGATOR = Quantity(
        type=str,
        shape=[],
        description='''
        Various methods to compute step directions in the PCG optimization
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_ALMO_SCF_ALMO_OPTIMIZER_PCG_EPS_ERROR = Quantity(
        type=str,
        shape=[],
        description='''
        Target value of the MAX norm of the error
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_ALMO_SCF_ALMO_OPTIMIZER_PCG_LIN_SEARCH_EPS_ERROR = Quantity(
        type=str,
        shape=[],
        description='''
        Target value of the gradient norm during the linear search
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_ALMO_SCF_ALMO_OPTIMIZER_PCG_LIN_SEARCH_STEP_SIZE_GUESS = Quantity(
        type=str,
        shape=[],
        description='''
        The size of the first step in the linear search
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_ALMO_SCF_ALMO_OPTIMIZER_PCG_MAX_ITER_OUTER_LOOP = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum number of iterations in the outer loop. Use the outer loop to update the
        preconditioner and reset the conjugator. This can speed up convergence
        significantly.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_ALMO_SCF_ALMO_OPTIMIZER_PCG_MAX_ITER = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum number of iterations
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_ALMO_SCF_ALMO_OPTIMIZER_PCG_PRECONDITIONER = Quantity(
        type=str,
        shape=[],
        description='''
        Select a preconditioner for the conjugate gradient optimization
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_ALMO_SCF_XALMO_OPTIMIZER_PCG(MSection):
    '''
    Controls the PCG optimization of extended ALMOs.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_ALMO_SCF_XALMO_OPTIMIZER_PCG_CONJUGATOR = Quantity(
        type=str,
        shape=[],
        description='''
        Various methods to compute step directions in the PCG optimization
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_ALMO_SCF_XALMO_OPTIMIZER_PCG_EPS_ERROR = Quantity(
        type=str,
        shape=[],
        description='''
        Target value of the MAX norm of the error
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_ALMO_SCF_XALMO_OPTIMIZER_PCG_LIN_SEARCH_EPS_ERROR = Quantity(
        type=str,
        shape=[],
        description='''
        Target value of the gradient norm during the linear search
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_ALMO_SCF_XALMO_OPTIMIZER_PCG_LIN_SEARCH_STEP_SIZE_GUESS = Quantity(
        type=str,
        shape=[],
        description='''
        The size of the first step in the linear search
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_ALMO_SCF_XALMO_OPTIMIZER_PCG_MAX_ITER_OUTER_LOOP = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum number of iterations in the outer loop. Use the outer loop to update the
        preconditioner and reset the conjugator. This can speed up convergence
        significantly.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_ALMO_SCF_XALMO_OPTIMIZER_PCG_MAX_ITER = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum number of iterations
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_ALMO_SCF_XALMO_OPTIMIZER_PCG_PRECONDITIONER = Quantity(
        type=str,
        shape=[],
        description='''
        Select a preconditioner for the conjugate gradient optimization
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_ALMO_SCF(MSection):
    '''
    Settings for a class of efficient linear scaling methods based on absolutely localized
    orbitals (ALMOs). ALMO methods are currently restricted to closed-shell molecular
    systems.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_ALMO_SCF_ALMO_ALGORITHM = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the algorithm to update block-diagonal ALMOs during the SCF procedure.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_ALMO_SCF_ALMO_SCF_GUESS = Quantity(
        type=str,
        shape=[],
        description='''
        The method to generate initial ALMOs.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_ALMO_SCF_DELOCALIZE_METHOD = Quantity(
        type=str,
        shape=[],
        description='''
        Methods to reintroduce electron delocalization, which is excluded with the block-
        diagonal ALMO reference. Electron delocalization can be computed using either
        fully delocalized MOs or spatially restricted ALMOs (called extended ALMOs or
        XALMOs). All methods below use a PCG optimizer controlled by XALMO_OPTIMIZER_PCG.
        The only exception is the non-iterative XALMO_1DIAG.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_ALMO_SCF_EPS_FILTER = Quantity(
        type=str,
        shape=[],
        description='''
        Threshold for the matrix sparsity filter
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_ALMO_SCF_XALMO_R_CUTOFF_FACTOR = Quantity(
        type=str,
        shape=[],
        description='''
        Controls the localization radius of XALMOs: R_cutoff =
        XALMO_R_CUTOFF_FACTOR*(radius(at1)+radius(at2))
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_ALMO_SCF_ALMO_OPTIMIZER_DIIS = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_ALMO_SCF_ALMO_OPTIMIZER_DIIS'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_ALMO_SCF_ALMO_OPTIMIZER_PCG = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_ALMO_SCF_ALMO_OPTIMIZER_PCG'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_ALMO_SCF_XALMO_OPTIMIZER_PCG = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_ALMO_SCF_XALMO_OPTIMIZER_PCG'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT_AUXILIARY_DENSITY_MATRIX_METHOD(MSection):
    '''
    Parameters needed for the ADMM method.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_AUXILIARY_DENSITY_MATRIX_METHOD_ADMM_PURIFICATION_METHOD = Quantity(
        type=str,
        shape=[],
        description='''
        Method that shall be used for wavefunction fitting. Use MO_DIAG for MD.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_AUXILIARY_DENSITY_MATRIX_METHOD_BLOCK_LIST = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies a list of atoms.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_AUXILIARY_DENSITY_MATRIX_METHOD_EPS_FILTER = Quantity(
        type=str,
        shape=[],
        description='''
        Define accuracy of DBCSR operations
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_AUXILIARY_DENSITY_MATRIX_METHOD_EXCH_CORRECTION_FUNC = Quantity(
        type=str,
        shape=[],
        description='''
        Exchange functional which is used for the ADMM correction.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_AUXILIARY_DENSITY_MATRIX_METHOD_EXCH_SCALING_MODEL = Quantity(
        type=str,
        shape=[],
        description='''
        Scaling of the exchange correction calculated by the auxiliary density matrix.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_AUXILIARY_DENSITY_MATRIX_METHOD_METHOD = Quantity(
        type=str,
        shape=[],
        description='''
        Method that shall be used for wavefunction fitting. Use BASIS_PROJECTION for MD.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_DENSITY_FITTING_PROGRAM_RUN_INFO(MSection):
    '''
    Controls the printing of basic information during the run
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_DENSITY_FITTING_PROGRAM_RUN_INFO_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_DENSITY_FITTING_PROGRAM_RUN_INFO_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_DENSITY_FITTING_PROGRAM_RUN_INFO_CONDITION_NUMBER = Quantity(
        type=str,
        shape=[],
        description='''
        Prints information regarding the condition numbers of the A matrix (to be
        inverted)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_DENSITY_FITTING_PROGRAM_RUN_INFO_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_DENSITY_FITTING_PROGRAM_RUN_INFO_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_DENSITY_FITTING_PROGRAM_RUN_INFO_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_DENSITY_FITTING(MSection):
    '''
    Setup parameters for density fitting (Bloechl charges or density derived  atomic point
    charges (DDAPC) charges)
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_DENSITY_FITTING_GCUT = Quantity(
        type=str,
        shape=[],
        description='''
        Cutoff for charge fit in G-space.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_DENSITY_FITTING_MIN_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the smallest radius of the gaussian used in the fit. All other radius
        are obtained with the progression factor.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_DENSITY_FITTING_NUM_GAUSS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the numbers of gaussian used to fit the QM density for each atomic site.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_DENSITY_FITTING_PFACTOR = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the progression factor for the gaussian exponent for each atomic site.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_DENSITY_FITTING_RADII = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies all the radius of the gaussian used in the fit for each atomic site. The
        use of this keyword disables all other keywords of this section.
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_DENSITY_FITTING_PROGRAM_RUN_INFO = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_DENSITY_FITTING_PROGRAM_RUN_INFO'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT_EFIELD_CONSTANT_ENV(MSection):
    '''
    parameters for a constant envelop
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_EFIELD_CONSTANT_ENV_END_STEP = Quantity(
        type=str,
        shape=[],
        description='''
        Last step the field is applied
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_EFIELD_CONSTANT_ENV_START_STEP = Quantity(
        type=str,
        shape=[],
        description='''
        First step the field is applied
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_EFIELD_GAUSSIAN_ENV(MSection):
    '''
    parameters for a gaussian envelop
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_EFIELD_GAUSSIAN_ENV_SIGMA = Quantity(
        type=str,
        shape=[],
        description='''
        Width of the gaussian
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_EFIELD_GAUSSIAN_ENV_T0 = Quantity(
        type=str,
        shape=[],
        description='''
        Center of the gaussian envelop (maximum of the gaussian)
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_EFIELD_RAMP_ENV(MSection):
    '''
    Parameters for an trapeziodal envelop
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_EFIELD_RAMP_ENV_END_STEP_IN = Quantity(
        type=str,
        shape=[],
        description='''
        Step when the field reaches the full strength
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_EFIELD_RAMP_ENV_END_STEP_OUT = Quantity(
        type=str,
        shape=[],
        description='''
        Step when the field disappears
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_EFIELD_RAMP_ENV_START_STEP_IN = Quantity(
        type=str,
        shape=[],
        description='''
        Step when the electric field starts to be applied
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_EFIELD_RAMP_ENV_START_STEP_OUT = Quantity(
        type=str,
        shape=[],
        description='''
        Step when the field starts to vanish
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_EFIELD(MSection):
    '''
    parameters for finite, time dependent, nonperiodic electric fields
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_EFIELD_ENVELOP = Quantity(
        type=str,
        shape=[],
        description='''
        Shape of the efield pulse
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_EFIELD_INTENSITY = Quantity(
        type=str,
        shape=[],
        description='''
        Intensity of the electric field in W*cm-2 which corresponds to a maximal amplitude
        in a.u. of sqrt(I/(3.50944*10^16))
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_EFIELD_PHASE = Quantity(
        type=str,
        shape=[],
        description='''
        phase offset of the cosine given in multiples of pi
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_EFIELD_POLARISATION = Quantity(
        type=str,
        shape=[],
        description='''
        Polarisation vector of electric field
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_EFIELD_WAVELENGTH = Quantity(
        type=str,
        shape=[],
        description='''
        Wavelength of efield field
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_EFIELD_CONSTANT_ENV = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_EFIELD_CONSTANT_ENV'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_EFIELD_GAUSSIAN_ENV = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_EFIELD_GAUSSIAN_ENV'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_EFIELD_RAMP_ENV = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_EFIELD_RAMP_ENV'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT_EXTERNAL_DENSITY(MSection):
    '''
    Section for the use of the ZMP technique on external densities.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_EXTERNAL_DENSITY_FERMI_AMALDI = Quantity(
        type=str,
        shape=[],
        description='''
        Add the Fermi-Amaldi contribution to the Hartree potential.It leads to a more
        stable convergence.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_EXTERNAL_DENSITY_FILE_DENSITY = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the filename containing the target density in *.cube format.In the MGRID
        section it must be imposed NGRID 1, as it works with onlyone grid. The number of
        points in each direction, and the spacing mustbe previously defined choosing the
        plane waves cut-off in section MGRIDkeyword CUTOFF, and the cube dimention in
        section SUBSYS / CELL / keyword ABC
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_EXTERNAL_DENSITY_LAMBDA = Quantity(
        type=str,
        shape=[],
        description='''
        Lagrange multiplier defined in the constraint ZMP method. When starting, usesmall
        values when starting from scratch (around 5,10). Then gradually increasethe values
        depending, restarting from the previous calculation with the smallervalue. To
        choose the progressive values of LAMBDA look at the convergence of the
        eigenvalues.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_EXTERNAL_DENSITY_ZMP_CONSTRAINT = Quantity(
        type=str,
        shape=[],
        description='''
        Specify which kind of constraint to solve the ZMP equation. The COULOMB
        defaultoption is more stable.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_EXTERNAL_POTENTIAL(MSection):
    '''
    Section controlling the presence of an electrostatic external potential dependent on
    the atomic positions (X,Y,Z)
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_EXTERNAL_POTENTIAL_DX = Quantity(
        type=str,
        shape=[],
        description='''
        Parameter used for computing the derivative with the Ridders method.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_EXTERNAL_POTENTIAL_ERROR_LIMIT = Quantity(
        type=str,
        shape=[],
        description='''
        Checks that the error in computing the derivative is not larger than the value
        set. In case prints a warning message.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_EXTERNAL_POTENTIAL_FUNCTION = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the functional form in mathematical notation. Variables must be the
        atomic coordinates (X,Y,Z) of the grid.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_EXTERNAL_POTENTIAL_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the parameters of the functional form
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_EXTERNAL_POTENTIAL_READ_FROM_CUBE = Quantity(
        type=str,
        shape=[],
        description='''
        Switch for reading the external potential from file pot.cube. The values of the
        potential must be on the grid points of the realspace grid.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_EXTERNAL_POTENTIAL_SCALING_FACTOR = Quantity(
        type=str,
        shape=[],
        description='''
        A factor for scaling the the external potential that is read from file.The value
        of the potential at each grid point is multiplied by this factor.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_EXTERNAL_POTENTIAL_STATIC = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the external potential as STATIC or time dependent. At the moment only
        static potentials are implemented.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_EXTERNAL_POTENTIAL_UNITS = Quantity(
        type=str,
        shape=[],
        description='''
        Optionally, allows to define valid CP2K unit strings for each parameter value. It
        is assumed that the corresponding parameter value is specified in this unit.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_EXTERNAL_POTENTIAL_VALUES = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the values of parameter of the functional form
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_EXTERNAL_VXC(MSection):
    '''
    SCF convergence with external v_xc calculated through previous ZMPcalculation
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_EXTERNAL_VXC_FILE_VXC = Quantity(
        type=str,
        shape=[],
        description='''
        The *.cube filename containing the v_xc potential. This works onlywith NGRID 1
        imposed in the MGRID section. The number of points in eachdirection, and the
        spacing must equal to those previously used in the ZMPcalculation and defined
        through the plane wave cut-off and the cube dimensionrespectively set in section
        MGRID / keyword CUTOFF, and in section SUBSYS /CELL / keyword ABC
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_KG_METHOD(MSection):
    '''
    Specifies the parameters for a Kim-Gordon-like partitioning into molecular subunits
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_KG_METHOD_COLORING_METHOD = Quantity(
        type=str,
        shape=[],
        description='''
        Which algorithm to use for coloring.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_KG_METHOD_TNADD_METHOD = Quantity(
        type=str,
        shape=[],
        description='''
        Algorithm to use for the calculation of the nonadditive kinetic energy.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_KPOINTS(MSection):
    '''
    Sets up the kpoints.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_KPOINTS_EPS_GEO = Quantity(
        type=str,
        shape=[],
        description='''
        Accuracy in symmetry determination.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_KPOINTS_FULL_GRID = Quantity(
        type=str,
        shape=[],
        description='''
        Use full non-reduced kpoint grid.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_KPOINTS_KPOINT = Quantity(
        type=str,
        shape=[],
        description='''
        Specify kpoint coordinates and weight.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_KPOINTS_PARALLEL_GROUP_SIZE = Quantity(
        type=str,
        shape=[],
        description='''
        Number of processors to be used for a single kpoint. Value=-1 (minimum number of
        processes). Value=0 (maximum number of processes). Value=n (exactly n processes).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_KPOINTS_SCHEME = Quantity(
        type=str,
        shape=[],
        description='''
        Kpoint scheme to be used.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_KPOINTS_SYMMETRY = Quantity(
        type=str,
        shape=[],
        description='''
        Use symmetry to reduce the number of kpoints.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_KPOINTS_VERBOSE = Quantity(
        type=str,
        shape=[],
        description='''
        Verbose output information.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_KPOINTS_WAVEFUNCTIONS = Quantity(
        type=str,
        shape=[],
        description='''
        Use real/complex wavefunctions if possible.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_LOCALIZE(MSection):
    '''
    Use one of the available methods to define the localization  and possibly to optimize
    it to a minimum or a maximum.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_LOCALIZE_CRAZY_SCALE = Quantity(
        type=str,
        shape=[],
        description='''
        scale angles
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LOCALIZE_CRAZY_USE_DIAG = Quantity(
        type=str,
        shape=[],
        description='''
        Use diagonalization (slow) or pade based calculation of matrix exponentials.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LOCALIZE_ENERGY_RANGE = Quantity(
        type=str,
        shape=[],
        description='''
        Select the orbitals to be localized within the given energy range.This type of
        selection cannot be added on top of the selection through a LIST. It reads to
        reals that are lower and higher boundaries of the energy range.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LOCALIZE_EPS_LOCALIZATION = Quantity(
        type=str,
        shape=[],
        description='''
        Tolerance used in the convergence criterium of the localization methods.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LOCALIZE_EPS_OCCUPATION = Quantity(
        type=str,
        shape=[],
        description='''
        Tolerance in the occupation number to select only fully occupied orbitals for the
        rotation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LOCALIZE_JACOBI_FALLBACK = Quantity(
        type=str,
        shape=[],
        description='''
        Use Jacobi method in case no convergence was achieved by using the crazy rotations
        method.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LOCALIZE_LIST_UNOCCUPIED = Quantity(
        type=str,
        shape=[],
        description='''
        Indexes of the unoccupied states to be localized, up to now only valid in
        combination with GPW. This keyword has to be present if unoccupied states should
        be localized. This keyword can be repeated several times(useful if you have to
        specify many indexes).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LOCALIZE_LIST = Quantity(
        type=str,
        shape=[],
        description='''
        Indexes of the occupied wfn to be localizedThis keyword can be repeated several
        times(useful if you have to specify many indexes).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LOCALIZE_LOCHOMO_RESTART_FILE_NAME = Quantity(
        type=str,
        shape=[],
        description='''
        File name where to read the MOS fromwhich to restart the localization procedure
        for occupied states
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LOCALIZE_LOCLUMO_RESTART_FILE_NAME = Quantity(
        type=str,
        shape=[],
        description='''
        File name where to read the MOS fromwhich to restart the localization procedure
        for unoccupied states
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LOCALIZE_MAX_CRAZY_ANGLE = Quantity(
        type=str,
        shape=[],
        description='''
        Largest allowed angle for the crazy rotations algorithm (smaller is slower but
        more stable).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LOCALIZE_MAX_ITER = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum number of iterations used for localization methods
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LOCALIZE_METHOD = Quantity(
        type=str,
        shape=[],
        description='''
        Method of optimization if any
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LOCALIZE_MIN_OR_MAX = Quantity(
        type=str,
        shape=[],
        description='''
        Requires the maximization of the spread of the wfn
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LOCALIZE_OPERATOR = Quantity(
        type=str,
        shape=[],
        description='''
        Type of opertator which defines the spread functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LOCALIZE_OUT_ITER_EACH = Quantity(
        type=str,
        shape=[],
        description='''
        Every how many iterations of the localization algorithm(Jacobi) the tolerance
        value is printed out
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LOCALIZE_RESTART = Quantity(
        type=str,
        shape=[],
        description='''
        Restart the localization from a set of orbitals read from a localization restart
        file.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LOCALIZE_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        controls the activation of the MOS localization procedure
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LOCALIZE_STATES = Quantity(
        type=str,
        shape=[],
        description='''
        Which states to localize, LUMO up to now only available in GPW
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LOCALIZE_USE_HISTORY = Quantity(
        type=str,
        shape=[],
        description='''
        Generate an improved initial guess based on a history of results, which is useful
        during MD.Will only work if the number of states to be localized remains constant.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_LOW_SPIN_ROKS(MSection):
    '''
    Specify the details of the low spin ROKS method.In particular, one can specify various
    terms added to the energy of the high spin roks configuration with a energy scaling
    factor, and a prescription of the spin state.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_LOW_SPIN_ROKS_ENERGY_SCALING = Quantity(
        type=str,
        shape=[],
        description='''
        The scaling factors for each term added to the total energy.This list should
        contain one number for each term added to the total energy.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LOW_SPIN_ROKS_SPIN_CONFIGURATION = Quantity(
        type=str,
        shape=[],
        description='''
        for each singly occupied orbital, specify if this should be an alpha (=1) or a
        beta (=2) orbitalThis keyword should be repeated, each repetition corresponding to
        an additional term.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_LS_SCF_CHEBYSHEV_DOS(MSection):
    '''
    Controls the printing of the Density of States (DOS).
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_CHEBYSHEV_DOS_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_CHEBYSHEV_DOS_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_CHEBYSHEV_DOS_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_CHEBYSHEV_DOS_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_CHEBYSHEV_DOS_N_GRIDPOINTS = Quantity(
        type=str,
        shape=[],
        description='''
        Number of points in the computed DOS
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_CHEBYSHEV_DOS_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_LS_SCF_CHEBYSHEV_PRINT_SPECIFIC_E_DENSITY_CUBE(MSection):
    '''
    Controls the printing of cube files with the electronic density (states) contributing
    to the density of states within the specific energy range (MIN_ENERGY ≤ E ≤
    MAX_ENERGY). MIN_ENERGY and MAX_ENERGY need to be specified explicitly.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_CHEBYSHEV_PRINT_SPECIFIC_E_DENSITY_CUBE_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_CHEBYSHEV_PRINT_SPECIFIC_E_DENSITY_CUBE_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_CHEBYSHEV_PRINT_SPECIFIC_E_DENSITY_CUBE_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_CHEBYSHEV_PRINT_SPECIFIC_E_DENSITY_CUBE_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_CHEBYSHEV_PRINT_SPECIFIC_E_DENSITY_CUBE_MAX_ENERGY = Quantity(
        type=str,
        shape=[],
        description='''
        Upper bounds of the energy ranges of interest.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_CHEBYSHEV_PRINT_SPECIFIC_E_DENSITY_CUBE_MIN_ENERGY = Quantity(
        type=str,
        shape=[],
        description='''
        Lower bounds of the energy ranges of interest.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_CHEBYSHEV_PRINT_SPECIFIC_E_DENSITY_CUBE_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_CHEBYSHEV_PRINT_SPECIFIC_E_DENSITY_CUBE_STRIDE = Quantity(
        type=str,
        shape=[],
        description='''
        The stride (X,Y,Z) used to write the cube file (larger values result in smaller
        cube files). You can provide 3 numbers (for X,Y,Z) or 1 number valid for all
        components.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_LS_SCF_CHEBYSHEV(MSection):
    '''
    Specifies the parameters needed for the chebyshev expansion based properties.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_CHEBYSHEV_N_CHEBYSHEV = Quantity(
        type=str,
        shape=[],
        description='''
        Order of the polynomial expansion.
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_LS_SCF_CHEBYSHEV_DOS = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_LS_SCF_CHEBYSHEV_DOS'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_LS_SCF_CHEBYSHEV_PRINT_SPECIFIC_E_DENSITY_CUBE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_LS_SCF_CHEBYSHEV_PRINT_SPECIFIC_E_DENSITY_CUBE'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT_LS_SCF_CURVY_STEPS(MSection):
    '''
    Specifies the parameters of the linear scaling SCF routines
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_CURVY_STEPS_FILTER_FACTOR_SCALE = Quantity(
        type=str,
        shape=[],
        description='''
        Allows for dynamic EPS_FILTER. Updates the filter factor every scf-newton step by
        FILTER_FACTOR=FILTER_FACTOR*FILTER_FACTOR_SCALE
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_CURVY_STEPS_FILTER_FACTOR = Quantity(
        type=str,
        shape=[],
        description='''
        Allows to set a seperate EPS_FILTER in the newton iterations. The new EPS is
        EPS_FILTER*FILTER_FACTOR.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_CURVY_STEPS_LINE_SEARCH = Quantity(
        type=str,
        shape=[],
        description='''
        Line serch type used in the curvy_setp optimization.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_CURVY_STEPS_MIN_FILTER = Quantity(
        type=str,
        shape=[],
        description='''
        Lowest EPS_FILTER in dynamic filtering. Given as multiple of EPS_FILTER:
        EPS_FILTER_MIN=EPS_FILTER*MIN_FILTER
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_CURVY_STEPS_MIN_HESSIAN_SHIFT = Quantity(
        type=str,
        shape=[],
        description='''
        Minimal eigenvalue shift for the Hessian in the Newton iteration. Useful for small
        band gap systems (0.5-1.0 recommended).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_CURVY_STEPS_N_BCH_HISTORY = Quantity(
        type=str,
        shape=[],
        description='''
        Number of stored matrices in the Baker-Campbell-Hausdorff series. Reduces the BCH
        evaluation during line search but can be memory intense.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_LS_SCF(MSection):
    '''
    Specifies the parameters of the linear scaling SCF routines
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_DYNAMIC_THRESHOLD = Quantity(
        type=str,
        shape=[],
        description='''
        Should the threshold for the purification be chosen dynamically
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_EPS_DIIS = Quantity(
        type=str,
        shape=[],
        description='''
        Threshold on the convergence to start using DIIS
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_EPS_FILTER = Quantity(
        type=str,
        shape=[],
        description='''
        Threshold used for filtering matrix operations.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_EPS_LANCZOS = Quantity(
        type=str,
        shape=[],
        description='''
        Threshold used for lanczos estimates.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_EPS_SCF = Quantity(
        type=str,
        shape=[],
        description='''
        target accuracy for the scf convergence. Change of the total energy per electron
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_EXTRAPOLATION_ORDER = Quantity(
        type=str,
        shape=[],
        description='''
        Number of previous matrices used for the ASPC extrapolation of the initial guess.
        0 implies that an atomic guess is used at each step. low (1-2) will result in a
        drift of the constant of motion during MD. high (>5) might be somewhat unstable,
        leading to more SCF iterations.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_FIXED_MU = Quantity(
        type=str,
        shape=[],
        description='''
        Should the calculation be performed at fixed chemical potential, or should it be
        found fixing the number of electrons
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_INI_DIIS = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration cycle to start DIIS Kohn-Sham matrix update
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_LS_DIIS = Quantity(
        type=str,
        shape=[],
        description='''
        Perform DIIS within linear scaling SCF
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_MATRIX_CLUSTER_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Specify how atomic blocks should be clustered in the used matrices, in order to
        improve flop rate, and possibly speedup the matrix multiply. Note that the atomic
        s_preconditioner can not be used.Furthermore, since screening is on matrix blocks,
        slightly more accurate results can be expected with molecular.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_MAX_DIIS = Quantity(
        type=str,
        shape=[],
        description='''
        Size of LS_DIIS buffer
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_MAX_ITER_LANCZOS = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum number of lanczos iterations.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_MAX_SCF = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum number of SCF iteration to be performed for one optimization
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_MIXING_FRACTION = Quantity(
        type=str,
        shape=[],
        description='''
        Mixing density matrices uses the specified fraction in the SCF procedure.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_MU = Quantity(
        type=str,
        shape=[],
        description='''
        Value (or initial guess) for the chemical potential, i.e. some suitable energy
        between HOMO and LUMO energy.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_NMIXING = Quantity(
        type=str,
        shape=[],
        description='''
        Minimal number of density mixing before start DIIS
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_NON_MONOTONIC = Quantity(
        type=str,
        shape=[],
        description='''
        Should the purification be performed non-monotonically. Relevant for TC2 only.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_PERFORM_MU_SCAN = Quantity(
        type=str,
        shape=[],
        description='''
        Do a scan of the chemical potential after the SCF
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_PURIFICATION_METHOD = Quantity(
        type=str,
        shape=[],
        description='''
        Scheme used to purify the Kohn-Sham matrix into the density matrix.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_REPORT_ALL_SPARSITIES = Quantity(
        type=str,
        shape=[],
        description='''
        Run the sparsity report at the end of the SCF
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_RESTART_READ = Quantity(
        type=str,
        shape=[],
        description='''
        Read the density matrix before the (first) SCF.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_RESTART_WRITE = Quantity(
        type=str,
        shape=[],
        description='''
        Write the density matrix at the end of the SCF (currently requires
        EXTRAPOLATION_ORDER>0). Files might be rather large.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_S_INVERSION = Quantity(
        type=str,
        shape=[],
        description='''
        Method used to compute the inverse of S.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_S_PRECONDITIONER = Quantity(
        type=str,
        shape=[],
        description='''
        Preconditions S with some appropriate form.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_SIGN_SQRT_ORDER = Quantity(
        type=str,
        shape=[],
        description='''
        Order of the sqrt iteration, should be 2..5, 3 or 5 recommended
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_LS_SCF_SINGLE_PRECISION_MATRICES = Quantity(
        type=str,
        shape=[],
        description='''
        Matrices used within the LS code can be either double or single precision.
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_LS_SCF_CHEBYSHEV = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_LS_SCF_CHEBYSHEV'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_LS_SCF_CURVY_STEPS = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_LS_SCF_CURVY_STEPS'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT_MGRID_INTERPOLATOR_CONV_INFO(MSection):
    '''
    if convergence information about the linear solver of the spline methods should be
    printed
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_MGRID_INTERPOLATOR_CONV_INFO_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_MGRID_INTERPOLATOR_CONV_INFO_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_MGRID_INTERPOLATOR_CONV_INFO_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_MGRID_INTERPOLATOR_CONV_INFO_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_MGRID_INTERPOLATOR_CONV_INFO_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_MGRID_INTERPOLATOR(MSection):
    '''
    kind of interpolation used between the multigrids
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_MGRID_INTERPOLATOR_AINT_PRECOND = Quantity(
        type=str,
        shape=[],
        description='''
        the approximate inverse to use to get the starting point for the linear solver of
        the spline3 methods
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_MGRID_INTERPOLATOR_EPS_R = Quantity(
        type=str,
        shape=[],
        description='''
        accuracy on the residual for spline3 the interpolators
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_MGRID_INTERPOLATOR_EPS_X = Quantity(
        type=str,
        shape=[],
        description='''
        accuracy on the solution for spline3 the interpolators
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_MGRID_INTERPOLATOR_KIND = Quantity(
        type=str,
        shape=[],
        description='''
        the interpolator to use
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_MGRID_INTERPOLATOR_MAX_ITER = Quantity(
        type=str,
        shape=[],
        description='''
        the maximum number of iterations
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_MGRID_INTERPOLATOR_PRECOND = Quantity(
        type=str,
        shape=[],
        description='''
        The preconditioner used for the linear solver of the spline3 methods
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_MGRID_INTERPOLATOR_SAFE_COMPUTATION = Quantity(
        type=str,
        shape=[],
        description='''
        if a non unrolled calculation is to be performed in parallel
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_MGRID_INTERPOLATOR_CONV_INFO = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_MGRID_INTERPOLATOR_CONV_INFO'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT_MGRID_RS_GRID(MSection):
    '''
    Set options that influence how the realspace grids are being distributed in parallel
    runs.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_MGRID_RS_GRID_DISTRIBUTION_LAYOUT = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the number of slices in the x, y and z directions.-1 specifies that any
        number of slices is OK.If a given distribution can not be satisfied, a replicated
        grid will result.Also see LOCK_DISTRIBUTION.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_MGRID_RS_GRID_DISTRIBUTION_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Parallelization strategy.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_MGRID_RS_GRID_HALO_REDUCTION_FACTOR = Quantity(
        type=str,
        shape=[],
        description='''
        Can be used to reduce the halo of the distributed grid (experimental features).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_MGRID_RS_GRID_LOCK_DISTRIBUTION = Quantity(
        type=str,
        shape=[],
        description='''
        Expert use only, only basic QS deals correctly with a non-default value.If the
        distribution is locked, a grid will have the same distribution asthe next finer
        multigrid (provided it is distributed).If unlocked, all grids can be distributed
        freely.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_MGRID_RS_GRID_MAX_DISTRIBUTED_LEVEL = Quantity(
        type=str,
        shape=[],
        description='''
        If the multigrid-level of a grid is larger than the parameter, it will not be
        distributed in the automatic scheme.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_MGRID_RS_GRID_MEMORY_FACTOR = Quantity(
        type=str,
        shape=[],
        description='''
        A grid will only be distributed if the memory usage for that grid (including halo)
        is smaller than a replicated grid by this parameter.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_MGRID(MSection):
    '''
    multigrid information
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_MGRID_COMMENSURATE = Quantity(
        type=str,
        shape=[],
        description='''
        If the grids should be commensurate. If true overrides the progression factor and
        the cutoffs of the sub grids
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_MGRID_CUTOFF = Quantity(
        type=str,
        shape=[],
        description='''
        The cutoff of the finest grid level. Default value for SE or DFTB calculation is
        1.0 [Ry].
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_MGRID_MULTIGRID_CUTOFF = Quantity(
        type=str,
        shape=[],
        description='''
        List of cutoff values to set up multigrids manually
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_MGRID_MULTIGRID_SET = Quantity(
        type=str,
        shape=[],
        description='''
        Activate a manual setting of the multigrids
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_MGRID_NGRIDS = Quantity(
        type=str,
        shape=[],
        description='''
        The number of multigrids to use
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_MGRID_PROGRESSION_FACTOR = Quantity(
        type=str,
        shape=[],
        description='''
        Factor used to find the cutoff of the multigrids that where not given explicitly
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_MGRID_REALSPACE = Quantity(
        type=str,
        shape=[],
        description='''
        If both rho and rho_gspace are needed
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_MGRID_REL_CUTOFF = Quantity(
        type=str,
        shape=[],
        description='''
        Determines the grid at which a Gaussian is mapped, giving the cutoff used for a
        gaussian with alpha=1. A value 50+-10Ry might be required for highly accurate
        results,  Or for simulations with a variable cell. Versions prior to 2.3 used a
        default of 30Ry.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_MGRID_SKIP_LOAD_BALANCE_DISTRIBUTED = Quantity(
        type=str,
        shape=[],
        description='''
        Skip load balancing on distributed multigrids, which might be memory intensive.If
        not explicitly specified, runs using more than 1024 MPI tasks will default to
        .TRUE.
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_MGRID_INTERPOLATOR = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_MGRID_INTERPOLATOR'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_MGRID_RS_GRID = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_MGRID_RS_GRID'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT_PERIODIC_EFIELD(MSection):
    '''
    parameters for finite periodic electric field computed using the Berry phase approach.
    IMPORTANT: Can only be used in combination  with OT. Can not be used in combination
    with RTP or EMD.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_PERIODIC_EFIELD_INTENSITY = Quantity(
        type=str,
        shape=[],
        description='''
        Intensity of the electric field in a.u
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_PERIODIC_EFIELD_POLARISATION = Quantity(
        type=str,
        shape=[],
        description='''
        Polarisation vector of electric field
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_POISSON_EWALD_MULTIPOLES(MSection):
    '''
    Enables the use of multipoles in the treatment of the electrostatics.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_EWALD_MULTIPOLES_EPS_POL = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the rmsd threshold for the derivatives of the energy towards the Cartesian
        dipoles components
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_EWALD_MULTIPOLES_MAX_IPOL_ITER = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the maximum number of iterations for induced dipoles
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_EWALD_MULTIPOLES_MAX_MULTIPOLE_EXPANSION = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the maximum level of multipoles expansion used  for the electrostatics.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_EWALD_MULTIPOLES_POL_SCF = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the method to obtain self consistent induced multipole moments.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_EWALD_MULTIPOLES_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Controls the activation of the Multipoles
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_POISSON_EWALD_RS_GRID(MSection):
    '''
    Set options that influence how the realspace grids are being distributed in parallel
    runs.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_EWALD_RS_GRID_DISTRIBUTION_LAYOUT = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the number of slices in the x, y and z directions.-1 specifies that any
        number of slices is OK.If a given distribution can not be satisfied, a replicated
        grid will result.Also see LOCK_DISTRIBUTION.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_EWALD_RS_GRID_DISTRIBUTION_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Parallelization strategy.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_EWALD_RS_GRID_HALO_REDUCTION_FACTOR = Quantity(
        type=str,
        shape=[],
        description='''
        Can be used to reduce the halo of the distributed grid (experimental features).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_EWALD_RS_GRID_LOCK_DISTRIBUTION = Quantity(
        type=str,
        shape=[],
        description='''
        Expert use only, only basic QS deals correctly with a non-default value.If the
        distribution is locked, a grid will have the same distribution asthe next finer
        multigrid (provided it is distributed).If unlocked, all grids can be distributed
        freely.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_EWALD_RS_GRID_MAX_DISTRIBUTED_LEVEL = Quantity(
        type=str,
        shape=[],
        description='''
        If the multigrid-level of a grid is larger than the parameter, it will not be
        distributed in the automatic scheme.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_EWALD_RS_GRID_MEMORY_FACTOR = Quantity(
        type=str,
        shape=[],
        description='''
        A grid will only be distributed if the memory usage for that grid (including halo)
        is smaller than a replicated grid by this parameter.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_POISSON_EWALD(MSection):
    '''
    Ewald parameters controlling electrostatic only for CLASSICAL MM.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_EWALD_ALPHA = Quantity(
        type=str,
        shape=[],
        description='''
        alpha parameter associated with Ewald (EWALD|PME|SPME). Recommended for small
        systems is is alpha = 3.5 / r_cut. Tuning alpha, r_cut and gmax is needed to
        obtain O(N**1.5) scaling for ewald.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_EWALD_EPSILON = Quantity(
        type=str,
        shape=[],
        description='''
        tolerance of gaussians for fft interpolation (PME only)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_EWALD_EWALD_ACCURACY = Quantity(
        type=str,
        shape=[],
        description='''
        Expected accuracy in the Ewald sum. This number affects only the calculation of
        the cutoff for the real-space term of the ewald summation (EWALD|PME|SPME) as well
        as the construction of the neighbor lists (if the cutoff for non-bonded terms is
        smaller than the value employed to compute the EWALD real-space term). This
        keyword has no effect on the reciprocal space term (which can be tuned
        independently).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_EWALD_EWALD_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        The type of ewald you want to perform.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_EWALD_GMAX = Quantity(
        type=str,
        shape=[],
        description='''
        number of grid points (SPME and EWALD). If a single number is specified,the same
        number of points is used for all three directions on the grid.If three numbers are
        given, each direction can have a different number of points.The number of points
        needs to be FFTable (which depends on the library used) and odd for EWALD.The
        optimal number depends e.g. on alpha and the size of the cell. 1 point per
        Angstrom is common.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_EWALD_NS_MAX = Quantity(
        type=str,
        shape=[],
        description='''
        number of grid points on small mesh (PME only), should be odd.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_EWALD_O_SPLINE = Quantity(
        type=str,
        shape=[],
        description='''
        order of the beta-Euler spline (SPME only)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_EWALD_RCUT = Quantity(
        type=str,
        shape=[],
        description='''
        Explicitly provide the real-space cutoff of the ewald summation (EWALD|PME|SPME).
        If present, overwrites the estimate of EWALD_ACCURACY and may affect the
        construction of the neighbor lists for non-bonded terms (in FIST), if the value
        specified is larger than the cutoff for non-bonded interactions.
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_POISSON_EWALD_MULTIPOLES = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_POISSON_EWALD_MULTIPOLES'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_POISSON_EWALD_RS_GRID = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_POISSON_EWALD_RS_GRID'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT_POISSON_MT(MSection):
    '''
    Sets up parameters of  Martyna-Tuckerman poisson solver. Note that exact results are
    only guaranteed if the unit cell is twice as large as charge density (and serious
    artefacts can result if the cell is much smaller).
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_MT_ALPHA = Quantity(
        type=str,
        shape=[],
        description='''
        Convergence parameter ALPHA*RMIN. Default value 7.0
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_MT_REL_CUTOFF = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the multiplicative factor for the CUTOFF keyword in MULTI_GRID  section.
        The result gives the cutoff at which the 1/r non-periodic FFT3D is
        evaluated.Default is 2.0
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_POISSON_MULTIPOLE_CHECK_SPLINE(MSection):
    '''
    Controls the checking of the G-space term Spline Interpolation.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_MULTIPOLE_CHECK_SPLINE_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_MULTIPOLE_CHECK_SPLINE_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_MULTIPOLE_CHECK_SPLINE_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_MULTIPOLE_CHECK_SPLINE_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_MULTIPOLE_CHECK_SPLINE_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_POISSON_MULTIPOLE_INTERPOLATOR_CONV_INFO(MSection):
    '''
    if convergence information about the linear solver of the spline methods should be
    printed
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_MULTIPOLE_INTERPOLATOR_CONV_INFO_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_MULTIPOLE_INTERPOLATOR_CONV_INFO_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_MULTIPOLE_INTERPOLATOR_CONV_INFO_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_MULTIPOLE_INTERPOLATOR_CONV_INFO_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_MULTIPOLE_INTERPOLATOR_CONV_INFO_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_POISSON_MULTIPOLE_INTERPOLATOR(MSection):
    '''
    controls the interpolation for the G-space term
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_MULTIPOLE_INTERPOLATOR_AINT_PRECOND = Quantity(
        type=str,
        shape=[],
        description='''
        the approximate inverse to use to get the starting point for the linear solver of
        the spline3 methods
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_MULTIPOLE_INTERPOLATOR_EPS_R = Quantity(
        type=str,
        shape=[],
        description='''
        accuracy on the residual for spline3 the interpolators
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_MULTIPOLE_INTERPOLATOR_EPS_X = Quantity(
        type=str,
        shape=[],
        description='''
        accuracy on the solution for spline3 the interpolators
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_MULTIPOLE_INTERPOLATOR_MAX_ITER = Quantity(
        type=str,
        shape=[],
        description='''
        the maximum number of iterations
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_MULTIPOLE_INTERPOLATOR_PRECOND = Quantity(
        type=str,
        shape=[],
        description='''
        The preconditioner used for the linear solver of the spline3 methods
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_POISSON_MULTIPOLE_INTERPOLATOR_CONV_INFO = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_POISSON_MULTIPOLE_INTERPOLATOR_CONV_INFO'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT_POISSON_MULTIPOLE_PROGRAM_RUN_INFO(MSection):
    '''
    Controls the printing of basic information during the run
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_MULTIPOLE_PROGRAM_RUN_INFO_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_MULTIPOLE_PROGRAM_RUN_INFO_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_MULTIPOLE_PROGRAM_RUN_INFO_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_MULTIPOLE_PROGRAM_RUN_INFO_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_MULTIPOLE_PROGRAM_RUN_INFO_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_POISSON_MULTIPOLE(MSection):
    '''
    This section is used to set up the decoupling of QM periodic images with the use of
    density derived atomic point charges.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_MULTIPOLE_ANALYTICAL_GTERM = Quantity(
        type=str,
        shape=[],
        description='''
        Evaluates the Gterm in the Ewald Scheme analytically instead of using Splines.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_MULTIPOLE_EWALD_PRECISION = Quantity(
        type=str,
        shape=[],
        description='''
        Precision achieved in the Ewald sum.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_MULTIPOLE_NGRIDS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the number of grid points used for the Interpolation of the G-space term
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_MULTIPOLE_RCUT = Quantity(
        type=str,
        shape=[],
        description='''
        Real space cutoff for the Ewald sum.
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_POISSON_MULTIPOLE_CHECK_SPLINE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_POISSON_MULTIPOLE_CHECK_SPLINE'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_POISSON_MULTIPOLE_INTERPOLATOR = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_POISSON_MULTIPOLE_INTERPOLATOR'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_POISSON_MULTIPOLE_PROGRAM_RUN_INFO = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_POISSON_MULTIPOLE_PROGRAM_RUN_INFO'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT_POISSON_WAVELET(MSection):
    '''
    Sets up parameters of  wavelet based poisson solver.This solver allows for non-
    periodic (PERIODIC NONE) boundary conditions and slab-boundary conditions (but only
    PERIODIC XZ).It does not require very large unit cells, only that the density goes to
    zero on the faces of the cell.The use of PREFERRED_FFT_LIBRARY FFTSG is required
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_WAVELET_SCF_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Type of scaling function used in the wavelet approach, the total energy depends on
        this choice,and the convergence with respect to cutoff depends on the selected
        scaling functions.Possible values are 8,14,16,20,24,30,40,50,60,100
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_POISSON(MSection):
    '''
    Sets up the poisson resolutor.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_PERIODIC = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the directions on wich apply PBC. Important notice,  this only applies to
        the electrostatics. See the CELL section to specify the periodicity used for e.g.
        the pair lists. Typically the settings should be the same.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_POISSON_POISSON_SOLVER = Quantity(
        type=str,
        shape=[],
        description='''
        Specify which kind of solver to use to solve the Poisson equation.
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_POISSON_EWALD = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_POISSON_EWALD'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_POISSON_MT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_POISSON_MT'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_POISSON_MULTIPOLE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_POISSON_MULTIPOLE'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_POISSON_WAVELET = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_POISSON_WAVELET'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT_QS_BECKE_RESTRAINT_PROGRAM_RUN_INFO(MSection):
    '''
    Controls the printing basic info about the method
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_QS_BECKE_RESTRAINT_PROGRAM_RUN_INFO_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_BECKE_RESTRAINT_PROGRAM_RUN_INFO_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_BECKE_RESTRAINT_PROGRAM_RUN_INFO_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_BECKE_RESTRAINT_PROGRAM_RUN_INFO_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_BECKE_RESTRAINT_PROGRAM_RUN_INFO_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_QS_BECKE_RESTRAINT(MSection):
    '''
    Use Becke weight population in a restraint/constraint
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_QS_BECKE_RESTRAINT_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the list of atoms that is summed in the restraint
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_BECKE_RESTRAINT_COEFF = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the the coefficient of the atom in the atom list (default is one)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_BECKE_RESTRAINT_FUNCTIONAL_FORM = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the functional form of the term added
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_BECKE_RESTRAINT_STRENGTH = Quantity(
        type=str,
        shape=[],
        description='''
        force constant of the restraint
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_BECKE_RESTRAINT_TARGET = Quantity(
        type=str,
        shape=[],
        description='''
        target value of the restraint
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_BECKE_RESTRAINT_TYPE_OF_DENSITY = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the type of density used for the fitting
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_QS_BECKE_RESTRAINT_PROGRAM_RUN_INFO = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_QS_BECKE_RESTRAINT_PROGRAM_RUN_INFO'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT_QS_DDAPC_RESTRAINT_PROGRAM_RUN_INFO(MSection):
    '''
    Controls the printing basic info about the method
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_QS_DDAPC_RESTRAINT_PROGRAM_RUN_INFO_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_DDAPC_RESTRAINT_PROGRAM_RUN_INFO_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_DDAPC_RESTRAINT_PROGRAM_RUN_INFO_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_DDAPC_RESTRAINT_PROGRAM_RUN_INFO_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_DDAPC_RESTRAINT_PROGRAM_RUN_INFO_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_QS_DDAPC_RESTRAINT(MSection):
    '''
    Use DDAPC charges in a restraint (check code for details)
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_QS_DDAPC_RESTRAINT_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the list of atoms that is summed in the restraint
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_DDAPC_RESTRAINT_COEFF = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the the coefficient of the atom in the atom list (default is one)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_DDAPC_RESTRAINT_FUNCTIONAL_FORM = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the functional form of the term added
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_DDAPC_RESTRAINT_STRENGTH = Quantity(
        type=str,
        shape=[],
        description='''
        force constant of the restraint
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_DDAPC_RESTRAINT_TARGET = Quantity(
        type=str,
        shape=[],
        description='''
        target value of the restraint
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_DDAPC_RESTRAINT_TYPE_OF_DENSITY = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the type of density used for the fitting
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_QS_DDAPC_RESTRAINT_PROGRAM_RUN_INFO = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_QS_DDAPC_RESTRAINT_PROGRAM_RUN_INFO'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT_QS_DFTB_PARAMETER(MSection):
    '''
    Information on where to find DFTB parameters
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_QS_DFTB_PARAMETER_COORDINATION_CUTOFF = Quantity(
        type=str,
        shape=[],
        description='''
        Define cutoff for coordination number calculation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_DFTB_PARAMETER_D3_SCALING = Quantity(
        type=str,
        shape=[],
        description='''
        Scaling parameters (s6,sr6,s8) for the D3 dispersion method,
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_DFTB_PARAMETER_DISPERSION_PARAMETER_FILE = Quantity(
        type=str,
        shape=[],
        description='''
        Specify file that contains the atomic dispersion parameters for the D3 method
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_DFTB_PARAMETER_DISPERSION_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Define radius of dispersion interaction
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_DFTB_PARAMETER_DISPERSION_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Use dispersion correction of the specified type. Dispersion correction has to be
        switched on in the DFTB section.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_DFTB_PARAMETER_HB_SR_PARAM = Quantity(
        type=str,
        shape=[],
        description='''
        Uses a modified version for the GAMMA within the SCC-DFTB scheme, specifically
        tuned for hydrogen bonds. Specify the exponent used in the exponential.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_DFTB_PARAMETER_PARAM_FILE_NAME = Quantity(
        type=str,
        shape=[],
        description='''
        Specify file that contains the names of Slater-Koster tables: A plain text file,
        each line has the format "ATOM1 ATOM2 filename.spl".
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_DFTB_PARAMETER_PARAM_FILE_PATH = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the directory with the DFTB parameter files. Used in combination with the
        filenames specified in the file given in PARAM_FILE_NAME.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_DFTB_PARAMETER_SK_FILE = Quantity(
        type=str,
        shape=[],
        description='''
        Define parameter file for atom pair
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_DFTB_PARAMETER_UFF_FORCE_FIELD = Quantity(
        type=str,
        shape=[],
        description='''
        Name of file with UFF parameters that will be used for the dispersion correction.
        Needs to be specified when DISPERSION==.TRUE., otherwise cp2k crashes with a
        Segmentation Fault.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_QS_DFTB(MSection):
    '''
    Parameters needed to set up the DFTB methods
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_QS_DFTB_DIAGONAL_DFTB3 = Quantity(
        type=str,
        shape=[],
        description='''
        Use a diagonal version of the 3rd order energy correction (DFTB3)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_DFTB_DISPERSION = Quantity(
        type=str,
        shape=[],
        description='''
        Use dispersion correction
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_DFTB_DO_EWALD = Quantity(
        type=str,
        shape=[],
        description='''
        Use Ewald type method instead of direct sum for Coulomb interaction
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_DFTB_EPS_DISP = Quantity(
        type=str,
        shape=[],
        description='''
        Define accuracy of dispersion interaction
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_DFTB_HB_SR_GAMMA = Quantity(
        type=str,
        shape=[],
        description='''
        Uses a modified version for the GAMMA within the SCC-DFTB scheme, specifically
        tuned for hydrogen bonds.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_DFTB_ORTHOGONAL_BASIS = Quantity(
        type=str,
        shape=[],
        description='''
        Assume orthogonal basis set
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_DFTB_SELF_CONSISTENT = Quantity(
        type=str,
        shape=[],
        description='''
        Use self-consistent method
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_QS_DFTB_PARAMETER = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_QS_DFTB_PARAMETER'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT_QS_DISTRIBUTION(MSection):
    '''
    can be used used to tune the parallel distribution of the data
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_QS_DISTRIBUTION_2D_MOLECULAR_DISTRIBUTION = Quantity(
        type=str,
        shape=[],
        description='''
        Distribute the atoms so that atoms belonging to a given molecule are on the same
        CPU for the 2D distribution. This might give rise to a worse distribution but
        reduces memory needs of finding the optimal distribution.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_DISTRIBUTION_BASIC_OPTIMIZATION = Quantity(
        type=str,
        shape=[],
        description='''
        Creates a distribution based on a few heuristics using only minimal memory and CPU
        time.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_DISTRIBUTION_BASIC_SPATIAL_OPTIMIZATION = Quantity(
        type=str,
        shape=[],
        description='''
        Creates a distribution with spatial info, using only minimal memory and CPU time.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_DISTRIBUTION_COST_MODEL = Quantity(
        type=str,
        shape=[],
        description='''
        The cost model that needs to be minimized
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_DISTRIBUTION_SKIP_OPTIMIZATION = Quantity(
        type=str,
        shape=[],
        description='''
        Do not optimize the distribution, go for something very simple. Might be useful if
        the optimization, which scales quadratically in system size, is too expensive.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_DISTRIBUTION_SYMMETRIC = Quantity(
        type=str,
        shape=[],
        description='''
        Take the symmetry of the distribution_2d into account.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_QS_HARRIS(MSection):
    '''
    This section specifies the flags for the calculation of the harris functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_QS_HARRIS_ACTIVATE = Quantity(
        type=str,
        shape=[],
        description='''
        Activates the harris functional.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_HARRIS_HARRIS_ENERGY = Quantity(
        type=str,
        shape=[],
        description='''
        Overrides the QS Energy.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_QS_MULLIKEN_RESTRAINT(MSection):
    '''
    Use mulliken charges in a restraint (check code for details)
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_QS_MULLIKEN_RESTRAINT_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the list of atoms that is summed in the restraint
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_MULLIKEN_RESTRAINT_STRENGTH = Quantity(
        type=str,
        shape=[],
        description='''
        force constant of the restraint
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_MULLIKEN_RESTRAINT_TARGET = Quantity(
        type=str,
        shape=[],
        description='''
        target value of the restraint
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_QS_OPTIMIZE_LRI_BASIS_CONSTRAIN_EXPONENTS(MSection):
    '''
    specifies constraints for the exponents of the lri auxiliary basis sets in the
    optimization.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_QS_OPTIMIZE_LRI_BASIS_CONSTRAIN_EXPONENTS_FERMI_EXP = Quantity(
        type=str,
        shape=[],
        description='''
        Exponent in the fermi-like constraint function.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_OPTIMIZE_LRI_BASIS_CONSTRAIN_EXPONENTS_SCALE = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the upper and lower boundaries as (1+scale)*exp and (1-scale)*exp. Fermi-
        like constraint function
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_QS_OPTIMIZE_LRI_BASIS(MSection):
    '''
    This section specifies the parameters for optimizing the lri auxiliary basis sets for
    LRIGPW. The Powell optimizer is used.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_QS_OPTIMIZE_LRI_BASIS_ACCURACY = Quantity(
        type=str,
        shape=[],
        description='''
        Target accuracy for the objective function (RHOEND)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_OPTIMIZE_LRI_BASIS_CONDITION_WEIGHT = Quantity(
        type=str,
        shape=[],
        description='''
        This keyword allows to give different weight factors to the condition number
        (LOG(cond) is used).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_OPTIMIZE_LRI_BASIS_DEGREES_OF_FREEDOM = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the degrees of freedom in the basis optimization.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_OPTIMIZE_LRI_BASIS_GEOMETRIC_SEQUENCE = Quantity(
        type=str,
        shape=[],
        description='''
        Exponents are assumed to be a geometric squence. Only the minimal and maximal
        exponents of one set are optimized and the other exponents are obtained by
        geometric progression.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_OPTIMIZE_LRI_BASIS_MAX_FUN = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum number of function evaluations
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_OPTIMIZE_LRI_BASIS_STEP_SIZE = Quantity(
        type=str,
        shape=[],
        description='''
        Initial step size for search algorithm (RHOBEG)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_OPTIMIZE_LRI_BASIS_USE_CONDITION_NUMBER = Quantity(
        type=str,
        shape=[],
        description='''
        Determines whether condition number should be part of optimization or not
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_QS_OPTIMIZE_LRI_BASIS_CONSTRAIN_EXPONENTS = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_QS_OPTIMIZE_LRI_BASIS_CONSTRAIN_EXPONENTS'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT_QS_S2_RESTRAINT(MSection):
    '''
    Use S2 in a re/constraint (OT only)
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_QS_S2_RESTRAINT_FUNCTIONAL_FORM = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the functional form of the term added
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_S2_RESTRAINT_STRENGTH = Quantity(
        type=str,
        shape=[],
        description='''
        force constant of the restraint
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_S2_RESTRAINT_TARGET = Quantity(
        type=str,
        shape=[],
        description='''
        target value of the restraint
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_QS_SCPTB(MSection):
    '''
    Parameters needed to set up the SCPTB methods
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_QS_SCPTB_COORDINATION_CUTOFF = Quantity(
        type=str,
        shape=[],
        description='''
        Define cutoff for coordination number calculation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_SCPTB_D3_SCALING = Quantity(
        type=str,
        shape=[],
        description='''
        Scaling parameters (s6,sr6,s8) for the D3 dispersion method,
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_SCPTB_DISPERSION_PARAMETER_FILE = Quantity(
        type=str,
        shape=[],
        description='''
        Specify file that contains the atomic dispersion parameters
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_SCPTB_DISPERSION_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Define radius of dispersion interaction
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_SCPTB_DISPERSION = Quantity(
        type=str,
        shape=[],
        description='''
        Use dispersion correction
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_SCPTB_DO_EWALD = Quantity(
        type=str,
        shape=[],
        description='''
        Use Ewald type method instead of direct sum for Coulomb interaction
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_SCPTB_DO_SCC = Quantity(
        type=str,
        shape=[],
        description='''
        Use self consistent charge method. Can be used together with DO_SCP to get TB
        method
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_SCPTB_DO_SCP = Quantity(
        type=str,
        shape=[],
        description='''
        Use SCP method. Can be used to switch off SCP to get a SCC-DFTB method
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_SCPTB_PAIR_CUTOFF = Quantity(
        type=str,
        shape=[],
        description='''
        Define cutoff for pair potential calculation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_SCPTB_PARAMETER_FILE_NAME = Quantity(
        type=str,
        shape=[],
        description='''
        Specify file that contains the atomic parameters
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_SCPTB_STO_NG = Quantity(
        type=str,
        shape=[],
        description='''
        Provides the order of the Slater orbital expansion of Gaussian-Type Orbitals.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_QS_SE_COULOMB(MSection):
    '''
    Setup parameters for the evaluation of the COULOMB term in SE calculations.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_QS_SE_COULOMB_CUTOFF = Quantity(
        type=str,
        shape=[],
        description='''
        Atomic Cutoff Radius Cutoff for the evaluation of the  Coulomb integrals. For non-
        periodic calculation the default value is exactly the full cell dimension, in
        order to evaluate all pair interactions. Instead, for periodic calculations the
        default numerical value is used.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_SE_COULOMB_RC_RANGE = Quantity(
        type=str,
        shape=[],
        description='''
        Range of cutoff switch function (tapering): 0.5*(1-TANH((r-r0)/RC_RANGE)), where
        r0=2.0*RC_TAPER-20.0*RC_RANGE.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_SE_COULOMB_RC_TAPER = Quantity(
        type=str,
        shape=[],
        description='''
        Atomic Cutoff Radius Cutoff for Tapering Coulomb integrals. If not specified it
        assumes the same value specified for the CUTOFF.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_QS_SE_EXCHANGE(MSection):
    '''
    Setup parameters for the evaluation of the EXCHANGE and  core Hamiltonian terms in SE
    calculations.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_QS_SE_EXCHANGE_CUTOFF = Quantity(
        type=str,
        shape=[],
        description='''
        Atomic Cutoff Radius Cutoff for the evaluation of the Exchange integrals. For non-
        periodic calculation the default value is exactly the full cell dimension, in
        order to evaluate all pair interactions. Instead, for periodic calculations the
        default is the minimum value between 1/4 of the cell dimension and the value
        specified in input (either explicitly defined or the default numerical value).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_SE_EXCHANGE_RC_RANGE = Quantity(
        type=str,
        shape=[],
        description='''
        Range of cutoff switch function (tapering): 0.5*(1-TANH((r-r0)/RC_RANGE)), where
        r0=2.0*RC_TAPER-20.0*RC_RANGE.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_SE_EXCHANGE_RC_TAPER = Quantity(
        type=str,
        shape=[],
        description='''
        Atomic Cutoff Radius Cutoff for Tapering Exchange integrals. If not specified it
        assumes the same value specified for the CUTOFF.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_QS_SE_GA(MSection):
    '''
    Sets up memory parameters for the storage of the integrals
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_QS_SE_GA_NCELLS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the number of linked cells for the neighbor list. Default value is number
        of processors
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_QS_SE_LR_CORRECTION(MSection):
    '''
    Setup parameters for the evaluation of the long-range correction term in SE
    calculations.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_QS_SE_LR_CORRECTION_CUTOFF = Quantity(
        type=str,
        shape=[],
        description='''
        Atomic Cutoff Radius Cutoff for the evaluation of the long-ranbe correction
        integrals.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_SE_LR_CORRECTION_RC_RANGE = Quantity(
        type=str,
        shape=[],
        description='''
        Range of cutoff switch function (tapering): 0.5*(1-TANH((r-r0)/RC_RANGE)), where
        r0=2.0*RC_TAPER-20.0*RC_RANGE.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_SE_LR_CORRECTION_RC_TAPER = Quantity(
        type=str,
        shape=[],
        description='''
        Atomic Cutoff Radius Cutoff for Tapering the long-range correction integrals. If
        not specified it assumes the same value specified for the CUTOFF.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_QS_SE_MEMORY(MSection):
    '''
    Sets up memory parameters for the storage of the integrals
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_QS_SE_MEMORY_COMPRESS = Quantity(
        type=str,
        shape=[],
        description='''
        Enables the compression of the integrals in memory.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_SE_MEMORY_EPS_STORAGE = Quantity(
        type=str,
        shape=[],
        description='''
        Storage threshold for compression is EPS_STORAGE
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_SE_MEMORY_MAX_MEMORY = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the maximum amount of memory [MB] used to store precomputed (possibly
        compressed) two-electron two-center integrals
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_QS_SE_NEIGHBOR_LISTS(MSection):
    '''
    This section specifies the input parameters for the construction of neighbor lists.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_QS_SE_NEIGHBOR_LISTS_GEO_CHECK = Quantity(
        type=str,
        shape=[],
        description='''
        This keyword enables the check that two atoms are never below the minimum value
        used to construct the splines during the construction of the neighbouring list.
        Disabling this keyword avoids CP2K to abort in case two atoms are below the
        minimum  value of the radius used to generate the splines.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_SE_NEIGHBOR_LISTS_NEIGHBOR_LISTS_FROM_SCRATCH = Quantity(
        type=str,
        shape=[],
        description='''
        This keyword enables the building of the neighbouring list from scratch.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_SE_NEIGHBOR_LISTS_VERLET_SKIN = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the Verlet Skin for the generation of the neighbor lists
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_QS_SE_SCREENING(MSection):
    '''
    Setup parameters for the tapering of the Coulomb/Exchange Screening in KDSO-D integral
    scheme,
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_QS_SE_SCREENING_RC_RANGE = Quantity(
        type=str,
        shape=[],
        description='''
        Range of cutoff switch function (tapering): 0.5*(1-TANH((r-r0)/RC_RANGE)), where
        r0=2*RC_TAPER-20*RC_RANGE.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_SE_SCREENING_RC_TAPER = Quantity(
        type=str,
        shape=[],
        description='''
        Atomic Cutoff Radius Cutoff for Tapering the screening term.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_QS_SE(MSection):
    '''
    Parameters needed to set up the Semi-empirical methods
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_QS_SE_ANALYTICAL_GRADIENTS = Quantity(
        type=str,
        shape=[],
        description='''
        Nuclear Gradients are computed analytically or numerically
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_SE_COORDINATION_CUTOFF = Quantity(
        type=str,
        shape=[],
        description='''
        Define cutoff for coordination number calculation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_SE_D3_SCALING = Quantity(
        type=str,
        shape=[],
        description='''
        Scaling parameters (s6,sr6,s8) for the D3 dispersion method,
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_SE_DELTA = Quantity(
        type=str,
        shape=[],
        description='''
        Step size in finite difference force calculation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_SE_DISPERSION_PARAMETER_FILE = Quantity(
        type=str,
        shape=[],
        description='''
        Specify file that contains the atomic dispersion parameters
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_SE_DISPERSION_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Define radius of dispersion interaction
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_SE_DISPERSION = Quantity(
        type=str,
        shape=[],
        description='''
        Use dispersion correction
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_SE_FORCE_KDSO_D_EXCHANGE = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords forces the usage of the KDSO-D integral screening for the Exchange
        integrals (default is to apply the screening only to the Coulomb integrals.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_SE_INTEGRAL_SCREENING = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the functional form for the
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_SE_ORTHOGONAL_BASIS = Quantity(
        type=str,
        shape=[],
        description='''
        Assume orthogonal basis set. This flag is overwritten by methods with fixed
        orthogonal/non-orthogonal basis set.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_SE_PERIODIC = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the type of treatment for the electrostatic long-range part in semi-
        empirical calculations.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_SE_SCP = Quantity(
        type=str,
        shape=[],
        description='''
        Perform a SCP-NDDO calculation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_SE_STO_NG = Quantity(
        type=str,
        shape=[],
        description='''
        Provides the order of the Slater orbital expansion of Gaussian-Type Orbitals.
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_QS_SE_COULOMB = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_QS_SE_COULOMB'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_QS_SE_EXCHANGE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_QS_SE_EXCHANGE'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_QS_SE_GA = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_QS_SE_GA'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_QS_SE_LR_CORRECTION = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_QS_SE_LR_CORRECTION'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_QS_SE_MEMORY = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_QS_SE_MEMORY'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_QS_SE_NEIGHBOR_LISTS = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_QS_SE_NEIGHBOR_LISTS'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_QS_SE_SCREENING = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_QS_SE_SCREENING'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT_QS(MSection):
    '''
    parameters needed to set up the Quickstep framework
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_QS_ALMO_SCF = Quantity(
        type=str,
        shape=[],
        description='''
        Perform ALMO SCF
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_ALPHA0_HARD = Quantity(
        type=str,
        shape=[],
        description='''
        GAPW: Exponent for hard compensation charge
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_BCSR_CODE = Quantity(
        type=str,
        shape=[],
        description='''
        Selects BCSR pathway.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_CHECK_BCSR_CODE = Quantity(
        type=str,
        shape=[],
        description='''
        Check the BCSR code on actual data, once per QS run.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_CORE_PPL = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the method used to calculate the local pseudopotential contribution.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_DEBUG_LRI_INTEGRALS = Quantity(
        type=str,
        shape=[],
        description='''
        Debug the integrals needed for LRIGPW.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_EPS_CORE_CHARGE = Quantity(
        type=str,
        shape=[],
        description='''
        Precision for mapping the core charges.Overrides EPS_DEFAULT/100.0 value
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_EPS_CPC = Quantity(
        type=str,
        shape=[],
        description='''
        Sets precision of the GAPW projection. Overrides EPS_DEFAULT value
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_EPS_DEFAULT = Quantity(
        type=str,
        shape=[],
        description='''
        Try setting all EPS_xxx to values leading to an energy correct up to EPS_DEFAULT
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_EPS_FILTER_MATRIX = Quantity(
        type=str,
        shape=[],
        description='''
        Sets the threshold for filtering matrix elements.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_EPS_GVG_RSPACE = Quantity(
        type=str,
        shape=[],
        description='''
        Sets precision of the realspace KS matrix element integration. Overrides
        SQRT(EPS_DEFAULT) value
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_EPS_KG_ORB = Quantity(
        type=str,
        shape=[],
        description='''
        Sets precision used in coloring the subsets for the Kim-Gordon method. Overrides
        SQRT(EPS_DEFAULT) value
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_EPS_PGF_ORB = Quantity(
        type=str,
        shape=[],
        description='''
        Sets precision of the overlap matrix elements. Overrides SQRT(EPS_DEFAULT) value
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_EPS_PPL = Quantity(
        type=str,
        shape=[],
        description='''
        Adjusts the precision for the local part of the pseudo potential.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_EPS_PPNL = Quantity(
        type=str,
        shape=[],
        description='''
        Sets precision of the non-local part of the pseudo potential. Overrides
        sqrt(EPS_DEFAULT) value
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_EPS_RHO_GSPACE = Quantity(
        type=str,
        shape=[],
        description='''
        Sets precision of the density mapping in gspace.Overrides EPS_DEFAULT
        value..Overrides EPS_RHO value
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_EPS_RHO_RSPACE = Quantity(
        type=str,
        shape=[],
        description='''
        Sets precision of the density mapping in rspace.Overrides EPS_DEFAULT
        value..Overrides EPS_RHO value
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_EPS_RHO = Quantity(
        type=str,
        shape=[],
        description='''
        Sets precision of the density mapping on the grids.Overrides EPS_DEFAULT value
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_EPSFIT = Quantity(
        type=str,
        shape=[],
        description='''
        GAPW: precision to give the extention of a hard gaussian
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_EPSISO = Quantity(
        type=str,
        shape=[],
        description='''
        GAPW: precision to determine an isolated projector
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_EPSRHO0 = Quantity(
        type=str,
        shape=[],
        description='''
        GAPW : precision to determine the range of V(rho0-rho0soft)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_EPSSVD = Quantity(
        type=str,
        shape=[],
        description='''
        GAPW: tolerance used in the singular value decomposition of the projector matrix
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_EXTRAPOLATION_ORDER = Quantity(
        type=str,
        shape=[],
        description='''
        Order for the PS or ASPC extrapolation (typically 2-4). Higher order might bring
        more accuracy, but comes, for large systems, also at some cost. In some cases, a
        high order extrapolation is not stable, and the order needs to be reduced.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_EXTRAPOLATION = Quantity(
        type=str,
        shape=[],
        description='''
        Extrapolation strategy for the wavefunction during e.g. MD.PS and ASPC are
        recommended, see also EXTRAPOLATION_ORDER.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_FORCE_PAW = Quantity(
        type=str,
        shape=[],
        description='''
        Use the GAPW scheme also for atoms with soft basis sets, i.e.  the local densities
        are computed even if hard and soft should be equal. If this keyword is not set to
        true, those atoms with soft basis sets are treated by a GPW scheme, i.e. the
        corresponding density contribution goes on the global grid and is expanded in PW.
        This option nullifies the effect of the GPW_TYPE in the atomic KIND
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_KG_METHOD = Quantity(
        type=str,
        shape=[],
        description='''
        Use a Kim-Gordon-like scheme.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_LADDN0 = Quantity(
        type=str,
        shape=[],
        description='''
        GAPW : integer added to the max L of the basis set, used to determine the maximum
        value of L for the compensation charge density.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_LMAXN0 = Quantity(
        type=str,
        shape=[],
        description='''
        GAPW : max L number for the expansion compensation densities in spherical
        gaussians
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_LMAXN1 = Quantity(
        type=str,
        shape=[],
        description='''
        GAPW : max L number for expansion of the atomic densities in spherical gaussians
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_LS_SCF = Quantity(
        type=str,
        shape=[],
        description='''
        Perform a linear scaling SCF
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_MAP_CONSISTENT = Quantity(
        type=str,
        shape=[],
        description='''
        Compute the exact derivative (Hks) of the energy with respect to the density
        matrix. This is slightly more expensive than using an approximate computation, but
        consistent mapping can improve the stability of the SCF procedure, especially for
        a tight EPS_SCF and a less tight EPS_DEFAULT.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_MAX_RAD_LOCAL = Quantity(
        type=str,
        shape=[],
        description='''
        GAPW : maximum radius of gaussian functions included in the generation of
        projectors
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_METHOD = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the electronic structure method that should be employed
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_PW_GRID_BLOCKED = Quantity(
        type=str,
        shape=[],
        description='''
        Can be used to set the distribution in g-space for the pw grids and their FFT.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_PW_GRID_LAYOUT = Quantity(
        type=str,
        shape=[],
        description='''
        Force a particular real-space layout for the plane waves grids. Numbers ≤ 0 mean
        that this dimension is free, incorrect layouts will be ignored. The default
        (/-1,-1/) causes CP2K to select a good value, i.e. plane distributed for large
        grids, more general distribution for small grids.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_PW_GRID = Quantity(
        type=str,
        shape=[],
        description='''
        What kind of PW_GRID should be employed
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_QUADRATURE = Quantity(
        type=str,
        shape=[],
        description='''
        GAPW: algorithm to construct the atomic radial grids
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_SCP = Quantity(
        type=str,
        shape=[],
        description='''
        Introduce additional self-consistent polarization through additional response
        basis functions (read in through AUX_BASIS.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_QS_TRANSPORT = Quantity(
        type=str,
        shape=[],
        description='''
        Perform transport calculations (coupling CP2K and OMEN)
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_QS_BECKE_RESTRAINT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_QS_BECKE_RESTRAINT'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_QS_DDAPC_RESTRAINT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_QS_DDAPC_RESTRAINT'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_QS_DFTB = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_QS_DFTB'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_QS_DISTRIBUTION = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_QS_DISTRIBUTION'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_QS_HARRIS = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_QS_HARRIS'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_QS_MULLIKEN_RESTRAINT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_QS_MULLIKEN_RESTRAINT'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_QS_OPTIMIZE_LRI_BASIS = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_QS_OPTIMIZE_LRI_BASIS'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_QS_S2_RESTRAINT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_QS_S2_RESTRAINT'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_QS_SCPTB = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_QS_SCPTB'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_QS_SE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_QS_SE'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT_REAL_TIME_PROPAGATION(MSection):
    '''
    Parameters needed to set up the real time propagation for the electron dynamics
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_REAL_TIME_PROPAGATION_ACCURACY_REFINEMENT = Quantity(
        type=str,
        shape=[],
        description='''
        If using density propagation some parts should be calculated with a higher
        accuracy than the rest to reduce numerical noise. This factor determines by how
        much the filtering threshold is reduced for these calculations.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_REAL_TIME_PROPAGATION_APPLY_DELTA_PULSE = Quantity(
        type=str,
        shape=[],
        description='''
        Applies a delta kick to the initial wfn (only RTP for now - the EMD  case is not
        yet implemented).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_REAL_TIME_PROPAGATION_ASPC_ORDER = Quantity(
        type=str,
        shape=[],
        description='''
        Speciefies how many steps will be used for extrapolation. One will be always used
        which is means X(t+dt)=X(t)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_REAL_TIME_PROPAGATION_CALCULATE_IDEMPOTENCY = Quantity(
        type=str,
        shape=[],
        description='''
        Calculates the idempotency. Costs some performance. Is for free if McWeeny is used
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_REAL_TIME_PROPAGATION_DELTA_PULSE_DIRECTION = Quantity(
        type=str,
        shape=[],
        description='''
        Direction of the applied electric field. The k vector is given as
        2*Pi*[i,j,k]*inv(h_mat), which for PERIODIC .FALSE. yields exp(ikr) periodic with
        the unit cell, only if DELTA_PULSE_SCALE is set to unity. For an orthorhombic cell
        [1,0,0] yields [2*Pi/L_x,0,0]. For small cells, this results in a very large kick.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_REAL_TIME_PROPAGATION_DELTA_PULSE_SCALE = Quantity(
        type=str,
        shape=[],
        description='''
        Scale the k vector, which for PERIODIC .FALSE. results in exp(ikr) no longer being
        periodic with the unit cell. The norm of k is the strength of the applied electric
        field in atomic units.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_REAL_TIME_PROPAGATION_DENSITY_PROPAGATION = Quantity(
        type=str,
        shape=[],
        description='''
        The density matrix is propagated instead of the molecular orbitals. This allows
        for a linear scaling simulation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_REAL_TIME_PROPAGATION_EPS_ITER = Quantity(
        type=str,
        shape=[],
        description='''
        Convergence criterium for the self consistent propagator loop.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_REAL_TIME_PROPAGATION_EXP_ACCURACY = Quantity(
        type=str,
        shape=[],
        description='''
        Accuracy for the taylor and pade approximation. This is only an upper bound bound
        since the norm used for the guess is an upper bound for the needed one.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_REAL_TIME_PROPAGATION_HFX_BALANCE_IN_CORE = Quantity(
        type=str,
        shape=[],
        description='''
        If HFX is used, this keyword forces a redistribution/recalculation of the
        integrals, balanced with respect to the in core steps.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_REAL_TIME_PROPAGATION_INITIAL_WFN = Quantity(
        type=str,
        shape=[],
        description='''
        Controls the initial WFN used for propagation.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_REAL_TIME_PROPAGATION_MAT_EXP = Quantity(
        type=str,
        shape=[],
        description='''
        Which method should be used to calculate the exponential in the propagator. For
        Ehrenfest MD only the Taylor method works, for real time propagation
        diagonalization works as well.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_REAL_TIME_PROPAGATION_MAX_ITER = Quantity(
        type=str,
        shape=[],
        description='''
        Maximal number of iterations for the self consistent propagator loop.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_REAL_TIME_PROPAGATION_MCWEENY_EPS = Quantity(
        type=str,
        shape=[],
        description='''
        Threshold after which McWeeny is terminated
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_REAL_TIME_PROPAGATION_MCWEENY_MAX_ITER = Quantity(
        type=str,
        shape=[],
        description='''
        Determines the maximum amount of McWeeny steps used after each converged step in
        density propagation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_REAL_TIME_PROPAGATION_ORTHONORMAL = Quantity(
        type=str,
        shape=[],
        description='''
        Performs rtp in the orthonormal basis, currently only works with density
        propagation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_REAL_TIME_PROPAGATION_PERIODIC = Quantity(
        type=str,
        shape=[],
        description='''
        Apply a delta-kick that is compatible with periodic boundary conditions for any
        value of DELTA_PULSE_SCALE. Uses perturbation theory for the preparation of the
        initial wfn. Note that the pulse is only applied when INITIAL_WFN is set to
        SCF_WFN, and not for restarts (RT_RESTART).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_REAL_TIME_PROPAGATION_PROPAGATOR = Quantity(
        type=str,
        shape=[],
        description='''
        Which propagator should be used for the orbitals
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_REAL_TIME_PROPAGATION_SC_CHECK_START = Quantity(
        type=str,
        shape=[],
        description='''
        Speciefies how many iteration steps will be done without a check for self
        consistency. Can save some time in big calculations.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_REAL_TIME_PROPAGATION_WRITE_RESTART = Quantity(
        type=str,
        shape=[],
        description='''
        For density propagation. It writes out the density matrix after each step
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_RELATIVISTIC(MSection):
    '''
    parameters needed and setup for relativistic calculations
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_RELATIVISTIC_DKH_ORDER = Quantity(
        type=str,
        shape=[],
        description='''
        The order of the DKH transformation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_RELATIVISTIC_METHOD = Quantity(
        type=str,
        shape=[],
        description='''
        type of relativistic correction used
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_RELATIVISTIC_POTENTIAL = Quantity(
        type=str,
        shape=[],
        description='''
        External potential used in DKH transformation, full 1/r or erfc(r)/r
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_RELATIVISTIC_TRANSFORMATION = Quantity(
        type=str,
        shape=[],
        description='''
        Type of DKH transformation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_RELATIVISTIC_Z_CUTOFF = Quantity(
        type=str,
        shape=[],
        description='''
        The minimal atomic number considered for atom transformation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_RELATIVISTIC_ZORA_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Type of ZORA method to be used
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_SCCS_ANDREUSSI(MSection):
    '''
    Define the parameters of the dielectric smoothing function proposed by Andreussi et
    al.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_SCCS_ANDREUSSI_RHO_MAX = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum density value used for the smoothing of the dielectric function
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCCS_ANDREUSSI_RHO_MIN = Quantity(
        type=str,
        shape=[],
        description='''
        Minimum density value used for the smoothing of the dielectric function
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_SCCS_FATTEBERT_GYGI(MSection):
    '''
    Define the parameters of the dielectric smoothing function proposed by Fattebert and
    Gygi
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_SCCS_FATTEBERT_GYGI_BETA = Quantity(
        type=str,
        shape=[],
        description='''
        Parameter β changes the width of the interface solute-solvent
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCCS_FATTEBERT_GYGI_RHO_ZERO = Quantity(
        type=str,
        shape=[],
        description='''
        Parameter ρ<sub>0</sub> defines the critical density in the middle of the
        interface solute-solvent
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_SCCS(MSection):
    '''
    Define the parameters for self-consistent continuum solvation (SCCS) model
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_SCCS_ALPHA = Quantity(
        type=str,
        shape=[],
        description='''
        Solvent specific tunable parameter for the calculation of the repulsion term
        <i>G<sup>rep</sup> = α S</i> where <i>S</i> is the (quantum) surface of the cavity
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCCS_BETA = Quantity(
        type=str,
        shape=[],
        description='''
        Solvent specific tunable parameter for the calculation of the dispersion term
        <i>G<sup>dis</sup> = β V</i> where <i>V</i> is the (quantum) volume of the cavity
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCCS_DELTA_RHO = Quantity(
        type=str,
        shape=[],
        description='''
        Numerical increment for the calculation of the (quantum) surface of the solute
        cavity
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCCS_DERIVATIVE_METHOD = Quantity(
        type=str,
        shape=[],
        description='''
        Method for the calculation of the numerical derivatives on the real-space grids
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCCS_DIELECTRIC_CONSTANT = Quantity(
        type=str,
        shape=[],
        description='''
        Dielectric constant of the solvent
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCCS_EPS_SCCS = Quantity(
        type=str,
        shape=[],
        description='''
        Tolerance for the convergence of the polarisation density, i.e. requested accuracy
        for the SCCS iteration cycle
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCCS_EPS_SCF = Quantity(
        type=str,
        shape=[],
        description='''
        The SCCS iteration cycle is activated only if the SCF iteration cycle is converged
        to this threshold value
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCCS_GAMMA = Quantity(
        type=str,
        shape=[],
        description='''
        Surface tension of the solvent used for the calculation of the cavitation term
        <i>G<sup>cav</sup> = γ S</i> where <i>S</i> is the (quantum) surface of the cavity
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCCS_MAX_ITER = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum number of SCCS iteration steps performed to converge within the given
        tolerance
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCCS_METHOD = Quantity(
        type=str,
        shape=[],
        description='''
        Method used for the smoothing of the dielectric function
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCCS_MIXING = Quantity(
        type=str,
        shape=[],
        description='''
        Mixing parameter (Hartree damping) employed during the iteration procedure
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCCS_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Controls the activation of the SCCS section
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_SCCS_ANDREUSSI = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_SCCS_ANDREUSSI'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_SCCS_FATTEBERT_GYGI = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_SCCS_FATTEBERT_GYGI'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_DAVIDSON(MSection):
    '''
    Settings for DAVIDSON
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_DAVIDSON_CONV_MOS_PERCENT = Quantity(
        type=str,
        shape=[],
        description='''
        Minimal percent of MOS that have to converge within the Davidson loop before the
        SCF iteration is completed and a new Hamiltonian is computed
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_DAVIDSON_ENERGY_GAP = Quantity(
        type=str,
        shape=[],
        description='''
        Should be an estimate for the energy gap [a.u.] (HOMO-LUMO) and is used in
        preconditioning, especially effective with the FULL_ALL preconditioner, in which
        case it should be an underestimate of the gap (0.001 doing normally fine). For the
        other preconditioners, making this value larger (0.2) will tame the preconditioner
        in case of poor initial guesses.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_DAVIDSON_FIRST_PREC = Quantity(
        type=str,
        shape=[],
        description='''
        First SCF iteration at which a Preconditioner is employed
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_DAVIDSON_NEW_PREC_EACH = Quantity(
        type=str,
        shape=[],
        description='''
        Number of SCF iterations after which a new Preconditioner is computed
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_DAVIDSON_PRECOND_SOLVER = Quantity(
        type=str,
        shape=[],
        description='''
        How the preconditioner is applied to the residual.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_DAVIDSON_PRECONDITIONER = Quantity(
        type=str,
        shape=[],
        description='''
        Type of preconditioner to be used with all minimization schemes.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_DAVIDSON_SPARSE_MOS = Quantity(
        type=str,
        shape=[],
        description='''
        Use MOS as sparse matrix and avoid as much as possible multiplications with full
        matrices
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_DIAG_SUB_SCF_MIXING(MSection):
    '''
    Define type and parameters for mixingprocedures to be applied to the density matrix.
    Normally, only one type of mixing method should be accepted. The mixing procedures
    activated by this section are only active for diagonalization methods, i.e. not with
    minimization methods based on OT.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_DIAG_SUB_SCF_MIXING_ALPHA = Quantity(
        type=str,
        shape=[],
        description='''
        Fraction of new density to be included
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_DIAG_SUB_SCF_MIXING_BETA = Quantity(
        type=str,
        shape=[],
        description='''
        Denominator parameter in Kerker damping introduced to suppress charge sloshing:
        rho_mix(g) =rho_in(g) + alpha*g^2/(g^2 + beta^2)*(rho_out(g)-rho_in(g))
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_DIAG_SUB_SCF_MIXING_BROY_W0 = Quantity(
        type=str,
        shape=[],
        description='''
        w0 parameter used in Broyden mixing
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_DIAG_SUB_SCF_MIXING_BROY_WMAX = Quantity(
        type=str,
        shape=[],
        description='''
        Settings for BROY_WMAX
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_DIAG_SUB_SCF_MIXING_BROY_WREF = Quantity(
        type=str,
        shape=[],
        description='''
        Settings for BROY_WREF
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_DIAG_SUB_SCF_MIXING_MAX_GVEC_EXP = Quantity(
        type=str,
        shape=[],
        description='''
        Restricts the G-space mixing to lower part of G-vector spectrum, up to a G0, by
        assigning the exponent of the Gaussian that can be represented by vectors smaller
        than G0 within a certain accuracy.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_DIAG_SUB_SCF_MIXING_MAX_STEP = Quantity(
        type=str,
        shape=[],
        description='''
        Upper bound for the magnitude of the unpredicted step size in the update by the
        multisecant mixing scheme
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_DIAG_SUB_SCF_MIXING_METHOD = Quantity(
        type=str,
        shape=[],
        description='''
        Mixing method to be applied
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_DIAG_SUB_SCF_MIXING_N_SIMPLE_MIX = Quantity(
        type=str,
        shape=[],
        description='''
        Number of kerker damping iterations before starting other mixing procedures
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_DIAG_SUB_SCF_MIXING_NBUFFER = Quantity(
        type=str,
        shape=[],
        description='''
        Number of previous steps stored for the actual mixing scheme
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_DIAG_SUB_SCF_MIXING_NMIXING = Quantity(
        type=str,
        shape=[],
        description='''
        Minimal number of density mixing (should be greater than 0),before starting DIIS
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_DIAG_SUB_SCF_MIXING_NSKIP = Quantity(
        type=str,
        shape=[],
        description='''
        Number of initial iteration for which the mixing is skipped
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_DIAG_SUB_SCF_MIXING_PULAY_ALPHA = Quantity(
        type=str,
        shape=[],
        description='''
        Fraction of new density to be added to the Pulay expansion
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_DIAG_SUB_SCF_MIXING_PULAY_BETA = Quantity(
        type=str,
        shape=[],
        description='''
        Fraction of residual contribution to be added to Pulay expansion
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_DIAG_SUB_SCF_MIXING_R_FACTOR = Quantity(
        type=str,
        shape=[],
        description='''
        Control factor for the magnitude of the unpredicted step size in the update by the
        multisecant mixing scheme
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_DIAG_SUB_SCF_MIXING_REGULARIZATION = Quantity(
        type=str,
        shape=[],
        description='''
        Regularization parameter to stabilize the inversion of the residual matrix {Yn^t
        Yn} in the multisecant mixing scheme (noise)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_DIAG_SUB_SCF_MIXING_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Controls the activation of the mixing procedure
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_DIAG_SUB_SCF(MSection):
    '''
    Activation of self-consistenf subspace refinement by diagonalization of H by adjusting
    the occupation but keeping the MOS unchanged.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_DIAG_SUB_SCF_EPS_ADAPT_SCF = Quantity(
        type=str,
        shape=[],
        description='''
        Required density matrix accuracy as compared to current SCF convergence
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_DIAG_SUB_SCF_EPS_ENE = Quantity(
        type=str,
        shape=[],
        description='''
        Required energy accuracy for convergence of subspace diagonalization
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_DIAG_SUB_SCF_EPS_SKIP_SUB_DIAG = Quantity(
        type=str,
        shape=[],
        description='''
        Level of convergence to be reached before starting the internal loop of subspace
        rotations. Above this threshold only the outer diagonalization method is used.  If
        negative the subspace rotation is started at the first iteration
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_DIAG_SUB_SCF_MAX_ITER = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum number of iterations for the SCF inner loop
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_DIAG_SUB_SCF_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        controls the activation of inner SCF loop to refine occupations in MOS subspace
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_DIAG_SUB_SCF_MIXING = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_DIAG_SUB_SCF_MIXING'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_KRYLOV(MSection):
    '''
    Settings for KRYLOV
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_KRYLOV_CHECK_MOS_CONV = Quantity(
        type=str,
        shape=[],
        description='''
        This requires to check the convergence of MOS also when standard diagonalization
        steps are performed, if the block krylov approach is active.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_KRYLOV_EPS_KRYLOV = Quantity(
        type=str,
        shape=[],
        description='''
        Convergence criterion for the MOs
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_KRYLOV_EPS_STD_DIAG = Quantity(
        type=str,
        shape=[],
        description='''
        Level of convergence to be reached before starting the Lanczos procedure. Above
        this threshold a standard diagonalization method is used.  If negative Lanczos is
        started at the first iteration
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_KRYLOV_NBLOCK = Quantity(
        type=str,
        shape=[],
        description='''
        Size of the block of vectors refined simultaneously by the Lanczos procedure
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_KRYLOV_NKRYLOV = Quantity(
        type=str,
        shape=[],
        description='''
        Dimension of the Krylov space used for the Lanczos refinement
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_OT(MSection):
    '''
    Sets the various options for the orbital transformation (OT) method. Default settings
    already provide an efficient, yet robust method. Most systems benefit from using the
    FULL_ALL preconditioner combined with a small value (0.001) of ENERGY_GAP.Well-behaved
    systems might benefit from using a DIIS minimizer.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_OT_ALGORITHM = Quantity(
        type=str,
        shape=[],
        description='''
        Algorithm to be used for OT
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_OT_BROYDEN_ADAPTIVE_SIGMA = Quantity(
        type=str,
        shape=[],
        description='''
        Enable adaptive curvature estimation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_OT_BROYDEN_BETA = Quantity(
        type=str,
        shape=[],
        description='''
        Underrelaxation for the broyden mixer
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_OT_BROYDEN_ENABLE_FLIP = Quantity(
        type=str,
        shape=[],
        description='''
        Ensure positive definite update
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_OT_BROYDEN_ETA = Quantity(
        type=str,
        shape=[],
        description='''
        Dampening of estimated energy curvature.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_OT_BROYDEN_FORGET_HISTORY = Quantity(
        type=str,
        shape=[],
        description='''
        Forget history on bad approximation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_OT_BROYDEN_GAMMA = Quantity(
        type=str,
        shape=[],
        description='''
        Backtracking parameter
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_OT_BROYDEN_OMEGA = Quantity(
        type=str,
        shape=[],
        description='''
        Growth limit of curvature.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_OT_BROYDEN_SIGMA_DECREASE = Quantity(
        type=str,
        shape=[],
        description='''
        Reduction of curvature on bad approximation.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_OT_BROYDEN_SIGMA_MIN = Quantity(
        type=str,
        shape=[],
        description='''
        Minimum adaptive curvature.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_OT_BROYDEN_SIGMA = Quantity(
        type=str,
        shape=[],
        description='''
        Curvature of energy functional.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_OT_ENERGIES = Quantity(
        type=str,
        shape=[],
        description='''
        Optimize orbital energies for use in Fermi-Dirac smearing (requires ROTATION and
        FD smearing to be active).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_OT_ENERGY_GAP = Quantity(
        type=str,
        shape=[],
        description='''
        Should be an estimate for the energy gap [a.u.] (HOMO-LUMO) and is used in
        preconditioning, especially effective with the FULL_ALL preconditioner, in which
        case it should be an underestimate of the gap (can be a small number, e.g. 0.002).
        FULL_SINGLE_INVERSE takes it as lower bound (values below 0.05 can cause stability
        issues). In general, heigher values will tame the preconditioner in case of poor
        initial guesses. A negative value will leave the choice to CP2K depending on type
        of preconditioner.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_OT_EPS_IRAC_FILTER_MATRIX = Quantity(
        type=str,
        shape=[],
        description='''
        Sets the threshold for filtering the matrices.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_OT_EPS_IRAC_QUICK_EXIT = Quantity(
        type=str,
        shape=[],
        description='''
        Only one extra refinement iteration is done when the norm is below this value.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_OT_EPS_IRAC_SWITCH = Quantity(
        type=str,
        shape=[],
        description='''
        The algorithm switches to the polynomial refinement when the norm is below this
        value.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_OT_EPS_IRAC = Quantity(
        type=str,
        shape=[],
        description='''
        Targeted accuracy during the refinement iteration.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_OT_EPS_TAYLOR = Quantity(
        type=str,
        shape=[],
        description='''
        Target accuracy of the taylor expansion for the matrix functions, should normally
        be kept as is.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_OT_GOLD_TARGET = Quantity(
        type=str,
        shape=[],
        description='''
        Target relative uncertainty in the location of the minimum for LINESEARCH GOLD
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_OT_IRAC_DEGREE = Quantity(
        type=str,
        shape=[],
        description='''
        The refinement polynomial degree (2, 3 or 4).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_OT_LINESEARCH = Quantity(
        type=str,
        shape=[],
        description='''
        1D line search algorithm to be used with the OT minimizer, in increasing order of
        robustness and cost. MINIMIZER CG combined with LINESEARCH GOLD should always find
        an electronic minimum.  Whereas the 2PNT minimizer is almost always OK, 3PNT might
        be needed for systems in which successive OT CG steps do not decrease the total
        energy.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_OT_MAX_IRAC = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum allowed refinement iteration.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_OT_MAX_TAYLOR = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum order of the Taylor expansion before diagonalisation is prefered, for
        large parallel runs a slightly higher order could sometimes result in a small
        speedup.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_OT_MINIMIZER = Quantity(
        type=str,
        shape=[],
        description='''
        Minimizer to be used with the OT method
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_OT_MIXED_PRECISION = Quantity(
        type=str,
        shape=[],
        description='''
        Uses a mixed precision algorithm.With a well behaved basis set (i.e. condition
        number less than 1/eps_sp)it provides double precision accuracy results and up to
        a 2 fold speedup for building and applying the preconditioner.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_OT_N_HISTORY_VEC = Quantity(
        type=str,
        shape=[],
        description='''
        Number of history vectors to be used with DIIS or BROYDEN
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_OT_NONDIAG_ENERGY_STRENGTH = Quantity(
        type=str,
        shape=[],
        description='''
        The prefactor for the non-diagonal energy penalty (FD smearing)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_OT_NONDIAG_ENERGY = Quantity(
        type=str,
        shape=[],
        description='''
        Add a non-diagonal energy penalty (FD smearing)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_OT_OCCUPATION_PRECONDITIONER = Quantity(
        type=str,
        shape=[],
        description='''
        Preconditioner with the occupation numbers (FD smearing)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_OT_ON_THE_FLY_LOC = Quantity(
        type=str,
        shape=[],
        description='''
        On the fly localization of the molecular orbitals. Can only be used with OT/IRAC.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_OT_ORTHO_IRAC = Quantity(
        type=str,
        shape=[],
        description='''
        The orthogonality method.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_OT_PRECOND_SOLVER = Quantity(
        type=str,
        shape=[],
        description='''
        How the preconditioner is applied to the residual.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_OT_PRECONDITIONER = Quantity(
        type=str,
        shape=[],
        description='''
        Type of preconditioner to be used with all minimization schemes. They differ in
        effectiveness, cost of construction, cost of application. Properly preconditioned
        minimization can be orders of magnitude faster than doing nothing.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_OT_ROTATION = Quantity(
        type=str,
        shape=[],
        description='''
        Introduce additional variables so that rotations of the occupied subspace are
        allowed as well, only needed for cases where the energy is not invariant under  a
        rotation of the occupied subspace such as non-singlet restricted calculations  or
        fractional occupations.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_OT_SAFE_DIIS = Quantity(
        type=str,
        shape=[],
        description='''
        Reject DIIS steps if they point away from the minimum, do SD in that case.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_OT_SCP_NDDO = Quantity(
        type=str,
        shape=[],
        description='''
        Introduce additional self-consistent polarization through response basis set =
        orbital basis set for NDDO.)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_OT_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        controls the activation of the ot method
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_OT_STEPSIZE = Quantity(
        type=str,
        shape=[],
        description='''
        Initial stepsize used for the line search, sometimes this parameter can be reduced
        to stablize DIIS or to improve the CG behavior in the first few steps. The optimal
        value depends on the quality of the preconditioner. A negative values leaves the
        choice to CP2K depending on the preconditioner.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION(MSection):
    '''
    Set up type and parameters for Kohn-Sham matrix diagonalization.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_ALGORITHM = Quantity(
        type=str,
        shape=[],
        description='''
        Algorithm to be used for diagonalization
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_EPS_ADAPT = Quantity(
        type=str,
        shape=[],
        description='''
        Required accuracy in iterative diagonalization as compared to current SCF
        convergence
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_EPS_ITER = Quantity(
        type=str,
        shape=[],
        description='''
        Required accuracy in iterative diagonalization
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_EPS_JACOBI = Quantity(
        type=str,
        shape=[],
        description='''
        Below this threshold value for the SCF convergence the pseudo-diagonalization
        method using Jacobi rotations is activated. This method is much faster than a real
        diagonalization and it is even speeding up while achieving full
        convergence.However, it needs a pre-converged wavefunction obtained by at least
        one real diagonalization which is further optimized while keeping the original
        eigenvalue spectrum. The MO eigenvalues are NOT updated. The method might be
        useful to speed up calculations for large systems e.g. using a semi-empirical
        method.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_JACOBI_THRESHOLD = Quantity(
        type=str,
        shape=[],
        description='''
        Controls the accuracy of the pseudo-diagonalization method using Jacobi rotations
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_MAX_ITER = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum number of iterations in iterative diagonalization
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        controls the activation of the diagonalization method
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_DAVIDSON = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_DAVIDSON'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_DIAG_SUB_SCF = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_DIAG_SUB_SCF'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_KRYLOV = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_KRYLOV'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_OT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION_OT'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT_SCF_MIXING(MSection):
    '''
    Define type and parameters for mixingprocedures to be applied to the density matrix.
    Normally, only one type of mixing method should be accepted. The mixing procedures
    activated by this section are only active for diagonalization methods, i.e. not with
    minimization methods based on OT.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_SCF_MIXING_ALPHA = Quantity(
        type=str,
        shape=[],
        description='''
        Fraction of new density to be included
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_MIXING_BETA = Quantity(
        type=str,
        shape=[],
        description='''
        Denominator parameter in Kerker damping introduced to suppress charge sloshing:
        rho_mix(g) =rho_in(g) + alpha*g^2/(g^2 + beta^2)*(rho_out(g)-rho_in(g))
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_MIXING_BROY_W0 = Quantity(
        type=str,
        shape=[],
        description='''
        w0 parameter used in Broyden mixing
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_MIXING_BROY_WMAX = Quantity(
        type=str,
        shape=[],
        description='''
        Settings for BROY_WMAX
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_MIXING_BROY_WREF = Quantity(
        type=str,
        shape=[],
        description='''
        Settings for BROY_WREF
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_MIXING_MAX_GVEC_EXP = Quantity(
        type=str,
        shape=[],
        description='''
        Restricts the G-space mixing to lower part of G-vector spectrum, up to a G0, by
        assigning the exponent of the Gaussian that can be represented by vectors smaller
        than G0 within a certain accuracy.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_MIXING_MAX_STEP = Quantity(
        type=str,
        shape=[],
        description='''
        Upper bound for the magnitude of the unpredicted step size in the update by the
        multisecant mixing scheme
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_MIXING_METHOD = Quantity(
        type=str,
        shape=[],
        description='''
        Mixing method to be applied
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_MIXING_N_SIMPLE_MIX = Quantity(
        type=str,
        shape=[],
        description='''
        Number of kerker damping iterations before starting other mixing procedures
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_MIXING_NBUFFER = Quantity(
        type=str,
        shape=[],
        description='''
        Number of previous steps stored for the actual mixing scheme
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_MIXING_NMIXING = Quantity(
        type=str,
        shape=[],
        description='''
        Minimal number of density mixing (should be greater than 0),before starting DIIS
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_MIXING_NSKIP = Quantity(
        type=str,
        shape=[],
        description='''
        Number of initial iteration for which the mixing is skipped
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_MIXING_PULAY_ALPHA = Quantity(
        type=str,
        shape=[],
        description='''
        Fraction of new density to be added to the Pulay expansion
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_MIXING_PULAY_BETA = Quantity(
        type=str,
        shape=[],
        description='''
        Fraction of residual contribution to be added to Pulay expansion
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_MIXING_R_FACTOR = Quantity(
        type=str,
        shape=[],
        description='''
        Control factor for the magnitude of the unpredicted step size in the update by the
        multisecant mixing scheme
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_MIXING_REGULARIZATION = Quantity(
        type=str,
        shape=[],
        description='''
        Regularization parameter to stabilize the inversion of the residual matrix {Yn^t
        Yn} in the multisecant mixing scheme (noise)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_MIXING_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Controls the activation of the mixing procedure
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_SCF_OT(MSection):
    '''
    Sets the various options for the orbital transformation (OT) method. Default settings
    already provide an efficient, yet robust method. Most systems benefit from using the
    FULL_ALL preconditioner combined with a small value (0.001) of ENERGY_GAP.Well-behaved
    systems might benefit from using a DIIS minimizer.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OT_ALGORITHM = Quantity(
        type=str,
        shape=[],
        description='''
        Algorithm to be used for OT
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OT_BROYDEN_ADAPTIVE_SIGMA = Quantity(
        type=str,
        shape=[],
        description='''
        Enable adaptive curvature estimation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OT_BROYDEN_BETA = Quantity(
        type=str,
        shape=[],
        description='''
        Underrelaxation for the broyden mixer
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OT_BROYDEN_ENABLE_FLIP = Quantity(
        type=str,
        shape=[],
        description='''
        Ensure positive definite update
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OT_BROYDEN_ETA = Quantity(
        type=str,
        shape=[],
        description='''
        Dampening of estimated energy curvature.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OT_BROYDEN_FORGET_HISTORY = Quantity(
        type=str,
        shape=[],
        description='''
        Forget history on bad approximation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OT_BROYDEN_GAMMA = Quantity(
        type=str,
        shape=[],
        description='''
        Backtracking parameter
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OT_BROYDEN_OMEGA = Quantity(
        type=str,
        shape=[],
        description='''
        Growth limit of curvature.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OT_BROYDEN_SIGMA_DECREASE = Quantity(
        type=str,
        shape=[],
        description='''
        Reduction of curvature on bad approximation.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OT_BROYDEN_SIGMA_MIN = Quantity(
        type=str,
        shape=[],
        description='''
        Minimum adaptive curvature.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OT_BROYDEN_SIGMA = Quantity(
        type=str,
        shape=[],
        description='''
        Curvature of energy functional.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OT_ENERGIES = Quantity(
        type=str,
        shape=[],
        description='''
        Optimize orbital energies for use in Fermi-Dirac smearing (requires ROTATION and
        FD smearing to be active).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OT_ENERGY_GAP = Quantity(
        type=str,
        shape=[],
        description='''
        Should be an estimate for the energy gap [a.u.] (HOMO-LUMO) and is used in
        preconditioning, especially effective with the FULL_ALL preconditioner, in which
        case it should be an underestimate of the gap (can be a small number, e.g. 0.002).
        FULL_SINGLE_INVERSE takes it as lower bound (values below 0.05 can cause stability
        issues). In general, heigher values will tame the preconditioner in case of poor
        initial guesses. A negative value will leave the choice to CP2K depending on type
        of preconditioner.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OT_EPS_IRAC_FILTER_MATRIX = Quantity(
        type=str,
        shape=[],
        description='''
        Sets the threshold for filtering the matrices.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OT_EPS_IRAC_QUICK_EXIT = Quantity(
        type=str,
        shape=[],
        description='''
        Only one extra refinement iteration is done when the norm is below this value.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OT_EPS_IRAC_SWITCH = Quantity(
        type=str,
        shape=[],
        description='''
        The algorithm switches to the polynomial refinement when the norm is below this
        value.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OT_EPS_IRAC = Quantity(
        type=str,
        shape=[],
        description='''
        Targeted accuracy during the refinement iteration.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OT_EPS_TAYLOR = Quantity(
        type=str,
        shape=[],
        description='''
        Target accuracy of the taylor expansion for the matrix functions, should normally
        be kept as is.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OT_GOLD_TARGET = Quantity(
        type=str,
        shape=[],
        description='''
        Target relative uncertainty in the location of the minimum for LINESEARCH GOLD
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OT_IRAC_DEGREE = Quantity(
        type=str,
        shape=[],
        description='''
        The refinement polynomial degree (2, 3 or 4).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OT_LINESEARCH = Quantity(
        type=str,
        shape=[],
        description='''
        1D line search algorithm to be used with the OT minimizer, in increasing order of
        robustness and cost. MINIMIZER CG combined with LINESEARCH GOLD should always find
        an electronic minimum.  Whereas the 2PNT minimizer is almost always OK, 3PNT might
        be needed for systems in which successive OT CG steps do not decrease the total
        energy.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OT_MAX_IRAC = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum allowed refinement iteration.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OT_MAX_TAYLOR = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum order of the Taylor expansion before diagonalisation is prefered, for
        large parallel runs a slightly higher order could sometimes result in a small
        speedup.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OT_MINIMIZER = Quantity(
        type=str,
        shape=[],
        description='''
        Minimizer to be used with the OT method
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OT_MIXED_PRECISION = Quantity(
        type=str,
        shape=[],
        description='''
        Uses a mixed precision algorithm.With a well behaved basis set (i.e. condition
        number less than 1/eps_sp)it provides double precision accuracy results and up to
        a 2 fold speedup for building and applying the preconditioner.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OT_N_HISTORY_VEC = Quantity(
        type=str,
        shape=[],
        description='''
        Number of history vectors to be used with DIIS or BROYDEN
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OT_NONDIAG_ENERGY_STRENGTH = Quantity(
        type=str,
        shape=[],
        description='''
        The prefactor for the non-diagonal energy penalty (FD smearing)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OT_NONDIAG_ENERGY = Quantity(
        type=str,
        shape=[],
        description='''
        Add a non-diagonal energy penalty (FD smearing)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OT_OCCUPATION_PRECONDITIONER = Quantity(
        type=str,
        shape=[],
        description='''
        Preconditioner with the occupation numbers (FD smearing)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OT_ON_THE_FLY_LOC = Quantity(
        type=str,
        shape=[],
        description='''
        On the fly localization of the molecular orbitals. Can only be used with OT/IRAC.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OT_ORTHO_IRAC = Quantity(
        type=str,
        shape=[],
        description='''
        The orthogonality method.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OT_PRECOND_SOLVER = Quantity(
        type=str,
        shape=[],
        description='''
        How the preconditioner is applied to the residual.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OT_PRECONDITIONER = Quantity(
        type=str,
        shape=[],
        description='''
        Type of preconditioner to be used with all minimization schemes. They differ in
        effectiveness, cost of construction, cost of application. Properly preconditioned
        minimization can be orders of magnitude faster than doing nothing.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OT_ROTATION = Quantity(
        type=str,
        shape=[],
        description='''
        Introduce additional variables so that rotations of the occupied subspace are
        allowed as well, only needed for cases where the energy is not invariant under  a
        rotation of the occupied subspace such as non-singlet restricted calculations  or
        fractional occupations.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OT_SAFE_DIIS = Quantity(
        type=str,
        shape=[],
        description='''
        Reject DIIS steps if they point away from the minimum, do SD in that case.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OT_SCP_NDDO = Quantity(
        type=str,
        shape=[],
        description='''
        Introduce additional self-consistent polarization through response basis set =
        orbital basis set for NDDO.)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OT_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        controls the activation of the ot method
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OT_STEPSIZE = Quantity(
        type=str,
        shape=[],
        description='''
        Initial stepsize used for the line search, sometimes this parameter can be reduced
        to stablize DIIS or to improve the CG behavior in the first few steps. The optimal
        value depends on the quality of the preconditioner. A negative values leaves the
        choice to CP2K depending on the preconditioner.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_SCF_OUTER_SCF(MSection):
    '''
    parameters controlling the outer SCF loop
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OUTER_SCF_BISECT_TRUST_COUNT = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum number of times the same point will be used in bisection, a small number
        guards against the effect of wrongly converged states.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OUTER_SCF_DIIS_BUFFER_LENGTH = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum number of DIIS vectors used
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OUTER_SCF_EPS_SCF = Quantity(
        type=str,
        shape=[],
        description='''
        The target gradient of the outer scf variables. Notice that the EPS_SCF of the
        inner loop also determines the value that can be reached in the outer loop,
        typically EPS_SCF of the outer loop must be smaller than EPS_SCF of the inner
        loop.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OUTER_SCF_EXTRAPOLATION_ORDER = Quantity(
        type=str,
        shape=[],
        description='''
        Number of past states used in the extrapolation of the variables during e.g. MD
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OUTER_SCF_MAX_SCF = Quantity(
        type=str,
        shape=[],
        description='''
        The maximum number of outer loops
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OUTER_SCF_OPTIMIZER = Quantity(
        type=str,
        shape=[],
        description='''
        Method used to bring the outer loop to a stationary point
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OUTER_SCF_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        controls the activation of the outer SCF loop
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OUTER_SCF_STEP_SIZE = Quantity(
        type=str,
        shape=[],
        description='''
        The initial step_size used in the optimizer (currently steepest descent).Note that
        in cases where a sadle point is sought for (DDAPC_CONSTRAINT), this can be
        negative
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_OUTER_SCF_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies which kind of outer SCF should be employed
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_SCF_SMEAR(MSection):
    '''
    Define the smearing of the MO occupation numbers
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_SCF_SMEAR_ELECTRONIC_TEMPERATURE = Quantity(
        type=str,
        shape=[],
        description='''
        Electronic temperature in the case of Fermi-Dirac smearing
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_SMEAR_EPS_FERMI_DIRAC = Quantity(
        type=str,
        shape=[],
        description='''
        Accuracy checks on occupation numbers use this as a tolerance
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_SMEAR_FIXED_MAGNETIC_MOMENT = Quantity(
        type=str,
        shape=[],
        description='''
        Imposed difference between the numbers of electrons of spin up and spin down: m =
        n(up) - n(down). A negative value (default) allows for a change of the magnetic
        moment. -1 specifically keeps an integer number of spin up and spin down
        electrons.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_SMEAR_LIST = Quantity(
        type=str,
        shape=[],
        description='''
        A list of fractional occupations to use. Must match the number of states and sum
        up to the correct number of electrons
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_SMEAR_METHOD = Quantity(
        type=str,
        shape=[],
        description='''
        Smearing method to be applied
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_SMEAR_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Controls the activation of smearing
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_SMEAR_WINDOW_SIZE = Quantity(
        type=str,
        shape=[],
        description='''
        Size of the energy window centred at the Fermi level
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_SCF(MSection):
    '''
    parameters needed perform an scf run
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_SCF_ADDED_MOS = Quantity(
        type=str,
        shape=[],
        description='''
        Number of additional MOS added for each spin
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_CHOLESKY = Quantity(
        type=str,
        shape=[],
        description='''
        If the cholesky method should be used for computing the inverse of S, and in this
        case calling which Lapack routines
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_EPS_DIIS = Quantity(
        type=str,
        shape=[],
        description='''
        Threshold on the convergence to start using DIAG/DIIS
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_EPS_EIGVAL = Quantity(
        type=str,
        shape=[],
        description='''
        Throw away linear combinations of basis functions with a small eigenvalue in S
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_EPS_LUMO = Quantity(
        type=str,
        shape=[],
        description='''
        target accuracy of the computation of the lumo energy
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_EPS_SCF_HISTORY = Quantity(
        type=str,
        shape=[],
        description='''
        target accuracy for the scf convergence after the history pipeline is filled
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_EPS_SCF = Quantity(
        type=str,
        shape=[],
        description='''
        target accuracy for the scf convergence
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_LEVEL_SHIFT = Quantity(
        type=str,
        shape=[],
        description='''
        Use level shifting to improve convergence
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_MAX_DIIS = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum number of DIIS vectors to be used
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_MAX_ITER_LUMO = Quantity(
        type=str,
        shape=[],
        description='''
        The maximum number of iteration for the lumo computation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_MAX_SCF_HISTORY = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum number of SCF iterations after the history pipeline is filled
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_MAX_SCF = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum number of SCF iteration to be performed for one optimization
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_NCOL_BLOCK = Quantity(
        type=str,
        shape=[],
        description='''
        Sets the number of columns in a scalapack block
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_NROW_BLOCK = Quantity(
        type=str,
        shape=[],
        description='''
        sets the number of rows in a scalapack block
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_ROKS_F = Quantity(
        type=str,
        shape=[],
        description='''
        Allows to define the parameter f for the general ROKS scheme.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_ROKS_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Allows to define all parameters for the high-spin ROKS scheme explicitly. The full
        set of 6 parameters has to be specified in the order acc, bcc, aoo, boo, avv, bvv
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_ROKS_SCHEME = Quantity(
        type=str,
        shape=[],
        description='''
        Selects the ROKS scheme when ROKS is applied.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCF_SCF_GUESS = Quantity(
        type=str,
        shape=[],
        description='''
        Change the initial guess for the wavefunction.
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_SCF_DIAGONALIZATION'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_SCF_MIXING = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_SCF_MIXING'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_SCF_OT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_SCF_OT'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_SCF_OUTER_SCF = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_SCF_OUTER_SCF'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_SCF_SMEAR = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_SCF_SMEAR'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT_SCRF_PROGRAM_RUN_INFO(MSection):
    '''
    Controls the printing basic info about the method
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_SCRF_PROGRAM_RUN_INFO_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCRF_PROGRAM_RUN_INFO_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCRF_PROGRAM_RUN_INFO_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCRF_PROGRAM_RUN_INFO_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCRF_PROGRAM_RUN_INFO_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_SCRF_SPHERE_CENTER(MSection):
    '''
    Defines the center of the sphere.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_SCRF_SPHERE_CENTER_ATOM_LIST = Quantity(
        type=str,
        shape=[],
        description='''
        Defines a list of atoms to define the center of the sphere
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCRF_SPHERE_CENTER_FIXED = Quantity(
        type=str,
        shape=[],
        description='''
        Specify if the center of the sphere should be fixed or allowed to move
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCRF_SPHERE_CENTER_WEIGHT_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the weight used to define the center of the sphere (if ATOM_LIST is
        provided)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCRF_SPHERE_CENTER_XYZ = Quantity(
        type=str,
        shape=[],
        description='''
        Coordinates of the center of the sphere
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_SCRF_SPHERE(MSection):
    '''
    Treats the implicit solvent environment like a sphere
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_SCRF_SPHERE_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Value of the spherical cavity in the dielectric medium
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_SCRF_SPHERE_CENTER = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_SCRF_SPHERE_CENTER'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT_SCRF(MSection):
    '''
    Adds an implicit solvation model to the DFT calculation. Know also as Self Consistent
    Reaction Field.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_SCRF_EPS_OUT = Quantity(
        type=str,
        shape=[],
        description='''
        Value of the dielectric constant outside the sphere
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SCRF_LMAX = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum value of L used in the multipole expansion
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_SCRF_PROGRAM_RUN_INFO = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_SCRF_PROGRAM_RUN_INFO'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_SCRF_SPHERE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_SCRF_SPHERE'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT_SIC(MSection):
    '''
    parameters for the self interaction correction
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_SIC_ORBITAL_SET = Quantity(
        type=str,
        shape=[],
        description='''
        Type of orbitals treated with the SIC
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SIC_SIC_METHOD = Quantity(
        type=str,
        shape=[],
        description='''
        Method used to remove the self interaction
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SIC_SIC_SCALING_A = Quantity(
        type=str,
        shape=[],
        description='''
        Scaling of the coulomb term in sic [experimental]
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SIC_SIC_SCALING_B = Quantity(
        type=str,
        shape=[],
        description='''
        Scaling of the xc term in sic [experimental]
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_SIC(MSection):
    '''
    parameters for the self interaction correction
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_SIC_ORBITAL_SET = Quantity(
        type=str,
        shape=[],
        description='''
        Type of orbitals treated with the SIC
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_SIC_SIC_METHOD = Quantity(
        type=str,
        shape=[],
        description='''
        Method used to remove the self interaction
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_SIC_SIC_SCALING_A = Quantity(
        type=str,
        shape=[],
        description='''
        Scaling of the coulomb term in sic [experimental]
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_SIC_SIC_SCALING_B = Quantity(
        type=str,
        shape=[],
        description='''
        Scaling of the xc term in sic [experimental]
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_ADIABATIC_RESCALING(MSection):
    '''
    Parameters for self interation corrected hybrid functionals
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_ADIABATIC_RESCALING_FUNCTIONAL_MODEL = Quantity(
        type=str,
        shape=[],
        description='''
        Which model for the coupling constant integration should be used.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_ADIABATIC_RESCALING_FUNCTIONAL_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Which Hybrid functional should be used. (Has to be consistent with the definitions
        in XC and HF).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_ADIABATIC_RESCALING_LAMBDA = Quantity(
        type=str,
        shape=[],
        description='''
        The point to be used along the adiabatic curve (0 &#60; λ &#60; 1)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_ADIABATIC_RESCALING_OMEGA = Quantity(
        type=str,
        shape=[],
        description='''
        Long-range parameter
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_HF_INFO(MSection):
    '''
    Controls the printing basic info about hf method
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_HF_INFO_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_HF_INFO_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_HF_INFO_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_HF_INFO_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_HF_INFO_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_HFX_RI(MSection):
    '''
    All parameters needed in a HFX RI calculation
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_HFX_RI_EPS_OPTIMIZATION = Quantity(
        type=str,
        shape=[],
        description='''
        Accuracy of iterative RI fit
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_HFX_RI_EPS_SCREENING = Quantity(
        type=str,
        shape=[],
        description='''
        Accuracy of geminal integral evaluation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_HFX_RI_MAX_ITER = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum number of iteration in RI fit
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_INTERACTION_POTENTIAL(MSection):
    '''
    Sets up interaction potential if requested
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_INTERACTION_POTENTIAL_CUTOFF_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Determines cutoff radius for the truncated 1/r potential. Only valid when doing
        truncated calculation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_INTERACTION_POTENTIAL_OMEGA = Quantity(
        type=str,
        shape=[],
        description='''
        Parameter for short/longrange interaction
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_INTERACTION_POTENTIAL_POTENTIAL_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Which interaction potential should be used (Coulomb, longrange or shortrange).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_INTERACTION_POTENTIAL_SCALE_COULOMB = Quantity(
        type=str,
        shape=[],
        description='''
        Scales Hartree-Fock contribution arising from a coulomb potential. Only valid when
        doing a mixed potential calculation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_INTERACTION_POTENTIAL_SCALE_GAUSSIAN = Quantity(
        type=str,
        shape=[],
        description='''
        Scales Hartree-Fock contribution arising from a gaussian potential. Only valid
        when doing a mixed potential calculation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_INTERACTION_POTENTIAL_SCALE_LONGRANGE = Quantity(
        type=str,
        shape=[],
        description='''
        Scales Hartree-Fock contribution arising from a longrange potential. Only valid
        when doing a mixed potential calculation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_INTERACTION_POTENTIAL_T_C_G_DATA = Quantity(
        type=str,
        shape=[],
        description='''
        Location of the file t_c_g.dat that contains the data for the evaluation of the
        truncated gamma function
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_LOAD_BALANCE(MSection):
    '''
    Parameters influencing the load balancing of the HF
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_LOAD_BALANCE_BLOCK_SIZE = Quantity(
        type=str,
        shape=[],
        description='''
        Determines the blocking used for the atomic quartet loops. A proper choice can
        speedup the calculation. The default (-1) is automatic.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_LOAD_BALANCE_NBINS = Quantity(
        type=str,
        shape=[],
        description='''
        Number of bins per process used to group atom quartets.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_LOAD_BALANCE_RANDOMIZE = Quantity(
        type=str,
        shape=[],
        description='''
        This flag controls the randomization of the bin assignment to processes. For
        highly ordered input structures with a bad load balance, setting this flag to TRUE
        might improve.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_MEMORY(MSection):
    '''
    Sets up memory parameters for the storage of the ERI's if requested
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_MEMORY_EPS_STORAGE_SCALING = Quantity(
        type=str,
        shape=[],
        description='''
        Scaling factor to scale eps_schwarz. Storage threshold for compression will be
        EPS_SCHWARZ*EPS_STORAGE_SCALING.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_MEMORY_MAX_DISK_SPACE = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the maximum amount of disk space [MB] used to store precomputed compressed
        four-center integrals. If 0, nothing is stored to disk
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_MEMORY_MAX_MEMORY = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the maximum amount of memory [MB] to be consumed by the full HFX module.
        All temporary buffers and helper arrays are subtracted from this number. What
        remains will be used for storage of integrals. NOTE: This number is assumed to
        represent the memory available to one MPI process. When running a threaded
        version, cp2k automatically takes care of distributing the memory among all the
        threads within a process.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_MEMORY_STORAGE_LOCATION = Quantity(
        type=str,
        shape=[],
        description='''
        Loaction where ERI's are stored if MAX_DISK_SPACE /=0 Expects a path to a
        directory.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_MEMORY_TREAT_FORCES_IN_CORE = Quantity(
        type=str,
        shape=[],
        description='''
        Determines whether the derivative ERI's should be stored to RAM or not. Only
        meaningful when performing Ehrenfest MD. Memory usage is defined via MAX_MEMORY,
        i.e. the memory is shared wit the energy ERI's.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_PERIODIC(MSection):
    '''
    Sets up periodic boundary condition parameters if requested
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_PERIODIC_NUMBER_OF_SHELLS = Quantity(
        type=str,
        shape=[],
        description='''
        Number of shells taken into account for periodicity. By default, cp2k tries to
        automatically evaluate this number. This algorithm might be to conservative,
        resulting in some overhead. You can try to adjust this number in order to make a
        calculation cheaper.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_SCREENING(MSection):
    '''
    Sets up screening parameters if requested
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_SCREENING_EPS_SCHWARZ_FORCES = Quantity(
        type=str,
        shape=[],
        description='''
        Screens the near field part of the electronic repulsion integrals using the
        Schwarz inequality for the given threshold. This will be approximately the
        accuracy of the forces,  and should normally be similar to EPS_SCF
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_SCREENING_EPS_SCHWARZ = Quantity(
        type=str,
        shape=[],
        description='''
        Screens the near field part of the electronic repulsion integrals using the
        Schwarz inequality for the given threshold.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_SCREENING_P_SCREEN_CORRECTION_FACTOR = Quantity(
        type=str,
        shape=[],
        description='''
        Recalculates integrals on the fly if the actual density matrix is larger by a
        given factor than the initial one. If the factor is set to 0.0_dp, this feature is
        disabled.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_SCREENING_SCREEN_ON_INITIAL_P = Quantity(
        type=str,
        shape=[],
        description='''
        Screen on an initial density matrix. For the first MD step this matrix must be
        provided by a Restart File.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_SCREENING_SCREEN_P_FORCES = Quantity(
        type=str,
        shape=[],
        description='''
        Screens the electronic repulsion integrals for the forces using the density
        matrix. This results in a significant speedup for large systems, but might require
        a somewhat tigher EPS_SCHWARZ_FORCES.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_HF(MSection):
    '''
    Sets up the Hartree-Fock parameters if requested
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_FRACTION = Quantity(
        type=str,
        shape=[],
        description='''
        The fraction of Hartree-Fock to add to the total energy. 1.0 implies standard
        Hartree-Fock if used with XC_FUNCTIONAL NONE. NOTE: In a mixed potential
        calculation this should be set to 1.0, otherwise all parts are multiplied with
        this factor.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_PW_HFX_BLOCKSIZE = Quantity(
        type=str,
        shape=[],
        description='''
        Improve the performance of pw_hfx at the cost of some additional memory by storing
        the realspace representation of PW_HFX_BLOCKSIZE states.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_PW_HFX = Quantity(
        type=str,
        shape=[],
        description='''
        Compute the Hartree-Fock energy also in the plane wave basis.The value is ignored,
        and intended for debugging only.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_TREAT_LSD_IN_CORE = Quantity(
        type=str,
        shape=[],
        description='''
        Determines how spin denisities are taken into account. If true, the beta spin
        density is included via a second in core call. If false, alpha and beta spins are
        done in one shot
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_HF_INFO = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_HF_INFO'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_HFX_RI = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_HFX_RI'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_INTERACTION_POTENTIAL = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_INTERACTION_POTENTIAL'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_LOAD_BALANCE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_LOAD_BALANCE'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_MEMORY = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_MEMORY'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_PERIODIC = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_PERIODIC'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_SCREENING = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_HF_SCREENING'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_VDW_POTENTIAL_NON_LOCAL(MSection):
    '''
    Information on the non local dispersion functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_VDW_POTENTIAL_NON_LOCAL_CUTOFF = Quantity(
        type=str,
        shape=[],
        description='''
        The cutoff of the FFT grid used in the calculation of the nonlocal vdW functional
        [Ry].
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_VDW_POTENTIAL_NON_LOCAL_KERNEL_FILE_NAME = Quantity(
        type=str,
        shape=[],
        description='''
        Name of the kernel data file, may include a path.vdW_kernel_table.dat is for DRSLL
        and LMKLL andrVV10_kernel_table.dat is for rVV10.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_VDW_POTENTIAL_NON_LOCAL_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Parameters b and C of the rVV10 functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_VDW_POTENTIAL_NON_LOCAL_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Type of functional (the corresponding kernel data file should be selected).Allows
        for common forms such as vdW-DF, vdW-DF2, optB88-vdW, rVV10.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_VDW_POTENTIAL_NON_LOCAL_VERBOSE_OUTPUT = Quantity(
        type=str,
        shape=[],
        description='''
        Extensive output for non local functionals
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_PRINT_DFTD(MSection):
    '''
    Controls the printing of some info about DFTD contributions
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_PRINT_DFTD_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_PRINT_DFTD_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_PRINT_DFTD_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_PRINT_DFTD_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_PRINT_DFTD_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_VDW_POTENTIAL_PAIR_POTENTIAL(MSection):
    '''
    Information on the pair potential to calculate dispersion
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_ATOM_COORDINATION_NUMBERS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the coordination number of a set of atoms for the C9 term in DFT-D3.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_ATOMPARM = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies parameters for atom types (in atomic units). If not provided default
        parameters are used (DFT-D2).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_CALCULATE_C9_TERM = Quantity(
        type=str,
        shape=[],
        description='''
        Calculate C9 terms in DFT-D3 model
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_D3_SCALING = Quantity(
        type=str,
        shape=[],
        description='''
        XC Functional dependent scaling parameters (s6,sr6,s8) for the DFT-D3 method, if
        set to zero CP2K attempts to guess the xc functional from REFERENCE_FUNCTIONAL and
        sets the associated scaling parameter.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_D3BJ_SCALING = Quantity(
        type=str,
        shape=[],
        description='''
        XC Functional dependent scaling parameters (s6,a1,s8,a2) for the DFT-D3(BJ)
        method, if set to zero CP2K attempts to guess the xc functional from
        REFERENCE_FUNCTIONAL and sets the associated scaling parameter.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_EPS_CN = Quantity(
        type=str,
        shape=[],
        description='''
        Cutoff value for coordination number function (DFT-D3 method)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_EXP_PRE = Quantity(
        type=str,
        shape=[],
        description='''
        Prefactor in exponential damping factor (DFT-D2 potential)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_KIND_COORDINATION_NUMBERS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the coordination number for a kind for the C9 term in DFT-D3.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_LONG_RANGE_CORRECTION = Quantity(
        type=str,
        shape=[],
        description='''
        Calculate a long range correction to the DFT-D3 model
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_PARAMETER_FILE_NAME = Quantity(
        type=str,
        shape=[],
        description='''
        Name of the parameter file, may include a path
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_R_CUTOFF = Quantity(
        type=str,
        shape=[],
        description='''
        Range of potential. The cutoff will be 2 times this value
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_REFERENCE_C9_TERM = Quantity(
        type=str,
        shape=[],
        description='''
        Calculate C9 terms in DFT-D3 model using reference coordination numbers
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_REFERENCE_FUNCTIONAL = Quantity(
        type=str,
        shape=[],
        description='''
        Use parameters for this specific density functional. For available D3 and D3(BJ)
        parameters see: http://www.thch.uni-bonn.de/tc/downloads/DFT-D3/functionals.html,
        http://www.thch.uni-bonn.de/tc/downloads/DFT-D3/functionalsbj.html
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_SCALING = Quantity(
        type=str,
        shape=[],
        description='''
        XC Functional dependent scaling parameter, if set to zero CP2K attempts to guess
        the xc functional that is in use and sets the associated scaling parameter.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Type of potential
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_VERBOSE_OUTPUT = Quantity(
        type=str,
        shape=[],
        description='''
        Extensive output for the DFT-D2 and DFT-D3 models
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_PRINT_DFTD = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_PRINT_DFTD'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_VDW_POTENTIAL(MSection):
    '''
    This section combines all possible additional dispersion corrections to the normal XC
    functionals. This can be more functionals or simple empirical pair potentials.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_VDW_POTENTIAL_POTENTIAL_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Type of dispersion/vdW functional or potential to use
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_VDW_POTENTIAL_NON_LOCAL = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_VDW_POTENTIAL_NON_LOCAL'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_VDW_POTENTIAL_PAIR_POTENTIAL = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_VDW_POTENTIAL_PAIR_POTENTIAL'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_CPHF(MSection):
    '''
    Parameters influencing the solution of the Z-vector equations in MP2 gradients
    calculations.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_CPHF_EPS_CONV = Quantity(
        type=str,
        shape=[],
        description='''
        Convergence threshold for the solution of the Z-vector equations. The Z-vector
        equations have the form of a linear system of equations Ax=b, convergence is
        achieved when |Ax-b|<=EPS_CONV.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_CPHF_MAX_ITER = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum number of iterations allowed for the solution of the Z-vector equations.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_DIRECT_CANONICAL(MSection):
    '''
    Parameters influencing the direct canonical method
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_DIRECT_CANONICAL_BIG_SEND = Quantity(
        type=str,
        shape=[],
        description='''
        Send big messages between processes (useful for >48 processors).
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_INTERACTION_POTENTIAL(MSection):
    '''
    Parameters the interaction potential in computing the biel integrals
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_INTERACTION_POTENTIAL_POTENTIAL_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Which interaction potential should be used (Coulomb, TShPSC operator).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_INTERACTION_POTENTIAL_TRUNCATION_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Determines truncation radius for the truncated TShPSC potential. Only valid when
        doing truncated calculation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_INTERACTION_POTENTIAL_TSHPSC_DATA = Quantity(
        type=str,
        shape=[],
        description='''
        Location of the file TShPSC.dat that contains the data for the evaluation of the
        TShPSC G0
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_MP2_INFO(MSection):
    '''
    Controls the printing basic info about MP2 method
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_MP2_INFO_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_MP2_INFO_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_MP2_INFO_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_MP2_INFO_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_MP2_INFO_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_OPT_RI_BASIS(MSection):
    '''
    Parameters influencing the optimization of the RI MP2 basis. Only exponents of non-
    contracted auxiliary basis can be optimized. An initial RI auxiliary basis has to be
    specified.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_OPT_RI_BASIS_BASIS_SIZE = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the size of the auxiliary basis set automatically generated as initial
        guess. This will be effective only if RI_AUX_BASIS_SET in the KIND section and
        NUM_FUNC are not specified.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_OPT_RI_BASIS_DELTA_I_REL = Quantity(
        type=str,
        shape=[],
        description='''
        Target accuracy in the relative deviation of the amplitudes calculated with and
        without RI approximation, (more details in Chem.Phys.Lett.294(1998)143).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_OPT_RI_BASIS_DELTA_RI = Quantity(
        type=str,
        shape=[],
        description='''
        Target accuracy in the absolute difference between the RI-MP2 and the exact MP2
        energy, DRI=ABS(E_MP2-E_RI-MP2).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_OPT_RI_BASIS_EPS_DERIV = Quantity(
        type=str,
        shape=[],
        description='''
        The derivatives of the MP2 energy with respect to the exponents of the basis are
        calculated numerically. The change in the exponent a_i employed for the numerical
        evaluation is defined as h_i=EPS_DERIV*a_i.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_OPT_RI_BASIS_MAX_ITER = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the maximum number of steps in the RI basis optimization.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_OPT_RI_BASIS_NUM_FUNC = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the number of function, for each angular momentum (s, p, d ...),
        employed in the automatically generated initial guess. This will be effective only
        if RI_AUX_BASIS_SET in the KIND section is not specified.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_LAPLACE(MSection):
    '''
    Parameters influencing the RI-SOS-MP2-Laplace method
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_LAPLACE_QUADRATURE_POINTS = Quantity(
        type=str,
        shape=[],
        description='''
        Number of quadrature points for the numerical integration in the RI-SOS-
        MP2-Laplace method.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_LAPLACE_SIZE_INTEG_GROUP = Quantity(
        type=str,
        shape=[],
        description='''
        Group size for the integration in the Laplace method, that is the number of
        processes involved in the computation of each integration point. SIZE_INTEG_GROUP
        has to be a multiple of GROUP_SIZE in the WF_CORRELATION section. The default (-1)
        is automatic.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_MP2(MSection):
    '''
    Parameters influencing the RI MP2 method
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_MP2_BLOCK_SIZE = Quantity(
        type=str,
        shape=[],
        description='''
        Determines the blocking used for communication in RI-MP2. Larger BLOCK_SIZE
        reduces communication but requires more memory. The default (-1) is automatic.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_MP2_EPS_CANONICAL = Quantity(
        type=str,
        shape=[],
        description='''
        Threshold for discriminate if a given ij pairs of the unrelaxed MP2 density matrix
        has to be calculated with a canonical reformulation based on the occupied
        eigenvalues differences.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_MP2_FREE_HFX_BUFFER = Quantity(
        type=str,
        shape=[],
        description='''
        Free the buffer containing the 4 center integrals used in the Hartree-Fock
        exchange calculation. This will be effective only for gradients calculations,
        since for the energy only case, the buffers are released by default. (Right now
        debugging only).
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_HF_INFO(MSection):
    '''
    Controls the printing basic info about hf method
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_HF_INFO_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_HF_INFO_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_HF_INFO_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_HF_INFO_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_HF_INFO_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_HFX_RI(MSection):
    '''
    All parameters needed in a HFX RI calculation
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_HFX_RI_EPS_OPTIMIZATION = Quantity(
        type=str,
        shape=[],
        description='''
        Accuracy of iterative RI fit
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_HFX_RI_EPS_SCREENING = Quantity(
        type=str,
        shape=[],
        description='''
        Accuracy of geminal integral evaluation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_HFX_RI_MAX_ITER = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum number of iteration in RI fit
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_INTERACTION_POTENTIAL(MSection):
    '''
    Sets up interaction potential if requested
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_INTERACTION_POTENTIAL_CUTOFF_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Determines cutoff radius for the truncated 1/r potential. Only valid when doing
        truncated calculation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_INTERACTION_POTENTIAL_OMEGA = Quantity(
        type=str,
        shape=[],
        description='''
        Parameter for short/longrange interaction
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_INTERACTION_POTENTIAL_POTENTIAL_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Which interaction potential should be used (Coulomb, longrange or shortrange).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_INTERACTION_POTENTIAL_SCALE_COULOMB = Quantity(
        type=str,
        shape=[],
        description='''
        Scales Hartree-Fock contribution arising from a coulomb potential. Only valid when
        doing a mixed potential calculation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_INTERACTION_POTENTIAL_SCALE_GAUSSIAN = Quantity(
        type=str,
        shape=[],
        description='''
        Scales Hartree-Fock contribution arising from a gaussian potential. Only valid
        when doing a mixed potential calculation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_INTERACTION_POTENTIAL_SCALE_LONGRANGE = Quantity(
        type=str,
        shape=[],
        description='''
        Scales Hartree-Fock contribution arising from a longrange potential. Only valid
        when doing a mixed potential calculation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_INTERACTION_POTENTIAL_T_C_G_DATA = Quantity(
        type=str,
        shape=[],
        description='''
        Location of the file t_c_g.dat that contains the data for the evaluation of the
        truncated gamma function
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_LOAD_BALANCE(MSection):
    '''
    Parameters influencing the load balancing of the HF
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_LOAD_BALANCE_BLOCK_SIZE = Quantity(
        type=str,
        shape=[],
        description='''
        Determines the blocking used for the atomic quartet loops. A proper choice can
        speedup the calculation. The default (-1) is automatic.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_LOAD_BALANCE_NBINS = Quantity(
        type=str,
        shape=[],
        description='''
        Number of bins per process used to group atom quartets.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_LOAD_BALANCE_RANDOMIZE = Quantity(
        type=str,
        shape=[],
        description='''
        This flag controls the randomization of the bin assignment to processes. For
        highly ordered input structures with a bad load balance, setting this flag to TRUE
        might improve.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_MEMORY(MSection):
    '''
    Sets up memory parameters for the storage of the ERI's if requested
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_MEMORY_EPS_STORAGE_SCALING = Quantity(
        type=str,
        shape=[],
        description='''
        Scaling factor to scale eps_schwarz. Storage threshold for compression will be
        EPS_SCHWARZ*EPS_STORAGE_SCALING.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_MEMORY_MAX_DISK_SPACE = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the maximum amount of disk space [MB] used to store precomputed compressed
        four-center integrals. If 0, nothing is stored to disk
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_MEMORY_MAX_MEMORY = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the maximum amount of memory [MB] to be consumed by the full HFX module.
        All temporary buffers and helper arrays are subtracted from this number. What
        remains will be used for storage of integrals. NOTE: This number is assumed to
        represent the memory available to one MPI process. When running a threaded
        version, cp2k automatically takes care of distributing the memory among all the
        threads within a process.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_MEMORY_STORAGE_LOCATION = Quantity(
        type=str,
        shape=[],
        description='''
        Loaction where ERI's are stored if MAX_DISK_SPACE /=0 Expects a path to a
        directory.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_MEMORY_TREAT_FORCES_IN_CORE = Quantity(
        type=str,
        shape=[],
        description='''
        Determines whether the derivative ERI's should be stored to RAM or not. Only
        meaningful when performing Ehrenfest MD. Memory usage is defined via MAX_MEMORY,
        i.e. the memory is shared wit the energy ERI's.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_PERIODIC(MSection):
    '''
    Sets up periodic boundary condition parameters if requested
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_PERIODIC_NUMBER_OF_SHELLS = Quantity(
        type=str,
        shape=[],
        description='''
        Number of shells taken into account for periodicity. By default, cp2k tries to
        automatically evaluate this number. This algorithm might be to conservative,
        resulting in some overhead. You can try to adjust this number in order to make a
        calculation cheaper.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_SCREENING(MSection):
    '''
    Sets up screening parameters if requested
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_SCREENING_EPS_SCHWARZ_FORCES = Quantity(
        type=str,
        shape=[],
        description='''
        Screens the near field part of the electronic repulsion integrals using the
        Schwarz inequality for the given threshold. This will be approximately the
        accuracy of the forces,  and should normally be similar to EPS_SCF
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_SCREENING_EPS_SCHWARZ = Quantity(
        type=str,
        shape=[],
        description='''
        Screens the near field part of the electronic repulsion integrals using the
        Schwarz inequality for the given threshold.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_SCREENING_P_SCREEN_CORRECTION_FACTOR = Quantity(
        type=str,
        shape=[],
        description='''
        Recalculates integrals on the fly if the actual density matrix is larger by a
        given factor than the initial one. If the factor is set to 0.0_dp, this feature is
        disabled.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_SCREENING_SCREEN_ON_INITIAL_P = Quantity(
        type=str,
        shape=[],
        description='''
        Screen on an initial density matrix. For the first MD step this matrix must be
        provided by a Restart File.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_SCREENING_SCREEN_P_FORCES = Quantity(
        type=str,
        shape=[],
        description='''
        Screens the electronic repulsion integrals for the forces using the density
        matrix. This results in a significant speedup for large systems, but might require
        a somewhat tigher EPS_SCHWARZ_FORCES.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF(MSection):
    '''
    Sets up the Hartree-Fock parameters if requested
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_FRACTION = Quantity(
        type=str,
        shape=[],
        description='''
        The fraction of Hartree-Fock to add to the total energy. 1.0 implies standard
        Hartree-Fock if used with XC_FUNCTIONAL NONE. NOTE: In a mixed potential
        calculation this should be set to 1.0, otherwise all parts are multiplied with
        this factor.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_PW_HFX_BLOCKSIZE = Quantity(
        type=str,
        shape=[],
        description='''
        Improve the performance of pw_hfx at the cost of some additional memory by storing
        the realspace representation of PW_HFX_BLOCKSIZE states.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_PW_HFX = Quantity(
        type=str,
        shape=[],
        description='''
        Compute the Hartree-Fock energy also in the plane wave basis.The value is ignored,
        and intended for debugging only.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_TREAT_LSD_IN_CORE = Quantity(
        type=str,
        shape=[],
        description='''
        Determines how spin denisities are taken into account. If true, the beta spin
        density is included via a second in core call. If false, alpha and beta spins are
        done in one shot
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_HF_INFO = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_HF_INFO'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_HFX_RI = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_HFX_RI'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_INTERACTION_POTENTIAL = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_INTERACTION_POTENTIAL'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_LOAD_BALANCE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_LOAD_BALANCE'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_MEMORY = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_MEMORY'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_PERIODIC = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_PERIODIC'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_SCREENING = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF_SCREENING'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA(MSection):
    '''
    Parameters influencing the RI RPA method
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_MINIMAX_QUADRATURE = Quantity(
        type=str,
        shape=[],
        description='''
        Use the Minimax quadrature scheme for performing the numerical integration.
        Maximum number of quadrature point limited to 20.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_MM_STYLE = Quantity(
        type=str,
        shape=[],
        description='''
        Matrix multiplication style for the Q matrix.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_QUADRATURE_POINTS = Quantity(
        type=str,
        shape=[],
        description='''
        Number of quadrature points for the numerical integration in the RI-RPA method.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_SIZE_FREQ_INTEG_GROUP = Quantity(
        type=str,
        shape=[],
        description='''
        Group size for frequency integration, that is the number of processes involved in
        the computation of each integration point. SIZE_FREQ_INTEG_GROUP has to be a
        multiple of GROUP_SIZE in the WF_CORRELATION section. The default (-1) is
        automatic.
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA_HF'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_WFC_GPW(MSection):
    '''
    Parameters for the GPW approach in Wavefunction-based Correlation methods
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_WFC_GPW_CUTOFF = Quantity(
        type=str,
        shape=[],
        description='''
        The cutoff of the finest grid level in the MP2 gpw integration.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_WFC_GPW_EPS_FILTER = Quantity(
        type=str,
        shape=[],
        description='''
        Determines a threshold for the DBCSR based multiply (usually 10 times smaller than
        EPS_GRID).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_WFC_GPW_EPS_GRID = Quantity(
        type=str,
        shape=[],
        description='''
        Determines a threshold for the GPW based integration
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_WFC_GPW_PRINT_LEVEL = Quantity(
        type=str,
        shape=[],
        description='''
        How much output is written by the individual groups.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_WFC_GPW_REL_CUTOFF = Quantity(
        type=str,
        shape=[],
        description='''
        Determines the grid at which a Gaussian is mapped.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION(MSection):
    '''
    Sets up the Wavefunction-based Correlation parameters if requested
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_CALC_COND_NUM = Quantity(
        type=str,
        shape=[],
        description='''
        Calculate the condition number of the (P|Q) matrix for the RI methods.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_COL_BLOCK = Quantity(
        type=str,
        shape=[],
        description='''
        Size of the column block used in the SCALAPACK block cyclic data
        distribution.Default is (COL_BLOCK=-1) is automatic. A proper choice can speedup
        the parallel matrix multiplication in the case of RI-RPA and RI-SOS-MP2-Laplace.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_GROUP_SIZE = Quantity(
        type=str,
        shape=[],
        description='''
        Group size used in the computation of the integrals. Default is to use all
        processors (GROUP_SIZE=-1).A smaller group size (for example the node size), might
        a better choice if the actual MP2 time is large compared to integral computation
        time. This is usually the case if the total number of processors is not too large.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_MEMORY = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum allowed total memory usage during MP2 methods [Mb].
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_METHOD = Quantity(
        type=str,
        shape=[],
        description='''
        Which method should be used to compute the MP2 energy
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_ROW_BLOCK = Quantity(
        type=str,
        shape=[],
        description='''
        Size of the row block used in the SCALAPACK block cyclic data distribution.Default
        is (ROW_BLOCK=-1) is automatic. A proper choice can speedup the parallel matrix
        multiplication in the case of RI-RPA and RI-SOS-MP2-Laplace.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_SCALE_S = Quantity(
        type=str,
        shape=[],
        description='''
        Scaling factor of the singlet energy component (opposite spin, OS).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_SCALE_T = Quantity(
        type=str,
        shape=[],
        description='''
        Scaling factor of the triplet energy component (same spin, SS).
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_CPHF = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_CPHF'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_DIRECT_CANONICAL = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_DIRECT_CANONICAL'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_INTERACTION_POTENTIAL = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_INTERACTION_POTENTIAL'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_MP2_INFO = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_MP2_INFO'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_OPT_RI_BASIS = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_OPT_RI_BASIS'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_LAPLACE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_LAPLACE'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_MP2 = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_MP2'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_RI_RPA'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_WFC_GPW = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION_WFC_GPW'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_BECKE88_LR_ADIABATIC(MSection):
    '''
    Uses the Becke 88 longrange exchange functional in an adiabatic fashion
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_BECKE88_LR_ADIABATIC_LAMBDA = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the parameter of the adiabatic curve
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_BECKE88_LR_ADIABATIC_OMEGA = Quantity(
        type=str,
        shape=[],
        description='''
        Potential parameter in erf(omega*r)/r
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_BECKE88_LR_ADIABATIC_SCALE_X = Quantity(
        type=str,
        shape=[],
        description='''
        scales the exchange part of the functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_BECKE88_LR_ADIABATIC_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_BECKE88_LR(MSection):
    '''
    Uses the Becke 88 longrange exchange functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_BECKE88_LR_OMEGA = Quantity(
        type=str,
        shape=[],
        description='''
        Potential parameter in erf(omega*r)/r
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_BECKE88_LR_SCALE_X = Quantity(
        type=str,
        shape=[],
        description='''
        scales the exchange part of the functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_BECKE88_LR_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_BECKE88(MSection):
    '''
    Uses the Becke 88 exchange functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_BECKE88_SCALE_X = Quantity(
        type=str,
        shape=[],
        description='''
        scales the exchange part of the functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_BECKE88_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_BECKE97(MSection):
    '''
    Uses the Becke 97 exchange correlation functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_BECKE97_PARAMETRIZATION = Quantity(
        type=str,
        shape=[],
        description='''
        switches between the B97 and Grimme parametrization
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_BECKE97_SCALE_C = Quantity(
        type=str,
        shape=[],
        description='''
        scales the correlation part of the functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_BECKE97_SCALE_X = Quantity(
        type=str,
        shape=[],
        description='''
        scales the exchange part of the functional, if -1 the default for the given
        parametrization is used
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_BECKE97_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_BECKE_ROUSSEL(MSection):
    '''
    Becke Roussel exchange hole model. Can be usedas long range correction with a
    truncated coulomb potential
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_BECKE_ROUSSEL_CUTOFF_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the cutoff radius for the truncation. If put to zero, the standard full
        range potential will be used
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_BECKE_ROUSSEL_GAMMA = Quantity(
        type=str,
        shape=[],
        description='''
        Parameter in the exchange hole. Usually this is put to 1.0 or 0.8
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_BECKE_ROUSSEL_SCALE_X = Quantity(
        type=str,
        shape=[],
        description='''
        scales the exchange part of the functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_BECKE_ROUSSEL_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_BEEF(MSection):
    '''
    Uses the BEEFvdW exchange functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_BEEF_SCALE_X = Quantity(
        type=str,
        shape=[],
        description='''
        scales the exchange part of the functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_BEEF_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_CS1(MSection):
    '''
    Uses the CS1 functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_CS1_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_GV09(MSection):
    '''
    Combination of three different exchange hole models
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_GV09_CUTOFF_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines cutoff for lower integration boundary
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_GV09_GAMMA = Quantity(
        type=str,
        shape=[],
        description='''
        Parameter for Becke Roussel hole
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_GV09_SCALE_X = Quantity(
        type=str,
        shape=[],
        description='''
        scales the exchange part of the functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_GV09_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_HCTH(MSection):
    '''
    Uses the HCTH class of functionals
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_HCTH_PARAMETER_SET = Quantity(
        type=str,
        shape=[],
        description='''
        Which version of the parameters should be used
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_HCTH_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_KE_GGA(MSection):
    '''
    Uses one of the KE_GGA functionals (optimized versions of some of these functionals
    might be available outside this section). These functionals are needed for the
    computation of the kinetic energy in the Kim-Gordon method.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_KE_GGA_FUNCTIONAL = Quantity(
        type=str,
        shape=[],
        description='''
        Which one of the KE_GGA functionals should be used
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_KE_GGA_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_KE_LIBXC(MSection):
    '''
    To be used for KG runs. Uses kinetic energy functionals from LIBXC, see also
    http://www.tddft.org/programs/octopus/wiki/index.php/Libxc
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_KE_LIBXC_FUNCTIONAL = Quantity(
        type=str,
        shape=[],
        description='''
        names of the functionals, see also
        http://www.tddft.org/programs/octopus/wiki/index.php/Libxc:manual .The precise
        list of available functionals depends on the version of libxc interfaced
        (currently 2.0.1).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_KE_LIBXC_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        parameters of the functionals
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_KE_LIBXC_SCALE = Quantity(
        type=str,
        shape=[],
        description='''
        scaling factors of the functionals
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_KE_LIBXC_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_LDA_HOLE_T_C_LR(MSection):
    '''
    LDA exchange hole model in truncated coulomb potential
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_LDA_HOLE_T_C_LR_CUTOFF_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines cutoff for lower integration boundary
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_LDA_HOLE_T_C_LR_SCALE_X = Quantity(
        type=str,
        shape=[],
        description='''
        scales the exchange part of the functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_LDA_HOLE_T_C_LR_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_LIBXC(MSection):
    '''
    Uses functionals from LIBXC, see also
    http://www.tddft.org/programs/octopus/wiki/index.php/Libxc
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_LIBXC_FUNCTIONAL = Quantity(
        type=str,
        shape=[],
        description='''
        names of the functionals, see also
        http://www.tddft.org/programs/octopus/wiki/index.php/Libxc:manual .The precise
        list of available functionals depends on the version of libxc interfaced
        (currently 2.0.1).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_LIBXC_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        parameters of the functionals
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_LIBXC_SCALE = Quantity(
        type=str,
        shape=[],
        description='''
        scaling factors of the functionals
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_LIBXC_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_LYP_ADIABATIC(MSection):
    '''
    Uses the LYP correlation functional in an adiabatic fashion
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_LYP_ADIABATIC_LAMBDA = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the parameter of the adiabatic curve.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_LYP_ADIABATIC_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_LYP(MSection):
    '''
    Uses the LYP functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_LYP_SCALE_C = Quantity(
        type=str,
        shape=[],
        description='''
        scales the correlation part of the functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_LYP_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_OPTX(MSection):
    '''
    Uses the OPTX functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_OPTX_SCALE_X = Quantity(
        type=str,
        shape=[],
        description='''
        scales the exchange part of the functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_OPTX_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_P86C(MSection):
    '''
    Uses the P86C functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_P86C_SCALE_C = Quantity(
        type=str,
        shape=[],
        description='''
        scales the correlation part of the functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_P86C_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_PADE(MSection):
    '''
    Uses the PADE functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_PADE_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_PBE_HOLE_T_C_LR(MSection):
    '''
    PBE exchange hole model in trucanted coulomb potential
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_PBE_HOLE_T_C_LR_CUTOFF_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines cutoff for lower integration boundary
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_PBE_HOLE_T_C_LR_SCALE_X = Quantity(
        type=str,
        shape=[],
        description='''
        scales the exchange part of the functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_PBE_HOLE_T_C_LR_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_PBE(MSection):
    '''
    Uses the PBE functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_PBE_PARAMETRIZATION = Quantity(
        type=str,
        shape=[],
        description='''
        switches between the different parametrizations of the functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_PBE_SCALE_C = Quantity(
        type=str,
        shape=[],
        description='''
        scales the correlation part of the functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_PBE_SCALE_X = Quantity(
        type=str,
        shape=[],
        description='''
        scales the exchange part of the functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_PBE_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_PW92(MSection):
    '''
    Uses the PerdewWang correlation functional.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_PW92_PARAMETRIZATION = Quantity(
        type=str,
        shape=[],
        description='''
        Which one of parametrizations should be used
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_PW92_SCALE = Quantity(
        type=str,
        shape=[],
        description='''
        Scaling of the energy functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_PW92_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_PZ81(MSection):
    '''
    Uses the PZ functional.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_PZ81_PARAMETRIZATION = Quantity(
        type=str,
        shape=[],
        description='''
        Which one of parametrizations should be used
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_PZ81_SCALE_C = Quantity(
        type=str,
        shape=[],
        description='''
        scales the correlation part of the functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_PZ81_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_TFW(MSection):
    '''
    Uses the TFW functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_TFW_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_TF(MSection):
    '''
    Uses the TF functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_TF_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_TPSS(MSection):
    '''
    Uses the TPSS functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_TPSS_SCALE_C = Quantity(
        type=str,
        shape=[],
        description='''
        scales the correlation part of the functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_TPSS_SCALE_X = Quantity(
        type=str,
        shape=[],
        description='''
        scales the exchange part of the functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_TPSS_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_VWN(MSection):
    '''
    Uses the VWN functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_VWN_FUNCTIONAL_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Which version of the VWN functional should be used
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_VWN_SCALE_C = Quantity(
        type=str,
        shape=[],
        description='''
        scales the correlation part of the functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_VWN_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_XALPHA(MSection):
    '''
    Uses the XALPHA (SLATER) functional.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_XALPHA_SCALE_X = Quantity(
        type=str,
        shape=[],
        description='''
        scales the exchange part of the functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_XALPHA_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_XALPHA_XA = Quantity(
        type=str,
        shape=[],
        description='''
        Value of the xa parameter (this does not change the exponent, just the mixing)
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_XGGA(MSection):
    '''
    Uses one of the XGGA functionals (optimized versions of some of these functionals
    might be available outside this section).
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_XGGA_FUNCTIONAL = Quantity(
        type=str,
        shape=[],
        description='''
        Which one of the XGGA functionals should be used
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_XGGA_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_XWPBE(MSection):
    '''
    Uses the short range PBE functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_XWPBE_OMEGA = Quantity(
        type=str,
        shape=[],
        description='''
        screening parameter
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_XWPBE_SCALE_X0 = Quantity(
        type=str,
        shape=[],
        description='''
        scales the exchange part of the original hole PBE-functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_XWPBE_SCALE_X = Quantity(
        type=str,
        shape=[],
        description='''
        scales the exchange part of the functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_XWPBE_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL(MSection):
    '''
    The xc functional to use
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Shortcut for the most common functional combinations.
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_BECKE88_LR_ADIABATIC = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_BECKE88_LR_ADIABATIC'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_BECKE88_LR = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_BECKE88_LR'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_BECKE88 = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_BECKE88'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_BECKE97 = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_BECKE97'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_BECKE_ROUSSEL = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_BECKE_ROUSSEL'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_BEEF = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_BEEF'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_CS1 = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_CS1'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_GV09 = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_GV09'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_HCTH = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_HCTH'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_KE_GGA = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_KE_GGA'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_KE_LIBXC = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_KE_LIBXC'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_LDA_HOLE_T_C_LR = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_LDA_HOLE_T_C_LR'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_LIBXC = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_LIBXC'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_LYP_ADIABATIC = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_LYP_ADIABATIC'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_LYP = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_LYP'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_OPTX = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_OPTX'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_P86C = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_P86C'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_PADE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_PADE'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_PBE_HOLE_T_C_LR = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_PBE_HOLE_T_C_LR'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_PBE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_PBE'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_PW92 = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_PW92'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_PZ81 = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_PZ81'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_TFW = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_TFW'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_TF = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_TF'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_TPSS = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_TPSS'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_VWN = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_VWN'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_XALPHA = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_XALPHA'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_XGGA = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_XGGA'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_XWPBE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL_XWPBE'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_GRID(MSection):
    '''
    The xc parameters used when calculating the xc on the grid
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_GRID_USE_FINER_GRID = Quantity(
        type=str,
        shape=[],
        description='''
        Uses a finer grid only to calculate the xc
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_GRID_XC_DERIV = Quantity(
        type=str,
        shape=[],
        description='''
        The method used to compute the derivatives
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_GRID_XC_SMOOTH_RHO = Quantity(
        type=str,
        shape=[],
        description='''
        The density smoothing used for the xc calculation
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_POTENTIAL_SAOP(MSection):
    '''
    Uses the SAOP potential
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_POTENTIAL_SAOP_ALPHA = Quantity(
        type=str,
        shape=[],
        description='''
        Value of the alpha parameter (default = 1.19).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_POTENTIAL_SAOP_BETA = Quantity(
        type=str,
        shape=[],
        description='''
        Value of the beta parameter (default = 0.01).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_POTENTIAL_SAOP_K_RHO = Quantity(
        type=str,
        shape=[],
        description='''
        Value of the K_rho parameter (default = 0.42).
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_POTENTIAL(MSection):
    '''
    The xc potential to use (CAREFUL: xc potential here refers to potentials that are not
    derived from an xc functional, but rather are modelled directly. Therefore there is no
    consistent xc energy available. To still get an energy expression, see ENERGY below
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_POTENTIAL_ENERGY = Quantity(
        type=str,
        shape=[],
        description='''
        How to determine the total energy.
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_POTENTIAL_SAOP = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_POTENTIAL_SAOP'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC(MSection):
    '''
    parameters needed calculate the xc potential
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_DENSITY_CUTOFF = Quantity(
        type=str,
        shape=[],
        description='''
        The cutoff on the density used by the xc calculation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_DENSITY_SMOOTH_CUTOFF_RANGE = Quantity(
        type=str,
        shape=[],
        description='''
        Parameter for the smoothing procedure inxc calculation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_FUNCTIONAL_ROUTINE = Quantity(
        type=str,
        shape=[],
        description='''
        Select the code for xc calculation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_GRADIENT_CUTOFF = Quantity(
        type=str,
        shape=[],
        description='''
        The cutoff on the gradient of the density used by the xc calculation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_XC_TAU_CUTOFF = Quantity(
        type=str,
        shape=[],
        description='''
        The cutoff on tau used by the xc calculation
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_ADIABATIC_RESCALING = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_ADIABATIC_RESCALING'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_HF = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_HF'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_VDW_POTENTIAL = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_VDW_POTENTIAL'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_WF_CORRELATION'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_FUNCTIONAL'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_GRID = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_GRID'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_POTENTIAL = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC_XC_POTENTIAL'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT(MSection):
    '''
    parameters needed to set up the Time Dependent Density Functional PT
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_CONVERGENCE = Quantity(
        type=str,
        shape=[],
        description='''
        The convergence of the eigenvalues
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_DIAG_METHOD = Quantity(
        type=str,
        shape=[],
        description='''
        Diagonalization method used in tddfpt
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_INVERT_S = Quantity(
        type=str,
        shape=[],
        description='''
        use the inverse of the overlap matrix
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_KERNEL = Quantity(
        type=str,
        shape=[],
        description='''
        compute the kernel (debug purpose only)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_LSD_SINGLETS = Quantity(
        type=str,
        shape=[],
        description='''
        compute singlets using lsd vxc kernel
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_MAX_KV = Quantity(
        type=str,
        shape=[],
        description='''
        maximal number of Krylov space vectors
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_NEV = Quantity(
        type=str,
        shape=[],
        description='''
        number of excitations to calculate
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_NLUMO = Quantity(
        type=str,
        shape=[],
        description='''
        number of additional unoccupied orbitals
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_NREORTHO = Quantity(
        type=str,
        shape=[],
        description='''
        number of reorthogonalization steps
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_OE_CORR = Quantity(
        type=str,
        shape=[],
        description='''
        Which type of orbital eigenvalue correction to use\\n(to yield better HOMO-LUMO
        energies)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_PRECONDITIONER = Quantity(
        type=str,
        shape=[],
        description='''
        use the preconditioner (only for Davidson)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_RES_ETYPE = Quantity(
        type=str,
        shape=[],
        description='''
        (singlets/triplets) for restricted calculation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TDDFPT_RESTARTS = Quantity(
        type=str,
        shape=[],
        description='''
        maximal number subspace search restarts
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_SIC = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_SIC'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT_XC'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT_TRANSPORT(MSection):
    '''
    Specifies the parameters for transport, sets parameters for the OMEN code, see also
    http://www.nano-tcad.ethz.ch/
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_TRANSPORT_BANDWIDTH = Quantity(
        type=str,
        shape=[],
        description='''
        The number of neighboring unit cells that one unit cell interacts with.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TRANSPORT_COLZERO_THRESHOLD = Quantity(
        type=str,
        shape=[],
        description='''
        The smallest number that is not zero in the full diagonalization part.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TRANSPORT_CORES_PER_NODE = Quantity(
        type=str,
        shape=[],
        description='''
        Number of cores per node.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TRANSPORT_DENSITY_MATRIX_CONSTRUCTION_METHOD = Quantity(
        type=str,
        shape=[],
        description='''
        Method used for constructing the density matrix
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TRANSPORT_ENERGY_INTERVAL = Quantity(
        type=str,
        shape=[],
        description='''
        Average distance for big intervals in energy vector.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TRANSPORT_EPS_DECAY = Quantity(
        type=str,
        shape=[],
        description='''
        The smallest imaginary part that a decaying eigenvalue may have not to be
        considered as propagating.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TRANSPORT_EPS_EIGVAL_DEGEN = Quantity(
        type=str,
        shape=[],
        description='''
        Filter for degenerate bands in the injection vector.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TRANSPORT_EPS_LIMIT = Quantity(
        type=str,
        shape=[],
        description='''
        The smallest eigenvalue that is kept.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TRANSPORT_EPS_MU = Quantity(
        type=str,
        shape=[],
        description='''
        Accuracy to which the Fermi level should be determined.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TRANSPORT_EPS_SINGULARITY_CURVATURES = Quantity(
        type=str,
        shape=[],
        description='''
        Filter for degenerate bands in the bandstructure.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TRANSPORT_MIN_INTERVAL = Quantity(
        type=str,
        shape=[],
        description='''
        Smallest enery distance in energy vector.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TRANSPORT_N_ABSCISSAE = Quantity(
        type=str,
        shape=[],
        description='''
        The number of abscissae per integration interval on the real axis.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TRANSPORT_N_CELLS = Quantity(
        type=str,
        shape=[],
        description='''
        The number of unit cells.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TRANSPORT_N_DOF = Quantity(
        type=str,
        shape=[],
        description='''
        Number of degrees of freedom for the contact unit cell.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TRANSPORT_N_KPOINTS = Quantity(
        type=str,
        shape=[],
        description='''
        The number of k points for determination of the singularities.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TRANSPORT_NUM_CONTACTS = Quantity(
        type=str,
        shape=[],
        description='''
        The number of contacts.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TRANSPORT_NUM_INTERVAL = Quantity(
        type=str,
        shape=[],
        description='''
        Max number of energy points per small interval.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TRANSPORT_TASKS_PER_POINT = Quantity(
        type=str,
        shape=[],
        description='''
        Number of tasks per energy point.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_TRANSPORT_TEMPERATURE = Quantity(
        type=str,
        shape=[],
        description='''
        Temperature.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XAS_LOCALIZE(MSection):
    '''
    Use one of the available methods to define the localization  and possibly to optimize
    it to a minimum or a maximum.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XAS_LOCALIZE_CRAZY_SCALE = Quantity(
        type=str,
        shape=[],
        description='''
        scale angles
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_LOCALIZE_CRAZY_USE_DIAG = Quantity(
        type=str,
        shape=[],
        description='''
        Use diagonalization (slow) or pade based calculation of matrix exponentials.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_LOCALIZE_ENERGY_RANGE = Quantity(
        type=str,
        shape=[],
        description='''
        Select the orbitals to be localized within the given energy range.This type of
        selection cannot be added on top of the selection through a LIST. It reads to
        reals that are lower and higher boundaries of the energy range.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_LOCALIZE_EPS_LOCALIZATION = Quantity(
        type=str,
        shape=[],
        description='''
        Tolerance used in the convergence criterium of the localization methods.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_LOCALIZE_EPS_OCCUPATION = Quantity(
        type=str,
        shape=[],
        description='''
        Tolerance in the occupation number to select only fully occupied orbitals for the
        rotation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_LOCALIZE_JACOBI_FALLBACK = Quantity(
        type=str,
        shape=[],
        description='''
        Use Jacobi method in case no convergence was achieved by using the crazy rotations
        method.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_LOCALIZE_LIST_UNOCCUPIED = Quantity(
        type=str,
        shape=[],
        description='''
        Indexes of the unoccupied states to be localized, up to now only valid in
        combination with GPW. This keyword has to be present if unoccupied states should
        be localized. This keyword can be repeated several times(useful if you have to
        specify many indexes).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_LOCALIZE_LIST = Quantity(
        type=str,
        shape=[],
        description='''
        Indexes of the occupied wfn to be localizedThis keyword can be repeated several
        times(useful if you have to specify many indexes).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_LOCALIZE_LOCHOMO_RESTART_FILE_NAME = Quantity(
        type=str,
        shape=[],
        description='''
        File name where to read the MOS fromwhich to restart the localization procedure
        for occupied states
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_LOCALIZE_LOCLUMO_RESTART_FILE_NAME = Quantity(
        type=str,
        shape=[],
        description='''
        File name where to read the MOS fromwhich to restart the localization procedure
        for unoccupied states
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_LOCALIZE_MAX_CRAZY_ANGLE = Quantity(
        type=str,
        shape=[],
        description='''
        Largest allowed angle for the crazy rotations algorithm (smaller is slower but
        more stable).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_LOCALIZE_MAX_ITER = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum number of iterations used for localization methods
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_LOCALIZE_METHOD = Quantity(
        type=str,
        shape=[],
        description='''
        Method of optimization if any
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_LOCALIZE_MIN_OR_MAX = Quantity(
        type=str,
        shape=[],
        description='''
        Requires the maximization of the spread of the wfn
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_LOCALIZE_OPERATOR = Quantity(
        type=str,
        shape=[],
        description='''
        Type of opertator which defines the spread functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_LOCALIZE_OUT_ITER_EACH = Quantity(
        type=str,
        shape=[],
        description='''
        Every how many iterations of the localization algorithm(Jacobi) the tolerance
        value is printed out
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_LOCALIZE_RESTART = Quantity(
        type=str,
        shape=[],
        description='''
        Restart the localization from a set of orbitals read from a localization restart
        file.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_LOCALIZE_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        controls the activation of the MOS localization procedure
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_LOCALIZE_STATES = Quantity(
        type=str,
        shape=[],
        description='''
        Which states to localize, LUMO up to now only available in GPW
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_LOCALIZE_USE_HISTORY = Quantity(
        type=str,
        shape=[],
        description='''
        Generate an improved initial guess based on a history of results, which is useful
        during MD.Will only work if the number of states to be localized remains constant.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_DAVIDSON(MSection):
    '''
    Settings for DAVIDSON
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_DAVIDSON_CONV_MOS_PERCENT = Quantity(
        type=str,
        shape=[],
        description='''
        Minimal percent of MOS that have to converge within the Davidson loop before the
        SCF iteration is completed and a new Hamiltonian is computed
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_DAVIDSON_ENERGY_GAP = Quantity(
        type=str,
        shape=[],
        description='''
        Should be an estimate for the energy gap [a.u.] (HOMO-LUMO) and is used in
        preconditioning, especially effective with the FULL_ALL preconditioner, in which
        case it should be an underestimate of the gap (0.001 doing normally fine). For the
        other preconditioners, making this value larger (0.2) will tame the preconditioner
        in case of poor initial guesses.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_DAVIDSON_FIRST_PREC = Quantity(
        type=str,
        shape=[],
        description='''
        First SCF iteration at which a Preconditioner is employed
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_DAVIDSON_NEW_PREC_EACH = Quantity(
        type=str,
        shape=[],
        description='''
        Number of SCF iterations after which a new Preconditioner is computed
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_DAVIDSON_PRECOND_SOLVER = Quantity(
        type=str,
        shape=[],
        description='''
        How the preconditioner is applied to the residual.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_DAVIDSON_PRECONDITIONER = Quantity(
        type=str,
        shape=[],
        description='''
        Type of preconditioner to be used with all minimization schemes.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_DAVIDSON_SPARSE_MOS = Quantity(
        type=str,
        shape=[],
        description='''
        Use MOS as sparse matrix and avoid as much as possible multiplications with full
        matrices
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_DIAG_SUB_SCF_MIXING(MSection):
    '''
    Define type and parameters for mixingprocedures to be applied to the density matrix.
    Normally, only one type of mixing method should be accepted. The mixing procedures
    activated by this section are only active for diagonalization methods, i.e. not with
    minimization methods based on OT.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_DIAG_SUB_SCF_MIXING_ALPHA = Quantity(
        type=str,
        shape=[],
        description='''
        Fraction of new density to be included
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_DIAG_SUB_SCF_MIXING_BETA = Quantity(
        type=str,
        shape=[],
        description='''
        Denominator parameter in Kerker damping introduced to suppress charge sloshing:
        rho_mix(g) =rho_in(g) + alpha*g^2/(g^2 + beta^2)*(rho_out(g)-rho_in(g))
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_DIAG_SUB_SCF_MIXING_BROY_W0 = Quantity(
        type=str,
        shape=[],
        description='''
        w0 parameter used in Broyden mixing
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_DIAG_SUB_SCF_MIXING_BROY_WMAX = Quantity(
        type=str,
        shape=[],
        description='''
        Settings for BROY_WMAX
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_DIAG_SUB_SCF_MIXING_BROY_WREF = Quantity(
        type=str,
        shape=[],
        description='''
        Settings for BROY_WREF
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_DIAG_SUB_SCF_MIXING_MAX_GVEC_EXP = Quantity(
        type=str,
        shape=[],
        description='''
        Restricts the G-space mixing to lower part of G-vector spectrum, up to a G0, by
        assigning the exponent of the Gaussian that can be represented by vectors smaller
        than G0 within a certain accuracy.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_DIAG_SUB_SCF_MIXING_MAX_STEP = Quantity(
        type=str,
        shape=[],
        description='''
        Upper bound for the magnitude of the unpredicted step size in the update by the
        multisecant mixing scheme
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_DIAG_SUB_SCF_MIXING_METHOD = Quantity(
        type=str,
        shape=[],
        description='''
        Mixing method to be applied
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_DIAG_SUB_SCF_MIXING_N_SIMPLE_MIX = Quantity(
        type=str,
        shape=[],
        description='''
        Number of kerker damping iterations before starting other mixing procedures
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_DIAG_SUB_SCF_MIXING_NBUFFER = Quantity(
        type=str,
        shape=[],
        description='''
        Number of previous steps stored for the actual mixing scheme
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_DIAG_SUB_SCF_MIXING_NMIXING = Quantity(
        type=str,
        shape=[],
        description='''
        Minimal number of density mixing (should be greater than 0),before starting DIIS
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_DIAG_SUB_SCF_MIXING_NSKIP = Quantity(
        type=str,
        shape=[],
        description='''
        Number of initial iteration for which the mixing is skipped
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_DIAG_SUB_SCF_MIXING_PULAY_ALPHA = Quantity(
        type=str,
        shape=[],
        description='''
        Fraction of new density to be added to the Pulay expansion
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_DIAG_SUB_SCF_MIXING_PULAY_BETA = Quantity(
        type=str,
        shape=[],
        description='''
        Fraction of residual contribution to be added to Pulay expansion
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_DIAG_SUB_SCF_MIXING_R_FACTOR = Quantity(
        type=str,
        shape=[],
        description='''
        Control factor for the magnitude of the unpredicted step size in the update by the
        multisecant mixing scheme
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_DIAG_SUB_SCF_MIXING_REGULARIZATION = Quantity(
        type=str,
        shape=[],
        description='''
        Regularization parameter to stabilize the inversion of the residual matrix {Yn^t
        Yn} in the multisecant mixing scheme (noise)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_DIAG_SUB_SCF_MIXING_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Controls the activation of the mixing procedure
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_DIAG_SUB_SCF(MSection):
    '''
    Activation of self-consistenf subspace refinement by diagonalization of H by adjusting
    the occupation but keeping the MOS unchanged.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_DIAG_SUB_SCF_EPS_ADAPT_SCF = Quantity(
        type=str,
        shape=[],
        description='''
        Required density matrix accuracy as compared to current SCF convergence
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_DIAG_SUB_SCF_EPS_ENE = Quantity(
        type=str,
        shape=[],
        description='''
        Required energy accuracy for convergence of subspace diagonalization
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_DIAG_SUB_SCF_EPS_SKIP_SUB_DIAG = Quantity(
        type=str,
        shape=[],
        description='''
        Level of convergence to be reached before starting the internal loop of subspace
        rotations. Above this threshold only the outer diagonalization method is used.  If
        negative the subspace rotation is started at the first iteration
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_DIAG_SUB_SCF_MAX_ITER = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum number of iterations for the SCF inner loop
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_DIAG_SUB_SCF_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        controls the activation of inner SCF loop to refine occupations in MOS subspace
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_DIAG_SUB_SCF_MIXING = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_DIAG_SUB_SCF_MIXING'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_KRYLOV(MSection):
    '''
    Settings for KRYLOV
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_KRYLOV_CHECK_MOS_CONV = Quantity(
        type=str,
        shape=[],
        description='''
        This requires to check the convergence of MOS also when standard diagonalization
        steps are performed, if the block krylov approach is active.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_KRYLOV_EPS_KRYLOV = Quantity(
        type=str,
        shape=[],
        description='''
        Convergence criterion for the MOs
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_KRYLOV_EPS_STD_DIAG = Quantity(
        type=str,
        shape=[],
        description='''
        Level of convergence to be reached before starting the Lanczos procedure. Above
        this threshold a standard diagonalization method is used.  If negative Lanczos is
        started at the first iteration
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_KRYLOV_NBLOCK = Quantity(
        type=str,
        shape=[],
        description='''
        Size of the block of vectors refined simultaneously by the Lanczos procedure
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_KRYLOV_NKRYLOV = Quantity(
        type=str,
        shape=[],
        description='''
        Dimension of the Krylov space used for the Lanczos refinement
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_OT(MSection):
    '''
    Sets the various options for the orbital transformation (OT) method. Default settings
    already provide an efficient, yet robust method. Most systems benefit from using the
    FULL_ALL preconditioner combined with a small value (0.001) of ENERGY_GAP.Well-behaved
    systems might benefit from using a DIIS minimizer.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_OT_ALGORITHM = Quantity(
        type=str,
        shape=[],
        description='''
        Algorithm to be used for OT
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_OT_BROYDEN_ADAPTIVE_SIGMA = Quantity(
        type=str,
        shape=[],
        description='''
        Enable adaptive curvature estimation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_OT_BROYDEN_BETA = Quantity(
        type=str,
        shape=[],
        description='''
        Underrelaxation for the broyden mixer
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_OT_BROYDEN_ENABLE_FLIP = Quantity(
        type=str,
        shape=[],
        description='''
        Ensure positive definite update
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_OT_BROYDEN_ETA = Quantity(
        type=str,
        shape=[],
        description='''
        Dampening of estimated energy curvature.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_OT_BROYDEN_FORGET_HISTORY = Quantity(
        type=str,
        shape=[],
        description='''
        Forget history on bad approximation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_OT_BROYDEN_GAMMA = Quantity(
        type=str,
        shape=[],
        description='''
        Backtracking parameter
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_OT_BROYDEN_OMEGA = Quantity(
        type=str,
        shape=[],
        description='''
        Growth limit of curvature.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_OT_BROYDEN_SIGMA_DECREASE = Quantity(
        type=str,
        shape=[],
        description='''
        Reduction of curvature on bad approximation.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_OT_BROYDEN_SIGMA_MIN = Quantity(
        type=str,
        shape=[],
        description='''
        Minimum adaptive curvature.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_OT_BROYDEN_SIGMA = Quantity(
        type=str,
        shape=[],
        description='''
        Curvature of energy functional.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_OT_ENERGIES = Quantity(
        type=str,
        shape=[],
        description='''
        Optimize orbital energies for use in Fermi-Dirac smearing (requires ROTATION and
        FD smearing to be active).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_OT_ENERGY_GAP = Quantity(
        type=str,
        shape=[],
        description='''
        Should be an estimate for the energy gap [a.u.] (HOMO-LUMO) and is used in
        preconditioning, especially effective with the FULL_ALL preconditioner, in which
        case it should be an underestimate of the gap (can be a small number, e.g. 0.002).
        FULL_SINGLE_INVERSE takes it as lower bound (values below 0.05 can cause stability
        issues). In general, heigher values will tame the preconditioner in case of poor
        initial guesses. A negative value will leave the choice to CP2K depending on type
        of preconditioner.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_OT_EPS_IRAC_FILTER_MATRIX = Quantity(
        type=str,
        shape=[],
        description='''
        Sets the threshold for filtering the matrices.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_OT_EPS_IRAC_QUICK_EXIT = Quantity(
        type=str,
        shape=[],
        description='''
        Only one extra refinement iteration is done when the norm is below this value.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_OT_EPS_IRAC_SWITCH = Quantity(
        type=str,
        shape=[],
        description='''
        The algorithm switches to the polynomial refinement when the norm is below this
        value.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_OT_EPS_IRAC = Quantity(
        type=str,
        shape=[],
        description='''
        Targeted accuracy during the refinement iteration.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_OT_EPS_TAYLOR = Quantity(
        type=str,
        shape=[],
        description='''
        Target accuracy of the taylor expansion for the matrix functions, should normally
        be kept as is.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_OT_GOLD_TARGET = Quantity(
        type=str,
        shape=[],
        description='''
        Target relative uncertainty in the location of the minimum for LINESEARCH GOLD
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_OT_IRAC_DEGREE = Quantity(
        type=str,
        shape=[],
        description='''
        The refinement polynomial degree (2, 3 or 4).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_OT_LINESEARCH = Quantity(
        type=str,
        shape=[],
        description='''
        1D line search algorithm to be used with the OT minimizer, in increasing order of
        robustness and cost. MINIMIZER CG combined with LINESEARCH GOLD should always find
        an electronic minimum.  Whereas the 2PNT minimizer is almost always OK, 3PNT might
        be needed for systems in which successive OT CG steps do not decrease the total
        energy.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_OT_MAX_IRAC = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum allowed refinement iteration.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_OT_MAX_TAYLOR = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum order of the Taylor expansion before diagonalisation is prefered, for
        large parallel runs a slightly higher order could sometimes result in a small
        speedup.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_OT_MINIMIZER = Quantity(
        type=str,
        shape=[],
        description='''
        Minimizer to be used with the OT method
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_OT_MIXED_PRECISION = Quantity(
        type=str,
        shape=[],
        description='''
        Uses a mixed precision algorithm.With a well behaved basis set (i.e. condition
        number less than 1/eps_sp)it provides double precision accuracy results and up to
        a 2 fold speedup for building and applying the preconditioner.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_OT_N_HISTORY_VEC = Quantity(
        type=str,
        shape=[],
        description='''
        Number of history vectors to be used with DIIS or BROYDEN
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_OT_NONDIAG_ENERGY_STRENGTH = Quantity(
        type=str,
        shape=[],
        description='''
        The prefactor for the non-diagonal energy penalty (FD smearing)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_OT_NONDIAG_ENERGY = Quantity(
        type=str,
        shape=[],
        description='''
        Add a non-diagonal energy penalty (FD smearing)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_OT_OCCUPATION_PRECONDITIONER = Quantity(
        type=str,
        shape=[],
        description='''
        Preconditioner with the occupation numbers (FD smearing)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_OT_ON_THE_FLY_LOC = Quantity(
        type=str,
        shape=[],
        description='''
        On the fly localization of the molecular orbitals. Can only be used with OT/IRAC.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_OT_ORTHO_IRAC = Quantity(
        type=str,
        shape=[],
        description='''
        The orthogonality method.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_OT_PRECOND_SOLVER = Quantity(
        type=str,
        shape=[],
        description='''
        How the preconditioner is applied to the residual.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_OT_PRECONDITIONER = Quantity(
        type=str,
        shape=[],
        description='''
        Type of preconditioner to be used with all minimization schemes. They differ in
        effectiveness, cost of construction, cost of application. Properly preconditioned
        minimization can be orders of magnitude faster than doing nothing.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_OT_ROTATION = Quantity(
        type=str,
        shape=[],
        description='''
        Introduce additional variables so that rotations of the occupied subspace are
        allowed as well, only needed for cases where the energy is not invariant under  a
        rotation of the occupied subspace such as non-singlet restricted calculations  or
        fractional occupations.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_OT_SAFE_DIIS = Quantity(
        type=str,
        shape=[],
        description='''
        Reject DIIS steps if they point away from the minimum, do SD in that case.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_OT_SCP_NDDO = Quantity(
        type=str,
        shape=[],
        description='''
        Introduce additional self-consistent polarization through response basis set =
        orbital basis set for NDDO.)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_OT_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        controls the activation of the ot method
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_OT_STEPSIZE = Quantity(
        type=str,
        shape=[],
        description='''
        Initial stepsize used for the line search, sometimes this parameter can be reduced
        to stablize DIIS or to improve the CG behavior in the first few steps. The optimal
        value depends on the quality of the preconditioner. A negative values leaves the
        choice to CP2K depending on the preconditioner.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION(MSection):
    '''
    Set up type and parameters for Kohn-Sham matrix diagonalization.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_ALGORITHM = Quantity(
        type=str,
        shape=[],
        description='''
        Algorithm to be used for diagonalization
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_EPS_ADAPT = Quantity(
        type=str,
        shape=[],
        description='''
        Required accuracy in iterative diagonalization as compared to current SCF
        convergence
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_EPS_ITER = Quantity(
        type=str,
        shape=[],
        description='''
        Required accuracy in iterative diagonalization
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_EPS_JACOBI = Quantity(
        type=str,
        shape=[],
        description='''
        Below this threshold value for the SCF convergence the pseudo-diagonalization
        method using Jacobi rotations is activated. This method is much faster than a real
        diagonalization and it is even speeding up while achieving full
        convergence.However, it needs a pre-converged wavefunction obtained by at least
        one real diagonalization which is further optimized while keeping the original
        eigenvalue spectrum. The MO eigenvalues are NOT updated. The method might be
        useful to speed up calculations for large systems e.g. using a semi-empirical
        method.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_JACOBI_THRESHOLD = Quantity(
        type=str,
        shape=[],
        description='''
        Controls the accuracy of the pseudo-diagonalization method using Jacobi rotations
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_MAX_ITER = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum number of iterations in iterative diagonalization
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        controls the activation of the diagonalization method
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_DAVIDSON = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_DAVIDSON'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_DIAG_SUB_SCF = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_DIAG_SUB_SCF'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_KRYLOV = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_KRYLOV'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_OT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION_OT'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT_XAS_SCF_MIXING(MSection):
    '''
    Define type and parameters for mixingprocedures to be applied to the density matrix.
    Normally, only one type of mixing method should be accepted. The mixing procedures
    activated by this section are only active for diagonalization methods, i.e. not with
    minimization methods based on OT.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_MIXING_ALPHA = Quantity(
        type=str,
        shape=[],
        description='''
        Fraction of new density to be included
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_MIXING_BETA = Quantity(
        type=str,
        shape=[],
        description='''
        Denominator parameter in Kerker damping introduced to suppress charge sloshing:
        rho_mix(g) =rho_in(g) + alpha*g^2/(g^2 + beta^2)*(rho_out(g)-rho_in(g))
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_MIXING_BROY_W0 = Quantity(
        type=str,
        shape=[],
        description='''
        w0 parameter used in Broyden mixing
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_MIXING_BROY_WMAX = Quantity(
        type=str,
        shape=[],
        description='''
        Settings for BROY_WMAX
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_MIXING_BROY_WREF = Quantity(
        type=str,
        shape=[],
        description='''
        Settings for BROY_WREF
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_MIXING_MAX_GVEC_EXP = Quantity(
        type=str,
        shape=[],
        description='''
        Restricts the G-space mixing to lower part of G-vector spectrum, up to a G0, by
        assigning the exponent of the Gaussian that can be represented by vectors smaller
        than G0 within a certain accuracy.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_MIXING_MAX_STEP = Quantity(
        type=str,
        shape=[],
        description='''
        Upper bound for the magnitude of the unpredicted step size in the update by the
        multisecant mixing scheme
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_MIXING_METHOD = Quantity(
        type=str,
        shape=[],
        description='''
        Mixing method to be applied
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_MIXING_N_SIMPLE_MIX = Quantity(
        type=str,
        shape=[],
        description='''
        Number of kerker damping iterations before starting other mixing procedures
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_MIXING_NBUFFER = Quantity(
        type=str,
        shape=[],
        description='''
        Number of previous steps stored for the actual mixing scheme
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_MIXING_NMIXING = Quantity(
        type=str,
        shape=[],
        description='''
        Minimal number of density mixing (should be greater than 0),before starting DIIS
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_MIXING_NSKIP = Quantity(
        type=str,
        shape=[],
        description='''
        Number of initial iteration for which the mixing is skipped
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_MIXING_PULAY_ALPHA = Quantity(
        type=str,
        shape=[],
        description='''
        Fraction of new density to be added to the Pulay expansion
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_MIXING_PULAY_BETA = Quantity(
        type=str,
        shape=[],
        description='''
        Fraction of residual contribution to be added to Pulay expansion
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_MIXING_R_FACTOR = Quantity(
        type=str,
        shape=[],
        description='''
        Control factor for the magnitude of the unpredicted step size in the update by the
        multisecant mixing scheme
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_MIXING_REGULARIZATION = Quantity(
        type=str,
        shape=[],
        description='''
        Regularization parameter to stabilize the inversion of the residual matrix {Yn^t
        Yn} in the multisecant mixing scheme (noise)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_MIXING_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Controls the activation of the mixing procedure
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XAS_SCF_OT(MSection):
    '''
    Sets the various options for the orbital transformation (OT) method. Default settings
    already provide an efficient, yet robust method. Most systems benefit from using the
    FULL_ALL preconditioner combined with a small value (0.001) of ENERGY_GAP.Well-behaved
    systems might benefit from using a DIIS minimizer.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OT_ALGORITHM = Quantity(
        type=str,
        shape=[],
        description='''
        Algorithm to be used for OT
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OT_BROYDEN_ADAPTIVE_SIGMA = Quantity(
        type=str,
        shape=[],
        description='''
        Enable adaptive curvature estimation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OT_BROYDEN_BETA = Quantity(
        type=str,
        shape=[],
        description='''
        Underrelaxation for the broyden mixer
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OT_BROYDEN_ENABLE_FLIP = Quantity(
        type=str,
        shape=[],
        description='''
        Ensure positive definite update
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OT_BROYDEN_ETA = Quantity(
        type=str,
        shape=[],
        description='''
        Dampening of estimated energy curvature.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OT_BROYDEN_FORGET_HISTORY = Quantity(
        type=str,
        shape=[],
        description='''
        Forget history on bad approximation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OT_BROYDEN_GAMMA = Quantity(
        type=str,
        shape=[],
        description='''
        Backtracking parameter
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OT_BROYDEN_OMEGA = Quantity(
        type=str,
        shape=[],
        description='''
        Growth limit of curvature.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OT_BROYDEN_SIGMA_DECREASE = Quantity(
        type=str,
        shape=[],
        description='''
        Reduction of curvature on bad approximation.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OT_BROYDEN_SIGMA_MIN = Quantity(
        type=str,
        shape=[],
        description='''
        Minimum adaptive curvature.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OT_BROYDEN_SIGMA = Quantity(
        type=str,
        shape=[],
        description='''
        Curvature of energy functional.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OT_ENERGIES = Quantity(
        type=str,
        shape=[],
        description='''
        Optimize orbital energies for use in Fermi-Dirac smearing (requires ROTATION and
        FD smearing to be active).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OT_ENERGY_GAP = Quantity(
        type=str,
        shape=[],
        description='''
        Should be an estimate for the energy gap [a.u.] (HOMO-LUMO) and is used in
        preconditioning, especially effective with the FULL_ALL preconditioner, in which
        case it should be an underestimate of the gap (can be a small number, e.g. 0.002).
        FULL_SINGLE_INVERSE takes it as lower bound (values below 0.05 can cause stability
        issues). In general, heigher values will tame the preconditioner in case of poor
        initial guesses. A negative value will leave the choice to CP2K depending on type
        of preconditioner.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OT_EPS_IRAC_FILTER_MATRIX = Quantity(
        type=str,
        shape=[],
        description='''
        Sets the threshold for filtering the matrices.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OT_EPS_IRAC_QUICK_EXIT = Quantity(
        type=str,
        shape=[],
        description='''
        Only one extra refinement iteration is done when the norm is below this value.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OT_EPS_IRAC_SWITCH = Quantity(
        type=str,
        shape=[],
        description='''
        The algorithm switches to the polynomial refinement when the norm is below this
        value.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OT_EPS_IRAC = Quantity(
        type=str,
        shape=[],
        description='''
        Targeted accuracy during the refinement iteration.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OT_EPS_TAYLOR = Quantity(
        type=str,
        shape=[],
        description='''
        Target accuracy of the taylor expansion for the matrix functions, should normally
        be kept as is.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OT_GOLD_TARGET = Quantity(
        type=str,
        shape=[],
        description='''
        Target relative uncertainty in the location of the minimum for LINESEARCH GOLD
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OT_IRAC_DEGREE = Quantity(
        type=str,
        shape=[],
        description='''
        The refinement polynomial degree (2, 3 or 4).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OT_LINESEARCH = Quantity(
        type=str,
        shape=[],
        description='''
        1D line search algorithm to be used with the OT minimizer, in increasing order of
        robustness and cost. MINIMIZER CG combined with LINESEARCH GOLD should always find
        an electronic minimum.  Whereas the 2PNT minimizer is almost always OK, 3PNT might
        be needed for systems in which successive OT CG steps do not decrease the total
        energy.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OT_MAX_IRAC = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum allowed refinement iteration.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OT_MAX_TAYLOR = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum order of the Taylor expansion before diagonalisation is prefered, for
        large parallel runs a slightly higher order could sometimes result in a small
        speedup.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OT_MINIMIZER = Quantity(
        type=str,
        shape=[],
        description='''
        Minimizer to be used with the OT method
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OT_MIXED_PRECISION = Quantity(
        type=str,
        shape=[],
        description='''
        Uses a mixed precision algorithm.With a well behaved basis set (i.e. condition
        number less than 1/eps_sp)it provides double precision accuracy results and up to
        a 2 fold speedup for building and applying the preconditioner.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OT_N_HISTORY_VEC = Quantity(
        type=str,
        shape=[],
        description='''
        Number of history vectors to be used with DIIS or BROYDEN
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OT_NONDIAG_ENERGY_STRENGTH = Quantity(
        type=str,
        shape=[],
        description='''
        The prefactor for the non-diagonal energy penalty (FD smearing)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OT_NONDIAG_ENERGY = Quantity(
        type=str,
        shape=[],
        description='''
        Add a non-diagonal energy penalty (FD smearing)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OT_OCCUPATION_PRECONDITIONER = Quantity(
        type=str,
        shape=[],
        description='''
        Preconditioner with the occupation numbers (FD smearing)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OT_ON_THE_FLY_LOC = Quantity(
        type=str,
        shape=[],
        description='''
        On the fly localization of the molecular orbitals. Can only be used with OT/IRAC.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OT_ORTHO_IRAC = Quantity(
        type=str,
        shape=[],
        description='''
        The orthogonality method.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OT_PRECOND_SOLVER = Quantity(
        type=str,
        shape=[],
        description='''
        How the preconditioner is applied to the residual.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OT_PRECONDITIONER = Quantity(
        type=str,
        shape=[],
        description='''
        Type of preconditioner to be used with all minimization schemes. They differ in
        effectiveness, cost of construction, cost of application. Properly preconditioned
        minimization can be orders of magnitude faster than doing nothing.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OT_ROTATION = Quantity(
        type=str,
        shape=[],
        description='''
        Introduce additional variables so that rotations of the occupied subspace are
        allowed as well, only needed for cases where the energy is not invariant under  a
        rotation of the occupied subspace such as non-singlet restricted calculations  or
        fractional occupations.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OT_SAFE_DIIS = Quantity(
        type=str,
        shape=[],
        description='''
        Reject DIIS steps if they point away from the minimum, do SD in that case.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OT_SCP_NDDO = Quantity(
        type=str,
        shape=[],
        description='''
        Introduce additional self-consistent polarization through response basis set =
        orbital basis set for NDDO.)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OT_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        controls the activation of the ot method
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OT_STEPSIZE = Quantity(
        type=str,
        shape=[],
        description='''
        Initial stepsize used for the line search, sometimes this parameter can be reduced
        to stablize DIIS or to improve the CG behavior in the first few steps. The optimal
        value depends on the quality of the preconditioner. A negative values leaves the
        choice to CP2K depending on the preconditioner.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XAS_SCF_OUTER_SCF(MSection):
    '''
    parameters controlling the outer SCF loop
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OUTER_SCF_BISECT_TRUST_COUNT = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum number of times the same point will be used in bisection, a small number
        guards against the effect of wrongly converged states.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OUTER_SCF_DIIS_BUFFER_LENGTH = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum number of DIIS vectors used
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OUTER_SCF_EPS_SCF = Quantity(
        type=str,
        shape=[],
        description='''
        The target gradient of the outer scf variables. Notice that the EPS_SCF of the
        inner loop also determines the value that can be reached in the outer loop,
        typically EPS_SCF of the outer loop must be smaller than EPS_SCF of the inner
        loop.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OUTER_SCF_EXTRAPOLATION_ORDER = Quantity(
        type=str,
        shape=[],
        description='''
        Number of past states used in the extrapolation of the variables during e.g. MD
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OUTER_SCF_MAX_SCF = Quantity(
        type=str,
        shape=[],
        description='''
        The maximum number of outer loops
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OUTER_SCF_OPTIMIZER = Quantity(
        type=str,
        shape=[],
        description='''
        Method used to bring the outer loop to a stationary point
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OUTER_SCF_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        controls the activation of the outer SCF loop
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OUTER_SCF_STEP_SIZE = Quantity(
        type=str,
        shape=[],
        description='''
        The initial step_size used in the optimizer (currently steepest descent).Note that
        in cases where a sadle point is sought for (DDAPC_CONSTRAINT), this can be
        negative
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_OUTER_SCF_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies which kind of outer SCF should be employed
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XAS_SCF_SMEAR(MSection):
    '''
    Define the smearing of the MO occupation numbers
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_SMEAR_ELECTRONIC_TEMPERATURE = Quantity(
        type=str,
        shape=[],
        description='''
        Electronic temperature in the case of Fermi-Dirac smearing
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_SMEAR_EPS_FERMI_DIRAC = Quantity(
        type=str,
        shape=[],
        description='''
        Accuracy checks on occupation numbers use this as a tolerance
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_SMEAR_FIXED_MAGNETIC_MOMENT = Quantity(
        type=str,
        shape=[],
        description='''
        Imposed difference between the numbers of electrons of spin up and spin down: m =
        n(up) - n(down). A negative value (default) allows for a change of the magnetic
        moment. -1 specifically keeps an integer number of spin up and spin down
        electrons.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_SMEAR_LIST = Quantity(
        type=str,
        shape=[],
        description='''
        A list of fractional occupations to use. Must match the number of states and sum
        up to the correct number of electrons
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_SMEAR_METHOD = Quantity(
        type=str,
        shape=[],
        description='''
        Smearing method to be applied
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_SMEAR_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Controls the activation of smearing
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_SMEAR_WINDOW_SIZE = Quantity(
        type=str,
        shape=[],
        description='''
        Size of the energy window centred at the Fermi level
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XAS_SCF(MSection):
    '''
    parameters needed perform an scf run
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_ADDED_MOS = Quantity(
        type=str,
        shape=[],
        description='''
        Number of additional MOS added for each spin
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_CHOLESKY = Quantity(
        type=str,
        shape=[],
        description='''
        If the cholesky method should be used for computing the inverse of S, and in this
        case calling which Lapack routines
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_EPS_DIIS = Quantity(
        type=str,
        shape=[],
        description='''
        Threshold on the convergence to start using DIAG/DIIS
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_EPS_EIGVAL = Quantity(
        type=str,
        shape=[],
        description='''
        Throw away linear combinations of basis functions with a small eigenvalue in S
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_EPS_LUMO = Quantity(
        type=str,
        shape=[],
        description='''
        target accuracy of the computation of the lumo energy
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_EPS_SCF_HISTORY = Quantity(
        type=str,
        shape=[],
        description='''
        target accuracy for the scf convergence after the history pipeline is filled
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_EPS_SCF = Quantity(
        type=str,
        shape=[],
        description='''
        target accuracy for the scf convergence
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_LEVEL_SHIFT = Quantity(
        type=str,
        shape=[],
        description='''
        Use level shifting to improve convergence
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_MAX_DIIS = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum number of DIIS vectors to be used
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_MAX_ITER_LUMO = Quantity(
        type=str,
        shape=[],
        description='''
        The maximum number of iteration for the lumo computation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_MAX_SCF_HISTORY = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum number of SCF iterations after the history pipeline is filled
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_MAX_SCF = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum number of SCF iteration to be performed for one optimization
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_NCOL_BLOCK = Quantity(
        type=str,
        shape=[],
        description='''
        Sets the number of columns in a scalapack block
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_NROW_BLOCK = Quantity(
        type=str,
        shape=[],
        description='''
        sets the number of rows in a scalapack block
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_ROKS_F = Quantity(
        type=str,
        shape=[],
        description='''
        Allows to define the parameter f for the general ROKS scheme.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_ROKS_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Allows to define all parameters for the high-spin ROKS scheme explicitly. The full
        set of 6 parameters has to be specified in the order acc, bcc, aoo, boo, avv, bvv
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_ROKS_SCHEME = Quantity(
        type=str,
        shape=[],
        description='''
        Selects the ROKS scheme when ROKS is applied.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SCF_SCF_GUESS = Quantity(
        type=str,
        shape=[],
        description='''
        Change the initial guess for the wavefunction.
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XAS_SCF_DIAGONALIZATION'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XAS_SCF_MIXING = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XAS_SCF_MIXING'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XAS_SCF_OT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XAS_SCF_OT'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XAS_SCF_OUTER_SCF = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XAS_SCF_OUTER_SCF'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XAS_SCF_SMEAR = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XAS_SCF_SMEAR'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT_XAS(MSection):
    '''
    Sets the method of choice to calculate core-level excitation spectra. The occupied
    states from  which we calculate the excitation should be specified. Localization of
    the orbitals may be useful.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XAS_ADDED_MOS = Quantity(
        type=str,
        shape=[],
        description='''
        Number of additional MOS added spin up only
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_ATOMS_LIST = Quantity(
        type=str,
        shape=[],
        description='''
        Indexes of the atoms to be excitedThis keyword can be repeated several
        times(useful if you have to specify many indexes).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_DIPOLE_FORM = Quantity(
        type=str,
        shape=[],
        description='''
        Type of integral to get the oscillator strengths in the diipole approximation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_EPS_ADDED = Quantity(
        type=str,
        shape=[],
        description='''
        target accuracy incalculation of the added orbitals
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_MAX_ITER_ADDED = Quantity(
        type=str,
        shape=[],
        description='''
        maximum number of iteration in calculation of added orbitals
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_METHOD = Quantity(
        type=str,
        shape=[],
        description='''
        Method to be used to calculate core-level excitation spectra
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_NGAUSS = Quantity(
        type=str,
        shape=[],
        description='''
        Number of gto's for the expansion of the stoof the type given by STATE_TYPE
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_RESTART = Quantity(
        type=str,
        shape=[],
        description='''
        Restart the excited state if the restart file exists
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        controls the activation of core-level spectroscopy simulations
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_STATE_SEARCH = Quantity(
        type=str,
        shape=[],
        description='''
        # of states where to look for the one to be excited
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_STATE_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Type of the orbitas that are excited for the xas spectra calculation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_WFN_RESTART_FILE_NAME = Quantity(
        type=str,
        shape=[],
        description='''
        Root of the file names where to read the MOS fromwhich to restart the calculation
        of the core level excited states
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_XAS_CORE = Quantity(
        type=str,
        shape=[],
        description='''
        Occupation of the core state in XAS calculation by TP_FLEX.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_XAS_TOT_EL = Quantity(
        type=str,
        shape=[],
        description='''
        Total number of electrons for spin channel alpha, in XAS calculation by TP_FLEX.If
        negative value, the number of electrons is set to GS number of electron  minus the
        amount subtracted from the core state
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_XES_CORE = Quantity(
        type=str,
        shape=[],
        description='''
        Occupation of the core state in XES calculation by TP_VAL.The homo is emptied by
        the same amount
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XAS_XES_EMPTY_HOMO = Quantity(
        type=str,
        shape=[],
        description='''
        Set the occupation of the HOMO in XES calculation by TP_VAL.The HOMO can be
        emptied or not, if the core is still full
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_XAS_LOCALIZE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XAS_LOCALIZE'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XAS_SCF = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XAS_SCF'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_ADIABATIC_RESCALING(MSection):
    '''
    Parameters for self interation corrected hybrid functionals
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_ADIABATIC_RESCALING_FUNCTIONAL_MODEL = Quantity(
        type=str,
        shape=[],
        description='''
        Which model for the coupling constant integration should be used.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_ADIABATIC_RESCALING_FUNCTIONAL_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Which Hybrid functional should be used. (Has to be consistent with the definitions
        in XC and HF).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_ADIABATIC_RESCALING_LAMBDA = Quantity(
        type=str,
        shape=[],
        description='''
        The point to be used along the adiabatic curve (0 &#60; λ &#60; 1)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_ADIABATIC_RESCALING_OMEGA = Quantity(
        type=str,
        shape=[],
        description='''
        Long-range parameter
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_HF_HF_INFO(MSection):
    '''
    Controls the printing basic info about hf method
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_HF_HF_INFO_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_HF_HF_INFO_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_HF_HF_INFO_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_HF_HF_INFO_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_HF_HF_INFO_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_HF_HFX_RI(MSection):
    '''
    All parameters needed in a HFX RI calculation
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_HF_HFX_RI_EPS_OPTIMIZATION = Quantity(
        type=str,
        shape=[],
        description='''
        Accuracy of iterative RI fit
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_HF_HFX_RI_EPS_SCREENING = Quantity(
        type=str,
        shape=[],
        description='''
        Accuracy of geminal integral evaluation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_HF_HFX_RI_MAX_ITER = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum number of iteration in RI fit
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_HF_INTERACTION_POTENTIAL(MSection):
    '''
    Sets up interaction potential if requested
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_HF_INTERACTION_POTENTIAL_CUTOFF_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Determines cutoff radius for the truncated 1/r potential. Only valid when doing
        truncated calculation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_HF_INTERACTION_POTENTIAL_OMEGA = Quantity(
        type=str,
        shape=[],
        description='''
        Parameter for short/longrange interaction
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_HF_INTERACTION_POTENTIAL_POTENTIAL_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Which interaction potential should be used (Coulomb, longrange or shortrange).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_HF_INTERACTION_POTENTIAL_SCALE_COULOMB = Quantity(
        type=str,
        shape=[],
        description='''
        Scales Hartree-Fock contribution arising from a coulomb potential. Only valid when
        doing a mixed potential calculation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_HF_INTERACTION_POTENTIAL_SCALE_GAUSSIAN = Quantity(
        type=str,
        shape=[],
        description='''
        Scales Hartree-Fock contribution arising from a gaussian potential. Only valid
        when doing a mixed potential calculation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_HF_INTERACTION_POTENTIAL_SCALE_LONGRANGE = Quantity(
        type=str,
        shape=[],
        description='''
        Scales Hartree-Fock contribution arising from a longrange potential. Only valid
        when doing a mixed potential calculation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_HF_INTERACTION_POTENTIAL_T_C_G_DATA = Quantity(
        type=str,
        shape=[],
        description='''
        Location of the file t_c_g.dat that contains the data for the evaluation of the
        truncated gamma function
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_HF_LOAD_BALANCE(MSection):
    '''
    Parameters influencing the load balancing of the HF
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_HF_LOAD_BALANCE_BLOCK_SIZE = Quantity(
        type=str,
        shape=[],
        description='''
        Determines the blocking used for the atomic quartet loops. A proper choice can
        speedup the calculation. The default (-1) is automatic.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_HF_LOAD_BALANCE_NBINS = Quantity(
        type=str,
        shape=[],
        description='''
        Number of bins per process used to group atom quartets.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_HF_LOAD_BALANCE_RANDOMIZE = Quantity(
        type=str,
        shape=[],
        description='''
        This flag controls the randomization of the bin assignment to processes. For
        highly ordered input structures with a bad load balance, setting this flag to TRUE
        might improve.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_HF_MEMORY(MSection):
    '''
    Sets up memory parameters for the storage of the ERI's if requested
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_HF_MEMORY_EPS_STORAGE_SCALING = Quantity(
        type=str,
        shape=[],
        description='''
        Scaling factor to scale eps_schwarz. Storage threshold for compression will be
        EPS_SCHWARZ*EPS_STORAGE_SCALING.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_HF_MEMORY_MAX_DISK_SPACE = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the maximum amount of disk space [MB] used to store precomputed compressed
        four-center integrals. If 0, nothing is stored to disk
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_HF_MEMORY_MAX_MEMORY = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the maximum amount of memory [MB] to be consumed by the full HFX module.
        All temporary buffers and helper arrays are subtracted from this number. What
        remains will be used for storage of integrals. NOTE: This number is assumed to
        represent the memory available to one MPI process. When running a threaded
        version, cp2k automatically takes care of distributing the memory among all the
        threads within a process.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_HF_MEMORY_STORAGE_LOCATION = Quantity(
        type=str,
        shape=[],
        description='''
        Loaction where ERI's are stored if MAX_DISK_SPACE /=0 Expects a path to a
        directory.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_HF_MEMORY_TREAT_FORCES_IN_CORE = Quantity(
        type=str,
        shape=[],
        description='''
        Determines whether the derivative ERI's should be stored to RAM or not. Only
        meaningful when performing Ehrenfest MD. Memory usage is defined via MAX_MEMORY,
        i.e. the memory is shared wit the energy ERI's.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_HF_PERIODIC(MSection):
    '''
    Sets up periodic boundary condition parameters if requested
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_HF_PERIODIC_NUMBER_OF_SHELLS = Quantity(
        type=str,
        shape=[],
        description='''
        Number of shells taken into account for periodicity. By default, cp2k tries to
        automatically evaluate this number. This algorithm might be to conservative,
        resulting in some overhead. You can try to adjust this number in order to make a
        calculation cheaper.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_HF_SCREENING(MSection):
    '''
    Sets up screening parameters if requested
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_HF_SCREENING_EPS_SCHWARZ_FORCES = Quantity(
        type=str,
        shape=[],
        description='''
        Screens the near field part of the electronic repulsion integrals using the
        Schwarz inequality for the given threshold. This will be approximately the
        accuracy of the forces,  and should normally be similar to EPS_SCF
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_HF_SCREENING_EPS_SCHWARZ = Quantity(
        type=str,
        shape=[],
        description='''
        Screens the near field part of the electronic repulsion integrals using the
        Schwarz inequality for the given threshold.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_HF_SCREENING_P_SCREEN_CORRECTION_FACTOR = Quantity(
        type=str,
        shape=[],
        description='''
        Recalculates integrals on the fly if the actual density matrix is larger by a
        given factor than the initial one. If the factor is set to 0.0_dp, this feature is
        disabled.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_HF_SCREENING_SCREEN_ON_INITIAL_P = Quantity(
        type=str,
        shape=[],
        description='''
        Screen on an initial density matrix. For the first MD step this matrix must be
        provided by a Restart File.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_HF_SCREENING_SCREEN_P_FORCES = Quantity(
        type=str,
        shape=[],
        description='''
        Screens the electronic repulsion integrals for the forces using the density
        matrix. This results in a significant speedup for large systems, but might require
        a somewhat tigher EPS_SCHWARZ_FORCES.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_HF(MSection):
    '''
    Sets up the Hartree-Fock parameters if requested
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_HF_FRACTION = Quantity(
        type=str,
        shape=[],
        description='''
        The fraction of Hartree-Fock to add to the total energy. 1.0 implies standard
        Hartree-Fock if used with XC_FUNCTIONAL NONE. NOTE: In a mixed potential
        calculation this should be set to 1.0, otherwise all parts are multiplied with
        this factor.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_HF_PW_HFX_BLOCKSIZE = Quantity(
        type=str,
        shape=[],
        description='''
        Improve the performance of pw_hfx at the cost of some additional memory by storing
        the realspace representation of PW_HFX_BLOCKSIZE states.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_HF_PW_HFX = Quantity(
        type=str,
        shape=[],
        description='''
        Compute the Hartree-Fock energy also in the plane wave basis.The value is ignored,
        and intended for debugging only.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_HF_TREAT_LSD_IN_CORE = Quantity(
        type=str,
        shape=[],
        description='''
        Determines how spin denisities are taken into account. If true, the beta spin
        density is included via a second in core call. If false, alpha and beta spins are
        done in one shot
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_HF_HF_INFO = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_HF_HF_INFO'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_HF_HFX_RI = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_HF_HFX_RI'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_HF_INTERACTION_POTENTIAL = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_HF_INTERACTION_POTENTIAL'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_HF_LOAD_BALANCE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_HF_LOAD_BALANCE'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_HF_MEMORY = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_HF_MEMORY'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_HF_PERIODIC = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_HF_PERIODIC'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_HF_SCREENING = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_HF_SCREENING'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_VDW_POTENTIAL_NON_LOCAL(MSection):
    '''
    Information on the non local dispersion functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_VDW_POTENTIAL_NON_LOCAL_CUTOFF = Quantity(
        type=str,
        shape=[],
        description='''
        The cutoff of the FFT grid used in the calculation of the nonlocal vdW functional
        [Ry].
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_VDW_POTENTIAL_NON_LOCAL_KERNEL_FILE_NAME = Quantity(
        type=str,
        shape=[],
        description='''
        Name of the kernel data file, may include a path.vdW_kernel_table.dat is for DRSLL
        and LMKLL andrVV10_kernel_table.dat is for rVV10.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_VDW_POTENTIAL_NON_LOCAL_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Parameters b and C of the rVV10 functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_VDW_POTENTIAL_NON_LOCAL_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Type of functional (the corresponding kernel data file should be selected).Allows
        for common forms such as vdW-DF, vdW-DF2, optB88-vdW, rVV10.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_VDW_POTENTIAL_NON_LOCAL_VERBOSE_OUTPUT = Quantity(
        type=str,
        shape=[],
        description='''
        Extensive output for non local functionals
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_PRINT_DFTD(MSection):
    '''
    Controls the printing of some info about DFTD contributions
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_PRINT_DFTD_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_PRINT_DFTD_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_PRINT_DFTD_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_PRINT_DFTD_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_PRINT_DFTD_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_VDW_POTENTIAL_PAIR_POTENTIAL(MSection):
    '''
    Information on the pair potential to calculate dispersion
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_ATOM_COORDINATION_NUMBERS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the coordination number of a set of atoms for the C9 term in DFT-D3.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_ATOMPARM = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies parameters for atom types (in atomic units). If not provided default
        parameters are used (DFT-D2).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_CALCULATE_C9_TERM = Quantity(
        type=str,
        shape=[],
        description='''
        Calculate C9 terms in DFT-D3 model
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_D3_SCALING = Quantity(
        type=str,
        shape=[],
        description='''
        XC Functional dependent scaling parameters (s6,sr6,s8) for the DFT-D3 method, if
        set to zero CP2K attempts to guess the xc functional from REFERENCE_FUNCTIONAL and
        sets the associated scaling parameter.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_D3BJ_SCALING = Quantity(
        type=str,
        shape=[],
        description='''
        XC Functional dependent scaling parameters (s6,a1,s8,a2) for the DFT-D3(BJ)
        method, if set to zero CP2K attempts to guess the xc functional from
        REFERENCE_FUNCTIONAL and sets the associated scaling parameter.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_EPS_CN = Quantity(
        type=str,
        shape=[],
        description='''
        Cutoff value for coordination number function (DFT-D3 method)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_EXP_PRE = Quantity(
        type=str,
        shape=[],
        description='''
        Prefactor in exponential damping factor (DFT-D2 potential)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_KIND_COORDINATION_NUMBERS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the coordination number for a kind for the C9 term in DFT-D3.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_LONG_RANGE_CORRECTION = Quantity(
        type=str,
        shape=[],
        description='''
        Calculate a long range correction to the DFT-D3 model
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_PARAMETER_FILE_NAME = Quantity(
        type=str,
        shape=[],
        description='''
        Name of the parameter file, may include a path
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_R_CUTOFF = Quantity(
        type=str,
        shape=[],
        description='''
        Range of potential. The cutoff will be 2 times this value
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_REFERENCE_C9_TERM = Quantity(
        type=str,
        shape=[],
        description='''
        Calculate C9 terms in DFT-D3 model using reference coordination numbers
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_REFERENCE_FUNCTIONAL = Quantity(
        type=str,
        shape=[],
        description='''
        Use parameters for this specific density functional. For available D3 and D3(BJ)
        parameters see: http://www.thch.uni-bonn.de/tc/downloads/DFT-D3/functionals.html,
        http://www.thch.uni-bonn.de/tc/downloads/DFT-D3/functionalsbj.html
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_SCALING = Quantity(
        type=str,
        shape=[],
        description='''
        XC Functional dependent scaling parameter, if set to zero CP2K attempts to guess
        the xc functional that is in use and sets the associated scaling parameter.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Type of potential
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_VERBOSE_OUTPUT = Quantity(
        type=str,
        shape=[],
        description='''
        Extensive output for the DFT-D2 and DFT-D3 models
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_PRINT_DFTD = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_VDW_POTENTIAL_PAIR_POTENTIAL_PRINT_DFTD'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_VDW_POTENTIAL(MSection):
    '''
    This section combines all possible additional dispersion corrections to the normal XC
    functionals. This can be more functionals or simple empirical pair potentials.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_VDW_POTENTIAL_POTENTIAL_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Type of dispersion/vdW functional or potential to use
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_VDW_POTENTIAL_NON_LOCAL = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_VDW_POTENTIAL_NON_LOCAL'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_VDW_POTENTIAL_PAIR_POTENTIAL = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_VDW_POTENTIAL_PAIR_POTENTIAL'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_CPHF(MSection):
    '''
    Parameters influencing the solution of the Z-vector equations in MP2 gradients
    calculations.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_CPHF_EPS_CONV = Quantity(
        type=str,
        shape=[],
        description='''
        Convergence threshold for the solution of the Z-vector equations. The Z-vector
        equations have the form of a linear system of equations Ax=b, convergence is
        achieved when |Ax-b|<=EPS_CONV.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_CPHF_MAX_ITER = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum number of iterations allowed for the solution of the Z-vector equations.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_DIRECT_CANONICAL(MSection):
    '''
    Parameters influencing the direct canonical method
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_DIRECT_CANONICAL_BIG_SEND = Quantity(
        type=str,
        shape=[],
        description='''
        Send big messages between processes (useful for >48 processors).
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_INTERACTION_POTENTIAL(MSection):
    '''
    Parameters the interaction potential in computing the biel integrals
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_INTERACTION_POTENTIAL_POTENTIAL_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Which interaction potential should be used (Coulomb, TShPSC operator).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_INTERACTION_POTENTIAL_TRUNCATION_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Determines truncation radius for the truncated TShPSC potential. Only valid when
        doing truncated calculation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_INTERACTION_POTENTIAL_TSHPSC_DATA = Quantity(
        type=str,
        shape=[],
        description='''
        Location of the file TShPSC.dat that contains the data for the evaluation of the
        TShPSC G0
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_MP2_INFO(MSection):
    '''
    Controls the printing basic info about MP2 method
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_MP2_INFO_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_MP2_INFO_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_MP2_INFO_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_MP2_INFO_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_MP2_INFO_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_OPT_RI_BASIS(MSection):
    '''
    Parameters influencing the optimization of the RI MP2 basis. Only exponents of non-
    contracted auxiliary basis can be optimized. An initial RI auxiliary basis has to be
    specified.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_OPT_RI_BASIS_BASIS_SIZE = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the size of the auxiliary basis set automatically generated as initial
        guess. This will be effective only if RI_AUX_BASIS_SET in the KIND section and
        NUM_FUNC are not specified.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_OPT_RI_BASIS_DELTA_I_REL = Quantity(
        type=str,
        shape=[],
        description='''
        Target accuracy in the relative deviation of the amplitudes calculated with and
        without RI approximation, (more details in Chem.Phys.Lett.294(1998)143).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_OPT_RI_BASIS_DELTA_RI = Quantity(
        type=str,
        shape=[],
        description='''
        Target accuracy in the absolute difference between the RI-MP2 and the exact MP2
        energy, DRI=ABS(E_MP2-E_RI-MP2).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_OPT_RI_BASIS_EPS_DERIV = Quantity(
        type=str,
        shape=[],
        description='''
        The derivatives of the MP2 energy with respect to the exponents of the basis are
        calculated numerically. The change in the exponent a_i employed for the numerical
        evaluation is defined as h_i=EPS_DERIV*a_i.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_OPT_RI_BASIS_MAX_ITER = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the maximum number of steps in the RI basis optimization.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_OPT_RI_BASIS_NUM_FUNC = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the number of function, for each angular momentum (s, p, d ...),
        employed in the automatically generated initial guess. This will be effective only
        if RI_AUX_BASIS_SET in the KIND section is not specified.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_LAPLACE(MSection):
    '''
    Parameters influencing the RI-SOS-MP2-Laplace method
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_LAPLACE_QUADRATURE_POINTS = Quantity(
        type=str,
        shape=[],
        description='''
        Number of quadrature points for the numerical integration in the RI-SOS-
        MP2-Laplace method.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_LAPLACE_SIZE_INTEG_GROUP = Quantity(
        type=str,
        shape=[],
        description='''
        Group size for the integration in the Laplace method, that is the number of
        processes involved in the computation of each integration point. SIZE_INTEG_GROUP
        has to be a multiple of GROUP_SIZE in the WF_CORRELATION section. The default (-1)
        is automatic.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_MP2(MSection):
    '''
    Parameters influencing the RI MP2 method
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_MP2_BLOCK_SIZE = Quantity(
        type=str,
        shape=[],
        description='''
        Determines the blocking used for communication in RI-MP2. Larger BLOCK_SIZE
        reduces communication but requires more memory. The default (-1) is automatic.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_MP2_EPS_CANONICAL = Quantity(
        type=str,
        shape=[],
        description='''
        Threshold for discriminate if a given ij pairs of the unrelaxed MP2 density matrix
        has to be calculated with a canonical reformulation based on the occupied
        eigenvalues differences.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_MP2_FREE_HFX_BUFFER = Quantity(
        type=str,
        shape=[],
        description='''
        Free the buffer containing the 4 center integrals used in the Hartree-Fock
        exchange calculation. This will be effective only for gradients calculations,
        since for the energy only case, the buffers are released by default. (Right now
        debugging only).
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_HF_INFO(MSection):
    '''
    Controls the printing basic info about hf method
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_HF_INFO_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_HF_INFO_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_HF_INFO_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_HF_INFO_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_HF_INFO_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_HFX_RI(MSection):
    '''
    All parameters needed in a HFX RI calculation
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_HFX_RI_EPS_OPTIMIZATION = Quantity(
        type=str,
        shape=[],
        description='''
        Accuracy of iterative RI fit
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_HFX_RI_EPS_SCREENING = Quantity(
        type=str,
        shape=[],
        description='''
        Accuracy of geminal integral evaluation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_HFX_RI_MAX_ITER = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum number of iteration in RI fit
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_INTERACTION_POTENTIAL(MSection):
    '''
    Sets up interaction potential if requested
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_INTERACTION_POTENTIAL_CUTOFF_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Determines cutoff radius for the truncated 1/r potential. Only valid when doing
        truncated calculation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_INTERACTION_POTENTIAL_OMEGA = Quantity(
        type=str,
        shape=[],
        description='''
        Parameter for short/longrange interaction
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_INTERACTION_POTENTIAL_POTENTIAL_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Which interaction potential should be used (Coulomb, longrange or shortrange).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_INTERACTION_POTENTIAL_SCALE_COULOMB = Quantity(
        type=str,
        shape=[],
        description='''
        Scales Hartree-Fock contribution arising from a coulomb potential. Only valid when
        doing a mixed potential calculation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_INTERACTION_POTENTIAL_SCALE_GAUSSIAN = Quantity(
        type=str,
        shape=[],
        description='''
        Scales Hartree-Fock contribution arising from a gaussian potential. Only valid
        when doing a mixed potential calculation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_INTERACTION_POTENTIAL_SCALE_LONGRANGE = Quantity(
        type=str,
        shape=[],
        description='''
        Scales Hartree-Fock contribution arising from a longrange potential. Only valid
        when doing a mixed potential calculation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_INTERACTION_POTENTIAL_T_C_G_DATA = Quantity(
        type=str,
        shape=[],
        description='''
        Location of the file t_c_g.dat that contains the data for the evaluation of the
        truncated gamma function
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_LOAD_BALANCE(MSection):
    '''
    Parameters influencing the load balancing of the HF
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_LOAD_BALANCE_BLOCK_SIZE = Quantity(
        type=str,
        shape=[],
        description='''
        Determines the blocking used for the atomic quartet loops. A proper choice can
        speedup the calculation. The default (-1) is automatic.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_LOAD_BALANCE_NBINS = Quantity(
        type=str,
        shape=[],
        description='''
        Number of bins per process used to group atom quartets.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_LOAD_BALANCE_RANDOMIZE = Quantity(
        type=str,
        shape=[],
        description='''
        This flag controls the randomization of the bin assignment to processes. For
        highly ordered input structures with a bad load balance, setting this flag to TRUE
        might improve.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_MEMORY(MSection):
    '''
    Sets up memory parameters for the storage of the ERI's if requested
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_MEMORY_EPS_STORAGE_SCALING = Quantity(
        type=str,
        shape=[],
        description='''
        Scaling factor to scale eps_schwarz. Storage threshold for compression will be
        EPS_SCHWARZ*EPS_STORAGE_SCALING.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_MEMORY_MAX_DISK_SPACE = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the maximum amount of disk space [MB] used to store precomputed compressed
        four-center integrals. If 0, nothing is stored to disk
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_MEMORY_MAX_MEMORY = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the maximum amount of memory [MB] to be consumed by the full HFX module.
        All temporary buffers and helper arrays are subtracted from this number. What
        remains will be used for storage of integrals. NOTE: This number is assumed to
        represent the memory available to one MPI process. When running a threaded
        version, cp2k automatically takes care of distributing the memory among all the
        threads within a process.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_MEMORY_STORAGE_LOCATION = Quantity(
        type=str,
        shape=[],
        description='''
        Loaction where ERI's are stored if MAX_DISK_SPACE /=0 Expects a path to a
        directory.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_MEMORY_TREAT_FORCES_IN_CORE = Quantity(
        type=str,
        shape=[],
        description='''
        Determines whether the derivative ERI's should be stored to RAM or not. Only
        meaningful when performing Ehrenfest MD. Memory usage is defined via MAX_MEMORY,
        i.e. the memory is shared wit the energy ERI's.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_PERIODIC(MSection):
    '''
    Sets up periodic boundary condition parameters if requested
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_PERIODIC_NUMBER_OF_SHELLS = Quantity(
        type=str,
        shape=[],
        description='''
        Number of shells taken into account for periodicity. By default, cp2k tries to
        automatically evaluate this number. This algorithm might be to conservative,
        resulting in some overhead. You can try to adjust this number in order to make a
        calculation cheaper.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_SCREENING(MSection):
    '''
    Sets up screening parameters if requested
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_SCREENING_EPS_SCHWARZ_FORCES = Quantity(
        type=str,
        shape=[],
        description='''
        Screens the near field part of the electronic repulsion integrals using the
        Schwarz inequality for the given threshold. This will be approximately the
        accuracy of the forces,  and should normally be similar to EPS_SCF
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_SCREENING_EPS_SCHWARZ = Quantity(
        type=str,
        shape=[],
        description='''
        Screens the near field part of the electronic repulsion integrals using the
        Schwarz inequality for the given threshold.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_SCREENING_P_SCREEN_CORRECTION_FACTOR = Quantity(
        type=str,
        shape=[],
        description='''
        Recalculates integrals on the fly if the actual density matrix is larger by a
        given factor than the initial one. If the factor is set to 0.0_dp, this feature is
        disabled.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_SCREENING_SCREEN_ON_INITIAL_P = Quantity(
        type=str,
        shape=[],
        description='''
        Screen on an initial density matrix. For the first MD step this matrix must be
        provided by a Restart File.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_SCREENING_SCREEN_P_FORCES = Quantity(
        type=str,
        shape=[],
        description='''
        Screens the electronic repulsion integrals for the forces using the density
        matrix. This results in a significant speedup for large systems, but might require
        a somewhat tigher EPS_SCHWARZ_FORCES.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF(MSection):
    '''
    Sets up the Hartree-Fock parameters if requested
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_FRACTION = Quantity(
        type=str,
        shape=[],
        description='''
        The fraction of Hartree-Fock to add to the total energy. 1.0 implies standard
        Hartree-Fock if used with XC_FUNCTIONAL NONE. NOTE: In a mixed potential
        calculation this should be set to 1.0, otherwise all parts are multiplied with
        this factor.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_PW_HFX_BLOCKSIZE = Quantity(
        type=str,
        shape=[],
        description='''
        Improve the performance of pw_hfx at the cost of some additional memory by storing
        the realspace representation of PW_HFX_BLOCKSIZE states.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_PW_HFX = Quantity(
        type=str,
        shape=[],
        description='''
        Compute the Hartree-Fock energy also in the plane wave basis.The value is ignored,
        and intended for debugging only.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_TREAT_LSD_IN_CORE = Quantity(
        type=str,
        shape=[],
        description='''
        Determines how spin denisities are taken into account. If true, the beta spin
        density is included via a second in core call. If false, alpha and beta spins are
        done in one shot
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_HF_INFO = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_HF_INFO'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_HFX_RI = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_HFX_RI'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_INTERACTION_POTENTIAL = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_INTERACTION_POTENTIAL'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_LOAD_BALANCE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_LOAD_BALANCE'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_MEMORY = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_MEMORY'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_PERIODIC = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_PERIODIC'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_SCREENING = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF_SCREENING'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA(MSection):
    '''
    Parameters influencing the RI RPA method
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_MINIMAX_QUADRATURE = Quantity(
        type=str,
        shape=[],
        description='''
        Use the Minimax quadrature scheme for performing the numerical integration.
        Maximum number of quadrature point limited to 20.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_MM_STYLE = Quantity(
        type=str,
        shape=[],
        description='''
        Matrix multiplication style for the Q matrix.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_QUADRATURE_POINTS = Quantity(
        type=str,
        shape=[],
        description='''
        Number of quadrature points for the numerical integration in the RI-RPA method.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_SIZE_FREQ_INTEG_GROUP = Quantity(
        type=str,
        shape=[],
        description='''
        Group size for frequency integration, that is the number of processes involved in
        the computation of each integration point. SIZE_FREQ_INTEG_GROUP has to be a
        multiple of GROUP_SIZE in the WF_CORRELATION section. The default (-1) is
        automatic.
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA_HF'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_WFC_GPW(MSection):
    '''
    Parameters for the GPW approach in Wavefunction-based Correlation methods
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_WFC_GPW_CUTOFF = Quantity(
        type=str,
        shape=[],
        description='''
        The cutoff of the finest grid level in the MP2 gpw integration.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_WFC_GPW_EPS_FILTER = Quantity(
        type=str,
        shape=[],
        description='''
        Determines a threshold for the DBCSR based multiply (usually 10 times smaller than
        EPS_GRID).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_WFC_GPW_EPS_GRID = Quantity(
        type=str,
        shape=[],
        description='''
        Determines a threshold for the GPW based integration
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_WFC_GPW_PRINT_LEVEL = Quantity(
        type=str,
        shape=[],
        description='''
        How much output is written by the individual groups.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_WFC_GPW_REL_CUTOFF = Quantity(
        type=str,
        shape=[],
        description='''
        Determines the grid at which a Gaussian is mapped.
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION(MSection):
    '''
    Sets up the Wavefunction-based Correlation parameters if requested
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_CALC_COND_NUM = Quantity(
        type=str,
        shape=[],
        description='''
        Calculate the condition number of the (P|Q) matrix for the RI methods.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_COL_BLOCK = Quantity(
        type=str,
        shape=[],
        description='''
        Size of the column block used in the SCALAPACK block cyclic data
        distribution.Default is (COL_BLOCK=-1) is automatic. A proper choice can speedup
        the parallel matrix multiplication in the case of RI-RPA and RI-SOS-MP2-Laplace.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_GROUP_SIZE = Quantity(
        type=str,
        shape=[],
        description='''
        Group size used in the computation of the integrals. Default is to use all
        processors (GROUP_SIZE=-1).A smaller group size (for example the node size), might
        a better choice if the actual MP2 time is large compared to integral computation
        time. This is usually the case if the total number of processors is not too large.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_MEMORY = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum allowed total memory usage during MP2 methods [Mb].
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_METHOD = Quantity(
        type=str,
        shape=[],
        description='''
        Which method should be used to compute the MP2 energy
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_ROW_BLOCK = Quantity(
        type=str,
        shape=[],
        description='''
        Size of the row block used in the SCALAPACK block cyclic data distribution.Default
        is (ROW_BLOCK=-1) is automatic. A proper choice can speedup the parallel matrix
        multiplication in the case of RI-RPA and RI-SOS-MP2-Laplace.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_SCALE_S = Quantity(
        type=str,
        shape=[],
        description='''
        Scaling factor of the singlet energy component (opposite spin, OS).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_SCALE_T = Quantity(
        type=str,
        shape=[],
        description='''
        Scaling factor of the triplet energy component (same spin, SS).
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_CPHF = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_CPHF'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_DIRECT_CANONICAL = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_DIRECT_CANONICAL'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_INTERACTION_POTENTIAL = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_INTERACTION_POTENTIAL'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_MP2_INFO = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_MP2_INFO'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_OPT_RI_BASIS = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_OPT_RI_BASIS'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_LAPLACE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_LAPLACE'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_MP2 = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_MP2'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_RI_RPA'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_WFC_GPW = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION_WFC_GPW'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_BECKE88_LR_ADIABATIC(MSection):
    '''
    Uses the Becke 88 longrange exchange functional in an adiabatic fashion
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_BECKE88_LR_ADIABATIC_LAMBDA = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the parameter of the adiabatic curve
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_BECKE88_LR_ADIABATIC_OMEGA = Quantity(
        type=str,
        shape=[],
        description='''
        Potential parameter in erf(omega*r)/r
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_BECKE88_LR_ADIABATIC_SCALE_X = Quantity(
        type=str,
        shape=[],
        description='''
        scales the exchange part of the functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_BECKE88_LR_ADIABATIC_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_BECKE88_LR(MSection):
    '''
    Uses the Becke 88 longrange exchange functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_BECKE88_LR_OMEGA = Quantity(
        type=str,
        shape=[],
        description='''
        Potential parameter in erf(omega*r)/r
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_BECKE88_LR_SCALE_X = Quantity(
        type=str,
        shape=[],
        description='''
        scales the exchange part of the functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_BECKE88_LR_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_BECKE88(MSection):
    '''
    Uses the Becke 88 exchange functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_BECKE88_SCALE_X = Quantity(
        type=str,
        shape=[],
        description='''
        scales the exchange part of the functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_BECKE88_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_BECKE97(MSection):
    '''
    Uses the Becke 97 exchange correlation functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_BECKE97_PARAMETRIZATION = Quantity(
        type=str,
        shape=[],
        description='''
        switches between the B97 and Grimme parametrization
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_BECKE97_SCALE_C = Quantity(
        type=str,
        shape=[],
        description='''
        scales the correlation part of the functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_BECKE97_SCALE_X = Quantity(
        type=str,
        shape=[],
        description='''
        scales the exchange part of the functional, if -1 the default for the given
        parametrization is used
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_BECKE97_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_BECKE_ROUSSEL(MSection):
    '''
    Becke Roussel exchange hole model. Can be usedas long range correction with a
    truncated coulomb potential
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_BECKE_ROUSSEL_CUTOFF_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the cutoff radius for the truncation. If put to zero, the standard full
        range potential will be used
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_BECKE_ROUSSEL_GAMMA = Quantity(
        type=str,
        shape=[],
        description='''
        Parameter in the exchange hole. Usually this is put to 1.0 or 0.8
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_BECKE_ROUSSEL_SCALE_X = Quantity(
        type=str,
        shape=[],
        description='''
        scales the exchange part of the functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_BECKE_ROUSSEL_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_BEEF(MSection):
    '''
    Uses the BEEFvdW exchange functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_BEEF_SCALE_X = Quantity(
        type=str,
        shape=[],
        description='''
        scales the exchange part of the functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_BEEF_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_CS1(MSection):
    '''
    Uses the CS1 functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_CS1_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_GV09(MSection):
    '''
    Combination of three different exchange hole models
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_GV09_CUTOFF_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines cutoff for lower integration boundary
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_GV09_GAMMA = Quantity(
        type=str,
        shape=[],
        description='''
        Parameter for Becke Roussel hole
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_GV09_SCALE_X = Quantity(
        type=str,
        shape=[],
        description='''
        scales the exchange part of the functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_GV09_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_HCTH(MSection):
    '''
    Uses the HCTH class of functionals
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_HCTH_PARAMETER_SET = Quantity(
        type=str,
        shape=[],
        description='''
        Which version of the parameters should be used
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_HCTH_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_KE_GGA(MSection):
    '''
    Uses one of the KE_GGA functionals (optimized versions of some of these functionals
    might be available outside this section). These functionals are needed for the
    computation of the kinetic energy in the Kim-Gordon method.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_KE_GGA_FUNCTIONAL = Quantity(
        type=str,
        shape=[],
        description='''
        Which one of the KE_GGA functionals should be used
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_KE_GGA_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_KE_LIBXC(MSection):
    '''
    To be used for KG runs. Uses kinetic energy functionals from LIBXC, see also
    http://www.tddft.org/programs/octopus/wiki/index.php/Libxc
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_KE_LIBXC_FUNCTIONAL = Quantity(
        type=str,
        shape=[],
        description='''
        names of the functionals, see also
        http://www.tddft.org/programs/octopus/wiki/index.php/Libxc:manual .The precise
        list of available functionals depends on the version of libxc interfaced
        (currently 2.0.1).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_KE_LIBXC_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        parameters of the functionals
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_KE_LIBXC_SCALE = Quantity(
        type=str,
        shape=[],
        description='''
        scaling factors of the functionals
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_KE_LIBXC_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_LDA_HOLE_T_C_LR(MSection):
    '''
    LDA exchange hole model in truncated coulomb potential
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_LDA_HOLE_T_C_LR_CUTOFF_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines cutoff for lower integration boundary
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_LDA_HOLE_T_C_LR_SCALE_X = Quantity(
        type=str,
        shape=[],
        description='''
        scales the exchange part of the functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_LDA_HOLE_T_C_LR_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_LIBXC(MSection):
    '''
    Uses functionals from LIBXC, see also
    http://www.tddft.org/programs/octopus/wiki/index.php/Libxc
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_LIBXC_FUNCTIONAL = Quantity(
        type=str,
        shape=[],
        description='''
        names of the functionals, see also
        http://www.tddft.org/programs/octopus/wiki/index.php/Libxc:manual .The precise
        list of available functionals depends on the version of libxc interfaced
        (currently 2.0.1).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_LIBXC_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        parameters of the functionals
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_LIBXC_SCALE = Quantity(
        type=str,
        shape=[],
        description='''
        scaling factors of the functionals
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_LIBXC_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_LYP_ADIABATIC(MSection):
    '''
    Uses the LYP correlation functional in an adiabatic fashion
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_LYP_ADIABATIC_LAMBDA = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the parameter of the adiabatic curve.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_LYP_ADIABATIC_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_LYP(MSection):
    '''
    Uses the LYP functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_LYP_SCALE_C = Quantity(
        type=str,
        shape=[],
        description='''
        scales the correlation part of the functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_LYP_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_OPTX(MSection):
    '''
    Uses the OPTX functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_OPTX_SCALE_X = Quantity(
        type=str,
        shape=[],
        description='''
        scales the exchange part of the functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_OPTX_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_P86C(MSection):
    '''
    Uses the P86C functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_P86C_SCALE_C = Quantity(
        type=str,
        shape=[],
        description='''
        scales the correlation part of the functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_P86C_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_PADE(MSection):
    '''
    Uses the PADE functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_PADE_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_PBE_HOLE_T_C_LR(MSection):
    '''
    PBE exchange hole model in trucanted coulomb potential
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_PBE_HOLE_T_C_LR_CUTOFF_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines cutoff for lower integration boundary
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_PBE_HOLE_T_C_LR_SCALE_X = Quantity(
        type=str,
        shape=[],
        description='''
        scales the exchange part of the functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_PBE_HOLE_T_C_LR_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_PBE(MSection):
    '''
    Uses the PBE functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_PBE_PARAMETRIZATION = Quantity(
        type=str,
        shape=[],
        description='''
        switches between the different parametrizations of the functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_PBE_SCALE_C = Quantity(
        type=str,
        shape=[],
        description='''
        scales the correlation part of the functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_PBE_SCALE_X = Quantity(
        type=str,
        shape=[],
        description='''
        scales the exchange part of the functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_PBE_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_PW92(MSection):
    '''
    Uses the PerdewWang correlation functional.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_PW92_PARAMETRIZATION = Quantity(
        type=str,
        shape=[],
        description='''
        Which one of parametrizations should be used
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_PW92_SCALE = Quantity(
        type=str,
        shape=[],
        description='''
        Scaling of the energy functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_PW92_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_PZ81(MSection):
    '''
    Uses the PZ functional.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_PZ81_PARAMETRIZATION = Quantity(
        type=str,
        shape=[],
        description='''
        Which one of parametrizations should be used
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_PZ81_SCALE_C = Quantity(
        type=str,
        shape=[],
        description='''
        scales the correlation part of the functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_PZ81_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_TFW(MSection):
    '''
    Uses the TFW functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_TFW_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_TF(MSection):
    '''
    Uses the TF functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_TF_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_TPSS(MSection):
    '''
    Uses the TPSS functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_TPSS_SCALE_C = Quantity(
        type=str,
        shape=[],
        description='''
        scales the correlation part of the functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_TPSS_SCALE_X = Quantity(
        type=str,
        shape=[],
        description='''
        scales the exchange part of the functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_TPSS_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_VWN(MSection):
    '''
    Uses the VWN functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_VWN_FUNCTIONAL_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Which version of the VWN functional should be used
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_VWN_SCALE_C = Quantity(
        type=str,
        shape=[],
        description='''
        scales the correlation part of the functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_VWN_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_XALPHA(MSection):
    '''
    Uses the XALPHA (SLATER) functional.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_XALPHA_SCALE_X = Quantity(
        type=str,
        shape=[],
        description='''
        scales the exchange part of the functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_XALPHA_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_XALPHA_XA = Quantity(
        type=str,
        shape=[],
        description='''
        Value of the xa parameter (this does not change the exponent, just the mixing)
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_XGGA(MSection):
    '''
    Uses one of the XGGA functionals (optimized versions of some of these functionals
    might be available outside this section).
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_XGGA_FUNCTIONAL = Quantity(
        type=str,
        shape=[],
        description='''
        Which one of the XGGA functionals should be used
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_XGGA_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_XWPBE(MSection):
    '''
    Uses the short range PBE functional
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_XWPBE_OMEGA = Quantity(
        type=str,
        shape=[],
        description='''
        screening parameter
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_XWPBE_SCALE_X0 = Quantity(
        type=str,
        shape=[],
        description='''
        scales the exchange part of the original hole PBE-functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_XWPBE_SCALE_X = Quantity(
        type=str,
        shape=[],
        description='''
        scales the exchange part of the functional
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_XWPBE_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        activates the functional
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL(MSection):
    '''
    The xc functional to use
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Shortcut for the most common functional combinations.
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_BECKE88_LR_ADIABATIC = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_BECKE88_LR_ADIABATIC'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_BECKE88_LR = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_BECKE88_LR'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_BECKE88 = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_BECKE88'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_BECKE97 = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_BECKE97'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_BECKE_ROUSSEL = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_BECKE_ROUSSEL'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_BEEF = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_BEEF'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_CS1 = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_CS1'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_GV09 = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_GV09'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_HCTH = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_HCTH'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_KE_GGA = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_KE_GGA'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_KE_LIBXC = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_KE_LIBXC'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_LDA_HOLE_T_C_LR = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_LDA_HOLE_T_C_LR'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_LIBXC = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_LIBXC'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_LYP_ADIABATIC = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_LYP_ADIABATIC'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_LYP = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_LYP'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_OPTX = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_OPTX'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_P86C = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_P86C'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_PADE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_PADE'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_PBE_HOLE_T_C_LR = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_PBE_HOLE_T_C_LR'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_PBE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_PBE'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_PW92 = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_PW92'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_PZ81 = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_PZ81'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_TFW = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_TFW'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_TF = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_TF'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_TPSS = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_TPSS'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_VWN = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_VWN'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_XALPHA = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_XALPHA'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_XGGA = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_XGGA'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_XWPBE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL_XWPBE'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_GRID(MSection):
    '''
    The xc parameters used when calculating the xc on the grid
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_GRID_USE_FINER_GRID = Quantity(
        type=str,
        shape=[],
        description='''
        Uses a finer grid only to calculate the xc
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_GRID_XC_DERIV = Quantity(
        type=str,
        shape=[],
        description='''
        The method used to compute the derivatives
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_GRID_XC_SMOOTH_RHO = Quantity(
        type=str,
        shape=[],
        description='''
        The density smoothing used for the xc calculation
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_POTENTIAL_SAOP(MSection):
    '''
    Uses the SAOP potential
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_POTENTIAL_SAOP_ALPHA = Quantity(
        type=str,
        shape=[],
        description='''
        Value of the alpha parameter (default = 1.19).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_POTENTIAL_SAOP_BETA = Quantity(
        type=str,
        shape=[],
        description='''
        Value of the beta parameter (default = 0.01).
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_POTENTIAL_SAOP_K_RHO = Quantity(
        type=str,
        shape=[],
        description='''
        Value of the K_rho parameter (default = 0.42).
        ''')


class x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_POTENTIAL(MSection):
    '''
    The xc potential to use (CAREFUL: xc potential here refers to potentials that are not
    derived from an xc functional, but rather are modelled directly. Therefore there is no
    consistent xc energy available. To still get an energy expression, see ENERGY below
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_XC_POTENTIAL_ENERGY = Quantity(
        type=str,
        shape=[],
        description='''
        How to determine the total energy.
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_POTENTIAL_SAOP = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_POTENTIAL_SAOP'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT_XC(MSection):
    '''
    parameters needed calculate the xc potential
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_XC_DENSITY_CUTOFF = Quantity(
        type=str,
        shape=[],
        description='''
        The cutoff on the density used by the xc calculation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_DENSITY_SMOOTH_CUTOFF_RANGE = Quantity(
        type=str,
        shape=[],
        description='''
        Parameter for the smoothing procedure inxc calculation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_FUNCTIONAL_ROUTINE = Quantity(
        type=str,
        shape=[],
        description='''
        Select the code for xc calculation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_GRADIENT_CUTOFF = Quantity(
        type=str,
        shape=[],
        description='''
        The cutoff on the gradient of the density used by the xc calculation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_XC_TAU_CUTOFF = Quantity(
        type=str,
        shape=[],
        description='''
        The cutoff on tau used by the xc calculation
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_ADIABATIC_RESCALING = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_ADIABATIC_RESCALING'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_HF = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_HF'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_VDW_POTENTIAL = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_VDW_POTENTIAL'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_WF_CORRELATION'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_FUNCTIONAL'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_GRID = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_GRID'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_POTENTIAL = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC_XC_POTENTIAL'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_DFT(MSection):
    '''
    parameter needed by dft programs
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_DFT_BASIS_SET_FILE_NAME = Quantity(
        type=str,
        shape=[],
        description='''
        Name of the basis set file, may include a path
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_CHARGE = Quantity(
        type=str,
        shape=[],
        description='''
        The total charge of the system
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_EXCITATIONS = Quantity(
        type=str,
        shape=[],
        description='''
        If excitations should be calculated
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_GEMINAL_FILE_NAME = Quantity(
        type=str,
        shape=[],
        description='''
        Name of the geminal basis set file, may include a path
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_MULTIPLICITY = Quantity(
        type=str,
        shape=[],
        description='''
        Two times the total spin plus one. Specify 3 for a triplet, 4 for a quartet, and
        so on. Default is 1 (singlet) for an even number and 2 (doublet) for an odd number
        of electrons.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_PLUS_U_METHOD = Quantity(
        type=str,
        shape=[],
        description='''
        Method employed for the calculation of the DFT+U contribution
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_POTENTIAL_FILE_NAME = Quantity(
        type=str,
        shape=[],
        description='''
        Name of the pseudo potential file, may include a path
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_RELAX_MULTIPLICITY = Quantity(
        type=str,
        shape=[],
        description='''
        Do not enforce the occupation of alpha and beta MOs due to the initially defined
        multiplicity, but rather follow an Aufbau principle. A threshold value greater
        than zero activates this option. Larger threshold values increase the probability
        for a spin flip. This option is only valid for unrestricted (i.e. spin polarised)
        Kohn-Sham (UKS) calculations.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_ROKS = Quantity(
        type=str,
        shape=[],
        description='''
        Requests a restricted open Kohn-Sham calculation
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SUBCELLS = Quantity(
        type=str,
        shape=[],
        description='''
        Read the grid size for subcell generation in the construction of neighbor lists.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SURF_DIP_DIR = Quantity(
        type=str,
        shape=[],
        description='''
        Cartesian axis parallel to surface normal.
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_SURFACE_DIPOLE_CORRECTION = Quantity(
        type=str,
        shape=[],
        description='''
        For slab calculations with asymmetric geometries, activate the correction of  the
        electrostatic potential with by compensating for the surface dipole. Implemented
        only for slabs with normal  parallel to one Cartesian axis. The normal direction
        is given by the keyword SURF_DIP_DIR
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_UKS = Quantity(
        type=str,
        shape=[],
        description='''
        Requests a spin-polarized calculation using alpha and beta orbitals, i.e. no spin
        restriction is applied
        ''')

    x_cp2k_input_FORCE_EVAL_DFT_WFN_RESTART_FILE_NAME = Quantity(
        type=str,
        shape=[],
        description='''
        Name of the wavefunction restart file, may include a path. If no file is
        specified, the default is to open the file as generated by the wfn restart print
        key.
        ''')

    x_cp2k_section_input_FORCE_EVAL_DFT_ALMO_SCF = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_ALMO_SCF'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_AUXILIARY_DENSITY_MATRIX_METHOD = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_AUXILIARY_DENSITY_MATRIX_METHOD'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_DENSITY_FITTING = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_DENSITY_FITTING'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_EFIELD = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_EFIELD'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_EXTERNAL_DENSITY = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_EXTERNAL_DENSITY'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_EXTERNAL_POTENTIAL = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_EXTERNAL_POTENTIAL'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_EXTERNAL_VXC = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_EXTERNAL_VXC'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_KG_METHOD = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_KG_METHOD'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_KPOINTS = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_KPOINTS'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_LOCALIZE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_LOCALIZE'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_LOW_SPIN_ROKS = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_LOW_SPIN_ROKS'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_LS_SCF = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_LS_SCF'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_MGRID = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_MGRID'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_PERIODIC_EFIELD = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_PERIODIC_EFIELD'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_POISSON = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_POISSON'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_QS = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_QS'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_REAL_TIME_PROPAGATION = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_REAL_TIME_PROPAGATION'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_RELATIVISTIC = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_RELATIVISTIC'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_SCCS = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_SCCS'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_SCF = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_SCF'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_SCRF = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_SCRF'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_SIC = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_SIC'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TDDFPT'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_TRANSPORT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_TRANSPORT'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XAS = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XAS'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_DFT_XC = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_DFT_XC'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_EIP(MSection):
    '''
    This section contains all information to run an Empirical Interatomic Potential (EIP)
    calculation.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_EIP_EIP_MODEL = Quantity(
        type=str,
        shape=[],
        description='''
        Selects the empirical interaction potential model
        ''')


class x_cp2k_section_input_FORCE_EVAL_EP(MSection):
    '''
    parameter needed by an ep calculation
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_EP_AT_PER_MOL = Quantity(
        type=str,
        shape=[],
        description='''
        Number of atoms in each molecule (at the moment only uniform system cam be
        handled)
        ''')

    x_cp2k_input_FORCE_EVAL_EP_COMP_INPUT = Quantity(
        type=str,
        shape=[],
        description='''
        Path to the input to be used for the component of the main system
        ''')

    x_cp2k_input_FORCE_EVAL_EP_E0_ONLY = Quantity(
        type=str,
        shape=[],
        description='''
        If only e0 should be calculated
        ''')

    x_cp2k_input_FORCE_EVAL_EP_EPS_LIN_SOLV = Quantity(
        type=str,
        shape=[],
        description='''
        Requested convergence of the linear solver (for psi1)
        ''')

    x_cp2k_input_FORCE_EVAL_EP_ROTATE = Quantity(
        type=str,
        shape=[],
        description='''
        If rotations from a unique set of coefficients should be used or if single
        molecule optimizations generate it
        ''')

    x_cp2k_input_FORCE_EVAL_EP_START_COEFFS = Quantity(
        type=str,
        shape=[],
        description='''
        Starting coefficients for roatation based ep
        ''')


class x_cp2k_section_input_FORCE_EVAL_EXTERNAL_POTENTIAL(MSection):
    '''
    Section controlling the presence of an external potential dependent  on the atomic
    positions (X,Y,Z)
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_EXTERNAL_POTENTIAL_ATOMS_LIST = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the atoms on which the external potential will act
        ''')

    x_cp2k_input_FORCE_EVAL_EXTERNAL_POTENTIAL_DX = Quantity(
        type=str,
        shape=[],
        description='''
        Parameter used for computing the derivative with the Ridders method.
        ''')

    x_cp2k_input_FORCE_EVAL_EXTERNAL_POTENTIAL_ERROR_LIMIT = Quantity(
        type=str,
        shape=[],
        description='''
        Checks that the error in computing the derivative is not larger than the value
        set. In case prints a warning message.
        ''')

    x_cp2k_input_FORCE_EVAL_EXTERNAL_POTENTIAL_FUNCTION = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the functional form in mathematical notation. Variables must be the
        atomic coordinates (X,Y,Z).
        ''')

    x_cp2k_input_FORCE_EVAL_EXTERNAL_POTENTIAL_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the parameters of the functional form
        ''')

    x_cp2k_input_FORCE_EVAL_EXTERNAL_POTENTIAL_UNITS = Quantity(
        type=str,
        shape=[],
        description='''
        Optionally, allows to define valid CP2K unit strings for each parameter value. It
        is assumed that the corresponding parameter value is specified in this unit.
        ''')

    x_cp2k_input_FORCE_EVAL_EXTERNAL_POTENTIAL_VALUES = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the values of  parameter of the functional form
        ''')


class x_cp2k_section_input_FORCE_EVAL_MIXED_COUPLING(MSection):
    '''
    Coupling between two force_eval: E=(E1+E2 - sqrt((E1-E2)**2+4*H12**2))/2
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MIXED_COUPLING_COUPLING_PARAMETER = Quantity(
        type=str,
        shape=[],
        description='''
        Coupling parameter H12 used in the coupling
        ''')


class x_cp2k_section_input_FORCE_EVAL_MIXED_GENERIC(MSection):
    '''
    User driven coupling between two or more force_eval.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MIXED_GENERIC_DX = Quantity(
        type=str,
        shape=[],
        description='''
        Parameter used for computing the derivative with the Ridders method.
        ''')

    x_cp2k_input_FORCE_EVAL_MIXED_GENERIC_ERROR_LIMIT = Quantity(
        type=str,
        shape=[],
        description='''
        Checks that the error in computing the derivative is not larger than the value
        set. In case prints a warning message.
        ''')

    x_cp2k_input_FORCE_EVAL_MIXED_GENERIC_MIXING_FUNCTION = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the mixing functional form in mathematical notation.
        ''')

    x_cp2k_input_FORCE_EVAL_MIXED_GENERIC_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the parameters of the functional form
        ''')

    x_cp2k_input_FORCE_EVAL_MIXED_GENERIC_UNITS = Quantity(
        type=str,
        shape=[],
        description='''
        Optionally, allows to define valid CP2K unit strings for each parameter value. It
        is assumed that the corresponding parameter value is specified in this unit.
        ''')

    x_cp2k_input_FORCE_EVAL_MIXED_GENERIC_VALUES = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the values of parameter of the functional form
        ''')

    x_cp2k_input_FORCE_EVAL_MIXED_GENERIC_VARIABLES = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the variables of the functional form. To allow an efficient mapping the
        order of the energy variables will be considered identical to the order of the
        force_eval in the force_eval_order list.
        ''')


class x_cp2k_section_input_FORCE_EVAL_MIXED_LINEAR(MSection):
    '''
    Linear combination between two force_eval:  F= lambda F1 + (1-lambda) F2
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MIXED_LINEAR_LAMBDA = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the mixing parameter lambda in the formula:
        ''')


class x_cp2k_section_input_FORCE_EVAL_MIXED_MAPPING_FORCE_EVAL_FRAGMENT(MSection):
    '''
    Fragment definition
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MIXED_MAPPING_FORCE_EVAL_FRAGMENT_DEFAULT_KEYWORD = Quantity(
        type=str,
        shape=[],
        description='''
        Starting and ending atomic index defining one fragment must be provided
        ''')

    x_cp2k_input_FORCE_EVAL_MIXED_MAPPING_FORCE_EVAL_FRAGMENT_MAP = Quantity(
        type=str,
        shape=[],
        description='''
        Provides the index of the fragment of the MIXED force_eval mapped on the locally
        defined fragment.
        ''')

    x_cp2k_input_FORCE_EVAL_MIXED_MAPPING_FORCE_EVAL_FRAGMENT_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the index of the fragment defined
        ''')


class x_cp2k_section_input_FORCE_EVAL_MIXED_MAPPING_FORCE_EVAL_MIXED_FRAGMENT(MSection):
    '''
    Fragment definition
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MIXED_MAPPING_FORCE_EVAL_MIXED_FRAGMENT_DEFAULT_KEYWORD = Quantity(
        type=str,
        shape=[],
        description='''
        Starting and ending atomic index defining one fragment must be provided
        ''')

    x_cp2k_input_FORCE_EVAL_MIXED_MAPPING_FORCE_EVAL_MIXED_FRAGMENT_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the index of the fragment defined
        ''')


class x_cp2k_section_input_FORCE_EVAL_MIXED_MAPPING_FORCE_EVAL_MIXED(MSection):
    '''
    Defines the fragments for the mixed force_eval (reference)
    '''

    m_def = Section(validate=False)

    x_cp2k_section_input_FORCE_EVAL_MIXED_MAPPING_FORCE_EVAL_MIXED_FRAGMENT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MIXED_MAPPING_FORCE_EVAL_MIXED_FRAGMENT'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_MIXED_MAPPING_FORCE_EVAL(MSection):
    '''
    Defines the fragments and the mapping for each force_eval (an integer index (ID) needs
    to be provided as parameter)
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MIXED_MAPPING_FORCE_EVAL_DEFINE_FRAGMENTS = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the fragments definition of the force_eval through the fragments of the
        force_eval_mixed. This avoids the pedantic definition of the fragments for the
        force_eval, assuming the order of the fragments for the specified force_eval is
        the same as the sequence  of integers provided. Easier to USE should be preferred
        to the specification of the single fragments.
        ''')

    x_cp2k_input_FORCE_EVAL_MIXED_MAPPING_FORCE_EVAL_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the index of the force_eval for which fragments and mappings are provided
        ''')

    x_cp2k_section_input_FORCE_EVAL_MIXED_MAPPING_FORCE_EVAL_FRAGMENT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MIXED_MAPPING_FORCE_EVAL_FRAGMENT'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_MIXED_MAPPING(MSection):
    '''
    Defines the mapping of atoms for the different force_eval with the mixed force_eval.
    The default is to have a mapping 1-1 between atom index (i.e. all force_eval share the
    same geometrical structure). The mapping is based on defining fragments and the
    mapping the  fragments between the several force_eval and the mixed force_eval
    '''

    m_def = Section(validate=False)

    x_cp2k_section_input_FORCE_EVAL_MIXED_MAPPING_FORCE_EVAL_MIXED = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MIXED_MAPPING_FORCE_EVAL_MIXED'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_MIXED_MAPPING_FORCE_EVAL = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MIXED_MAPPING_FORCE_EVAL'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_MIXED_RESTRAINT(MSection):
    '''
    Restraint between two force_eval: E = E1 + k*(E1-E2-t)**2
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MIXED_RESTRAINT_RESTRAINT_STRENGTH = Quantity(
        type=str,
        shape=[],
        description='''
        Strength of the restraint (k) in k*(E1-E2-t)**2
        ''')

    x_cp2k_input_FORCE_EVAL_MIXED_RESTRAINT_RESTRAINT_TARGET = Quantity(
        type=str,
        shape=[],
        description='''
        Target value of the restraint (t)
        ''')


class x_cp2k_section_input_FORCE_EVAL_MIXED(MSection):
    '''
    This section contains all information to run with a hamiltonian defined by a mixing of
    force_evals
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MIXED_GROUP_PARTITION = Quantity(
        type=str,
        shape=[],
        description='''
        gives the exact number of processors for each group. If not specified processors
        allocated will be equally distributed for the specified subforce_eval, trying to
        build a number of groups equal to the number of subforce_eval specified.
        ''')

    x_cp2k_input_FORCE_EVAL_MIXED_MIXING_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        The type of mixing to be employed
        ''')

    x_cp2k_input_FORCE_EVAL_MIXED_NGROUPS = Quantity(
        type=str,
        shape=[],
        description='''
        Gives the wanted number of groups. If not specified the number of groups is set to
        the number of subforce_eval defined.
        ''')

    x_cp2k_section_input_FORCE_EVAL_MIXED_COUPLING = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MIXED_COUPLING'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_MIXED_GENERIC = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MIXED_GENERIC'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_MIXED_LINEAR = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MIXED_LINEAR'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_MIXED_MAPPING = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MIXED_MAPPING'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_MIXED_RESTRAINT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MIXED_RESTRAINT'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_BEND_UB(MSection):
    '''
    Specifies the Urey-Bradley potential between the external atoms defining the angle
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_BEND_UB_CS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the cubic stretch term.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_BEND_UB_KIND = Quantity(
        type=str,
        shape=[],
        description='''
        Define the kind of Urey-Bradleypotential.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_BEND_UB_K = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the force constant of the potential. For MORSE potentials 2 numbers are
        expected. For QUARTIC potentials 3 numbers are expected.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_BEND_UB_R0 = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the equilibrium distance.
        ''')


class x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_BEND(MSection):
    '''
    Specifies the bend potential of the MM system.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_BEND_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the atomic kinds involved in the bend.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_BEND_CB = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the the cubic force constant of the bend
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_BEND_KBS12 = Quantity(
        type=str,
        shape=[],
        description='''
        Mixed bend stretch parameter
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_BEND_KBS32 = Quantity(
        type=str,
        shape=[],
        description='''
        Mixed bend stretch parameter
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_BEND_KIND = Quantity(
        type=str,
        shape=[],
        description='''
        Define the kind of bend potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_BEND_KSS = Quantity(
        type=str,
        shape=[],
        description='''
        Mixed bend stretch parameter
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_BEND_K = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the force constant of the potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_BEND_R012 = Quantity(
        type=str,
        shape=[],
        description='''
        Mixed bend stretch parameter
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_BEND_R032 = Quantity(
        type=str,
        shape=[],
        description='''
        Mixed bend stretch parameter
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_BEND_THETA0 = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the equilibrium angle.
        ''')

    x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_BEND_UB = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_BEND_UB'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_BOND(MSection):
    '''
    Specifies the bond potential
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_BOND_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the atomic kinds involved in the bond.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_BOND_CS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the cubic stretch term.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_BOND_KIND = Quantity(
        type=str,
        shape=[],
        description='''
        Define the kind of Bondpotential.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_BOND_K = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the force constant of the potential. For MORSE potentials 2 numbers are
        expected. For QUARTIC potentials 3 numbers are expected.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_BOND_R0 = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the equilibrium distance.
        ''')


class x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_CHARGES(MSection):
    '''
    Allow to specify an array of classical charges, thus avoiding the packing and
    permitting the usage of different charges for same atomic types.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_CHARGES_DEFAULT_KEYWORD = Quantity(
        type=str,
        shape=[],
        description='''
        Value of the charge for the individual atom. Order MUST reflect the one specified
        for the geometry.
        ''')


class x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_CHARGE(MSection):
    '''
    This section specifies the charge of the MM atoms
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_CHARGE_ATOM = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the atomic kind of the charge.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_CHARGE_CHARGE = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the charge of the MM atom in electron charge unit.
        ''')


class x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_DIPOLE_DAMPING(MSection):
    '''
    This section specifies optional electric field damping for the polarizable atoms.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_DIPOLE_DAMPING_ATOM = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the atomic kind for this damping function.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_DIPOLE_DAMPING_BIJ = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the BIJ parameter for this damping.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_DIPOLE_DAMPING_CIJ = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the CIJ parameter for this damping.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_DIPOLE_DAMPING_ORDER = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the order for this damping.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_DIPOLE_DAMPING_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the damping type.
        ''')


class x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_DIPOLE(MSection):
    '''
    This section specifies that we will perform an SCF dipole calculation of the MM atoms.
    Needs KEYWORD POL_SCF in POISSON secton
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_DIPOLE_APOL = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the isotropic polarizability of the MM atom.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_DIPOLE_ATOM = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the atomic kind of the scf dipole.
        ''')

    x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_DIPOLE_DAMPING = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_DIPOLE_DAMPING'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_IMPROPER(MSection):
    '''
    Specifies the improper torsion potential of the MM system.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_IMPROPER_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the atomic kinds involved in the improper tors.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_IMPROPER_KIND = Quantity(
        type=str,
        shape=[],
        description='''
        Define the kind of improper torsion potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_IMPROPER_K = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the force constant of the potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_IMPROPER_PHI0 = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the phase of the potential.
        ''')


class x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BMHFTD(MSection):
    '''
    This section specifies the input parameters for BMHFTD potential type.Functional form:
    V(r) = A*exp(-B*r) - f_6*(r)C/r^6 - f_8(r)*D/r^8.where f_order(r)=1-exp(-BD * r) *
    \\sum_{k=0}^order (BD * r)^k / k! .(Tang-Toennies damping function)No values available
    inside cp2k.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BMHFTD_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the atomic kind involved in the BMHFTD nonbond potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BMHFTD_A = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the A parameter of the dispersion-damped Fumi-Tosi Potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BMHFTD_BD = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the BD parameter of the dispersion-damped Fumi-Tosi Potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BMHFTD_B = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the B parameter of the dispersion-damped Fumi-Tosi Potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BMHFTD_C = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the C parameter of the dispersion-damped Fumi-Tosi Potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BMHFTD_D = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the D parameter of the dispersion-damped Fumi-Tosi Potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BMHFTD_MAP_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the kinds for which internally is defined the BMHFTD nonbond potential at
        the moment no species included.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BMHFTD_ORDER = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the order for this damping.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BMHFTD_RCUT = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the cutoff parameter of the BMHFTD potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BMHFTD_RMAX = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the upper bound of the potential. If not set the range is the full range
        generate by the spline
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BMHFTD_RMIN = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the lower bound of the potential. If not set the range is the full range
        generate by the spline
        ''')


class x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BMHFT(MSection):
    '''
    This section specifies the input parameters for BMHFT potential type.Functional form:
    V(r) = A * EXP(-B*r) - C/r^6 - D/r^8.Values available inside cp2k only for the Na/Cl
    pair.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BMHFT_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the atomic kind involved in the BMHFT nonbond potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BMHFT_A = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the A parameter of the Fumi-Tosi Potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BMHFT_B = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the B parameter of the Fumi-Tosi Potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BMHFT_C = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the C parameter of the Fumi-Tosi Potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BMHFT_D = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the D parameter of the Fumi-Tosi Potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BMHFT_MAP_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the kinds for which internally is defined the BMHFT nonbond potential at
        the moment only Na and Cl.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BMHFT_RCUT = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the cutoff parameter of the BMHFT potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BMHFT_RMAX = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the upper bound of the potential. If not set the range is the full range
        generate by the spline
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BMHFT_RMIN = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the lower bound of the potential. If not set the range is the full range
        generate by the spline
        ''')


class x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BUCK4RANGES(MSection):
    '''
    This section specifies the input parameters for the Buckingham 4-ranges potential
    type. Functional form:<ul><li>V(r) = A*EXP(-B*r) for r &#60;
    r<sub>1</sub></li><li>V(r) = Sum_n POLY1(n)*r<sup>n</sup> for r<sub>1</sub> ≤ r &#60;
    r<sub>2</sub></li><li>V(r) = Sum_n POLY2(n)*r<sup>n</sup> for r<sub>2</sub> ≤ r &#60;
    r<sub>3</sub></li><li>V(r) = -C/r<sup>6</sup> for r ≥ r<sub>3</sub></li></ul>
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BUCK4RANGES_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the atomic kind involved in the nonbond potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BUCK4RANGES_A = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the A parameter of the Buckingham potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BUCK4RANGES_B = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the B parameter of the Buckingham potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BUCK4RANGES_C = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the C parameter of the Buckingham  potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BUCK4RANGES_POLY1 = Quantity(
        type=str,
        shape=[],
        description='''
        Coefficients of the polynomial used in the second rangeThis keyword can be
        repeated several times.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BUCK4RANGES_POLY2 = Quantity(
        type=str,
        shape=[],
        description='''
        Coefficients of the polynomial used in the third rangeThis keyword can be repeated
        several times.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BUCK4RANGES_R1 = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the upper bound of the first range
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BUCK4RANGES_R2 = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the upper bound of the second range
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BUCK4RANGES_R3 = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the upper bound of the third range
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BUCK4RANGES_RCUT = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the cutoff parameter of the Buckingham potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BUCK4RANGES_RMAX = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the upper bound of the potential. If not set the range is the full range
        generate by the spline
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BUCK4RANGES_RMIN = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the lower bound of the potential. If not set the range is the full range
        generate by the spline
        ''')


class x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BUCKMORSE(MSection):
    '''
    This section specifies the input parameters for Buckingham plus Morse potential type
    Functional Form: V(r) =
    F0*(B1+B2)*EXP([A1+A2-r]/[B1+B2])-C/r^6+D*{EXP[-2*beta*(r-R0)]-2*EXP[-beta*(r-R0)]}.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BUCKMORSE_A1 = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the A1 parameter of Buckingham+Morse potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BUCKMORSE_A2 = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the A2 parameter of Buckingham+Morse potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BUCKMORSE_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the atomic kind involved in the nonbond potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BUCKMORSE_B1 = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the B1 parameter of Buckingham+Morse potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BUCKMORSE_B2 = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the B2 parameter of Buckingham+Morse potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BUCKMORSE_BETA = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the width for the Morse part
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BUCKMORSE_C = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the C parameter of Buckingham+Morse  potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BUCKMORSE_D = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the amplitude for the Morse part
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BUCKMORSE_F0 = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the f0 parameter of Buckingham+Morse potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BUCKMORSE_R0 = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the equilibrium distance for the Morse part
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BUCKMORSE_RCUT = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the cutoff parameter of the Buckingham potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BUCKMORSE_RMAX = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the upper bound of the potential. If not set the range is the full range
        generate by the spline
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BUCKMORSE_RMIN = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the lower bound of the potential. If not set the range is the full range
        generate by the spline
        ''')


class x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_EAM(MSection):
    '''
    This section specifies the input parameters for EAM potential type.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_EAM_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the atomic kind involved in the nonbond potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_EAM_PARM_FILE_NAME = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the filename that contains the tabulated EAM potential. File structure:
        the first line of the potential file contains a title. The second line contains:
        atomic number, mass and lattice constant. These information are parsed but not
        used in CP2K. The third line contains: dr: increment of r for the tabulated values
        of density and phi (assuming r starts in 0) [angstrom]; drho: increment of density
        for the tabulated values of the embedding function (assuming rho starts in 0)
        [au_c]; cutoff: cutoff of the EAM potential; npoints: number of points in
        tabulated. Follow in order npoints lines for rho [au_c] and its derivative
        [au_c*angstrom^-1]; npoints lines for PHI [ev] and its derivative [ev*angstrom^-1]
        and npoint lines for the embedded function [ev] and its derivative [ev*au_c^-1].
        ''')


class x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_GENPOT(MSection):
    '''
    This section specifies the input parameters for a generic potential type.A functional
    form is specified. Mathematical Operators recognized are +, -, *, /, ** or
    alternatively ^, whereas symbols for brackets must be (). The function parser
    recognizes the (single argument) Fortran 90 intrinsic functions abs, exp, log10, log,
    sqrt, sinh, cosh, tanh, sin, cos, tan, asin, acos, atan. Parsing for INTRINSIC
    functions is CASE INsensitive.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_GENPOT_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the atomic kind involved in the generic potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_GENPOT_FUNCTION = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the functional form in mathematical notation.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_GENPOT_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the parameters of the functional form
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_GENPOT_RCUT = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the cutoff parameter of the generic potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_GENPOT_RMAX = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the upper bound of the potential. If not set the range is the full range
        generate by the spline
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_GENPOT_RMIN = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the lower bound of the potential. If not set the range is the full range
        generate by the spline
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_GENPOT_UNITS = Quantity(
        type=str,
        shape=[],
        description='''
        Optionally, allows to define valid CP2K unit strings for each parameter value. It
        is assumed that the corresponding parameter value is specified in this unit.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_GENPOT_VALUES = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the values of parameter of the functional form
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_GENPOT_VARIABLES = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the variable of the functional form.
        ''')


class x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_GOODWIN(MSection):
    '''
    This section specifies the input parameters for GOODWIN potential type.Functional
    form: V(r) = EXP(M*(-(r/DC)**MC+(D/DC)**MC))*VR0*(D/r)**M.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_GOODWIN_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the atomic kind involved in the nonbond potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_GOODWIN_DC = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the DC parameter of the Goodwin potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_GOODWIN_D = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the D parameter of the Goodwin potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_GOODWIN_MC = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the MC parameter of the Goodwin potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_GOODWIN_M = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the M parameter of the Goodwin potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_GOODWIN_RCUT = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the cutoff parameter of the Goodwin potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_GOODWIN_RMAX = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the upper bound of the potential. If not set the range is the full range
        generate by the spline
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_GOODWIN_RMIN = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the lower bound of the potential. If not set the range is the full range
        generate by the spline
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_GOODWIN_VR0 = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the VR0 parameter of the Goodwin potential
        ''')


class x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_IPBV(MSection):
    '''
    This section specifies the input parameters for IPBV potential type.Functional form:
    Implicit table function.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_IPBV_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the atomic kind involved in the IPBV nonbond potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_IPBV_RCUT = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the cutoff parameter of the IPBV potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_IPBV_RMAX = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the upper bound of the potential. If not set the range is the full range
        generate by the spline
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_IPBV_RMIN = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the lower bound of the potential. If not set the range is the full range
        generate by the spline
        ''')


class x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_LENNARD_JONES(MSection):
    '''
    This section specifies the input parameters for LENNARD-JONES potential
    type.Functional form: V(r) = 4.0 * EPSILON * [(SIGMA/r)^12-(SIGMA/r)^6].
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_LENNARD_JONES_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the atomic kind involved in the nonbond potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_LENNARD_JONES_EPSILON = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the EPSILON parameter of the LJ potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_LENNARD_JONES_RCUT = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the cutoff parameter of the LJ potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_LENNARD_JONES_RMAX = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the upper bound of the potential. If not set the range is the full range
        generate by the spline
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_LENNARD_JONES_RMIN = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the lower bound of the potential. If not set the range is the full range
        generate by the spline
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_LENNARD_JONES_SIGMA = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the SIGMA parameter of the LJ potential
        ''')


class x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_QUIP(MSection):
    '''
    This section specifies the input parameters for QUIP potential type. Mainly intended
    for things like GAP corrections to DFT to achieve correlated-wavefunction-like
    accuracy. Requires linking with quip library from <a href="http://www.libatoms.org"
    target="_blank">http://www.libatoms.org</a> .
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_QUIP_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the atomic kinds involved in the QUIP potential. For more than 2 elements,
        &QUIP section must be repeated until each element has been mentioned at least
        once. Set IGNORE_MISSING_CRITICAL_PARAMS to T in enclosing &FORCEFIELD section to
        avoid having to list every pair of elements separately.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_QUIP_CALC_ARGS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the potential calculation arguments for the QUIP potential.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_QUIP_INIT_ARGS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the potential initialization arguments for the QUIP potential. If blank
        (default) first potential defined in QUIP parameter file will be used.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_QUIP_PARM_FILE_NAME = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the filename that contains the QUIP potential.
        ''')


class x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_SIEPMANN(MSection):
    '''
    This section specifies the input parameters for the Siepmann-Sprik potential type.
    Consist of 4 terms: T1+T2+T3+T4. The terms T1=A/rij^alpha and T2=-C/rij^6 have to be
    given via the GENPOT section. The terms T3+T4 are obtained from the SIEPMANN section.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_SIEPMANN_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the atomic kind involved in the nonbond potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_SIEPMANN_BETA = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the beta parameter of Siepmann potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_SIEPMANN_B = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the B parameter of Siepmann potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_SIEPMANN_D = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the D parameter of Siepmann potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_SIEPMANN_E = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the E parameter of Siepmann potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_SIEPMANN_F = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the F parameter of Siepmann potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_SIEPMANN_RCUT = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the cutoff parameter of Siepmann potential
        ''')


class x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_TERSOFF(MSection):
    '''
    This section specifies the input parameters for Tersoff potential type.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_TERSOFF_ALPHA = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the alpha parameter of Tersoff potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_TERSOFF_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the atomic kind involved in the nonbond potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_TERSOFF_A = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the A parameter of Tersoff potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_TERSOFF_BETA = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the beta parameter of Tersoff potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_TERSOFF_BIGD = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the D parameter of Tersoff potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_TERSOFF_BIGR = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the bigR parameter of Tersoff potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_TERSOFF_B = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the B parameter of Tersoff potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_TERSOFF_C = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the c parameter of Tersoff potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_TERSOFF_D = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the d parameter of Tersoff potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_TERSOFF_H = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the h parameter of Tersoff potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_TERSOFF_LAMBDA1 = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the lambda1 parameter of Tersoff potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_TERSOFF_LAMBDA2 = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the lambda2 parameter of Tersoff potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_TERSOFF_LAMBDA3 = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the lambda3 parameter of Tersoff potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_TERSOFF_N = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the n parameter of Tersoff potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_TERSOFF_RCUT = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the cutoff parameter of the tersoff potential.  This parameter is in
        principle already defined by the values of  bigD and bigR. But it is necessary to
        define it when using the tersoff  in conjuction with other potentials (for the
        same atomic pair) in order to have the same consistent definition of RCUT for all
        potentials.
        ''')


class x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_WILLIAMS(MSection):
    '''
    This section specifies the input parameters for WILLIAMS potential type.Functional
    form: V(r) = A*EXP(-B*r) - C / r^6 .
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_WILLIAMS_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the atomic kind involved in the nonbond potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_WILLIAMS_A = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the A parameter of the Williams potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_WILLIAMS_B = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the B parameter of the Williams potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_WILLIAMS_C = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the C parameter of the Williams potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_WILLIAMS_RCUT = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the cutoff parameter of the Williams potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_WILLIAMS_RMAX = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the upper bound of the potential. If not set the range is the full range
        generate by the spline
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_WILLIAMS_RMIN = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the lower bound of the potential. If not set the range is the full range
        generate by the spline
        ''')


class x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED14_GENPOT(MSection):
    '''
    This section specifies the input parameters for a generic potential type.A functional
    form is specified. Mathematical Operators recognized are +, -, *, /, ** or
    alternatively ^, whereas symbols for brackets must be (). The function parser
    recognizes the (single argument) Fortran 90 intrinsic functions abs, exp, log10, log,
    sqrt, sinh, cosh, tanh, sin, cos, tan, asin, acos, atan. Parsing for INTRINSIC
    functions is CASE INsensitive.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED14_GENPOT_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the atomic kind involved in the generic potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED14_GENPOT_FUNCTION = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the functional form in mathematical notation.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED14_GENPOT_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the parameters of the functional form
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED14_GENPOT_RCUT = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the cutoff parameter of the generic potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED14_GENPOT_RMAX = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the upper bound of the potential. If not set the range is the full range
        generate by the spline
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED14_GENPOT_RMIN = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the lower bound of the potential. If not set the range is the full range
        generate by the spline
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED14_GENPOT_UNITS = Quantity(
        type=str,
        shape=[],
        description='''
        Optionally, allows to define valid CP2K unit strings for each parameter value. It
        is assumed that the corresponding parameter value is specified in this unit.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED14_GENPOT_VALUES = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the values of parameter of the functional form
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED14_GENPOT_VARIABLES = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the variable of the functional form.
        ''')


class x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED14_GOODWIN(MSection):
    '''
    This section specifies the input parameters for GOODWIN potential type.Functional
    form: V(r) = EXP(M*(-(r/DC)**MC+(D/DC)**MC))*VR0*(D/r)**M.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED14_GOODWIN_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the atomic kind involved in the nonbond potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED14_GOODWIN_DC = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the DC parameter of the Goodwin potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED14_GOODWIN_D = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the D parameter of the Goodwin potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED14_GOODWIN_MC = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the MC parameter of the Goodwin potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED14_GOODWIN_M = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the M parameter of the Goodwin potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED14_GOODWIN_RCUT = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the cutoff parameter of the Goodwin potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED14_GOODWIN_RMAX = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the upper bound of the potential. If not set the range is the full range
        generate by the spline
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED14_GOODWIN_RMIN = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the lower bound of the potential. If not set the range is the full range
        generate by the spline
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED14_GOODWIN_VR0 = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the VR0 parameter of the Goodwin potential
        ''')


class x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED14_LENNARD_JONES(MSection):
    '''
    This section specifies the input parameters for LENNARD-JONES potential
    type.Functional form: V(r) = 4.0 * EPSILON * [(SIGMA/r)^12-(SIGMA/r)^6].
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED14_LENNARD_JONES_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the atomic kind involved in the nonbond potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED14_LENNARD_JONES_EPSILON = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the EPSILON parameter of the LJ potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED14_LENNARD_JONES_RCUT = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the cutoff parameter of the LJ potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED14_LENNARD_JONES_RMAX = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the upper bound of the potential. If not set the range is the full range
        generate by the spline
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED14_LENNARD_JONES_RMIN = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the lower bound of the potential. If not set the range is the full range
        generate by the spline
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED14_LENNARD_JONES_SIGMA = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the SIGMA parameter of the LJ potential
        ''')


class x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED14_WILLIAMS(MSection):
    '''
    This section specifies the input parameters for WILLIAMS potential type.Functional
    form: V(r) = A*EXP(-B*r) - C / r^6 .
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED14_WILLIAMS_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the atomic kind involved in the nonbond potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED14_WILLIAMS_A = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the A parameter of the Williams potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED14_WILLIAMS_B = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the B parameter of the Williams potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED14_WILLIAMS_C = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the C parameter of the Williams potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED14_WILLIAMS_RCUT = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the cutoff parameter of the Williams potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED14_WILLIAMS_RMAX = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the upper bound of the potential. If not set the range is the full range
        generate by the spline
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED14_WILLIAMS_RMIN = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the lower bound of the potential. If not set the range is the full range
        generate by the spline
        ''')


class x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED14(MSection):
    '''
    This section specifies the input parameters for 1-4 NON-BONDED interactions.
    '''

    m_def = Section(validate=False)

    x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED14_GENPOT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED14_GENPOT'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED14_GOODWIN = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED14_GOODWIN'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED14_LENNARD_JONES = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED14_LENNARD_JONES'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED14_WILLIAMS = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED14_WILLIAMS'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED(MSection):
    '''
    This section specifies the input parameters for NON-BONDED interactions.
    '''

    m_def = Section(validate=False)

    x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BMHFTD = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BMHFTD'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BMHFT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BMHFT'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BUCK4RANGES = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BUCK4RANGES'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BUCKMORSE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_BUCKMORSE'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_EAM = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_EAM'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_GENPOT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_GENPOT'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_GOODWIN = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_GOODWIN'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_IPBV = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_IPBV'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_LENNARD_JONES = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_LENNARD_JONES'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_QUIP = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_QUIP'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_SIEPMANN = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_SIEPMANN'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_TERSOFF = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_TERSOFF'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_WILLIAMS = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED_WILLIAMS'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_OPBEND(MSection):
    '''
    Specifies the out of plane bend potential of the MM system.(Only defined for atom
    quadruples which are also defined as an improper pattern in the topology.)
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_OPBEND_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the atomic kinds involved in the opbend.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_OPBEND_KIND = Quantity(
        type=str,
        shape=[],
        description='''
        Define the kind of out of plane bend potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_OPBEND_K = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the force constant of the potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_OPBEND_PHI0 = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the phase of the potential.
        ''')


class x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_QUADRUPOLE(MSection):
    '''
    This section specifies that we will perform an SCF quadrupole calculation of the MM
    atoms. Needs KEYWORD POL_SCF in POISSON secton
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_QUADRUPOLE_ATOM = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the atomic kind of the scf quadrupole.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_QUADRUPOLE_CPOL = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the isotropic polarizability of the MM atom.
        ''')


class x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_SHELL(MSection):
    '''
    This section specifies the parameters for shell-model potentials
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_SHELL_CORE_CHARGE = Quantity(
        type=str,
        shape=[],
        description='''
        Partial charge assigned to the core (electron charge units)
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_SHELL_K2_SPRING = Quantity(
        type=str,
        shape=[],
        description='''
        Force constant k2 of the spring potential 1/2*k2*r^2 + 1/24*k4*r^4 binding a core-
        shell pair when a core-shell potential is employed.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_SHELL_K4_SPRING = Quantity(
        type=str,
        shape=[],
        description='''
        Force constant k4 of the spring potential 1/2*k2*r^2 + 1/24*k4*r^4 binding a core-
        shell pair when a core-shell potential is employed. By default a harmonic spring
        potential is used, i.e. k4 is zero.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_SHELL_MASS_FRACTION = Quantity(
        type=str,
        shape=[],
        description='''
        Fraction of the mass of the atom to be assigned to the shell
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_SHELL_MAX_DISTANCE = Quantity(
        type=str,
        shape=[],
        description='''
        Assign a maximum elongation of the spring, if negative no limit is imposed
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_SHELL_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        The kind for which the shell potential parameters are given
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_SHELL_SHELL_CHARGE = Quantity(
        type=str,
        shape=[],
        description='''
        Partial charge assigned to the shell (electron charge units)
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_SHELL_SHELL_CUTOFF = Quantity(
        type=str,
        shape=[],
        description='''
        Define a screening function to exclude some neighbors  of the shell when
        electrostatic interaction are considered, if negative no screening is operated
        ''')


class x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_SPLINE(MSection):
    '''
    specifies parameters to set up the splines used in the nonboned interactions (both
    pair body potential and many body potential)
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_SPLINE_EMAX_ACCURACY = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the maximum value of energy used to check the accuracy requested through
        EPS_SPLINE. Energy values larger than EMAX_ACCURACY generally do not  satisfy the
        requested accuracy
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_SPLINE_EMAX_SPLINE = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the maximum value of the potential up to which splines will be constructed
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_SPLINE_EPS_SPLINE = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the threshold for the choice of the number of points used in the splines
        (comparing the splined value with the  analytically evaluated one)
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_SPLINE_NPOINTS = Quantity(
        type=str,
        shape=[],
        description='''
        Override the default search for an accurate spline by specifying a fixed number of
        spline points.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_SPLINE_R0_NB = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the minimum value of the distance interval  that brackets the value of
        emax_spline.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_SPLINE_RCUT_NB = Quantity(
        type=str,
        shape=[],
        description='''
        Cutoff radius for nonbonded interactions. This value overrides  the value
        specified in the potential definition and is global for all potentials.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_SPLINE_UNIQUE_SPLINE = Quantity(
        type=str,
        shape=[],
        description='''
        For few potentials (Lennard-Jones) one global optimal spline is generated instead
        of different optimal splines for each kind of potential
        ''')


class x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_TORSION(MSection):
    '''
    Specifies the torsion potential of the MM system.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_TORSION_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the atomic kinds involved in the tors.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_TORSION_KIND = Quantity(
        type=str,
        shape=[],
        description='''
        Define the kind of torsion potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_TORSION_K = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the force constant of the potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_TORSION_M = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the multiplicity of the potential.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_TORSION_PHI0 = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the phase of the potential.
        ''')


class x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD(MSection):
    '''
    Section specifying information regarding how to set up properly a force_field for the
    classical calculations.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_DO_NONBONDED = Quantity(
        type=str,
        shape=[],
        description='''
        Controls the computation of all the real-sapce (short-range) nonbonded
        interactions. This also includes the real-space corrections for excluded or scaled
        1-2, 1-3 and 1-4 interactions. When set to F, the neighborlists are not created
        and all interactions that depend on them are not computed.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_EI_SCALE14 = Quantity(
        type=str,
        shape=[],
        description='''
        Scaling factor for the electrostatics 1-4
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_IGNORE_MISSING_CRITICAL_PARAMS = Quantity(
        type=str,
        shape=[],
        description='''
        Do not abort when critical force-field parameters are missing. CP2K will run as if
        the terms containing the missing parameters are zero.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_MULTIPLE_POTENTIAL = Quantity(
        type=str,
        shape=[],
        description='''
        Enables the possibility to define NONBONDED and NONBONDED14 as a sum of different
        kinds of potential. Useful for piecewise defined potentials.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_PARM_FILE_NAME = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the filename that contains the parameters of the FF.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_PARMTYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Define the kind of torsion potential
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_SHIFT_CUTOFF = Quantity(
        type=str,
        shape=[],
        description='''
        Add a constant energy shift to the real-space non-bonding interactions (both Van
        der Waals and electrostatic) such that the energy at the cutoff radius is zero.
        This makes the non-bonding interactions continuous at the cutoff.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_VDW_SCALE14 = Quantity(
        type=str,
        shape=[],
        description='''
        Scaling factor for the VDW 1-4
        ''')

    x_cp2k_input_FORCE_EVAL_MM_FORCEFIELD_ZBL_SCATTERING = Quantity(
        type=str,
        shape=[],
        description='''
        A short range repulsive potential is added, to simulate collisions and scattering.
        ''')

    x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_BEND = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_BEND'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_BOND = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_BOND'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_CHARGES = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_CHARGES'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_CHARGE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_CHARGE'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_DIPOLE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_DIPOLE'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_IMPROPER = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_IMPROPER'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED14 = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED14'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_NONBONDED'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_OPBEND = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_OPBEND'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_QUADRUPOLE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_QUADRUPOLE'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_SHELL = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_SHELL'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_SPLINE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_SPLINE'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_TORSION = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD_TORSION'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_MM_NEIGHBOR_LISTS(MSection):
    '''
    This section specifies the input parameters for the construction of neighbor lists.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MM_NEIGHBOR_LISTS_GEO_CHECK = Quantity(
        type=str,
        shape=[],
        description='''
        This keyword enables the check that two atoms are never below the minimum value
        used to construct the splines during the construction of the neighbouring list.
        Disabling this keyword avoids CP2K to abort in case two atoms are below the
        minimum  value of the radius used to generate the splines.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_NEIGHBOR_LISTS_NEIGHBOR_LISTS_FROM_SCRATCH = Quantity(
        type=str,
        shape=[],
        description='''
        This keyword enables the building of the neighbouring list from scratch.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_NEIGHBOR_LISTS_VERLET_SKIN = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the Verlet Skin for the generation of the neighbor lists
        ''')


class x_cp2k_section_input_FORCE_EVAL_MM_POISSON_EWALD_MULTIPOLES(MSection):
    '''
    Enables the use of multipoles in the treatment of the electrostatics.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MM_POISSON_EWALD_MULTIPOLES_EPS_POL = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the rmsd threshold for the derivatives of the energy towards the Cartesian
        dipoles components
        ''')

    x_cp2k_input_FORCE_EVAL_MM_POISSON_EWALD_MULTIPOLES_MAX_IPOL_ITER = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the maximum number of iterations for induced dipoles
        ''')

    x_cp2k_input_FORCE_EVAL_MM_POISSON_EWALD_MULTIPOLES_MAX_MULTIPOLE_EXPANSION = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the maximum level of multipoles expansion used  for the electrostatics.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_POISSON_EWALD_MULTIPOLES_POL_SCF = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the method to obtain self consistent induced multipole moments.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_POISSON_EWALD_MULTIPOLES_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Controls the activation of the Multipoles
        ''')


class x_cp2k_section_input_FORCE_EVAL_MM_POISSON_EWALD_RS_GRID(MSection):
    '''
    Set options that influence how the realspace grids are being distributed in parallel
    runs.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MM_POISSON_EWALD_RS_GRID_DISTRIBUTION_LAYOUT = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the number of slices in the x, y and z directions.-1 specifies that any
        number of slices is OK.If a given distribution can not be satisfied, a replicated
        grid will result.Also see LOCK_DISTRIBUTION.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_POISSON_EWALD_RS_GRID_DISTRIBUTION_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Parallelization strategy.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_POISSON_EWALD_RS_GRID_HALO_REDUCTION_FACTOR = Quantity(
        type=str,
        shape=[],
        description='''
        Can be used to reduce the halo of the distributed grid (experimental features).
        ''')

    x_cp2k_input_FORCE_EVAL_MM_POISSON_EWALD_RS_GRID_LOCK_DISTRIBUTION = Quantity(
        type=str,
        shape=[],
        description='''
        Expert use only, only basic QS deals correctly with a non-default value.If the
        distribution is locked, a grid will have the same distribution asthe next finer
        multigrid (provided it is distributed).If unlocked, all grids can be distributed
        freely.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_POISSON_EWALD_RS_GRID_MAX_DISTRIBUTED_LEVEL = Quantity(
        type=str,
        shape=[],
        description='''
        If the multigrid-level of a grid is larger than the parameter, it will not be
        distributed in the automatic scheme.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_POISSON_EWALD_RS_GRID_MEMORY_FACTOR = Quantity(
        type=str,
        shape=[],
        description='''
        A grid will only be distributed if the memory usage for that grid (including halo)
        is smaller than a replicated grid by this parameter.
        ''')


class x_cp2k_section_input_FORCE_EVAL_MM_POISSON_EWALD(MSection):
    '''
    Ewald parameters controlling electrostatic only for CLASSICAL MM.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MM_POISSON_EWALD_ALPHA = Quantity(
        type=str,
        shape=[],
        description='''
        alpha parameter associated with Ewald (EWALD|PME|SPME). Recommended for small
        systems is is alpha = 3.5 / r_cut. Tuning alpha, r_cut and gmax is needed to
        obtain O(N**1.5) scaling for ewald.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_POISSON_EWALD_EPSILON = Quantity(
        type=str,
        shape=[],
        description='''
        tolerance of gaussians for fft interpolation (PME only)
        ''')

    x_cp2k_input_FORCE_EVAL_MM_POISSON_EWALD_EWALD_ACCURACY = Quantity(
        type=str,
        shape=[],
        description='''
        Expected accuracy in the Ewald sum. This number affects only the calculation of
        the cutoff for the real-space term of the ewald summation (EWALD|PME|SPME) as well
        as the construction of the neighbor lists (if the cutoff for non-bonded terms is
        smaller than the value employed to compute the EWALD real-space term). This
        keyword has no effect on the reciprocal space term (which can be tuned
        independently).
        ''')

    x_cp2k_input_FORCE_EVAL_MM_POISSON_EWALD_EWALD_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        The type of ewald you want to perform.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_POISSON_EWALD_GMAX = Quantity(
        type=str,
        shape=[],
        description='''
        number of grid points (SPME and EWALD). If a single number is specified,the same
        number of points is used for all three directions on the grid.If three numbers are
        given, each direction can have a different number of points.The number of points
        needs to be FFTable (which depends on the library used) and odd for EWALD.The
        optimal number depends e.g. on alpha and the size of the cell. 1 point per
        Angstrom is common.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_POISSON_EWALD_NS_MAX = Quantity(
        type=str,
        shape=[],
        description='''
        number of grid points on small mesh (PME only), should be odd.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_POISSON_EWALD_O_SPLINE = Quantity(
        type=str,
        shape=[],
        description='''
        order of the beta-Euler spline (SPME only)
        ''')

    x_cp2k_input_FORCE_EVAL_MM_POISSON_EWALD_RCUT = Quantity(
        type=str,
        shape=[],
        description='''
        Explicitly provide the real-space cutoff of the ewald summation (EWALD|PME|SPME).
        If present, overwrites the estimate of EWALD_ACCURACY and may affect the
        construction of the neighbor lists for non-bonded terms (in FIST), if the value
        specified is larger than the cutoff for non-bonded interactions.
        ''')

    x_cp2k_section_input_FORCE_EVAL_MM_POISSON_EWALD_MULTIPOLES = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MM_POISSON_EWALD_MULTIPOLES'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_MM_POISSON_EWALD_RS_GRID = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MM_POISSON_EWALD_RS_GRID'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_MM_POISSON_MT(MSection):
    '''
    Sets up parameters of  Martyna-Tuckerman poisson solver. Note that exact results are
    only guaranteed if the unit cell is twice as large as charge density (and serious
    artefacts can result if the cell is much smaller).
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MM_POISSON_MT_ALPHA = Quantity(
        type=str,
        shape=[],
        description='''
        Convergence parameter ALPHA*RMIN. Default value 7.0
        ''')

    x_cp2k_input_FORCE_EVAL_MM_POISSON_MT_REL_CUTOFF = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the multiplicative factor for the CUTOFF keyword in MULTI_GRID  section.
        The result gives the cutoff at which the 1/r non-periodic FFT3D is
        evaluated.Default is 2.0
        ''')


class x_cp2k_section_input_FORCE_EVAL_MM_POISSON_MULTIPOLE_CHECK_SPLINE(MSection):
    '''
    Controls the checking of the G-space term Spline Interpolation.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MM_POISSON_MULTIPOLE_CHECK_SPLINE_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_POISSON_MULTIPOLE_CHECK_SPLINE_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_MM_POISSON_MULTIPOLE_CHECK_SPLINE_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_POISSON_MULTIPOLE_CHECK_SPLINE_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_MM_POISSON_MULTIPOLE_CHECK_SPLINE_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_FORCE_EVAL_MM_POISSON_MULTIPOLE_INTERPOLATOR_CONV_INFO(MSection):
    '''
    if convergence information about the linear solver of the spline methods should be
    printed
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MM_POISSON_MULTIPOLE_INTERPOLATOR_CONV_INFO_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_POISSON_MULTIPOLE_INTERPOLATOR_CONV_INFO_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_MM_POISSON_MULTIPOLE_INTERPOLATOR_CONV_INFO_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_POISSON_MULTIPOLE_INTERPOLATOR_CONV_INFO_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_MM_POISSON_MULTIPOLE_INTERPOLATOR_CONV_INFO_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_FORCE_EVAL_MM_POISSON_MULTIPOLE_INTERPOLATOR(MSection):
    '''
    controls the interpolation for the G-space term
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MM_POISSON_MULTIPOLE_INTERPOLATOR_AINT_PRECOND = Quantity(
        type=str,
        shape=[],
        description='''
        the approximate inverse to use to get the starting point for the linear solver of
        the spline3 methods
        ''')

    x_cp2k_input_FORCE_EVAL_MM_POISSON_MULTIPOLE_INTERPOLATOR_EPS_R = Quantity(
        type=str,
        shape=[],
        description='''
        accuracy on the residual for spline3 the interpolators
        ''')

    x_cp2k_input_FORCE_EVAL_MM_POISSON_MULTIPOLE_INTERPOLATOR_EPS_X = Quantity(
        type=str,
        shape=[],
        description='''
        accuracy on the solution for spline3 the interpolators
        ''')

    x_cp2k_input_FORCE_EVAL_MM_POISSON_MULTIPOLE_INTERPOLATOR_MAX_ITER = Quantity(
        type=str,
        shape=[],
        description='''
        the maximum number of iterations
        ''')

    x_cp2k_input_FORCE_EVAL_MM_POISSON_MULTIPOLE_INTERPOLATOR_PRECOND = Quantity(
        type=str,
        shape=[],
        description='''
        The preconditioner used for the linear solver of the spline3 methods
        ''')

    x_cp2k_section_input_FORCE_EVAL_MM_POISSON_MULTIPOLE_INTERPOLATOR_CONV_INFO = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MM_POISSON_MULTIPOLE_INTERPOLATOR_CONV_INFO'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_MM_POISSON_MULTIPOLE_PROGRAM_RUN_INFO(MSection):
    '''
    Controls the printing of basic information during the run
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MM_POISSON_MULTIPOLE_PROGRAM_RUN_INFO_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_POISSON_MULTIPOLE_PROGRAM_RUN_INFO_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_MM_POISSON_MULTIPOLE_PROGRAM_RUN_INFO_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_POISSON_MULTIPOLE_PROGRAM_RUN_INFO_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_MM_POISSON_MULTIPOLE_PROGRAM_RUN_INFO_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_FORCE_EVAL_MM_POISSON_MULTIPOLE(MSection):
    '''
    This section is used to set up the decoupling of QM periodic images with the use of
    density derived atomic point charges.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MM_POISSON_MULTIPOLE_ANALYTICAL_GTERM = Quantity(
        type=str,
        shape=[],
        description='''
        Evaluates the Gterm in the Ewald Scheme analytically instead of using Splines.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_POISSON_MULTIPOLE_EWALD_PRECISION = Quantity(
        type=str,
        shape=[],
        description='''
        Precision achieved in the Ewald sum.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_POISSON_MULTIPOLE_NGRIDS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the number of grid points used for the Interpolation of the G-space term
        ''')

    x_cp2k_input_FORCE_EVAL_MM_POISSON_MULTIPOLE_RCUT = Quantity(
        type=str,
        shape=[],
        description='''
        Real space cutoff for the Ewald sum.
        ''')

    x_cp2k_section_input_FORCE_EVAL_MM_POISSON_MULTIPOLE_CHECK_SPLINE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MM_POISSON_MULTIPOLE_CHECK_SPLINE'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_MM_POISSON_MULTIPOLE_INTERPOLATOR = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MM_POISSON_MULTIPOLE_INTERPOLATOR'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_MM_POISSON_MULTIPOLE_PROGRAM_RUN_INFO = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MM_POISSON_MULTIPOLE_PROGRAM_RUN_INFO'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_MM_POISSON_WAVELET(MSection):
    '''
    Sets up parameters of  wavelet based poisson solver.This solver allows for non-
    periodic (PERIODIC NONE) boundary conditions and slab-boundary conditions (but only
    PERIODIC XZ).It does not require very large unit cells, only that the density goes to
    zero on the faces of the cell.The use of PREFERRED_FFT_LIBRARY FFTSG is required
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MM_POISSON_WAVELET_SCF_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Type of scaling function used in the wavelet approach, the total energy depends on
        this choice,and the convergence with respect to cutoff depends on the selected
        scaling functions.Possible values are 8,14,16,20,24,30,40,50,60,100
        ''')


class x_cp2k_section_input_FORCE_EVAL_MM_POISSON(MSection):
    '''
    Sets up the poisson resolutor.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_MM_POISSON_PERIODIC = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the directions on wich apply PBC. Important notice,  this only applies to
        the electrostatics. See the CELL section to specify the periodicity used for e.g.
        the pair lists. Typically the settings should be the same.
        ''')

    x_cp2k_input_FORCE_EVAL_MM_POISSON_POISSON_SOLVER = Quantity(
        type=str,
        shape=[],
        description='''
        Specify which kind of solver to use to solve the Poisson equation.
        ''')

    x_cp2k_section_input_FORCE_EVAL_MM_POISSON_EWALD = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MM_POISSON_EWALD'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_MM_POISSON_MT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MM_POISSON_MT'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_MM_POISSON_MULTIPOLE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MM_POISSON_MULTIPOLE'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_MM_POISSON_WAVELET = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MM_POISSON_WAVELET'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_MM(MSection):
    '''
    This section contains all information to run a MM calculation.
    '''

    m_def = Section(validate=False)

    x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MM_FORCEFIELD'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_MM_NEIGHBOR_LISTS = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MM_NEIGHBOR_LISTS'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_MM_POISSON = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_MM_POISSON'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_PRINT_DISTRIBUTION_EACH(MSection):
    '''
    This section specifies how often this proprety is printed.Each keyword inside this
    section is mapping to a specific iteration level and the value of each of these
    keywords is matched with the iteration level during the calculation. How to handle the
    last iteration is treated separately in ADD_LAST (this mean that each iteration level
    (MD, GEO_OPT, etc..), though equal to 0, might print the last iteration). If an
    iteration level is specified that is not present in the flow of the calculation it is
    just ignored.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION_EACH_BAND = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Band Calculation Steps
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION_EACH_BSSE = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Basis Set Superposition Error (BSSE) Calculation
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION_EACH_CELL_OPT = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Cell optimization steps.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION_EACH_EP_LIN_SOLVER = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Energy Perturbation (EP) linear solver
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION_EACH_GEO_OPT = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Geometry optimization steps.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION_EACH_JUST_ENERGY = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for an ENERGY/ENERGY_FORCE calculation.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION_EACH_MD = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the MD steps.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION_EACH_METADYNAMICS = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the METADYNAMICS steps (number of hills added).
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION_EACH_POWELL_OPT = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for POWELL based optimization steps.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION_EACH_QS_SCF = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the SCF Steps.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION_EACH_REPLICA_EVAL = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the evaluation of the Replica Environment
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION_EACH_ROT_OPT = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Rotational optimization steps in the Dimer Calculation.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION_EACH_SHELL_OPT = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Shell-Core distances optimization steps
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION_EACH_SPLINE_FIND_COEFFS = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the solution of the coefficients of the splines
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION_EACH_XAS_SCF = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the X-Ray Absorption Spectroscopy (XAS) SCF Steps.
        ''')


class x_cp2k_section_input_FORCE_EVAL_PRINT_DISTRIBUTION1D_EACH(MSection):
    '''
    This section specifies how often this proprety is printed.Each keyword inside this
    section is mapping to a specific iteration level and the value of each of these
    keywords is matched with the iteration level during the calculation. How to handle the
    last iteration is treated separately in ADD_LAST (this mean that each iteration level
    (MD, GEO_OPT, etc..), though equal to 0, might print the last iteration). If an
    iteration level is specified that is not present in the flow of the calculation it is
    just ignored.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION1D_EACH_BAND = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Band Calculation Steps
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION1D_EACH_BSSE = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Basis Set Superposition Error (BSSE) Calculation
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION1D_EACH_CELL_OPT = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Cell optimization steps.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION1D_EACH_EP_LIN_SOLVER = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Energy Perturbation (EP) linear solver
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION1D_EACH_GEO_OPT = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Geometry optimization steps.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION1D_EACH_JUST_ENERGY = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for an ENERGY/ENERGY_FORCE calculation.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION1D_EACH_MD = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the MD steps.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION1D_EACH_METADYNAMICS = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the METADYNAMICS steps (number of hills added).
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION1D_EACH_POWELL_OPT = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for POWELL based optimization steps.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION1D_EACH_QS_SCF = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the SCF Steps.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION1D_EACH_REPLICA_EVAL = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the evaluation of the Replica Environment
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION1D_EACH_ROT_OPT = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Rotational optimization steps in the Dimer Calculation.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION1D_EACH_SHELL_OPT = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Shell-Core distances optimization steps
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION1D_EACH_SPLINE_FIND_COEFFS = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the solution of the coefficients of the splines
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION1D_EACH_XAS_SCF = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the X-Ray Absorption Spectroscopy (XAS) SCF Steps.
        ''')


class x_cp2k_section_input_FORCE_EVAL_PRINT_DISTRIBUTION1D(MSection):
    '''
    Each node prints out its distribution info ...
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION1D_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION1D_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION1D_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION1D_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION1D_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')

    x_cp2k_section_input_FORCE_EVAL_PRINT_DISTRIBUTION1D_EACH = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PRINT_DISTRIBUTION1D_EACH'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_PRINT_DISTRIBUTION2D_EACH(MSection):
    '''
    This section specifies how often this proprety is printed.Each keyword inside this
    section is mapping to a specific iteration level and the value of each of these
    keywords is matched with the iteration level during the calculation. How to handle the
    last iteration is treated separately in ADD_LAST (this mean that each iteration level
    (MD, GEO_OPT, etc..), though equal to 0, might print the last iteration). If an
    iteration level is specified that is not present in the flow of the calculation it is
    just ignored.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION2D_EACH_BAND = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Band Calculation Steps
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION2D_EACH_BSSE = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Basis Set Superposition Error (BSSE) Calculation
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION2D_EACH_CELL_OPT = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Cell optimization steps.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION2D_EACH_EP_LIN_SOLVER = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Energy Perturbation (EP) linear solver
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION2D_EACH_GEO_OPT = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Geometry optimization steps.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION2D_EACH_JUST_ENERGY = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for an ENERGY/ENERGY_FORCE calculation.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION2D_EACH_MD = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the MD steps.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION2D_EACH_METADYNAMICS = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the METADYNAMICS steps (number of hills added).
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION2D_EACH_POWELL_OPT = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for POWELL based optimization steps.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION2D_EACH_QS_SCF = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the SCF Steps.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION2D_EACH_REPLICA_EVAL = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the evaluation of the Replica Environment
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION2D_EACH_ROT_OPT = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Rotational optimization steps in the Dimer Calculation.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION2D_EACH_SHELL_OPT = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Shell-Core distances optimization steps
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION2D_EACH_SPLINE_FIND_COEFFS = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the solution of the coefficients of the splines
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION2D_EACH_XAS_SCF = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the X-Ray Absorption Spectroscopy (XAS) SCF Steps.
        ''')


class x_cp2k_section_input_FORCE_EVAL_PRINT_DISTRIBUTION2D(MSection):
    '''
    Controls the printing of the distribution of matrix blocks,...
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION2D_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION2D_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION2D_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION2D_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION2D_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')

    x_cp2k_section_input_FORCE_EVAL_PRINT_DISTRIBUTION2D_EACH = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PRINT_DISTRIBUTION2D_EACH'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_PRINT_DISTRIBUTION(MSection):
    '''
    Controls the printing of the distribution of molecules, atoms, ...
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_DISTRIBUTION_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')

    x_cp2k_section_input_FORCE_EVAL_PRINT_DISTRIBUTION_EACH = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PRINT_DISTRIBUTION_EACH'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_PRINT_FORCES_EACH(MSection):
    '''
    This section specifies how often this proprety is printed.Each keyword inside this
    section is mapping to a specific iteration level and the value of each of these
    keywords is matched with the iteration level during the calculation. How to handle the
    last iteration is treated separately in ADD_LAST (this mean that each iteration level
    (MD, GEO_OPT, etc..), though equal to 0, might print the last iteration). If an
    iteration level is specified that is not present in the flow of the calculation it is
    just ignored.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PRINT_FORCES_EACH_BAND = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Band Calculation Steps
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_FORCES_EACH_BSSE = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Basis Set Superposition Error (BSSE) Calculation
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_FORCES_EACH_CELL_OPT = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Cell optimization steps.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_FORCES_EACH_EP_LIN_SOLVER = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Energy Perturbation (EP) linear solver
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_FORCES_EACH_GEO_OPT = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Geometry optimization steps.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_FORCES_EACH_JUST_ENERGY = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for an ENERGY/ENERGY_FORCE calculation.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_FORCES_EACH_MD = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the MD steps.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_FORCES_EACH_METADYNAMICS = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the METADYNAMICS steps (number of hills added).
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_FORCES_EACH_POWELL_OPT = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for POWELL based optimization steps.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_FORCES_EACH_QS_SCF = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the SCF Steps.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_FORCES_EACH_REPLICA_EVAL = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the evaluation of the Replica Environment
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_FORCES_EACH_ROT_OPT = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Rotational optimization steps in the Dimer Calculation.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_FORCES_EACH_SHELL_OPT = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Shell-Core distances optimization steps
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_FORCES_EACH_SPLINE_FIND_COEFFS = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the solution of the coefficients of the splines
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_FORCES_EACH_XAS_SCF = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the X-Ray Absorption Spectroscopy (XAS) SCF Steps.
        ''')


class x_cp2k_section_input_FORCE_EVAL_PRINT_FORCES(MSection):
    '''
    Controls the printing of the forces after each force evaluation
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PRINT_FORCES_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_FORCES_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_FORCES_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_FORCES_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_FORCES_NDIGITS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the number of digits used for the printing of the forces
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_FORCES_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')

    x_cp2k_section_input_FORCE_EVAL_PRINT_FORCES_EACH = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PRINT_FORCES_EACH'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_PRINT_GRID_INFORMATION_EACH(MSection):
    '''
    This section specifies how often this proprety is printed.Each keyword inside this
    section is mapping to a specific iteration level and the value of each of these
    keywords is matched with the iteration level during the calculation. How to handle the
    last iteration is treated separately in ADD_LAST (this mean that each iteration level
    (MD, GEO_OPT, etc..), though equal to 0, might print the last iteration). If an
    iteration level is specified that is not present in the flow of the calculation it is
    just ignored.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PRINT_GRID_INFORMATION_EACH_BAND = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Band Calculation Steps
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_GRID_INFORMATION_EACH_BSSE = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Basis Set Superposition Error (BSSE) Calculation
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_GRID_INFORMATION_EACH_CELL_OPT = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Cell optimization steps.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_GRID_INFORMATION_EACH_EP_LIN_SOLVER = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Energy Perturbation (EP) linear solver
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_GRID_INFORMATION_EACH_GEO_OPT = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Geometry optimization steps.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_GRID_INFORMATION_EACH_JUST_ENERGY = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for an ENERGY/ENERGY_FORCE calculation.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_GRID_INFORMATION_EACH_MD = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the MD steps.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_GRID_INFORMATION_EACH_METADYNAMICS = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the METADYNAMICS steps (number of hills added).
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_GRID_INFORMATION_EACH_POWELL_OPT = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for POWELL based optimization steps.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_GRID_INFORMATION_EACH_QS_SCF = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the SCF Steps.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_GRID_INFORMATION_EACH_REPLICA_EVAL = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the evaluation of the Replica Environment
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_GRID_INFORMATION_EACH_ROT_OPT = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Rotational optimization steps in the Dimer Calculation.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_GRID_INFORMATION_EACH_SHELL_OPT = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Shell-Core distances optimization steps
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_GRID_INFORMATION_EACH_SPLINE_FIND_COEFFS = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the solution of the coefficients of the splines
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_GRID_INFORMATION_EACH_XAS_SCF = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the X-Ray Absorption Spectroscopy (XAS) SCF Steps.
        ''')


class x_cp2k_section_input_FORCE_EVAL_PRINT_GRID_INFORMATION(MSection):
    '''
    Controls the printing of information regarding the PW and RS grid structures.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PRINT_GRID_INFORMATION_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_GRID_INFORMATION_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_GRID_INFORMATION_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_GRID_INFORMATION_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_GRID_INFORMATION_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')

    x_cp2k_section_input_FORCE_EVAL_PRINT_GRID_INFORMATION_EACH = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PRINT_GRID_INFORMATION_EACH'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_PRINT_PROGRAM_RUN_INFO_EACH(MSection):
    '''
    This section specifies how often this proprety is printed.Each keyword inside this
    section is mapping to a specific iteration level and the value of each of these
    keywords is matched with the iteration level during the calculation. How to handle the
    last iteration is treated separately in ADD_LAST (this mean that each iteration level
    (MD, GEO_OPT, etc..), though equal to 0, might print the last iteration). If an
    iteration level is specified that is not present in the flow of the calculation it is
    just ignored.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PRINT_PROGRAM_RUN_INFO_EACH_BAND = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Band Calculation Steps
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_PROGRAM_RUN_INFO_EACH_BSSE = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Basis Set Superposition Error (BSSE) Calculation
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_PROGRAM_RUN_INFO_EACH_CELL_OPT = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Cell optimization steps.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_PROGRAM_RUN_INFO_EACH_EP_LIN_SOLVER = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Energy Perturbation (EP) linear solver
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_PROGRAM_RUN_INFO_EACH_GEO_OPT = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Geometry optimization steps.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_PROGRAM_RUN_INFO_EACH_JUST_ENERGY = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for an ENERGY/ENERGY_FORCE calculation.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_PROGRAM_RUN_INFO_EACH_MD = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the MD steps.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_PROGRAM_RUN_INFO_EACH_METADYNAMICS = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the METADYNAMICS steps (number of hills added).
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_PROGRAM_RUN_INFO_EACH_POWELL_OPT = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for POWELL based optimization steps.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_PROGRAM_RUN_INFO_EACH_QS_SCF = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the SCF Steps.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_PROGRAM_RUN_INFO_EACH_REPLICA_EVAL = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the evaluation of the Replica Environment
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_PROGRAM_RUN_INFO_EACH_ROT_OPT = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Rotational optimization steps in the Dimer Calculation.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_PROGRAM_RUN_INFO_EACH_SHELL_OPT = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Shell-Core distances optimization steps
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_PROGRAM_RUN_INFO_EACH_SPLINE_FIND_COEFFS = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the solution of the coefficients of the splines
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_PROGRAM_RUN_INFO_EACH_XAS_SCF = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the X-Ray Absorption Spectroscopy (XAS) SCF Steps.
        ''')


class x_cp2k_section_input_FORCE_EVAL_PRINT_PROGRAM_RUN_INFO(MSection):
    '''
    Controls the printing of basic information generated by force_eval
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PRINT_PROGRAM_RUN_INFO_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_PROGRAM_RUN_INFO_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_PROGRAM_RUN_INFO_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_PROGRAM_RUN_INFO_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_PROGRAM_RUN_INFO_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')

    x_cp2k_section_input_FORCE_EVAL_PRINT_PROGRAM_RUN_INFO_EACH = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PRINT_PROGRAM_RUN_INFO_EACH'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_PRINT_STRESS_TENSOR_EACH(MSection):
    '''
    This section specifies how often this proprety is printed.Each keyword inside this
    section is mapping to a specific iteration level and the value of each of these
    keywords is matched with the iteration level during the calculation. How to handle the
    last iteration is treated separately in ADD_LAST (this mean that each iteration level
    (MD, GEO_OPT, etc..), though equal to 0, might print the last iteration). If an
    iteration level is specified that is not present in the flow of the calculation it is
    just ignored.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PRINT_STRESS_TENSOR_EACH_BAND = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Band Calculation Steps
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_STRESS_TENSOR_EACH_BSSE = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Basis Set Superposition Error (BSSE) Calculation
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_STRESS_TENSOR_EACH_CELL_OPT = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Cell optimization steps.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_STRESS_TENSOR_EACH_EP_LIN_SOLVER = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Energy Perturbation (EP) linear solver
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_STRESS_TENSOR_EACH_GEO_OPT = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Geometry optimization steps.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_STRESS_TENSOR_EACH_JUST_ENERGY = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for an ENERGY/ENERGY_FORCE calculation.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_STRESS_TENSOR_EACH_MD = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the MD steps.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_STRESS_TENSOR_EACH_METADYNAMICS = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the METADYNAMICS steps (number of hills added).
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_STRESS_TENSOR_EACH_POWELL_OPT = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for POWELL based optimization steps.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_STRESS_TENSOR_EACH_QS_SCF = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the SCF Steps.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_STRESS_TENSOR_EACH_REPLICA_EVAL = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the evaluation of the Replica Environment
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_STRESS_TENSOR_EACH_ROT_OPT = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Rotational optimization steps in the Dimer Calculation.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_STRESS_TENSOR_EACH_SHELL_OPT = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Shell-Core distances optimization steps
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_STRESS_TENSOR_EACH_SPLINE_FIND_COEFFS = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the solution of the coefficients of the splines
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_STRESS_TENSOR_EACH_XAS_SCF = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the X-Ray Absorption Spectroscopy (XAS) SCF Steps.
        ''')


class x_cp2k_section_input_FORCE_EVAL_PRINT_STRESS_TENSOR(MSection):
    '''
    Controls the printing of the stress tensor
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PRINT_STRESS_TENSOR_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_STRESS_TENSOR_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_STRESS_TENSOR_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_STRESS_TENSOR_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_STRESS_TENSOR_NDIGITS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the number of digits used for the printing of the stress tensor
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_STRESS_TENSOR_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')

    x_cp2k_section_input_FORCE_EVAL_PRINT_STRESS_TENSOR_EACH = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PRINT_STRESS_TENSOR_EACH'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_PRINT_TOTAL_NUMBERS_EACH(MSection):
    '''
    This section specifies how often this proprety is printed.Each keyword inside this
    section is mapping to a specific iteration level and the value of each of these
    keywords is matched with the iteration level during the calculation. How to handle the
    last iteration is treated separately in ADD_LAST (this mean that each iteration level
    (MD, GEO_OPT, etc..), though equal to 0, might print the last iteration). If an
    iteration level is specified that is not present in the flow of the calculation it is
    just ignored.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PRINT_TOTAL_NUMBERS_EACH_BAND = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Band Calculation Steps
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_TOTAL_NUMBERS_EACH_BSSE = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Basis Set Superposition Error (BSSE) Calculation
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_TOTAL_NUMBERS_EACH_CELL_OPT = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Cell optimization steps.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_TOTAL_NUMBERS_EACH_EP_LIN_SOLVER = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Energy Perturbation (EP) linear solver
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_TOTAL_NUMBERS_EACH_GEO_OPT = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Geometry optimization steps.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_TOTAL_NUMBERS_EACH_JUST_ENERGY = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for an ENERGY/ENERGY_FORCE calculation.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_TOTAL_NUMBERS_EACH_MD = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the MD steps.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_TOTAL_NUMBERS_EACH_METADYNAMICS = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the METADYNAMICS steps (number of hills added).
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_TOTAL_NUMBERS_EACH_POWELL_OPT = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for POWELL based optimization steps.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_TOTAL_NUMBERS_EACH_QS_SCF = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the SCF Steps.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_TOTAL_NUMBERS_EACH_REPLICA_EVAL = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the evaluation of the Replica Environment
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_TOTAL_NUMBERS_EACH_ROT_OPT = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Rotational optimization steps in the Dimer Calculation.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_TOTAL_NUMBERS_EACH_SHELL_OPT = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the Shell-Core distances optimization steps
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_TOTAL_NUMBERS_EACH_SPLINE_FIND_COEFFS = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the solution of the coefficients of the splines
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_TOTAL_NUMBERS_EACH_XAS_SCF = Quantity(
        type=str,
        shape=[],
        description='''
        Iteration level for the X-Ray Absorption Spectroscopy (XAS) SCF Steps.
        ''')


class x_cp2k_section_input_FORCE_EVAL_PRINT_TOTAL_NUMBERS(MSection):
    '''
    Controls the printing of the total number of atoms, kinds,...
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PRINT_TOTAL_NUMBERS_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_TOTAL_NUMBERS_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_TOTAL_NUMBERS_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_TOTAL_NUMBERS_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_PRINT_TOTAL_NUMBERS_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')

    x_cp2k_section_input_FORCE_EVAL_PRINT_TOTAL_NUMBERS_EACH = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PRINT_TOTAL_NUMBERS_EACH'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_PRINT(MSection):
    '''
    Properties that you want to output and that are common to all methods
    '''

    m_def = Section(validate=False)

    x_cp2k_section_input_FORCE_EVAL_PRINT_DISTRIBUTION1D = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PRINT_DISTRIBUTION1D'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_PRINT_DISTRIBUTION2D = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PRINT_DISTRIBUTION2D'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_PRINT_DISTRIBUTION = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PRINT_DISTRIBUTION'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_PRINT_FORCES = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PRINT_FORCES'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_PRINT_GRID_INFORMATION = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PRINT_GRID_INFORMATION'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_PRINT_PROGRAM_RUN_INFO = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PRINT_PROGRAM_RUN_INFO'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_PRINT_STRESS_TENSOR = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PRINT_STRESS_TENSOR'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_PRINT_TOTAL_NUMBERS = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PRINT_TOTAL_NUMBERS'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_PROPERTIES_ATOMIC(MSection):
    '''
    Controls the calculation of atomic properties. Printing is controled by FORCE_EVAL /
    PRINT / PROGRAM_RUN_INFO
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ATOMIC_ENERGY = Quantity(
        type=str,
        shape=[],
        description='''
        Calculate atomic energies
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ATOMIC_PRESSURE = Quantity(
        type=str,
        shape=[],
        description='''
        Calculate atomic pressure tensors
        ''')


class x_cp2k_section_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_BECKE_RESTRAINT_A_PROGRAM_RUN_INFO(MSection):
    '''
    Controls the printing basic info about the method
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_BECKE_RESTRAINT_A_PROGRAM_RUN_INFO_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_BECKE_RESTRAINT_A_PROGRAM_RUN_INFO_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_BECKE_RESTRAINT_A_PROGRAM_RUN_INFO_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_BECKE_RESTRAINT_A_PROGRAM_RUN_INFO_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_BECKE_RESTRAINT_A_PROGRAM_RUN_INFO_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_BECKE_RESTRAINT_A(MSection):
    '''
    Use DDAPC charges in a restraint (check code for details), section can be repeated,
    but only one constraint is possible at the moment.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_BECKE_RESTRAINT_A_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the list of atoms that is summed in the restraint
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_BECKE_RESTRAINT_A_COEFF = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the the coefficient of the atom in the atom list (default is one),
        currently DDAPC only
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_BECKE_RESTRAINT_A_FUNCTIONAL_FORM = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the functional form of the term added
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_BECKE_RESTRAINT_A_STRENGTH = Quantity(
        type=str,
        shape=[],
        description='''
        force constant of the restraint
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_BECKE_RESTRAINT_A_TARGET = Quantity(
        type=str,
        shape=[],
        description='''
        target value of the restraint
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_BECKE_RESTRAINT_A_TYPE_OF_DENSITY = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the type of density used for the fitting
        ''')

    x_cp2k_section_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_BECKE_RESTRAINT_A_PROGRAM_RUN_INFO = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_BECKE_RESTRAINT_A_PROGRAM_RUN_INFO'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_BECKE_RESTRAINT_B_PROGRAM_RUN_INFO(MSection):
    '''
    Controls the printing basic info about the method
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_BECKE_RESTRAINT_B_PROGRAM_RUN_INFO_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_BECKE_RESTRAINT_B_PROGRAM_RUN_INFO_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_BECKE_RESTRAINT_B_PROGRAM_RUN_INFO_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_BECKE_RESTRAINT_B_PROGRAM_RUN_INFO_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_BECKE_RESTRAINT_B_PROGRAM_RUN_INFO_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_BECKE_RESTRAINT_B(MSection):
    '''
    Use DDAPC charges in a restraint (check code for details), section can be repeated,
    but only one constraint is possible at the moment.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_BECKE_RESTRAINT_B_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the list of atoms that is summed in the restraint
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_BECKE_RESTRAINT_B_COEFF = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the the coefficient of the atom in the atom list (default is one),
        currently DDAPC only
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_BECKE_RESTRAINT_B_FUNCTIONAL_FORM = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the functional form of the term added
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_BECKE_RESTRAINT_B_STRENGTH = Quantity(
        type=str,
        shape=[],
        description='''
        force constant of the restraint
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_BECKE_RESTRAINT_B_TARGET = Quantity(
        type=str,
        shape=[],
        description='''
        target value of the restraint
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_BECKE_RESTRAINT_B_TYPE_OF_DENSITY = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the type of density used for the fitting
        ''')

    x_cp2k_section_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_BECKE_RESTRAINT_B_PROGRAM_RUN_INFO = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_BECKE_RESTRAINT_B_PROGRAM_RUN_INFO'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_DDAPC_RESTRAINT_A_PROGRAM_RUN_INFO(MSection):
    '''
    Controls the printing basic info about the method
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_DDAPC_RESTRAINT_A_PROGRAM_RUN_INFO_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_DDAPC_RESTRAINT_A_PROGRAM_RUN_INFO_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_DDAPC_RESTRAINT_A_PROGRAM_RUN_INFO_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_DDAPC_RESTRAINT_A_PROGRAM_RUN_INFO_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_DDAPC_RESTRAINT_A_PROGRAM_RUN_INFO_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_DDAPC_RESTRAINT_A(MSection):
    '''
    Use DDAPC charges in a restraint (check code for details), section can be repeated,
    but only one constraint is possible at the moment.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_DDAPC_RESTRAINT_A_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the list of atoms that is summed in the restraint
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_DDAPC_RESTRAINT_A_COEFF = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the the coefficient of the atom in the atom list (default is one),
        currently DDAPC only
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_DDAPC_RESTRAINT_A_FUNCTIONAL_FORM = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the functional form of the term added
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_DDAPC_RESTRAINT_A_STRENGTH = Quantity(
        type=str,
        shape=[],
        description='''
        force constant of the restraint
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_DDAPC_RESTRAINT_A_TARGET = Quantity(
        type=str,
        shape=[],
        description='''
        target value of the restraint
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_DDAPC_RESTRAINT_A_TYPE_OF_DENSITY = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the type of density used for the fitting
        ''')

    x_cp2k_section_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_DDAPC_RESTRAINT_A_PROGRAM_RUN_INFO = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_DDAPC_RESTRAINT_A_PROGRAM_RUN_INFO'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_DDAPC_RESTRAINT_B_PROGRAM_RUN_INFO(MSection):
    '''
    Controls the printing basic info about the method
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_DDAPC_RESTRAINT_B_PROGRAM_RUN_INFO_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_DDAPC_RESTRAINT_B_PROGRAM_RUN_INFO_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_DDAPC_RESTRAINT_B_PROGRAM_RUN_INFO_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_DDAPC_RESTRAINT_B_PROGRAM_RUN_INFO_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_DDAPC_RESTRAINT_B_PROGRAM_RUN_INFO_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_DDAPC_RESTRAINT_B(MSection):
    '''
    Use DDAPC charges in a restraint (check code for details), section can be repeated,
    but only one constraint is possible at the moment.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_DDAPC_RESTRAINT_B_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the list of atoms that is summed in the restraint
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_DDAPC_RESTRAINT_B_COEFF = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the the coefficient of the atom in the atom list (default is one),
        currently DDAPC only
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_DDAPC_RESTRAINT_B_FUNCTIONAL_FORM = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the functional form of the term added
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_DDAPC_RESTRAINT_B_STRENGTH = Quantity(
        type=str,
        shape=[],
        description='''
        force constant of the restraint
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_DDAPC_RESTRAINT_B_TARGET = Quantity(
        type=str,
        shape=[],
        description='''
        target value of the restraint
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_DDAPC_RESTRAINT_B_TYPE_OF_DENSITY = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the type of density used for the fitting
        ''')

    x_cp2k_section_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_DDAPC_RESTRAINT_B_PROGRAM_RUN_INFO = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_DDAPC_RESTRAINT_B_PROGRAM_RUN_INFO'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_PROGRAM_RUN_INFO(MSection):
    '''
    Controls the printing basic info about the method
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_PROGRAM_RUN_INFO_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_PROGRAM_RUN_INFO_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_PROGRAM_RUN_INFO_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_PROGRAM_RUN_INFO_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_PROGRAM_RUN_INFO_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_FORCE_EVAL_PROPERTIES_ET_COUPLING(MSection):
    '''
    specifies the two constraints/restraints for extracting ET coupling elements
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_TYPE_OF_CONSTRAINT = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the type of constraint
        ''')

    x_cp2k_section_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_BECKE_RESTRAINT_A = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_BECKE_RESTRAINT_A'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_BECKE_RESTRAINT_B = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_BECKE_RESTRAINT_B'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_DDAPC_RESTRAINT_A = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_DDAPC_RESTRAINT_A'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_DDAPC_RESTRAINT_B = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_DDAPC_RESTRAINT_B'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_PROGRAM_RUN_INFO = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PROPERTIES_ET_COUPLING_PROGRAM_RUN_INFO'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_PROPERTIES_FIT_CHARGE(MSection):
    '''
    This section is used to print the density derived atomic point charges.The fit of the
    charges is controlled through the DENSITY_FITTING section
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PROPERTIES_FIT_CHARGE_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_FIT_CHARGE_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_FIT_CHARGE_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_FIT_CHARGE_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_FIT_CHARGE_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_FIT_CHARGE_TYPE_OF_DENSITY = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the type of density used for the fitting
        ''')


class x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_CURRENT_INTERPOLATOR_CONV_INFO(MSection):
    '''
    if convergence information about the linear solver of the spline methods should be
    printed
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_CURRENT_INTERPOLATOR_CONV_INFO_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_CURRENT_INTERPOLATOR_CONV_INFO_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_CURRENT_INTERPOLATOR_CONV_INFO_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_CURRENT_INTERPOLATOR_CONV_INFO_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_CURRENT_INTERPOLATOR_CONV_INFO_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_CURRENT_INTERPOLATOR(MSection):
    '''
    kind of interpolation used between the multigrids
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_CURRENT_INTERPOLATOR_AINT_PRECOND = Quantity(
        type=str,
        shape=[],
        description='''
        the approximate inverse to use to get the starting point for the linear solver of
        the spline3 methods
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_CURRENT_INTERPOLATOR_EPS_R = Quantity(
        type=str,
        shape=[],
        description='''
        accuracy on the residual for spline3 the interpolators
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_CURRENT_INTERPOLATOR_EPS_X = Quantity(
        type=str,
        shape=[],
        description='''
        accuracy on the solution for spline3 the interpolators
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_CURRENT_INTERPOLATOR_KIND = Quantity(
        type=str,
        shape=[],
        description='''
        the interpolator to use
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_CURRENT_INTERPOLATOR_MAX_ITER = Quantity(
        type=str,
        shape=[],
        description='''
        the maximum number of iterations
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_CURRENT_INTERPOLATOR_PRECOND = Quantity(
        type=str,
        shape=[],
        description='''
        The preconditioner used for the linear solver of the spline3 methods
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_CURRENT_INTERPOLATOR_SAFE_COMPUTATION = Quantity(
        type=str,
        shape=[],
        description='''
        if a non unrolled calculation is to be performed in parallel
        ''')

    x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_CURRENT_INTERPOLATOR_CONV_INFO = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_CURRENT_INTERPOLATOR_CONV_INFO'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_CURRENT(MSection):
    '''
    The induced current density is calculated by DFPT.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_CURRENT_CHI_PBC = Quantity(
        type=str,
        shape=[],
        description='''
        Calculate the succeptibility correction to the shift with PBC
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_CURRENT_COMMON_CENTER = Quantity(
        type=str,
        shape=[],
        description='''
        The common center
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_CURRENT_GAUGE_ATOM_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Build the gauge=atom using only the atoms within this radius.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_CURRENT_GAUGE = Quantity(
        type=str,
        shape=[],
        description='''
        The gauge used to compute the induced current within GAPW.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_CURRENT_NBOX = Quantity(
        type=str,
        shape=[],
        description='''
        How many boxes along each directions
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_CURRENT_ORBITAL_CENTER = Quantity(
        type=str,
        shape=[],
        description='''
        The orbital center.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_CURRENT_RESTART_CURRENT = Quantity(
        type=str,
        shape=[],
        description='''
        Restart the induced current density calculation from a previous run (not working
        yet).
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_CURRENT_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        controls the activation of the induced current calculation
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_CURRENT_SELECTED_STATES_ATOM_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Select all the states included in the given radius arround each atoms in
        SELECTED_STATES_ON_ATOM_LIST.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_CURRENT_SELECTED_STATES_ON_ATOM_LIST = Quantity(
        type=str,
        shape=[],
        description='''
        Indexes of the atoms for selecting the states to be used for the response
        calculations.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_CURRENT_USE_OLD_GAUGE_ATOM = Quantity(
        type=str,
        shape=[],
        description='''
        Use the old way to compute the gauge.
        ''')

    x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_CURRENT_INTERPOLATOR = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_CURRENT_INTERPOLATOR'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_EPR_INTERPOLATOR_CONV_INFO(MSection):
    '''
    if convergence information about the linear solver of the spline methods should be
    printed
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_EPR_INTERPOLATOR_CONV_INFO_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_EPR_INTERPOLATOR_CONV_INFO_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_EPR_INTERPOLATOR_CONV_INFO_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_EPR_INTERPOLATOR_CONV_INFO_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_EPR_INTERPOLATOR_CONV_INFO_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_EPR_INTERPOLATOR(MSection):
    '''
    kind of interpolation used between the multigrids
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_EPR_INTERPOLATOR_AINT_PRECOND = Quantity(
        type=str,
        shape=[],
        description='''
        the approximate inverse to use to get the starting point for the linear solver of
        the spline3 methods
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_EPR_INTERPOLATOR_EPS_R = Quantity(
        type=str,
        shape=[],
        description='''
        accuracy on the residual for spline3 the interpolators
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_EPR_INTERPOLATOR_EPS_X = Quantity(
        type=str,
        shape=[],
        description='''
        accuracy on the solution for spline3 the interpolators
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_EPR_INTERPOLATOR_KIND = Quantity(
        type=str,
        shape=[],
        description='''
        the interpolator to use
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_EPR_INTERPOLATOR_MAX_ITER = Quantity(
        type=str,
        shape=[],
        description='''
        the maximum number of iterations
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_EPR_INTERPOLATOR_PRECOND = Quantity(
        type=str,
        shape=[],
        description='''
        The preconditioner used for the linear solver of the spline3 methods
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_EPR_INTERPOLATOR_SAFE_COMPUTATION = Quantity(
        type=str,
        shape=[],
        description='''
        if a non unrolled calculation is to be performed in parallel
        ''')

    x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_EPR_INTERPOLATOR_CONV_INFO = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_EPR_INTERPOLATOR_CONV_INFO'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_EPR(MSection):
    '''
    The g tensor is calculated by DFPT
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_EPR_RESTART_EPR = Quantity(
        type=str,
        shape=[],
        description='''
        Restart the EPR calculation from a previous run (NOT WORKING)
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_EPR_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        controls the activation of the epr calculation
        ''')

    x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_EPR_INTERPOLATOR = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_EPR_INTERPOLATOR'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_LOCALIZE(MSection):
    '''
    Use one of the available methods to define the localization  and possibly to optimize
    it to a minimum or a maximum.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_LOCALIZE_CRAZY_SCALE = Quantity(
        type=str,
        shape=[],
        description='''
        scale angles
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_LOCALIZE_CRAZY_USE_DIAG = Quantity(
        type=str,
        shape=[],
        description='''
        Use diagonalization (slow) or pade based calculation of matrix exponentials.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_LOCALIZE_ENERGY_RANGE = Quantity(
        type=str,
        shape=[],
        description='''
        Select the orbitals to be localized within the given energy range.This type of
        selection cannot be added on top of the selection through a LIST. It reads to
        reals that are lower and higher boundaries of the energy range.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_LOCALIZE_EPS_LOCALIZATION = Quantity(
        type=str,
        shape=[],
        description='''
        Tolerance used in the convergence criterium of the localization methods.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_LOCALIZE_EPS_OCCUPATION = Quantity(
        type=str,
        shape=[],
        description='''
        Tolerance in the occupation number to select only fully occupied orbitals for the
        rotation
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_LOCALIZE_JACOBI_FALLBACK = Quantity(
        type=str,
        shape=[],
        description='''
        Use Jacobi method in case no convergence was achieved by using the crazy rotations
        method.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_LOCALIZE_LIST_UNOCCUPIED = Quantity(
        type=str,
        shape=[],
        description='''
        Indexes of the unoccupied states to be localized, up to now only valid in
        combination with GPW. This keyword has to be present if unoccupied states should
        be localized. This keyword can be repeated several times(useful if you have to
        specify many indexes).
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_LOCALIZE_LIST = Quantity(
        type=str,
        shape=[],
        description='''
        Indexes of the occupied wfn to be localizedThis keyword can be repeated several
        times(useful if you have to specify many indexes).
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_LOCALIZE_LOCHOMO_RESTART_FILE_NAME = Quantity(
        type=str,
        shape=[],
        description='''
        File name where to read the MOS fromwhich to restart the localization procedure
        for occupied states
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_LOCALIZE_LOCLUMO_RESTART_FILE_NAME = Quantity(
        type=str,
        shape=[],
        description='''
        File name where to read the MOS fromwhich to restart the localization procedure
        for unoccupied states
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_LOCALIZE_MAX_CRAZY_ANGLE = Quantity(
        type=str,
        shape=[],
        description='''
        Largest allowed angle for the crazy rotations algorithm (smaller is slower but
        more stable).
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_LOCALIZE_MAX_ITER = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum number of iterations used for localization methods
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_LOCALIZE_METHOD = Quantity(
        type=str,
        shape=[],
        description='''
        Method of optimization if any
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_LOCALIZE_MIN_OR_MAX = Quantity(
        type=str,
        shape=[],
        description='''
        Requires the maximization of the spread of the wfn
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_LOCALIZE_OPERATOR = Quantity(
        type=str,
        shape=[],
        description='''
        Type of opertator which defines the spread functional
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_LOCALIZE_OUT_ITER_EACH = Quantity(
        type=str,
        shape=[],
        description='''
        Every how many iterations of the localization algorithm(Jacobi) the tolerance
        value is printed out
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_LOCALIZE_RESTART = Quantity(
        type=str,
        shape=[],
        description='''
        Restart the localization from a set of orbitals read from a localization restart
        file.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_LOCALIZE_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        controls the activation of the MOS localization procedure
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_LOCALIZE_STATES = Quantity(
        type=str,
        shape=[],
        description='''
        Which states to localize, LUMO up to now only available in GPW
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_LOCALIZE_USE_HISTORY = Quantity(
        type=str,
        shape=[],
        description='''
        Generate an improved initial guess based on a history of results, which is useful
        during MD.Will only work if the number of states to be localized remains constant.
        ''')


class x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_NMR_INTERPOLATOR_CONV_INFO(MSection):
    '''
    if convergence information about the linear solver of the spline methods should be
    printed
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_NMR_INTERPOLATOR_CONV_INFO_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_NMR_INTERPOLATOR_CONV_INFO_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_NMR_INTERPOLATOR_CONV_INFO_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_NMR_INTERPOLATOR_CONV_INFO_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_NMR_INTERPOLATOR_CONV_INFO_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_NMR_INTERPOLATOR(MSection):
    '''
    kind of interpolation used between the multigrids
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_NMR_INTERPOLATOR_AINT_PRECOND = Quantity(
        type=str,
        shape=[],
        description='''
        the approximate inverse to use to get the starting point for the linear solver of
        the spline3 methods
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_NMR_INTERPOLATOR_EPS_R = Quantity(
        type=str,
        shape=[],
        description='''
        accuracy on the residual for spline3 the interpolators
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_NMR_INTERPOLATOR_EPS_X = Quantity(
        type=str,
        shape=[],
        description='''
        accuracy on the solution for spline3 the interpolators
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_NMR_INTERPOLATOR_KIND = Quantity(
        type=str,
        shape=[],
        description='''
        the interpolator to use
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_NMR_INTERPOLATOR_MAX_ITER = Quantity(
        type=str,
        shape=[],
        description='''
        the maximum number of iterations
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_NMR_INTERPOLATOR_PRECOND = Quantity(
        type=str,
        shape=[],
        description='''
        The preconditioner used for the linear solver of the spline3 methods
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_NMR_INTERPOLATOR_SAFE_COMPUTATION = Quantity(
        type=str,
        shape=[],
        description='''
        if a non unrolled calculation is to be performed in parallel
        ''')

    x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_NMR_INTERPOLATOR_CONV_INFO = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_NMR_INTERPOLATOR_CONV_INFO'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_NMR(MSection):
    '''
    The chemical shift is calculated by DFPT.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_NMR_INTERPOLATE_SHIFT = Quantity(
        type=str,
        shape=[],
        description='''
        Calculate the soft part of the chemical shift by interpolation
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_NMR_NICS_FILE_NAME = Quantity(
        type=str,
        shape=[],
        description='''
        Name of the file with the NICS points coordinates
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_NMR_NICS = Quantity(
        type=str,
        shape=[],
        description='''
        Calculate the chemical shift in a set of points   given from an external file
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_NMR_RESTART_NMR = Quantity(
        type=str,
        shape=[],
        description='''
        Restart the NMR calculation from a previous run (NOT WORKING YET)
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_NMR_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        controls the activation of the nmr calculation
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_NMR_SHIFT_GAPW_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        While computing the local part of the shift (GAPW), the integration is restricted
        to nuclei that are within this radius.
        ''')

    x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_NMR_INTERPOLATOR = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_NMR_INTERPOLATOR'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_POLAR_INTERPOLATOR_CONV_INFO(MSection):
    '''
    if convergence information about the linear solver of the spline methods should be
    printed
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_POLAR_INTERPOLATOR_CONV_INFO_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_POLAR_INTERPOLATOR_CONV_INFO_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_POLAR_INTERPOLATOR_CONV_INFO_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_POLAR_INTERPOLATOR_CONV_INFO_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_POLAR_INTERPOLATOR_CONV_INFO_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_POLAR_INTERPOLATOR(MSection):
    '''
    kind of interpolation used between the multigrids
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_POLAR_INTERPOLATOR_AINT_PRECOND = Quantity(
        type=str,
        shape=[],
        description='''
        the approximate inverse to use to get the starting point for the linear solver of
        the spline3 methods
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_POLAR_INTERPOLATOR_EPS_R = Quantity(
        type=str,
        shape=[],
        description='''
        accuracy on the residual for spline3 the interpolators
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_POLAR_INTERPOLATOR_EPS_X = Quantity(
        type=str,
        shape=[],
        description='''
        accuracy on the solution for spline3 the interpolators
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_POLAR_INTERPOLATOR_KIND = Quantity(
        type=str,
        shape=[],
        description='''
        the interpolator to use
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_POLAR_INTERPOLATOR_MAX_ITER = Quantity(
        type=str,
        shape=[],
        description='''
        the maximum number of iterations
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_POLAR_INTERPOLATOR_PRECOND = Quantity(
        type=str,
        shape=[],
        description='''
        The preconditioner used for the linear solver of the spline3 methods
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_POLAR_INTERPOLATOR_SAFE_COMPUTATION = Quantity(
        type=str,
        shape=[],
        description='''
        if a non unrolled calculation is to be performed in parallel
        ''')

    x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_POLAR_INTERPOLATOR_CONV_INFO = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_POLAR_INTERPOLATOR_CONV_INFO'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_POLAR(MSection):
    '''
    Compute polarizabilities.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_POLAR_DO_RAMAN = Quantity(
        type=str,
        shape=[],
        description='''
        Compute the electric-dipole--electric-dipole polarizability
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_POLAR_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        controls the activation of the polarizability calculation
        ''')

    x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_POLAR_INTERPOLATOR = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_POLAR_INTERPOLATOR'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_SPINSPIN_INTERPOLATOR_CONV_INFO(MSection):
    '''
    if convergence information about the linear solver of the spline methods should be
    printed
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_SPINSPIN_INTERPOLATOR_CONV_INFO_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_SPINSPIN_INTERPOLATOR_CONV_INFO_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_SPINSPIN_INTERPOLATOR_CONV_INFO_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_SPINSPIN_INTERPOLATOR_CONV_INFO_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_SPINSPIN_INTERPOLATOR_CONV_INFO_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_SPINSPIN_INTERPOLATOR(MSection):
    '''
    kind of interpolation used between the multigrids
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_SPINSPIN_INTERPOLATOR_AINT_PRECOND = Quantity(
        type=str,
        shape=[],
        description='''
        the approximate inverse to use to get the starting point for the linear solver of
        the spline3 methods
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_SPINSPIN_INTERPOLATOR_EPS_R = Quantity(
        type=str,
        shape=[],
        description='''
        accuracy on the residual for spline3 the interpolators
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_SPINSPIN_INTERPOLATOR_EPS_X = Quantity(
        type=str,
        shape=[],
        description='''
        accuracy on the solution for spline3 the interpolators
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_SPINSPIN_INTERPOLATOR_KIND = Quantity(
        type=str,
        shape=[],
        description='''
        the interpolator to use
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_SPINSPIN_INTERPOLATOR_MAX_ITER = Quantity(
        type=str,
        shape=[],
        description='''
        the maximum number of iterations
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_SPINSPIN_INTERPOLATOR_PRECOND = Quantity(
        type=str,
        shape=[],
        description='''
        The preconditioner used for the linear solver of the spline3 methods
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_SPINSPIN_INTERPOLATOR_SAFE_COMPUTATION = Quantity(
        type=str,
        shape=[],
        description='''
        if a non unrolled calculation is to be performed in parallel
        ''')

    x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_SPINSPIN_INTERPOLATOR_CONV_INFO = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_SPINSPIN_INTERPOLATOR_CONV_INFO'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_SPINSPIN(MSection):
    '''
    Compute indirect spin-spin coupling constants.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_SPINSPIN_DO_DSO = Quantity(
        type=str,
        shape=[],
        description='''
        Compute the diamagnetic spin-orbit contribution (NOT YET IMPLEMENTED)
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_SPINSPIN_DO_FC = Quantity(
        type=str,
        shape=[],
        description='''
        Compute the Fermi contact contribution
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_SPINSPIN_DO_PSO = Quantity(
        type=str,
        shape=[],
        description='''
        Compute the paramagnetic spin-orbit contribution
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_SPINSPIN_DO_SD = Quantity(
        type=str,
        shape=[],
        description='''
        Compute the spin-dipolar contribution
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_SPINSPIN_ISSC_ON_ATOM_LIST = Quantity(
        type=str,
        shape=[],
        description='''
        Atoms for which the issc is computed.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_SPINSPIN_RESTART_SPINSPIN = Quantity(
        type=str,
        shape=[],
        description='''
        Restart the spin-spin calculation from a previous run (NOT WORKING YET)
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_SPINSPIN_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        controls the activation of the nmr calculation
        ''')

    x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_SPINSPIN_INTERPOLATOR = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_SPINSPIN_INTERPOLATOR'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES(MSection):
    '''
    The linear response is used to calculate one of the  following properties: nmr, epr,
    raman, ...
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_ENERGY_GAP = Quantity(
        type=str,
        shape=[],
        description='''
        Energy gap estimate [a.u.] for preconditioning
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_EPS = Quantity(
        type=str,
        shape=[],
        description='''
        target accuracy for the convergence of the conjugate gradient.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_MAX_ITER = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum number of conjugate gradient iteration to be performed for one
        optimization.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_PRECONDITIONER = Quantity(
        type=str,
        shape=[],
        description='''
        Type of preconditioner to be used with all minimization schemes. They differ in
        effectiveness, cost of construction, cost of application. Properly preconditioned
        minimization can be orders of magnitude faster than doing nothing.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_RESTART_EVERY = Quantity(
        type=str,
        shape=[],
        description='''
        Restart the conjugate gradient after the specified number of iterations.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_RESTART = Quantity(
        type=str,
        shape=[],
        description='''
        Restart the response calculation if the restart file exists
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_LINRES_WFN_RESTART_FILE_NAME = Quantity(
        type=str,
        shape=[],
        description='''
        Root of the file names where to read the response functions fromwhich to restart
        the calculation of the linear response
        ''')

    x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_CURRENT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_CURRENT'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_EPR = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_EPR'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_LOCALIZE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_LOCALIZE'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_NMR = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_NMR'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_POLAR = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_POLAR'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_SPINSPIN = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES_SPINSPIN'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_PROPERTIES_RESP_CONSTRAINT(MSection):
    '''
    specifies a linear constraint on the fitted charges.This can be used to give equal
    values to equivalent atoms.sum over atom_list c_i * q_i = t
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PROPERTIES_RESP_CONSTRAINT_ATOM_COEF = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the coefficient of the atom in this linear constraint
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_RESP_CONSTRAINT_ATOM_LIST = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the list of atoms involved in this constraint
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_RESP_CONSTRAINT_EQUAL_CHARGES = Quantity(
        type=str,
        shape=[],
        description='''
        All atoms in ATOM_LIST are constrained to have the same charges. When using this
        keyword, TARGET and ATOM_COEF do not need to be set and will be ignored. Instead
        of using this keyword, the constraint section could be repeated.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_RESP_CONSTRAINT_TARGET = Quantity(
        type=str,
        shape=[],
        description='''
        the target value for the constraint
        ''')


class x_cp2k_section_input_FORCE_EVAL_PROPERTIES_RESP_NONPERIODIC_SYS(MSection):
    '''
    Specifies the parameter for sampling the RESP fitting points for non-periodic systems,
    i.e. systems that do not involve surfaces. This section can be used with periodic and
    nonperiodic Poisson solvers, it only affects the sampling of grid points. All grid
    points in the shell defined by rmin and rmax are accepted for fitting.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PROPERTIES_RESP_NONPERIODIC_SYS_RMAX_KIND = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the maximum distance a fit point is away from an atom of a given kind
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_RESP_NONPERIODIC_SYS_RMAX = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the maximum distance a fit point is away from an atom. Valid for all
        atomic kinds for which no RMAX_KIND are specified.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_RESP_NONPERIODIC_SYS_RMIN_KIND = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the minimum distance a fit point is away from an atom of a given kind
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_RESP_NONPERIODIC_SYS_RMIN = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the minimum distance a fit point is away from an atom. Valid for all
        atomic kinds for which no RMIN_KIND are specified.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_RESP_NONPERIODIC_SYS_X_HI = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the upper boundary of the box along X used to sample the potential.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_RESP_NONPERIODIC_SYS_X_LOW = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the lower boundary of the box along X used to sample the potential.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_RESP_NONPERIODIC_SYS_Y_HI = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the upper boundary of the box along Y used to sample the potential.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_RESP_NONPERIODIC_SYS_Y_LOW = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the lower boundary of the box along Y used to sample the potential.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_RESP_NONPERIODIC_SYS_Z_HI = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the upper boundary of the box along Z used to sample the potential.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_RESP_NONPERIODIC_SYS_Z_LOW = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the lower boundary of the box along Z used to sample the potential.
        ''')


class x_cp2k_section_input_FORCE_EVAL_PROPERTIES_RESP_PERIODIC_SYS(MSection):
    '''
    Specifies the parameter for sampling the RESP fitting points for periodic systems,
    i.e. systems that involve surfaces. This section can only be used with periodic
    Poisson solver and cell. To see, which grid points were used, switch on
    COORD_FIT_POINTS in the PRINT section.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PROPERTIES_RESP_PERIODIC_SYS_ATOM_LIST = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the list of indexes of atoms used to define the region for the RESP
        fitting. The list should contain indexes of atoms of the first surface layer.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_RESP_PERIODIC_SYS_LENGTH = Quantity(
        type=str,
        shape=[],
        description='''
        Length of the sampling box, i.e. a box of this length and the height specified by
        RANGE is defined above each surface atom given in ATOM_LIST. The grid points in
        the boxes are accepted as fitting point. Should be in the range of the nearest
        neighbour distance (a bit larger to be on the safe side). Allows for a refined
        sampling of grid points in case of corrugated surfaces.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_RESP_PERIODIC_SYS_RANGE = Quantity(
        type=str,
        shape=[],
        description='''
        Range where the fitting points are sampled. A range of 3 to 5 Angstroms means that
        the fitting points are sampled in the region of 3 to 5 Angstroms above the surface
        which is defined by atom indexes given in ATOM_LIST.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_RESP_PERIODIC_SYS_SURF_DIRECTION = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies what above the surface means. Defines the direction.
        ''')


class x_cp2k_section_input_FORCE_EVAL_PROPERTIES_RESP_RESTRAINT(MSection):
    '''
    specifies a restraint on the fitted charges.This can be used to restrain values to
    zero.s*(sum over atom_list q_i - t)**2
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PROPERTIES_RESP_RESTRAINT_ATOM_COEF = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the coefficient of the atom in this linear restraint. If given, the
        restraint will be: s*(sum over atom_list c_i * q_i - t)**2
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_RESP_RESTRAINT_ATOM_LIST = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the list of atoms involved in this restraint
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_RESP_RESTRAINT_STRENGTH = Quantity(
        type=str,
        shape=[],
        description='''
        the target value for the constraint
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_RESP_RESTRAINT_TARGET = Quantity(
        type=str,
        shape=[],
        description='''
        the target value for the restraint
        ''')


class x_cp2k_section_input_FORCE_EVAL_PROPERTIES_RESP(MSection):
    '''
    Requests a RESP fit of charges. When using a periodic Poisson solver and a periodic
    cell, the periodic RESP routines are used. If the Hartree potential matches with the
    one of an isolated system (i.e. isolated Poisson solver and big, nonperiodic cells),
    the nonperiodic RESP routines are automatically used. The subsections NONPERIODIC_SYS
    and PERIODIC_SYS do not request a nonperiodic or periodic solution, they only
    determine the sampling of the fitting points. All restraints are harmonic!
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_PROPERTIES_RESP_INTEGER_TOTAL_CHARGE = Quantity(
        type=str,
        shape=[],
        description='''
        Forces the total charge to be integer
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_RESP_RESTRAIN_HEAVIES_STRENGTH = Quantity(
        type=str,
        shape=[],
        description='''
        If defined, enforce the restraint of non-hydrogen atoms to zero. Its value is the
        strength of the restraint on the heavy atoms.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_RESP_RESTRAIN_HEAVIES_TO_ZERO = Quantity(
        type=str,
        shape=[],
        description='''
        Restrain non-hydrogen atoms to zero.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_RESP_STRIDE = Quantity(
        type=str,
        shape=[],
        description='''
        The stride (X,Y,Z) used to write the cube file (larger values result in smaller
        cube files). You can provide 3 numbers (for X,Y,Z) or 1 number valid for all
        components.
        ''')

    x_cp2k_input_FORCE_EVAL_PROPERTIES_RESP_WIDTH = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the value of the width of the Gaussian charge distribution carried by
        each atom. Needs only to be specified when using a periodic Poisson solver.
        ''')

    x_cp2k_section_input_FORCE_EVAL_PROPERTIES_RESP_CONSTRAINT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PROPERTIES_RESP_CONSTRAINT'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_PROPERTIES_RESP_NONPERIODIC_SYS = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PROPERTIES_RESP_NONPERIODIC_SYS'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_PROPERTIES_RESP_PERIODIC_SYS = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PROPERTIES_RESP_PERIODIC_SYS'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_PROPERTIES_RESP_RESTRAINT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PROPERTIES_RESP_RESTRAINT'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_PROPERTIES(MSection):
    '''
    This section is used to set up the PROPERTIES calculation.
    '''

    m_def = Section(validate=False)

    x_cp2k_section_input_FORCE_EVAL_PROPERTIES_ATOMIC = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PROPERTIES_ATOMIC'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_PROPERTIES_ET_COUPLING = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PROPERTIES_ET_COUPLING'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_PROPERTIES_FIT_CHARGE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PROPERTIES_FIT_CHARGE'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PROPERTIES_LINRES'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_PROPERTIES_RESP = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_PROPERTIES_RESP'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_QMMM_CELL_CELL_REF(MSection):
    '''
    Input parameters needed to set up the CELL_REF.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_CELL_CELL_REF_ABC = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the lengths of the cell vectors A, B, and C, which defines the diagonal
        elements of h matrix for an orthorhombic cell. For non-orthorhombic cells it is
        possible either to specify the angles ALPHA, BETA, GAMMA via ALPHA_BETA_GAMMA
        keyword or alternatively use the keywords A, B, and C. The convention is that A
        lies along the X-axis, B is in the XY plane.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_CELL_CELL_REF_ALPHA_BETA_GAMMA = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the angles between the vectors A, B and C when using the ABC keyword. The
        convention is that A lies along the X-axis, B is in the XY plane. ALPHA is the
        angle between B and C, BETA is the angle between A and C and GAMMA the angle
        between A and B.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_CELL_CELL_REF_A = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the Cartesian components for the cell vector A. This defines the first
        column of the h matrix.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_CELL_CELL_REF_B = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the Cartesian components for the cell vector B. This defines the second
        column of the h matrix.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_CELL_CELL_REF_CELL_FILE_FORMAT = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the format of the cell file (if used)
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_CELL_CELL_REF_CELL_FILE_NAME = Quantity(
        type=str,
        shape=[],
        description='''
        Possibility to read the cell from an external file
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_CELL_CELL_REF_C = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the Cartesian components for the cell vector C. This defines the third
        column of the h matrix.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_CELL_CELL_REF_MULTIPLE_UNIT_CELL = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the numbers of repetition in space (X, Y, Z) of the defined cell,
        assuming it as a unit cell. This keyword affects only the CELL specification. The
        same keyword in SUBSYS%TOPOLOGY%MULTIPLE_UNIT_CELL should be modified in order to
        affect the coordinates specification.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_CELL_CELL_REF_PERIODIC = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the directions for which periodic boundary conditions (PBC) will be
        applied. Important notice: This applies to the generation of the pair lists as
        well as to the application of the PBCs to positions. See the POISSON section to
        specify the periodicity used for the electrostatics. Typically the settings should
        be the same.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_CELL_CELL_REF_SYMMETRY = Quantity(
        type=str,
        shape=[],
        description='''
        Imposes an initial cell symmetry.
        ''')


class x_cp2k_section_input_FORCE_EVAL_QMMM_CELL(MSection):
    '''
    Input parameters needed to set up the CELL.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_CELL_ABC = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the lengths of the cell vectors A, B, and C, which defines the diagonal
        elements of h matrix for an orthorhombic cell. For non-orthorhombic cells it is
        possible either to specify the angles ALPHA, BETA, GAMMA via ALPHA_BETA_GAMMA
        keyword or alternatively use the keywords A, B, and C. The convention is that A
        lies along the X-axis, B is in the XY plane.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_CELL_ALPHA_BETA_GAMMA = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the angles between the vectors A, B and C when using the ABC keyword. The
        convention is that A lies along the X-axis, B is in the XY plane. ALPHA is the
        angle between B and C, BETA is the angle between A and C and GAMMA the angle
        between A and B.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_CELL_A = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the Cartesian components for the cell vector A. This defines the first
        column of the h matrix.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_CELL_B = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the Cartesian components for the cell vector B. This defines the second
        column of the h matrix.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_CELL_CELL_FILE_FORMAT = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the format of the cell file (if used)
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_CELL_CELL_FILE_NAME = Quantity(
        type=str,
        shape=[],
        description='''
        Possibility to read the cell from an external file
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_CELL_C = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the Cartesian components for the cell vector C. This defines the third
        column of the h matrix.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_CELL_MULTIPLE_UNIT_CELL = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the numbers of repetition in space (X, Y, Z) of the defined cell,
        assuming it as a unit cell. This keyword affects only the CELL specification. The
        same keyword in SUBSYS%TOPOLOGY%MULTIPLE_UNIT_CELL should be modified in order to
        affect the coordinates specification.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_CELL_PERIODIC = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the directions for which periodic boundary conditions (PBC) will be
        applied. Important notice: This applies to the generation of the pair lists as
        well as to the application of the PBCs to positions. See the POISSON section to
        specify the periodicity used for the electrostatics. Typically the settings should
        be the same.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_CELL_SYMMETRY = Quantity(
        type=str,
        shape=[],
        description='''
        Imposes an initial cell symmetry.
        ''')

    x_cp2k_section_input_FORCE_EVAL_QMMM_CELL_CELL_REF = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_CELL_CELL_REF'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_LINKS_LINK_ADD_MM_CHARGE(MSection):
    '''
    Specify information to add a classical charge before the QM/MM energies and forces
    evaluation
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_LINKS_LINK_ADD_MM_CHARGE_ALPHA = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the scaling factor that defines the movement along the defined direction
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_LINKS_LINK_ADD_MM_CHARGE_ATOM_INDEX_1 = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the index of the first atom defining the direction along which the atom
        will be added
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_LINKS_LINK_ADD_MM_CHARGE_ATOM_INDEX_2 = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the index of the second atom defining the direction along which  the
        atom will be added
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_LINKS_LINK_ADD_MM_CHARGE_CHARGE = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the charge for the added source of QM/MM potential
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_LINKS_LINK_ADD_MM_CHARGE_CORR_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the correction radius used for the QM/MM electrostatic coupling for the
        added source
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_LINKS_LINK_ADD_MM_CHARGE_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the radius used for the QM/MM electrostatic coupling for the added
        source
        ''')


class x_cp2k_section_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_LINKS_LINK_MOVE_MM_CHARGE(MSection):
    '''
    Specify information to move a classical charge before the QM/MM energies and forces
    evaluation
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_LINKS_LINK_MOVE_MM_CHARGE_ALPHA = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the scaling factor that defines the movement along the defined direction
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_LINKS_LINK_MOVE_MM_CHARGE_ATOM_INDEX_1 = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the index of the MM atom involved in the QM/MM link to be moved
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_LINKS_LINK_MOVE_MM_CHARGE_ATOM_INDEX_2 = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the index of the second atom defining the direction along which  the
        atom will be moved
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_LINKS_LINK_MOVE_MM_CHARGE_CORR_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the correction radius used for the QM/MM electrostatic coupling after
        movement
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_LINKS_LINK_MOVE_MM_CHARGE_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the radius used for the QM/MM electrostatic coupling after movement
        ''')


class x_cp2k_section_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_LINKS_LINK(MSection):
    '''
    Specify information on the QM/MM link treatment
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_LINKS_LINK_ALPHA_IMOMM = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the scaling factor to be used for projecting the forces on the capping
        hydrogen in the IMOMM QM/MM link scheme to the MM atom of the link. A good guess
        can be derived from the bond distances of the forcefield: alpha = r_eq(QM-MM) /
        r_eq(QM-H).
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_LINKS_LINK_CORR_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Overwrite the specification of the correction radius only for the MM atom involved
        in the link.Default is to use the same correction radius as for the specified
        type.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_LINKS_LINK_FIST_SCALE_FACTOR = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the scaling factor for the MM charge involved in the link QM/MM. This
        keyword modifies the MM charge in FIST. The modified charge will be used then also
        for the generation of the QM/MM potential. Default 1.0 i.e. no charge rescaling of
        the MM atom of the QM/MM link bond.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_LINKS_LINK_LINK_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the method to use to treat the defined QM/MM link
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_LINKS_LINK_MM_INDEX = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the index of the MM atom involved in the QM/MM link, Default hydrogen.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_LINKS_LINK_QM_INDEX = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the index of the QM atom involved in the QM/MM link
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_LINKS_LINK_QM_KIND = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the element of the QM capping atom involved in the QM/MM link
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_LINKS_LINK_QMMM_SCALE_FACTOR = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the scaling factor for the MM charge involved in the link QM/MM. This
        keyword affects only the QM/MM potential, it doesn't affect the electrostatic in
        the classical part of the code. Default 1.0 i.e. no charge rescaling of the MM
        atom of the QM/MM link bond.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_LINKS_LINK_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Overwrite the specification of the radius only for the MM atom involved in the
        link.Default is to use the same radius as for the specified type.
        ''')

    x_cp2k_section_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_LINKS_LINK_ADD_MM_CHARGE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_LINKS_LINK_ADD_MM_CHARGE'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_LINKS_LINK_MOVE_MM_CHARGE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_LINKS_LINK_MOVE_MM_CHARGE'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_LINKS(MSection):
    '''
    Information about possible links for automatic covalent bond breaking for the buffer
    QM/MM calculation.Ignored - need to implement buffer selection by atom and walking of
    connectivity data.
    '''

    m_def = Section(validate=False)

    x_cp2k_section_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_LINKS_LINK = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_LINKS_LINK'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_NON_ADAPTIVE_LINK_ADD_MM_CHARGE(MSection):
    '''
    Specify information to add a classical charge before the QM/MM energies and forces
    evaluation
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_NON_ADAPTIVE_LINK_ADD_MM_CHARGE_ALPHA = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the scaling factor that defines the movement along the defined direction
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_NON_ADAPTIVE_LINK_ADD_MM_CHARGE_ATOM_INDEX_1 = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the index of the first atom defining the direction along which the atom
        will be added
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_NON_ADAPTIVE_LINK_ADD_MM_CHARGE_ATOM_INDEX_2 = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the index of the second atom defining the direction along which  the
        atom will be added
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_NON_ADAPTIVE_LINK_ADD_MM_CHARGE_CHARGE = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the charge for the added source of QM/MM potential
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_NON_ADAPTIVE_LINK_ADD_MM_CHARGE_CORR_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the correction radius used for the QM/MM electrostatic coupling for the
        added source
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_NON_ADAPTIVE_LINK_ADD_MM_CHARGE_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the radius used for the QM/MM electrostatic coupling for the added
        source
        ''')


class x_cp2k_section_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_NON_ADAPTIVE_LINK_MOVE_MM_CHARGE(MSection):
    '''
    Specify information to move a classical charge before the QM/MM energies and forces
    evaluation
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_NON_ADAPTIVE_LINK_MOVE_MM_CHARGE_ALPHA = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the scaling factor that defines the movement along the defined direction
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_NON_ADAPTIVE_LINK_MOVE_MM_CHARGE_ATOM_INDEX_1 = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the index of the MM atom involved in the QM/MM link to be moved
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_NON_ADAPTIVE_LINK_MOVE_MM_CHARGE_ATOM_INDEX_2 = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the index of the second atom defining the direction along which  the
        atom will be moved
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_NON_ADAPTIVE_LINK_MOVE_MM_CHARGE_CORR_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the correction radius used for the QM/MM electrostatic coupling after
        movement
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_NON_ADAPTIVE_LINK_MOVE_MM_CHARGE_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the radius used for the QM/MM electrostatic coupling after movement
        ''')


class x_cp2k_section_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_NON_ADAPTIVE_LINK(MSection):
    '''
    Specify information on the QM/MM link treatment
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_NON_ADAPTIVE_LINK_ALPHA_IMOMM = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the scaling factor to be used for projecting the forces on the capping
        hydrogen in the IMOMM QM/MM link scheme to the MM atom of the link. A good guess
        can be derived from the bond distances of the forcefield: alpha = r_eq(QM-MM) /
        r_eq(QM-H).
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_NON_ADAPTIVE_LINK_CORR_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Overwrite the specification of the correction radius only for the MM atom involved
        in the link.Default is to use the same correction radius as for the specified
        type.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_NON_ADAPTIVE_LINK_FIST_SCALE_FACTOR = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the scaling factor for the MM charge involved in the link QM/MM. This
        keyword modifies the MM charge in FIST. The modified charge will be used then also
        for the generation of the QM/MM potential. Default 1.0 i.e. no charge rescaling of
        the MM atom of the QM/MM link bond.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_NON_ADAPTIVE_LINK_LINK_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the method to use to treat the defined QM/MM link
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_NON_ADAPTIVE_LINK_MM_INDEX = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the index of the MM atom involved in the QM/MM link, Default hydrogen.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_NON_ADAPTIVE_LINK_QM_INDEX = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the index of the QM atom involved in the QM/MM link
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_NON_ADAPTIVE_LINK_QM_KIND = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the element of the QM capping atom involved in the QM/MM link
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_NON_ADAPTIVE_LINK_QMMM_SCALE_FACTOR = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the scaling factor for the MM charge involved in the link QM/MM. This
        keyword affects only the QM/MM potential, it doesn't affect the electrostatic in
        the classical part of the code. Default 1.0 i.e. no charge rescaling of the MM
        atom of the QM/MM link bond.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_NON_ADAPTIVE_LINK_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Overwrite the specification of the radius only for the MM atom involved in the
        link.Default is to use the same radius as for the specified type.
        ''')

    x_cp2k_section_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_NON_ADAPTIVE_LINK_ADD_MM_CHARGE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_NON_ADAPTIVE_LINK_ADD_MM_CHARGE'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_NON_ADAPTIVE_LINK_MOVE_MM_CHARGE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_NON_ADAPTIVE_LINK_MOVE_MM_CHARGE'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_NON_ADAPTIVE_QM_KIND(MSection):
    '''
    Information about the qm kind in the qm/mm scheme
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_NON_ADAPTIVE_QM_KIND_MM_INDEX = Quantity(
        type=str,
        shape=[],
        description='''
        The indexes of the mm atoms that have this kind. This keyword can be repeated
        several times (useful if you have to specify many indexes).
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_NON_ADAPTIVE_QM_KIND_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        The qm kind
        ''')


class x_cp2k_section_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_NON_ADAPTIVE(MSection):
    '''
    List of atoms always in buffer region, non-adaptively, and any needed LINK sections
    '''

    m_def = Section(validate=False)

    x_cp2k_section_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_NON_ADAPTIVE_LINK = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_NON_ADAPTIVE_LINK'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_NON_ADAPTIVE_QM_KIND = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_NON_ADAPTIVE_QM_KIND'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_QMMM_FORCE_MIXING_QM_NON_ADAPTIVE_QM_KIND(MSection):
    '''
    Information about the qm kind in the qm/mm scheme
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_QM_NON_ADAPTIVE_QM_KIND_MM_INDEX = Quantity(
        type=str,
        shape=[],
        description='''
        The indexes of the mm atoms that have this kind. This keyword can be repeated
        several times (useful if you have to specify many indexes).
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_QM_NON_ADAPTIVE_QM_KIND_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        The qm kind
        ''')


class x_cp2k_section_input_FORCE_EVAL_QMMM_FORCE_MIXING_QM_NON_ADAPTIVE(MSection):
    '''
    List of atoms always in QM region, non-adaptively
    '''

    m_def = Section(validate=False)

    x_cp2k_section_input_FORCE_EVAL_QMMM_FORCE_MIXING_QM_NON_ADAPTIVE_QM_KIND = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_FORCE_MIXING_QM_NON_ADAPTIVE_QM_KIND'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_QMMM_FORCE_MIXING_RESTART_INFO(MSection):
    '''
    This section provides information about old force-mixing indices and labels, for
    restarts.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_RESTART_INFO_INDICES = Quantity(
        type=str,
        shape=[],
        description='''
        Indices of atoms in previous step QM regions.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_RESTART_INFO_LABELS = Quantity(
        type=str,
        shape=[],
        description='''
        Labels of atoms in previous step QM regions.
        ''')


class x_cp2k_section_input_FORCE_EVAL_QMMM_FORCE_MIXING(MSection):
    '''
    This section enables and defines parameters for force-mixing based QM/MM, which
    actually does two conventional QM/MM calculations, on a small  and a large QM region,
    and combines the MM forces from one and QM  forces from the other to create a complete
    set of forces.  Energy is  not conserved (although the QM/MM energy from the large QM
    region calculation is reported)  so a proper thermostat (i.e. massive, and able to
    handle dissipation, such as  Adaptive Langevin (AD_LANGEVIN)) must be used. For some
    propagation algorithms  (NVT and REFTRAJ MD ensembles) algorithm is adaptive,
    including molecules hysteretically based on their instantaneous distance from the core
    region.  Information on core/QM/buffer labels can be written in PDB file using
    MOTION&PRINT&FORCE_MIXING_LABELS.  Will fail if calculation requires a  meaningfull
    stress, or an energy that is consistent with the forces.  For GEO_OPT this means  only
    MOTION&GEO_OPT&TYPE CG, MOTION&GEO_OPT&CG&LINE_SEARCH&TYPE 2PNT, and
    MOTION&GEO_OPT&CG&LINE_SEARCH&2PNT&LINMIN_GRAD_ONLY T
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_ADAPTIVE_EXCLUDE_MOLECULES = Quantity(
        type=str,
        shape=[],
        description='''
        List of molecule names to exclude from adaptive regions (e.g. big things like
        proteins)
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_EXTENDED_DELTA_CHARGE = Quantity(
        type=str,
        shape=[],
        description='''
        Additional net charge in extended region relative to core (core charge is
        specified in DFT section, as usual for a convetional QM/MM calculation)
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_MAX_N_QM = Quantity(
        type=str,
        shape=[],
        description='''
        Maximum number of QM atoms, for detection of runaway adaptive selection.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_MOMENTUM_CONSERVATION_REGION = Quantity(
        type=str,
        shape=[],
        description='''
        Region to apply correction force to for momentum conservation
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_MOMENTUM_CONSERVATION_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        How to apply force to get momentum conservation
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_QM_EXTENDED_SEED_IS_ONLY_CORE_LIST = Quantity(
        type=str,
        shape=[],
        description='''
        Makes the extended QM zone be defined hysterestically  by distance from QM core
        list (i.e. atoms specified explicitly by  user) instead of from full QM core
        region (specified by user + hysteretic  selection + unbreakable bonds)
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_QM_KIND_ELEMENT_MAPPING = Quantity(
        type=str,
        shape=[],
        description='''
        Mapping from elements to QM_KINDs for adaptively included atoms.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_R_BUF = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the inner and outer radii of buffer region.  All atoms within this
        distance (hysteretically) of any QM atoms  will be buffer atoms in the force-
        mixing calculation.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_R_CORE = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the inner and outer radii of core QM region.  All molecules with any atoms
        within this distance (hysteretically) of any atoms  specified as QM in enclosing
        QM/MM section  will be core QM atoms in the force-mixing calculation.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_R_QM = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the inner and outer radii of QM dynamics region.  All molecules with atoms
        within this distance (hysteretically) of any atoms in  core will follow QM
        dynamics in the force-mixing calculation.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCE_MIXING_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Enables force-mixing
        ''')

    x_cp2k_section_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_LINKS = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_LINKS'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_NON_ADAPTIVE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_FORCE_MIXING_BUFFER_NON_ADAPTIVE'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_QMMM_FORCE_MIXING_QM_NON_ADAPTIVE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_FORCE_MIXING_QM_NON_ADAPTIVE'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_QMMM_FORCE_MIXING_RESTART_INFO = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_FORCE_MIXING_RESTART_INFO'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED_GENPOT(MSection):
    '''
    This section specifies the input parameters for a generic potential type.A functional
    form is specified. Mathematical Operators recognized are +, -, *, /, ** or
    alternatively ^, whereas symbols for brackets must be (). The function parser
    recognizes the (single argument) Fortran 90 intrinsic functions abs, exp, log10, log,
    sqrt, sinh, cosh, tanh, sin, cos, tan, asin, acos, atan. Parsing for INTRINSIC
    functions is CASE INsensitive.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED_GENPOT_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the atomic kind involved in the generic potential
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED_GENPOT_FUNCTION = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the functional form in mathematical notation.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED_GENPOT_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the parameters of the functional form
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED_GENPOT_RCUT = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the cutoff parameter of the generic potential
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED_GENPOT_RMAX = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the upper bound of the potential. If not set the range is the full range
        generate by the spline
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED_GENPOT_RMIN = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the lower bound of the potential. If not set the range is the full range
        generate by the spline
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED_GENPOT_UNITS = Quantity(
        type=str,
        shape=[],
        description='''
        Optionally, allows to define valid CP2K unit strings for each parameter value. It
        is assumed that the corresponding parameter value is specified in this unit.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED_GENPOT_VALUES = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the values of parameter of the functional form
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED_GENPOT_VARIABLES = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the variable of the functional form.
        ''')


class x_cp2k_section_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED_GOODWIN(MSection):
    '''
    This section specifies the input parameters for GOODWIN potential type.Functional
    form: V(r) = EXP(M*(-(r/DC)**MC+(D/DC)**MC))*VR0*(D/r)**M.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED_GOODWIN_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the atomic kind involved in the nonbond potential
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED_GOODWIN_DC = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the DC parameter of the Goodwin potential
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED_GOODWIN_D = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the D parameter of the Goodwin potential
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED_GOODWIN_MC = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the MC parameter of the Goodwin potential
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED_GOODWIN_M = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the M parameter of the Goodwin potential
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED_GOODWIN_RCUT = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the cutoff parameter of the Goodwin potential
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED_GOODWIN_RMAX = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the upper bound of the potential. If not set the range is the full range
        generate by the spline
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED_GOODWIN_RMIN = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the lower bound of the potential. If not set the range is the full range
        generate by the spline
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED_GOODWIN_VR0 = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the VR0 parameter of the Goodwin potential
        ''')


class x_cp2k_section_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED_LENNARD_JONES(MSection):
    '''
    This section specifies the input parameters for LENNARD-JONES potential
    type.Functional form: V(r) = 4.0 * EPSILON * [(SIGMA/r)^12-(SIGMA/r)^6].
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED_LENNARD_JONES_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the atomic kind involved in the nonbond potential
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED_LENNARD_JONES_EPSILON = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the EPSILON parameter of the LJ potential
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED_LENNARD_JONES_RCUT = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the cutoff parameter of the LJ potential
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED_LENNARD_JONES_RMAX = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the upper bound of the potential. If not set the range is the full range
        generate by the spline
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED_LENNARD_JONES_RMIN = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the lower bound of the potential. If not set the range is the full range
        generate by the spline
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED_LENNARD_JONES_SIGMA = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the SIGMA parameter of the LJ potential
        ''')


class x_cp2k_section_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED_WILLIAMS(MSection):
    '''
    This section specifies the input parameters for WILLIAMS potential type.Functional
    form: V(r) = A*EXP(-B*r) - C / r^6 .
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED_WILLIAMS_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the atomic kind involved in the nonbond potential
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED_WILLIAMS_A = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the A parameter of the Williams potential
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED_WILLIAMS_B = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the B parameter of the Williams potential
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED_WILLIAMS_C = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the C parameter of the Williams potential
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED_WILLIAMS_RCUT = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the cutoff parameter of the Williams potential
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED_WILLIAMS_RMAX = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the upper bound of the potential. If not set the range is the full range
        generate by the spline
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED_WILLIAMS_RMIN = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the lower bound of the potential. If not set the range is the full range
        generate by the spline
        ''')


class x_cp2k_section_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED14_GENPOT(MSection):
    '''
    This section specifies the input parameters for a generic potential type.A functional
    form is specified. Mathematical Operators recognized are +, -, *, /, ** or
    alternatively ^, whereas symbols for brackets must be (). The function parser
    recognizes the (single argument) Fortran 90 intrinsic functions abs, exp, log10, log,
    sqrt, sinh, cosh, tanh, sin, cos, tan, asin, acos, atan. Parsing for INTRINSIC
    functions is CASE INsensitive.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED14_GENPOT_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the atomic kind involved in the generic potential
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED14_GENPOT_FUNCTION = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the functional form in mathematical notation.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED14_GENPOT_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the parameters of the functional form
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED14_GENPOT_RCUT = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the cutoff parameter of the generic potential
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED14_GENPOT_RMAX = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the upper bound of the potential. If not set the range is the full range
        generate by the spline
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED14_GENPOT_RMIN = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the lower bound of the potential. If not set the range is the full range
        generate by the spline
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED14_GENPOT_UNITS = Quantity(
        type=str,
        shape=[],
        description='''
        Optionally, allows to define valid CP2K unit strings for each parameter value. It
        is assumed that the corresponding parameter value is specified in this unit.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED14_GENPOT_VALUES = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the values of parameter of the functional form
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED14_GENPOT_VARIABLES = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the variable of the functional form.
        ''')


class x_cp2k_section_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED14_GOODWIN(MSection):
    '''
    This section specifies the input parameters for GOODWIN potential type.Functional
    form: V(r) = EXP(M*(-(r/DC)**MC+(D/DC)**MC))*VR0*(D/r)**M.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED14_GOODWIN_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the atomic kind involved in the nonbond potential
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED14_GOODWIN_DC = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the DC parameter of the Goodwin potential
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED14_GOODWIN_D = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the D parameter of the Goodwin potential
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED14_GOODWIN_MC = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the MC parameter of the Goodwin potential
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED14_GOODWIN_M = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the M parameter of the Goodwin potential
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED14_GOODWIN_RCUT = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the cutoff parameter of the Goodwin potential
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED14_GOODWIN_RMAX = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the upper bound of the potential. If not set the range is the full range
        generate by the spline
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED14_GOODWIN_RMIN = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the lower bound of the potential. If not set the range is the full range
        generate by the spline
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED14_GOODWIN_VR0 = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the VR0 parameter of the Goodwin potential
        ''')


class x_cp2k_section_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED14_LENNARD_JONES(MSection):
    '''
    This section specifies the input parameters for LENNARD-JONES potential
    type.Functional form: V(r) = 4.0 * EPSILON * [(SIGMA/r)^12-(SIGMA/r)^6].
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED14_LENNARD_JONES_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the atomic kind involved in the nonbond potential
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED14_LENNARD_JONES_EPSILON = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the EPSILON parameter of the LJ potential
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED14_LENNARD_JONES_RCUT = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the cutoff parameter of the LJ potential
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED14_LENNARD_JONES_RMAX = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the upper bound of the potential. If not set the range is the full range
        generate by the spline
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED14_LENNARD_JONES_RMIN = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the lower bound of the potential. If not set the range is the full range
        generate by the spline
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED14_LENNARD_JONES_SIGMA = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the SIGMA parameter of the LJ potential
        ''')


class x_cp2k_section_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED14_WILLIAMS(MSection):
    '''
    This section specifies the input parameters for WILLIAMS potential type.Functional
    form: V(r) = A*EXP(-B*r) - C / r^6 .
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED14_WILLIAMS_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the atomic kind involved in the nonbond potential
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED14_WILLIAMS_A = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the A parameter of the Williams potential
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED14_WILLIAMS_B = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the B parameter of the Williams potential
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED14_WILLIAMS_C = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the C parameter of the Williams potential
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED14_WILLIAMS_RCUT = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the cutoff parameter of the Williams potential
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED14_WILLIAMS_RMAX = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the upper bound of the potential. If not set the range is the full range
        generate by the spline
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED14_WILLIAMS_RMIN = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the lower bound of the potential. If not set the range is the full range
        generate by the spline
        ''')


class x_cp2k_section_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED14(MSection):
    '''
    This section specifies the input parameters for 1-4 NON-BONDED interactions.
    '''

    m_def = Section(validate=False)

    x_cp2k_section_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED14_GENPOT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED14_GENPOT'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED14_GOODWIN = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED14_GOODWIN'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED14_LENNARD_JONES = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED14_LENNARD_JONES'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED14_WILLIAMS = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED14_WILLIAMS'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED(MSection):
    '''
    Specify information on the QM/MM non-bonded forcefield
    '''

    m_def = Section(validate=False)

    x_cp2k_section_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED_GENPOT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED_GENPOT'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED_GOODWIN = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED_GOODWIN'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED_LENNARD_JONES = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED_LENNARD_JONES'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED_WILLIAMS = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED_WILLIAMS'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_QMMM_FORCEFIELD(MSection):
    '''
    Specify information on the QM/MM forcefield
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_FORCEFIELD_MULTIPLE_POTENTIAL = Quantity(
        type=str,
        shape=[],
        description='''
        Enables the possibility to define NONBONDED and NONBONDED14 as a sum of different
        kinds of potential. Useful for piecewise defined potentials.
        ''')

    x_cp2k_section_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED14 = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED14'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_FORCEFIELD_NONBONDED'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_QMMM_IMAGE_CHARGE(MSection):
    '''
    Inclusion of polarization effects within the image charge approach for systems where
    QM molecules are physisorbed on e.g. metal surfaces described by MM. QM box size has
    to be equal to MM box size.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_IMAGE_CHARGE_DETERM_COEFF = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies how the coefficients are determined.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_IMAGE_CHARGE_EXT_POTENTIAL = Quantity(
        type=str,
        shape=[],
        description='''
        External potential applied to the metal electrode
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_IMAGE_CHARGE_IMAGE_RESTART_FILE_NAME = Quantity(
        type=str,
        shape=[],
        description='''
        File name where to read the image matrix used as preconditioner in the iterative
        scheme
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_IMAGE_CHARGE_MM_ATOM_LIST = Quantity(
        type=str,
        shape=[],
        description='''
        List of MM atoms carrying an induced Gaussian charge. If this keyword is not
        given, all MM atoms will carry an image charge.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_IMAGE_CHARGE_RESTART_IMAGE_MATRIX = Quantity(
        type=str,
        shape=[],
        description='''
        Restart the image matrix. Useful when calculating coefficients iteratively (the
        image matrix is used as preconditioner in that case)
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_IMAGE_CHARGE_WIDTH = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the value of the width of the (induced) Gaussian charge distribution
        carried by each MM atom.
        ''')


class x_cp2k_section_input_FORCE_EVAL_QMMM_INTERPOLATOR_CONV_INFO(MSection):
    '''
    if convergence information about the linear solver of the spline methods should be
    printed
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_INTERPOLATOR_CONV_INFO_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_INTERPOLATOR_CONV_INFO_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_INTERPOLATOR_CONV_INFO_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_INTERPOLATOR_CONV_INFO_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_INTERPOLATOR_CONV_INFO_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_FORCE_EVAL_QMMM_INTERPOLATOR_SPL_COEFFS(MSection):
    '''
    outputs a cube with the coefficents calculated for the spline interpolation
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_INTERPOLATOR_SPL_COEFFS_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_INTERPOLATOR_SPL_COEFFS_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_INTERPOLATOR_SPL_COEFFS_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_INTERPOLATOR_SPL_COEFFS_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_INTERPOLATOR_SPL_COEFFS_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_FORCE_EVAL_QMMM_INTERPOLATOR(MSection):
    '''
    kind of interpolation used between the multigrids
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_INTERPOLATOR_AINT_PRECOND = Quantity(
        type=str,
        shape=[],
        description='''
        the approximate inverse to use to get the starting point for the linear solver of
        the spline3 methods
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_INTERPOLATOR_EPS_R = Quantity(
        type=str,
        shape=[],
        description='''
        accuracy on the residual for spline3 the interpolators
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_INTERPOLATOR_EPS_X = Quantity(
        type=str,
        shape=[],
        description='''
        accuracy on the solution for spline3 the interpolators
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_INTERPOLATOR_KIND = Quantity(
        type=str,
        shape=[],
        description='''
        the interpolator to use
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_INTERPOLATOR_MAX_ITER = Quantity(
        type=str,
        shape=[],
        description='''
        the maximum number of iterations
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_INTERPOLATOR_PRECOND = Quantity(
        type=str,
        shape=[],
        description='''
        The preconditioner used for the linear solver of the spline3 methods
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_INTERPOLATOR_SAFE_COMPUTATION = Quantity(
        type=str,
        shape=[],
        description='''
        if a non unrolled calculation is to be performed in parallel
        ''')

    x_cp2k_section_input_FORCE_EVAL_QMMM_INTERPOLATOR_CONV_INFO = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_INTERPOLATOR_CONV_INFO'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_QMMM_INTERPOLATOR_SPL_COEFFS = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_INTERPOLATOR_SPL_COEFFS'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_QMMM_LINK_ADD_MM_CHARGE(MSection):
    '''
    Specify information to add a classical charge before the QM/MM energies and forces
    evaluation
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_LINK_ADD_MM_CHARGE_ALPHA = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the scaling factor that defines the movement along the defined direction
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_LINK_ADD_MM_CHARGE_ATOM_INDEX_1 = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the index of the first atom defining the direction along which the atom
        will be added
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_LINK_ADD_MM_CHARGE_ATOM_INDEX_2 = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the index of the second atom defining the direction along which  the
        atom will be added
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_LINK_ADD_MM_CHARGE_CHARGE = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the charge for the added source of QM/MM potential
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_LINK_ADD_MM_CHARGE_CORR_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the correction radius used for the QM/MM electrostatic coupling for the
        added source
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_LINK_ADD_MM_CHARGE_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the radius used for the QM/MM electrostatic coupling for the added
        source
        ''')


class x_cp2k_section_input_FORCE_EVAL_QMMM_LINK_MOVE_MM_CHARGE(MSection):
    '''
    Specify information to move a classical charge before the QM/MM energies and forces
    evaluation
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_LINK_MOVE_MM_CHARGE_ALPHA = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the scaling factor that defines the movement along the defined direction
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_LINK_MOVE_MM_CHARGE_ATOM_INDEX_1 = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the index of the MM atom involved in the QM/MM link to be moved
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_LINK_MOVE_MM_CHARGE_ATOM_INDEX_2 = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the index of the second atom defining the direction along which  the
        atom will be moved
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_LINK_MOVE_MM_CHARGE_CORR_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the correction radius used for the QM/MM electrostatic coupling after
        movement
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_LINK_MOVE_MM_CHARGE_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the radius used for the QM/MM electrostatic coupling after movement
        ''')


class x_cp2k_section_input_FORCE_EVAL_QMMM_LINK(MSection):
    '''
    Specify information on the QM/MM link treatment
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_LINK_ALPHA_IMOMM = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the scaling factor to be used for projecting the forces on the capping
        hydrogen in the IMOMM QM/MM link scheme to the MM atom of the link. A good guess
        can be derived from the bond distances of the forcefield: alpha = r_eq(QM-MM) /
        r_eq(QM-H).
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_LINK_CORR_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Overwrite the specification of the correction radius only for the MM atom involved
        in the link.Default is to use the same correction radius as for the specified
        type.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_LINK_FIST_SCALE_FACTOR = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the scaling factor for the MM charge involved in the link QM/MM. This
        keyword modifies the MM charge in FIST. The modified charge will be used then also
        for the generation of the QM/MM potential. Default 1.0 i.e. no charge rescaling of
        the MM atom of the QM/MM link bond.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_LINK_LINK_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the method to use to treat the defined QM/MM link
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_LINK_MM_INDEX = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the index of the MM atom involved in the QM/MM link, Default hydrogen.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_LINK_QM_INDEX = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the index of the QM atom involved in the QM/MM link
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_LINK_QM_KIND = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the element of the QM capping atom involved in the QM/MM link
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_LINK_QMMM_SCALE_FACTOR = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the scaling factor for the MM charge involved in the link QM/MM. This
        keyword affects only the QM/MM potential, it doesn't affect the electrostatic in
        the classical part of the code. Default 1.0 i.e. no charge rescaling of the MM
        atom of the QM/MM link bond.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_LINK_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Overwrite the specification of the radius only for the MM atom involved in the
        link.Default is to use the same radius as for the specified type.
        ''')

    x_cp2k_section_input_FORCE_EVAL_QMMM_LINK_ADD_MM_CHARGE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_LINK_ADD_MM_CHARGE'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_QMMM_LINK_MOVE_MM_CHARGE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_LINK_MOVE_MM_CHARGE'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_QMMM_MM_KIND(MSection):
    '''
    Information about the mm kind in the qm/mm scheme
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_MM_KIND_CORR_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the correction radius of the atomic kinds The correction radius is
        connected to the use of the compatibility keyword.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_MM_KIND_RADIUS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the radius of the atomic kinds
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_MM_KIND_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        The MM  kind
        ''')


class x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_CHECK_SPLINE(MSection):
    '''
    Controls the checking of the G-space term Spline Interpolation.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_CHECK_SPLINE_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_CHECK_SPLINE_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_CHECK_SPLINE_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_CHECK_SPLINE_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_CHECK_SPLINE_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_INTERPOLATOR_CONV_INFO(MSection):
    '''
    if convergence information about the linear solver of the spline methods should be
    printed
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_INTERPOLATOR_CONV_INFO_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_INTERPOLATOR_CONV_INFO_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_INTERPOLATOR_CONV_INFO_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_INTERPOLATOR_CONV_INFO_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_INTERPOLATOR_CONV_INFO_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_INTERPOLATOR(MSection):
    '''
    controls the interpolation for the G-space term
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_INTERPOLATOR_AINT_PRECOND = Quantity(
        type=str,
        shape=[],
        description='''
        the approximate inverse to use to get the starting point for the linear solver of
        the spline3 methods
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_INTERPOLATOR_EPS_R = Quantity(
        type=str,
        shape=[],
        description='''
        accuracy on the residual for spline3 the interpolators
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_INTERPOLATOR_EPS_X = Quantity(
        type=str,
        shape=[],
        description='''
        accuracy on the solution for spline3 the interpolators
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_INTERPOLATOR_MAX_ITER = Quantity(
        type=str,
        shape=[],
        description='''
        the maximum number of iterations
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_INTERPOLATOR_PRECOND = Quantity(
        type=str,
        shape=[],
        description='''
        The preconditioner used for the linear solver of the spline3 methods
        ''')

    x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_INTERPOLATOR_CONV_INFO = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_INTERPOLATOR_CONV_INFO'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_MULTIPOLE_CHECK_SPLINE(MSection):
    '''
    Controls the checking of the G-space term Spline Interpolation.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_MULTIPOLE_CHECK_SPLINE_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_MULTIPOLE_CHECK_SPLINE_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_MULTIPOLE_CHECK_SPLINE_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_MULTIPOLE_CHECK_SPLINE_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_MULTIPOLE_CHECK_SPLINE_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_MULTIPOLE_INTERPOLATOR_CONV_INFO(MSection):
    '''
    if convergence information about the linear solver of the spline methods should be
    printed
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_MULTIPOLE_INTERPOLATOR_CONV_INFO_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_MULTIPOLE_INTERPOLATOR_CONV_INFO_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_MULTIPOLE_INTERPOLATOR_CONV_INFO_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_MULTIPOLE_INTERPOLATOR_CONV_INFO_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_MULTIPOLE_INTERPOLATOR_CONV_INFO_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_MULTIPOLE_INTERPOLATOR(MSection):
    '''
    controls the interpolation for the G-space term
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_MULTIPOLE_INTERPOLATOR_AINT_PRECOND = Quantity(
        type=str,
        shape=[],
        description='''
        the approximate inverse to use to get the starting point for the linear solver of
        the spline3 methods
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_MULTIPOLE_INTERPOLATOR_EPS_R = Quantity(
        type=str,
        shape=[],
        description='''
        accuracy on the residual for spline3 the interpolators
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_MULTIPOLE_INTERPOLATOR_EPS_X = Quantity(
        type=str,
        shape=[],
        description='''
        accuracy on the solution for spline3 the interpolators
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_MULTIPOLE_INTERPOLATOR_MAX_ITER = Quantity(
        type=str,
        shape=[],
        description='''
        the maximum number of iterations
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_MULTIPOLE_INTERPOLATOR_PRECOND = Quantity(
        type=str,
        shape=[],
        description='''
        The preconditioner used for the linear solver of the spline3 methods
        ''')

    x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_MULTIPOLE_INTERPOLATOR_CONV_INFO = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_MULTIPOLE_INTERPOLATOR_CONV_INFO'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_MULTIPOLE_PROGRAM_RUN_INFO(MSection):
    '''
    Controls the printing of basic information during the run
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_MULTIPOLE_PROGRAM_RUN_INFO_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_MULTIPOLE_PROGRAM_RUN_INFO_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_MULTIPOLE_PROGRAM_RUN_INFO_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_MULTIPOLE_PROGRAM_RUN_INFO_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_MULTIPOLE_PROGRAM_RUN_INFO_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_MULTIPOLE(MSection):
    '''
    This section is used to set up the decoupling of QM periodic images with the use of
    density derived atomic point charges. Switched on by default even if not explicitly
    given. Can be switched off if e.g. QM and MM box are of the same size.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_MULTIPOLE_ANALYTICAL_GTERM = Quantity(
        type=str,
        shape=[],
        description='''
        Evaluates the Gterm in the Ewald Scheme analytically instead of using Splines.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_MULTIPOLE_EWALD_PRECISION = Quantity(
        type=str,
        shape=[],
        description='''
        Precision achieved in the Ewald sum.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_MULTIPOLE_NGRIDS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the number of grid points used for the Interpolation of the G-space term
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_MULTIPOLE_RCUT = Quantity(
        type=str,
        shape=[],
        description='''
        Real space cutoff for the Ewald sum.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_MULTIPOLE_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the usage of the multipole section
        ''')

    x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_MULTIPOLE_CHECK_SPLINE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_MULTIPOLE_CHECK_SPLINE'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_MULTIPOLE_INTERPOLATOR = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_MULTIPOLE_INTERPOLATOR'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_MULTIPOLE_PROGRAM_RUN_INFO = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_MULTIPOLE_PROGRAM_RUN_INFO'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_EWALD_MULTIPOLES(MSection):
    '''
    Enables the use of multipoles in the treatment of the electrostatics.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_EWALD_MULTIPOLES_EPS_POL = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the rmsd threshold for the derivatives of the energy towards the Cartesian
        dipoles components
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_EWALD_MULTIPOLES_MAX_IPOL_ITER = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the maximum number of iterations for induced dipoles
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_EWALD_MULTIPOLES_MAX_MULTIPOLE_EXPANSION = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the maximum level of multipoles expansion used  for the electrostatics.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_EWALD_MULTIPOLES_POL_SCF = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the method to obtain self consistent induced multipole moments.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_EWALD_MULTIPOLES_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Controls the activation of the Multipoles
        ''')


class x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_EWALD_RS_GRID(MSection):
    '''
    Set options that influence how the realspace grids are being distributed in parallel
    runs.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_EWALD_RS_GRID_DISTRIBUTION_LAYOUT = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the number of slices in the x, y and z directions.-1 specifies that any
        number of slices is OK.If a given distribution can not be satisfied, a replicated
        grid will result.Also see LOCK_DISTRIBUTION.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_EWALD_RS_GRID_DISTRIBUTION_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Parallelization strategy.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_EWALD_RS_GRID_HALO_REDUCTION_FACTOR = Quantity(
        type=str,
        shape=[],
        description='''
        Can be used to reduce the halo of the distributed grid (experimental features).
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_EWALD_RS_GRID_LOCK_DISTRIBUTION = Quantity(
        type=str,
        shape=[],
        description='''
        Expert use only, only basic QS deals correctly with a non-default value.If the
        distribution is locked, a grid will have the same distribution asthe next finer
        multigrid (provided it is distributed).If unlocked, all grids can be distributed
        freely.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_EWALD_RS_GRID_MAX_DISTRIBUTED_LEVEL = Quantity(
        type=str,
        shape=[],
        description='''
        If the multigrid-level of a grid is larger than the parameter, it will not be
        distributed in the automatic scheme.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_EWALD_RS_GRID_MEMORY_FACTOR = Quantity(
        type=str,
        shape=[],
        description='''
        A grid will only be distributed if the memory usage for that grid (including halo)
        is smaller than a replicated grid by this parameter.
        ''')


class x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_EWALD(MSection):
    '''
    Ewald parameters controlling electrostatic only for CLASSICAL MM.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_EWALD_ALPHA = Quantity(
        type=str,
        shape=[],
        description='''
        alpha parameter associated with Ewald (EWALD|PME|SPME). Recommended for small
        systems is is alpha = 3.5 / r_cut. Tuning alpha, r_cut and gmax is needed to
        obtain O(N**1.5) scaling for ewald.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_EWALD_EPSILON = Quantity(
        type=str,
        shape=[],
        description='''
        tolerance of gaussians for fft interpolation (PME only)
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_EWALD_EWALD_ACCURACY = Quantity(
        type=str,
        shape=[],
        description='''
        Expected accuracy in the Ewald sum. This number affects only the calculation of
        the cutoff for the real-space term of the ewald summation (EWALD|PME|SPME) as well
        as the construction of the neighbor lists (if the cutoff for non-bonded terms is
        smaller than the value employed to compute the EWALD real-space term). This
        keyword has no effect on the reciprocal space term (which can be tuned
        independently).
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_EWALD_EWALD_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        The type of ewald you want to perform.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_EWALD_GMAX = Quantity(
        type=str,
        shape=[],
        description='''
        number of grid points (SPME and EWALD). If a single number is specified,the same
        number of points is used for all three directions on the grid.If three numbers are
        given, each direction can have a different number of points.The number of points
        needs to be FFTable (which depends on the library used) and odd for EWALD.The
        optimal number depends e.g. on alpha and the size of the cell. 1 point per
        Angstrom is common.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_EWALD_NS_MAX = Quantity(
        type=str,
        shape=[],
        description='''
        number of grid points on small mesh (PME only), should be odd.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_EWALD_O_SPLINE = Quantity(
        type=str,
        shape=[],
        description='''
        order of the beta-Euler spline (SPME only)
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_EWALD_RCUT = Quantity(
        type=str,
        shape=[],
        description='''
        Explicitly provide the real-space cutoff of the ewald summation (EWALD|PME|SPME).
        If present, overwrites the estimate of EWALD_ACCURACY and may affect the
        construction of the neighbor lists for non-bonded terms (in FIST), if the value
        specified is larger than the cutoff for non-bonded interactions.
        ''')

    x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_EWALD_MULTIPOLES = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_EWALD_MULTIPOLES'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_EWALD_RS_GRID = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_EWALD_RS_GRID'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_MT(MSection):
    '''
    Sets up parameters of  Martyna-Tuckerman poisson solver. Note that exact results are
    only guaranteed if the unit cell is twice as large as charge density (and serious
    artefacts can result if the cell is much smaller).
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_MT_ALPHA = Quantity(
        type=str,
        shape=[],
        description='''
        Convergence parameter ALPHA*RMIN. Default value 7.0
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_MT_REL_CUTOFF = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the multiplicative factor for the CUTOFF keyword in MULTI_GRID  section.
        The result gives the cutoff at which the 1/r non-periodic FFT3D is
        evaluated.Default is 2.0
        ''')


class x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_MULTIPOLE_CHECK_SPLINE(MSection):
    '''
    Controls the checking of the G-space term Spline Interpolation.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_MULTIPOLE_CHECK_SPLINE_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_MULTIPOLE_CHECK_SPLINE_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_MULTIPOLE_CHECK_SPLINE_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_MULTIPOLE_CHECK_SPLINE_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_MULTIPOLE_CHECK_SPLINE_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_MULTIPOLE_INTERPOLATOR_CONV_INFO(MSection):
    '''
    if convergence information about the linear solver of the spline methods should be
    printed
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_MULTIPOLE_INTERPOLATOR_CONV_INFO_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_MULTIPOLE_INTERPOLATOR_CONV_INFO_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_MULTIPOLE_INTERPOLATOR_CONV_INFO_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_MULTIPOLE_INTERPOLATOR_CONV_INFO_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_MULTIPOLE_INTERPOLATOR_CONV_INFO_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_MULTIPOLE_INTERPOLATOR(MSection):
    '''
    controls the interpolation for the G-space term
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_MULTIPOLE_INTERPOLATOR_AINT_PRECOND = Quantity(
        type=str,
        shape=[],
        description='''
        the approximate inverse to use to get the starting point for the linear solver of
        the spline3 methods
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_MULTIPOLE_INTERPOLATOR_EPS_R = Quantity(
        type=str,
        shape=[],
        description='''
        accuracy on the residual for spline3 the interpolators
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_MULTIPOLE_INTERPOLATOR_EPS_X = Quantity(
        type=str,
        shape=[],
        description='''
        accuracy on the solution for spline3 the interpolators
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_MULTIPOLE_INTERPOLATOR_MAX_ITER = Quantity(
        type=str,
        shape=[],
        description='''
        the maximum number of iterations
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_MULTIPOLE_INTERPOLATOR_PRECOND = Quantity(
        type=str,
        shape=[],
        description='''
        The preconditioner used for the linear solver of the spline3 methods
        ''')

    x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_MULTIPOLE_INTERPOLATOR_CONV_INFO = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_MULTIPOLE_INTERPOLATOR_CONV_INFO'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_MULTIPOLE_PROGRAM_RUN_INFO(MSection):
    '''
    Controls the printing of basic information during the run
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_MULTIPOLE_PROGRAM_RUN_INFO_ADD_LAST = Quantity(
        type=str,
        shape=[],
        description='''
        If the last iteration should be added, and if it should be marked symbolically
        (with lowercase letter l) or with the iteration number. Not every iteration level
        is able to identify the last iteration early enough to be able to output. When
        this keyword is activated all iteration levels are checked for the last iteration
        step.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_MULTIPOLE_PROGRAM_RUN_INFO_COMMON_ITERATION_LEVELS = Quantity(
        type=str,
        shape=[],
        description='''
        How many iterations levels should be written in the same file (no extra
        information about the actual iteration level is written to the file)
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_MULTIPOLE_PROGRAM_RUN_INFO_FILENAME = Quantity(
        type=str,
        shape=[],
        description='''
        controls part of the filename for output.  use __STD_OUT__ (exactly as written
        here) for the screen or standard logger.  use filename to obtain projectname-
        filename.  use ./filename to get filename. A middle name (if present), iteration
        numbers and extension are always added to the filename. if you want to avoid it
        use =filename, in this case the filename is always exactly as typed. Please note
        that this can lead to clashes of filenames.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_MULTIPOLE_PROGRAM_RUN_INFO_LOG_PRINT_KEY = Quantity(
        type=str,
        shape=[],
        description='''
        This keywords enables the logger for the print_key (a message is printed on screen
        everytime data, controlled by this print_key, are written)
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_MULTIPOLE_PROGRAM_RUN_INFO_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Level starting at which this proprety is printed
        ''')


class x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_MULTIPOLE(MSection):
    '''
    This section is used to set up the decoupling of QM periodic images with the use of
    density derived atomic point charges.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_MULTIPOLE_ANALYTICAL_GTERM = Quantity(
        type=str,
        shape=[],
        description='''
        Evaluates the Gterm in the Ewald Scheme analytically instead of using Splines.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_MULTIPOLE_EWALD_PRECISION = Quantity(
        type=str,
        shape=[],
        description='''
        Precision achieved in the Ewald sum.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_MULTIPOLE_NGRIDS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the number of grid points used for the Interpolation of the G-space term
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_MULTIPOLE_RCUT = Quantity(
        type=str,
        shape=[],
        description='''
        Real space cutoff for the Ewald sum.
        ''')

    x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_MULTIPOLE_CHECK_SPLINE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_MULTIPOLE_CHECK_SPLINE'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_MULTIPOLE_INTERPOLATOR = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_MULTIPOLE_INTERPOLATOR'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_MULTIPOLE_PROGRAM_RUN_INFO = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_MULTIPOLE_PROGRAM_RUN_INFO'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_WAVELET(MSection):
    '''
    Sets up parameters of  wavelet based poisson solver.This solver allows for non-
    periodic (PERIODIC NONE) boundary conditions and slab-boundary conditions (but only
    PERIODIC XZ).It does not require very large unit cells, only that the density goes to
    zero on the faces of the cell.The use of PREFERRED_FFT_LIBRARY FFTSG is required
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_WAVELET_SCF_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Type of scaling function used in the wavelet approach, the total energy depends on
        this choice,and the convergence with respect to cutoff depends on the selected
        scaling functions.Possible values are 8,14,16,20,24,30,40,50,60,100
        ''')


class x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_POISSON(MSection):
    '''
    Sets up the poisson resolutor.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_PERIODIC = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the directions on wich apply PBC. Important notice,  this only applies to
        the electrostatics. See the CELL section to specify the periodicity used for e.g.
        the pair lists. Typically the settings should be the same.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_POISSON_SOLVER = Quantity(
        type=str,
        shape=[],
        description='''
        Specify which kind of solver to use to solve the Poisson equation.
        ''')

    x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_EWALD = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_EWALD'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_MT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_MT'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_MULTIPOLE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_MULTIPOLE'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_WAVELET = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_POISSON_WAVELET'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC(MSection):
    '''
    Specify parameters for QM/MM periodic boundary conditions calculations
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_GMAX = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the maximum value of G in the reciprocal space over which perform the
        Ewald sum.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_NGRIDS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the number of grid points used for the Interpolation of the G-space term
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PERIODIC_REPLICA = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the number of replica to take into consideration for the real part of
        the calculation. Default is letting the qmmm module decide how many replica you
        really need.
        ''')

    x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_CHECK_SPLINE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_CHECK_SPLINE'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_INTERPOLATOR = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_INTERPOLATOR'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_MULTIPOLE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_MULTIPOLE'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_POISSON = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC_POISSON'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_QMMM_QM_KIND(MSection):
    '''
    Information about the qm kind in the qm/mm scheme
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_QM_KIND_MM_INDEX = Quantity(
        type=str,
        shape=[],
        description='''
        The indexes of the mm atoms that have this kind. This keyword can be repeated
        several times (useful if you have to specify many indexes).
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_QM_KIND_SECTION_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        The qm kind
        ''')


class x_cp2k_section_input_FORCE_EVAL_QMMM_WALLS(MSection):
    '''
    Enables Walls for the QM box. This can be used to avoid that QM  atoms move out of the
    QM box.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_WALLS_K = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the value of the the force constant for the quadratic wall
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_WALLS_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the type of wall
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_WALLS_WALL_SKIN = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the value of the skin of the Wall in each dimension. The wall's effect is
        felt when atoms fall within the skin of the Wall.
        ''')


class x_cp2k_section_input_FORCE_EVAL_QMMM(MSection):
    '''
    Input for QM/MM calculations.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_QMMM_CENTER_GRID = Quantity(
        type=str,
        shape=[],
        description='''
        This keyword specifies whether the QM system is centered in units of the grid
        spacing.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_CENTER_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        How to do the centering
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_CENTER = Quantity(
        type=str,
        shape=[],
        description='''
        This keyword sets when the qm system is automatically centered.  Default is
        EVERY_STEP.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_DELTA_CHARGE = Quantity(
        type=str,
        shape=[],
        description='''
        Additional net charge relative to that specified in DFT section.  Used
        automatically by force mixing
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_E_COUPL = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the type of the QM - MM electrostatic coupling.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_EPS_MM_RSPACE = Quantity(
        type=str,
        shape=[],
        description='''
        Set the threshold for the collocation of the GEEP gaussian functions.this keyword
        affects only the GAUSS E_COUPLING.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_INITIAL_TRANSLATION_VECTOR = Quantity(
        type=str,
        shape=[],
        description='''
        This keyword specify the initial translation vector to be applied to the system.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_MM_POTENTIAL_FILE_NAME = Quantity(
        type=str,
        shape=[],
        description='''
        Name of the file containing the potential expansion in gaussians. See the
        USE_GEEP_LIB keyword.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_NOCOMPATIBILITY = Quantity(
        type=str,
        shape=[],
        description='''
        This keyword disables the compatibility of QM/MM potential between CPMD and CP2K
        implementations. The compatibility is achieved using an MM potential of the form:
        Erf[x/rc]/x + (1/rc -2/(pi^1/2*rc))*Exp[-(x/rc)^2] .This keyword has effect only
        selecting GAUSS E_COUPLING type.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_PARALLEL_SCHEME = Quantity(
        type=str,
        shape=[],
        description='''
        Chooses the parallel_scheme for the long range Potential
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_SPHERICAL_CUTOFF = Quantity(
        type=str,
        shape=[],
        description='''
        Set the spherical cutoff for the QMMM electrostatic interaction. This acts like a
        charge multiplicative factor dependent on cutoff. For MM atoms farther than the
        SPHERICAL_CUTOFF(1) their charge is zero. The switch is performed with a smooth
        function: 0.5*(1-TANH((r-[SPH_CUT(1)-20*SPH_CUT(2)])/(SPH_CUT(2)))). Two values
        are required: the first one is the distance cutoff. The second one controls the
        stiffness of the smoothing.
        ''')

    x_cp2k_input_FORCE_EVAL_QMMM_USE_GEEP_LIB = Quantity(
        type=str,
        shape=[],
        description='''
        This keyword enables the use of the internal GEEP library to generate the gaussian
        expansion of the MM potential. Using this keyword there's no need to provide the
        MM_POTENTIAL_FILENAME. It expects a number from 2 to 15 (the number of gaussian
        funtions to be used in the expansion.
        ''')

    x_cp2k_section_input_FORCE_EVAL_QMMM_CELL = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_CELL'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_QMMM_FORCE_MIXING = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_FORCE_MIXING'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_QMMM_FORCEFIELD = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_FORCEFIELD'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_QMMM_IMAGE_CHARGE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_IMAGE_CHARGE'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_QMMM_INTERPOLATOR = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_INTERPOLATOR'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_QMMM_LINK = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_LINK'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_QMMM_MM_KIND = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_MM_KIND'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_PERIODIC'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_QMMM_QM_KIND = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_QM_KIND'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_QMMM_WALLS = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_QMMM_WALLS'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_RESCALE_FORCES(MSection):
    '''
    Section controlling the rescaling of forces. Useful when starting from quite bad
    geometries with unphysically large forces.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_RESCALE_FORCES_MAX_FORCE = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the Maximum Values of the force. If the force of one atom exceed this
        value it's rescaled to the MAX_FORCE value.
        ''')


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_CELL_CELL_REF(MSection):
    '''
    Input parameters needed to set up the CELL_REF.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_CELL_CELL_REF_ABC = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the lengths of the cell vectors A, B, and C, which defines the diagonal
        elements of h matrix for an orthorhombic cell. For non-orthorhombic cells it is
        possible either to specify the angles ALPHA, BETA, GAMMA via ALPHA_BETA_GAMMA
        keyword or alternatively use the keywords A, B, and C. The convention is that A
        lies along the X-axis, B is in the XY plane.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_CELL_CELL_REF_ALPHA_BETA_GAMMA = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the angles between the vectors A, B and C when using the ABC keyword. The
        convention is that A lies along the X-axis, B is in the XY plane. ALPHA is the
        angle between B and C, BETA is the angle between A and C and GAMMA the angle
        between A and B.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_CELL_CELL_REF_A = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the Cartesian components for the cell vector A. This defines the first
        column of the h matrix.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_CELL_CELL_REF_B = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the Cartesian components for the cell vector B. This defines the second
        column of the h matrix.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_CELL_CELL_REF_CELL_FILE_FORMAT = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the format of the cell file (if used)
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_CELL_CELL_REF_CELL_FILE_NAME = Quantity(
        type=str,
        shape=[],
        description='''
        Possibility to read the cell from an external file
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_CELL_CELL_REF_C = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the Cartesian components for the cell vector C. This defines the third
        column of the h matrix.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_CELL_CELL_REF_MULTIPLE_UNIT_CELL = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the numbers of repetition in space (X, Y, Z) of the defined cell,
        assuming it as a unit cell. This keyword affects only the CELL specification. The
        same keyword in SUBSYS%TOPOLOGY%MULTIPLE_UNIT_CELL should be modified in order to
        affect the coordinates specification.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_CELL_CELL_REF_PERIODIC = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the directions for which periodic boundary conditions (PBC) will be
        applied. Important notice: This applies to the generation of the pair lists as
        well as to the application of the PBCs to positions. See the POISSON section to
        specify the periodicity used for the electrostatics. Typically the settings should
        be the same.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_CELL_CELL_REF_SYMMETRY = Quantity(
        type=str,
        shape=[],
        description='''
        Imposes an initial cell symmetry.
        ''')


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_CELL(MSection):
    '''
    Input parameters needed to set up the CELL.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_CELL_ABC = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the lengths of the cell vectors A, B, and C, which defines the diagonal
        elements of h matrix for an orthorhombic cell. For non-orthorhombic cells it is
        possible either to specify the angles ALPHA, BETA, GAMMA via ALPHA_BETA_GAMMA
        keyword or alternatively use the keywords A, B, and C. The convention is that A
        lies along the X-axis, B is in the XY plane.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_CELL_ALPHA_BETA_GAMMA = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the angles between the vectors A, B and C when using the ABC keyword. The
        convention is that A lies along the X-axis, B is in the XY plane. ALPHA is the
        angle between B and C, BETA is the angle between A and C and GAMMA the angle
        between A and B.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_CELL_A = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the Cartesian components for the cell vector A. This defines the first
        column of the h matrix.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_CELL_B = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the Cartesian components for the cell vector B. This defines the second
        column of the h matrix.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_CELL_CELL_FILE_FORMAT = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the format of the cell file (if used)
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_CELL_CELL_FILE_NAME = Quantity(
        type=str,
        shape=[],
        description='''
        Possibility to read the cell from an external file
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_CELL_C = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the Cartesian components for the cell vector C. This defines the third
        column of the h matrix.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_CELL_MULTIPLE_UNIT_CELL = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the numbers of repetition in space (X, Y, Z) of the defined cell,
        assuming it as a unit cell. This keyword affects only the CELL specification. The
        same keyword in SUBSYS%TOPOLOGY%MULTIPLE_UNIT_CELL should be modified in order to
        affect the coordinates specification.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_CELL_PERIODIC = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the directions for which periodic boundary conditions (PBC) will be
        applied. Important notice: This applies to the generation of the pair lists as
        well as to the application of the PBCs to positions. See the POISSON section to
        specify the periodicity used for the electrostatics. Typically the settings should
        be the same.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_CELL_SYMMETRY = Quantity(
        type=str,
        shape=[],
        description='''
        Imposes an initial cell symmetry.
        ''')

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_CELL_CELL_REF = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_CELL_CELL_REF'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_ANGLE_POINT(MSection):
    '''
    Enables the possibility to use geometrical centers instead of single atoms to define
    colvars
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_ANGLE_POINT_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the indexes of atoms defining the geometrical center
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_ANGLE_POINT_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Chooses the type of geometrical point
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_ANGLE_POINT_WEIGHTS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the weights for a weighted geometrical center. Default is 1/natoms for
        every atom
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_ANGLE_POINT_XYZ = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the xyz of the fixed point (if the case)
        ''')


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_ANGLE_PLANE_PLANE_PLANE(MSection):
    '''
    This section defines the plane. When using this colvar, two plane section must be
    defined!
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_ANGLE_PLANE_PLANE_PLANE_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the indexes of 3 atoms/points defining the plane.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_ANGLE_PLANE_PLANE_PLANE_DEF_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Specify how the plane is defined: either by 3 atoms or by a fixed normal vector.
        At least one plane must be defined through atoms.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_ANGLE_PLANE_PLANE_PLANE_NORMAL_VECTOR = Quantity(
        type=str,
        shape=[],
        description='''
        Alternatively to 3 atoms/points one can define one of the two, planes by defining
        its NORMAL vector.
        ''')


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_ANGLE_PLANE_PLANE_POINT(MSection):
    '''
    Enables the possibility to use geometrical centers instead of single atoms to define
    colvars
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_ANGLE_PLANE_PLANE_POINT_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the indexes of atoms defining the geometrical center
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_ANGLE_PLANE_PLANE_POINT_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Chooses the type of geometrical point
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_ANGLE_PLANE_PLANE_POINT_WEIGHTS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the weights for a weighted geometrical center. Default is 1/natoms for
        every atom
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_ANGLE_PLANE_PLANE_POINT_XYZ = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the xyz of the fixed point (if the case)
        ''')


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_ANGLE_PLANE_PLANE(MSection):
    '''
    This section defines the angle between two planes as a collective variables.
    '''

    m_def = Section(validate=False)

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_ANGLE_PLANE_PLANE_PLANE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_ANGLE_PLANE_PLANE_PLANE'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_ANGLE_PLANE_PLANE_POINT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_ANGLE_PLANE_PLANE_POINT'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_ANGLE(MSection):
    '''
    Section to define the angle as a collective variables.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_ANGLE_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the indexes of atoms/points defining the angle.
        ''')

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_ANGLE_POINT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_ANGLE_POINT'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_BOND_ROTATION_POINT(MSection):
    '''
    Enables the possibility to use geometrical centers instead of single atoms to define
    colvars
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_BOND_ROTATION_POINT_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the indexes of atoms defining the geometrical center
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_BOND_ROTATION_POINT_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Chooses the type of geometrical point
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_BOND_ROTATION_POINT_WEIGHTS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the weights for a weighted geometrical center. Default is 1/natoms for
        every atom
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_BOND_ROTATION_POINT_XYZ = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the xyz of the fixed point (if the case)
        ''')


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_BOND_ROTATION(MSection):
    '''
    Section to define the rotation of a bond/line with respect toanother bond/line
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_BOND_ROTATION_P1_BOND1 = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the index of atom/point defining the first pointof the first bond/line.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_BOND_ROTATION_P1_BOND2 = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the index of atom/point defining the first pointof the second bond/line.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_BOND_ROTATION_P2_BOND1 = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the index of atom/point defining the second pointof the first bond/line.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_BOND_ROTATION_P2_BOND2 = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the index of atom/point defining the second pointof the second
        bond/line.
        ''')

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_BOND_ROTATION_POINT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_BOND_ROTATION_POINT'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COLVAR_FUNC_INFO(MSection):
    '''
    Specify further data possibly used by colvars, depending on the starting geometry, for
    computing the functions value.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COLVAR_FUNC_INFO_DEFAULT_KEYWORD = Quantity(
        type=str,
        shape=[],
        description='''
        Colvar function data. The order is an internal order. So if you decide to
        edit/modify/add these values by hand you should know very well what you are
        doing.!
        ''')


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_ANGLE_POINT(MSection):
    '''
    Enables the possibility to use geometrical centers instead of single atoms to define
    colvars
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_ANGLE_POINT_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the indexes of atoms defining the geometrical center
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_ANGLE_POINT_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Chooses the type of geometrical point
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_ANGLE_POINT_WEIGHTS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the weights for a weighted geometrical center. Default is 1/natoms for
        every atom
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_ANGLE_POINT_XYZ = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the xyz of the fixed point (if the case)
        ''')


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_ANGLE_PLANE_PLANE_PLANE(MSection):
    '''
    This section defines the plane. When using this colvar, two plane section must be
    defined!
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_ANGLE_PLANE_PLANE_PLANE_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the indexes of 3 atoms/points defining the plane.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_ANGLE_PLANE_PLANE_PLANE_DEF_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Specify how the plane is defined: either by 3 atoms or by a fixed normal vector.
        At least one plane must be defined through atoms.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_ANGLE_PLANE_PLANE_PLANE_NORMAL_VECTOR = Quantity(
        type=str,
        shape=[],
        description='''
        Alternatively to 3 atoms/points one can define one of the two, planes by defining
        its NORMAL vector.
        ''')


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_ANGLE_PLANE_PLANE_POINT(MSection):
    '''
    Enables the possibility to use geometrical centers instead of single atoms to define
    colvars
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_ANGLE_PLANE_PLANE_POINT_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the indexes of atoms defining the geometrical center
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_ANGLE_PLANE_PLANE_POINT_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Chooses the type of geometrical point
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_ANGLE_PLANE_PLANE_POINT_WEIGHTS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the weights for a weighted geometrical center. Default is 1/natoms for
        every atom
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_ANGLE_PLANE_PLANE_POINT_XYZ = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the xyz of the fixed point (if the case)
        ''')


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_ANGLE_PLANE_PLANE(MSection):
    '''
    This section defines the angle between two planes as a collective variables.
    '''

    m_def = Section(validate=False)

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_ANGLE_PLANE_PLANE_PLANE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_ANGLE_PLANE_PLANE_PLANE'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_ANGLE_PLANE_PLANE_POINT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_ANGLE_PLANE_PLANE_POINT'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_ANGLE(MSection):
    '''
    Section to define the angle as a collective variables.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_ANGLE_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the indexes of atoms/points defining the angle.
        ''')

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_ANGLE_POINT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_ANGLE_POINT'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_BOND_ROTATION_POINT(MSection):
    '''
    Enables the possibility to use geometrical centers instead of single atoms to define
    colvars
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_BOND_ROTATION_POINT_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the indexes of atoms defining the geometrical center
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_BOND_ROTATION_POINT_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Chooses the type of geometrical point
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_BOND_ROTATION_POINT_WEIGHTS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the weights for a weighted geometrical center. Default is 1/natoms for
        every atom
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_BOND_ROTATION_POINT_XYZ = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the xyz of the fixed point (if the case)
        ''')


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_BOND_ROTATION(MSection):
    '''
    Section to define the rotation of a bond/line with respect toanother bond/line
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_BOND_ROTATION_P1_BOND1 = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the index of atom/point defining the first pointof the first bond/line.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_BOND_ROTATION_P1_BOND2 = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the index of atom/point defining the first pointof the second bond/line.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_BOND_ROTATION_P2_BOND1 = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the index of atom/point defining the second pointof the first bond/line.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_BOND_ROTATION_P2_BOND2 = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the index of atom/point defining the second pointof the second
        bond/line.
        ''')

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_BOND_ROTATION_POINT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_BOND_ROTATION_POINT'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_COLVAR_FUNC_INFO(MSection):
    '''
    Specify further data possibly used by colvars, depending on the starting geometry, for
    computing the functions value.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_COLVAR_FUNC_INFO_DEFAULT_KEYWORD = Quantity(
        type=str,
        shape=[],
        description='''
        Colvar function data. The order is an internal order. So if you decide to
        edit/modify/add these values by hand you should know very well what you are
        doing.!
        ''')


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_CONDITIONED_DISTANCE_POINT(MSection):
    '''
    Enables the possibility to use geometrical centers instead of single atoms to define
    colvars
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_CONDITIONED_DISTANCE_POINT_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the indexes of atoms defining the geometrical center
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_CONDITIONED_DISTANCE_POINT_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Chooses the type of geometrical point
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_CONDITIONED_DISTANCE_POINT_WEIGHTS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the weights for a weighted geometrical center. Default is 1/natoms for
        every atom
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_CONDITIONED_DISTANCE_POINT_XYZ = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the xyz of the fixed point (if the case)
        ''')


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_CONDITIONED_DISTANCE(MSection):
    '''
    Section to define the conditioned distance as a collective variable.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_CONDITIONED_DISTANCE_ATOMS_DISTANCE = Quantity(
        type=str,
        shape=[],
        description='''
        Specify indexes of atoms/points from which the distance is computed.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_CONDITIONED_DISTANCE_ATOMS_FROM = Quantity(
        type=str,
        shape=[],
        description='''
        Specify indexes of atoms/points building the coordination variable.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_CONDITIONED_DISTANCE_ATOMS_TO = Quantity(
        type=str,
        shape=[],
        description='''
        Specify indexes of atoms/points building the coordination variable.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_CONDITIONED_DISTANCE_KINDS_FROM = Quantity(
        type=str,
        shape=[],
        description='''
        Specify alternatively kinds of atoms building the coordination variable.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_CONDITIONED_DISTANCE_KINDS_TO = Quantity(
        type=str,
        shape=[],
        description='''
        Specify alternatively kinds of atoms building the coordination variable.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_CONDITIONED_DISTANCE_LAMBDA = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the lambda parameter at the exponent of the conditioned distance function.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_CONDITIONED_DISTANCE_ND = Quantity(
        type=str,
        shape=[],
        description='''
        Sets the value of the denominator of the exponential factorin the coordination
        FUNCTION.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_CONDITIONED_DISTANCE_NN = Quantity(
        type=str,
        shape=[],
        description='''
        Sets the value of the numerator of the exponential factorin the coordination
        FUNCTION.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_CONDITIONED_DISTANCE_R0 = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the R0 parameter in the coordination function.
        ''')

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_CONDITIONED_DISTANCE_POINT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_CONDITIONED_DISTANCE_POINT'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_COORDINATION_POINT(MSection):
    '''
    Enables the possibility to use geometrical centers instead of single atoms to define
    colvars
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_COORDINATION_POINT_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the indexes of atoms defining the geometrical center
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_COORDINATION_POINT_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Chooses the type of geometrical point
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_COORDINATION_POINT_WEIGHTS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the weights for a weighted geometrical center. Default is 1/natoms for
        every atom
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_COORDINATION_POINT_XYZ = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the xyz of the fixed point (if the case)
        ''')


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_COORDINATION(MSection):
    '''
    Section to define the coordination number as a collective variable.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_COORDINATION_ATOMS_FROM = Quantity(
        type=str,
        shape=[],
        description='''
        Specify indexes of atoms/points building the coordination variable.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_COORDINATION_ATOMS_TO_B = Quantity(
        type=str,
        shape=[],
        description='''
        For the CV given by the multiplication of two coorination numbers, here specify
        indexes of the third set of atoms/points.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_COORDINATION_ATOMS_TO = Quantity(
        type=str,
        shape=[],
        description='''
        Specify indexes of atoms/points building the coordination variable.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_COORDINATION_KINDS_FROM = Quantity(
        type=str,
        shape=[],
        description='''
        Specify alternatively kinds of atoms building the coordination variable.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_COORDINATION_KINDS_TO_B = Quantity(
        type=str,
        shape=[],
        description='''
        For the CV given by the multiplication of two coorination numbers, here specify
        alternatively kinds of atoms building the coordination variable.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_COORDINATION_KINDS_TO = Quantity(
        type=str,
        shape=[],
        description='''
        Specify alternatively kinds of atoms building the coordination variable.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_COORDINATION_ND_B = Quantity(
        type=str,
        shape=[],
        description='''
        For the CV given by the multiplication of two coorination numbers,Sets the value
        of the denominator of the exponential factorin the coordination FUNCTION.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_COORDINATION_ND = Quantity(
        type=str,
        shape=[],
        description='''
        Sets the value of the denominator of the exponential factorin the coordination
        FUNCTION.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_COORDINATION_NN_B = Quantity(
        type=str,
        shape=[],
        description='''
        For the CV given by the multiplication of two coorination numbers,Sets the value
        of the numerator of the exponential factorin the coordination FUNCTION.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_COORDINATION_NN = Quantity(
        type=str,
        shape=[],
        description='''
        Sets the value of the numerator of the exponential factorin the coordination
        FUNCTION.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_COORDINATION_R0_B = Quantity(
        type=str,
        shape=[],
        description='''
        For the CV given by the multiplication of two coorination numbers, specify the R0
        parameter in the second coordination function.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_COORDINATION_R0 = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the R0 parameter in the coordination function.
        ''')

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_COORDINATION_POINT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_COORDINATION_POINT'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_DISTANCE_POINT(MSection):
    '''
    Enables the possibility to use geometrical centers instead of single atoms to define
    colvars
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_DISTANCE_POINT_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the indexes of atoms defining the geometrical center
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_DISTANCE_POINT_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Chooses the type of geometrical point
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_DISTANCE_POINT_WEIGHTS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the weights for a weighted geometrical center. Default is 1/natoms for
        every atom
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_DISTANCE_POINT_XYZ = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the xyz of the fixed point (if the case)
        ''')


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_DISTANCE_FUNCTION_POINT(MSection):
    '''
    Enables the possibility to use geometrical centers instead of single atoms to define
    colvars
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_DISTANCE_FUNCTION_POINT_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the indexes of atoms defining the geometrical center
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_DISTANCE_FUNCTION_POINT_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Chooses the type of geometrical point
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_DISTANCE_FUNCTION_POINT_WEIGHTS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the weights for a weighted geometrical center. Default is 1/natoms for
        every atom
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_DISTANCE_FUNCTION_POINT_XYZ = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the xyz of the fixed point (if the case)
        ''')


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_DISTANCE_FUNCTION(MSection):
    '''
    Section to define functions between two distances as collective variables. The
    function is defined as d1+coeff*d2
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_DISTANCE_FUNCTION_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the indexes of atoms/points for the two bonds d1=(1-2) d2=(3-4).
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_DISTANCE_FUNCTION_COEFFICIENT = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the coefficient in the function for the constraint. -1.0 has to be used
        for distance difference, 1.0 for distance addition
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_DISTANCE_FUNCTION_PBC = Quantity(
        type=str,
        shape=[],
        description='''
        Whether periodic boundary conditions should be applied on the atomic position
        before computing the colvar or not.
        ''')

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_DISTANCE_FUNCTION_POINT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_DISTANCE_FUNCTION_POINT'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_DISTANCE_POINT_PLANE_POINT(MSection):
    '''
    Enables the possibility to use geometrical centers instead of single atoms to define
    colvars
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_DISTANCE_POINT_PLANE_POINT_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the indexes of atoms defining the geometrical center
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_DISTANCE_POINT_PLANE_POINT_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Chooses the type of geometrical point
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_DISTANCE_POINT_PLANE_POINT_WEIGHTS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the weights for a weighted geometrical center. Default is 1/natoms for
        every atom
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_DISTANCE_POINT_PLANE_POINT_XYZ = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the xyz of the fixed point (if the case)
        ''')


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_DISTANCE_POINT_PLANE(MSection):
    '''
    Section to define the distance of a point from a plane as a collective variables.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_DISTANCE_POINT_PLANE_ATOM_POINT = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the atom/point index defining the point.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_DISTANCE_POINT_PLANE_ATOMS_PLANE = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the indexes of atoms/points defining the plane.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_DISTANCE_POINT_PLANE_PBC = Quantity(
        type=str,
        shape=[],
        description='''
        Whether periodic boundary conditions should be applied on the atomic position
        before computing the colvar or not.
        ''')

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_DISTANCE_POINT_PLANE_POINT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_DISTANCE_POINT_PLANE_POINT'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_DISTANCE(MSection):
    '''
    Section to define the distance as a collective variables.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_DISTANCE_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the indexes of atoms/points defining the distance.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_DISTANCE_AXIS = Quantity(
        type=str,
        shape=[],
        description='''
        Define the axes along which the colvar should be evaluated
        ''')

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_DISTANCE_POINT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_DISTANCE_POINT'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_GYRATION_RADIUS_POINT(MSection):
    '''
    Enables the possibility to use geometrical centers instead of single atoms to define
    colvars
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_GYRATION_RADIUS_POINT_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the indexes of atoms defining the geometrical center
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_GYRATION_RADIUS_POINT_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Chooses the type of geometrical point
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_GYRATION_RADIUS_POINT_WEIGHTS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the weights for a weighted geometrical center. Default is 1/natoms for
        every atom
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_GYRATION_RADIUS_POINT_XYZ = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the xyz of the fixed point (if the case)
        ''')


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_GYRATION_RADIUS(MSection):
    '''
    Section to define the gyration radius as a collective variable.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_GYRATION_RADIUS_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specify indexes of atoms/points defyining the gyration radius variable.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_GYRATION_RADIUS_KINDS = Quantity(
        type=str,
        shape=[],
        description='''
        Specify alternatively kinds of atoms defining the gyration radius.
        ''')

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_GYRATION_RADIUS_POINT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_GYRATION_RADIUS_POINT'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_HBP_POINT(MSection):
    '''
    Enables the possibility to use geometrical centers instead of single atoms to define
    colvars
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_HBP_POINT_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the indexes of atoms defining the geometrical center
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_HBP_POINT_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Chooses the type of geometrical point
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_HBP_POINT_WEIGHTS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the weights for a weighted geometrical center. Default is 1/natoms for
        every atom
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_HBP_POINT_XYZ = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the xyz of the fixed point (if the case)
        ''')


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_HBP(MSection):
    '''
    Section to define the hbond wannier centre as a collective variables.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_HBP_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the indexes of atoms/points defining the bond (Od, H, Oa).
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_HBP_NPOINTS = Quantity(
        type=str,
        shape=[],
        description='''
        The number of points in the path
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_HBP_RCUT = Quantity(
        type=str,
        shape=[],
        description='''
        Parameter used for computing the cutoff radius for searching the wannier centres
        around an atom
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_HBP_SHIFT = Quantity(
        type=str,
        shape=[],
        description='''
        Parameter used for shifting each term in the sum
        ''')

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_HBP_POINT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_HBP_POINT'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_HYDRONIUM_POINT(MSection):
    '''
    Enables the possibility to use geometrical centers instead of single atoms to define
    colvars
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_HYDRONIUM_POINT_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the indexes of atoms defining the geometrical center
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_HYDRONIUM_POINT_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Chooses the type of geometrical point
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_HYDRONIUM_POINT_WEIGHTS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the weights for a weighted geometrical center. Default is 1/natoms for
        every atom
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_HYDRONIUM_POINT_XYZ = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the xyz of the fixed point (if the case)
        ''')


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_HYDRONIUM(MSection):
    '''
    Section to define the formation of a hydronium as a collective variable.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_HYDRONIUM_HYDROGENS = Quantity(
        type=str,
        shape=[],
        description='''
        Specify indexes of atoms building the coordination variable.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_HYDRONIUM_LAMBDA = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the LAMBDA parameter in the hydronium function.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_HYDRONIUM_NH = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the NH parameter in the hydronium function.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_HYDRONIUM_OXYGENS = Quantity(
        type=str,
        shape=[],
        description='''
        Specify indexes of atoms building the coordination variable.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_HYDRONIUM_PNH = Quantity(
        type=str,
        shape=[],
        description='''
        Sets the value of the numerator of the exponential factorin the coordination
        FUNCTION.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_HYDRONIUM_PNO = Quantity(
        type=str,
        shape=[],
        description='''
        Sets the value of the numerator of the exponential factorin the coordination
        FUNCTION.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_HYDRONIUM_P = Quantity(
        type=str,
        shape=[],
        description='''
        Sets the value of the numerator of the exponential factorin the coordination
        FUNCTION.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_HYDRONIUM_QNH = Quantity(
        type=str,
        shape=[],
        description='''
        Sets the value of the denominator of the exponential factorin the coordination
        FUNCTION.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_HYDRONIUM_QNO = Quantity(
        type=str,
        shape=[],
        description='''
        Sets the value of the denominator of the exponential factorin the coordination
        FUNCTION.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_HYDRONIUM_Q = Quantity(
        type=str,
        shape=[],
        description='''
        Sets the value of the denominator of the exponential factorin the coordination
        FUNCTION.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_HYDRONIUM_ROH = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the ROH parameter in the coordination function.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_HYDRONIUM_ROO = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the ROO parameter in the coordination function.
        ''')

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_HYDRONIUM_POINT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_HYDRONIUM_POINT'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_POPULATION_POINT(MSection):
    '''
    Enables the possibility to use geometrical centers instead of single atoms to define
    colvars
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_POPULATION_POINT_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the indexes of atoms defining the geometrical center
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_POPULATION_POINT_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Chooses the type of geometrical point
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_POPULATION_POINT_WEIGHTS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the weights for a weighted geometrical center. Default is 1/natoms for
        every atom
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_POPULATION_POINT_XYZ = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the xyz of the fixed point (if the case)
        ''')


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_POPULATION(MSection):
    '''
    Section to define the population of specie as a collective variable.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_POPULATION_ATOMS_FROM = Quantity(
        type=str,
        shape=[],
        description='''
        Specify indexes of atoms/points building the coordination variable.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_POPULATION_ATOMS_TO = Quantity(
        type=str,
        shape=[],
        description='''
        Specify indexes of atoms/points building the coordination variable.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_POPULATION_KINDS_FROM = Quantity(
        type=str,
        shape=[],
        description='''
        Specify alternatively kinds of atoms building the coordination variable.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_POPULATION_KINDS_TO = Quantity(
        type=str,
        shape=[],
        description='''
        Specify alternatively kinds of atoms building the coordination variable.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_POPULATION_N0 = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the n0 parameter that sets the coordination of the species.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_POPULATION_ND = Quantity(
        type=str,
        shape=[],
        description='''
        Sets the value of the denominator of the exponential factorin the coordination
        FUNCTION.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_POPULATION_NN = Quantity(
        type=str,
        shape=[],
        description='''
        Sets the value of the numerator of the exponential factorin the coordination
        FUNCTION.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_POPULATION_R0 = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the R0 parameter in the coordination function.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_POPULATION_SIGMA = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the gaussian width of used to build the population istogram.
        ''')

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_POPULATION_POINT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_POPULATION_POINT'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_QPARM_POINT(MSection):
    '''
    Enables the possibility to use geometrical centers instead of single atoms to define
    colvars
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_QPARM_POINT_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the indexes of atoms defining the geometrical center
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_QPARM_POINT_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Chooses the type of geometrical point
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_QPARM_POINT_WEIGHTS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the weights for a weighted geometrical center. Default is 1/natoms for
        every atom
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_QPARM_POINT_XYZ = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the xyz of the fixed point (if the case)
        ''')


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_QPARM(MSection):
    '''
    Section to define the Q parameter (crystalline order parameter) as a collective
    variable.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_QPARM_ALPHA = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the width of the Fermi-Dirac style smearing around RCUT.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_QPARM_ATOMS_FROM = Quantity(
        type=str,
        shape=[],
        description='''
        Specify indexes of atoms/points building the coordination variable.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_QPARM_ATOMS_TO = Quantity(
        type=str,
        shape=[],
        description='''
        Specify indexes of atoms/points building the coordination variable.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_QPARM_L = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the L spherical harmonics from Ylm.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_QPARM_RCUT = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the distance cutoff for neighbors.
        ''')

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_QPARM_POINT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_QPARM_POINT'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_RING_PUCKERING_POINT(MSection):
    '''
    Enables the possibility to use geometrical centers instead of single atoms to define
    colvars
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_RING_PUCKERING_POINT_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the indexes of atoms defining the geometrical center
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_RING_PUCKERING_POINT_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Chooses the type of geometrical point
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_RING_PUCKERING_POINT_WEIGHTS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the weights for a weighted geometrical center. Default is 1/natoms for
        every atom
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_RING_PUCKERING_POINT_XYZ = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the xyz of the fixed point (if the case)
        ''')


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_RING_PUCKERING(MSection):
    '''
    Section to define general ring puckering collective variables.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_RING_PUCKERING_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the indexes of atoms/points defining the ring.At least 4 Atoms are
        needed.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_RING_PUCKERING_COORDINATE = Quantity(
        type=str,
        shape=[],
        description='''
        Indicate the coordinate to be used. Follow the Cremer-Pople definition for a N
        ring.0 is the total puckering variable Q,2..[N/2] are puckering
        coordinates.-2..-[N/2-1] are puckering angles.
        ''')

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_RING_PUCKERING_POINT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_RING_PUCKERING_POINT'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_RMSD_FRAME_COORD(MSection):
    '''
    The positions for RMSD used for restart
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_RMSD_FRAME_COORD_DEFAULT_KEYWORD = Quantity(
        type=str,
        shape=[],
        description='''
        Specify positions of the system
        ''')


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_RMSD_FRAME(MSection):
    '''
    Specify coordinates of the frame (number of frames can be either 1 or 2)
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_RMSD_FRAME_COORD_FILE_NAME = Quantity(
        type=str,
        shape=[],
        description='''
        Name of the xyz file with coordinates (alternative to &COORD section)
        ''')

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_RMSD_FRAME_COORD = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_RMSD_FRAME_COORD'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_RMSD(MSection):
    '''
    Section to define a CV as function of RMSD computed with respect to given reference
    configurations. For 2 configurations the colvar is equal to: ss = (RMSDA-
    RMSDB)/(RMSDA+RMSDB), while if only 1 configuration is given, then the colvar is just
    the RMSD from that frame.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_RMSD_ALIGN_FRAMES = Quantity(
        type=str,
        shape=[],
        description='''
        Whether the reference frames should be aligned to minimize the RMSD
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_RMSD_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specify indexes of atoms building the subset.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_RMSD_SUBSET_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Define the subsytem used to compute the RMSD
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_RMSD_WEIGHTS = Quantity(
        type=str,
        shape=[],
        description='''
        Specify weights of atoms building the subset.
        ''')

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_RMSD_FRAME = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_RMSD_FRAME'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_TORSION_POINT(MSection):
    '''
    Enables the possibility to use geometrical centers instead of single atoms to define
    colvars
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_TORSION_POINT_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the indexes of atoms defining the geometrical center
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_TORSION_POINT_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Chooses the type of geometrical point
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_TORSION_POINT_WEIGHTS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the weights for a weighted geometrical center. Default is 1/natoms for
        every atom
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_TORSION_POINT_XYZ = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the xyz of the fixed point (if the case)
        ''')


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_TORSION(MSection):
    '''
    Section to define the torsion as a collective variables.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_TORSION_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the indexes of atoms/points defining the torsion.
        ''')

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_TORSION_POINT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_TORSION_POINT'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_U_MIXED(MSection):
    '''
    This section allows to use any function of the energy subsystems  in a mixed_env
    calculation as a collective variable.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_U_MIXED_DX = Quantity(
        type=str,
        shape=[],
        description='''
        Parameter used for computing the derivative with the Ridders method.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_U_MIXED_ENERGY_FUNCTION = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the functional form of the collective variable in mathematical notation.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_U_MIXED_ERROR_LIMIT = Quantity(
        type=str,
        shape=[],
        description='''
        Checks that the error in computing the derivative is not larger than the value
        set. In case prints a warning message.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_U_MIXED_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the parameters of the functional form
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_U_MIXED_UNITS = Quantity(
        type=str,
        shape=[],
        description='''
        Optionally, allows to define valid CP2K unit strings for each parameter value. It
        is assumed that the corresponding parameter value is specified in this unit.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_U_MIXED_VALUES = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the values of parameter of the functional form
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_U_MIXED_VARIABLES = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the variables of the functional form. To allow an efficient mapping the
        order of the energy variables will be considered identical to the order of the
        force_eval in the force_eval_order list.
        ''')


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_U(MSection):
    '''
    Section to define the energy as a generalized collective variable.
    '''

    m_def = Section(validate=False)

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_U_MIXED = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_U_MIXED'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_WC_POINT(MSection):
    '''
    Enables the possibility to use geometrical centers instead of single atoms to define
    colvars
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_WC_POINT_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the indexes of atoms defining the geometrical center
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_WC_POINT_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Chooses the type of geometrical point
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_WC_POINT_WEIGHTS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the weights for a weighted geometrical center. Default is 1/natoms for
        every atom
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_WC_POINT_XYZ = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the xyz of the fixed point (if the case)
        ''')


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_WC(MSection):
    '''
    Section to define the hbond wannier centre as a collective variables.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_WC_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the indexes of atoms/points defining the bond (Od, H, Oa).
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_WC_RCUT = Quantity(
        type=str,
        shape=[],
        description='''
        Parameter used for computing the cutoff radius for searching the wannier centres
        around an atom
        ''')

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_WC_POINT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_WC_POINT'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_XYZ_DIAG_POINT(MSection):
    '''
    Enables the possibility to use geometrical centers instead of single atoms to define
    colvars
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_XYZ_DIAG_POINT_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the indexes of atoms defining the geometrical center
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_XYZ_DIAG_POINT_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Chooses the type of geometrical point
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_XYZ_DIAG_POINT_WEIGHTS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the weights for a weighted geometrical center. Default is 1/natoms for
        every atom
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_XYZ_DIAG_POINT_XYZ = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the xyz of the fixed point (if the case)
        ''')


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_XYZ_DIAG(MSection):
    '''
    Section to define the distance of an atom from its starting position
    ((X-X(0))^2+(Y-Y(0))^2+(Z-Z(0))^2) or part of its components as a collective
    variable.If absolute_position is specified, instead the CV is represented by the
    instantaneous position of the atom (only available for X, Y or Z components).
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_XYZ_DIAG_ABSOLUTE_POSITION = Quantity(
        type=str,
        shape=[],
        description='''
        If enabled, the absolute position of the atoms will be used.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_XYZ_DIAG_ATOM = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the index of the atom/point.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_XYZ_DIAG_COMPONENT = Quantity(
        type=str,
        shape=[],
        description='''
        Define the component of the position vector which will be used as a colvar.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_XYZ_DIAG_PBC = Quantity(
        type=str,
        shape=[],
        description='''
        Whether periodic boundary conditions should be applied on the atomic position
        before computing the colvar or not.
        ''')

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_XYZ_DIAG_POINT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_XYZ_DIAG_POINT'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_XYZ_OUTERDIAG_POINT(MSection):
    '''
    Enables the possibility to use geometrical centers instead of single atoms to define
    colvars
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_XYZ_OUTERDIAG_POINT_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the indexes of atoms defining the geometrical center
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_XYZ_OUTERDIAG_POINT_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Chooses the type of geometrical point
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_XYZ_OUTERDIAG_POINT_WEIGHTS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the weights for a weighted geometrical center. Default is 1/natoms for
        every atom
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_XYZ_OUTERDIAG_POINT_XYZ = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the xyz of the fixed point (if the case)
        ''')


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_XYZ_OUTERDIAG(MSection):
    '''
    Section to define the cross term (XA-XA(0))*(XB-XB(0))+(XA-XA(0))*(YB-YB(0)) or part
    of its components as a collective variable. The final term is given by the product  of
    the components of A with the components of B.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_XYZ_OUTERDIAG_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the index of the atoms/points A and B.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_XYZ_OUTERDIAG_COMPONENT_A = Quantity(
        type=str,
        shape=[],
        description='''
        Define the component of the position vector which will be used as a colvar for
        atom A.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_XYZ_OUTERDIAG_COMPONENT_B = Quantity(
        type=str,
        shape=[],
        description='''
        Define the component of the position vector which will be used as a colvar for
        atom B.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_XYZ_OUTERDIAG_PBC = Quantity(
        type=str,
        shape=[],
        description='''
        Whether periodic boundary conditions should be applied on the atomic position
        before computing the colvar or not.
        ''')

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_XYZ_OUTERDIAG_POINT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_XYZ_OUTERDIAG_POINT'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR(MSection):
    '''
    This section specifies the nature of the collective variables.
    '''

    m_def = Section(validate=False)

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_ANGLE_PLANE_PLANE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_ANGLE_PLANE_PLANE'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_ANGLE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_ANGLE'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_BOND_ROTATION = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_BOND_ROTATION'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_COLVAR_FUNC_INFO = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_COLVAR_FUNC_INFO'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_CONDITIONED_DISTANCE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_CONDITIONED_DISTANCE'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_COORDINATION = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_COORDINATION'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_DISTANCE_FUNCTION = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_DISTANCE_FUNCTION'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_DISTANCE_POINT_PLANE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_DISTANCE_POINT_PLANE'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_DISTANCE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_DISTANCE'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_GYRATION_RADIUS = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_GYRATION_RADIUS'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_HBP = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_HBP'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_HYDRONIUM = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_HYDRONIUM'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_POPULATION = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_POPULATION'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_QPARM = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_QPARM'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_RING_PUCKERING = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_RING_PUCKERING'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_RMSD = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_RMSD'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_TORSION = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_TORSION'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_U = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_U'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_WC = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_WC'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_XYZ_DIAG = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_XYZ_DIAG'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_XYZ_OUTERDIAG = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR_XYZ_OUTERDIAG'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR(MSection):
    '''
    Allows the possibility to combine several COLVARs into one COLVAR with a generic
    function.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_DX = Quantity(
        type=str,
        shape=[],
        description='''
        Parameter used for computing the derivative of the combination of COLVARs with the
        Ridders method.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_ERROR_LIMIT = Quantity(
        type=str,
        shape=[],
        description='''
        Checks that the error in computing the derivative is not larger than the value
        set. In case prints a warning message.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_FUNCTION = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the function used to combine different COLVARs into one.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_PARAMETERS = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the parameters of the functional form
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_VALUES = Quantity(
        type=str,
        shape=[],
        description='''
        Defines the values of  parameter of the functional form
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_VARIABLES = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the name of the variable that parametrises the FUNCTION defining how
        COLVARS should be combined. The matching follows the same order of the COLVARS
        definition in the input file.
        ''')

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COMBINE_COLVAR_COLVAR'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_CONDITIONED_DISTANCE_POINT(MSection):
    '''
    Enables the possibility to use geometrical centers instead of single atoms to define
    colvars
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_CONDITIONED_DISTANCE_POINT_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the indexes of atoms defining the geometrical center
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_CONDITIONED_DISTANCE_POINT_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Chooses the type of geometrical point
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_CONDITIONED_DISTANCE_POINT_WEIGHTS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the weights for a weighted geometrical center. Default is 1/natoms for
        every atom
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_CONDITIONED_DISTANCE_POINT_XYZ = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the xyz of the fixed point (if the case)
        ''')


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_CONDITIONED_DISTANCE(MSection):
    '''
    Section to define the conditioned distance as a collective variable.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_CONDITIONED_DISTANCE_ATOMS_DISTANCE = Quantity(
        type=str,
        shape=[],
        description='''
        Specify indexes of atoms/points from which the distance is computed.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_CONDITIONED_DISTANCE_ATOMS_FROM = Quantity(
        type=str,
        shape=[],
        description='''
        Specify indexes of atoms/points building the coordination variable.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_CONDITIONED_DISTANCE_ATOMS_TO = Quantity(
        type=str,
        shape=[],
        description='''
        Specify indexes of atoms/points building the coordination variable.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_CONDITIONED_DISTANCE_KINDS_FROM = Quantity(
        type=str,
        shape=[],
        description='''
        Specify alternatively kinds of atoms building the coordination variable.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_CONDITIONED_DISTANCE_KINDS_TO = Quantity(
        type=str,
        shape=[],
        description='''
        Specify alternatively kinds of atoms building the coordination variable.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_CONDITIONED_DISTANCE_LAMBDA = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the lambda parameter at the exponent of the conditioned distance function.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_CONDITIONED_DISTANCE_ND = Quantity(
        type=str,
        shape=[],
        description='''
        Sets the value of the denominator of the exponential factorin the coordination
        FUNCTION.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_CONDITIONED_DISTANCE_NN = Quantity(
        type=str,
        shape=[],
        description='''
        Sets the value of the numerator of the exponential factorin the coordination
        FUNCTION.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_CONDITIONED_DISTANCE_R0 = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the R0 parameter in the coordination function.
        ''')

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_CONDITIONED_DISTANCE_POINT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_CONDITIONED_DISTANCE_POINT'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COORDINATION_POINT(MSection):
    '''
    Enables the possibility to use geometrical centers instead of single atoms to define
    colvars
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COORDINATION_POINT_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the indexes of atoms defining the geometrical center
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COORDINATION_POINT_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Chooses the type of geometrical point
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COORDINATION_POINT_WEIGHTS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the weights for a weighted geometrical center. Default is 1/natoms for
        every atom
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COORDINATION_POINT_XYZ = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the xyz of the fixed point (if the case)
        ''')


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COORDINATION(MSection):
    '''
    Section to define the coordination number as a collective variable.
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COORDINATION_ATOMS_FROM = Quantity(
        type=str,
        shape=[],
        description='''
        Specify indexes of atoms/points building the coordination variable.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COORDINATION_ATOMS_TO_B = Quantity(
        type=str,
        shape=[],
        description='''
        For the CV given by the multiplication of two coorination numbers, here specify
        indexes of the third set of atoms/points.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COORDINATION_ATOMS_TO = Quantity(
        type=str,
        shape=[],
        description='''
        Specify indexes of atoms/points building the coordination variable.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COORDINATION_KINDS_FROM = Quantity(
        type=str,
        shape=[],
        description='''
        Specify alternatively kinds of atoms building the coordination variable.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COORDINATION_KINDS_TO_B = Quantity(
        type=str,
        shape=[],
        description='''
        For the CV given by the multiplication of two coorination numbers, here specify
        alternatively kinds of atoms building the coordination variable.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COORDINATION_KINDS_TO = Quantity(
        type=str,
        shape=[],
        description='''
        Specify alternatively kinds of atoms building the coordination variable.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COORDINATION_ND_B = Quantity(
        type=str,
        shape=[],
        description='''
        For the CV given by the multiplication of two coorination numbers,Sets the value
        of the denominator of the exponential factorin the coordination FUNCTION.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COORDINATION_ND = Quantity(
        type=str,
        shape=[],
        description='''
        Sets the value of the denominator of the exponential factorin the coordination
        FUNCTION.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COORDINATION_NN_B = Quantity(
        type=str,
        shape=[],
        description='''
        For the CV given by the multiplication of two coorination numbers,Sets the value
        of the numerator of the exponential factorin the coordination FUNCTION.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COORDINATION_NN = Quantity(
        type=str,
        shape=[],
        description='''
        Sets the value of the numerator of the exponential factorin the coordination
        FUNCTION.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COORDINATION_R0_B = Quantity(
        type=str,
        shape=[],
        description='''
        For the CV given by the multiplication of two coorination numbers, specify the R0
        parameter in the second coordination function.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_COORDINATION_R0 = Quantity(
        type=str,
        shape=[],
        description='''
        Specify the R0 parameter in the coordination function.
        ''')

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COORDINATION_POINT = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_COORDINATION_POINT'),
        repeats=True)


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_DISTANCE_POINT(MSection):
    '''
    Enables the possibility to use geometrical centers instead of single atoms to define
    colvars
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_DISTANCE_POINT_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the indexes of atoms defining the geometrical center
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_DISTANCE_POINT_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Chooses the type of geometrical point
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_DISTANCE_POINT_WEIGHTS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the weights for a weighted geometrical center. Default is 1/natoms for
        every atom
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_DISTANCE_POINT_XYZ = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the xyz of the fixed point (if the case)
        ''')


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_DISTANCE_FROM_PATH_COLVAR_ANGLE_POINT(MSection):
    '''
    Enables the possibility to use geometrical centers instead of single atoms to define
    colvars
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_DISTANCE_FROM_PATH_COLVAR_ANGLE_POINT_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the indexes of atoms defining the geometrical center
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_DISTANCE_FROM_PATH_COLVAR_ANGLE_POINT_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Chooses the type of geometrical point
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_DISTANCE_FROM_PATH_COLVAR_ANGLE_POINT_WEIGHTS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the weights for a weighted geometrical center. Default is 1/natoms for
        every atom
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_DISTANCE_FROM_PATH_COLVAR_ANGLE_POINT_XYZ = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the xyz of the fixed point (if the case)
        ''')


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_DISTANCE_FROM_PATH_COLVAR_ANGLE_PLANE_PLANE_PLANE(MSection):
    '''
    This section defines the plane. When using this colvar, two plane section must be
    defined!
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_DISTANCE_FROM_PATH_COLVAR_ANGLE_PLANE_PLANE_PLANE_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the indexes of 3 atoms/points defining the plane.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_DISTANCE_FROM_PATH_COLVAR_ANGLE_PLANE_PLANE_PLANE_DEF_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Specify how the plane is defined: either by 3 atoms or by a fixed normal vector.
        At least one plane must be defined through atoms.
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_DISTANCE_FROM_PATH_COLVAR_ANGLE_PLANE_PLANE_PLANE_NORMAL_VECTOR = Quantity(
        type=str,
        shape=[],
        description='''
        Alternatively to 3 atoms/points one can define one of the two, planes by defining
        its NORMAL vector.
        ''')


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_DISTANCE_FROM_PATH_COLVAR_ANGLE_PLANE_PLANE_POINT(MSection):
    '''
    Enables the possibility to use geometrical centers instead of single atoms to define
    colvars
    '''

    m_def = Section(validate=False)

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_DISTANCE_FROM_PATH_COLVAR_ANGLE_PLANE_PLANE_POINT_ATOMS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the indexes of atoms defining the geometrical center
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_DISTANCE_FROM_PATH_COLVAR_ANGLE_PLANE_PLANE_POINT_TYPE = Quantity(
        type=str,
        shape=[],
        description='''
        Chooses the type of geometrical point
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_DISTANCE_FROM_PATH_COLVAR_ANGLE_PLANE_PLANE_POINT_WEIGHTS = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the weights for a weighted geometrical center. Default is 1/natoms for
        every atom
        ''')

    x_cp2k_input_FORCE_EVAL_SUBSYS_COLVAR_DISTANCE_FROM_PATH_COLVAR_ANGLE_PLANE_PLANE_POINT_XYZ = Quantity(
        type=str,
        shape=[],
        description='''
        Specifies the xyz of the fixed point (if the case)
        ''')


class x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_DISTANCE_FROM_PATH_COLVAR_ANGLE_PLANE_PLANE(MSection):
    '''
    This section defines the angle between two planes as a collective variables.
    '''

    m_def = Section(validate=False)

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_DISTANCE_FROM_PATH_COLVAR_ANGLE_PLANE_PLANE_PLANE = SubSection(
        sub_section=SectionProxy('x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_DISTANCE_FROM_PATH_COLVAR_ANGLE_PLANE_PLANE_PLANE'),
        repeats=True)

    x_cp2k_section_input_FORCE_EVAL_SUBSYS_COLVAR_DISTANCE_FROM_