#
# Copyright The NOMAD Authors.
#
# This file is part of NOMAD.
# See https://nomad-lab.eu for further info.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

import re
import numpy as np

from nomad.units import ureg
from nomad.parsing.file_parser import BasicParser


class FleurParser(BasicParser):
    def __init__(self):
        re_f = r'\-*\d+\.\d+E*\-*\+*\d*'

        def get_forces(val):
            forces = re.findall(rf'FX\_TOT\=\s*({re_f}) FY\_TOT\=\s*({re_f}) FZ\_TOT\=\s*({re_f})', val)
            return np.array(forces, dtype=np.dtype(np.float64))

        super().__init__(
            specifications=dict(
                name='parsers/fleur', code_name='fleur',
                code_homepage='https://www.flapw.de/', domain='dft',
                mainfile_contents_re=r'This output is generated by fleur.'),
            units_mapping=dict(length=ureg.bohr, energy=ureg.hartree),
            program_version=r'This output is generated by (fleur[\w\.]+)',
            lattice_vectors=r'bravais matrices of real and reciprocal lattices\s*([\s\S]+?)\n *\n',
            atom_labels_atom_positions_scaled=rf'([A-Z][a-z]* +\d+ +\d+ +\d+ +\d+ +{re_f} +{re_f}\s*{re_f} +{re_f} +{re_f} +{re_f}[\s\S]+?)\-\-\-\-\-',
            energy_reference_fermi=(rf'\n *Fermi\s*\:\s*({re_f})', lambda x: [x]),
            energy_total=rf'\-\-\-\-\> total energy\= *({re_f})',
            energy_free=rf'\-\-\-\-\> free energy\= *({re_f})',
            atom_forces=(rf'TOTAL FORCES ON ATOMS \*+\s+([\s\S]+?)\n *\n', get_forces))
