"""Displays crash information from Nipype crash files. For certain crash files,
one can rerun a failed node in a temp directory.

Examples:

nipype_display_crash crashfile.pklz
nipype_display_crash crashfile.pklz -r -i
nipype_display_crash crashfile.pklz -r -i

"""

def display_crash_files(crashfile, rerun, debug, directory):
    """display crash file content and rerun if required"""

    from nipype.utils.filemanip import loadcrash
    crash_data = loadcrash(crashfile)
    node = None
    if 'node' in crash_data:
        node = crash_data['node']
    tb = crash_data['traceback']
    print("\n")
    print("File: %s" % crashfile)
    if node:
        print("Node: %s" % node)
        if node.base_dir:
            print("Working directory: %s" % node.output_dir())
        else:
            print("Node crashed before execution")
        print("\n")
        print("Node inputs:")
        print(node.inputs)
        print("\n")
    print("Traceback: ")
    print(''.join(tb))
    print ("\n")

    if rerun:
        if node is None:
            print("No node in crashfile. Cannot rerun")
            return
        print("Rerunning node")
        node.base_dir = directory
        node.config = {'execution': {'crashdump_dir': '/tmp'}}
        try:
            node.run()
        except:
            if debug and debug != 'ipython':
                import pdb
                pdb.post_mortem()
            else:
                raise
        print("\n")

if __name__ == "__main__":
    from argparse import ArgumentParser, RawTextHelpFormatter
    defstr = ' (default %(default)s)'
    parser = ArgumentParser(prog='nipype_display_crash',
                            description=__doc__,
                            formatter_class=RawTextHelpFormatter)
    parser.add_argument('crashfile', metavar='f', type=str,
                        help='crash file to display')
    parser.add_argument('-r','--rerun', dest='rerun',
                        default=False, action="store_true",
                        help='rerun crashed node' + defstr)
    group = parser.add_mutually_exclusive_group()
    group.add_argument('-d','--debug', dest='debug',
                        default=False, action="store_true",
                        help='enable python debugger when re-executing' + defstr)
    group.add_argument('-i','--ipydebug', dest='ipydebug',
                        default=False, action="store_true",
                        help='enable ipython debugger when re-executing' + defstr)
    parser.add_argument('--dir', dest='directory',
                        default=None,
                        help='Directory to run the node in' + defstr)
    args = parser.parse_args()
    debug = 'ipython' if args.ipydebug else args.debug
    if debug == 'ipython':
        import sys
        from IPython.core import ultratb
        sys.excepthook = ultratb.FormattedTB(mode='Verbose',
                                             color_scheme='Linux',
                                             call_pdb=1)
    display_crash_files(args.crashfile, args.rerun,
                        debug, args.directory)
