"use strict";
// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
Object.defineProperty(exports, "__esModule", { value: true });
var apputils_1 = require("@jupyterlab/apputils");
var coreutils_1 = require("@jupyterlab/coreutils");
var rendermime_1 = require("@jupyterlab/rendermime");
var notebook_1 = require("@jupyterlab/notebook");
var algorithm_1 = require("@phosphor/algorithm");
var disposable_1 = require("@phosphor/disposable");
var actions_1 = require("./actions");
/**
 * Error message if the nbdime API is unavailable.
 */
var serverMissingMsg = 'Unable to query nbdime API. Is the server extension enabled?';
var INITIAL_NETWORK_RETRY = 2; // ms
var NBDiffExtension = /** @class */ (function () {
    /**
     *
     */
    function NBDiffExtension(commands) {
        this.commands = commands;
    }
    /**
     * Create a new extension object.
     */
    NBDiffExtension.prototype.createNew = function (nb, context) {
        // Create extension here
        // Add buttons to toolbar
        var buttons = [];
        var insertionPoint = -1;
        algorithm_1.find(nb.toolbar.children(), function (tbb, index) {
            if (tbb.hasClass('jp-Notebook-toolbarCellType')) {
                insertionPoint = index;
                return true;
            }
            return false;
        });
        var i = 1;
        for (var _i = 0, _a = [CommandIDs.diffNotebookCheckpoint, CommandIDs.diffNotebookGit]; _i < _a.length; _i++) {
            var id = _a[_i];
            var button = apputils_1.Toolbar.createFromCommand(this.commands, id);
            if (button === null) {
                throw new Error('Cannot create button, command not registered!');
            }
            if (insertionPoint >= 0) {
                nb.toolbar.insertItem(insertionPoint + i++, this.commands.label(id), button);
            }
            else {
                nb.toolbar.addItem(this.commands.label(id), button);
            }
            buttons.push(button);
        }
        return new disposable_1.DisposableDelegate(function () {
            // Cleanup extension here
            for (var _i = 0, buttons_1 = buttons; _i < buttons_1.length; _i++) {
                var btn = buttons_1[_i];
                btn.dispose();
            }
        });
    };
    return NBDiffExtension;
}());
exports.NBDiffExtension = NBDiffExtension;
var CommandIDs;
(function (CommandIDs) {
    CommandIDs.diffNotebook = 'nbdime:diff';
    CommandIDs.diffNotebookGit = 'nbdime:diff-git';
    CommandIDs.diffNotebookCheckpoint = 'nbdime:diff-checkpoint';
})(CommandIDs = exports.CommandIDs || (exports.CommandIDs = {}));
function addCommands(app, tracker, rendermime) {
    var commands = app.commands, shell = app.shell;
    // Whether we have our server extension available
    var hasAPI = true;
    /**
     * Whether there is an active notebook.
     */
    function hasWidget() {
        return tracker.currentWidget !== null;
    }
    /**
     * Whether there is an active notebook.
     */
    function baseEnabled() {
        return hasAPI && tracker.currentWidget !== null;
    }
    // This allows quicker checking, but if someone creates/removes
    // a repo during the session, this will become incorrect
    var lut_known_git = {};
    var networkRetry = INITIAL_NETWORK_RETRY;
    /**
     * Whether the notebook is in a git repository.
     */
    function hasGitNotebook() {
        if (!baseEnabled()) {
            return false;
        }
        var path = tracker.currentWidget.context.path;
        var dir = coreutils_1.PathExt.dirname(path);
        var known_git = lut_known_git[dir];
        if (known_git === undefined) {
            var inGitPromise = actions_1.isNbInGit({ path: dir });
            inGitPromise.then(function (inGit) {
                networkRetry = INITIAL_NETWORK_RETRY;
                lut_known_git[dir] = inGit;
                // Only update if false, since it is left enabled while waiting
                if (!inGit) {
                    commands.notifyCommandChanged(CommandIDs.diffNotebookGit);
                }
            });
            inGitPromise.catch(function (reason) {
                hasAPI = reason.status !== undefined && reason.status !== 404;
                setTimeout(function () {
                    networkRetry *= 2;
                    commands.notifyCommandChanged(CommandIDs.diffNotebook);
                    commands.notifyCommandChanged(CommandIDs.diffNotebookCheckpoint);
                    commands.notifyCommandChanged(CommandIDs.diffNotebookGit);
                }, networkRetry);
            });
            // Leave button enabled while unsure
            return true;
        }
        return known_git;
    }
    function erroredGen(text) {
        return function () {
            if (hasAPI) {
                return text;
            }
            return serverMissingMsg;
        };
    }
    commands.addCommand(CommandIDs.diffNotebook, {
        execute: function (args) {
            // TODO: Check args for base/remote
            // if missing, prompt with dialog.
            //let content = current.notebook;
            //diffNotebook({base, remote});
        },
        label: erroredGen('Notebook diff'),
        caption: erroredGen('Display nbdiff between two notebooks'),
        isEnabled: baseEnabled,
        icon: 'action-notebook-diff action-notebook-diff-notebooks',
        iconLabel: 'nbdiff',
    });
    commands.addCommand(CommandIDs.diffNotebookCheckpoint, {
        execute: function (args) {
            var current = tracker.currentWidget;
            if (!current) {
                return;
            }
            var widget = actions_1.diffNotebookCheckpoint({ path: current.context.path, rendermime: rendermime });
            shell.addToMainArea(widget);
            if (args['activate'] !== false) {
                shell.activateById(widget.id);
            }
        },
        label: erroredGen('Notebook checkpoint diff'),
        caption: erroredGen('Display nbdiff from checkpoint to currently saved version'),
        isEnabled: baseEnabled,
        iconClass: 'fa fa-clock-o action-notebook-diff action-notebook-diff-checkpoint',
    });
    commands.addCommand(CommandIDs.diffNotebookGit, {
        execute: function (args) {
            var current = tracker.currentWidget;
            if (!current) {
                return;
            }
            var widget = actions_1.diffNotebookGit({ path: current.context.path, rendermime: rendermime });
            shell.addToMainArea(widget);
            if (args['activate'] !== false) {
                shell.activateById(widget.id);
            }
        },
        label: erroredGen('Notebook Git diff'),
        caption: erroredGen('Display nbdiff from git HEAD to currently saved version'),
        isEnabled: hasGitNotebook,
        iconClass: 'fa fa-git action-notebook-diff action-notebook-diff-git',
    });
}
/**
 * The notebook diff provider.
 */
var nbDiffProvider = {
    id: 'jupyter.extensions.nbdime',
    requires: [notebook_1.INotebookTracker, rendermime_1.IRenderMimeRegistry],
    activate: activateWidgetExtension,
    autoStart: true
};
exports.default = nbDiffProvider;
/**
 * Activate the widget extension.
 */
function activateWidgetExtension(app, tracker, rendermime) {
    var commands = app.commands, docRegistry = app.docRegistry;
    var extension = new NBDiffExtension(commands);
    docRegistry.addWidgetExtension('Notebook', extension);
    addCommands(app, tracker, rendermime);
    // Update the command registry when the notebook state changes.
    tracker.currentChanged.connect(function () {
        commands.notifyCommandChanged(CommandIDs.diffNotebookGit);
        if (tracker.size <= 1) {
            commands.notifyCommandChanged(CommandIDs.diffNotebook);
            commands.notifyCommandChanged(CommandIDs.diffNotebookCheckpoint);
        }
    });
}
//# sourceMappingURL=plugin.js.map