# Copyright (c) 2008-2015 MetPy Developers.
# Distributed under the terms of the BSD 3-Clause License.
# SPDX-License-Identifier: BSD-3-Clause
r"""Collection of utilities for testing

    This includes:
    * unit-aware test functions
    * code for testing matplotlib figures
"""

import numpy.testing
from matplotlib import style
from pint import DimensionalityError
from .units import units

# Our lowest supported matplotlib doesn't have the classic style, so fallback to empty list
test_style = 'classic' if 'classic' in style.available else []


def check_and_drop_units(actual, desired):
    r"""Check that the units on the passed in arrays are compatible; return the magnitudes.

    Parameters
    ----------
    actual : `pint.Quantity` or array-like

    desired : `pint.Quantity` or array-like

    Returns
    -------
    actual, desired
        array-like versions of `actual` and `desired` once they have been
        coerced to compatible units.

    Raises
    ------
    AssertionError
        If the units on the passed in objects are not compatible.
    """
    try:
        # If the desired result has units, add dimensionless units if necessary, then
        # ensure that this is compatible to the desired result.
        if hasattr(desired, 'units'):
            if not hasattr(actual, 'units'):
                actual = units.Quantity(actual, 'dimensionless')
            actual = actual.to(desired.units)
        # Otherwise, the desired result has no units. Convert the actual result to
        # dimensionless units if it is a united quantity.
        else:
            if hasattr(actual, 'units'):
                actual = actual.to('dimensionless')
    except DimensionalityError:
        raise AssertionError('Units are not compatible: %s should be %s' %
                             (actual.units, desired.units))
    except AttributeError:
        pass

    if hasattr(actual, 'magnitude'):
        actual = actual.magnitude
    if hasattr(desired, 'magnitude'):
        desired = desired.magnitude

    return actual, desired


def assert_almost_equal(actual, desired, decimal=7):
    'numpy.testing.assert_almost_equal that first handles units'
    actual, desired = check_and_drop_units(actual, desired)
    numpy.testing.assert_almost_equal(actual, desired, decimal)


def assert_array_almost_equal(actual, desired, decimal=7):
    'numpy.testing.assert_array_almost_equal that first handles units'
    actual, desired = check_and_drop_units(actual, desired)
    numpy.testing.assert_array_almost_equal(actual, desired, decimal)


def assert_array_equal(actual, desired):
    'numpy.testing.assert_array_equal that first handles units'
    actual, desired = check_and_drop_units(actual, desired)
    numpy.testing.assert_array_equal(actual, desired)


def make_figure(*args, **kwargs):
    'Create an Agg figure for testing'
    from matplotlib.figure import Figure
    from matplotlib.backends.backend_agg import FigureCanvasAgg
    if 'dpi' not in kwargs:
        kwargs['dpi'] = 100
    fig = Figure(*args, **kwargs)
    fig.canvas = FigureCanvasAgg(fig)
    return fig


def hide_tick_labels(ax):
    'Hide the ticklabels on an axes'
    ax.xaxis.set_ticklabels([])
    ax.yaxis.set_ticklabels([])
