/*=========================================================================
*
*  Copyright Insight Software Consortium
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*         http://www.apache.org/licenses/LICENSE-2.0.txt
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
*=========================================================================*/
#ifndef sitkN4BiasFieldCorrectionImageFilter_h
#define sitkN4BiasFieldCorrectionImageFilter_h

/*
 * WARNING: DO NOT EDIT THIS FILE!
 * THIS FILE IS AUTOMATICALLY GENERATED BY THE SIMPLEITK BUILD PROCESS.
 * Please look at sitkImageFilterTemplate.h.in to make changes.
 */

#include <memory>

#include "sitkBasicFilters.h"
#include "sitkImageFilter.h"

namespace itk {
  namespace simple {

    /**\class N4BiasFieldCorrectionImageFilter
\brief Implementation of the N4 bias field correction algorithm.

The nonparametric nonuniform intensity normalization (N3) algorithm, as introduced by Sled et al. in 1998 is a method for correcting nonuniformity associated with MR images. The algorithm assumes a simple parametric model (Gaussian) for the bias field and does not require tissue class segmentation. In addition, there are only a couple of parameters to tune with the default values performing quite well. N3 has been publicly available as a set of perl scripts (http://www.bic.mni.mcgill.ca/ServicesSoftwareAdvancedImageProcessingTools/HomePage )

The N4 algorithm, encapsulated with this class, is a variation of the original N3 algorithm with the additional benefits of an improved B-spline fitting routine which allows for multiple resolutions to be used during the correction process. We also modify the iterative update component of algorithm such that the residual bias field is continually updated

Notes for the user:
\li Since much of the image manipulation is done in the log space of the intensities, input images with negative and small values (< 1) can produce poor results.

\li The original authors recommend performing the bias field correction on a downsampled version of the original image.

\li A binary mask or a weighted image can be supplied. If a binary mask is specified, those voxels in the input image which correspond to the voxels in the mask image are used to estimate the bias field. If a UseMaskLabel value is set to true, only voxels in the MaskImage that match the MaskLabel will be used; otherwise, all non-zero voxels in the MaskImage will be masked. If a confidence image is specified, the input voxels are weighted in the b-spline fitting routine according to the confidence voxel values.

\li The filter returns the corrected image. If the bias field is wanted, one can reconstruct it using the class itkBSplineControlPointImageFilter. See the IJ article and the test file for an example.

\li The 'Z' parameter in Sled's 1998 paper is the square root of the class variable 'm_WienerFilterNoise'.



The basic algorithm iterates between sharpening the intensity histogram of the corrected input image and spatially smoothing those results with a B-spline scalar field estimate of the bias field.

\author Nicholas J. Tustison

Contributed by Nicholas J. Tustison, James C. Gee in the Insight Journal paper: https://hdl.handle.net/10380/3053 

\par REFERENCE


J.G. Sled, A.P. Zijdenbos and A.C. Evans. "A Nonparametric Method for
Automatic Correction of Intensity Nonuniformity in Data" IEEE Transactions on Medical Imaging, Vol 17, No 1. Feb 1998.

N.J. Tustison, B.B. Avants, P.A. Cook, Y. Zheng, A. Egan, P.A. Yushkevich, and J.C. Gee. "N4ITK: Improved N3 Bias Correction" IEEE Transactions on Medical Imaging, 29(6):1310-1320, June 2010.
\sa itk::simple::N4BiasFieldCorrection for the procedural interface
\sa itk::N4BiasFieldCorrectionImageFilter for the Doxygen on the original ITK class.
     */
    class SITKBasicFilters_EXPORT N4BiasFieldCorrectionImageFilter : public ImageFilter<0> {
    public:
      typedef N4BiasFieldCorrectionImageFilter Self;

      /** Destructor */
      virtual ~N4BiasFieldCorrectionImageFilter();

      /** Default Constructor that takes no arguments and initializes
       * default parameters */
      N4BiasFieldCorrectionImageFilter();

      /** Define the pixels types supported by this filter */
      typedef RealPixelIDTypeList  PixelIDTypeList;



      /**
       * Set the convergence threshold. Convergence is determined by the coefficient of variation of the difference image between the current bias field estimate and the previous estimate. If this value is less than the specified threshold, the algorithm proceeds to the next fitting level or terminates if it is at the last level.
       */
      SITK_RETURN_SELF_TYPE_HEADER SetConvergenceThreshold ( double ConvergenceThreshold ) { this->m_ConvergenceThreshold = ConvergenceThreshold; return *this; }

      /**
       * Get the convergence threshold. Convergence is determined by the coefficient of variation of the difference image between the current bias field estimate and the previous estimate. If this value is less than the specified threshold, the algorithm proceeds to the next fitting level or terminates if it is at the last level.
       */
        double GetConvergenceThreshold() const { return this->m_ConvergenceThreshold; }

      /**
       * Set the maximum number of iterations specified at each fitting level. Default = 50.
       */
      SITK_RETURN_SELF_TYPE_HEADER SetMaximumNumberOfIterations ( std::vector<uint32_t> MaximumNumberOfIterations ) { this->m_MaximumNumberOfIterations = MaximumNumberOfIterations; return *this; }

      /**
       * Get the maximum number of iterations specified at each fitting level. Default = 50.
       */
        std::vector<uint32_t> GetMaximumNumberOfIterations() const { return this->m_MaximumNumberOfIterations; }

      /**
       * Set the full width at half maximum parameter characterizing the width of the Gaussian deconvolution. Default = 0.15.
       */
      SITK_RETURN_SELF_TYPE_HEADER SetBiasFieldFullWidthAtHalfMaximum ( double BiasFieldFullWidthAtHalfMaximum ) { this->m_BiasFieldFullWidthAtHalfMaximum = BiasFieldFullWidthAtHalfMaximum; return *this; }

      /**
       * Get the full width at half maximum parameter characterizing the width of the Gaussian deconvolution. Default = 0.15.
       */
        double GetBiasFieldFullWidthAtHalfMaximum() const { return this->m_BiasFieldFullWidthAtHalfMaximum; }

      /**
       * Set the noise estimate defining the Wiener filter. Default = 0.01.
       */
      SITK_RETURN_SELF_TYPE_HEADER SetWienerFilterNoise ( double WienerFilterNoise ) { this->m_WienerFilterNoise = WienerFilterNoise; return *this; }

      /**
       * Get the noise estimate defining the Wiener filter. Default = 0.01.
       */
        double GetWienerFilterNoise() const { return this->m_WienerFilterNoise; }

      /**
       * Set number of bins defining the log input intensity histogram. Default = 200.
       */
      SITK_RETURN_SELF_TYPE_HEADER SetNumberOfHistogramBins ( uint32_t NumberOfHistogramBins ) { this->m_NumberOfHistogramBins = NumberOfHistogramBins; return *this; }

      /**
       * Get number of bins defining the log input intensity histogram. Default = 200.
       */
        uint32_t GetNumberOfHistogramBins() const { return this->m_NumberOfHistogramBins; }

      /**
       * Set the control point grid size defining the B-spline estimate of the scalar bias field. In each dimension, the B-spline mesh size is equal to the number of control points in that dimension minus the spline order. Default = 4 control points in each dimension for a mesh size of 1 in each dimension.
       */
      SITK_RETURN_SELF_TYPE_HEADER SetNumberOfControlPoints ( const std::vector<uint32_t> & NumberOfControlPoints ) { this->m_NumberOfControlPoints = NumberOfControlPoints; return *this; }

      /** Set the values of the NumberOfControlPoints vector all to value */
      SITK_RETURN_SELF_TYPE_HEADER SetNumberOfControlPoints( uint32_t value ) { this->m_NumberOfControlPoints = std::vector<uint32_t>(3, value); return *this; }


      /**
       * Get the control point grid size defining the B-spline estimate of the scalar bias field. In each dimension, the B-spline mesh size is equal to the number of control points in that dimension minus the spline order. Default = 4 control points in each dimension for a mesh size of 1 in each dimension.
       */
        std::vector<uint32_t> GetNumberOfControlPoints() const { return this->m_NumberOfControlPoints; }

      /**
       * Set the spline order defining the bias field estimate. Default = 3.
       */
      SITK_RETURN_SELF_TYPE_HEADER SetSplineOrder ( uint32_t SplineOrder ) { this->m_SplineOrder = SplineOrder; return *this; }

      /**
       * Get the spline order defining the bias field estimate. Default = 3.
       */
        uint32_t GetSplineOrder() const { return this->m_SplineOrder; }

      /**
       * Use a mask label for identifying mask functionality. See SetMaskLabel. Defaults to true.
       */
      SITK_RETURN_SELF_TYPE_HEADER SetUseMaskLabel ( bool UseMaskLabel ) { this->m_UseMaskLabel = UseMaskLabel; return *this; }

      /** Set the value of UseMaskLabel to true or false respectfully. */
      SITK_RETURN_SELF_TYPE_HEADER UseMaskLabelOn() { return this->SetUseMaskLabel(true); }
      SITK_RETURN_SELF_TYPE_HEADER UseMaskLabelOff() { return this->SetUseMaskLabel(false); }

      /**
       * Use a mask label for identifying mask functionality. See SetMaskLabel. Defaults to true.
       */
        bool GetUseMaskLabel() const { return this->m_UseMaskLabel; }

      /**
       * DeprecatedSet/Get mask label value. If a binary mask image is specified and if UseMaskValue is true, only those input image voxels corresponding with mask image values equal to MaskLabel are used in estimating the bias field. If a MaskImage is specified and UseMaskLabel is false, all input image voxels corresponding to non-zero voxels in the MaskImage are used in estimating the bias field. Default = 1.
       */
      SITK_RETURN_SELF_TYPE_HEADER SetMaskLabel ( uint8_t MaskLabel ) { this->m_MaskLabel = MaskLabel; return *this; }

      /**
       * DeprecatedSet/Get mask label value. If a binary mask image is specified and if UseMaskValue is true, only those input image voxels corresponding with mask image values equal to MaskLabel are used in estimating the bias field. If a MaskImage is specified and UseMaskLabel is false, all input image voxels corresponding to non-zero voxels in the MaskImage are used in estimating the bias field. Default = 1.
       */
        uint8_t GetMaskLabel() const { return this->m_MaskLabel; }
      /** Name of this class */
      std::string GetName() const { return std::string ("N4BiasFieldCorrectionImageFilter"); }

      /** Print ourselves out */
      std::string ToString() const;


      /** Execute the filter on the input image */
      Image Execute ( const Image & image, const Image & maskImage );
      Image Execute ( const Image & image );


      /** Execute the filter on the input image with the given parameters */
      Image Execute ( const Image & image, const Image & maskImage, double convergenceThreshold, std::vector<uint32_t> maximumNumberOfIterations, double biasFieldFullWidthAtHalfMaximum, double wienerFilterNoise, uint32_t numberOfHistogramBins, const std::vector<uint32_t> & numberOfControlPoints, uint32_t splineOrder, bool useMaskLabel, uint8_t maskLabel );
      Image Execute ( const Image & image, double convergenceThreshold, std::vector<uint32_t> maximumNumberOfIterations, double biasFieldFullWidthAtHalfMaximum, double wienerFilterNoise, uint32_t numberOfHistogramBins, const std::vector<uint32_t> & numberOfControlPoints, uint32_t splineOrder, bool useMaskLabel, uint8_t maskLabel );

    private:

      /** Setup for member function dispatching */

      typedef Image (Self::*MemberFunctionType)( const Image * image, const Image * maskImage );
      template <class TImageType> Image ExecuteInternal ( const Image * image, const Image * maskImage );


      friend struct detail::MemberFunctionAddressor<MemberFunctionType>;

      nsstd::auto_ptr<detail::MemberFunctionFactory<MemberFunctionType> > m_MemberFactory;


      /*  */
      double  m_ConvergenceThreshold;
      /* 	odo this variable needs to be printed */
      std::vector<uint32_t>  m_MaximumNumberOfIterations;
      /*  */
      double  m_BiasFieldFullWidthAtHalfMaximum;
      /*  */
      double  m_WienerFilterNoise;
      /*  */
      uint32_t  m_NumberOfHistogramBins;
      /*  */
      std::vector<uint32_t>  m_NumberOfControlPoints;
      /*  */
      uint32_t  m_SplineOrder;
      bool  m_UseMaskLabel;
      uint8_t  m_MaskLabel;
    };


    /**
     * \brief Implementation of the N4 bias field correction algorithm.
     *
     * This function directly calls the execute method of N4BiasFieldCorrectionImageFilter
     * in order to support a procedural API
     *
     * \sa itk::simple::N4BiasFieldCorrectionImageFilter for the object oriented interface
     */
     SITKBasicFilters_EXPORT Image N4BiasFieldCorrection ( const Image & image, const Image & maskImage, double convergenceThreshold = 0.001, std::vector<uint32_t> maximumNumberOfIterations = std::vector<uint32_t>(4,50), double biasFieldFullWidthAtHalfMaximum = 0.15, double wienerFilterNoise = 0.01, uint32_t numberOfHistogramBins = 200u, const std::vector<uint32_t> & numberOfControlPoints = std::vector<uint32_t>(3, 4), uint32_t splineOrder = 3u, bool useMaskLabel = true, uint8_t maskLabel = 1 );
     SITKBasicFilters_EXPORT Image N4BiasFieldCorrection ( const Image & image, double convergenceThreshold = 0.001, std::vector<uint32_t> maximumNumberOfIterations = std::vector<uint32_t>(4,50), double biasFieldFullWidthAtHalfMaximum = 0.15, double wienerFilterNoise = 0.01, uint32_t numberOfHistogramBins = 200u, const std::vector<uint32_t> & numberOfControlPoints = std::vector<uint32_t>(3, 4), uint32_t splineOrder = 3u, bool useMaskLabel = true, uint8_t maskLabel = 1 );
  }
}
#endif
