/*=========================================================================
 *
 *  Copyright Insight Software Consortium
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *=========================================================================*/
#ifndef itkExpandWithZerosImageFilter_hxx
#define itkExpandWithZerosImageFilter_hxx

#include "itkExpandWithZerosImageFilter.h"
#include "itkImageScanlineIterator.h"
#include "itkImageRegionConstIteratorWithIndex.h"
#include "itkObjectFactory.h"
#include "itkNumericTraits.h"
#include "itkProgressReporter.h"

namespace itk
{
/**
 * Default constructor
 */
template< typename TInputImage, typename TOutputImage >
ExpandWithZerosImageFilter< TInputImage, TOutputImage >
::ExpandWithZerosImageFilter()
{
  // Set default factors to 1
  for ( unsigned int j = 0; j < ImageDimension; j++ )
    {
    m_ExpandFactors[j] = 1;
    }
}

/**
 * Standard "PrintSelf" method
 */
template< typename TInputImage, typename TOutputImage >
void
ExpandWithZerosImageFilter< TInputImage, TOutputImage >
::PrintSelf(std::ostream & os, Indent indent) const
{
  Superclass::PrintSelf(os, indent);

  unsigned int j;
  os << indent << "ExpandFactors: [";
  for ( j = 0; j < ImageDimension - 1; j++ )
    {
    os << m_ExpandFactors[j] << ", ";
    }
  os << m_ExpandFactors[j] << "]" << std::endl;
}

/**
 * Set expand factors from a single unsigned int
 */
template< typename TInputImage, typename TOutputImage >
void
ExpandWithZerosImageFilter< TInputImage, TOutputImage >
::SetExpandFactors(
  const unsigned int factor)
{
  unsigned int j;

  for ( j = 0; j < ImageDimension; j++ )
    {
    if ( factor != m_ExpandFactors[j] )
      {
      break;
      }
    }
  if ( j < ImageDimension )
    {
    this->Modified();
    for ( j = 0; j < ImageDimension; j++ )
      {
      m_ExpandFactors[j] = factor;
      if ( m_ExpandFactors[j] < 1 )
        {
        m_ExpandFactors[j] = 1;
        }
      }
    }
}

/**
 * BeforeThreadedGenerateData
 */
template< typename TInputImage, typename TOutputImage >
void
ExpandWithZerosImageFilter< TInputImage, TOutputImage >
::BeforeThreadedGenerateData()
{
}

/**
 * ThreadedGenerateData
 */
template< typename TInputImage, typename TOutputImage >
void
ExpandWithZerosImageFilter< TInputImage, TOutputImage >
::ThreadedGenerateData(const OutputImageRegionType & outputRegionForThread,
  ThreadIdType threadId)
{
  // Get the input and output pointers
  OutputImagePointer outputPtr    = this->GetOutput();
  const InputImageType * inputPtr = this->GetInput();

  // Iterator for walking the output
  typedef ImageScanlineIterator< TOutputImage > OutputIterator;

  OutputIterator outIt(outputPtr, outputRegionForThread);

  // Report progress on a per scanline basis
  const SizeValueType size0 = outputRegionForThread.GetSize(0);
  if ( size0 == 0 )
    {
    return;
    }
  const size_t numberOfLinesToProcess = outputRegionForThread.GetNumberOfPixels() / size0;
  ProgressReporter progress( this, threadId, static_cast< SizeValueType >( numberOfLinesToProcess ) );

  const typename OutputImageType::IndexType outputOriginIndex = outputPtr->GetLargestPossibleRegion().GetIndex();
  // Walk the output region, and interpolate the input image
  while ( !outIt.IsAtEnd() )
    {
    while ( !outIt.IsAtEndOfLine() )
      {
      const typename OutputImageType::IndexType outputIndex = outIt.GetIndex();
      bool indexIsMultipleOfFactor(true);
      for ( unsigned int j = 0; j < ImageDimension; j++ )
        {
        // Check that index (normalized to start with zero) is multiple of ExpandFactors
        if ( (outputIndex[j] - outputOriginIndex[j]) % m_ExpandFactors[j] != 0 )
          {
          indexIsMultipleOfFactor = false;
          break;
          }
        }
      // Copy value from input if index is multiplo, or set it to zero otherwise.
      if ( indexIsMultipleOfFactor )
        {
        // Determine the input pixel location associated with this output
        // pixel at the start of the scanline.
        //
        // Don't need to check for division by zero because the factors are
        // clamped to be minimum for 1.
        typename InputImageType::IndexType inputIndex;
        for ( unsigned int j = 0; j < ImageDimension; j++ )
          {
          inputIndex[j] = outputIndex[j] / m_ExpandFactors[j];
          }
        outIt.Set( static_cast< OutputPixelType >(
            inputPtr->GetPixel(inputIndex) ) );
        }
      else
        {
        outIt.Set( NumericTraits< OutputPixelType >::ZeroValue());
        }
      ++outIt;
      }

    outIt.NextLine();
    progress.CompletedPixel();
    }
}

/**
 * GenerateInputRequesteRegion
 */
template< typename TInputImage, typename TOutputImage >
void
ExpandWithZerosImageFilter< TInputImage, TOutputImage >
::GenerateInputRequestedRegion()
{
  // Call the superclass' implementation of this method
  Superclass::GenerateInputRequestedRegion();

  // Get pointers to the input and output
  InputImagePointer inputPtr =
    const_cast< TInputImage * >( this->GetInput() );
  OutputImagePointer outputPtr = this->GetOutput();

  if ( !inputPtr || !outputPtr )
    {
    return;
    }

  // We need to compute the input requested region (size and start index)
  unsigned int i;
  const typename TOutputImage::SizeType & outputRequestedRegionSize =
    outputPtr->GetRequestedRegion().GetSize();
  const typename TOutputImage::IndexType & outputRequestedRegionStartIndex =
    outputPtr->GetRequestedRegion().GetIndex();

  typename TInputImage::SizeType inputRequestedRegionSize;
  typename TInputImage::IndexType inputRequestedRegionStartIndex;
  /**
   * inputRequestedSize = (outputRequestedSize / ExpandFactor) + 1)
   * The extra 1 above is to take care of edge effects when streaming.
   */
  for ( i = 0; i < TInputImage::ImageDimension; i++ )
    {
    inputRequestedRegionSize[i] =
      static_cast<SizeValueType>(
          std::ceil( static_cast< double >(outputRequestedRegionSize[i])
            / static_cast< double >(m_ExpandFactors[i]) ) + 1 );

    inputRequestedRegionStartIndex[i] =
      static_cast<SizeValueType>(
          std::floor( static_cast< double >(outputRequestedRegionStartIndex[i])
            / static_cast< double >(m_ExpandFactors[i]) ) );
    }

  typename TInputImage::RegionType inputRequestedRegion;
  inputRequestedRegion.SetSize(inputRequestedRegionSize);
  inputRequestedRegion.SetIndex(inputRequestedRegionStartIndex);

  // Make sure the requested region is within largest possible.
  inputRequestedRegion.Crop( inputPtr->GetLargestPossibleRegion() );

  // Set the input requested region.
  inputPtr->SetRequestedRegion(inputRequestedRegion);
}

/**
 * GenerateOutputInformation
 */
template< typename TInputImage, typename TOutputImage >
void
ExpandWithZerosImageFilter< TInputImage, TOutputImage >
::GenerateOutputInformation()
{
  // Call the superclass' implementation of this method
  Superclass::GenerateOutputInformation();

  // Get pointers to the input and output
  InputImagePointer inputPtr =
    const_cast< TInputImage * >( this->GetInput() );
  OutputImagePointer outputPtr = this->GetOutput();

  if ( !inputPtr || !outputPtr )
    {
    return;
    }

  // We need to compute the output spacing, the output image size, and the
  // output image start index
  const typename TInputImage::SpacingType &
  inputSpacing = inputPtr->GetSpacing();
  const typename TInputImage::SizeType & inputSize =
    inputPtr->GetLargestPossibleRegion().GetSize();
  const typename TInputImage::IndexType & inputStartIndex =
    inputPtr->GetLargestPossibleRegion().GetIndex();
  const typename TInputImage::PointType &
  inputOrigin = inputPtr->GetOrigin();

  typename TOutputImage::SpacingType outputSpacing;
  typename TOutputImage::SizeType outputSize;
  typename TOutputImage::IndexType outputStartIndex;
  typename TOutputImage::PointType outputOrigin;

  typename TInputImage::SpacingType inputOriginShift;
  for ( unsigned int i = 0; i < TOutputImage::ImageDimension; i++ )
    {
    outputSpacing[i]    = inputSpacing[i] / (float)m_ExpandFactors[i];
    outputSize[i]       = inputSize[i] * (SizeValueType)m_ExpandFactors[i];
    outputStartIndex[i] = inputStartIndex[i] * (IndexValueType)m_ExpandFactors[i];
    const double fraction = static_cast< double >(m_ExpandFactors[i] - 1)
      / static_cast< double >(m_ExpandFactors[i]);
    inputOriginShift[i] = -( inputSpacing[i] / 2.0 ) * fraction;
    }

  const typename TInputImage::DirectionType inputDirection   = inputPtr->GetDirection();
  const typename TOutputImage::SpacingType outputOriginShift = inputDirection * inputOriginShift;

  outputOrigin = inputOrigin + outputOriginShift;

  outputPtr->SetSpacing(outputSpacing);
  outputPtr->SetOrigin(outputOrigin);

  typename TOutputImage::RegionType outputLargestPossibleRegion;
  outputLargestPossibleRegion.SetSize(outputSize);
  outputLargestPossibleRegion.SetIndex(outputStartIndex);

  outputPtr->SetLargestPossibleRegion(outputLargestPossibleRegion);
}
} // end namespace itk

#endif
