import argparse
import sys

from libcomcat.search import search, count
from libcomcat.utils import get_detail_data_frame, get_summary_data_frame, maketime
from impactutils.time.ancient_time import HistoricTime


def get_parser():
    desc = '''Download basic earthquake information in line format (csv, tab, etc.).

    To download basic event information (time,lat,lon,depth,magnitude) and
    moment tensor components for a box around New Zealand during 2013:

    %(prog)s nz.csv -b 163.213 -178.945 -48.980 -32.324 -s 2013-01-01 -e 2014-01-01 -f csv

    To expand the results to include preferred moment tensors:

    %(prog)s nz.xlsx -o preferred -b 163.213 -178.945 -48.980 -32.324 -s 2013-01-01 -e 2014-01-01 -f excel

    To expand the results to include ALL moment tensors:

    %(prog)s nz.xlsx -o all -b 163.213 -178.945 -48.980 -32.324 -s 2013-01-01 -e 2014-01-01 -f excel

    To expand the results to include preferred focal mechanisms:

    %(prog)s nz.xlsx -a preferred -b 163.213 -178.945 -48.980 -32.324 -s 2013-01-01 -e 2014-01-01 -f excel

    To expand the results to include ALL focal mechanisms:

    %(prog)s nz.xlsx -a all -b 163.213 -178.945 -48.980 -32.324 -s 2013-01-01 -e 2014-01-01 -f excel

    To include all magnitudes (including source and type) for that same search, add the -g flag:

    %(prog)s nz.csv -o -b 163.213 -178.945 -48.980 -32.324 -s 2013-01-01 -e 2014-01-01 -g -f csv

    To print the number of events that would be returned from the above query,
    and the maximum number of events supported by ONE ComCat query*:

    %(prog)s tmp.csv -x -o -b 163.213 -178.945 -48.980 -32.324 -s 2013-01-01 -e 2014-01-01

    To download events with fractional days, use the ISO 8601 combined date
    time format (YYYY-mm-ddTHH:MM:SS, YYYY-mm-ddTHH:MM:SS.s): %(prog)s -s
    2015-01-01T00:00:00 -e 2015-01-01T01:15:00

    NOTES: 

    Any start or end time where only date is specified (YYYY-mm-dd) will
    be translated to the beginning of that day.  Thus, a start time of
    "2015-01-01" becomes "2015-01-01T:00:00:00" and an end time of "2015-01-02"
    becomes ""2015-01-02T:00:00:00".
    
    Events which do not have a value for a given field (moment tensor
    components, for example), will have the string "nan" instead.

    Note that when specifying a search box that crosses the -180/180 meridian,
    you simply specify longitudes as you would if you were not crossing that
    meridian (i.e., lonmin=179, lonmax=-179).  The program will resolve the
    discrepancy.

    When specifying the -a or -o flags, whether "preferred" or "all", 
    a new column will be added for each different source and algorithm
    used to create the moment tensor or focal mechanism.  For example, 
    columns containing NEIC W-phase moment tensor solution information should be 
    prepended by "us_Mww" (network_method) to distinguish them from, say "nc_mwr"
    (Northern California regional moment tensor).  If "all" is chosen, then there
    will be many such columns of information, each prepended with the network 
    and method, WHERE SUCH METHOD IS AVAILABLE IN THE DATABASE.
    
    The ComCat API has a returned event limit of 20,000.  
    Queries that exceed this ComCat limit ARE supported by this
    software, by breaking up one large request into a number of
    smaller ones.  However, large queries, when also configured to
    retrieve moment tensor parameters, nodal plane angles, or moment
    tensor type can take a very long time to download. This delay is 
    caused by the fact that when this program has to retrieve moment 
    tensor parameters, nodal plane angles, or moment tensor type, it 
    must open a URL for EACH event and parse the data it finds.  If 
    these parameters are not requested, then the same request will 
    return in much less time (~10 minutes or less for a 20,000 event query).
    Queries for all magnitude solutions will take even more time, as this
    requires parsing an XML file for each event and extracting the magnitude
    values and associated source and type.
    '''
    parser = argparse.ArgumentParser(
        description=desc, formatter_class=argparse.RawDescriptionHelpFormatter)
    # positional arguments
    parser.add_argument('filename',
                        metavar='FILENAME', help='Output filename.')
    # optional arguments
    parser.add_argument('-b', '--bounds', metavar=('lonmin', 'lonmax', 'latmin', 'latmax'),
                        dest='bounds', type=float, nargs=4,
                        help='Bounds to constrain event search [lonmin lonmax latmin latmax]')
    parser.add_argument('-r', '--radius', dest='radius', metavar=('lat', 'lon', 'rmax'), type=float,
                        nargs=3, help='Search radius in KM (use instead of bounding box)')
    parser.add_argument('-s', '--start-time', dest='startTime', type=maketime,
                        help='Start time for search (defaults to ~30 days ago).  YYYY-mm-dd, YYYY-mm-ddTHH:MM:SS, or YYYY-mm-ddTHH:MM:SS.s')
    parser.add_argument('-e', '--end-time', dest='endTime', type=maketime,
                        help='End time for search (defaults to current date/time).  YYYY-mm-dd, YYYY-mm-ddTHH:MM:SS, or YYYY-mm-ddTHH:MM:SS.s')
    parser.add_argument('-t', '--time-after', dest='after', type=maketime,
                        help='Limit to events after specified time. YYYY-mm-dd or YYYY-mm-ddTHH:MM:SS')
    parser.add_argument('-m', '--mag-range', metavar=('minmag', 'maxmag'), dest='magRange', type=float, nargs=2,
                        help='Min/max (authoritative) magnitude to restrict search.')
    parser.add_argument('-c', '--catalog', dest='catalog',
                        help='Source catalog from which products derive (atlas, centennial, etc.)')
    parser.add_argument('-n', '--contributor', dest='contributor',
                        help='Source contributor (who loaded product) (us, nc, etc.)')

    parser.add_argument('-o', '--get-moment-components', dest='getComponents', choices=['none', 'preferred', 'all'],
                        default='none',
                        help='Extract preferred or all moment-tensor components (including type and derived hypocenter) where available.')
    parser.add_argument('-a', '--get-focal-angles', dest='getAngles', choices=['none', 'preferred', 'all'],
                        default='none',
                        help='Extract preferred or all focal-mechanism angles (strike,dip,rake) where available.')

    supphelp = """Extract moment tensor supplemental information (duration, derived origin, 
percent double couple) when available."""
    parser.add_argument('--get-moment-supplement', dest='getMomentSupplement', action='store_true',
                        help=supphelp)

    parser.add_argument('-g', '--get-all-magnitudes', dest='getAllMags', action='store_true',
                        help='Extract all magnitudes (with sources),authoritative listed first.')
    parser.add_argument('-f', '--format', dest='format', choices=['csv', 'tab', 'excel'], default='csv',
                        metavar='FORMAT', help='Output format.')

    prodhelp = '''Limit the search to only those events containing products of type PRODUCT. 
                  See the full list here: https://usgs.github.io/pdl/userguide/products/index.html
'''
    parser.add_argument('-p', '--product-type', dest='limitByProductType', metavar='PRODUCT',
                        help=prodhelp)
    parser.add_argument('-x', '--count', dest='getCount', action='store_true',
                        help='Just return the number of events in search and maximum allowed.')
    parser.add_argument('-v', '--verbose', dest='verbose', action='store_true',
                        help='Print progress')
    parser.add_argument('-d', '--debug', dest='debug', action='store_true',
                        help='Check the USGS development server (only valid inside USGS network).')
    return parser


def main(args):
    latitude = None
    longitude = None
    radiuskm = None
    lonmin = latmin = lonmax = latmax = None
    if args.radius:
        latitude = args.radius[0]
        longitude = args.radius[1]
        radiuskm = args.radius[2]

    if args.bounds:
        lonmin, lonmax, latmin, latmax = args.bounds
        # fix longitude bounds when crossing dateline
        if lonmin > lonmax and lonmax >= -180:
            lonmin -= 360
    else:
        lonmin, lonmax, latmin, latmax = None, None, None, None

    minmag = 0.0
    maxmag = 9.9
    if args.magRange:
        minmag = args.magRange[0]
        maxmag = args.magRange[1]

    if args.getCount:
        nevents = count(starttime=args.startTime,
                        endtime=args.endTime,
                        updatedafter=args.after,
                        minlatitude=latmin,
                        maxlatitude=latmax,
                        minlongitude=lonmin,
                        maxlongitude=lonmax,
                        latitude=latitude,
                        longitude=longitude,
                        maxradiuskm=radiuskm,
                        catalog=args.catalog,
                        contributor=args.contributor,
                        maxmagnitude=maxmag,
                        minmagnitude=minmag,
                        producttype=args.limitByProductType,
                        verbose=args.verbose)
        print('There are %i events matching input criteria.' % nevents)
        sys.exit(0)

    if args.bounds and args.radius:
        print('Please specify either a bounding box OR radius search.')
        sys.exit(1)

    events = search(starttime=args.startTime,
                    endtime=args.endTime,
                    updatedafter=args.after,
                    minlatitude=latmin,
                    maxlatitude=latmax,
                    minlongitude=lonmin,
                    maxlongitude=lonmax,
                    latitude=latitude,
                    longitude=longitude,
                    maxradiuskm=radiuskm,
                    catalog=args.catalog,
                    contributor=args.contributor,
                    maxmagnitude=maxmag,
                    minmagnitude=minmag,
                    producttype=args.limitByProductType,
                    verbose=args.verbose)

    if not len(events):
        print('No events found matching your search criteria. Exiting.')
        sys.exit(0)

    if args.getAngles != 'none' or args.getAllMags or args.getComponents != 'none':
        if args.verbose:
            sys.stderr.write(
                'Fetched %i events...creating table.\n' % (len(events)))

        df = get_detail_data_frame(events, get_all_magnitudes=args.getAllMags,
                                   get_tensors=args.getComponents,
                                   get_focals=args.getAngles,
                                   get_moment_supplement=args.getMomentSupplement,
                                   verbose=args.verbose)
    else:
        if args.verbose:
            sys.stderr.write(
                'Fetched %i events...creating summary table.\n' % (len(events)))
        df = get_summary_data_frame(events)

    # order the columns so that at least the initial parameters come the way we want them...
    first_columns = list(events[0].toDict().keys())
    col_list = list(df.columns)
    for column in first_columns:
        col_list.remove(column)
    df = df[first_columns + col_list]

    if args.verbose:
        sys.stderr.write('Created table...saving %i records to %s.\n' %
                         (len(df), args.filename))
    if args.format == 'csv':
        df.to_csv(args.filename, index=False, chunksize=1000)
    elif args.format == 'tab':
        df.to_csv(args.filename, sep='\t', index=False)
    else:
        df.to_excel(args.filename, index=False)
    print('%i records saved to %s.' % (len(df), args.filename))
    sys.exit(0)


if __name__ == '__main__':
    parser = get_parser()
    pargs = parser.parse_args()

    main(pargs)
