def example(extract='m', estimate=False):
    import larch.numba as lx
    from larch import P, X

    hh, pp, tour, skims, emp = lx.example(200, ['hh', 'pp', 'tour', 'skims', 'emp'])

    logsums_file = lx.example(202, output_file='/tmp/logsums.zarr.zip')
    logsums = lx.DataArray.from_zarr('/tmp/logsums.zarr.zip')

    ca = lx.Dataset(
        {'logsum': logsums},
        caseid='TOURID',
        alts=skims.TAZ_ID,
    )
    ca

    emp.info()

    tree = lx.DataTree(
        base=ca,
        tour=tour.rename_axis(index='TOUR_ID'),
        hh=hh.set_index("HHID"),
        person=pp.set_index('PERSONID'),
        emp=emp,
        skims=lx.Dataset.from_omx(skims),
        relationships=(
            "base.TAZ_ID @ emp.TAZ",
            "base.TOURID -> tour.TOUR_ID",
            "tour.HHID @ hh.HHID",
            "tour.PERSONID @ person.PERSONID",
            "hh.HOMETAZ @ skims.otaz",
            "base.TAZ_ID @ skims.dtaz"
        ),
    )

    m = lx.Model(datatree=tree)
    m.title = "Exampville Work Tour Destination Choice v1"

    m.quantity_ca = (
            + P.EmpRetail_HighInc * X('RETAIL_EMP * (INCOME>50000)')
            + P.EmpNonRetail_HighInc * X('NONRETAIL_EMP') * X("INCOME>50000")
            + P.EmpRetail_LowInc * X('RETAIL_EMP') * X("INCOME<=50000")
            + P.EmpNonRetail_LowInc * X('NONRETAIL_EMP') * X("INCOME<=50000")
    )

    m.quantity_scale = P.Theta


    m.utility_ca = (
        + P.logsum * X.logsum
        + P.distance * X.AUTO_DIST
    )

    m.choice_co_code = "tour.DTAZ"

    m.lock_values(
        EmpRetail_HighInc=0,
        EmpRetail_LowInc=0,
    )

    m.loglike()
    if not estimate:
        if isinstance(extract, str):
            return locals()[extract]
        else:
            _locals = locals()
            return [_locals.get(i) for i in extract]
    m.maximize_loglike()

    m.calculate_parameter_covariance()

    m.distribution_on_idca_variable('AUTO_DIST')

    m.distribution_on_idca_variable('AUTO_DIST', bins=40, range=(0,10))

    m.distribution_on_idca_variable(
        'AUTO_DIST',
        style='kde',
    )

    m.distribution_on_idca_variable(
        'AUTO_DIST',
        xlabel="Distance (miles)",
        bins=26,
        subselector='INCOME<10000',
        range=(0,13),
        header='Destination Distance, Very Low Income (<$10k) Households',
    )

    report = lx.Reporter(title=m.title)

    report << '# Parameter Summary' << m.parameter_summary()

    report << "# Estimation Statistics" << m.estimation_statistics()

    report << "# Utility Functions" << m.utility_functions()

    figure = m.distribution_on_idca_variable(
        'AUTO_DIST', 
        xlabel="Distance (miles)",
        style='kde',
        header='Destination Distance',
    )
    report << "# Visualization"
    report << figure

    report.save(
        '/tmp/exampville_dest_choice.html',
        overwrite=True,
        metadata=m,
    )
    if isinstance(extract, str):
        return locals()[extract]
    else:
        _locals = locals()
        return [_locals.get(i) for i in extract]
