"""
Core API functions and initialization routines.
"""

import configparser
import os
import sys
import logging

from . import backend
from .util import platform_ as platform
from .backends import fail


log = logging.getLogger(__name__)

_keyring_backend = None


def set_keyring(keyring):
    """Set current keyring backend."""
    global _keyring_backend
    if not isinstance(keyring, backend.KeyringBackend):
        raise TypeError("The keyring must be a subclass of KeyringBackend")
    _keyring_backend = keyring
    keyring.set_properties_from_env()


def get_keyring():
    """Get current keyring backend."""
    if _keyring_backend is None:
        init_backend()
    return _keyring_backend


def disable():
    """
    Configure the null keyring as the default.
    """
    root = platform.config_root()
    try:
        os.makedirs(root)
    except OSError:
        pass
    filename = os.path.join(root, 'keyringrc.cfg')
    if os.path.exists(filename):
        msg = "Refusing to overwrite {filename}".format(**locals())
        raise RuntimeError(msg)
    with open(filename, 'w') as file:
        file.write('[backend]\ndefault-keyring=keyring.backends.null.Keyring')


def get_password(service_name, username):
    """Get password from the specified service."""
    return get_keyring().get_password(service_name, username)


def set_password(service_name, username, password):
    """Set password for the user in the specified service."""
    get_keyring().set_password(service_name, username, password)


def delete_password(service_name, username):
    """Delete the password for the user in the specified service."""
    get_keyring().delete_password(service_name, username)


def get_credential(service_name, username):
    """Get a Credential for the specified service."""
    return get_keyring().get_credential(service_name, username)


def recommended(backend):
    return backend.priority >= 1


def init_backend(limit=None):
    """
    Load a detected backend.
    """
    set_keyring(_detect_backend(limit))


def _detect_backend(limit=None):
    """
    Return a keyring specified in the config file or infer the best available.

    Limit, if supplied, should be a callable taking a backend and returning
    True if that backend should be included for consideration.
    """

    # save the limit for the chainer to honor
    backend._limit = limit
    return (
        load_env()
        or load_config()
        or max(
            # all keyrings passing the limit filter
            filter(limit, backend.get_all_keyring()),
            default=fail.Keyring(),
            key=backend.by_priority,
        )
    )


def _load_keyring_class(keyring_name):
    """
    Load the keyring class indicated by name.

    These popular names are tested to ensure their presence.

    >>> popular_names = [
    ...      'keyring.backends.Windows.WinVaultKeyring',
    ...      'keyring.backends.macOS.Keyring',
    ...      'keyring.backends.kwallet.DBusKeyring',
    ...      'keyring.backends.SecretService.Keyring',
    ...  ]
    >>> list(map(_load_keyring_class, popular_names))
    [...]

    These legacy names are retained for compatibility.

    >>> legacy_names = [
    ...  ]
    >>> list(map(_load_keyring_class, legacy_names))
    [...]
    """
    module_name, sep, class_name = keyring_name.rpartition('.')
    __import__(module_name)
    module = sys.modules[module_name]
    return getattr(module, class_name)


def load_keyring(keyring_name):
    """
    Load the specified keyring by name (a fully-qualified name to the
    keyring, such as 'keyring.backends.file.PlaintextKeyring')
    """
    class_ = _load_keyring_class(keyring_name)
    # invoke the priority to ensure it is viable, or raise a RuntimeError
    class_.priority
    return class_()


def load_env():
    """Load a keyring configured in the environment variable."""
    try:
        return load_keyring(os.environ['PYTHON_KEYRING_BACKEND'])
    except KeyError:
        pass


def load_config():
    """Load a keyring using the config file in the config root."""

    filename = 'keyringrc.cfg'

    keyring_cfg = os.path.join(platform.config_root(), filename)

    if not os.path.exists(keyring_cfg):
        return

    config = configparser.RawConfigParser()
    config.read(keyring_cfg)
    _load_keyring_path(config)

    # load the keyring class name, and then load this keyring
    try:
        if config.has_section("backend"):
            keyring_name = config.get("backend", "default-keyring").strip()
        else:
            raise configparser.NoOptionError('backend', 'default-keyring')

    except (configparser.NoOptionError, ImportError):
        logger = logging.getLogger('keyring')
        logger.warning(
            "Keyring config file contains incorrect values.\n"
            + "Config file: %s" % keyring_cfg
        )
        return

    return load_keyring(keyring_name)


def _load_keyring_path(config):
    "load the keyring-path option (if present)"
    try:
        path = config.get("backend", "keyring-path").strip()
        sys.path.insert(0, path)
    except (configparser.NoOptionError, configparser.NoSectionError):
        pass
