#   Licensed under the Apache License, Version 2.0 (the "License");
#   you may not use this file except in compliance with the License.
#   You may obtain a copy of the License at
#
#       http://www.apache.org/licenses/LICENSE-2.0
#
#   Unless required by applicable law or agreed to in writing, software
#   distributed under the License is distributed on an "AS IS" BASIS,
#   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#   See the License for the specific language governing permissions and
#   limitations under the License.

import os, winreg, sys


class WindowsRegistry(object):

    def __init__(self):
        self.handle = winreg.ConnectRegistry(None, winreg.HKEY_LOCAL_MACHINE)

    def get(self, key, name):

        handle = None
        try:
            handle = winreg.OpenKey(self.handle, key)
            return winreg.QueryValueEx(handle, name)[0]
        finally:
            if handle is not None:
                handle.Close()

    def close(self):
        self.handle.Close()


def get_jvm_dll_directory():
    """
    Checks JCC_JDK, JAVA_HOME, windows registry to find jvm.dll

    Raises:
        ValueError: If no jvm.dll could be found

    Returns:
        str: Path to jvm.dll, None if not found
    """

    # First check environment variable JCC_JDK
    path = os.getenv('JCC_JDK')

    #  check environment variable JAVA_HOME
    if path is None:
        path = os.getenv('JAVA_HOME')

    # Check Windows Registry if nothing yet
    if path is None:
        jre_key = r"SOFTWARE\JavaSoft\Java Runtime Environment"
        jdk_key = r"SOFTWARE\JavaSoft\Java Development Kit"
        current_key = r"%s\%s"

        registry = None
        try:
            registry = WindowsRegistry()

            try: # try JRE
                version = registry.get(jre_key, "CurrentVersion")
                path = registry.get(current_key %(jre_key, version), "JavaHome")
                if not os.path.exists(path):
                    path = None
            except:
                path = None

            if not path:
                try: # try JDK
                    version = registry.get(jdk_key, "CurrentVersion")
                    path = registry.get(current_key %(jdk_key, version), "JavaHome")
                    if os.path.exists(path):
                        path = os.path.abspath(os.path.join(path, "jre"))
                    else:
                        path = None
                except:
                    path = None

        finally:
            if registry is not None:
                registry.close()

    # Traverse the found path to identify if there is a jvm.dll somewhere there
    if path:
        for location in (('bin', 'client'),('bin', 'server'),('jre', 'bin', 'client'),('jre', 'bin', 'server')):
            jvm_path = os.path.abspath(os.path.join(path, *location))
            if os.path.exists(os.path.join(jvm_path, "jvm.dll")):
                return jvm_path

    return None


def add_jvm_dll_directory_to_path():
    """
    Triggers the search for jvm.dll
    And adds the dll search path for python.

    Raises:
        ValueError: If no jvm.dll could be found

    Returns:
       bool: True if successful
    """

    dll_path = get_jvm_dll_directory()

    if dll_path is None:
        raise ValueError("jvm.dll could not be found")

    if sys.version_info.major == 3 and sys.version_info.minor >=8:
        # From python 3.8 use add_dll_directory instead of PATH
        os.add_dll_directory(dll_path)

    else:
        path = os.environ['Path'].split(os.pathsep)
        path.append(dll_path)
        os.environ['Path'] = os.pathsep.join(path)

    return True
