# (C) British Crown Copyright 2013 - 2017, Met Office
#
# This file is part of Iris.
#
# Iris is free software: you can redistribute it and/or modify it under
# the terms of the GNU Lesser General Public License as published by the
# Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Iris is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with Iris.  If not, see <http://www.gnu.org/licenses/>.
"""Unit tests for the `iris.cube.Cube` class."""

from __future__ import (absolute_import, division, print_function)
from six.moves import (filter, input, map, range, zip)  # noqa

# Import iris.tests first so that some things can be initialised before
# importing anything else.
import iris.tests as tests

from itertools import permutations

import biggus
import numpy as np
import numpy.ma as ma

import iris.aux_factory
import iris.coords
import iris.exceptions
from iris import FUTURE
from iris.analysis import WeightedAggregator, Aggregator
from iris.analysis import MEAN
from iris.cube import Cube
from iris.coords import AuxCoord, DimCoord, CellMeasure
from iris.exceptions import CoordinateNotFoundError, CellMeasureNotFoundError
from iris.tests import mock
import iris.tests.stock as stock


class Test___init___data(tests.IrisTest):
    def test_ndarray(self):
        # np.ndarray should be allowed through
        data = np.arange(12).reshape(3, 4)
        cube = Cube(data)
        self.assertEqual(type(cube.data), np.ndarray)
        self.assertArrayEqual(cube.data, data)

    def test_masked(self):
        # np.ma.MaskedArray should be allowed through
        data = np.ma.masked_greater(np.arange(12).reshape(3, 4), 1)
        cube = Cube(data)
        self.assertEqual(type(cube.data), np.ma.MaskedArray)
        self.assertMaskedArrayEqual(cube.data, data)

    def test_matrix(self):
        # Subclasses of np.ndarray should be coerced back to np.ndarray.
        # (Except for np.ma.MaskedArray.)
        data = np.matrix([[1, 2, 3], [4, 5, 6]])
        cube = Cube(data)
        self.assertEqual(type(cube.data), np.ndarray)
        self.assertArrayEqual(cube.data, data)

    def test_default_share_data(self):
        cube = Cube(np.arange(1))
        self.assertFalse(cube.share_data)


class Test_extract(tests.IrisTest):
    def test_scalar_cube_exists(self):
        # Ensure that extract is able to extract a scalar cube.
        constraint = iris.Constraint(name='a1')
        cube = Cube(1, long_name='a1')
        res = cube.extract(constraint)
        self.assertIs(res, cube)

    def test_scalar_cube_noexists(self):
        # Ensure that extract does not return a non-matching scalar cube.
        constraint = iris.Constraint(name='a2')
        cube = Cube(1, long_name='a1')
        res = cube.extract(constraint)
        self.assertIs(res, None)

    def test_scalar_cube_coord_match(self):
        # Ensure that extract is able to extract a scalar cube according to
        # constrained scalar coordinate.
        constraint = iris.Constraint(scalar_coord=0)
        cube = Cube(1, long_name='a1')
        coord = iris.coords.AuxCoord(0, long_name='scalar_coord')
        cube.add_aux_coord(coord, None)
        res = cube.extract(constraint)
        self.assertIs(res, cube)

    def test_scalar_cube_coord_nomatch(self):
        # Ensure that extract is not extracting a scalar cube with scalar
        # coordinate that does not match the constraint.
        constraint = iris.Constraint(scalar_coord=1)
        cube = Cube(1, long_name='a1')
        coord = iris.coords.AuxCoord(0, long_name='scalar_coord')
        cube.add_aux_coord(coord, None)
        res = cube.extract(constraint)
        self.assertIs(res, None)

    def test_1d_cube_exists(self):
        # Ensure that extract is able to extract from a 1d cube.
        constraint = iris.Constraint(name='a1')
        cube = Cube([1], long_name='a1')
        res = cube.extract(constraint)
        self.assertIs(res, cube)

    def test_1d_cube_noexists(self):
        # Ensure that extract does not return a non-matching 1d cube.
        constraint = iris.Constraint(name='a2')
        cube = Cube([1], long_name='a1')
        res = cube.extract(constraint)
        self.assertIs(res, None)


class Test_xml(tests.IrisTest):
    def test_checksum_ignores_masked_values(self):
        # Mask out an single element.
        data = np.ma.arange(12).reshape(3, 4)
        data[1, 2] = np.ma.masked
        cube = Cube(data)
        self.assertCML(cube)

        # If we change the underlying value before masking it, the
        # checksum should be unaffected.
        data = np.ma.arange(12).reshape(3, 4)
        data[1, 2] = 42
        data[1, 2] = np.ma.masked
        cube = Cube(data)
        self.assertCML(cube)

    def test_byteorder_default(self):
        cube = Cube(np.arange(3))
        self.assertIn('byteorder', cube.xml())

    def test_byteorder_false(self):
        cube = Cube(np.arange(3))
        self.assertNotIn('byteorder', cube.xml(byteorder=False))

    def test_byteorder_true(self):
        cube = Cube(np.arange(3))
        self.assertIn('byteorder', cube.xml(byteorder=True))


class Test_collapsed__lazy(tests.IrisTest):
    def setUp(self):
        self.data = np.arange(6.0).reshape((2, 3))
        self.lazydata = biggus.NumpyArrayAdapter(self.data)
        cube = Cube(self.lazydata)
        for i_dim, name in enumerate(('y', 'x')):
            npts = cube.shape[i_dim]
            coord = DimCoord(np.arange(npts), long_name=name)
            cube.add_dim_coord(coord, i_dim)
        self.cube = cube

    def test_dim0_lazy(self):
        cube_collapsed = self.cube.collapsed('y', MEAN)
        self.assertTrue(cube_collapsed.has_lazy_data())
        self.assertArrayAlmostEqual(cube_collapsed.data, [1.5, 2.5, 3.5])
        self.assertFalse(cube_collapsed.has_lazy_data())

    def test_dim1_lazy(self):
        cube_collapsed = self.cube.collapsed('x', MEAN)
        self.assertTrue(cube_collapsed.has_lazy_data())
        self.assertArrayAlmostEqual(cube_collapsed.data, [1.0, 4.0])
        self.assertFalse(cube_collapsed.has_lazy_data())

    def test_fail_multidims(self):
        # Check that MEAN produces a suitable error message for multiple dims.
        # N.B. non-lazy op can do this
        self.cube.collapsed(('x', 'y'), MEAN)

    def test_non_lazy_aggregator(self):
        # An aggregator which doesn't have a lazy function should still work.
        dummy_agg = Aggregator('custom_op',
                               lambda x, axis=None: np.mean(x, axis=axis))
        result = self.cube.collapsed('x', dummy_agg)
        self.assertFalse(result.has_lazy_data())
        self.assertArrayEqual(result.data, np.mean(self.data, axis=1))


class Test_collapsed__warning(tests.IrisTest):
    def setUp(self):
        self.cube = Cube([[1, 2], [1, 2]])
        lat = DimCoord([1, 2], standard_name='latitude')
        lon = DimCoord([1, 2], standard_name='longitude')
        grid_lat = AuxCoord([1, 2], standard_name='grid_latitude')
        grid_lon = AuxCoord([1, 2], standard_name='grid_longitude')
        wibble = AuxCoord([1, 2], long_name='wibble')

        self.cube.add_dim_coord(lat, 0)
        self.cube.add_dim_coord(lon, 1)
        self.cube.add_aux_coord(grid_lat, 0)
        self.cube.add_aux_coord(grid_lon, 1)
        self.cube.add_aux_coord(wibble, 1)

    def _aggregator(self, uses_weighting):
        # Returns a mock aggregator with a mocked method (uses_weighting)
        # which returns the given True/False condition.
        aggregator = mock.Mock(spec=WeightedAggregator, lazy_func=None)
        aggregator.cell_method = None
        aggregator.uses_weighting = mock.Mock(return_value=uses_weighting)

        return aggregator

    def _assert_warn_collapse_without_weight(self, coords, warn):
        # Ensure that warning is raised.
        msg = "Collapsing spatial coordinate {!r} without weighting"
        for coord in coords:
            self.assertIn(mock.call(msg.format(coord)), warn.call_args_list)

    def _assert_nowarn_collapse_without_weight(self, coords, warn):
        # Ensure that warning is not rised.
        msg = "Collapsing spatial coordinate {!r} without weighting"
        for coord in coords:
            self.assertNotIn(mock.call(msg.format(coord)), warn.call_args_list)

    def test_lat_lon_noweighted_aggregator(self):
        # Collapse latitude coordinate with unweighted aggregator.
        aggregator = mock.Mock(spec=Aggregator, lazy_func=None)
        aggregator.cell_method = None
        coords = ['latitude', 'longitude']

        with mock.patch('warnings.warn') as warn:
            self.cube.collapsed(coords, aggregator, somekeyword='bla')

        self._assert_nowarn_collapse_without_weight(coords, warn)

    def test_lat_lon_weighted_aggregator(self):
        # Collapse latitude coordinate with weighted aggregator without
        # providing weights.
        aggregator = self._aggregator(False)
        coords = ['latitude', 'longitude']

        with mock.patch('warnings.warn') as warn:
            self.cube.collapsed(coords, aggregator)

        coords = [coord for coord in coords if 'latitude' in coord]
        self._assert_warn_collapse_without_weight(coords, warn)

    def test_lat_lon_weighted_aggregator_with_weights(self):
        # Collapse latitude coordinate with a weighted aggregators and
        # providing suitable weights.
        weights = np.array([[0.1, 0.5], [0.3, 0.2]])
        aggregator = self._aggregator(True)
        coords = ['latitude', 'longitude']

        with mock.patch('warnings.warn') as warn:
            self.cube.collapsed(coords, aggregator, weights=weights)

        self._assert_nowarn_collapse_without_weight(coords, warn)

    def test_lat_lon_weighted_aggregator_alt(self):
        # Collapse grid_latitude coordinate with weighted aggregator without
        # providing weights.  Tests coordinate matching logic.
        aggregator = self._aggregator(False)
        coords = ['grid_latitude', 'grid_longitude']

        with mock.patch('warnings.warn') as warn:
            self.cube.collapsed(coords, aggregator)

        coords = [coord for coord in coords if 'latitude' in coord]
        self._assert_warn_collapse_without_weight(coords, warn)

    def test_no_lat_weighted_aggregator_mixed(self):
        # Collapse grid_latitude and an unmatched coordinate (not lat/lon)
        # with weighted aggregator without providing weights.
        # Tests coordinate matching logic.
        aggregator = self._aggregator(False)
        coords = ['wibble']

        with mock.patch('warnings.warn') as warn:
            self.cube.collapsed(coords, aggregator)

        self._assert_nowarn_collapse_without_weight(coords, warn)


class Test_summary(tests.IrisTest):
    def test_cell_datetime_objects(self):
        # Check the scalar coordinate summary still works even when
        # iris.FUTURE.cell_datetime_objects is True.
        cube = Cube(0)
        cube.add_aux_coord(AuxCoord(42, units='hours since epoch'))
        with FUTURE.context(cell_datetime_objects=True):
            summary = cube.summary()
        self.assertIn('1970-01-02 18:00:00', summary)


class Test_is_compatible(tests.IrisTest):
    def setUp(self):
        self.test_cube = Cube([1.])
        self.other_cube = self.test_cube.copy()

    def test_noncommon_array_attrs_compatible(self):
        # Non-common array attributes should be ok.
        self.test_cube.attributes['array_test'] = np.array([1.0, 2, 3])
        self.assertTrue(self.test_cube.is_compatible(self.other_cube))

    def test_matching_array_attrs_compatible(self):
        # Matching array attributes should be ok.
        self.test_cube.attributes['array_test'] = np.array([1.0, 2, 3])
        self.other_cube.attributes['array_test'] = np.array([1.0, 2, 3])
        self.assertTrue(self.test_cube.is_compatible(self.other_cube))

    def test_different_array_attrs_incompatible(self):
        # Differing array attributes should make the cubes incompatible.
        self.test_cube.attributes['array_test'] = np.array([1.0, 2, 3])
        self.other_cube.attributes['array_test'] = np.array([1.0, 2, 777.7])
        self.assertFalse(self.test_cube.is_compatible(self.other_cube))


class Test_aggregated_by(tests.IrisTest):
    def setUp(self):
        self.cube = Cube(np.arange(11))
        val_coord = AuxCoord([0, 0, 0, 1, 1, 2, 0, 0, 2, 0, 1],
                             long_name="val")
        label_coord = AuxCoord(['alpha', 'alpha', 'beta',
                                'beta', 'alpha', 'gamma',
                                'alpha', 'alpha', 'alpha',
                                'gamma', 'beta'],
                               long_name='label', units='no_unit')
        self.cube.add_aux_coord(val_coord, 0)
        self.cube.add_aux_coord(label_coord, 0)
        self.mock_agg = mock.Mock(spec=Aggregator)
        self.mock_agg.cell_method = []
        self.mock_agg.aggregate = mock.Mock(
            return_value=mock.Mock(dtype='object'))
        self.mock_agg.aggregate_shape = mock.Mock(return_value=())
        self.mock_agg.post_process = mock.Mock(side_effect=lambda x, y, z: x)

    def test_string_coord_agg_by_label(self):
        # Aggregate a cube on a string coordinate label where label
        # and val entries are not in step; the resulting cube has a val
        # coord of bounded cells and a label coord of single string entries.
        res_cube = self.cube.aggregated_by('label', self.mock_agg)
        val_coord = AuxCoord(np.array([1., 0.5, 1.]),
                             bounds=np.array([[0, 2], [0, 1], [2, 0]]),
                             long_name='val')
        label_coord = AuxCoord(np.array(['alpha', 'beta', 'gamma']),
                               long_name='label', units='no_unit')
        self.assertEqual(res_cube.coord('val'), val_coord)
        self.assertEqual(res_cube.coord('label'), label_coord)

    def test_string_coord_agg_by_val(self):
        # Aggregate a cube on a numeric coordinate val where label
        # and val entries are not in step; the resulting cube has a label
        # coord with serialised labels from the aggregated cells.
        res_cube = self.cube.aggregated_by('val', self.mock_agg)
        val_coord = AuxCoord(np.array([0,  1,  2]), long_name='val')
        exp0 = 'alpha|alpha|beta|alpha|alpha|gamma'
        exp1 = 'beta|alpha|beta'
        exp2 = 'gamma|alpha'
        label_coord = AuxCoord(np.array((exp0, exp1, exp2)),
                               long_name='label', units='no_unit')
        self.assertEqual(res_cube.coord('val'), val_coord)
        self.assertEqual(res_cube.coord('label'), label_coord)

    def test_single_string_aggregation(self):
        aux_coords = [(AuxCoord(['a', 'b', 'a'], long_name='foo'), 0),
                      (AuxCoord(['a', 'a', 'a'], long_name='bar'), 0)]
        cube = iris.cube.Cube(np.arange(12).reshape(3, 4),
                              aux_coords_and_dims=aux_coords)
        result = cube.aggregated_by('foo', MEAN)
        self.assertEqual(result.shape, (2, 4))
        self.assertEqual(result.coord('bar'),
                         AuxCoord(['a|a', 'a'], long_name='bar'))


class Test_rolling_window(tests.IrisTest):
    def setUp(self):
        self.cube = Cube(np.arange(6))
        val_coord = DimCoord([0, 1, 2, 3, 4, 5], long_name="val")
        month_coord = AuxCoord(['jan', 'feb', 'mar', 'apr', 'may', 'jun'],
                               long_name='month')
        self.cube.add_dim_coord(val_coord, 0)
        self.cube.add_aux_coord(month_coord, 0)
        self.mock_agg = mock.Mock(spec=Aggregator)
        self.mock_agg.aggregate = mock.Mock(
            return_value=np.empty([4]))
        self.mock_agg.post_process = mock.Mock(side_effect=lambda x, y, z: x)

    def test_string_coord(self):
        # Rolling window on a cube that contains a string coordinate.
        res_cube = self.cube.rolling_window('val', self.mock_agg, 3)
        val_coord = DimCoord(np.array([1, 2, 3, 4]),
                             bounds=np.array([[0, 2], [1, 3], [2, 4], [3, 5]]),
                             long_name='val')
        month_coord = AuxCoord(
            np.array(['jan|feb|mar', 'feb|mar|apr', 'mar|apr|may',
                      'apr|may|jun']),
            bounds=np.array([['jan', 'mar'], ['feb', 'apr'],
                             ['mar', 'may'], ['apr', 'jun']]),
            long_name='month')
        self.assertEqual(res_cube.coord('val'), val_coord)
        self.assertEqual(res_cube.coord('month'), month_coord)

    def test_kwargs(self):
        # Rolling window with missing data not tolerated
        window = 2
        self.cube.data = np.ma.array(self.cube.data,
                                     mask=([True, False, False,
                                            False, True, False]))
        res_cube = self.cube.rolling_window('val', iris.analysis.MEAN,
                                            window, mdtol=0)
        expected_result = np.ma.array([-99., 1.5, 2.5, -99., -99.],
                                      mask=[True, False, False, True, True],
                                      dtype=np.float64)
        self.assertMaskedArrayEqual(expected_result, res_cube.data)


class Test_slices_dim_order(tests.IrisTest):
    '''
    This class tests the capability of iris.cube.Cube.slices(), including its
    ability to correctly re-order the dimensions.
    '''
    def setUp(self):
        '''
        setup a 4D iris cube, each dimension is length 1.
        The dimensions are;
            dim1: time
            dim2: height
            dim3: latitude
            dim4: longitude
        '''
        self.cube = iris.cube.Cube(np.array([[[[8.]]]]))
        self.cube.add_dim_coord(iris.coords.DimCoord([0], "time"), [0])
        self.cube.add_dim_coord(iris.coords.DimCoord([0], "height"), [1])
        self.cube.add_dim_coord(iris.coords.DimCoord([0], "latitude"), [2])
        self.cube.add_dim_coord(iris.coords.DimCoord([0], "longitude"), [3])

    @staticmethod
    def expected_cube_setup(dim1name, dim2name, dim3name):
        '''
        input:
        ------
            dim1name: str
                name of the first dimension coordinate
            dim2name: str
                name of the second dimension coordinate
            dim3name: str
                name of the third dimension coordinate
        output:
        ------
            cube: iris cube
                iris cube with the specified axis holding the data 8
        '''
        cube = iris.cube.Cube(np.array([[[8.]]]))
        cube.add_dim_coord(iris.coords.DimCoord([0], dim1name), [0])
        cube.add_dim_coord(iris.coords.DimCoord([0], dim2name), [1])
        cube.add_dim_coord(iris.coords.DimCoord([0], dim3name), [2])
        return cube

    def check_order(self, dim1, dim2, dim3, dim_to_remove):
        '''
        does two things:
        (1) slices the 4D cube in dim1, dim2, dim3 (and removes the scalar
        coordinate) and
        (2) sets up a 3D cube with dim1, dim2, dim3.
        input:
        -----
            dim1: str
                name of first dimension
            dim2: str
                name of second dimension
            dim3: str
                name of third dimension
            dim_to_remove: str
                name of the dimension that transforms into a scalar coordinate
                when slicing the cube.
        output:
        ------
            sliced_cube: 3D cube
                the cube that results if slicing the original cube
            expected_cube: 3D cube
                a cube set up with the axis corresponding to the dims
        '''
        sliced_cube = next(self.cube.slices([dim1, dim2, dim3]))
        sliced_cube.remove_coord(dim_to_remove)
        expected_cube = self.expected_cube_setup(dim1, dim2, dim3)
        self.assertEqual(sliced_cube, expected_cube)

    def test_all_permutations(self):
        for perm in permutations(["time", "height", "latitude", "longitude"]):
            self.check_order(*perm)


@tests.skip_data
class Test_slices_over(tests.IrisTest):
    def setUp(self):
        self.cube = stock.realistic_4d()
        # Define expected iterators for 1D and 2D test cases.
        self.exp_iter_1d = range(
            len(self.cube.coord('model_level_number').points))
        self.exp_iter_2d = np.ndindex(6, 70, 1, 1)
        # Define maximum number of interations for particularly long
        # (and so time-consuming) iterators.
        self.long_iterator_max = 5

    def test_1d_slice_coord_given(self):
        res = self.cube.slices_over(self.cube.coord('model_level_number'))
        for i, res_cube in zip(self.exp_iter_1d, res):
            expected = self.cube[:, i]
            self.assertEqual(res_cube, expected)

    def test_1d_slice_nonexistent_coord_given(self):
        with self.assertRaises(CoordinateNotFoundError):
            res = self.cube.slices_over(self.cube.coord('wibble'))

    def test_1d_slice_coord_name_given(self):
        res = self.cube.slices_over('model_level_number')
        for i, res_cube in zip(self.exp_iter_1d, res):
            expected = self.cube[:, i]
            self.assertEqual(res_cube, expected)

    def test_1d_slice_nonexistent_coord_name_given(self):
        with self.assertRaises(CoordinateNotFoundError):
            res = self.cube.slices_over('wibble')

    def test_1d_slice_dimension_given(self):
        res = self.cube.slices_over(1)
        for i, res_cube in zip(self.exp_iter_1d, res):
            expected = self.cube[:, i]
            self.assertEqual(res_cube, expected)

    def test_1d_slice_nonexistent_dimension_given(self):
        with self.assertRaisesRegexp(ValueError, 'iterator over a dimension'):
            res = self.cube.slices_over(self.cube.ndim + 1)

    def test_2d_slice_coord_given(self):
        # Slicing over these two dimensions returns 420 2D cubes, so only check
        # cubes up to `self.long_iterator_max` to keep test runtime sensible.
        res = self.cube.slices_over([self.cube.coord('time'),
                                     self.cube.coord('model_level_number')])
        for ct in range(self.long_iterator_max):
            indices = list(next(self.exp_iter_2d))
            # Replace the dimensions not iterated over with spanning slices.
            indices[2] = indices[3] = slice(None)
            expected = self.cube[tuple(indices)]
            self.assertEqual(next(res), expected)

    def test_2d_slice_nonexistent_coord_given(self):
        with self.assertRaises(CoordinateNotFoundError):
            res = self.cube.slices_over([self.cube.coord('time'),
                                         self.cube.coord('wibble')])

    def test_2d_slice_coord_name_given(self):
        # Slicing over these two dimensions returns 420 2D cubes, so only check
        # cubes up to `self.long_iterator_max` to keep test runtime sensible.
        res = self.cube.slices_over(['time', 'model_level_number'])
        for ct in range(self.long_iterator_max):
            indices = list(next(self.exp_iter_2d))
            # Replace the dimensions not iterated over with spanning slices.
            indices[2] = indices[3] = slice(None)
            expected = self.cube[tuple(indices)]
            self.assertEqual(next(res), expected)

    def test_2d_slice_nonexistent_coord_name_given(self):
        with self.assertRaises(CoordinateNotFoundError):
            res = self.cube.slices_over(['time', 'wibble'])

    def test_2d_slice_dimension_given(self):
        # Slicing over these two dimensions returns 420 2D cubes, so only check
        # cubes up to `self.long_iterator_max` to keep test runtime sensible.
        res = self.cube.slices_over([0, 1])
        for ct in range(self.long_iterator_max):
            indices = list(next(self.exp_iter_2d))
            # Replace the dimensions not iterated over with spanning slices.
            indices[2] = indices[3] = slice(None)
            expected = self.cube[tuple(indices)]
            self.assertEqual(next(res), expected)

    def test_2d_slice_reversed_dimension_given(self):
        # Confirm that reversing the order of the dimensions returns the same
        # results as the above test.
        res = self.cube.slices_over([1, 0])
        for ct in range(self.long_iterator_max):
            indices = list(next(self.exp_iter_2d))
            # Replace the dimensions not iterated over with spanning slices.
            indices[2] = indices[3] = slice(None)
            expected = self.cube[tuple(indices)]
            self.assertEqual(next(res), expected)

    def test_2d_slice_nonexistent_dimension_given(self):
        with self.assertRaisesRegexp(ValueError, 'iterator over a dimension'):
            res = self.cube.slices_over([0, self.cube.ndim + 1])

    def test_multidim_slice_coord_given(self):
        # Slicing over surface altitude returns 100x100 2D cubes, so only check
        # cubes up to `self.long_iterator_max` to keep test runtime sensible.
        res = self.cube.slices_over('surface_altitude')
        # Define special ndindex iterator for the different dims sliced over.
        nditer = np.ndindex(1, 1, 100, 100)
        for ct in range(self.long_iterator_max):
            indices = list(next(nditer))
            # Replace the dimensions not iterated over with spanning slices.
            indices[0] = indices[1] = slice(None)
            expected = self.cube[tuple(indices)]
            self.assertEqual(next(res), expected)

    def test_duplicate_coordinate_given(self):
        res = self.cube.slices_over([1, 1])
        for i, res_cube in zip(self.exp_iter_1d, res):
            expected = self.cube[:, i]
            self.assertEqual(res_cube, expected)

    def test_non_orthogonal_coordinates_given(self):
        res = self.cube.slices_over(['model_level_number', 'sigma'])
        for i, res_cube in zip(self.exp_iter_1d, res):
            expected = self.cube[:, i]
            self.assertEqual(res_cube, expected)

    def test_nodimension(self):
        # Slicing over no dimension should return the whole cube.
        res = self.cube.slices_over([])
        self.assertEqual(next(res), self.cube)


def create_cube(lon_min, lon_max, bounds=False):
    n_lons = max(lon_min, lon_max) - min(lon_max, lon_min)
    data = np.arange(4 * 3 * n_lons, dtype='f4').reshape(4, 3, -1)
    data = biggus.NumpyArrayAdapter(data)
    cube = Cube(data, standard_name='x_wind', units='ms-1')
    cube.add_dim_coord(iris.coords.DimCoord([0, 20, 40, 80],
                                            long_name='level_height',
                                            units='m'), 0)
    cube.add_aux_coord(iris.coords.AuxCoord([1.0, 0.9, 0.8, 0.6],
                                            long_name='sigma'), 0)
    cube.add_dim_coord(iris.coords.DimCoord([-45, 0, 45], 'latitude',
                                            units='degrees'), 1)
    step = 1 if lon_max > lon_min else -1
    circular = (abs(lon_max - lon_min) == 360)
    cube.add_dim_coord(iris.coords.DimCoord(np.arange(lon_min, lon_max, step),
                                            'longitude', units='degrees',
                                            circular=circular), 2)
    if bounds:
        cube.coord('longitude').guess_bounds()
    cube.add_aux_coord(iris.coords.AuxCoord(
        np.arange(3 * n_lons).reshape(3, -1) * 10, 'surface_altitude',
        units='m'), [1, 2])
    cube.add_aux_factory(iris.aux_factory.HybridHeightFactory(
        cube.coord('level_height'), cube.coord('sigma'),
        cube.coord('surface_altitude')))
    return cube


# Ensure all the other coordinates and factories are correctly preserved.
class Test_intersection__Metadata(tests.IrisTest):
    def test_metadata(self):
        cube = create_cube(0, 360)
        result = cube.intersection(longitude=(170, 190))
        self.assertCMLApproxData(result)

    def test_metadata_wrapped(self):
        cube = create_cube(-180, 180)
        result = cube.intersection(longitude=(170, 190))
        self.assertCMLApproxData(result)


# Explicitly check the handling of `circular` on the result.
class Test_intersection__Circular(tests.IrisTest):
    def test_regional(self):
        cube = create_cube(0, 360)
        result = cube.intersection(longitude=(170, 190))
        self.assertFalse(result.coord('longitude').circular)

    def test_regional_wrapped(self):
        cube = create_cube(-180, 180)
        result = cube.intersection(longitude=(170, 190))
        self.assertFalse(result.coord('longitude').circular)

    def test_global(self):
        cube = create_cube(-180, 180)
        result = cube.intersection(longitude=(-180, 180))
        self.assertTrue(result.coord('longitude').circular)

    def test_global_wrapped(self):
        cube = create_cube(-180, 180)
        result = cube.intersection(longitude=(10, 370))
        self.assertTrue(result.coord('longitude').circular)


# Check the various error conditions.
class Test_intersection__Invalid(tests.IrisTest):
    def test_reversed_min_max(self):
        cube = create_cube(0, 360)
        with self.assertRaises(ValueError):
            cube.intersection(longitude=(30, 10))

    def test_dest_too_large(self):
        cube = create_cube(0, 360)
        with self.assertRaises(ValueError):
            cube.intersection(longitude=(30, 500))

    def test_src_too_large(self):
        cube = create_cube(0, 400)
        with self.assertRaises(ValueError):
            cube.intersection(longitude=(10, 30))

    def test_missing_coord(self):
        cube = create_cube(0, 360)
        with self.assertRaises(iris.exceptions.CoordinateNotFoundError):
            cube.intersection(parrots=(10, 30))

    def test_multi_dim_coord(self):
        cube = create_cube(0, 360)
        with self.assertRaises(iris.exceptions.CoordinateMultiDimError):
            cube.intersection(surface_altitude=(10, 30))

    def test_null_region(self):
        # 10 <= v < 10
        cube = create_cube(0, 360)
        with self.assertRaises(IndexError):
            cube.intersection(longitude=(10, 10, False, False))


class Test_intersection__Lazy(tests.IrisTest):
    def test_real_data(self):
        cube = create_cube(0, 360)
        cube.data
        result = cube.intersection(longitude=(170, 190))
        self.assertFalse(result.has_lazy_data())
        self.assertArrayEqual(result.coord('longitude').points,
                              np.arange(170, 191))
        self.assertEqual(result.data[0, 0, 0], 170)
        self.assertEqual(result.data[0, 0, -1], 190)

    def test_real_data_wrapped(self):
        cube = create_cube(-180, 180)
        cube.data
        result = cube.intersection(longitude=(170, 190))
        self.assertFalse(result.has_lazy_data())
        self.assertArrayEqual(result.coord('longitude').points,
                              np.arange(170, 191))
        self.assertEqual(result.data[0, 0, 0], 350)
        self.assertEqual(result.data[0, 0, -1], 10)

    def test_lazy_data(self):
        cube = create_cube(0, 360)
        result = cube.intersection(longitude=(170, 190))
        self.assertTrue(result.has_lazy_data())
        self.assertArrayEqual(result.coord('longitude').points,
                              np.arange(170, 191))
        self.assertEqual(result.data[0, 0, 0], 170)
        self.assertEqual(result.data[0, 0, -1], 190)

    def test_lazy_data_wrapped(self):
        cube = create_cube(-180, 180)
        result = cube.intersection(longitude=(170, 190))
        self.assertTrue(result.has_lazy_data())
        self.assertArrayEqual(result.coord('longitude').points,
                              np.arange(170, 191))
        self.assertEqual(result.data[0, 0, 0], 350)
        self.assertEqual(result.data[0, 0, -1], 10)


class Test_intersection_Points(tests.IrisTest):
    def test_ignore_bounds(self):
        cube = create_cube(0, 30, bounds=True)
        result = cube.intersection(longitude=(9.5, 12.5), ignore_bounds=True)
        self.assertArrayEqual(result.coord('longitude').points,
                              np.arange(10, 13))
        self.assertArrayEqual(result.coord('longitude').bounds[0],
                              [9.5, 10.5])
        self.assertArrayEqual(result.coord('longitude').bounds[-1],
                              [11.5, 12.5])


# Check what happens with a regional, points-only circular intersection
# coordinate.
class Test_intersection__RegionalSrcModulus(tests.IrisTest):
    def test_request_subset(self):
        cube = create_cube(40, 60)
        result = cube.intersection(longitude=(45, 50))
        self.assertArrayEqual(result.coord('longitude').points,
                              np.arange(45, 51))
        self.assertArrayEqual(result.data[0, 0], np.arange(5, 11))

    def test_request_left(self):
        cube = create_cube(40, 60)
        result = cube.intersection(longitude=(35, 45))
        self.assertArrayEqual(result.coord('longitude').points,
                              np.arange(40, 46))
        self.assertArrayEqual(result.data[0, 0], np.arange(0, 6))

    def test_request_right(self):
        cube = create_cube(40, 60)
        result = cube.intersection(longitude=(55, 65))
        self.assertArrayEqual(result.coord('longitude').points,
                              np.arange(55, 60))
        self.assertArrayEqual(result.data[0, 0], np.arange(15, 20))

    def test_request_superset(self):
        cube = create_cube(40, 60)
        result = cube.intersection(longitude=(35, 65))
        self.assertArrayEqual(result.coord('longitude').points,
                              np.arange(40, 60))
        self.assertArrayEqual(result.data[0, 0], np.arange(0, 20))

    def test_request_subset_modulus(self):
        cube = create_cube(40, 60)
        result = cube.intersection(longitude=(45 + 360, 50 + 360))
        self.assertArrayEqual(result.coord('longitude').points,
                              np.arange(45 + 360, 51 + 360))
        self.assertArrayEqual(result.data[0, 0], np.arange(5, 11))

    def test_request_left_modulus(self):
        cube = create_cube(40, 60)
        result = cube.intersection(longitude=(35 + 360, 45 + 360))
        self.assertArrayEqual(result.coord('longitude').points,
                              np.arange(40 + 360, 46 + 360))
        self.assertArrayEqual(result.data[0, 0], np.arange(0, 6))

    def test_request_right_modulus(self):
        cube = create_cube(40, 60)
        result = cube.intersection(longitude=(55 + 360, 65 + 360))
        self.assertArrayEqual(result.coord('longitude').points,
                              np.arange(55 + 360, 60 + 360))
        self.assertArrayEqual(result.data[0, 0], np.arange(15, 20))

    def test_request_superset_modulus(self):
        cube = create_cube(40, 60)
        result = cube.intersection(longitude=(35 + 360, 65 + 360))
        self.assertArrayEqual(result.coord('longitude').points,
                              np.arange(40 + 360, 60 + 360))
        self.assertArrayEqual(result.data[0, 0], np.arange(0, 20))

    def test_tolerance_f4(self):
        cube = create_cube(0, 5)
        cube.coord('longitude').points = np.array([0., 3.74999905, 7.49999809,
                                                   11.24999714, 14.99999619],
                                                  dtype='f4')
        result = cube.intersection(longitude=(0, 5))

    def test_tolerance_f8(self):
        cube = create_cube(0, 5)
        cube.coord('longitude').points = np.array([0., 3.74999905, 7.49999809,
                                                   11.24999714, 14.99999619],
                                                  dtype='f8')
        result = cube.intersection(longitude=(0, 5))


# Check what happens with a global, points-only circular intersection
# coordinate.
class Test_intersection__GlobalSrcModulus(tests.IrisTest):
    def test_global_wrapped_extreme_increasing_base_period(self):
        # Ensure that we can correctly handle points defined at (base + period)
        cube = create_cube(-180., 180.)
        lons = cube.coord('longitude')
        # Redefine longitude so that points at (base + period)
        lons.points = np.linspace(-180., 180, lons.points.size)
        result = cube.intersection(longitude=(lons.points.min(),
                                              lons.points.max()))
        self.assertArrayEqual(result.data, cube.data)

    def test_global_wrapped_extreme_decreasing_base_period(self):
        # Ensure that we can correctly handle points defined at (base + period)
        cube = create_cube(180., -180.)
        lons = cube.coord('longitude')
        # Redefine longitude so that points at (base + period)
        lons.points = np.linspace(180., -180., lons.points.size)
        result = cube.intersection(longitude=(lons.points.min(),
                                              lons.points.max()))
        self.assertArrayEqual(result.data, cube.data)

    def test_global(self):
        cube = create_cube(0, 360)
        result = cube.intersection(longitude=(0, 360))
        self.assertEqual(result.coord('longitude').points[0], 0)
        self.assertEqual(result.coord('longitude').points[-1], 359)
        self.assertEqual(result.data[0, 0, 0], 0)
        self.assertEqual(result.data[0, 0, -1], 359)

    def test_global_wrapped(self):
        cube = create_cube(0, 360)
        result = cube.intersection(longitude=(-180, 180))
        self.assertEqual(result.coord('longitude').points[0], -180)
        self.assertEqual(result.coord('longitude').points[-1], 179)
        self.assertEqual(result.data[0, 0, 0], 180)
        self.assertEqual(result.data[0, 0, -1], 179)

    def test_aux_coord(self):
        cube = create_cube(0, 360)
        cube.replace_coord(iris.coords.AuxCoord.from_coord(
            cube.coord('longitude')))
        result = cube.intersection(longitude=(0, 360))
        self.assertEqual(result.coord('longitude').points[0], 0)
        self.assertEqual(result.coord('longitude').points[-1], 359)
        self.assertEqual(result.data[0, 0, 0], 0)
        self.assertEqual(result.data[0, 0, -1], 359)

    def test_aux_coord_wrapped(self):
        cube = create_cube(0, 360)
        cube.replace_coord(iris.coords.AuxCoord.from_coord(
            cube.coord('longitude')))
        result = cube.intersection(longitude=(-180, 180))
        self.assertEqual(result.coord('longitude').points[0], 0)
        self.assertEqual(result.coord('longitude').points[-1], -1)
        self.assertEqual(result.data[0, 0, 0], 0)
        self.assertEqual(result.data[0, 0, -1], 359)

    def test_aux_coord_non_contiguous_wrapped(self):
        cube = create_cube(0, 360)
        coord = iris.coords.AuxCoord.from_coord(cube.coord('longitude'))
        coord.points = (coord.points * 1.5) % 360
        cube.replace_coord(coord)
        result = cube.intersection(longitude=(-90, 90))
        self.assertEqual(result.coord('longitude').points[0], 0)
        self.assertEqual(result.coord('longitude').points[-1], 90)
        self.assertEqual(result.data[0, 0, 0], 0)
        self.assertEqual(result.data[0, 0, -1], 300)

    def test_decrementing(self):
        cube = create_cube(360, 0)
        result = cube.intersection(longitude=(40, 60))
        self.assertEqual(result.coord('longitude').points[0], 60)
        self.assertEqual(result.coord('longitude').points[-1], 40)
        self.assertEqual(result.data[0, 0, 0], 300)
        self.assertEqual(result.data[0, 0, -1], 320)

    def test_decrementing_wrapped(self):
        cube = create_cube(360, 0)
        result = cube.intersection(longitude=(-10, 10))
        self.assertEqual(result.coord('longitude').points[0], 10)
        self.assertEqual(result.coord('longitude').points[-1], -10)
        self.assertEqual(result.data[0, 0, 0], 350)
        self.assertEqual(result.data[0, 0, -1], 10)

    def test_no_wrap_after_modulus(self):
        cube = create_cube(0, 360)
        result = cube.intersection(longitude=(170 + 360, 190 + 360))
        self.assertEqual(result.coord('longitude').points[0], 170 + 360)
        self.assertEqual(result.coord('longitude').points[-1], 190 + 360)
        self.assertEqual(result.data[0, 0, 0], 170)
        self.assertEqual(result.data[0, 0, -1], 190)

    def test_wrap_after_modulus(self):
        cube = create_cube(-180, 180)
        result = cube.intersection(longitude=(170 + 360, 190 + 360))
        self.assertEqual(result.coord('longitude').points[0], 170 + 360)
        self.assertEqual(result.coord('longitude').points[-1], 190 + 360)
        self.assertEqual(result.data[0, 0, 0], 350)
        self.assertEqual(result.data[0, 0, -1], 10)

    def test_select_by_coord(self):
        cube = create_cube(0, 360)
        coord = iris.coords.DimCoord(0, 'longitude', units='degrees')
        result = cube.intersection(iris.coords.CoordExtent(coord, 10, 30))
        self.assertEqual(result.coord('longitude').points[0], 10)
        self.assertEqual(result.coord('longitude').points[-1], 30)
        self.assertEqual(result.data[0, 0, 0], 10)
        self.assertEqual(result.data[0, 0, -1], 30)

    def test_inclusive_exclusive(self):
        cube = create_cube(0, 360)
        result = cube.intersection(longitude=(170, 190, True, False))
        self.assertEqual(result.coord('longitude').points[0], 170)
        self.assertEqual(result.coord('longitude').points[-1], 189)
        self.assertEqual(result.data[0, 0, 0], 170)
        self.assertEqual(result.data[0, 0, -1], 189)

    def test_exclusive_inclusive(self):
        cube = create_cube(0, 360)
        result = cube.intersection(longitude=(170, 190, False))
        self.assertEqual(result.coord('longitude').points[0], 171)
        self.assertEqual(result.coord('longitude').points[-1], 190)
        self.assertEqual(result.data[0, 0, 0], 171)
        self.assertEqual(result.data[0, 0, -1], 190)

    def test_exclusive_exclusive(self):
        cube = create_cube(0, 360)
        result = cube.intersection(longitude=(170, 190, False, False))
        self.assertEqual(result.coord('longitude').points[0], 171)
        self.assertEqual(result.coord('longitude').points[-1], 189)
        self.assertEqual(result.data[0, 0, 0], 171)
        self.assertEqual(result.data[0, 0, -1], 189)

    def test_single_point(self):
        # 10 <= v <= 10
        cube = create_cube(0, 360)
        result = cube.intersection(longitude=(10, 10))
        self.assertEqual(result.coord('longitude').points[0], 10)
        self.assertEqual(result.coord('longitude').points[-1], 10)
        self.assertEqual(result.data[0, 0, 0], 10)
        self.assertEqual(result.data[0, 0, -1], 10)

    def test_two_points(self):
        # -1.5 <= v <= 0.5
        cube = create_cube(0, 360)
        result = cube.intersection(longitude=(-1.5, 0.5))
        self.assertEqual(result.coord('longitude').points[0], -1)
        self.assertEqual(result.coord('longitude').points[-1], 0)
        self.assertEqual(result.data[0, 0, 0], 359)
        self.assertEqual(result.data[0, 0, -1], 0)

    def test_wrap_radians(self):
        cube = create_cube(0, 360)
        cube.coord('longitude').convert_units('radians')
        result = cube.intersection(longitude=(-1, 0.5))
        self.assertEqual(result.coord('longitude').points[0],
                         -0.99483767363676634)
        self.assertEqual(result.coord('longitude').points[-1],
                         0.48869219055841207)
        self.assertEqual(result.data[0, 0, 0], 303)
        self.assertEqual(result.data[0, 0, -1], 28)

    def test_tolerance_bug(self):
        # Floating point changes introduced by wrapping mean
        # the resulting coordinate values are not equal to their
        # equivalents. This led to a bug that this test checks.
        cube = create_cube(0, 400)
        cube.coord('longitude').points = np.linspace(-179.55, 179.55, 400)
        result = cube.intersection(longitude=(125, 145))
        self.assertArrayAlmostEqual(result.coord('longitude').points,
                                    cube.coord('longitude').points[339:361])

    def test_tolerance_bug_wrapped(self):
        cube = create_cube(0, 400)
        cube.coord('longitude').points = np.linspace(-179.55, 179.55, 400)
        result = cube.intersection(longitude=(-190, -170))
        # Expected result is the last 11 and first 11 points.
        expected = np.append(cube.coord('longitude').points[389:] - 360.,
                             cube.coord('longitude').points[:11])
        self.assertArrayAlmostEqual(result.coord('longitude').points,
                                    expected)


# Check what happens with a global, points-and-bounds circular
# intersection coordinate.
class Test_intersection__ModulusBounds(tests.IrisTest):
    def test_global_wrapped_extreme_increasing_base_period(self):
        # Ensure that we can correctly handle bounds defined at (base + period)
        cube = create_cube(-180., 180., bounds=True)
        lons = cube.coord('longitude')
        result = cube.intersection(longitude=(lons.bounds.min(),
                                              lons.bounds.max()))
        self.assertArrayEqual(result.data, cube.data)

    def test_global_wrapped_extreme_decreasing_base_period(self):
        # Ensure that we can correctly handle bounds defined at (base + period)
        cube = create_cube(180., -180., bounds=True)
        lons = cube.coord('longitude')
        result = cube.intersection(longitude=(lons.bounds.min(),
                                              lons.bounds.max()))
        self.assertArrayEqual(result.data, cube.data)

    def test_misaligned_points_inside(self):
        cube = create_cube(0, 360, bounds=True)
        result = cube.intersection(longitude=(169.75, 190.25))
        self.assertArrayEqual(result.coord('longitude').bounds[0],
                              [169.5, 170.5])
        self.assertArrayEqual(result.coord('longitude').bounds[-1],
                              [189.5, 190.5])
        self.assertEqual(result.data[0, 0, 0], 170)
        self.assertEqual(result.data[0, 0, -1], 190)

    def test_misaligned_points_outside(self):
        cube = create_cube(0, 360, bounds=True)
        result = cube.intersection(longitude=(170.25, 189.75))
        self.assertArrayEqual(result.coord('longitude').bounds[0],
                              [169.5, 170.5])
        self.assertArrayEqual(result.coord('longitude').bounds[-1],
                              [189.5, 190.5])
        self.assertEqual(result.data[0, 0, 0], 170)
        self.assertEqual(result.data[0, 0, -1], 190)

    def test_misaligned_bounds(self):
        cube = create_cube(-180, 180, bounds=True)
        result = cube.intersection(longitude=(0, 360))
        self.assertArrayEqual(result.coord('longitude').bounds[0],
                              [-0.5,  0.5])
        self.assertArrayEqual(result.coord('longitude').bounds[-1],
                              [358.5,  359.5])
        self.assertEqual(result.data[0, 0, 0], 180)
        self.assertEqual(result.data[0, 0, -1], 179)

    def test_misaligned_bounds_decreasing(self):
        cube = create_cube(180, -180, bounds=True)
        result = cube.intersection(longitude=(0, 360))
        self.assertArrayEqual(result.coord('longitude').bounds[0],
                              [359.5,  358.5])
        self.assertArrayEqual(result.coord('longitude').points[-1], 0)
        self.assertArrayEqual(result.coord('longitude').bounds[-1],
                              [0.5, -0.5])
        self.assertEqual(result.data[0, 0, 0], 181)
        self.assertEqual(result.data[0, 0, -1], 180)

    def test_aligned_inclusive(self):
        cube = create_cube(0, 360, bounds=True)
        result = cube.intersection(longitude=(170.5, 189.5))
        self.assertArrayEqual(result.coord('longitude').bounds[0],
                              [169.5, 170.5])
        self.assertArrayEqual(result.coord('longitude').bounds[-1],
                              [189.5, 190.5])
        self.assertEqual(result.data[0, 0, 0], 170)
        self.assertEqual(result.data[0, 0, -1], 190)

    def test_aligned_exclusive(self):
        cube = create_cube(0, 360, bounds=True)
        result = cube.intersection(longitude=(170.5, 189.5, False, False))
        self.assertArrayEqual(result.coord('longitude').bounds[0],
                              [170.5, 171.5])
        self.assertArrayEqual(result.coord('longitude').bounds[-1],
                              [188.5, 189.5])
        self.assertEqual(result.data[0, 0, 0], 171)
        self.assertEqual(result.data[0, 0, -1], 189)

    def test_negative_misaligned_points_inside(self):
        cube = create_cube(0, 360, bounds=True)
        result = cube.intersection(longitude=(-10.25, 10.25))
        self.assertArrayEqual(result.coord('longitude').bounds[0],
                              [-10.5, -9.5])
        self.assertArrayEqual(result.coord('longitude').bounds[-1],
                              [9.5, 10.5])
        self.assertEqual(result.data[0, 0, 0], 350)
        self.assertEqual(result.data[0, 0, -1], 10)

    def test_negative_misaligned_points_outside(self):
        cube = create_cube(0, 360, bounds=True)
        result = cube.intersection(longitude=(-9.75, 9.75))
        self.assertArrayEqual(result.coord('longitude').bounds[0],
                              [-10.5, -9.5])
        self.assertArrayEqual(result.coord('longitude').bounds[-1],
                              [9.5, 10.5])
        self.assertEqual(result.data[0, 0, 0], 350)
        self.assertEqual(result.data[0, 0, -1], 10)

    def test_negative_aligned_inclusive(self):
        cube = create_cube(0, 360, bounds=True)
        result = cube.intersection(longitude=(-10.5, 10.5))
        self.assertArrayEqual(result.coord('longitude').bounds[0],
                              [-11.5, -10.5])
        self.assertArrayEqual(result.coord('longitude').bounds[-1],
                              [10.5, 11.5])
        self.assertEqual(result.data[0, 0, 0], 349)
        self.assertEqual(result.data[0, 0, -1], 11)

    def test_negative_aligned_exclusive(self):
        cube = create_cube(0, 360, bounds=True)
        result = cube.intersection(longitude=(-10.5, 10.5, False, False))
        self.assertArrayEqual(result.coord('longitude').bounds[0],
                              [-10.5, -9.5])
        self.assertArrayEqual(result.coord('longitude').bounds[-1],
                              [9.5, 10.5])
        self.assertEqual(result.data[0, 0, 0], 350)
        self.assertEqual(result.data[0, 0, -1], 10)

    def test_decrementing(self):
        cube = create_cube(360, 0, bounds=True)
        result = cube.intersection(longitude=(40, 60))
        self.assertArrayEqual(result.coord('longitude').bounds[0],
                              [60.5, 59.5])
        self.assertArrayEqual(result.coord('longitude').bounds[-1],
                              [40.5, 39.5])
        self.assertEqual(result.data[0, 0, 0], 300)
        self.assertEqual(result.data[0, 0, -1], 320)

    def test_decrementing_wrapped(self):
        cube = create_cube(360, 0, bounds=True)
        result = cube.intersection(longitude=(-10, 10))
        self.assertArrayEqual(result.coord('longitude').bounds[0],
                              [10.5, 9.5])
        self.assertArrayEqual(result.coord('longitude').bounds[-1],
                              [-9.5, -10.5])
        self.assertEqual(result.data[0, 0, 0], 350)
        self.assertEqual(result.data[0, 0, -1], 10)

    def test_numerical_tolerance(self):
        # test the tolerance on the coordinate value is not causing a
        # modulus wrapping
        cube = create_cube(28.5, 68.5, bounds=True)
        result = cube.intersection(longitude=(27.74, 68.61))
        self.assertAlmostEqual(result.coord('longitude').points[0], 28.5)
        self.assertAlmostEqual(result.coord('longitude').points[-1], 67.5)


def unrolled_cube():
    data = np.arange(5, dtype='f4')
    cube = Cube(data)
    cube.add_aux_coord(iris.coords.AuxCoord([5.0, 10.0, 8.0, 5.0, 3.0],
                                            'longitude', units='degrees'), 0)
    cube.add_aux_coord(iris.coords.AuxCoord([1.0, 3.0, -2.0, -1.0, -4.0],
                                            'latitude'), 0)
    return cube


# Check what happens with a "unrolled" scatter-point data with a circular
# intersection coordinate.
class Test_intersection__ScatterModulus(tests.IrisTest):
    def test_subset(self):
        cube = unrolled_cube()
        result = cube.intersection(longitude=(5, 8))
        self.assertArrayEqual(result.coord('longitude').points, [5, 8, 5])
        self.assertArrayEqual(result.data, [0, 2, 3])

    def test_subset_wrapped(self):
        cube = unrolled_cube()
        result = cube.intersection(longitude=(5 + 360, 8 + 360))
        self.assertArrayEqual(result.coord('longitude').points,
                              [365, 368, 365])
        self.assertArrayEqual(result.data, [0, 2, 3])

    def test_superset(self):
        cube = unrolled_cube()
        result = cube.intersection(longitude=(0, 15))
        self.assertArrayEqual(result.coord('longitude').points,
                              [5, 10, 8, 5, 3])
        self.assertArrayEqual(result.data, np.arange(5))


# Test the API of the cube interpolation method.
class Test_interpolate(tests.IrisTest):
    def setUp(self):
        self.cube = stock.simple_2d()

        self.scheme = mock.Mock(name='interpolation scheme')
        self.interpolator = mock.Mock(name='interpolator')
        self.interpolator.return_value = mock.sentinel.RESULT
        self.scheme.interpolator.return_value = self.interpolator
        self.collapse_coord = True

    def test_api(self):
        sample_points = (('foo', 0.5), ('bar', 0.6))
        result = self.cube.interpolate(sample_points, self.scheme,
                                       self.collapse_coord)
        self.scheme.interpolator.assert_called_once_with(
            self.cube, ('foo', 'bar'))
        self.interpolator.assert_called_once_with(
            (0.5, 0.6), collapse_scalar=self.collapse_coord)
        self.assertIs(result, mock.sentinel.RESULT)


class Test_regrid(tests.IrisTest):
    def test(self):
        # Test that Cube.regrid() just defers to the regridder of the
        # given scheme.

        # Define a fake scheme and its associated regridder which just
        # capture their arguments and return them in place of the
        # regridded cube.
        class FakeRegridder(object):
            def __init__(self, *args):
                self.args = args

            def __call__(self, cube):
                return self.args + (cube,)

        class FakeScheme(object):
            def regridder(self, src, target):
                return FakeRegridder(self, src, target)

        cube = Cube(0)
        scheme = FakeScheme()
        result = cube.regrid(mock.sentinel.TARGET, scheme)
        self.assertEqual(result, (scheme, cube, mock.sentinel.TARGET, cube))


class Test_copy(tests.IrisTest):
    def _check_copy(self, cube, cube_copy):
        self.assertIsNot(cube_copy, cube)
        self.assertEqual(cube_copy, cube)
        self.assertIsNot(cube_copy.data, cube.data)
        if isinstance(cube.data, np.ma.MaskedArray):
            self.assertMaskedArrayEqual(cube_copy.data, cube.data)
            if cube.data.mask is not ma.nomask:
                # "No mask" is a constant : all other cases must be distinct.
                self.assertIsNot(cube_copy.data.mask, cube.data.mask)
        else:
            self.assertArrayEqual(cube_copy.data, cube.data)

    def test(self):
        cube = stock.simple_3d()
        self._check_copy(cube, cube.copy())

    def test__masked_emptymask(self):
        cube = Cube(np.ma.array([0, 1]))
        self._check_copy(cube, cube.copy())

    def test__masked_arraymask(self):
        cube = Cube(np.ma.array([0, 1], mask=[True, False]))
        self._check_copy(cube, cube.copy())

    def test__scalar(self):
        cube = Cube(0)
        self._check_copy(cube, cube.copy())

    def test__masked_scalar_emptymask(self):
        cube = Cube(np.ma.array(0))
        self._check_copy(cube, cube.copy())

    def test__masked_scalar_arraymask(self):
        cube = Cube(np.ma.array(0, mask=False))
        self._check_copy(cube, cube.copy())

    def test__lazy(self):
        cube = Cube(biggus.NumpyArrayAdapter(np.array([1, 0])))
        self._check_copy(cube, cube.copy())


class Test_dtype(tests.IrisTest):
    def test_int8(self):
        cube = Cube(np.array([0, 1], dtype=np.int8))
        self.assertEqual(cube.dtype, np.int8)

    def test_float32(self):
        cube = Cube(np.array([0, 1], dtype=np.float32))
        self.assertEqual(cube.dtype, np.float32)

    def test_lazy(self):
        data = np.arange(6, dtype=np.float32).reshape(2, 3)
        lazydata = biggus.NumpyArrayAdapter(data)
        cube = Cube(lazydata)
        self.assertEqual(cube.dtype, np.float32)
        # Check that accessing the dtype does not trigger loading of the data.
        self.assertTrue(cube.has_lazy_data())


class TestSubset(tests.IrisTest):
    def test_scalar_coordinate(self):
        cube = Cube(0, long_name='apricot', units='1')
        cube.add_aux_coord(DimCoord([0], long_name='banana', units='1'))
        result = cube.subset(cube.coord('banana'))
        self.assertEqual(cube, result)

    def test_dimensional_coordinate(self):
        cube = Cube(np.zeros((4)), long_name='tinned_peach', units='1')
        cube.add_dim_coord(DimCoord([0, 1, 2, 3],
                                    long_name='sixteen_ton_weight',
                                    units='1'),
                           0)
        result = cube.subset(cube.coord('sixteen_ton_weight'))
        self.assertEqual(cube, result)

    def test_missing_coordinate(self):
        cube = Cube(0, long_name='raspberry', units='1')
        cube.add_aux_coord(DimCoord([0], long_name='loganberry', units='1'))
        bad_coord = DimCoord([0], long_name='tiger', units='1')
        self.assertRaises(CoordinateNotFoundError, cube.subset, bad_coord)

    def test_different_coordinate(self):
        cube = Cube(0, long_name='raspberry', units='1')
        cube.add_aux_coord(DimCoord([0], long_name='loganberry', units='1'))
        different_coord = DimCoord([2], long_name='loganberry', units='1')
        result = cube.subset(different_coord)
        self.assertEqual(result, None)

    def test_not_coordinate(self):
        cube = Cube(0, long_name='peach', units='1')
        cube.add_aux_coord(DimCoord([0], long_name='crocodile', units='1'))
        self.assertRaises(ValueError, cube.subset, 'Pointed Stick')


class Test_add_metadata(tests.IrisTest):
    def test_add_dim_coord(self):
        cube = Cube(np.arange(3))
        x_coord = DimCoord(points=np.array([2, 3, 4]),
                           long_name='x')
        cube.add_dim_coord(x_coord, 0)
        self.assertEqual(cube.coord('x'), x_coord)

    def test_add_aux_coord(self):
        cube = Cube(np.arange(6).reshape(2, 3))
        x_coord = AuxCoord(points=np.arange(6).reshape(2, 3),
                           long_name='x')
        cube.add_aux_coord(x_coord, [0, 1])
        self.assertEqual(cube.coord('x'), x_coord)

    def test_add_cell_measure(self):
        cube = Cube(np.arange(6).reshape(2, 3))
        a_cell_measure = CellMeasure(data=np.arange(6).reshape(2, 3),
                                     long_name='area', measure='area')
        cube.add_cell_measure(a_cell_measure, [0, 1])
        self.assertEqual(cube.cell_measure('area'), a_cell_measure)


class Test_remove_metadata(tests.IrisTest):
    def setUp(self):
        cube = Cube(np.arange(6).reshape(2, 3))
        x_coord = DimCoord(points=np.array([2, 3, 4]),
                           long_name='x')
        cube.add_dim_coord(x_coord, 1)
        z_coord = AuxCoord(points=np.arange(6).reshape(2, 3),
                           long_name='z')
        cube.add_aux_coord(z_coord, [0, 1])
        a_cell_measure = CellMeasure(data=np.arange(6).reshape(2, 3),
                                     long_name='area', measure='area')
        self.b_cell_measure = CellMeasure(data=np.arange(6).reshape(2, 3),
                                          long_name='other_area',
                                          measure='area')
        cube.add_cell_measure(a_cell_measure, [0, 1])
        cube.add_cell_measure(self.b_cell_measure, [0, 1])
        self.cube = cube

    def test_remove_dim_coord(self):
        self.cube.remove_coord(self.cube.coord('x'))
        self.assertEqual(self.cube.coords('x'), [])

    def test_remove_aux_coord(self):
        self.cube.remove_coord(self.cube.coord('z'))
        self.assertEqual(self.cube.coords('z'), [])

    def test_remove_cell_measure(self):
        self.cube.remove_cell_measure(self.cube.cell_measure('area'))
        self.assertEqual(self.cube._cell_measures_and_dims,
                         [[self.b_cell_measure, (0, 1)]])


class Test_share_data(tests.IrisTest):
    def setter_lazy_data(self):
        cube = Cube(biggus.NumpyArrayAdapter(np.arange(6).reshape(2, 3)))
        cube.share_data = True
        self.assertFalse(cube.has_lazy_data())
        self.assertTrue(cube._share_data)

    def setter_realised_data(self):
        cube = Cube(np.arange(6).reshape(2, 3))
        cube.share_data = True
        self.assertFalse(cube.has_lazy_data())
        self.assertTrue(cube._share_data)


class Test___getitem__no_share_data(tests.IrisTest):
    def test_lazy_array(self):
        cube = Cube(biggus.NumpyArrayAdapter(np.arange(6).reshape(2, 3)))
        cube2 = cube[1:]
        self.assertTrue(cube2.has_lazy_data())
        cube.data
        self.assertTrue(cube2.has_lazy_data())

    def test_ndarray(self):
        cube = Cube(np.arange(6).reshape(2, 3))
        cube2 = cube[1:]
        self.assertIsNot(cube.data.base, cube2.data.base)


class Test___getitem__share_data(tests.IrisTest):
    def test_lazy_array(self):
        cube = Cube(biggus.NumpyArrayAdapter(np.arange(6).reshape(2, 3)))
        cube.share_data = True
        cube2 = cube[1:]
        self.assertFalse(cube.has_lazy_data())
        self.assertFalse(cube2.has_lazy_data())
        self.assertIs(cube.data.base, cube2.data.base)

    def test_ndarray(self):
        cube = Cube(np.arange(6).reshape(2, 3))
        cube.share_data = True
        cube2 = cube[1:]
        self.assertIs(cube.data.base, cube2.data.base)


class Test__getitem_CellMeasure(tests.IrisTest):
    def setUp(self):
        cube = Cube(np.arange(6).reshape(2, 3))
        x_coord = DimCoord(points=np.array([2, 3, 4]),
                           long_name='x')
        cube.add_dim_coord(x_coord, 1)
        y_coord = DimCoord(points=np.array([5, 6]),
                           long_name='y')
        cube.add_dim_coord(y_coord, 0)
        z_coord = AuxCoord(points=np.arange(6).reshape(2, 3),
                           long_name='z')
        cube.add_aux_coord(z_coord, [0, 1])
        a_cell_measure = CellMeasure(data=np.arange(6).reshape(2, 3),
                                     long_name='area', measure='area')
        cube.add_cell_measure(a_cell_measure, [0, 1])
        self.cube = cube

    def test_cell_measure_2d(self):
        result = self.cube[0:2, 0:2]
        self.assertEqual(len(result.cell_measures()), 1)
        self.assertEqual(result.shape,
                         result.cell_measures()[0].data.shape)

    def test_cell_measure_1d(self):
        result = self.cube[0, 0:2]
        self.assertEqual(len(result.cell_measures()), 1)
        self.assertEqual(result.shape,
                         result.cell_measures()[0].data.shape)


class TestCellMeasures(tests.IrisTest):
    def setUp(self):
        cube = Cube(np.arange(6).reshape(2, 3))
        x_coord = DimCoord(points=np.array([2, 3, 4]),
                           long_name='x')
        cube.add_dim_coord(x_coord, 1)
        z_coord = AuxCoord(points=np.arange(6).reshape(2, 3),
                           long_name='z')
        cube.add_aux_coord(z_coord, [0, 1])
        self.a_cell_measure = CellMeasure(data=np.arange(6).reshape(2, 3),
                                          long_name='area', measure='area',
                                          units='m2')
        cube.add_cell_measure(self.a_cell_measure, [0, 1])
        self.cube = cube

    def test_get_cell_measure(self):
        cm = self.cube.cell_measure('area')
        self.assertEqual(cm, self.a_cell_measure)

    def test_get_cell_measures(self):
        cms = self.cube.cell_measures()
        self.assertEqual(len(cms), 1)
        self.assertEqual(cms[0], self.a_cell_measure)

    def test_get_cell_measures_obj(self):
        cms = self.cube.cell_measures(self.a_cell_measure)
        self.assertEqual(len(cms), 1)
        self.assertEqual(cms[0], self.a_cell_measure)

    def test_fail_get_cell_measure(self):
        with self.assertRaises(CellMeasureNotFoundError):
            cm = self.cube.cell_measure('notarea')

    def test_fail_get_cell_measures_obj(self):
        a_cell_measure = self.a_cell_measure.copy()
        a_cell_measure.units = 'km2'
        with self.assertRaises(CellMeasureNotFoundError):
            cms = self.cube.cell_measure(a_cell_measure)

    def test_cell_measure_dims(self):
        cm_dims = self.cube.cell_measure_dims(self.a_cell_measure)
        self.assertEqual(cm_dims, (0, 1))

    def test_fail_cell_measure_dims(self):
        a_cell_measure = self.a_cell_measure.copy()
        a_cell_measure.units = 'km2'
        with self.assertRaises(CellMeasureNotFoundError):
            cm_dims = self.cube.cell_measure_dims(a_cell_measure)


class Test_transpose(tests.IrisTest):
    def test_lazy_data(self):
        data = np.arange(12).reshape(3, 4)
        cube = Cube(biggus.NumpyArrayAdapter(data))
        cube.transpose()
        self.assertTrue(cube.has_lazy_data())
        self.assertArrayEqual(data.T, cube.data)

    def test_not_lazy_data(self):
        data = np.arange(12).reshape(3, 4)
        cube = Cube(data)
        cube.transpose()
        self.assertFalse(cube.has_lazy_data())
        self.assertIs(data.base, cube.data.base)
        self.assertArrayEqual(data.T, cube.data)


if __name__ == '__main__':
    tests.main()
