"""Support for SleepIQ sensors."""
from __future__ import annotations

from homeassistant.components.sensor import SensorEntity
from homeassistant.core import HomeAssistant
from homeassistant.helpers.entity_platform import AddEntitiesCallback
from homeassistant.helpers.typing import ConfigType, DiscoveryInfoType

from . import SleepIQSensor
from .const import DOMAIN, SENSOR_TYPES, SIDES, SLEEP_NUMBER

ICON = "mdi:bed"


def setup_platform(
    hass: HomeAssistant,
    config: ConfigType,
    add_entities: AddEntitiesCallback,
    discovery_info: DiscoveryInfoType | None = None,
) -> None:
    """Set up the SleepIQ sensors."""
    if discovery_info is None:
        return

    data = hass.data[DOMAIN]
    data.update()

    dev = []
    for bed_id, bed in data.beds.items():
        for side in SIDES:
            if getattr(bed, side) is not None:
                dev.append(SleepNumberSensor(data, bed_id, side))
    add_entities(dev)


class SleepNumberSensor(SleepIQSensor, SensorEntity):
    """Implementation of a SleepIQ sensor."""

    def __init__(self, sleepiq_data, bed_id, side):
        """Initialize the sensor."""
        SleepIQSensor.__init__(self, sleepiq_data, bed_id, side)

        self._state = None
        self.type = SLEEP_NUMBER
        self._name = SENSOR_TYPES[self.type]

        self.update()

    @property
    def native_value(self):
        """Return the state of the sensor."""
        return self._state

    @property
    def icon(self):
        """Icon to use in the frontend, if any."""
        return ICON

    def update(self):
        """Get the latest data from SleepIQ and updates the states."""
        SleepIQSensor.update(self)
        self._state = self.side.sleep_number
