from importlib import import_module

from .nlp import print_tree
from .parser_en import ParserEN
from .text import edge_text


def create_parser(lang=None, parser_class=None, lemmas=False,
                  resolve_corefs=False, beta='repair', normalize=True,
                  post_process=True):
    """Creates and returns a parser (as an instanceof a subclass of Parser)
    for the language specified in the parameter. If parser_class is specified,
    then the parser specified by this class is instantiated instead. Throws
    exception if language is not implemented.

    Available parsers:
    'en' -- English

    Keyword argument:
    parser_class -- specify an external parser class.
    lemmas -- if True, lemma edges are generated by the parser.
    resolve_corefs -- if True, coreference resolution is performed.
    (default: False)
    beta -- beta stage mode, current options are 'strict' and 'repair'
    (default: 'repair')
    normalize -- perform hyperedge normalization (default: True)
    post_process -- perform hyperedge post-processing (default: True)
    """
    if not lang and not parser_class:
        raise RuntimeError(
            'Either "lang" or "parser_class" must be specified.')

    if parser_class:
        package = None
        if parser_class[0] == '.':
            parser_class = parser_class[1:]
            package = '.'
        path_parts = parser_class.split('.')
        module_name = '.'.join(path_parts[:-1])
        class_name = path_parts[-1]
        class_obj = getattr(import_module(module_name, package=package),
                            class_name)
        parser = class_obj(lemmas=lemmas, resolve_corefs=resolve_corefs,
                           beta=beta, normalize=normalize,
                           post_process=post_process)
        if lang and parser.lang != lang:
            raise RuntimeError(
                'Specified language and parser class do not match.')
        return parser
    elif lang == 'en':
        return ParserEN(lemmas=lemmas, resolve_corefs=resolve_corefs,
                        beta=beta, normalize=normalize,
                        post_process=post_process)
    else:
        raise RuntimeError('Unknown parser: {}'.format(lang))


def parser_lang(parser_class):
    package = None
    if parser_class[0] == '.':
        parser_class = parser_class[1:]
        package = '.'
    path_parts = parser_class.split('.')
    module_name = '.'.join(path_parts[:-1])
    return getattr(import_module(module_name, package=package), 'LANG')
