/*     The GPI core node library is licensed under
 * either the BSD 3-clause or the LGPL v. 3.
 *
 *     Under either license, the following additional term applies:
 *
 *         NO CLINICAL USE.  THE SOFTWARE IS NOT INTENDED FOR COMMERCIAL
 * PURPOSES AND SHOULD BE USED ONLY FOR NON-COMMERCIAL RESEARCH PURPOSES.  THE
 * SOFTWARE MAY NOT IN ANY EVENT BE USED FOR ANY CLINICAL OR DIAGNOSTIC
 * PURPOSES.  YOU ACKNOWLEDGE AND AGREE THAT THE SOFTWARE IS NOT INTENDED FOR
 * USE IN ANY HIGH RISK OR STRICT LIABILITY ACTIVITY, INCLUDING BUT NOT LIMITED
 * TO LIFE SUPPORT OR EMERGENCY MEDICAL OPERATIONS OR USES.  LICENSOR MAKES NO
 * WARRANTY AND HAS NOR LIABILITY ARISING FROM ANY USE OF THE SOFTWARE IN ANY
 * HIGH RISK OR STRICT LIABILITY ACTIVITIES.
 *
 *     If you elect to license the GPI core node library under the LGPL the
 * following applies:
 *
 *         This file is part of the GPI core node library.
 *
 *         The GPI core node library is free software: you can redistribute it
 * and/or modify it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the License,
 * or (at your option) any later version. GPI core node library is distributed
 * in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even
 * the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 *
 *         You should have received a copy of the GNU Lesser General Public
 * License along with the GPI core node library. If not, see
 * <http://www.gnu.org/licenses/>.
 */


/* spiralfill.c
 *
 * Author: Jim Pipe
 * Date: 2020 oct
 * Brief: takes coords generated by spiralgenmrtud_gen.c and rotates them to their final
 *          2D orientations. 
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>

/***
extern "C"
{
    #include "spiralgencf_gen.c"
}
***/

/* the base spiral sequence */

#ifndef MAX
#define UNDEFMAX
#define MAX(a,b) (((a)<(b))?(b):(a))
#endif

#ifndef MIN
#define UNDEFMIN
#define MIN(a,b) (((a)>(b))?(b):(a))
#endif

#define Phi_PI 3.141592653589793
void spiralfill(double* spparams, int maxarray, float* gxarray, float* gyarray, float* kxarray, float* kyarray,
                  Array<double> &garray, Array<double> &karray, Array<double> &ktmp,
                  double xdel, double ydel)
  {

  uint64_t i, j;
  double betastep, beta, cb, sb;

  double ix,iy;
  int fix, fiy;
  int ix1, ix2, ix3, ix4;
  int iy1, iy2, iy3, iy4;
  double delx, dely;
  double delx_sq, dely_sq;
  double dgrast;
  double a0,a1,a2;
  double f1,f2,f3,f4;

  dgrast = 1./SPGRAST;

  betastep = -2.*M_PI/(double)(garray.size(2)); //DHW match psd add -
  for(j=0;j<garray.size(2);j++) { // arms
    beta = (double)(j)*betastep;
    cb = cos(beta);
    sb = sin(beta);
    for(i=0;i<garray.size(1);i++) { // points
      garray(0,i,j) = cb*gxarray[i] - sb*gyarray[i];
      garray(1,i,j) = cb*gyarray[i] + sb*gxarray[i];
      } // i
    for(i=0;i<ktmp.size(1);i++) { // points
      ktmp(0,i,j) = cb*kxarray[i] - sb*kyarray[i];
      ktmp(1,i,j) = cb*kyarray[i] + sb*kxarray[i];
      } // i
    } // j
//*************************************************************************************
// interpolate ktmp to kout
//*************************************************************************************
// For interpoloating along kx:
// We first find ix where kout(i,...) = ktmp(ix,....).
// ix is a real number between ix2 = floor(ix) and ix3 = ceiling (ix)
// and delx = ix - ix2, i.e.
//
//                  delx
//                |<---->|
//    *           *      O    *           *
//   ix1         ix2     ix  ix3         ix4
//
// Now if
// f1 = ktmp(ix1,...)
// f2 = ktmp(ix2,...)
// f3 = ktmp(ix3,...)
// f4 = ktmp(ix4,...)
//
// We do a simple quadratic interopolation based on f1, f2, and f3, i.e.

// f(ix) = 0.5*(f1+f3-2f2)*delx^2 + 0.5*(f3-f1)*delx + f2;

// and another similar quadratic interopolation based on f2, f3, and f4.
// We then take their average. This average gives the coeffificents a0, a1, and a2
// as a function of f1, f2, f3, and f4.  (simple math :-)
// This interpolation has the nice
// feature that it passes exactly through f2 and f3
// It's only piecewise smooth, but I think that's OK.

  int64_t M = ktmp.size(1)-1; //DHW
  for(i=0;i<karray.size(1);i++) { // points
// Note these times are in usec
    ix = (((double)(i)*spparams[spDWELL])-xdel)*dgrast;
    iy = (((double)(i)*spparams[spDWELL])-ydel)*dgrast;

    fix = floor(ix);
    fiy = floor(iy);

    ix1 = MIN(MAX(0,fix - 1),M);
    ix2 = MIN(MAX(0,fix),M);
    ix3 = MIN(MAX(0,fix + 1),M);
    ix4 = MIN(MAX(0,fix + 2),M);
    delx = MAX(0.,ix-(double)(ix2));
    delx_sq = delx*delx;

    iy1 = MIN(MAX(0,fiy - 1),M);
    iy2 = MIN(MAX(0,fiy),M);
    iy3 = MIN(MAX(0,fiy + 1),M);
    iy4 = MIN(MAX(0,fiy + 2),M);
    dely = MAX(0.,iy-(double)(iy2));
    dely_sq = dely*dely;

    for(j=0;j<karray.size(2);j++) { // arms
// X first
        f1 = ktmp(0,ix1,j);
        f2 = ktmp(0,ix2,j);
        f3 = ktmp(0,ix3,j);
        f4 = ktmp(0,ix4,j);
        a2 = 0.25*(f1-f2-f3+f4);
        a1 = 0.25*(-f1 - 3.*f2 + 5.*f3 - f4);
        a0 = f2;
        karray(0,i,j) = a2*delx_sq + a1*delx + a0;
// Y second
        f1 = ktmp(1,iy1,j);
        f2 = ktmp(1,iy2,j);
        f3 = ktmp(1,iy3,j);
        f4 = ktmp(1,iy4,j);
        a2 = 0.25*(f1-f2-f3+f4);
        a1 = 0.25*(-f1 - 3.*f2 + 5.*f3 - f4);
        a0 = f2;
        karray(1,i,j) = a2*dely_sq + a1*dely + a0;
    } } // i j

  } // spiralfill


/* undo common macro names */
#ifdef UNDEFMAX
#undef MAX
#endif

#ifdef UNDEFMIN
#undef MIN
#endif
