
import os
import sys

this_dir = os.path.dirname(sys.argv[0])
pkg_dir = os.path.abspath(os.path.join(this_dir, '..'))
sys.path.insert(0, pkg_dir)

from numpy import *
from gala import imio, morpho, evaluate as ev
from scipy.ndimage import binary_dilation, generate_binary_structure
import getopt
import errno
import json
import argparse


larva_gt='/groups/flyem/proj/data/larva_fib_groundtruth_round2.h5'
medulla_gt='/groups/flyem/proj/data/larva_fib_groundtruth_round2.h5'

class Error(Exception):
    """Base-class for exceptions in this module."""

class UsageError(Error):
    def __init__(self, msg):
        self.msg = msg

def main(argv):
    parser = argparse.ArgumentParser(description='Perform VI accuracy comparison between two stacks')
    parser.add_argument('--stack1', type=str, help='Stack compared against base stack (stack path or h5)', dest='stack1', required=True)
    parser.add_argument('--stackbase', type=str, help='Base stack (stack path or h5)', dest='stackbase', required=True)
    parser.add_argument('--synapsejson', type=str, help='JSON file with synapse annotations', dest='synapsejson', default='')
    
    parser.add_argument('--relabel1', action='store_true', default=False, help='Relabel ids in stack 1 to prevent memory issues', dest='relabel1')
    parser.add_argument('--relabelbase', action='store_true', default=False, help='Relabel ids in stack base to prevent memory issues', dest='relabelbase')
    parser.add_argument('--filtersize', type=int, default=1000, help='Size below which comparisons are not made', dest='filtersize')
    parser.add_argument('--VIthres', type=float, default=0.02, help='VI difference below which is unimportant', dest='VIthres')
    parser.add_argument('--stack1_outjson', type=str, default='stack1_merged.json', help='output json file name for stack 1 (outputs over-merged bodies in stack 1)', dest='stack1_outjson')
    parser.add_argument('--stackbase_outjson', type=str, default='stackbase_merged.json', help='output json file name for stack base (outputs over-merged bodies in stack base)', dest='stackbase_outjson')

    parser.add_argument('--dilate1', type=int, default=0, help='Create and dilate edges on stack 1', dest='dilate1')
    parser.add_argument('--dilatebase', type=int, default=0, help='Create and dilate edges on base image', dest='dilatebase')

    args = parser.parse_args()


    filtersize = args.filtersize
    if args.synapsejson != '':
        filtersize = 0

    stack1=args.stack1
    stack1_int = None
    if stack1.endswith('.h5'):
        stack1_int = imio.read_image_stack(stack1)
    else: 
        stack1_int = imio.raveler_to_labeled_volume(stack1)
    if stack1_int.ndim == 4:
        stack1_int = stack1_int[:,:,:,0] + (stack1_int[:,:,:,1] << 8) + (stack1_int[:,:,:,2] << 16)

    stack1_int = morpho.remove_small_connected_components(stack1_int, min_size=filtersize, in_place=True)
   
    if args.dilate1 > 0:
        conn = 1 # the connectivity, can be in {1, 2, ..., seg.ndim}
        strel = generate_binary_structure(stack1_int.ndim, conn)        
        bound = morpho.seg_to_bdry(stack1_int)
        bound = binary_dilation(bound, strel, iterations=args.dilate1)
        stack1_int[bound] = 0
 
    if args.relabel1:
        stack1_int, dummy1, dummy2 = ev.relabel_from_one(stack1_int)
    print("Imported first stack")


    stackbase=args.stackbase
    stackbase_int = None
    if stackbase.endswith('.h5'):
        stackbase_int = imio.read_image_stack(stackbase)
    else:
        stackbase_int = imio.raveler_to_labeled_volume(stackbase)
    if stackbase_int.ndim == 4:
        stackbase_int = stackbase_int[:,:,:,0] + (stackbase_int[:,:,:,1] << 8) + (stackbase_int[:,:,:,2] << 16)
 
    stackbase_int = morpho.remove_small_connected_components(stackbase_int, min_size=filtersize, in_place=True)

    if args.dilatebase > 0:
        conn = 1 # the connectivity, can be in {1, 2, ..., seg.ndim}
        strel = generate_binary_structure(stackbase_int.ndim, conn)        
        bound = morpho.seg_to_bdry(stackbase_int)
        bound = binary_dilation(bound, strel, iterations=args.dilatebase)
        stackbase_int[bound] = 0

    if args.relabelbase:
        stackbase_int, dummy1, dummy2 = ev.relabel_from_one(stackbase_int)
    print("Imported base stack")
    
    if args.synapsejson != '':
        synaptic_vi, synaptic_comps = ev.make_synaptic_functions(args.synapsejson, [ev.split_vi_mem, ev.split_vi_mem])
        merge, split, stack1_bodies2, stack1_vis2, gt_bodies2, gt_vis2 = synaptic_vi(stack1_int, stackbase_int)
        print("SynMergeSplit: " + str((merge, split)))
    else:
        merge, split, stack1_bodies2, stack1_vis2, gt_bodies2, gt_vis2 = ev.split_vi_mem(stack1_int, stackbase_int)
        print("MergeSplit: " + str((merge, split)))
        
    data=[]
    database=[]
    accum = 0
    if merge > args.VIthres:
        for bodyid in stack1_bodies2:
            if accum > (merge - args.VIthres):
                break        
            accum += stack1_vis2[bodyid]
            data.append({'status' : 'not sure', 'body ID' : int(bodyid), 'vi contrib' : stack1_vis2[bodyid]})    

    accum = 0
    if split > args.VIthres:
        for bodyid in gt_bodies2:
            if accum > (split - args.VIthres):
                break        
            accum += gt_vis2[bodyid]
            database.append({'status' : 'not sure', 'body ID' : int(bodyid), 'vi contrib' : gt_vis2[bodyid]})    

    json_out_data = {'data' : data}
    json_out_data2 = {'data' : database}
    
    stack1_fp = open(args.stack1_outjson, "w")
    stackbase_fp = open(args.stackbase_outjson, "w")

    stack1_fp.write(json.dumps(json_out_data, sort_keys=False, indent=2)) 
    stackbase_fp.write(json.dumps(json_out_data2, sort_keys=False, indent=2)) 

sys.exit(main(sys.argv))
