# -*- coding: utf-8 -*-
"""
Improved logging facilities:

* info and debug messages are printed to stdout instead of stderr.
* if level is set to debug, will show source of each message.
* ability to filter messages with a regexp.
* provides a context manager to capture log messages in a list.

This code can be put anywhere in your project, then import 
set_log_level()
"""

# The code in this file is inspired by similar code from the Vispy project

from __future__ import print_function, absolute_import, with_statement, unicode_literals, division

import re
import sys
import logging


MODULE_NAME = __name__.split('.')[0]

logging_types = dict(debug=logging.DEBUG, info=logging.INFO,
                     warning=logging.WARNING, error=logging.ERROR,
                     critical=logging.CRITICAL)


class _Formatter(logging.Formatter):
    """Formatter that optionally prepends caller """
    
    def __init__(self):
        super(_Formatter, self).__init__('%(levelname)s %(name)s: %(message)s')
        self.prepend_caller = False
    
    def format(self, record):
        out = logging.Formatter.format(self, record)
        if self.prepend_caller:
            part1, part2 = out.split(':', 1)
            out = part1 + ' ' + record.funcName + '():' + part2
        return out


class _Handler(logging.StreamHandler):
    """ Stream handler that prints INFO and lower to stdout
    """
    
    def emit(self, record):
        if record.levelno >= logging.WARNING:
            self.stream = sys.stderr
        else:
            self.stream = sys.stdout
        super(_Handler, self).emit(record)


class _MatchFilter(object):
    """ To filter records on regexp matches.
    """
    def __init__(self):
        self.match = None
    
    def filter(self, record):
        match = self.match
        if not match:
            return True
        elif isinstance(match, basestring):
            return (match in record.name or
                    match in record.getMessage() or
                    match in record.funcName)
        else:
            return (re.search(match, record.name) or
                    re.search(match, record.getMessage()) or
                    re.search(match, record.funcName))


class _CaptureFilter(object):
    """ To collect records in the capture_log context.
    """
    def __init__(self):
        self.records = []
    
    def filter(self, record):
        self.records.append(_formatter.format(record))
        return False


def set_log_level(level, match=None):
    """Set the logging level and match filter

    Parameters:
        level (str, int): The verbosity of messages to print.
            If a str, it can be either DEBUG, INFO, WARNING, ERROR, or
            CRITICAL. Note that these are for convenience and are equivalent
            to passing in logging.DEBUG, etc.
        match (str, regexp, None): String to match. Only those messages
            that contain ``match`` as a substring (and has the
            appropriate ``level``) will be displayed. Match can also be
            a compiled regexp.
    
    Notes
    -----
    If level is DEBUG, the method emitting the log message will be
    prepended to each log message. Note that if ``level`` is DEBUG or
    if the ``match`` option is used, a small overhead is added to each
    logged message.
    """
    if isinstance(level, basestring):
        level = level.lower()
        if level not in logging_types:
            raise ValueError('Invalid argument "%s"' % level)
        level = logging_types[level]
    elif not isinstance(level, int):
        raise TypeError('log level must be an int or string')
    logger.setLevel(level)
    _filter.match = match
    _formatter.prepend_caller = level <= logging.DEBUG


class capture_log(object):
    """ Context manager to capture log messages. Useful for testing.
    Usage:
    
    .. code-block:: python
    
        with capture_log(level, match) as log:
            ...
        # log is a list strings (as they would have appeared in the console)
    """
    
    def __init__(self, level, match=None):
        self._args = level, match
    
    def __enter__(self):
        self._old_args = logger.level, _filter.match
        set_log_level(*self._args)
        self._filter = _CaptureFilter()
        _handler.addFilter(self._filter)
        return self._filter.records
    
    def __exit__(self, type, value, traceback):
        _handler.removeFilter(self._filter)
        set_log_level(*self._old_args)


# Create logger

logger = logging.getLogger(MODULE_NAME)
logger.propagate = False
logger.setLevel(logging.INFO)

_handler = _Handler()
_filter = _MatchFilter()
_formatter = _Formatter()

logger.addHandler(_handler)
_handler.addFilter(_filter)
_handler.setFormatter(_formatter)
