"""Dictionaries storing the generic beat-coefficients for simulating segmented
photoiodes.

Builtin pdtypes:
* SINGLE - single element photodiode
* XSPLIT - measuring horizontal/yaw motions
* YSPLIT - measuring vertical/pitch motions

Use the function `construct_segment_beat_matrix` to convert these dictionary definitions
into a dense or sparse CSR matrix for using in calculations.
"""
import numpy as np
import numbers
from finesse.cymath.sparsemath import CSRMatrix


def get_pdtype(pdtype: str) -> dict:
    """Returns a pdtype dictionary definition in this module for a given string name."""
    return globals()[pdtype.upper()] if isinstance(pdtype, str) else pdtype


def construct_segment_beat_matrix(
    mode_index_map, beats_dict, sparse_output=False, assert_hermitian=True
):
    """Takes a pdtype definition dict and converts it into a matrix suitable to use with
    a KnmMatrix.

    Parameters
    ----------
    mode_index_map : dict[(n,m), index]
        Dictionary of mode indices and index

    beats_dict : str, pdtype style dict
        A dictionary in the pdtype style describing a particular segmentation
        type for a sensor. If a string is given it will try and get a builtin
        pdtype dict

    sparse_output : bool, optional
        Return a sparse CSR matrix object rather than a dense one

    assert_hermitian : bool, optional
        If true then the generated matrix from the pdtype definition must be Hermitian

    Returns
    -------
    2D Complex matrix of size (N, N) with N=len(mode_index_map) or a
    CSRMatrix if requested.

    Examples
    --------
    >>> pdtypes.construct_segment_beat_matrix(model.mode_index_map, pdtype.YSPLIT)
    array([[0.        +0.j, 0.79788456+0.j, 0.        +0.j],
           [0.79788456+0.j, 0.        +0.j, 0.        +0.j],
           [0.        +0.j, 0.        +0.j, 0.        +0.j]])
    """
    # TODO ddb - don't need to generate the dense matrix first to then
    # convert it into a sparse representation, would be more efficient
    # to go straight to sparse.

    if isinstance(beats_dict, str):
        beats_dict = get_pdtype(beats_dict)

    N = len(mode_index_map)
    _beats = np.zeros((N, N), dtype=complex)
    maxtem = max(max(mode_index_map, key=lambda x: max(x)))

    for (n1, m1, n2, m2), factor in beats_dict.items():
        if all(isinstance(k, numbers.Number) for k in (n1, m1, n2, m2)):
            idx = mode_index_map.get((n1, m1), -1)
            jdx = mode_index_map.get((n2, m2), -1)
            if idx >= 0 and jdx >= 0:
                _beats[idx, jdx] = factor
                _beats[jdx, idx] = factor
        elif all(k in ("x", "y") for k in (m1, m2)) and all(
            isinstance(k, numbers.Number) for k in (n1, n2)
        ):
            for m in range(maxtem + 1):
                idx = mode_index_map.get((n1, m), -1)
                jdx = mode_index_map.get((n2, m), -1)
                if idx >= 0 and jdx >= 0:
                    _beats[idx, jdx] = factor
                    _beats[jdx, idx] = factor
        elif all(k in ("x", "y") for k in (n1, n2)) and all(
            isinstance(k, numbers.Number) for k in (m1, m2)
        ):
            for n in range(maxtem + 1):
                idx = mode_index_map.get((n, m1), -1)
                jdx = mode_index_map.get((n, m2), -1)
                if idx >= 0 and jdx >= 0:
                    _beats[idx, jdx] = factor
                    _beats[jdx, idx] = factor
        elif n1 == n2 and m1 == m2:
            for n in range(maxtem + 1):
                for m in range(maxtem + 1):
                    idx = mode_index_map.get((n, m), -1)
                    jdx = mode_index_map.get((n, m), -1)
                    if idx >= 0 and jdx >= 0:
                        _beats[idx, jdx] = factor
                        _beats[jdx, idx] = factor
        elif n1 == m2 and m1 == n2:
            for n in range(maxtem + 1):
                for m in range(maxtem + 1):
                    idx = mode_index_map.get((n, m), -1)
                    jdx = mode_index_map.get((m, n), -1)
                    if idx >= 0 and jdx >= 0:
                        _beats[idx, jdx] = factor
                        _beats[jdx, idx] = factor
        else:
            raise NotImplementedError()

    if assert_hermitian:
        if np.all(_beats - _beats.T.conj() != 0):
            raise Exception(f"Matrix generated by pdtype was not Hermitian:\n{_beats}")

    if sparse_output:
        return CSRMatrix(_beats)
    else:
        return _beats


SINGLE = {
    ("x", "y", "x", "y"): 1,
}

# split detector in x-direction, for up to TEM-order 40
XSPLIT = {
    (0, "x", 1, "x"): 0.797884560802865,
    (0, "x", 3, "x"): -0.32573500793528,
    (0, "x", 5, "x"): 0.218509686118416,
    (0, "x", 7, "x"): -0.168583882836184,
    (0, "x", 9, "x"): 0.139074607877595,
    (0, "x", 11, "x"): -0.119342192152758,
    (0, "x", 13, "x"): 0.105105246952602,
    (0, "x", 15, "x"): -0.0942883643366128,
    (0, "x", 17, "x"): 0.085756077667627,
    (0, "x", 19, "x"): -0.0788316791424408,
    (0, "x", 21, "x"): 0.0730852493544752,
    (0, "x", 23, "x"): -0.0682297483293215,
    (0, "x", 25, "x"): 0.0640657596627177,
    (0, "x", 27, "x"): -0.0604501569509354,
    (0, "x", 29, "x"): 0.0572773850389327,
    (0, "x", 31, "x"): -0.0544677839262821,
    (0, "x", 33, "x"): 0.051960035687153,
    (0, "x", 35, "x"): -0.0497061241097501,
    (0, "x", 37, "x"): 0.0476678793880178,
    (0, "x", 39, "x"): -0.0458145530435871,
    (2, "x", 1, "x"): 0.564189583547756,
    (2, "x", 3, "x"): 0.690988298942671,
    (2, "x", 5, "x"): -0.257516134682126,
    (2, "x", 7, "x"): 0.166889529453114,
    (2, "x", 9, "x"): -0.126437912127138,
    (2, "x", 11, "x"): 0.103140489653524,
    (2, "x", 13, "x"): -0.0878334751963756,
    (2, "x", 15, "x"): 0.0769291636262386,
    (2, "x", 17, "x"): -0.0687238645863045,
    (2, "x", 19, "x"): 0.0623003460579352,
    (2, "x", 21, "x"): -0.0571189780993917,
    (2, "x", 23, "x"): 0.0528405479815828,
    (2, "x", 25, "x"): -0.0492405794558427,
    (2, "x", 27, "x"): 0.0461642931761014,
    (2, "x", 29, "x"): -0.0435013182859334,
    (2, "x", 31, "x"): 0.0411707144994762,
    (2, "x", 33, "x"): -0.0391116996228281,
    (2, "x", 35, "x"): 0.0372776912078078,
    (2, "x", 37, "x"): -0.0356323539463384,
    (2, "x", 39, "x"): 0.0341469044387,
    (4, "x", 1, "x"): -0.16286750396764,
    (4, "x", 3, "x"): 0.598413420602149,
    (4, "x", 5, "x"): 0.669046543557289,
    (4, "x", 7, "x"): -0.240884286886713,
    (4, "x", 9, "x"): 0.153297821464992,
    (4, "x", 11, "x"): -0.114842936826923,
    (4, "x", 13, "x"): 0.092969581005561,
    (4, "x", 15, "x"): -0.0787358118089783,
    (4, "x", 17, "x"): 0.0686730145130556,
    (4, "x", 19, "x"): -0.0611475066641648,
    (4, "x", 21, "x"): 0.0552860726690191,
    (4, "x", 23, "x"): -0.0505782313126719,
    (4, "x", 25, "x"): 0.0467048872492379,
    (4, "x", 27, "x"): -0.0434559246068004,
    (4, "x", 29, "x"): 0.0406871064724294,
    (4, "x", 31, "x"): -0.0382959872150584,
    (4, "x", 33, "x"): 0.0362077065246626,
    (4, "x", 35, "x"): -0.0343662293597736,
    (4, "x", 37, "x"): 0.0327287372726021,
    (4, "x", 39, "x"): -0.031261920230486,
    (6, "x", 1, "x"): 0.0892062058076385,
    (6, "x", 3, "x"): -0.18209140509868,
    (6, "x", 5, "x"): 0.61075313987865,
    (6, "x", 7, "x"): 0.659688788381993,
    (6, "x", 9, "x"): -0.233235207868822,
    (6, "x", 11, "x"): 0.146771489830575,
    (6, "x", 13, "x"): -0.109117578596892,
    (6, "x", 15, "x"): 0.0878479967266388,
    (6, "x", 17, "x"): -0.0740877073983095,
    (6, "x", 19, "x"): 0.0644074398753087,
    (6, "x", 21, "x"): -0.0571982549981952,
    (6, "x", 23, "x"): 0.0516033260749633,
    (6, "x", 25, "x"): -0.0471234847486845,
    (6, "x", 27, "x"): 0.043447712204836,
    (6, "x", 29, "x"): -0.0403718224901522,
    (6, "x", 31, "x"): 0.0377560369073105,
    (6, "x", 33, "x"): -0.0355013303057689,
    (6, "x", 35, "x"): 0.0335355132263835,
    (6, "x", 37, "x"): -0.031804668469598,
    (6, "x", 39, "x"): 0.0302676798757954,
    (8, "x", 1, "x"): -0.059603403376112,
    (8, "x", 3, "x"): 0.102198547643328,
    (8, "x", 5, "x"): -0.190435749776861,
    (8, "x", 7, "x"): 0.617082357005358,
    (8, "x", 9, "x"): 0.6545146787836,
    (8, "x", 11, "x"): -0.228820262121828,
    (8, "x", 13, "x"): 0.142898207894381,
    (8, "x", 15, "x"): -0.10565264117106,
    (8, "x", 17, "x"): 0.0847033053948983,
    (8, "x", 19, "x"): -0.0712017602555549,
    (8, "x", 21, "x"): 0.0617354634590125,
    (8, "x", 23, "x"): -0.0547065570873058,
    (8, "x", 25, "x"): 0.0492658641354183,
    (8, "x", 27, "x"): -0.0449196778881935,
    (8, "x", 29, "x"): 0.0413609897651131,
    (8, "x", 31, "x"): -0.0383886289102033,
    (8, "x", 33, "x"): 0.0358651299902981,
    (8, "x", 35, "x"): -0.0336932742842578,
    (8, "x", 37, "x"): 0.0318023047767947,
    (8, "x", 39, "x"): -0.030139455674167,
    (10, "x", 1, "x"): 0.0439792525587995,
    (10, "x", 3, "x"): -0.0692528965958894,
    (10, "x", 5, "x"): 0.108397929099021,
    (10, "x", 7, "x"): -0.195138575204209,
    (10, "x", 9, "x"): 0.620927144090899,
    (10, "x", 11, "x"): 0.651233882791557,
    (10, "x", 13, "x"): -0.225941905251249,
    (10, "x", 15, "x"): 0.140323254503473,
    (10, "x", 17, "x"): -0.103315642865758,
    (10, "x", 19, "x"): 0.0825585698009622,
    (10, "x", 21, "x"): -0.0692160218229681,
    (10, "x", 23, "x"): 0.0598836888491238,
    (10, "x", 25, "x"): -0.0529693961318701,
    (10, "x", 27, "x"): 0.0476280244212015,
    (10, "x", 29, "x"): -0.0433688465158433,
    (10, "x", 31, "x"): 0.0398870940423755,
    (10, "x", 33, "x"): -0.0369833150045722,
    (10, "x", 35, "x"): 0.0345213862957191,
    (10, "x", 37, "x"): -0.0324051535576467,
    (10, "x", 39, "x"): 0.0305647153283757,
    (12, "x", 1, "x"): -0.0344511233825375,
    (12, "x", 3, "x"): 0.051570244826762,
    (12, "x", 5, "x"): -0.0741307969600372,
    (12, "x", 7, "x"): 0.112098577001675,
    (12, "x", 9, "x"): -0.198164159898117,
    (12, "x", 11, "x"): 0.623508972874051,
    (12, "x", 13, "x"): 0.648968714596981,
    (12, "x", 15, "x"): -0.223915479517027,
    (12, "x", 17, "x"): 0.13848407598893,
    (12, "x", 19, "x"): -0.101627759192188,
    (12, "x", 21, "x"): 0.0809958057142225,
    (12, "x", 23, "x"): -0.0677586792014329,
    (12, "x", 25, "x"): 0.058516541236311,
    (12, "x", 27, "x"): -0.0516804098923711,
    (12, "x", 29, "x"): 0.0464075103789091,
    (12, "x", 31, "x"): -0.0422088782527226,
    (12, "x", 33, "x"): 0.0387810976838852,
    (12, "x", 35, "x"): -0.0359257728661409,
    (12, "x", 37, "x"): 0.0335076182428655,
    (12, "x", 39, "x"): -0.031431152699668,
    (14, "x", 1, "x"): 0.0280905588320645,
    (14, "x", 3, "x"): -0.0406589983848392,
    (14, "x", 5, "x"): 0.0555599515078389,
    (14, "x", 7, "x"): -0.0771577797725184,
    (14, "x", 9, "x"): 0.114573457526737,
    (14, "x", 11, "x"): -0.200276093365814,
    (14, "x", 13, "x"): 0.625361901098279,
    (14, "x", 15, "x"): 0.647311065109173,
    (14, "x", 17, "x"): -0.222410991173016,
    (14, "x", 19, "x"): 0.137103342840115,
    (14, "x", 21, "x"): -0.100349371709367,
    (14, "x", 23, "x"): 0.0798036519532965,
    (14, "x", 25, "x"): -0.0666403046237177,
    (14, "x", 27, "x"): 0.0574621027762285,
    (14, "x", 29, "x"): -0.0506819820916291,
    (14, "x", 31, "x"): 0.0454586096116469,
    (14, "x", 33, "x"): -0.041304127205318,
    (14, "x", 35, "x"): 0.0379159829603672,
    (14, "x", 37, "x"): -0.0350964661555485,
    (14, "x", 39, "x"): 0.0327108419265415,
    (16, "x", 1, "x"): -0.0235720910841532,
    (16, "x", 3, "x"): 0.0333113049961062,
    (16, "x", 5, "x"): -0.0440146568671265,
    (16, "x", 7, "x"): 0.0581059881284756,
    (16, "x", 9, "x"): -0.0792394808782953,
    (16, "x", 11, "x"): 0.116349896137392,
    (16, "x", 13, "x"): -0.201834685692187,
    (16, "x", 15, "x"): 0.626756243745903,
    (16, "x", 17, "x"): 0.646045548619932,
    (16, "x", 19, "x"): -0.221249568137138,
    (16, "x", 21, "x"): 0.136028005901464,
    (16, "x", 23, "x"): -0.0993465691012448,
    (16, "x", 25, "x"): 0.0788629080536757,
    (16, "x", 27, "x"): -0.0657533402767068,
    (16, "x", 29, "x"): 0.0566222517097102,
    (16, "x", 31, "x"): -0.0498837907575016,
    (16, "x", 33, "x"): 0.04469755498707,
    (16, "x", 35, "x"): -0.0405764129180567,
    (16, "x", 37, "x"): 0.0372183888740113,
    (16, "x", 39, "x"): -0.034426235332043,
    (18, "x", 1, "x"): 0.0202129013489131,
    (18, "x", 3, "x"): -0.0280564002314289,
    (18, "x", 5, "x"): 0.0361938566085091,
    (18, "x", 7, "x"): -0.0462018029426718,
    (18, "x", 9, "x"): 0.0598942816336477,
    (18, "x", 11, "x"): -0.0807655532556404,
    (18, "x", 13, "x"): 0.117688834298236,
    (18, "x", 15, "x"): -0.203032528170905,
    (18, "x", 17, "x"): 0.627843419313627,
    (18, "x", 19, "x"): 0.645047794281474,
    (18, "x", 21, "x"): -0.220325766323103,
    (18, "x", 23, "x"): 0.135166515247699,
    (18, "x", 25, "x"): -0.0985383906746981,
    (18, "x", 27, "x"): 0.07810092971977,
    (18, "x", 29, "x"): -0.0650318354039826,
    (18, "x", 31, "x"): 0.055936535430677,
    (18, "x", 33, "x"): -0.0492299775566192,
    (18, "x", 35, "x"): 0.0440723835688313,
    (18, "x", 37, "x"): -0.0399771169972641,
    (18, "x", 39, "x"): 0.0366426000734971,
    (20, "x", 1, "x"): -0.017627299334295,
    (20, "x", 3, "x"): 0.0241288202745342,
    (20, "x", 5, "x"): -0.0305737533320824,
    (20, "x", 7, "x"): 0.0381039552927518,
    (20, "x", 9, "x"): -0.047763592814719,
    (20, "x", 11, "x"): 0.0612270740454677,
    (20, "x", 13, "x"): -0.0819349188989436,
    (20, "x", 15, "x"): 0.118734977843307,
    (20, "x", 17, "x"): -0.203982022959719,
    (20, "x", 19, "x"): 0.628714819343417,
    (20, "x", 21, "x"): 0.644240979964118,
    (20, "x", 23, "x"): -0.219573371070635,
    (20, "x", 25, "x"): 0.134460680056461,
    (20, "x", 27, "x"): -0.0978729043007541,
    (20, "x", 29, "x"): 0.0774707909003214,
    (20, "x", 31, "x"): -0.0644329526695553,
    (20, "x", 33, "x"): 0.0553655154050687,
    (20, "x", 35, "x"): -0.0486839720121728,
    (20, "x", 37, "x"): 0.0435489765855488,
    (20, "x", 39, "x"): -0.039474240161498,
    (22, "x", 1, "x"): 0.0155818275174025,
    (22, "x", 3, "x"): -0.0210925805324099,
    (22, "x", 5, "x"): 0.0263566009981115,
    (22, "x", 7, "x"): -0.0322641671658076,
    (22, "x", 9, "x"): 0.0394861351020212,
    (22, "x", 11, "x"): -0.0489431183977767,
    (22, "x", 13, "x"): 0.0622619711245261,
    (22, "x", 15, "x"): -0.082860766578907,
    (22, "x", 17, "x"): 0.119575292488569,
    (22, "x", 19, "x"): -0.204753224284902,
    (22, "x", 21, "x"): 0.629428862525661,
    (22, "x", 23, "x"): 0.643575097636349,
    (22, "x", 25, "x"): -0.218948694495961,
    (22, "x", 27, "x"): 0.133871712354208,
    (22, "x", 29, "x"): -0.0973152193947713,
    (22, "x", 31, "x"): 0.0769407678634122,
    (22, "x", 33, "x"): -0.0639275883754644,
    (22, "x", 35, "x"): 0.0548822891765173,
    (22, "x", 37, "x"): -0.0482207457593204,
    (22, "x", 39, "x"): 0.0431039197585667,
    (24, "x", 1, "x"): -0.0139273390571125,
    (24, "x", 3, "x"): 0.0186819548996952,
    (24, "x", 5, "x"): -0.0230856985072204,
    (24, "x", 7, "x"): 0.0278689812889355,
    (24, "x", 9, "x"): -0.0335007876120844,
    (24, "x", 11, "x"): 0.040541449001796,
    (24, "x", 13, "x"): -0.0498690376104398,
    (24, "x", 15, "x"): 0.0630903264429405,
    (24, "x", 17, "x"): -0.0836125971039459,
    (24, "x", 19, "x"): 0.120265288362839,
    (24, "x", 21, "x"): -0.205392081451324,
    (24, "x", 23, "x"): 0.630024630631138,
    (24, "x", 25, "x"): 0.643016196013222,
    (24, "x", 27, "x"): -0.218421741398637,
    (24, "x", 29, "x"): 0.133372747843751,
    (24, "x", 31, "x"): -0.0968410038781554,
    (24, "x", 33, "x"): 0.0764886144798256,
    (24, "x", 35, "x"): -0.0634952419012075,
    (24, "x", 37, "x"): 0.054467836681766,
    (24, "x", 39, "x"): -0.0478225494651678,
    (26, "x", 1, "x"): 0.012564329179744,
    (26, "x", 3, "x"): -0.0167261931797469,
    (26, "x", 5, "x"): 0.0204814479513874,
    (26, "x", 7, "x"): -0.0244511756080565,
    (26, "x", 9, "x"): 0.0289854926054433,
    (26, "x", 11, "x"): -0.0344536065949141,
    (26, "x", 13, "x"): 0.0413774431197777,
    (26, "x", 15, "x"): -0.0506169449451649,
    (26, "x", 17, "x"): 0.0637691420834689,
    (26, "x", 19, "x"): -0.0842355840892829,
    (26, "x", 21, "x"): 0.120842103606662,
    (26, "x", 23, "x"): -0.205929992668735,
    (26, "x", 25, "x"): 0.63052925596685,
    (26, "x", 27, "x"): 0.64254041629879,
    (26, "x", 29, "x"): -0.217971231998968,
    (26, "x", 31, "x"): 0.1329445838244,
    (26, "x", 33, "x"): -0.0964327593462296,
    (26, "x", 35, "x"): 0.0760982532525654,
    (26, "x", 37, "x"): -0.0631210368417068,
    (26, "x", 39, "x"): 0.0541083090782501,
    (28, "x", 1, "x"): -0.0114240058576427,
    (28, "x", 3, "x"): 0.015110811991687,
    (28, "x", 5, "x"): -0.0183634797872995,
    (28, "x", 7, "x"): 0.021723856102418,
    (28, "x", 9, "x"): -0.025467063571133,
    (28, "x", 11, "x"): 0.0298524441527061,
    (28, "x", 13, "x"): -0.0352142635297349,
    (28, "x", 15, "x"): 0.0420579527183231,
    (28, "x", 17, "x"): -0.0512345899753054,
    (28, "x", 19, "x"): 0.0643359900496595,
    (28, "x", 21, "x"): -0.0847604214666164,
    (28, "x", 23, "x"): 0.121331543048508,
    (28, "x", 25, "x"): -0.206389145953734,
    (28, "x", 27, "x"): 0.630962163162352,
    (28, "x", 29, "x"): 0.642130502115889,
    (28, "x", 31, "x"): -0.21758164950347,
    (28, "x", 33, "x"): 0.132573125753267,
    (28, "x", 35, "x"): -0.0960775730220675,
    (28, "x", 37, "x"): 0.0757577671900808,
    (28, "x", 39, "x"): -0.0627939055909658,
    (30, "x", 1, "x"): 0.0104573719402441,
    (30, "x", 3, "x"): -0.0137563247003482,
    (30, "x", 5, "x"): 0.0166104416612905,
    (30, "x", 7, "x"): -0.019501442987836,
    (30, "x", 9, "x"): 0.0226543470950928,
    (30, "x", 11, "x"): -0.0262611407020985,
    (30, "x", 13, "x"): 0.0305491632478923,
    (30, "x", 15, "x"): -0.0358375732209661,
    (30, "x", 17, "x"): 0.0426236826218798,
    (30, "x", 19, "x"): -0.0517537930024467,
    (30, "x", 21, "x"): 0.0648167139702787,
    (30, "x", 23, "x"): -0.0852087230070191,
    (30, "x", 25, "x"): 0.121752104250785,
    (30, "x", 27, "x"): -0.206785667252914,
    (30, "x", 29, "x"): 0.63133762416741,
    (30, "x", 31, "x"): 0.641773663798743,
    (30, "x", 33, "x"): -0.217241412157705,
    (30, "x", 35, "x"): 0.132247792438937,
    (30, "x", 37, "x"): -0.0957657043526452,
    (30, "x", 39, "x"): 0.0754581308127959,
    (32, "x", 1, "x"): -0.00962863484261943,
    (32, "x", 3, "x"): 0.0126059053586902,
    (32, "x", 5, "x"): -0.0151378181055341,
    (32, "x", 7, "x"): 0.0176587692986935,
    (32, "x", 9, "x"): -0.020358644867144,
    (32, "x", 11, "x"): 0.0233858818137396,
    (32, "x", 13, "x"): -0.0269029867321868,
    (32, "x", 15, "x"): 0.0311233823964002,
    (32, "x", 17, "x"): -0.0363587480299401,
    (32, "x", 19, "x"): 0.0431019952858045,
    (32, "x", 21, "x"): -0.0521966573355218,
    (32, "x", 23, "x"): 0.0652297096005878,
    (32, "x", 25, "x"): -0.0855961631739456,
    (32, "x", 27, "x"): 0.122117397913667,
    (32, "x", 29, "x"): -0.207131559327254,
    (32, "x", 31, "x"): 0.631666360260589,
    (32, "x", 33, "x"): 0.641460221209205,
    (32, "x", 35, "x"): -0.216941699788514,
    (32, "x", 37, "x"): 0.131960484273278,
    (32, "x", 39, "x"): -0.0954896651000627,
    (34, "x", 1, "x"): 0.00891107260430101,
    (34, "x", 3, "x"): -0.0116179059849555,
    (34, "x", 5, "x"): 0.0138850216154384,
    (34, "x", 7, "x"): -0.0161084661466783,
    (34, "x", 9, "x"): 0.0184524571478648,
    (34, "x", 11, "x"): -0.0210359786164796,
    (34, "x", 13, "x"): 0.0239801731757788,
    (34, "x", 15, "x"): -0.0274346624154534,
    (34, "x", 17, "x"): 0.0316059442338158,
    (34, "x", 19, "x"): -0.0368016236511537,
    (34, "x", 21, "x"): 0.0435120491147015,
    (34, "x", 23, "x"): -0.0525790555417618,
    (34, "x", 25, "x"): 0.0655884458140045,
    (34, "x", 27, "x"): -0.0859343937562171,
    (34, "x", 29, "x"): 0.122437664390516,
    (34, "x", 31, "x"): -0.207435940915892,
    (34, "x", 33, "x"): 0.631956581438438,
    (34, "x", 35, "x"): 0.641182712988902,
    (34, "x", 37, "x"): -0.216675676757992,
    (34, "x", 39, "x"): 0.131704895258842,
    (36, "x", 1, "x"): -0.00828435401829169,
    (36, "x", 3, "x"): 0.0107611425267978,
    (36, "x", 5, "x"): -0.0128075374964671,
    (36, "x", 7, "x"): 0.0147877713476547,
    (36, "x", 9, "x"): -0.0168466371421289,
    (36, "x", 11, "x"): 0.0190824142643028,
    (36, "x", 13, "x"): -0.0215887026965906,
    (36, "x", 15, "x"): 0.0244746617600977,
    (36, "x", 17, "x"): -0.0278834727283041,
    (36, "x", 19, "x"): 0.0320178443629164,
    (36, "x", 21, "x"): -0.0371829978128156,
    (36, "x", 23, "x"): 0.043867702084051,
    (36, "x", 25, "x"): -0.0529127015843396,
    (36, "x", 27, "x"): 0.0659030205003435,
    (36, "x", 29, "x"): -0.0862322608322223,
    (36, "x", 31, "x"): 0.122720757634075,
    (36, "x", 33, "x"): -0.20770586418026,
    (36, "x", 35, "x"): 0.632214680931101,
    (36, "x", 37, "x"): 0.640935295378691,
    (36, "x", 39, "x"): -0.216437963382246,
    (38, "x", 1, "x"): 0.00773275113661403,
    (38, "x", 3, "x"): -0.0100118271418192,
    (38, "x", 5, "x"): 0.0118719607936816,
    (38, "x", 7, "x"): -0.0136504853925913,
    (38, "x", 9, "x"): 0.0154770452283758,
    (38, "x", 11, "x"): -0.017434866570091,
    (38, "x", 13, "x"): 0.019598527229861,
    (38, "x", 15, "x"): -0.022050438024718,
    (38, "x", 17, "x"): 0.0248937447207395,
    (38, "x", 19, "x"): -0.0282680905210534,
    (38, "x", 21, "x"): 0.0323739582618915,
    (38, "x", 23, "x"): -0.0375150942942534,
    (38, "x", 25, "x"): 0.0441792493621718,
    (38, "x", 27, "x"): -0.0532064407664412,
    (38, "x", 29, "x"): 0.0661811587512815,
    (38, "x", 31, "x"): -0.0864966038327677,
    (38, "x", 33, "x"): 0.122972802996398,
    (38, "x", 35, "x"): -0.207946868938393,
    (38, "x", 37, "x"): 0.632445711335645,
    (38, "x", 39, "x"): 0.64071332643857,
    (40, "x", 1, "x"): -0.00724391688001674,
    (40, "x", 3, "x"): 0.00935151491502463,
    (40, "x", 5, "x"): -0.0110527578939448,
    (40, "x", 7, "x"): 0.012661878924004,
    (40, "x", 9, "x"): -0.014296399105208,
    (40, "x", 11, "x"): 0.0160282719391012,
    (40, "x", 13, "x"): -0.0179185139453499,
    (40, "x", 15, "x"): 0.0200312179442163,
    (40, "x", 17, "x"): -0.0224431596178727,
    (40, "x", 19, "x"): 0.0252541692908214,
    (40, "x", 21, "x"): -0.0286018042974914,
    (40, "x", 23, "x"): 0.0326851654679604,
    (40, "x", 25, "x"): -0.0378070449564994,
    (40, "x", 27, "x"): 0.0444545121459287,
    (40, "x", 29, "x"): -0.0534670839867032,
    (40, "x", 31, "x"): 0.0664288744972003,
    (40, "x", 33, "x"): -0.0867327955058567,
    (40, "x", 35, "x"): 0.123198648554883,
    (40, "x", 37, "x"): -0.208163367559173,
    (40, "x", 39, "x"): 0.632653718481981,
}

# split detector in y-direction, for up to TEM-order 40
YSPLIT = {
    ("x", 0, "x", 1): 0.797884560802865,
    ("x", 0, "x", 3): -0.32573500793528,
    ("x", 0, "x", 5): 0.218509686118416,
    ("x", 0, "x", 7): -0.168583882836184,
    ("x", 0, "x", 9): 0.139074607877595,
    ("x", 0, "x", 11): -0.119342192152758,
    ("x", 0, "x", 13): 0.105105246952602,
    ("x", 0, "x", 15): -0.0942883643366128,
    ("x", 0, "x", 17): 0.085756077667627,
    ("x", 0, "x", 19): -0.0788316791424408,
    ("x", 0, "x", 21): 0.0730852493544752,
    ("x", 0, "x", 23): -0.0682297483293215,
    ("x", 0, "x", 25): 0.0640657596627177,
    ("x", 0, "x", 27): -0.0604501569509354,
    ("x", 0, "x", 29): 0.0572773850389327,
    ("x", 0, "x", 31): -0.0544677839262821,
    ("x", 0, "x", 33): 0.051960035687153,
    ("x", 0, "x", 35): -0.0497061241097501,
    ("x", 0, "x", 37): 0.0476678793880178,
    ("x", 0, "x", 39): -0.0458145530435871,
    ("x", 2, "x", 1): 0.564189583547756,
    ("x", 2, "x", 3): 0.690988298942671,
    ("x", 2, "x", 5): -0.257516134682126,
    ("x", 2, "x", 7): 0.166889529453114,
    ("x", 2, "x", 9): -0.126437912127138,
    ("x", 2, "x", 11): 0.103140489653524,
    ("x", 2, "x", 13): -0.0878334751963756,
    ("x", 2, "x", 15): 0.0769291636262386,
    ("x", 2, "x", 17): -0.0687238645863045,
    ("x", 2, "x", 19): 0.0623003460579352,
    ("x", 2, "x", 21): -0.0571189780993917,
    ("x", 2, "x", 23): 0.0528405479815828,
    ("x", 2, "x", 25): -0.0492405794558427,
    ("x", 2, "x", 27): 0.0461642931761014,
    ("x", 2, "x", 29): -0.0435013182859334,
    ("x", 2, "x", 31): 0.0411707144994762,
    ("x", 2, "x", 33): -0.0391116996228281,
    ("x", 2, "x", 35): 0.0372776912078078,
    ("x", 2, "x", 37): -0.0356323539463384,
    ("x", 2, "x", 39): 0.0341469044387,
    ("x", 4, "x", 1): -0.16286750396764,
    ("x", 4, "x", 3): 0.598413420602149,
    ("x", 4, "x", 5): 0.669046543557289,
    ("x", 4, "x", 7): -0.240884286886713,
    ("x", 4, "x", 9): 0.153297821464992,
    ("x", 4, "x", 11): -0.114842936826923,
    ("x", 4, "x", 13): 0.092969581005561,
    ("x", 4, "x", 15): -0.0787358118089783,
    ("x", 4, "x", 17): 0.0686730145130556,
    ("x", 4, "x", 19): -0.0611475066641648,
    ("x", 4, "x", 21): 0.0552860726690191,
    ("x", 4, "x", 23): -0.0505782313126719,
    ("x", 4, "x", 25): 0.0467048872492379,
    ("x", 4, "x", 27): -0.0434559246068004,
    ("x", 4, "x", 29): 0.0406871064724294,
    ("x", 4, "x", 31): -0.0382959872150584,
    ("x", 4, "x", 33): 0.0362077065246626,
    ("x", 4, "x", 35): -0.0343662293597736,
    ("x", 4, "x", 37): 0.0327287372726021,
    ("x", 4, "x", 39): -0.031261920230486,
    ("x", 6, "x", 1): 0.0892062058076385,
    ("x", 6, "x", 3): -0.18209140509868,
    ("x", 6, "x", 5): 0.61075313987865,
    ("x", 6, "x", 7): 0.659688788381993,
    ("x", 6, "x", 9): -0.233235207868822,
    ("x", 6, "x", 11): 0.146771489830575,
    ("x", 6, "x", 13): -0.109117578596892,
    ("x", 6, "x", 15): 0.0878479967266388,
    ("x", 6, "x", 17): -0.0740877073983095,
    ("x", 6, "x", 19): 0.0644074398753087,
    ("x", 6, "x", 21): -0.0571982549981952,
    ("x", 6, "x", 23): 0.0516033260749633,
    ("x", 6, "x", 25): -0.0471234847486845,
    ("x", 6, "x", 27): 0.043447712204836,
    ("x", 6, "x", 29): -0.0403718224901522,
    ("x", 6, "x", 31): 0.0377560369073105,
    ("x", 6, "x", 33): -0.0355013303057689,
    ("x", 6, "x", 35): 0.0335355132263835,
    ("x", 6, "x", 37): -0.031804668469598,
    ("x", 6, "x", 39): 0.0302676798757954,
    ("x", 8, "x", 1): -0.059603403376112,
    ("x", 8, "x", 3): 0.102198547643328,
    ("x", 8, "x", 5): -0.190435749776861,
    ("x", 8, "x", 7): 0.617082357005358,
    ("x", 8, "x", 9): 0.6545146787836,
    ("x", 8, "x", 11): -0.228820262121828,
    ("x", 8, "x", 13): 0.142898207894381,
    ("x", 8, "x", 15): -0.10565264117106,
    ("x", 8, "x", 17): 0.0847033053948983,
    ("x", 8, "x", 19): -0.0712017602555549,
    ("x", 8, "x", 21): 0.0617354634590125,
    ("x", 8, "x", 23): -0.0547065570873058,
    ("x", 8, "x", 25): 0.0492658641354183,
    ("x", 8, "x", 27): -0.0449196778881935,
    ("x", 8, "x", 29): 0.0413609897651131,
    ("x", 8, "x", 31): -0.0383886289102033,
    ("x", 8, "x", 33): 0.0358651299902981,
    ("x", 8, "x", 35): -0.0336932742842578,
    ("x", 8, "x", 37): 0.0318023047767947,
    ("x", 8, "x", 39): -0.030139455674167,
    ("x", 10, "x", 1): 0.0439792525587995,
    ("x", 10, "x", 3): -0.0692528965958894,
    ("x", 10, "x", 5): 0.108397929099021,
    ("x", 10, "x", 7): -0.195138575204209,
    ("x", 10, "x", 9): 0.620927144090899,
    ("x", 10, "x", 11): 0.651233882791557,
    ("x", 10, "x", 13): -0.225941905251249,
    ("x", 10, "x", 15): 0.140323254503473,
    ("x", 10, "x", 17): -0.103315642865758,
    ("x", 10, "x", 19): 0.0825585698009622,
    ("x", 10, "x", 21): -0.0692160218229681,
    ("x", 10, "x", 23): 0.0598836888491238,
    ("x", 10, "x", 25): -0.0529693961318701,
    ("x", 10, "x", 27): 0.0476280244212015,
    ("x", 10, "x", 29): -0.0433688465158433,
    ("x", 10, "x", 31): 0.0398870940423755,
    ("x", 10, "x", 33): -0.0369833150045722,
    ("x", 10, "x", 35): 0.0345213862957191,
    ("x", 10, "x", 37): -0.0324051535576467,
    ("x", 10, "x", 39): 0.0305647153283757,
    ("x", 12, "x", 1): -0.0344511233825375,
    ("x", 12, "x", 3): 0.051570244826762,
    ("x", 12, "x", 5): -0.0741307969600372,
    ("x", 12, "x", 7): 0.112098577001675,
    ("x", 12, "x", 9): -0.198164159898117,
    ("x", 12, "x", 11): 0.623508972874051,
    ("x", 12, "x", 13): 0.648968714596981,
    ("x", 12, "x", 15): -0.223915479517027,
    ("x", 12, "x", 17): 0.13848407598893,
    ("x", 12, "x", 19): -0.101627759192188,
    ("x", 12, "x", 21): 0.0809958057142225,
    ("x", 12, "x", 23): -0.0677586792014329,
    ("x", 12, "x", 25): 0.058516541236311,
    ("x", 12, "x", 27): -0.0516804098923711,
    ("x", 12, "x", 29): 0.0464075103789091,
    ("x", 12, "x", 31): -0.0422088782527226,
    ("x", 12, "x", 33): 0.0387810976838852,
    ("x", 12, "x", 35): -0.0359257728661409,
    ("x", 12, "x", 37): 0.0335076182428655,
    ("x", 12, "x", 39): -0.031431152699668,
    ("x", 14, "x", 1): 0.0280905588320645,
    ("x", 14, "x", 3): -0.0406589983848392,
    ("x", 14, "x", 5): 0.0555599515078389,
    ("x", 14, "x", 7): -0.0771577797725184,
    ("x", 14, "x", 9): 0.114573457526737,
    ("x", 14, "x", 11): -0.200276093365814,
    ("x", 14, "x", 13): 0.625361901098279,
    ("x", 14, "x", 15): 0.647311065109173,
    ("x", 14, "x", 17): -0.222410991173016,
    ("x", 14, "x", 19): 0.137103342840115,
    ("x", 14, "x", 21): -0.100349371709367,
    ("x", 14, "x", 23): 0.0798036519532965,
    ("x", 14, "x", 25): -0.0666403046237177,
    ("x", 14, "x", 27): 0.0574621027762285,
    ("x", 14, "x", 29): -0.0506819820916291,
    ("x", 14, "x", 31): 0.0454586096116469,
    ("x", 14, "x", 33): -0.041304127205318,
    ("x", 14, "x", 35): 0.0379159829603672,
    ("x", 14, "x", 37): -0.0350964661555485,
    ("x", 14, "x", 39): 0.0327108419265415,
    ("x", 16, "x", 1): -0.0235720910841532,
    ("x", 16, "x", 3): 0.0333113049961062,
    ("x", 16, "x", 5): -0.0440146568671265,
    ("x", 16, "x", 7): 0.0581059881284756,
    ("x", 16, "x", 9): -0.0792394808782953,
    ("x", 16, "x", 11): 0.116349896137392,
    ("x", 16, "x", 13): -0.201834685692187,
    ("x", 16, "x", 15): 0.626756243745903,
    ("x", 16, "x", 17): 0.646045548619932,
    ("x", 16, "x", 19): -0.221249568137138,
    ("x", 16, "x", 21): 0.136028005901464,
    ("x", 16, "x", 23): -0.0993465691012448,
    ("x", 16, "x", 25): 0.0788629080536757,
    ("x", 16, "x", 27): -0.0657533402767068,
    ("x", 16, "x", 29): 0.0566222517097102,
    ("x", 16, "x", 31): -0.0498837907575016,
    ("x", 16, "x", 33): 0.04469755498707,
    ("x", 16, "x", 35): -0.0405764129180567,
    ("x", 16, "x", 37): 0.0372183888740113,
    ("x", 16, "x", 39): -0.034426235332043,
    ("x", 18, "x", 1): 0.0202129013489131,
    ("x", 18, "x", 3): -0.0280564002314289,
    ("x", 18, "x", 5): 0.0361938566085091,
    ("x", 18, "x", 7): -0.0462018029426718,
    ("x", 18, "x", 9): 0.0598942816336477,
    ("x", 18, "x", 11): -0.0807655532556404,
    ("x", 18, "x", 13): 0.117688834298236,
    ("x", 18, "x", 15): -0.203032528170905,
    ("x", 18, "x", 17): 0.627843419313627,
    ("x", 18, "x", 19): 0.645047794281474,
    ("x", 18, "x", 21): -0.220325766323103,
    ("x", 18, "x", 23): 0.135166515247699,
    ("x", 18, "x", 25): -0.0985383906746981,
    ("x", 18, "x", 27): 0.07810092971977,
    ("x", 18, "x", 29): -0.0650318354039826,
    ("x", 18, "x", 31): 0.055936535430677,
    ("x", 18, "x", 33): -0.0492299775566192,
    ("x", 18, "x", 35): 0.0440723835688313,
    ("x", 18, "x", 37): -0.0399771169972641,
    ("x", 18, "x", 39): 0.0366426000734971,
    ("x", 20, "x", 1): -0.017627299334295,
    ("x", 20, "x", 3): 0.0241288202745342,
    ("x", 20, "x", 5): -0.0305737533320824,
    ("x", 20, "x", 7): 0.0381039552927518,
    ("x", 20, "x", 9): -0.047763592814719,
    ("x", 20, "x", 11): 0.0612270740454677,
    ("x", 20, "x", 13): -0.0819349188989436,
    ("x", 20, "x", 15): 0.118734977843307,
    ("x", 20, "x", 17): -0.203982022959719,
    ("x", 20, "x", 19): 0.628714819343417,
    ("x", 20, "x", 21): 0.644240979964118,
    ("x", 20, "x", 23): -0.219573371070635,
    ("x", 20, "x", 25): 0.134460680056461,
    ("x", 20, "x", 27): -0.0978729043007541,
    ("x", 20, "x", 29): 0.0774707909003214,
    ("x", 20, "x", 31): -0.0644329526695553,
    ("x", 20, "x", 33): 0.0553655154050687,
    ("x", 20, "x", 35): -0.0486839720121728,
    ("x", 20, "x", 37): 0.0435489765855488,
    ("x", 20, "x", 39): -0.039474240161498,
    ("x", 22, "x", 1): 0.0155818275174025,
    ("x", 22, "x", 3): -0.0210925805324099,
    ("x", 22, "x", 5): 0.0263566009981115,
    ("x", 22, "x", 7): -0.0322641671658076,
    ("x", 22, "x", 9): 0.0394861351020212,
    ("x", 22, "x", 11): -0.0489431183977767,
    ("x", 22, "x", 13): 0.0622619711245261,
    ("x", 22, "x", 15): -0.082860766578907,
    ("x", 22, "x", 17): 0.119575292488569,
    ("x", 22, "x", 19): -0.204753224284902,
    ("x", 22, "x", 21): 0.629428862525661,
    ("x", 22, "x", 23): 0.643575097636349,
    ("x", 22, "x", 25): -0.218948694495961,
    ("x", 22, "x", 27): 0.133871712354208,
    ("x", 22, "x", 29): -0.0973152193947713,
    ("x", 22, "x", 31): 0.0769407678634122,
    ("x", 22, "x", 33): -0.0639275883754644,
    ("x", 22, "x", 35): 0.0548822891765173,
    ("x", 22, "x", 37): -0.0482207457593204,
    ("x", 22, "x", 39): 0.0431039197585667,
    ("x", 24, "x", 1): -0.0139273390571125,
    ("x", 24, "x", 3): 0.0186819548996952,
    ("x", 24, "x", 5): -0.0230856985072204,
    ("x", 24, "x", 7): 0.0278689812889355,
    ("x", 24, "x", 9): -0.0335007876120844,
    ("x", 24, "x", 11): 0.040541449001796,
    ("x", 24, "x", 13): -0.0498690376104398,
    ("x", 24, "x", 15): 0.0630903264429405,
    ("x", 24, "x", 17): -0.0836125971039459,
    ("x", 24, "x", 19): 0.120265288362839,
    ("x", 24, "x", 21): -0.205392081451324,
    ("x", 24, "x", 23): 0.630024630631138,
    ("x", 24, "x", 25): 0.643016196013222,
    ("x", 24, "x", 27): -0.218421741398637,
    ("x", 24, "x", 29): 0.133372747843751,
    ("x", 24, "x", 31): -0.0968410038781554,
    ("x", 24, "x", 33): 0.0764886144798256,
    ("x", 24, "x", 35): -0.0634952419012075,
    ("x", 24, "x", 37): 0.054467836681766,
    ("x", 24, "x", 39): -0.0478225494651678,
    ("x", 26, "x", 1): 0.012564329179744,
    ("x", 26, "x", 3): -0.0167261931797469,
    ("x", 26, "x", 5): 0.0204814479513874,
    ("x", 26, "x", 7): -0.0244511756080565,
    ("x", 26, "x", 9): 0.0289854926054433,
    ("x", 26, "x", 11): -0.0344536065949141,
    ("x", 26, "x", 13): 0.0413774431197777,
    ("x", 26, "x", 15): -0.0506169449451649,
    ("x", 26, "x", 17): 0.0637691420834689,
    ("x", 26, "x", 19): -0.0842355840892829,
    ("x", 26, "x", 21): 0.120842103606662,
    ("x", 26, "x", 23): -0.205929992668735,
    ("x", 26, "x", 25): 0.63052925596685,
    ("x", 26, "x", 27): 0.64254041629879,
    ("x", 26, "x", 29): -0.217971231998968,
    ("x", 26, "x", 31): 0.1329445838244,
    ("x", 26, "x", 33): -0.0964327593462296,
    ("x", 26, "x", 35): 0.0760982532525654,
    ("x", 26, "x", 37): -0.0631210368417068,
    ("x", 26, "x", 39): 0.0541083090782501,
    ("x", 28, "x", 1): -0.0114240058576427,
    ("x", 28, "x", 3): 0.015110811991687,
    ("x", 28, "x", 5): -0.0183634797872995,
    ("x", 28, "x", 7): 0.021723856102418,
    ("x", 28, "x", 9): -0.025467063571133,
    ("x", 28, "x", 11): 0.0298524441527061,
    ("x", 28, "x", 13): -0.0352142635297349,
    ("x", 28, "x", 15): 0.0420579527183231,
    ("x", 28, "x", 17): -0.0512345899753054,
    ("x", 28, "x", 19): 0.0643359900496595,
    ("x", 28, "x", 21): -0.0847604214666164,
    ("x", 28, "x", 23): 0.121331543048508,
    ("x", 28, "x", 25): -0.206389145953734,
    ("x", 28, "x", 27): 0.630962163162352,
    ("x", 28, "x", 29): 0.642130502115889,
    ("x", 28, "x", 31): -0.21758164950347,
    ("x", 28, "x", 33): 0.132573125753267,
    ("x", 28, "x", 35): -0.0960775730220675,
    ("x", 28, "x", 37): 0.0757577671900808,
    ("x", 28, "x", 39): -0.0627939055909658,
    ("x", 30, "x", 1): 0.0104573719402441,
    ("x", 30, "x", 3): -0.0137563247003482,
    ("x", 30, "x", 5): 0.0166104416612905,
    ("x", 30, "x", 7): -0.019501442987836,
    ("x", 30, "x", 9): 0.0226543470950928,
    ("x", 30, "x", 11): -0.0262611407020985,
    ("x", 30, "x", 13): 0.0305491632478923,
    ("x", 30, "x", 15): -0.0358375732209661,
    ("x", 30, "x", 17): 0.0426236826218798,
    ("x", 30, "x", 19): -0.0517537930024467,
    ("x", 30, "x", 21): 0.0648167139702787,
    ("x", 30, "x", 23): -0.0852087230070191,
    ("x", 30, "x", 25): 0.121752104250785,
    ("x", 30, "x", 27): -0.206785667252914,
    ("x", 30, "x", 29): 0.63133762416741,
    ("x", 30, "x", 31): 0.641773663798743,
    ("x", 30, "x", 33): -0.217241412157705,
    ("x", 30, "x", 35): 0.132247792438937,
    ("x", 30, "x", 37): -0.0957657043526452,
    ("x", 30, "x", 39): 0.0754581308127959,
    ("x", 32, "x", 1): -0.00962863484261943,
    ("x", 32, "x", 3): 0.0126059053586902,
    ("x", 32, "x", 5): -0.0151378181055341,
    ("x", 32, "x", 7): 0.0176587692986935,
    ("x", 32, "x", 9): -0.020358644867144,
    ("x", 32, "x", 11): 0.0233858818137396,
    ("x", 32, "x", 13): -0.0269029867321868,
    ("x", 32, "x", 15): 0.0311233823964002,
    ("x", 32, "x", 17): -0.0363587480299401,
    ("x", 32, "x", 19): 0.0431019952858045,
    ("x", 32, "x", 21): -0.0521966573355218,
    ("x", 32, "x", 23): 0.0652297096005878,
    ("x", 32, "x", 25): -0.0855961631739456,
    ("x", 32, "x", 27): 0.122117397913667,
    ("x", 32, "x", 29): -0.207131559327254,
    ("x", 32, "x", 31): 0.631666360260589,
    ("x", 32, "x", 33): 0.641460221209205,
    ("x", 32, "x", 35): -0.216941699788514,
    ("x", 32, "x", 37): 0.131960484273278,
    ("x", 32, "x", 39): -0.0954896651000627,
    ("x", 34, "x", 1): 0.00891107260430101,
    ("x", 34, "x", 3): -0.0116179059849555,
    ("x", 34, "x", 5): 0.0138850216154384,
    ("x", 34, "x", 7): -0.0161084661466783,
    ("x", 34, "x", 9): 0.0184524571478648,
    ("x", 34, "x", 11): -0.0210359786164796,
    ("x", 34, "x", 13): 0.0239801731757788,
    ("x", 34, "x", 15): -0.0274346624154534,
    ("x", 34, "x", 17): 0.0316059442338158,
    ("x", 34, "x", 19): -0.0368016236511537,
    ("x", 34, "x", 21): 0.0435120491147015,
    ("x", 34, "x", 23): -0.0525790555417618,
    ("x", 34, "x", 25): 0.0655884458140045,
    ("x", 34, "x", 27): -0.0859343937562171,
    ("x", 34, "x", 29): 0.122437664390516,
    ("x", 34, "x", 31): -0.207435940915892,
    ("x", 34, "x", 33): 0.631956581438438,
    ("x", 34, "x", 35): 0.641182712988902,
    ("x", 34, "x", 37): -0.216675676757992,
    ("x", 34, "x", 39): 0.131704895258842,
    ("x", 36, "x", 1): -0.00828435401829169,
    ("x", 36, "x", 3): 0.0107611425267978,
    ("x", 36, "x", 5): -0.0128075374964671,
    ("x", 36, "x", 7): 0.0147877713476547,
    ("x", 36, "x", 9): -0.0168466371421289,
    ("x", 36, "x", 11): 0.0190824142643028,
    ("x", 36, "x", 13): -0.0215887026965906,
    ("x", 36, "x", 15): 0.0244746617600977,
    ("x", 36, "x", 17): -0.0278834727283041,
    ("x", 36, "x", 19): 0.0320178443629164,
    ("x", 36, "x", 21): -0.0371829978128156,
    ("x", 36, "x", 23): 0.043867702084051,
    ("x", 36, "x", 25): -0.0529127015843396,
    ("x", 36, "x", 27): 0.0659030205003435,
    ("x", 36, "x", 29): -0.0862322608322223,
    ("x", 36, "x", 31): 0.122720757634075,
    ("x", 36, "x", 33): -0.20770586418026,
    ("x", 36, "x", 35): 0.632214680931101,
    ("x", 36, "x", 37): 0.640935295378691,
    ("x", 36, "x", 39): -0.216437963382246,
    ("x", 38, "x", 1): 0.00773275113661403,
    ("x", 38, "x", 3): -0.0100118271418192,
    ("x", 38, "x", 5): 0.0118719607936816,
    ("x", 38, "x", 7): -0.0136504853925913,
    ("x", 38, "x", 9): 0.0154770452283758,
    ("x", 38, "x", 11): -0.017434866570091,
    ("x", 38, "x", 13): 0.019598527229861,
    ("x", 38, "x", 15): -0.022050438024718,
    ("x", 38, "x", 17): 0.0248937447207395,
    ("x", 38, "x", 19): -0.0282680905210534,
    ("x", 38, "x", 21): 0.0323739582618915,
    ("x", 38, "x", 23): -0.0375150942942534,
    ("x", 38, "x", 25): 0.0441792493621718,
    ("x", 38, "x", 27): -0.0532064407664412,
    ("x", 38, "x", 29): 0.0661811587512815,
    ("x", 38, "x", 31): -0.0864966038327677,
    ("x", 38, "x", 33): 0.122972802996398,
    ("x", 38, "x", 35): -0.207946868938393,
    ("x", 38, "x", 37): 0.632445711335645,
    ("x", 38, "x", 39): 0.64071332643857,
    ("x", 40, "x", 1): -0.00724391688001674,
    ("x", 40, "x", 3): 0.00935151491502463,
    ("x", 40, "x", 5): -0.0110527578939448,
    ("x", 40, "x", 7): 0.012661878924004,
    ("x", 40, "x", 9): -0.014296399105208,
    ("x", 40, "x", 11): 0.0160282719391012,
    ("x", 40, "x", 13): -0.0179185139453499,
    ("x", 40, "x", 15): 0.0200312179442163,
    ("x", 40, "x", 17): -0.0224431596178727,
    ("x", 40, "x", 19): 0.0252541692908214,
    ("x", 40, "x", 21): -0.0286018042974914,
    ("x", 40, "x", 23): 0.0326851654679604,
    ("x", 40, "x", 25): -0.0378070449564994,
    ("x", 40, "x", 27): 0.0444545121459287,
    ("x", 40, "x", 29): -0.0534670839867032,
    ("x", 40, "x", 31): 0.0664288744972003,
    ("x", 40, "x", 33): -0.0867327955058567,
    ("x", 40, "x", 35): 0.123198648554883,
    ("x", 40, "x", 37): -0.208163367559173,
    ("x", 40, "x", 39): 0.632653718481981,
}
