# Copyright (c) 2019-2021, Manfred Moitzi
# License: MIT License
from functools import partial
from ezdxf.lldxf.hdrvars import SingleValue, Point2D, Point3D, HeaderVarDef
from ezdxf.lldxf.const import (
    DXF12,
    DXF2000,
    DXF2004,
    DXF2007,
    DXF2010,
    DXF2013,
    DXF2018,
)

HEADER_VAR_MAP = {
    "$ACADVER": HeaderVarDef(
        name="$ACADVER",
        code=1,
        factory=partial(SingleValue, code=1),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=0,
        default="AC1032",
    ),
    "$ACADMAINTVER": HeaderVarDef(
        name="$ACADMAINTVER",
        # group code changed to 90 in DXF R2018+, this fact is handled in:
        # ezdxf.sections.header.HeaderSection.export_dxf()
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=100,
        default=4,
    ),
    "$DWGCODEPAGE": HeaderVarDef(
        name="$DWGCODEPAGE",
        code=3,
        factory=partial(SingleValue, code=3),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=200,
        default="ANSI_1252",
    ),
    "$LASTSAVEDBY": HeaderVarDef(
        name="$LASTSAVEDBY",
        code=1,
        factory=partial(SingleValue, code=1),
        mindxf=DXF2004,
        maxdxf=DXF2018,
        priority=300,
        default="ezdxf",
    ),
    "$REQUIREDVERSIONS": HeaderVarDef(
        name="$REQUIREDVERSIONS",
        code=160,
        factory=partial(SingleValue, code=160),
        mindxf=DXF2013,
        maxdxf=DXF2018,
        priority=400,
        default=0,
    ),
    "$INSBASE": HeaderVarDef(
        name="$INSBASE",
        code=10,
        factory=Point3D,
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=500,
        default=(0.0, 0.0, 0.0),
    ),
    "$EXTMIN": HeaderVarDef(
        name="$EXTMIN",
        code=10,
        factory=Point3D,
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=600,
        default=(1e20, 1e20, 1e20),
    ),
    "$EXTMAX": HeaderVarDef(
        name="$EXTMAX",
        code=10,
        factory=Point3D,
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=700,
        default=(-1e20, -1e20, -1e20),
    ),
    "$LIMMIN": HeaderVarDef(
        name="$LIMMIN",
        code=10,
        factory=Point2D,
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=800,
        default=(0.0, 0.0),
    ),
    "$LIMMAX": HeaderVarDef(
        name="$LIMMAX",
        code=10,
        factory=Point2D,
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=900,
        default=(420.0, 297.0),
    ),
    "$ORTHOMODE": HeaderVarDef(
        name="$ORTHOMODE",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=1000,
        default=0,
    ),
    "$REGENMODE": HeaderVarDef(
        name="$REGENMODE",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=1100,
        default=1,
    ),
    "$FILLMODE": HeaderVarDef(
        name="$FILLMODE",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=1200,
        default=1,
    ),
    "$DRAGMODE": HeaderVarDef(
        name="$DRAGMODE",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF12,
        priority=1250,
        default=2,
    ),
    "$QTEXTMODE": HeaderVarDef(
        name="$QTEXTMODE",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=1300,
        default=0,
    ),
    "$MIRRTEXT": HeaderVarDef(
        name="$MIRRTEXT",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=1400,
        default=1,
    ),
    "$OSMODE": HeaderVarDef(
        name="$OSMODE",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF12,
        priority=1400,
        default=20583,
    ),
    "$LTSCALE": HeaderVarDef(
        name="$LTSCALE",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=1500,
        default=1.0,
    ),
    "$ATTMODE": HeaderVarDef(
        name="$ATTMODE",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=1600,
        default=1,
    ),
    "$TEXTSIZE": HeaderVarDef(
        name="$TEXTSIZE",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=1700,
        default=2.5,
    ),
    "$TRACEWID": HeaderVarDef(
        name="$TRACEWID",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=1800,
        default=1.0,
    ),
    "$TEXTSTYLE": HeaderVarDef(
        name="$TEXTSTYLE",
        code=7,
        factory=partial(SingleValue, code=7),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=1900,
        default="Standard",
    ),
    "$CLAYER": HeaderVarDef(
        name="$CLAYER",
        code=8,
        factory=partial(SingleValue, code=8),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=2000,
        default="0",
    ),
    "$CELTYPE": HeaderVarDef(
        name="$CELTYPE",
        code=6,
        factory=partial(SingleValue, code=6),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=2100,
        default="ByLayer",
    ),
    "$CECOLOR": HeaderVarDef(
        name="$CECOLOR",
        code=62,
        factory=partial(SingleValue, code=62),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=2200,
        default=256,
    ),
    "$CELTSCALE": HeaderVarDef(
        name="$CELTSCALE",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=2300,
        default=1.0,
    ),
    "$DISPSILH": HeaderVarDef(
        name="$DISPSILH",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=2400,
        default=0,
    ),
    "$DIMSCALE": HeaderVarDef(
        name="$DIMSCALE",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=2500,
        default=1.0,
    ),
    "$DIMASZ": HeaderVarDef(
        name="$DIMASZ",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=2600,
        default=2.5,
    ),
    "$DIMEXO": HeaderVarDef(
        name="$DIMEXO",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=2700,
        default=0.625,
    ),
    "$DIMDLI": HeaderVarDef(
        name="$DIMDLI",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=2800,
        default=3.75,
    ),
    "$DIMRND": HeaderVarDef(
        name="$DIMRND",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=2900,
        default=0.0,
    ),
    "$DIMDLE": HeaderVarDef(
        name="$DIMDLE",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=3000,
        default=0.0,
    ),
    "$DIMEXE": HeaderVarDef(
        name="$DIMEXE",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=3100,
        default=1.25,
    ),
    "$DIMTP": HeaderVarDef(
        name="$DIMTP",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=3200,
        default=0.0,
    ),
    "$DIMTM": HeaderVarDef(
        name="$DIMTM",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=3300,
        default=0.0,
    ),
    "$DIMTXT": HeaderVarDef(
        name="$DIMTXT",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=3400,
        default=2.5,
    ),
    "$DIMCEN": HeaderVarDef(
        name="$DIMCEN",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=3500,
        default=2.5,
    ),
    "$DIMTSZ": HeaderVarDef(
        name="$DIMTSZ",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=3600,
        default=0.0,
    ),
    "$DIMTOL": HeaderVarDef(
        name="$DIMTOL",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=3700,
        default=0,
    ),
    "$DIMLIM": HeaderVarDef(
        name="$DIMLIM",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=3800,
        default=0,
    ),
    "$DIMTIH": HeaderVarDef(
        name="$DIMTIH",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=3900,
        default=0,
    ),
    "$DIMTOH": HeaderVarDef(
        name="$DIMTOH",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=4000,
        default=0,
    ),
    "$DIMSE1": HeaderVarDef(
        name="$DIMSE1",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=4100,
        default=0,
    ),
    "$DIMSE2": HeaderVarDef(
        name="$DIMSE2",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=4200,
        default=0,
    ),
    "$DIMTAD": HeaderVarDef(
        name="$DIMTAD",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=4300,
        default=1,
    ),
    "$DIMZIN": HeaderVarDef(
        name="$DIMZIN",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=4400,
        default=8,
    ),
    "$DIMBLK": HeaderVarDef(
        name="$DIMBLK",
        code=1,
        factory=partial(SingleValue, code=1),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=4500,
        default="",
    ),
    "$DIMASO": HeaderVarDef(
        name="$DIMASO",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=4600,
        default=1,
    ),
    "$DIMSHO": HeaderVarDef(
        name="$DIMSHO",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=4700,
        default=1,
    ),
    "$DIMPOST": HeaderVarDef(
        name="$DIMPOST",
        code=1,
        factory=partial(SingleValue, code=1),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=4800,
        default="",
    ),
    "$DIMAPOST": HeaderVarDef(
        name="$DIMAPOST",
        code=1,
        factory=partial(SingleValue, code=1),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=4900,
        default="",
    ),
    "$DIMALT": HeaderVarDef(
        name="$DIMALT",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=5000,
        default=0,
    ),
    "$DIMALTD": HeaderVarDef(
        name="$DIMALTD",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=5100,
        default=3,
    ),
    "$DIMALTF": HeaderVarDef(
        name="$DIMALTF",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=5200,
        default=0.03937007874,
    ),
    "$DIMLFAC": HeaderVarDef(
        name="$DIMLFAC",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=5300,
        default=1.0,
    ),
    "$DIMTOFL": HeaderVarDef(
        name="$DIMTOFL",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=5400,
        default=1,
    ),
    "$DIMTVP": HeaderVarDef(
        name="$DIMTVP",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=5500,
        default=0.0,
    ),
    "$DIMTIX": HeaderVarDef(
        name="$DIMTIX",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=5600,
        default=0,
    ),
    "$DIMSOXD": HeaderVarDef(
        name="$DIMSOXD",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=5700,
        default=0,
    ),
    "$DIMSAH": HeaderVarDef(
        name="$DIMSAH",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=5800,
        default=0,
    ),
    "$DIMBLK1": HeaderVarDef(
        name="$DIMBLK1",
        code=1,
        factory=partial(SingleValue, code=1),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=5900,
        default="",
    ),
    "$DIMBLK2": HeaderVarDef(
        name="$DIMBLK2",
        code=1,
        factory=partial(SingleValue, code=1),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=6000,
        default="",
    ),
    "$DIMSTYLE": HeaderVarDef(
        name="$DIMSTYLE",
        code=2,
        factory=partial(SingleValue, code=2),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=6100,
        default="ISO-25",
    ),
    "$DIMCLRD": HeaderVarDef(
        name="$DIMCLRD",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=6200,
        default=0,
    ),
    "$DIMCLRE": HeaderVarDef(
        name="$DIMCLRE",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=6300,
        default=0,
    ),
    "$DIMCLRT": HeaderVarDef(
        name="$DIMCLRT",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=6400,
        default=0,
    ),
    "$DIMTFAC": HeaderVarDef(
        name="$DIMTFAC",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=6500,
        default=1.0,
    ),
    "$DIMGAP": HeaderVarDef(
        name="$DIMGAP",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=6600,
        default=0.625,
    ),
    "$DIMJUST": HeaderVarDef(
        name="$DIMJUST",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=6700,
        default=0,
    ),
    "$DIMSD1": HeaderVarDef(
        name="$DIMSD1",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=6800,
        default=0,
    ),
    "$DIMSD2": HeaderVarDef(
        name="$DIMSD2",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=6900,
        default=0,
    ),
    "$DIMTOLJ": HeaderVarDef(
        name="$DIMTOLJ",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=7000,
        default=0,
    ),
    "$DIMTZIN": HeaderVarDef(
        name="$DIMTZIN",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=7100,
        default=8,
    ),
    "$DIMALTZ": HeaderVarDef(
        name="$DIMALTZ",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=7200,
        default=0,
    ),
    "$DIMALTTZ": HeaderVarDef(
        name="$DIMALTTZ",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=7300,
        default=0,
    ),
    "$DIMUPT": HeaderVarDef(
        name="$DIMUPT",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=7400,
        default=0,
    ),
    "$DIMDEC": HeaderVarDef(
        name="$DIMDEC",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=7500,
        default=2,
    ),
    "$DIMTDEC": HeaderVarDef(
        name="$DIMTDEC",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=7600,
        default=2,
    ),
    "$DIMALTU": HeaderVarDef(
        name="$DIMALTU",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=7700,
        default=2,
    ),
    "$DIMALTTD": HeaderVarDef(
        name="$DIMALTTD",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=7800,
        default=3,
    ),
    "$DIMTXSTY": HeaderVarDef(
        name="$DIMTXSTY",
        code=7,
        factory=partial(SingleValue, code=7),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=7900,
        default="Standard",
    ),
    "$DIMAUNIT": HeaderVarDef(
        name="$DIMAUNIT",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=8000,
        default=0,
    ),
    "$DIMADEC": HeaderVarDef(
        name="$DIMADEC",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=8100,
        default=0,
    ),
    "$DIMALTRND": HeaderVarDef(
        name="$DIMALTRND",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=8200,
        default=0.0,
    ),
    "$DIMAZIN": HeaderVarDef(
        name="$DIMAZIN",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=8300,
        default=0,
    ),
    "$DIMDSEP": HeaderVarDef(
        name="$DIMDSEP",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=8400,
        default=44,
    ),
    "$DIMATFIT": HeaderVarDef(
        name="$DIMATFIT",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=8500,
        default=3,
    ),
    "$DIMFRAC": HeaderVarDef(
        name="$DIMFRAC",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=8600,
        default=0,
    ),
    "$DIMLDRBLK": HeaderVarDef(
        name="$DIMLDRBLK",
        code=1,
        factory=partial(SingleValue, code=1),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=8700,
        default="",
    ),
    "$DIMLUNIT": HeaderVarDef(
        name="$DIMLUNIT",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=8800,
        default=2,
    ),
    "$COORDS": HeaderVarDef(
        name="$COORDS",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF12,
        priority=8800,
        default=1,
    ),
    "$DIMLWD": HeaderVarDef(
        name="$DIMLWD",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=8900,
        default=-2,
    ),
    "$DIMLWE": HeaderVarDef(
        name="$DIMLWE",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=9000,
        default=-2,
    ),
    "$DIMTMOVE": HeaderVarDef(
        name="$DIMTMOVE",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=9100,
        default=0,
    ),
    "$DIMFXL": HeaderVarDef(
        name="$DIMFXL",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF2007,
        maxdxf=DXF2018,
        priority=9200,
        default=1.0,
    ),
    "$ATTDIA": HeaderVarDef(
        name="$ATTDIA",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF12,
        priority=9200,
        default=0,
    ),
    "$DIMFXLON": HeaderVarDef(
        name="$DIMFXLON",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF2007,
        maxdxf=DXF2018,
        priority=9300,
        default=0,
    ),
    "$ATTREQ": HeaderVarDef(
        name="$ATTREQ",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF12,
        priority=9300,
        default=1,
    ),
    "$DIMJOGANG": HeaderVarDef(
        name="$DIMJOGANG",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF2007,
        maxdxf=DXF2018,
        priority=9400,
        default=0.785398163397,
    ),
    "$DIMTFILL": HeaderVarDef(
        name="$DIMTFILL",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF2007,
        maxdxf=DXF2018,
        priority=9500,
        default=0,
    ),
    "$DIMTFILLCLR": HeaderVarDef(
        name="$DIMTFILLCLR",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF2007,
        maxdxf=DXF2018,
        priority=9600,
        default=0,
    ),
    "$DIMARCSYM": HeaderVarDef(
        name="$DIMARCSYM",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF2007,
        maxdxf=DXF2018,
        priority=9700,
        default=0,
    ),
    "$DIMLTYPE": HeaderVarDef(
        name="$DIMLTYPE",
        code=6,
        factory=partial(SingleValue, code=6),
        mindxf=DXF2007,
        maxdxf=DXF2018,
        priority=9800,
        default="",
    ),
    "$DIMLTEX1": HeaderVarDef(
        name="$DIMLTEX1",
        code=6,
        factory=partial(SingleValue, code=6),
        mindxf=DXF2007,
        maxdxf=DXF2018,
        priority=9900,
        default="",
    ),
    "$DIMLTEX2": HeaderVarDef(
        name="$DIMLTEX2",
        code=6,
        factory=partial(SingleValue, code=6),
        mindxf=DXF2007,
        maxdxf=DXF2018,
        priority=10000,
        default="",
    ),
    "$DIMTXTDIRECTION": HeaderVarDef(
        name="$DIMTXTDIRECTION",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF2010,
        maxdxf=DXF2018,
        priority=10100,
        default=0,
    ),
    "$LUNITS": HeaderVarDef(
        name="$LUNITS",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=10200,
        default=2,
    ),
    "$LUPREC": HeaderVarDef(
        name="$LUPREC",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=10300,
        default=4,
    ),
    "$SKETCHINC": HeaderVarDef(
        name="$SKETCHINC",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=10400,
        default=1.0,
    ),
    "$FILLETRAD": HeaderVarDef(
        name="$FILLETRAD",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=10500,
        default=10.0,
    ),
    "$AUNITS": HeaderVarDef(
        name="$AUNITS",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=10600,
        default=0,
    ),
    "$AUPREC": HeaderVarDef(
        name="$AUPREC",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=10700,
        default=2,
    ),
    "$MENU": HeaderVarDef(
        name="$MENU",
        code=1,
        factory=partial(SingleValue, code=1),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=10800,
        default=".",
    ),
    "$ELEVATION": HeaderVarDef(
        name="$ELEVATION",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=10900,
        default=0.0,
    ),
    "$PELEVATION": HeaderVarDef(
        name="$PELEVATION",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=11000,
        default=0.0,
    ),
    "$THICKNESS": HeaderVarDef(
        name="$THICKNESS",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=11100,
        default=0.0,
    ),
    "$LIMCHECK": HeaderVarDef(
        name="$LIMCHECK",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=11200,
        default=0,
    ),
    "$CHAMFERA": HeaderVarDef(
        name="$CHAMFERA",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=11300,
        default=0.0,
    ),
    "$CHAMFERB": HeaderVarDef(
        name="$CHAMFERB",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=11400,
        default=0.0,
    ),
    "$CHAMFERC": HeaderVarDef(
        name="$CHAMFERC",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=11500,
        default=0.0,
    ),
    "$CHAMFERD": HeaderVarDef(
        name="$CHAMFERD",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=11600,
        default=0.0,
    ),
    "$SKPOLY": HeaderVarDef(
        name="$SKPOLY",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=11700,
        default=0,
    ),
    "$TDCREATE": HeaderVarDef(
        name="$TDCREATE",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=11800,
        default=2458532.195663565,
    ),
    "$TDUCREATE": HeaderVarDef(
        name="$TDUCREATE",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=11900,
        default=2458532.153996898,
    ),
    "$TDUPDATE": HeaderVarDef(
        name="$TDUPDATE",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=12000,
        default=2458532.196097766,
    ),
    "$TDUUPDATE": HeaderVarDef(
        name="$TDUUPDATE",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=12100,
        default=2458532.1544311,
    ),
    "$TDINDWG": HeaderVarDef(
        name="$TDINDWG",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=12200,
        default=0.0,
    ),
    "$TDUSRTIMER": HeaderVarDef(
        name="$TDUSRTIMER",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=12300,
        default=0.0,
    ),
    "$USRTIMER": HeaderVarDef(
        name="$USRTIMER",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=12400,
        default=1,
    ),
    "$ANGBASE": HeaderVarDef(
        name="$ANGBASE",
        code=50,
        factory=partial(SingleValue, code=50),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=12500,
        default=0.0,
    ),
    "$ANGDIR": HeaderVarDef(
        name="$ANGDIR",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=12600,
        default=0,
    ),
    "$PDMODE": HeaderVarDef(
        name="$PDMODE",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=12700,
        default=0,
    ),
    "$PDSIZE": HeaderVarDef(
        name="$PDSIZE",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=12800,
        default=0.0,
    ),
    "$PLINEWID": HeaderVarDef(
        name="$PLINEWID",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=12900,
        default=0.0,
    ),
    "$SPLFRAME": HeaderVarDef(
        name="$SPLFRAME",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=13000,
        default=0,
    ),
    "$SPLINETYPE": HeaderVarDef(
        name="$SPLINETYPE",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=13100,
        default=6,
    ),
    "$SPLINESEGS": HeaderVarDef(
        name="$SPLINESEGS",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=13200,
        default=8,
    ),
    "$HANDLING": HeaderVarDef(
        name="$HANDLING",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF12,
        priority=9400,
        default=1,
    ),
    "$HANDSEED": HeaderVarDef(
        name="$HANDSEED",
        code=5,
        factory=partial(SingleValue, code=5),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=13300,
        default="100",
    ),
    "$SURFTAB1": HeaderVarDef(
        name="$SURFTAB1",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=13400,
        default=6,
    ),
    "$SURFTAB2": HeaderVarDef(
        name="$SURFTAB2",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=13500,
        default=6,
    ),
    "$SURFTYPE": HeaderVarDef(
        name="$SURFTYPE",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=13600,
        default=6,
    ),
    "$SURFU": HeaderVarDef(
        name="$SURFU",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=13700,
        default=6,
    ),
    "$SURFV": HeaderVarDef(
        name="$SURFV",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=13800,
        default=6,
    ),
    "$UCSBASE": HeaderVarDef(
        name="$UCSBASE",
        code=2,
        factory=partial(SingleValue, code=2),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=13900,
        default="",
    ),
    "$UCSNAME": HeaderVarDef(
        name="$UCSNAME",
        code=2,
        factory=partial(SingleValue, code=2),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=14000,
        default="",
    ),
    "$UCSORG": HeaderVarDef(
        name="$UCSORG",
        code=10,
        factory=Point3D,
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=14100,
        default=(0.0, 0.0, 0.0),
    ),
    "$UCSXDIR": HeaderVarDef(
        name="$UCSXDIR",
        code=10,
        factory=Point3D,
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=14200,
        default=(1.0, 0.0, 0.0),
    ),
    "$UCSYDIR": HeaderVarDef(
        name="$UCSYDIR",
        code=10,
        factory=Point3D,
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=14300,
        default=(0.0, 1.0, 0.0),
    ),
    "$UCSORTHOREF": HeaderVarDef(
        name="$UCSORTHOREF",
        code=2,
        factory=partial(SingleValue, code=2),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=14400,
        default="",
    ),
    "$UCSORTHOVIEW": HeaderVarDef(
        name="$UCSORTHOVIEW",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=14500,
        default=0,
    ),
    "$UCSORGTOP": HeaderVarDef(
        name="$UCSORGTOP",
        code=10,
        factory=Point3D,
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=14600,
        default=(0.0, 0.0, 0.0),
    ),
    "$UCSORGBOTTOM": HeaderVarDef(
        name="$UCSORGBOTTOM",
        code=10,
        factory=Point3D,
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=14700,
        default=(0.0, 0.0, 0.0),
    ),
    "$UCSORGLEFT": HeaderVarDef(
        name="$UCSORGLEFT",
        code=10,
        factory=Point3D,
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=14800,
        default=(0.0, 0.0, 0.0),
    ),
    "$UCSORGRIGHT": HeaderVarDef(
        name="$UCSORGRIGHT",
        code=10,
        factory=Point3D,
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=14900,
        default=(0.0, 0.0, 0.0),
    ),
    "$UCSORGFRONT": HeaderVarDef(
        name="$UCSORGFRONT",
        code=10,
        factory=Point3D,
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=15000,
        default=(0.0, 0.0, 0.0),
    ),
    "$UCSORGBACK": HeaderVarDef(
        name="$UCSORGBACK",
        code=10,
        factory=Point3D,
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=15100,
        default=(0.0, 0.0, 0.0),
    ),
    "$PUCSBASE": HeaderVarDef(
        name="$PUCSBASE",
        code=2,
        factory=partial(SingleValue, code=2),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=15200,
        default="",
    ),
    "$PUCSNAME": HeaderVarDef(
        name="$PUCSNAME",
        code=2,
        factory=partial(SingleValue, code=2),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=15300,
        default="",
    ),
    "$PUCSORG": HeaderVarDef(
        name="$PUCSORG",
        code=10,
        factory=Point3D,
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=15400,
        default=(0.0, 0.0, 0.0),
    ),
    "$PUCSXDIR": HeaderVarDef(
        name="$PUCSXDIR",
        code=10,
        factory=Point3D,
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=15500,
        default=(1.0, 0.0, 0.0),
    ),
    "$PUCSYDIR": HeaderVarDef(
        name="$PUCSYDIR",
        code=10,
        factory=Point3D,
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=15600,
        default=(0.0, 1.0, 0.0),
    ),
    "$PUCSORTHOREF": HeaderVarDef(
        name="$PUCSORTHOREF",
        code=2,
        factory=partial(SingleValue, code=2),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=15700,
        default="",
    ),
    "$PUCSORTHOVIEW": HeaderVarDef(
        name="$PUCSORTHOVIEW",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=15800,
        default=0,
    ),
    "$PUCSORGTOP": HeaderVarDef(
        name="$PUCSORGTOP",
        code=10,
        factory=Point3D,
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=15900,
        default=(0.0, 0.0, 0.0),
    ),
    "$PUCSORGBOTTOM": HeaderVarDef(
        name="$PUCSORGBOTTOM",
        code=10,
        factory=Point3D,
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=16000,
        default=(0.0, 0.0, 0.0),
    ),
    "$PUCSORGLEFT": HeaderVarDef(
        name="$PUCSORGLEFT",
        code=10,
        factory=Point3D,
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=16100,
        default=(0.0, 0.0, 0.0),
    ),
    "$PUCSORGRIGHT": HeaderVarDef(
        name="$PUCSORGRIGHT",
        code=10,
        factory=Point3D,
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=16200,
        default=(0.0, 0.0, 0.0),
    ),
    "$PUCSORGFRONT": HeaderVarDef(
        name="$PUCSORGFRONT",
        code=10,
        factory=Point3D,
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=16300,
        default=(0.0, 0.0, 0.0),
    ),
    "$PUCSORGBACK": HeaderVarDef(
        name="$PUCSORGBACK",
        code=10,
        factory=Point3D,
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=16400,
        default=(0.0, 0.0, 0.0),
    ),
    "$USERI1": HeaderVarDef(
        name="$USERI1",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=16500,
        default=0,
    ),
    "$USERI2": HeaderVarDef(
        name="$USERI2",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=16600,
        default=0,
    ),
    "$USERI3": HeaderVarDef(
        name="$USERI3",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=16700,
        default=0,
    ),
    "$USERI4": HeaderVarDef(
        name="$USERI4",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=16800,
        default=0,
    ),
    "$USERI5": HeaderVarDef(
        name="$USERI5",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=16900,
        default=0,
    ),
    "$USERR1": HeaderVarDef(
        name="$USERR1",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=17000,
        default=0.0,
    ),
    "$USERR2": HeaderVarDef(
        name="$USERR2",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=17100,
        default=0.0,
    ),
    "$USERR3": HeaderVarDef(
        name="$USERR3",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=17200,
        default=0.0,
    ),
    "$USERR4": HeaderVarDef(
        name="$USERR4",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=17300,
        default=0.0,
    ),
    "$USERR5": HeaderVarDef(
        name="$USERR5",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=17400,
        default=0.0,
    ),
    "$WORLDVIEW": HeaderVarDef(
        name="$WORLDVIEW",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=17500,
        default=1,
    ),
    "$SHADEDGE": HeaderVarDef(
        name="$SHADEDGE",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=17600,
        default=3,
    ),
    "$SHADEDIF": HeaderVarDef(
        name="$SHADEDIF",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=17700,
        default=70,
    ),
    "$TILEMODE": HeaderVarDef(
        name="$TILEMODE",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=17800,
        default=1,
    ),
    "$MAXACTVP": HeaderVarDef(
        name="$MAXACTVP",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=17900,
        default=64,
    ),
    "$PINSBASE": HeaderVarDef(
        name="$PINSBASE",
        code=10,
        factory=Point3D,
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=18000,
        default=(0.0, 0.0, 0.0),
    ),
    "$PLIMCHECK": HeaderVarDef(
        name="$PLIMCHECK",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=18100,
        default=0,
    ),
    "$PEXTMIN": HeaderVarDef(
        name="$PEXTMIN",
        code=10,
        factory=Point3D,
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=18200,
        default=(1e20, 1e20, 1e20),
    ),
    "$PEXTMAX": HeaderVarDef(
        name="$PEXTMAX",
        code=10,
        factory=Point3D,
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=18300,
        default=(-1e20, -1e20, -1e20),
    ),
    "$PLIMMIN": HeaderVarDef(
        name="$PLIMMIN",
        code=10,
        factory=Point2D,
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=18400,
        default=(0.0, 0.0),
    ),
    "$PLIMMAX": HeaderVarDef(
        name="$PLIMMAX",
        code=10,
        factory=Point2D,
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=18500,
        default=(420.0, 297.0),
    ),
    "$UNITMODE": HeaderVarDef(
        name="$UNITMODE",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=18600,
        default=0,
    ),
    "$VISRETAIN": HeaderVarDef(
        name="$VISRETAIN",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=18700,
        default=1,
    ),
    "$PLINEGEN": HeaderVarDef(
        name="$PLINEGEN",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=18800,
        default=0,
    ),
    "$PSLTSCALE": HeaderVarDef(
        name="$PSLTSCALE",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF12,
        maxdxf=DXF2018,
        priority=18900,
        default=1,
    ),
    "$TREEDEPTH": HeaderVarDef(
        name="$TREEDEPTH",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=19000,
        default=3020,
    ),
    "$CMLSTYLE": HeaderVarDef(
        name="$CMLSTYLE",
        code=2,
        factory=partial(SingleValue, code=2),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=19100,
        default="Standard",
    ),
    "$CMLJUST": HeaderVarDef(
        name="$CMLJUST",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=19200,
        default=0,
    ),
    "$CMLSCALE": HeaderVarDef(
        name="$CMLSCALE",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=19300,
        default=20.0,
    ),
    "$PROXYGRAPHICS": HeaderVarDef(
        name="$PROXYGRAPHICS",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=19400,
        default=1,
    ),
    "$MEASUREMENT": HeaderVarDef(
        name="$MEASUREMENT",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=19500,
        default=1,
    ),
    "$CELWEIGHT": HeaderVarDef(
        name="$CELWEIGHT",
        code=370,
        factory=partial(SingleValue, code=370),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=19600,
        default=-1,
    ),
    "$ENDCAPS": HeaderVarDef(
        name="$ENDCAPS",
        code=280,
        factory=partial(SingleValue, code=280),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=19700,
        default=0,
    ),
    "$JOINSTYLE": HeaderVarDef(
        name="$JOINSTYLE",
        code=280,
        factory=partial(SingleValue, code=280),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=19800,
        default=0,
    ),
    "$LWDISPLAY": HeaderVarDef(
        name="$LWDISPLAY",
        code=290,
        factory=partial(SingleValue, code=290),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=19900,
        default=0,
    ),
    "$INSUNITS": HeaderVarDef(
        name="$INSUNITS",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=20000,
        default=6,
    ),
    "$HYPERLINKBASE": HeaderVarDef(
        name="$HYPERLINKBASE",
        code=1,
        factory=partial(SingleValue, code=1),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=20100,
        default="",
    ),
    "$STYLESHEET": HeaderVarDef(
        name="$STYLESHEET",
        code=1,
        factory=partial(SingleValue, code=1),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=20200,
        default="",
    ),
    "$XEDIT": HeaderVarDef(
        name="$XEDIT",
        code=290,
        factory=partial(SingleValue, code=290),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=20300,
        default=1,
    ),
    "$CEPSNTYPE": HeaderVarDef(
        name="$CEPSNTYPE",
        code=380,
        factory=partial(SingleValue, code=380),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=20400,
        default=0,
    ),
    "$PSTYLEMODE": HeaderVarDef(
        name="$PSTYLEMODE",
        code=290,
        factory=partial(SingleValue, code=290),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=20500,
        default=1,
    ),
    "$FINGERPRINTGUID": HeaderVarDef(
        name="$FINGERPRINTGUID",
        code=2,
        factory=partial(SingleValue, code=2),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=20600,
        default="{6d50f5e5-c3bb-4eb1-8520-1109eb28619f}",
    ),
    "$VERSIONGUID": HeaderVarDef(
        name="$VERSIONGUID",
        code=2,
        factory=partial(SingleValue, code=2),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=20700,
        default="{FAEB1C32-E019-11D5-929B-00C0DF256EC4}",
    ),
    "$EXTNAMES": HeaderVarDef(
        name="$EXTNAMES",
        code=290,
        factory=partial(SingleValue, code=290),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=20800,
        default=1,
    ),
    "$PSVPSCALE": HeaderVarDef(
        name="$PSVPSCALE",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=20900,
        default=0.0,
    ),
    "$OLESTARTUP": HeaderVarDef(
        name="$OLESTARTUP",
        code=290,
        factory=partial(SingleValue, code=290),
        mindxf=DXF2000,
        maxdxf=DXF2018,
        priority=21000,
        default=0,
    ),
    "$SORTENTS": HeaderVarDef(
        name="$SORTENTS",
        code=280,
        factory=partial(SingleValue, code=280),
        mindxf=DXF2004,
        maxdxf=DXF2018,
        priority=21100,
        default=127,
    ),
    "$INDEXCTL": HeaderVarDef(
        name="$INDEXCTL",
        code=280,
        factory=partial(SingleValue, code=280),
        mindxf=DXF2004,
        maxdxf=DXF2018,
        priority=21200,
        default=0,
    ),
    "$HIDETEXT": HeaderVarDef(
        name="$HIDETEXT",
        code=280,
        factory=partial(SingleValue, code=280),
        mindxf=DXF2004,
        maxdxf=DXF2018,
        priority=21300,
        default=1,
    ),
    "$XCLIPFRAME": HeaderVarDef(
        name="$XCLIPFRAME",
        code=280,  # 2004 & 2007 = 290
        factory=partial(SingleValue, code=280),  # 2004 & 2007 = 290
        mindxf=DXF2004,
        maxdxf=DXF2018,
        priority=21400,
        default=0,
    ),
    "$HALOGAP": HeaderVarDef(
        name="$HALOGAP",
        code=280,
        factory=partial(SingleValue, code=280),
        mindxf=DXF2004,
        maxdxf=DXF2018,
        priority=21500,
        default=0,
    ),
    "$OBSCOLOR": HeaderVarDef(
        name="$OBSCOLOR",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF2004,
        maxdxf=DXF2018,
        priority=21600,
        default=257,
    ),
    "$OBSLTYPE": HeaderVarDef(
        name="$OBSLTYPE",
        code=280,
        factory=partial(SingleValue, code=280),
        mindxf=DXF2004,
        maxdxf=DXF2018,
        priority=21700,
        default=0,
    ),
    "$INTERSECTIONDISPLAY": HeaderVarDef(
        name="$INTERSECTIONDISPLAY",
        code=280,
        factory=partial(SingleValue, code=280),
        mindxf=DXF2004,
        maxdxf=DXF2018,
        priority=21800,
        default=0,
    ),
    "$INTERSECTIONCOLOR": HeaderVarDef(
        name="$INTERSECTIONCOLOR",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF2004,
        maxdxf=DXF2018,
        priority=21900,
        default=257,
    ),
    "$DIMASSOC": HeaderVarDef(
        name="$DIMASSOC",
        code=280,
        factory=partial(SingleValue, code=280),
        mindxf=DXF2004,
        maxdxf=DXF2018,
        priority=22000,
        default=2,
    ),
    "$PROJECTNAME": HeaderVarDef(
        name="$PROJECTNAME",
        code=1,
        factory=partial(SingleValue, code=1),
        mindxf=DXF2004,
        maxdxf=DXF2018,
        priority=22100,
        default="",
    ),
    "$CAMERADISPLAY": HeaderVarDef(
        name="$CAMERADISPLAY",
        code=290,
        factory=partial(SingleValue, code=290),
        mindxf=DXF2007,
        maxdxf=DXF2018,
        priority=22200,
        default=0,
    ),
    "$LENSLENGTH": HeaderVarDef(
        name="$LENSLENGTH",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF2007,
        maxdxf=DXF2018,
        priority=22300,
        default=50.0,
    ),
    "$CAMERAHEIGHT": HeaderVarDef(
        name="$CAMERAHEIGHT",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF2007,
        maxdxf=DXF2018,
        priority=22400,
        default=0.0,
    ),
    "$STEPSPERSEC": HeaderVarDef(
        name="$STEPSPERSEC",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF2007,
        maxdxf=DXF2018,
        priority=22500,
        default=24.0,
    ),
    "$STEPSIZE": HeaderVarDef(
        name="$STEPSIZE",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF2007,
        maxdxf=DXF2018,
        priority=22600,
        default=100.0,
    ),
    "$3DDWFPREC": HeaderVarDef(
        name="$3DDWFPREC",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF2007,
        maxdxf=DXF2018,
        priority=22700,
        default=2.0,
    ),
    "$PSOLWIDTH": HeaderVarDef(
        name="$PSOLWIDTH",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF2007,
        maxdxf=DXF2018,
        priority=22800,
        default=0.005,
    ),
    "$PSOLHEIGHT": HeaderVarDef(
        name="$PSOLHEIGHT",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF2007,
        maxdxf=DXF2018,
        priority=22900,
        default=0.08,
    ),
    "$LOFTANG1": HeaderVarDef(
        name="$LOFTANG1",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF2007,
        maxdxf=DXF2018,
        priority=23000,
        default=1.570796326795,
    ),
    "$LOFTANG2": HeaderVarDef(
        name="$LOFTANG2",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF2007,
        maxdxf=DXF2018,
        priority=23100,
        default=1.570796326795,
    ),
    "$LOFTMAG1": HeaderVarDef(
        name="$LOFTMAG1",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF2007,
        maxdxf=DXF2018,
        priority=23200,
        default=0.0,
    ),
    "$LOFTMAG2": HeaderVarDef(
        name="$LOFTMAG2",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF2007,
        maxdxf=DXF2018,
        priority=23300,
        default=0.0,
    ),
    "$LOFTPARAM": HeaderVarDef(
        name="$LOFTPARAM",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF2007,
        maxdxf=DXF2018,
        priority=23400,
        default=7,
    ),
    "$LOFTNORMALS": HeaderVarDef(
        name="$LOFTNORMALS",
        code=280,
        factory=partial(SingleValue, code=280),
        mindxf=DXF2007,
        maxdxf=DXF2018,
        priority=23500,
        default=1,
    ),
    "$LATITUDE": HeaderVarDef(
        name="$LATITUDE",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF2007,
        maxdxf=DXF2018,
        priority=23600,
        default=37.795,
    ),
    "$LONGITUDE": HeaderVarDef(
        name="$LONGITUDE",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF2007,
        maxdxf=DXF2018,
        priority=23700,
        default=-122.394,
    ),
    "$NORTHDIRECTION": HeaderVarDef(
        name="$NORTHDIRECTION",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF2007,
        maxdxf=DXF2018,
        priority=23800,
        default=0.0,
    ),
    "$TIMEZONE": HeaderVarDef(
        name="$TIMEZONE",
        code=70,
        factory=partial(SingleValue, code=70),
        mindxf=DXF2007,
        maxdxf=DXF2018,
        priority=23900,
        default=-8000,
    ),
    "$LIGHTGLYPHDISPLAY": HeaderVarDef(
        name="$LIGHTGLYPHDISPLAY",
        code=280,
        factory=partial(SingleValue, code=280),
        mindxf=DXF2007,
        maxdxf=DXF2018,
        priority=24000,
        default=1,
    ),
    "$TILEMODELIGHTSYNCH": HeaderVarDef(
        name="$TILEMODELIGHTSYNCH",
        code=280,
        factory=partial(SingleValue, code=280),
        mindxf=DXF2007,
        maxdxf=DXF2018,
        priority=24100,
        default=1,
    ),
    "$CMATERIAL": HeaderVarDef(
        name="$CMATERIAL",
        code=347,
        factory=partial(SingleValue, code=347),
        mindxf=DXF2007,
        maxdxf=DXF2018,
        priority=24200,
        default="45",
    ),  # default value '0' crashes BricsCAD
    "$SOLIDHIST": HeaderVarDef(
        name="$SOLIDHIST",
        code=280,
        factory=partial(SingleValue, code=280),
        mindxf=DXF2007,
        maxdxf=DXF2018,
        priority=24300,
        default=0,
    ),
    "$SHOWHIST": HeaderVarDef(
        name="$SHOWHIST",
        code=280,
        factory=partial(SingleValue, code=280),
        mindxf=DXF2007,
        maxdxf=DXF2018,
        priority=24400,
        default=1,
    ),
    "$DWFFRAME": HeaderVarDef(
        name="$DWFFRAME",
        code=280,
        factory=partial(SingleValue, code=280),
        mindxf=DXF2007,
        maxdxf=DXF2018,
        priority=24500,
        default=2,
    ),
    "$DGNFRAME": HeaderVarDef(
        name="$DGNFRAME",
        code=280,
        factory=partial(SingleValue, code=280),
        mindxf=DXF2007,
        maxdxf=DXF2018,
        priority=24600,
        default=2,
    ),
    "$REALWORLDSCALE": HeaderVarDef(
        name="$REALWORLDSCALE",
        code=290,
        factory=partial(SingleValue, code=290),
        mindxf=DXF2007,
        maxdxf=DXF2018,
        priority=24700,
        default=1,
    ),
    "$INTERFERECOLOR": HeaderVarDef(
        name="$INTERFERECOLOR",
        code=62,
        factory=partial(SingleValue, code=62),
        mindxf=DXF2007,
        maxdxf=DXF2018,
        priority=24800,
        default=256,
    ),
    "$INTERFEREOBJVS": HeaderVarDef(
        # Handle to a VisualStyle, if pointing to an invalid or non-existing VS,
        # copy to clipboard in AutoCAD can fail.
        name="$INTERFEREOBJVS",
        code=345,
        factory=partial(SingleValue, code=345),
        mindxf=DXF2007,
        maxdxf=DXF2018,
        priority=24810,
        default=None,
    ),  # will not be written, if not set
    "$INTERFEREVPVS": HeaderVarDef(
        # Handle to a VisualStyle, if pointing to an invalid or non-existing VS,
        # copy to clipboard in AutoCAD can fail.
        name="$INTERFEREVPVS",
        code=346,
        factory=partial(SingleValue, code=346),
        mindxf=DXF2007,
        maxdxf=DXF2018,
        priority=24820,
        default=None,
    ),  # will not be written, if not set
    "$CSHADOW": HeaderVarDef(
        name="$CSHADOW",
        code=280,
        factory=partial(SingleValue, code=280),
        mindxf=DXF2007,
        maxdxf=DXF2018,
        priority=24900,
        default=0,
    ),
    "$SHADOWPLANELOCATION": HeaderVarDef(
        name="$SHADOWPLANELOCATION",
        code=40,
        factory=partial(SingleValue, code=40),
        mindxf=DXF2007,
        maxdxf=DXF2018,
        priority=25000,
        default=0.0,
    ),
}


def version_specific_group_code(name: str, dxfversion: str) -> int:
    group_code = HEADER_VAR_MAP[name].code
    # The HEADER_VAR_MAP contains the group codes for the latest DXF version.
    # This section adjust changed group codes for older DXF versions:
    if name == "$ACADMAINTVER":
        group_code = 70 if dxfversion < DXF2018 else 90
    elif name == "$XCLIPFRAME":
        group_code = 290 if dxfversion < DXF2010 else 280
    return group_code
