from __future__ import unicode_literals

import logging
import os
import sys
import re
from ctypes import CDLL, CFUNCTYPE, c_char_p, c_int

from django.contrib.gis.gdal.error import GDALException
from django.core.exceptions import ImproperlyConfigured

logger = logging.getLogger('django.contrib.gis')

# This loads the GDAL/OGR C library.
if os.name == 'posix':
    platform = os.uname()[0]
    if platform == 'Linux':
        libname = 'libgdal.so'
    elif platform == 'Darwin':
        libname = 'libgdal.1.dylib'
    lgdal = CDLL(os.path.join(sys.prefix, 'lib', libname))
elif os.name == 'nt':
    # On Windows, the GDAL binaries have some OSR routines exported with
    # STDCALL, while others are not.  Thus, the library will also need to
    # be loaded up as WinDLL for said OSR functions that require the
    # different calling convention.
    from ctypes import WinDLL
    libname = os.path.join(sys.prefix, 'Library', 'bin', 'gdal111.dll')
    lgdal = CDLL(libname)
    lwingdal = WinDLL(libname)
else:
    raise Exception('No valid platform name. Found {}'.format(os.name))


def std_call(func):
    """
    Returns the correct STDCALL function for certain OSR routines on Win32
    platforms.
    """
    if os.name == 'nt':
        return lwingdal[func]
    else:
        return lgdal[func]

# #### Version-information functions. ####

# Returns GDAL library version information with the given key.
_version_info = std_call('GDALVersionInfo')
_version_info.argtypes = [c_char_p]
_version_info.restype = c_char_p


def gdal_version():
    "Returns only the GDAL version number information."
    return _version_info(b'RELEASE_NAME')


def gdal_full_version():
    "Returns the full GDAL version information."
    return _version_info('')

version_regex = re.compile(r'^(?P<major>\d+)\.(?P<minor>\d+)(\.(?P<subminor>\d+))?')


def gdal_version_info():
    ver = gdal_version().decode()
    m = version_regex.match(ver)
    if not m:
        raise GDALException('Could not parse GDAL version string "%s"' % ver)
    return {key: m.group(key) for key in ('major', 'minor', 'subminor')}

_verinfo = gdal_version_info()
GDAL_MAJOR_VERSION = int(_verinfo['major'])
GDAL_MINOR_VERSION = int(_verinfo['minor'])
GDAL_SUBMINOR_VERSION = _verinfo['subminor'] and int(_verinfo['subminor'])
GDAL_VERSION = (GDAL_MAJOR_VERSION, GDAL_MINOR_VERSION, GDAL_SUBMINOR_VERSION)
del _verinfo

# Set library error handling so as errors are logged
CPLErrorHandler = CFUNCTYPE(None, c_int, c_int, c_char_p)


def err_handler(error_class, error_number, message):
    logger.error('GDAL_ERROR %d: %s' % (error_number, message))
err_handler = CPLErrorHandler(err_handler)


def function(name, args, restype):
    func = std_call(name)
    func.argtypes = args
    func.restype = restype
    return func

set_error_handler = function('CPLSetErrorHandler', [CPLErrorHandler], CPLErrorHandler)
set_error_handler(err_handler)
