"""
The gRPC integration traces the client and server using the interceptor pattern.


Enabling
~~~~~~~~

The gRPC integration is enabled automatically when using
:ref:`ddtrace-run<ddtracerun>` or :ref:`patch_all()<patch_all>`.

Or use :ref:`patch()<patch>` to manually enable the integration::

    from ddtrace import patch
    patch(grpc=True)

    # use grpc like usual


Global Configuration
~~~~~~~~~~~~~~~~~~~~

.. py:data:: ddtrace.config.grpc["service"]

   The service name reported by default for gRPC client instances.

   This option can also be set with the ``DD_GRPC_SERVICE`` environment
   variable.

   Default: ``"grpc-client"``

.. py:data:: ddtrace.config.grpc_server["service"]

   The service name reported by default for gRPC server instances.

   This option can also be set with the ``DD_SERVICE`` or
   ``DD_GRPC_SERVER_SERVICE`` environment variables.

   Default: ``"grpc-server"``


Instance Configuration
~~~~~~~~~~~~~~~~~~~~~~

To configure the gRPC integration on an per-channel basis use the
``Pin`` API::

    import grpc
    from ddtrace import Pin, patch, Tracer

    patch(grpc=True)
    custom_tracer = Tracer()

    # override the pin on the client
    Pin.override(grpc.Channel, service='mygrpc', tracer=custom_tracer)
    with grpc.insecure_channel('localhost:50051') as channel:
        # create stubs and send requests
        pass

To configure the gRPC integration on the server use the ``Pin`` API::

    import grpc
    from grpc.framework.foundation import logging_pool

    from ddtrace import Pin, patch, Tracer

    patch(grpc=True)
    custom_tracer = Tracer()

    # override the pin on the server
    Pin.override(grpc.Server, service='mygrpc', tracer=custom_tracer)
    server = grpc.server(logging_pool.pool(2))
    server.add_insecure_port('localhost:50051')
    add_MyServicer_to_server(MyServicer(), server)
    server.start()
"""


from ...utils.importlib import require_modules


required_modules = ['grpc']

with require_modules(required_modules) as missing_modules:
    if not missing_modules:
        from .patch import patch
        from .patch import unpatch

        __all__ = ['patch', 'unpatch']
