#ifndef DCGP_RNG_HPP
#define DCGP_RNG_HPP

#include <mutex>
#include <random>

namespace dcgp
{
namespace detail
{

// DCGP makes use of the 32-bit Mersenne Twister by Matsumoto and Nishimura, 1998.
using random_engine_type = std::mt19937;

template <typename = void>
struct random_device_statics {
    /// DCGP random engine
    static random_engine_type global_rng;
    /// Mutex protecting access to PaGMO random engine
    static std::mutex global_rng_mutex;
};

template <typename T>
random_engine_type random_device_statics<T>::global_rng(static_cast<random_engine_type::result_type>(std::random_device()()));

template <typename T>
std::mutex random_device_statics<T>::global_rng_mutex;

} // end namespace detail

/// Thread-safe random device
/**
 * This class intends to be a thread-safe substitute for std::random_device,
 * allowing, at the same time, precise global seed control throughout PaGMO.
 * It offers the user access to a global Pseudo Random Sequence generated by the
 * 32-bit Mersenne Twister by Matsumoto and Nishimura, 1998.
 * Such a PRS can be accessed by all PaGMO classes via the static method
 * random_device::next. The seed of this global Pseudo Random Sequence can
 * be set by the method random_device::set_seed, else by default is initialized
 * once at run-time using std::random_device.
 *
 * In PaGMO, all classes that contain a random engine (thus that generate
 * random numbers from variates), by default should contain something like:
 * @code{.unparsed}
 * #include <pagmo/rng.hpp>
 * class class_using_random {
 * explicit class_using_random(args ...... , unsigned int seed = pagmo::random_device::next()) : m_e(seed),
 * m_seed(seed);
 * private:
 *    // Random engine
 *    mutable detail::random_engine_type               m_e;
 *    // Seed
 *    unsigned int                                     m_seed;
 * }
 * @endcode
 */
class random_device : public detail::random_device_statics<>
{
public:
    /// Next element of the Pseudo Random Sequence
    /**
     * This static method returns the next element of the PRS.
     *
     * @returns the next element of the PRS
     */
    static unsigned int next()
    {
        std::lock_guard<std::mutex> lock(global_rng_mutex);
        return static_cast<unsigned int>(global_rng());
    }
    /// Sets the seed for the PRS
    /**
     * This static method sets a new seed for the PRS, so that all the
     * following calls to random_device::next() will always repeat the same
     * numbers.
     *
     * @param seed The new seed to be used
     */
    static void set_seed(unsigned int seed)
    {
        std::lock_guard<std::mutex> lock(global_rng_mutex);
        global_rng.seed(static_cast<detail::random_engine_type::result_type>(seed));
    }
};

} // end namespace dcgp

#endif