import os
import sys
import logging


def load_setup_py_data(setup_file, from_recipe_dir=False, recipe_dir=None, work_dir=None,
                       permit_undefined_jinja=True):
    _setuptools_data = {}
    log = logging.getLogger(__name__)

    def setup(**kw):
        _setuptools_data.update(kw)

    import setuptools
    import distutils.core

    cd_to_work = False
    path_backup = sys.path

    if from_recipe_dir and recipe_dir:
        setup_file = os.path.abspath(os.path.join(recipe_dir, setup_file))
    elif os.path.exists(work_dir):
        cd_to_work = True
        cwd = os.getcwd()
        os.chdir(work_dir)
        if not os.path.isabs(setup_file):
            setup_file = os.path.join(work_dir, setup_file)
        # this is very important - or else if versioneer or otherwise is in the start folder,
        # things will pick up the wrong versioneer/whatever!
        sys.path.insert(0, work_dir)
    else:
        message = ("Did not find setup.py file in manually specified location, and source "
                  "not downloaded yet.")
        if permit_undefined_jinja:
            log.debug(message)
            return {}
        else:
            raise RuntimeError(message)

    # Patch setuptools, distutils
    setuptools_setup = setuptools.setup
    distutils_setup = distutils.core.setup
    numpy_setup = None

    versioneer = None
    if 'versioneer' in sys.modules:
        versioneer = sys.modules['versioneer']
        del sys.modules['versioneer']

    try:
        import numpy.distutils.core
        numpy_setup = numpy.distutils.core.setup
        numpy.distutils.core.setup = setup
    except ImportError:
        log.debug("Failed to import numpy for setup patch.  Is numpy installed?")

    setuptools.setup = distutils.core.setup = setup
    ns = {
        '__name__': '__main__',
        '__doc__': None,
        '__file__': setup_file,
    }
    if os.path.isfile(setup_file):
        code = compile(open(setup_file).read(), setup_file, 'exec', dont_inherit=1)
        exec(code, ns, ns)
    else:
        if not permit_undefined_jinja:
            raise TypeError('{} is not a file that can be read'.format(setup_file))

    sys.modules['versioneer'] = versioneer

    distutils.core.setup = distutils_setup
    setuptools.setup = setuptools_setup
    if numpy_setup:
        numpy.distutils.core.setup = numpy_setup
    if cd_to_work:
        os.chdir(cwd)
    # remove our workdir from sys.path
    sys.path = path_backup
    return _setuptools_data


if __name__ == '__main__':
    import json
    import argparse
    parser = argparse.ArgumentParser(description='run setup.py file to obtain metadata')
    parser.add_argument('work_dir', help=('path to work dir, where we\'ll write the output data '
                                    'json, and potentially also where setup.py should be found'))
    parser.add_argument('setup_file', help='path or filename of setup.py file')
    parser.add_argument('--from-recipe-dir', help=('look for setup.py file in recipe '
                                                   'dir (as opposed to work dir)'),
                        default=False, action="store_true")
    parser.add_argument('--recipe-dir', help=('(optional) path to recipe dir, where '
                                            'setup.py should be found'))

    parser.add_argument('--permit-undefined-jinja', help=('look for setup.py file in recipe '
                                                   'dir (as opposed to work dir)'),
                        default=False, action="store_true")
    args = parser.parse_args()
    # we get back a dict of the setup data
    data = load_setup_py_data(**args.__dict__)
    with open(os.path.join(args.work_dir, 'conda_build_loaded_setup_py.json'), 'w') as f:
        # need to remove the cmdclass as it is not json serializable
        if 'cmdclass' in data:
            del data['cmdclass']
        json.dump(data, f)
