# -*- coding: utf-8 -*-
from __future__ import absolute_import, division, print_function, unicode_literals

from errno import EINVAL, EXDEV
from logging import getLogger
from os import rename as os_rename, utime
from os.path import dirname, isdir, lexists
import re

from . import exp_backoff_fn
from .delete import rm_rf
from .link import islink
from ...common.path import expand

log = getLogger(__name__)

SHEBANG_REGEX = re.compile(br'^(#!((?:\\ |[^ \n\r])+)(.*))')


class CancelOperation(Exception):
    pass


def update_file_in_place_as_binary(file_full_path, callback):
    # callback should be a callable that takes one positional argument, which is the
    #   content of the file before updating
    # this method updates the file in-place, without releasing the file lock
    fh = None
    try:
        fh = exp_backoff_fn(open, file_full_path, 'rb+')
        log.trace("in-place update path locked for %s", file_full_path)
        data = fh.read()
        fh.seek(0)
        try:
            fh.write(callback(data))
            fh.truncate()
        except CancelOperation:
            pass  # NOQA
    finally:
        if fh:
            fh.close()


def _copy_then_remove(source_path, destination_path):
    from .create import copy, create_hard_link_or_copy
    if islink(source_path):
        copy(source_path, destination_path)
    else:
        create_hard_link_or_copy(source_path, destination_path)
    rm_rf(source_path)


def rename(source_path, destination_path, force=False):
    if lexists(destination_path) and force:
        rm_rf(destination_path)
    if lexists(source_path):
        log.trace("renaming %s => %s", source_path, destination_path)
        try:
            os_rename(source_path, destination_path)
        except EnvironmentError as e:
            if e.errno in (EINVAL, EXDEV):
                # see https://github.com/conda/conda/issues/6711
                log.trace("Could not rename do to errno [%s]. Falling back to copy/remove.",
                          e.errno)
                _copy_then_remove(source_path, destination_path)
            else:
                raise
    else:
        log.trace("cannot rename; source path does not exist '%s'", source_path)


def backoff_rename(source_path, destination_path, force=False):
    exp_backoff_fn(rename, source_path, destination_path, force)


def touch(path):
    # returns
    #   True if the file did not exist but was created
    #   False if the file already existed
    # raises permissions errors such as EPERM and EACCES
    path = expand(path)
    log.trace("touching path %s", path)
    if lexists(path):
        utime(path, None)
        return True
    else:
        assert isdir(dirname(path))
        try:
            fh = open(path, 'a')
        except:
            raise
        else:
            fh.close()
            return False
