#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
ASTM G-173 Solar Spectral Irradiance
====================================

Defines the solar spectral irradiance spectral power distribution used in
various plotting objects.

References
----------
.. [1]  Renewable Resource Data Center. (2003). Reference Solar Spectral
        Irradiance: ASTM G-173. Retrieved August 23, 2014, from
        http://rredc.nrel.gov/solar/spectra/am1.5/ASTMG173/ASTMG173.html
"""

from __future__ import division, unicode_literals

from colour.colorimetry import SpectralPowerDistribution

__author__ = 'Colour Developers'
__copyright__ = 'Copyright (C) 2013-2017 - Colour Developers'
__license__ = 'New BSD License - http://opensource.org/licenses/BSD-3-Clause'
__maintainer__ = 'Colour Developers'
__email__ = 'colour-science@googlegroups.com'
__status__ = 'Production'

__all__ = ['ASTM_G_173_ETR_DATA',
           'ASTM_G_173_ETR']

ASTM_G_173_ETR_DATA = {
    280.0: 0.08200,
    280.5: 0.09900,
    281.0: 0.15000,
    281.5: 0.21200,
    282.0: 0.26700,
    282.5: 0.30300,
    283.0: 0.32500,
    283.5: 0.32300,
    284.0: 0.29900,
    284.5: 0.25024,
    285.0: 0.17589,
    285.5: 0.15500,
    286.0: 0.24200,
    286.5: 0.33300,
    287.0: 0.36200,
    287.5: 0.33900,
    288.0: 0.31100,
    288.5: 0.32500,
    289.0: 0.39200,
    289.5: 0.47900,
    290.0: 0.56300,
    290.5: 0.60600,
    291.0: 0.61800,
    291.5: 0.59800,
    292.0: 0.56700,
    292.5: 0.52900,
    293.0: 0.53800,
    293.5: 0.54900,
    294.0: 0.53300,
    294.5: 0.52000,
    295.0: 0.52700,
    295.5: 0.55900,
    296.0: 0.57300,
    296.5: 0.52100,
    297.0: 0.47800,
    297.5: 0.52900,
    298.0: 0.52800,
    298.5: 0.46800,
    299.0: 0.47200,
    299.5: 0.50100,
    300.0: 0.45794,
    300.5: 0.43300,
    301.0: 0.46300,
    301.5: 0.47756,
    302.0: 0.44900,
    302.5: 0.50800,
    303.0: 0.61200,
    303.5: 0.64600,
    304.0: 0.62100,
    304.5: 0.62600,
    305.0: 0.64200,
    305.5: 0.61100,
    306.0: 0.56500,
    306.5: 0.57500,
    307.0: 0.60500,
    307.5: 0.63100,
    308.0: 0.64500,
    308.5: 0.64050,
    309.0: 0.58000,
    309.5: 0.50000,
    310.0: 0.53300,
    310.5: 0.65200,
    311.0: 0.76200,
    311.5: 0.74400,
    312.0: 0.70615,
    312.5: 0.67600,
    313.0: 0.69400,
    313.5: 0.72400,
    314.0: 0.71700,
    314.5: 0.67600,
    315.0: 0.68450,
    315.5: 0.63200,
    316.0: 0.58700,
    316.5: 0.64900,
    317.0: 0.73900,
    317.5: 0.80189,
    318.0: 0.72361,
    318.5: 0.68000,
    319.0: 0.70580,
    319.5: 0.73318,
    320.0: 0.77500,
    320.5: 0.82600,
    321.0: 0.76546,
    321.5: 0.71900,
    322.0: 0.73513,
    322.5: 0.71152,
    323.0: 0.64900,
    323.5: 0.68138,
    324.0: 0.74050,
    324.5: 0.78800,
    325.0: 0.82918,
    325.5: 0.91377,
    326.0: 0.99835,
    326.5: 1.01660,
    327.0: 1.00470,
    327.5: 0.98679,
    328.0: 0.95681,
    328.5: 0.93400,
    329.0: 1.00460,
    329.5: 1.09580,
    330.0: 1.10980,
    330.5: 1.04290,
    331.0: 0.99100,
    331.5: 0.99396,
    332.0: 0.99277,
    332.5: 0.98003,
    333.0: 0.96394,
    333.5: 0.93700,
    334.0: 0.95557,
    334.5: 0.98862,
    335.0: 1.00970,
    335.5: 0.97453,
    336.0: 0.88979,
    336.5: 0.82900,
    337.0: 0.81800,
    337.5: 0.87732,
    338.0: 0.92529,
    338.5: 0.95783,
    339.0: 0.96863,
    339.5: 0.98900,
    340.0: 1.05440,
    340.5: 1.04630,
    341.0: 0.97100,
    341.5: 0.95900,
    342.0: 0.99570,
    342.5: 1.02770,
    343.0: 1.04170,
    343.5: 0.98301,
    344.0: 0.85416,
    344.5: 0.81300,
    345.0: 0.91583,
    345.5: 0.97021,
    346.0: 0.94269,
    346.5: 0.95594,
    347.0: 0.96954,
    347.5: 0.93318,
    348.0: 0.92472,
    348.5: 0.93721,
    349.0: 0.89900,
    349.5: 0.91969,
    350.0: 1.01220,
    350.5: 1.08490,
    351.0: 1.05340,
    351.5: 1.01290,
    352.0: 0.98383,
    352.5: 0.92600,
    353.0: 0.98012,
    353.5: 1.07520,
    354.0: 1.13460,
    354.5: 1.14400,
    355.0: 1.14060,
    355.5: 1.09640,
    356.0: 1.02500,
    356.5: 0.95914,
    357.0: 0.84200,
    357.5: 0.85015,
    358.0: 0.78916,
    358.5: 0.73100,
    359.0: 0.85805,
    359.5: 1.03210,
    360.0: 1.08900,
    360.5: 1.02650,
    361.0: 0.94150,
    361.5: 0.91800,
    362.0: 0.95800,
    362.5: 1.04500,
    363.0: 1.07100,
    363.5: 1.03800,
    364.0: 1.07160,
    364.5: 1.05900,
    365.0: 1.09700,
    365.5: 1.20410,
    366.0: 1.28600,
    366.5: 1.28430,
    367.0: 1.25770,
    367.5: 1.23170,
    368.0: 1.15700,
    368.5: 1.14590,
    369.0: 1.19440,
    369.5: 1.27950,
    370.0: 1.29340,
    370.5: 1.16600,
    371.0: 1.18110,
    371.5: 1.22490,
    372.0: 1.14440,
    372.5: 1.08760,
    373.0: 1.04450,
    373.5: 0.93900,
    374.0: 0.93442,
    374.5: 0.92500,
    375.0: 0.98500,
    375.5: 1.08740,
    376.0: 1.12400,
    376.5: 1.10400,
    377.0: 1.18130,
    377.5: 1.31490,
    378.0: 1.41440,
    378.5: 1.37650,
    379.0: 1.22600,
    379.5: 1.09800,
    380.0: 1.15200,
    380.5: 1.23100,
    381.0: 1.24900,
    381.5: 1.12240,
    382.0: 0.95426,
    382.5: 0.82313,
    383.0: 0.73603,
    383.5: 0.71095,
    384.0: 0.82100,
    384.5: 0.98620,
    385.0: 1.08020,
    385.5: 1.02960,
    386.0: 0.99113,
    386.5: 1.02790,
    387.0: 1.03540,
    387.5: 1.01860,
    388.0: 1.00670,
    388.5: 0.99743,
    389.0: 1.08100,
    389.5: 1.19580,
    390.0: 1.25190,
    390.5: 1.26010,
    391.0: 1.33220,
    391.5: 1.34900,
    392.0: 1.24000,
    392.5: 1.03120,
    393.0: 0.74500,
    393.5: 0.59120,
    394.0: 0.76675,
    394.5: 1.05570,
    395.0: 1.24500,
    395.5: 1.32450,
    396.0: 1.16260,
    396.5: 0.84330,
    397.0: 0.65200,
    397.5: 0.96142,
    398.0: 1.30100,
    398.5: 1.53420,
    399.0: 1.62450,
    399.5: 1.67170,
    400.0: 1.68850,
    401.0: 1.75200,
    402.0: 1.81400,
    403.0: 1.74000,
    404.0: 1.76300,
    405.0: 1.71500,
    406.0: 1.66600,
    407.0: 1.63000,
    408.0: 1.69900,
    409.0: 1.80900,
    410.0: 1.53700,
    411.0: 1.71500,
    412.0: 1.81600,
    413.0: 1.73920,
    414.0: 1.71440,
    415.0: 1.76880,
    416.0: 1.81500,
    417.0: 1.76600,
    418.0: 1.68500,
    419.0: 1.74900,
    420.0: 1.59900,
    421.0: 1.81100,
    422.0: 1.78200,
    423.0: 1.72100,
    424.0: 1.70800,
    425.0: 1.75500,
    426.0: 1.69900,
    427.0: 1.63800,
    428.0: 1.65100,
    429.0: 1.52300,
    430.0: 1.21200,
    431.0: 1.09900,
    432.0: 1.82200,
    433.0: 1.69130,
    434.0: 1.56000,
    435.0: 1.70900,
    436.0: 1.86800,
    437.0: 1.90000,
    438.0: 1.66300,
    439.0: 1.60100,
    440.0: 1.83000,
    441.0: 1.79900,
    442.0: 1.92200,
    443.0: 1.94900,
    444.0: 1.89410,
    445.0: 1.96500,
    446.0: 1.75570,
    447.0: 1.99000,
    448.0: 2.01400,
    449.0: 2.00100,
    450.0: 2.06900,
    451.0: 2.14200,
    452.0: 2.04700,
    453.0: 1.88640,
    454.0: 2.01800,
    455.0: 2.00100,
    456.0: 2.06300,
    457.0: 2.07700,
    458.0: 2.03200,
    459.0: 2.01200,
    460.0: 1.99730,
    461.0: 2.06390,
    462.0: 2.07800,
    463.0: 2.08400,
    464.0: 2.01500,
    465.0: 1.98400,
    466.0: 2.02100,
    467.0: 1.93100,
    468.0: 2.01200,
    469.0: 2.01800,
    470.0: 1.93900,
    471.0: 1.96900,
    472.0: 2.07000,
    473.0: 1.98820,
    474.0: 2.01200,
    475.0: 2.08000,
    476.0: 2.01200,
    477.0: 2.02500,
    478.0: 2.08600,
    479.0: 2.04000,
    480.0: 2.06800,
    481.0: 2.06100,
    482.0: 2.06230,
    483.0: 2.03100,
    484.0: 1.98900,
    485.0: 1.97900,
    486.0: 1.60100,
    487.0: 1.78900,
    488.0: 1.93500,
    489.0: 1.82240,
    490.0: 2.03200,
    491.0: 1.94900,
    492.0: 1.85600,
    493.0: 1.98300,
    494.0: 1.93390,
    495.0: 2.05100,
    496.0: 1.94900,
    497.0: 1.98000,
    498.0: 1.92400,
    499.0: 1.92300,
    500.0: 1.91600,
    501.0: 1.85800,
    502.0: 1.86000,
    503.0: 1.94900,
    504.0: 1.83300,
    505.0: 1.94720,
    506.0: 2.02500,
    507.0: 1.93540,
    508.0: 1.88000,
    509.0: 1.96500,
    510.0: 1.91000,
    511.0: 1.94100,
    512.0: 1.98900,
    513.0: 1.86600,
    514.0: 1.82400,
    515.0: 1.87500,
    516.0: 1.89100,
    517.0: 1.53900,
    518.0: 1.75900,
    519.0: 1.70400,
    520.0: 1.86000,
    521.0: 1.87300,
    522.0: 1.91500,
    523.0: 1.80400,
    524.0: 1.94100,
    525.0: 1.92800,
    526.0: 1.87400,
    527.0: 1.64100,
    528.0: 1.88000,
    529.0: 1.96900,
    530.0: 1.89200,
    531.0: 1.99500,
    532.0: 1.95800,
    533.0: 1.74700,
    534.0: 1.86900,
    535.0: 1.89500,
    536.0: 1.97400,
    537.0: 1.82400,
    538.0: 1.91300,
    539.0: 1.86400,
    540.0: 1.80000,
    541.0: 1.73400,
    542.0: 1.88800,
    543.0: 1.85100,
    544.0: 1.91900,
    545.0: 1.87400,
    546.0: 1.86090,
    547.0: 1.88200,
    548.0: 1.82600,
    549.0: 1.88000,
    550.0: 1.86300,
    551.0: 1.85900,
    552.0: 1.89600,
    553.0: 1.84200,
    554.0: 1.87800,
    555.0: 1.88900,
    556.0: 1.85700,
    557.0: 1.81200,
    558.0: 1.85300,
    559.0: 1.75500,
    560.0: 1.78600,
    561.0: 1.89000,
    562.0: 1.80100,
    563.0: 1.87100,
    564.0: 1.83600,
    565.0: 1.84900,
    566.0: 1.75000,
    567.0: 1.86800,
    568.0: 1.85900,
    569.0: 1.83100,
    570.0: 1.82800,
    571.0: 1.76200,
    572.0: 1.87200,
    573.0: 1.88100,
    574.0: 1.87300,
    575.0: 1.83400,
    576.0: 1.81800,
    577.0: 1.86200,
    578.0: 1.79900,
    579.0: 1.81600,
    580.0: 1.83400,
    581.0: 1.83300,
    582.0: 1.85200,
    583.0: 1.86300,
    584.0: 1.85400,
    585.0: 1.83600,
    586.0: 1.79200,
    587.0: 1.83800,
    588.0: 1.82100,
    589.0: 1.62400,
    590.0: 1.72180,
    591.0: 1.80900,
    592.0: 1.78800,
    593.0: 1.79200,
    594.0: 1.78900,
    595.0: 1.77800,
    596.0: 1.79600,
    597.0: 1.80600,
    598.0: 1.77200,
    599.0: 1.76400,
    600.0: 1.77000,
    601.0: 1.74200,
    602.0: 1.71500,
    603.0: 1.74900,
    604.0: 1.77900,
    605.0: 1.77300,
    606.0: 1.75800,
    607.0: 1.76200,
    608.0: 1.75100,
    609.0: 1.73400,
    610.0: 1.72400,
    611.0: 1.71200,
    612.0: 1.73600,
    613.0: 1.71000,
    614.0: 1.65500,
    615.0: 1.71200,
    616.0: 1.66400,
    617.0: 1.64100,
    618.0: 1.70200,
    619.0: 1.70900,
    620.0: 1.71100,
    621.0: 1.72400,
    622.0: 1.67840,
    623.0: 1.68200,
    624.0: 1.66700,
    625.0: 1.64400,
    626.0: 1.64000,
    627.0: 1.69300,
    628.0: 1.69300,
    629.0: 1.68700,
    630.0: 1.66500,
    631.0: 1.65900,
    632.0: 1.59010,
    633.0: 1.67400,
    634.0: 1.63700,
    635.0: 1.65200,
    636.0: 1.60930,
    637.0: 1.66100,
    638.0: 1.66500,
    639.0: 1.65300,
    640.0: 1.61300,
    641.0: 1.61000,
    642.0: 1.60900,
    643.0: 1.62500,
    644.0: 1.61400,
    645.0: 1.62700,
    646.0: 1.59100,
    647.0: 1.60600,
    648.0: 1.60200,
    649.0: 1.55100,
    650.0: 1.52600,
    651.0: 1.61300,
    652.0: 1.59100,
    653.0: 1.59800,
    654.0: 1.57500,
    655.0: 1.52300,
    656.0: 1.32330,
    657.0: 1.38400,
    658.0: 1.53900,
    659.0: 1.54200,
    660.0: 1.55800,
    661.0: 1.56600,
    662.0: 1.57100,
    663.0: 1.56300,
    664.0: 1.55400,
    665.0: 1.56700,
    666.0: 1.55500,
    667.0: 1.53540,
    668.0: 1.53480,
    669.0: 1.55800,
    670.0: 1.53400,
    671.0: 1.52900,
    672.0: 1.50600,
    673.0: 1.51700,
    674.0: 1.51300,
    675.0: 1.49900,
    676.0: 1.51500,
    677.0: 1.50000,
    678.0: 1.50700,
    679.0: 1.49300,
    680.0: 1.49400,
    681.0: 1.48700,
    682.0: 1.49300,
    683.0: 1.47600,
    684.0: 1.46600,
    685.0: 1.46500,
    686.0: 1.43300,
    687.0: 1.47200,
    688.0: 1.47600,
    689.0: 1.47800,
    690.0: 1.47900,
    691.0: 1.46800,
    692.0: 1.45400,
    693.0: 1.45800,
    694.0: 1.45700,
    695.0: 1.43500,
    696.0: 1.44200,
    697.0: 1.43800,
    698.0: 1.41700,
    699.0: 1.43400,
    700.0: 1.42200,
    701.0: 1.41310,
    702.0: 1.39870,
    703.0: 1.40950,
    704.0: 1.41870,
    705.0: 1.43300,
    706.0: 1.41380,
    707.0: 1.40400,
    708.0: 1.39900,
    709.0: 1.39000,
    710.0: 1.40400,
    711.0: 1.39700,
    712.0: 1.38180,
    713.0: 1.37020,
    714.0: 1.38190,
    715.0: 1.35020,
    716.0: 1.36940,
    717.0: 1.36500,
    718.0: 1.35700,
    719.0: 1.30100,
    720.0: 1.34870,
    721.0: 1.34800,
    722.0: 1.36000,
    723.0: 1.35100,
    724.0: 1.36070,
    725.0: 1.34650,
    726.0: 1.34290,
    727.0: 1.34440,
    728.0: 1.33700,
    729.0: 1.27960,
    730.0: 1.33570,
    731.0: 1.31040,
    732.0: 1.33330,
    733.0: 1.31080,
    734.0: 1.33900,
    735.0: 1.32710,
    736.0: 1.31000,
    737.0: 1.31200,
    738.0: 1.30000,
    739.0: 1.26460,
    740.0: 1.28300,
    741.0: 1.27070,
    742.0: 1.26490,
    743.0: 1.28920,
    744.0: 1.29550,
    745.0: 1.29200,
    746.0: 1.28920,
    747.0: 1.28900,
    748.0: 1.28080,
    749.0: 1.27600,
    750.0: 1.27400,
    751.0: 1.26800,
    752.0: 1.27200,
    753.0: 1.26500,
    754.0: 1.28090,
    755.0: 1.27710,
    756.0: 1.26100,
    757.0: 1.25980,
    758.0: 1.26800,
    759.0: 1.25000,
    760.0: 1.25900,
    761.0: 1.24870,
    762.0: 1.24600,
    763.0: 1.25430,
    764.0: 1.25660,
    765.0: 1.24520,
    766.0: 1.20000,
    767.0: 1.21200,
    768.0: 1.23100,
    769.0: 1.21420,
    770.0: 1.21460,
    771.0: 1.20730,
    772.0: 1.21200,
    773.0: 1.21000,
    774.0: 1.20900,
    775.0: 1.20800,
    776.0: 1.21020,
    777.0: 1.20280,
    778.0: 1.20140,
    779.0: 1.20670,
    780.0: 1.19300,
    781.0: 1.18980,
    782.0: 1.19500,
    783.0: 1.18960,
    784.0: 1.18100,
    785.0: 1.18740,
    786.0: 1.18900,
    787.0: 1.18770,
    788.0: 1.18000,
    789.0: 1.18120,
    790.0: 1.17040,
    791.0: 1.16320,
    792.0: 1.15000,
    793.0: 1.14490,
    794.0: 1.12600,
    795.0: 1.12800,
    796.0: 1.14270,
    797.0: 1.15800,
    798.0: 1.15010,
    799.0: 1.13500,
    800.0: 1.12480,
    801.0: 1.14270,
    802.0: 1.14220,
    803.0: 1.12410,
    804.0: 1.13420,
    805.0: 1.10010,
    806.0: 1.13090,
    807.0: 1.12300,
    808.0: 1.11620,
    809.0: 1.09800,
    810.0: 1.11000,
    811.0: 1.11430,
    812.0: 1.11600,
    813.0: 1.11800,
    814.0: 1.11740,
    815.0: 1.10910,
    816.0: 1.10500,
    817.0: 1.10830,
    818.0: 1.08610,
    819.0: 1.07380,
    820.0: 1.07400,
    821.0: 1.07600,
    822.0: 1.05600,
    823.0: 1.07600,
    824.0: 1.07150,
    825.0: 1.06990,
    826.0: 1.07590,
    827.0: 1.07620,
    828.0: 1.07490,
    829.0: 1.06200,
    830.0: 1.05630,
    831.0: 1.05600,
    832.0: 1.05500,
    833.0: 1.03270,
    834.0: 1.04590,
    835.0: 1.04760,
    836.0: 1.05970,
    837.0: 1.05110,
    838.0: 1.05490,
    839.0: 1.04000,
    840.0: 1.05000,
    841.0: 1.05200,
    842.0: 1.02870,
    843.0: 1.02710,
    844.0: 1.01400,
    845.0: 1.03600,
    846.0: 1.03700,
    847.0: 1.00600,
    848.0: 1.01100,
    849.0: 1.00500,
    850.0: 0.91000,
    851.0: 0.99800,
    852.0: 0.99000,
    853.0: 0.98100,
    854.0: 0.86900,
    855.0: 0.92700,
    856.0: 0.99017,
    857.0: 1.00560,
    858.0: 1.00700,
    859.0: 1.00500,
    860.0: 1.00000,
    861.0: 0.99900,
    862.0: 1.00600,
    863.0: 1.01200,
    864.0: 0.99006,
    865.0: 0.97354,
    866.0: 0.85800,
    867.0: 0.92500,
    868.0: 0.96900,
    869.0: 0.95900,
    870.0: 0.97700,
    871.0: 0.96317,
    872.0: 0.97600,
    873.0: 0.96614,
    874.0: 0.94913,
    875.0: 0.93562,
    876.0: 0.96200,
    877.0: 0.96500,
    878.0: 0.96200,
    879.0: 0.94571,
    880.0: 0.94856,
    881.0: 0.91963,
    882.0: 0.94408,
    883.0: 0.93930,
    884.0: 0.94352,
    885.0: 0.95469,
    886.0: 0.92558,
    887.0: 0.92300,
    888.0: 0.94066,
    889.0: 0.94367,
    890.0: 0.94236,
    891.0: 0.93928,
    892.0: 0.93259,
    893.0: 0.91800,
    894.0: 0.92400,
    895.0: 0.92600,
    896.0: 0.93425,
    897.0: 0.92686,
    898.0: 0.92378,
    899.0: 0.91396,
    900.0: 0.91378,
    901.0: 0.89834,
    902.0: 0.87720,
    903.0: 0.92233,
    904.0: 0.92100,
    905.0: 0.91800,
    906.0: 0.90780,
    907.0: 0.91450,
    908.0: 0.90143,
    909.0: 0.88552,
    910.0: 0.89496,
    911.0: 0.89730,
    912.0: 0.89016,
    913.0: 0.90020,
    914.0: 0.89820,
    915.0: 0.88830,
    916.0: 0.89470,
    917.0: 0.88860,
    918.0: 0.88810,
    919.0: 0.89210,
    920.0: 0.88540,
    921.0: 0.87030,
    922.0: 0.86280,
    923.0: 0.83335,
    924.0: 0.86957,
    925.0: 0.87642,
    926.0: 0.84561,
    927.0: 0.87809,
    928.0: 0.87110,
    929.0: 0.86954,
    930.0: 0.86866,
    931.0: 0.87078,
    932.0: 0.86514,
    933.0: 0.86958,
    934.0: 0.86709,
    935.0: 0.85601,
    936.0: 0.85222,
    937.0: 0.85368,
    938.0: 0.85936,
    939.0: 0.85676,
    940.0: 0.84000,
    941.0: 0.83153,
    942.0: 0.80616,
    943.0: 0.83814,
    944.0: 0.81854,
    945.0: 0.82441,
    946.0: 0.83138,
    947.0: 0.82809,
    948.0: 0.83137,
    949.0: 0.83303,
    950.0: 0.82867,
    951.0: 0.82019,
    952.0: 0.82402,
    953.0: 0.82050,
    954.0: 0.81200,
    955.0: 0.76900,
    956.0: 0.80200,
    957.0: 0.80583,
    958.0: 0.81147,
    959.0: 0.81483,
    960.0: 0.80627,
    961.0: 0.79982,
    962.0: 0.80173,
    963.0: 0.79761,
    964.0: 0.78808,
    965.0: 0.79579,
    966.0: 0.78598,
    967.0: 0.79503,
    968.0: 0.78526,
    969.0: 0.78789,
    970.0: 0.79140,
    971.0: 0.78186,
    972.0: 0.78785,
    973.0: 0.78149,
    974.0: 0.76882,
    975.0: 0.77348,
    976.0: 0.78054,
    977.0: 0.77208,
    978.0: 0.77800,
    979.0: 0.76650,
    980.0: 0.77512,
    981.0: 0.77300,
    982.0: 0.77879,
    983.0: 0.77541,
    984.0: 0.76881,
    985.0: 0.77081,
    986.0: 0.76620,
    987.0: 0.76606,
    988.0: 0.76368,
    989.0: 0.75500,
    990.0: 0.75694,
    991.0: 0.76161,
    992.0: 0.75714,
    993.0: 0.75639,
    994.0: 0.75851,
    995.0: 0.75392,
    996.0: 0.75347,
    997.0: 0.74300,
    998.0: 0.74300,
    999.0: 0.74200,
    1000.0: 0.74255,
    1001.0: 0.74741,
    1002.0: 0.74677,
    1003.0: 0.73836,
    1004.0: 0.73133,
    1005.0: 0.68365,
    1006.0: 0.71770,
    1007.0: 0.73093,
    1008.0: 0.73339,
    1009.0: 0.72934,
    1010.0: 0.73055,
    1011.0: 0.72562,
    1012.0: 0.72496,
    1013.0: 0.71960,
    1014.0: 0.72265,
    1015.0: 0.71062,
    1016.0: 0.71419,
    1017.0: 0.70659,
    1018.0: 0.71566,
    1019.0: 0.69606,
    1020.0: 0.70127,
    1021.0: 0.70300,
    1022.0: 0.69300,
    1023.0: 0.69900,
    1024.0: 0.70057,
    1025.0: 0.69918,
    1026.0: 0.69943,
    1027.0: 0.69594,
    1028.0: 0.69722,
    1029.0: 0.69000,
    1030.0: 0.69208,
    1031.0: 0.68985,
    1032.0: 0.68960,
    1033.0: 0.67800,
    1034.0: 0.68181,
    1035.0: 0.68400,
    1036.0: 0.68431,
    1037.0: 0.67731,
    1038.0: 0.67400,
    1039.0: 0.67900,
    1040.0: 0.67440,
    1041.0: 0.67490,
    1042.0: 0.67516,
    1043.0: 0.66860,
    1044.0: 0.67199,
    1045.0: 0.66846,
    1046.0: 0.65131,
    1047.0: 0.66155,
    1048.0: 0.66781,
    1049.0: 0.66491,
    1050.0: 0.66117,
    1051.0: 0.66182,
    1052.0: 0.65848,
    1053.0: 0.65698,
    1054.0: 0.65474,
    1055.0: 0.65753,
    1056.0: 0.65629,
    1057.0: 0.65540,
    1058.0: 0.64937,
    1059.0: 0.63046,
    1060.0: 0.64831,
    1061.0: 0.63387,
    1062.0: 0.64651,
    1063.0: 0.63669,
    1064.0: 0.64621,
    1065.0: 0.64533,
    1066.0: 0.63139,
    1067.0: 0.63517,
    1068.0: 0.63591,
    1069.0: 0.60084,
    1070.0: 0.62165,
    1071.0: 0.62821,
    1072.0: 0.63131,
    1073.0: 0.61796,
    1074.0: 0.63063,
    1075.0: 0.61768,
    1076.0: 0.62468,
    1077.0: 0.62426,
    1078.0: 0.62531,
    1079.0: 0.61203,
    1080.0: 0.62250,
    1081.0: 0.60726,
    1082.0: 0.60249,
    1083.0: 0.60900,
    1084.0: 0.61392,
    1085.0: 0.61181,
    1086.0: 0.61570,
    1087.0: 0.59145,
    1088.0: 0.60728,
    1089.0: 0.60908,
    1090.0: 0.60442,
    1091.0: 0.60799,
    1092.0: 0.59692,
    1093.0: 0.59221,
    1094.0: 0.56065,
    1095.0: 0.58252,
    1096.0: 0.58600,
    1097.0: 0.59100,
    1098.0: 0.58700,
    1099.0: 0.58216,
    1100.0: 0.60000,
    1101.0: 0.59900,
    1102.0: 0.58202,
    1103.0: 0.59500,
    1104.0: 0.59300,
    1105.0: 0.59100,
    1106.0: 0.59000,
    1107.0: 0.58600,
    1108.0: 0.58650,
    1109.0: 0.58550,
    1110.0: 0.58700,
    1111.0: 0.58250,
    1112.0: 0.58342,
    1113.0: 0.57800,
    1114.0: 0.58400,
    1115.0: 0.58200,
    1116.0: 0.57500,
    1117.0: 0.58300,
    1118.0: 0.57222,
    1119.0: 0.57151,
    1120.0: 0.56850,
    1121.0: 0.57200,
    1122.0: 0.57500,
    1123.0: 0.57400,
    1124.0: 0.57250,
    1125.0: 0.57012,
    1126.0: 0.55300,
    1127.0: 0.56179,
    1128.0: 0.56829,
    1129.0: 0.56837,
    1130.0: 0.56401,
    1131.0: 0.56800,
    1132.0: 0.57000,
    1133.0: 0.56186,
    1134.0: 0.56346,
    1135.0: 0.56250,
    1136.0: 0.56481,
    1137.0: 0.55197,
    1138.0: 0.54400,
    1139.0: 0.55300,
    1140.0: 0.55573,
    1141.0: 0.54303,
    1142.0: 0.55531,
    1143.0: 0.54960,
    1144.0: 0.54501,
    1145.0: 0.55001,
    1146.0: 0.55407,
    1147.0: 0.55408,
    1148.0: 0.55032,
    1149.0: 0.54788,
    1150.0: 0.54623,
    1151.0: 0.54531,
    1152.0: 0.54170,
    1153.0: 0.54308,
    1154.0: 0.54651,
    1155.0: 0.54731,
    1156.0: 0.54022,
    1157.0: 0.54370,
    1158.0: 0.54334,
    1159.0: 0.53548,
    1160.0: 0.52928,
    1161.0: 0.51899,
    1162.0: 0.53336,
    1163.0: 0.53708,
    1164.0: 0.52684,
    1165.0: 0.53482,
    1166.0: 0.52227,
    1167.0: 0.52984,
    1168.0: 0.53415,
    1169.0: 0.51640,
    1170.0: 0.52875,
    1171.0: 0.53074,
    1172.0: 0.53063,
    1173.0: 0.52686,
    1174.0: 0.52760,
    1175.0: 0.51852,
    1176.0: 0.51036,
    1177.0: 0.52296,
    1178.0: 0.51703,
    1179.0: 0.51504,
    1180.0: 0.52199,
    1181.0: 0.51843,
    1182.0: 0.51743,
    1183.0: 0.48821,
    1184.0: 0.51098,
    1185.0: 0.51605,
    1186.0: 0.51519,
    1187.0: 0.51455,
    1188.0: 0.51012,
    1189.0: 0.48402,
    1190.0: 0.50583,
    1191.0: 0.51354,
    1192.0: 0.51175,
    1193.0: 0.51064,
    1194.0: 0.51127,
    1195.0: 0.50592,
    1196.0: 0.50561,
    1197.0: 0.50286,
    1198.0: 0.49217,
    1199.0: 0.47434,
    1200.0: 0.50005,
    1201.0: 0.50579,
    1202.0: 0.50705,
    1203.0: 0.48836,
    1204.0: 0.48696,
    1205.0: 0.50195,
    1206.0: 0.49869,
    1207.0: 0.50143,
    1208.0: 0.48962,
    1209.0: 0.46636,
    1210.0: 0.49225,
    1211.0: 0.47909,
    1212.0: 0.49064,
    1213.0: 0.49190,
    1214.0: 0.48932,
    1215.0: 0.49044,
    1216.0: 0.48878,
    1217.0: 0.48764,
    1218.0: 0.48599,
    1219.0: 0.48234,
    1220.0: 0.48433,
    1221.0: 0.48343,
    1222.0: 0.48133,
    1223.0: 0.47600,
    1224.0: 0.48224,
    1225.0: 0.47925,
    1226.0: 0.48026,
    1227.0: 0.46841,
    1228.0: 0.47443,
    1229.0: 0.47725,
    1230.0: 0.47502,
    1231.0: 0.47425,
    1232.0: 0.46896,
    1233.0: 0.47118,
    1234.0: 0.46946,
    1235.0: 0.46644,
    1236.0: 0.46863,
    1237.0: 0.46790,
    1238.0: 0.46748,
    1239.0: 0.46235,
    1240.0: 0.45968,
    1241.0: 0.46439,
    1242.0: 0.46169,
    1243.0: 0.45655,
    1244.0: 0.45535,
    1245.0: 0.45600,
    1246.0: 0.45875,
    1247.0: 0.45775,
    1248.0: 0.46076,
    1249.0: 0.46013,
    1250.0: 0.45860,
    1251.0: 0.45638,
    1252.0: 0.45195,
    1253.0: 0.45371,
    1254.0: 0.45358,
    1255.0: 0.45304,
    1256.0: 0.45192,
    1257.0: 0.44880,
    1258.0: 0.45116,
    1259.0: 0.44797,
    1260.0: 0.45018,
    1261.0: 0.44925,
    1262.0: 0.44252,
    1263.0: 0.44825,
    1264.0: 0.44198,
    1265.0: 0.44267,
    1266.0: 0.44615,
    1267.0: 0.43912,
    1268.0: 0.43594,
    1269.0: 0.44042,
    1270.0: 0.44212,
    1271.0: 0.44290,
    1272.0: 0.44130,
    1273.0: 0.44084,
    1274.0: 0.43807,
    1275.0: 0.44072,
    1276.0: 0.44192,
    1277.0: 0.44188,
    1278.0: 0.44074,
    1279.0: 0.43795,
    1280.0: 0.43543,
    1281.0: 0.42267,
    1282.0: 0.37589,
    1283.0: 0.41159,
    1284.0: 0.42667,
    1285.0: 0.42909,
    1286.0: 0.42878,
    1287.0: 0.43036,
    1288.0: 0.42684,
    1289.0: 0.42755,
    1290.0: 0.41851,
    1291.0: 0.42159,
    1292.0: 0.42278,
    1293.0: 0.42261,
    1294.0: 0.41615,
    1295.0: 0.41944,
    1296.0: 0.41979,
    1297.0: 0.41718,
    1298.0: 0.41263,
    1299.0: 0.41701,
    1300.0: 0.41537,
    1301.0: 0.41404,
    1302.0: 0.40955,
    1303.0: 0.40768,
    1304.0: 0.40632,
    1305.0: 0.41028,
    1306.0: 0.41039,
    1307.0: 0.40958,
    1308.0: 0.40873,
    1309.0: 0.40907,
    1310.0: 0.40714,
    1311.0: 0.40611,
    1312.0: 0.40315,
    1313.0: 0.39084,
    1314.0: 0.40271,
    1315.0: 0.39162,
    1316.0: 0.39421,
    1317.0: 0.40116,
    1318.0: 0.39547,
    1319.0: 0.40093,
    1320.0: 0.39875,
    1321.0: 0.39545,
    1322.0: 0.39538,
    1323.0: 0.39502,
    1324.0: 0.39516,
    1325.0: 0.39369,
    1326.0: 0.39229,
    1327.0: 0.38877,
    1328.0: 0.39062,
    1329.0: 0.37950,
    1330.0: 0.38548,
    1331.0: 0.38765,
    1332.0: 0.37850,
    1333.0: 0.38699,
    1334.0: 0.38684,
    1335.0: 0.38563,
    1336.0: 0.38231,
    1337.0: 0.38285,
    1338.0: 0.38094,
    1339.0: 0.37570,
    1340.0: 0.37439,
    1341.0: 0.38168,
    1342.0: 0.38020,
    1343.0: 0.37977,
    1344.0: 0.37986,
    1345.0: 0.37953,
    1346.0: 0.37554,
    1347.0: 0.37760,
    1348.0: 0.37664,
    1349.0: 0.37550,
    1350.0: 0.37081,
    1351.0: 0.37167,
    1352.0: 0.37533,
    1353.0: 0.37401,
    1354.0: 0.37294,
    1355.0: 0.36827,
    1356.0: 0.36876,
    1357.0: 0.36439,
    1358.0: 0.36997,
    1359.0: 0.36903,
    1360.0: 0.36464,
    1361.0: 0.37027,
    1362.0: 0.36936,
    1363.0: 0.36185,
    1364.0: 0.36683,
    1365.0: 0.36629,
    1366.0: 0.36592,
    1367.0: 0.36242,
    1368.0: 0.36557,
    1369.0: 0.35886,
    1370.0: 0.35912,
    1371.0: 0.35949,
    1372.0: 0.36313,
    1373.0: 0.36294,
    1374.0: 0.36291,
    1375.0: 0.35749,
    1376.0: 0.35350,
    1377.0: 0.35790,
    1378.0: 0.35652,
    1379.0: 0.36035,
    1380.0: 0.35682,
    1381.0: 0.35784,
    1382.0: 0.35274,
    1383.0: 0.35376,
    1384.0: 0.35594,
    1385.0: 0.35242,
    1386.0: 0.35493,
    1387.0: 0.35136,
    1388.0: 0.35277,
    1389.0: 0.35406,
    1390.0: 0.34736,
    1391.0: 0.35242,
    1392.0: 0.35179,
    1393.0: 0.35008,
    1394.0: 0.34760,
    1395.0: 0.34754,
    1396.0: 0.34943,
    1397.0: 0.34932,
    1398.0: 0.34952,
    1399.0: 0.34886,
    1400.0: 0.33896,
    1401.0: 0.33929,
    1402.0: 0.34411,
    1403.0: 0.34014,
    1404.0: 0.34106,
    1405.0: 0.34012,
    1406.0: 0.34164,
    1407.0: 0.34085,
    1408.0: 0.34007,
    1409.0: 0.34118,
    1410.0: 0.34046,
    1411.0: 0.33767,
    1412.0: 0.32920,
    1413.0: 0.33290,
    1414.0: 0.33704,
    1415.0: 0.33707,
    1416.0: 0.33503,
    1417.0: 0.33748,
    1418.0: 0.33773,
    1419.0: 0.33391,
    1420.0: 0.33270,
    1421.0: 0.33594,
    1422.0: 0.32875,
    1423.0: 0.32884,
    1424.0: 0.32772,
    1425.0: 0.33256,
    1426.0: 0.32306,
    1427.0: 0.32745,
    1428.0: 0.33036,
    1429.0: 0.32347,
    1430.0: 0.32277,
    1431.0: 0.32421,
    1432.0: 0.33114,
    1433.0: 0.33067,
    1434.0: 0.32509,
    1435.0: 0.32731,
    1436.0: 0.32662,
    1437.0: 0.32786,
    1438.0: 0.32601,
    1439.0: 0.32856,
    1440.0: 0.31269,
    1441.0: 0.31695,
    1442.0: 0.31584,
    1443.0: 0.31780,
    1444.0: 0.31911,
    1445.0: 0.31929,
    1446.0: 0.32135,
    1447.0: 0.31760,
    1448.0: 0.32096,
    1449.0: 0.32026,
    1450.0: 0.31774,
    1451.0: 0.31367,
    1452.0: 0.31355,
    1453.0: 0.31539,
    1454.0: 0.31282,
    1455.0: 0.31210,
    1456.0: 0.30956,
    1457.0: 0.31241,
    1458.0: 0.31612,
    1459.0: 0.31746,
    1460.0: 0.31367,
    1461.0: 0.31308,
    1462.0: 0.31018,
    1463.0: 0.30823,
    1464.0: 0.30820,
    1465.0: 0.31158,
    1466.0: 0.30696,
    1467.0: 0.31322,
    1468.0: 0.30750,
    1469.0: 0.31201,
    1470.0: 0.31010,
    1471.0: 0.30414,
    1472.0: 0.30581,
    1473.0: 0.30552,
    1474.0: 0.29705,
    1475.0: 0.30349,
    1476.0: 0.30369,
    1477.0: 0.30250,
    1478.0: 0.30228,
    1479.0: 0.30461,
    1480.0: 0.30626,
    1481.0: 0.30179,
    1482.0: 0.30462,
    1483.0: 0.29813,
    1484.0: 0.30172,
    1485.0: 0.30607,
    1486.0: 0.30168,
    1487.0: 0.29989,
    1488.0: 0.26525,
    1489.0: 0.29910,
    1490.0: 0.30093,
    1491.0: 0.30175,
    1492.0: 0.29773,
    1493.0: 0.30200,
    1494.0: 0.30190,
    1495.0: 0.30074,
    1496.0: 0.29140,
    1497.0: 0.29684,
    1498.0: 0.29946,
    1499.0: 0.29528,
    1500.0: 0.30077,
    1501.0: 0.29990,
    1502.0: 0.29743,
    1503.0: 0.26920,
    1504.0: 0.28319,
    1505.0: 0.27149,
    1506.0: 0.28367,
    1507.0: 0.29376,
    1508.0: 0.29028,
    1509.0: 0.28805,
    1510.0: 0.29394,
    1511.0: 0.29446,
    1512.0: 0.28845,
    1513.0: 0.28729,
    1514.0: 0.28500,
    1515.0: 0.28807,
    1516.0: 0.28720,
    1517.0: 0.28526,
    1518.0: 0.28970,
    1519.0: 0.28495,
    1520.0: 0.28786,
    1521.0: 0.28083,
    1522.0: 0.28031,
    1523.0: 0.28448,
    1524.0: 0.28460,
    1525.0: 0.27846,
    1526.0: 0.28596,
    1527.0: 0.28297,
    1528.0: 0.28490,
    1529.0: 0.28290,
    1530.0: 0.26731,
    1531.0: 0.28272,
    1532.0: 0.28263,
    1533.0: 0.28041,
    1534.0: 0.27543,
    1535.0: 0.27754,
    1536.0: 0.28058,
    1537.0: 0.28096,
    1538.0: 0.27338,
    1539.0: 0.27808,
    1540.0: 0.26857,
    1541.0: 0.27297,
    1542.0: 0.27536,
    1543.0: 0.27499,
    1544.0: 0.27716,
    1545.0: 0.27768,
    1546.0: 0.27722,
    1547.0: 0.27354,
    1548.0: 0.26698,
    1549.0: 0.27267,
    1550.0: 0.26961,
    1551.0: 0.27075,
    1552.0: 0.27400,
    1553.0: 0.27090,
    1554.0: 0.26495,
    1555.0: 0.26712,
    1556.0: 0.26292,
    1557.0: 0.27013,
    1558.0: 0.26805,
    1559.0: 0.26780,
    1560.0: 0.26546,
    1561.0: 0.26968,
    1562.0: 0.26833,
    1563.0: 0.26688,
    1564.0: 0.26307,
    1565.0: 0.26682,
    1566.0: 0.26262,
    1567.0: 0.26352,
    1568.0: 0.26042,
    1569.0: 0.26275,
    1570.0: 0.26068,
    1571.0: 0.26668,
    1572.0: 0.26694,
    1573.0: 0.26344,
    1574.0: 0.26221,
    1575.0: 0.24738,
    1576.0: 0.26031,
    1577.0: 0.23690,
    1578.0: 0.25723,
    1579.0: 0.26167,
    1580.0: 0.26381,
    1581.0: 0.26483,
    1582.0: 0.25412,
    1583.0: 0.25771,
    1584.0: 0.25449,
    1585.0: 0.26198,
    1586.0: 0.25817,
    1587.0: 0.25492,
    1588.0: 0.25048,
    1589.0: 0.23152,
    1590.0: 0.24154,
    1591.0: 0.24132,
    1592.0: 0.25286,
    1593.0: 0.25751,
    1594.0: 0.25643,
    1595.0: 0.25765,
    1596.0: 0.24452,
    1597.0: 0.24854,
    1598.0: 0.25629,
    1599.0: 0.24977,
    1600.0: 0.25259,
    1601.0: 0.24494,
    1602.0: 0.24857,
    1603.0: 0.24901,
    1604.0: 0.25002,
    1605.0: 0.24703,
    1606.0: 0.24748,
    1607.0: 0.24544,
    1608.0: 0.24999,
    1609.0: 0.24858,
    1610.0: 0.23312,
    1611.0: 0.24078,
    1612.0: 0.24125,
    1613.0: 0.24581,
    1614.0: 0.24350,
    1615.0: 0.24434,
    1616.0: 0.23248,
    1617.0: 0.23644,
    1618.0: 0.24415,
    1619.0: 0.24224,
    1620.0: 0.23407,
    1621.0: 0.23386,
    1622.0: 0.23702,
    1623.0: 0.24166,
    1624.0: 0.24240,
    1625.0: 0.23746,
    1626.0: 0.23958,
    1627.0: 0.24037,
    1628.0: 0.24149,
    1629.0: 0.24111,
    1630.0: 0.23655,
    1631.0: 0.23932,
    1632.0: 0.23884,
    1633.0: 0.23585,
    1634.0: 0.23276,
    1635.0: 0.23735,
    1636.0: 0.23676,
    1637.0: 0.22747,
    1638.0: 0.22708,
    1639.0: 0.22205,
    1640.0: 0.22571,
    1641.0: 0.22393,
    1642.0: 0.22402,
    1643.0: 0.22879,
    1644.0: 0.22572,
    1645.0: 0.22512,
    1646.0: 0.22899,
    1647.0: 0.22910,
    1648.0: 0.22898,
    1649.0: 0.22354,
    1650.0: 0.22840,
    1651.0: 0.22369,
    1652.0: 0.22802,
    1653.0: 0.22533,
    1654.0: 0.22654,
    1655.0: 0.22390,
    1656.0: 0.22607,
    1657.0: 0.22564,
    1658.0: 0.22532,
    1659.0: 0.22221,
    1660.0: 0.22412,
    1661.0: 0.22543,
    1662.0: 0.22421,
    1663.0: 0.22265,
    1664.0: 0.22252,
    1665.0: 0.22137,
    1666.0: 0.21853,
    1667.0: 0.22016,
    1668.0: 0.21617,
    1669.0: 0.21809,
    1670.0: 0.22208,
    1671.0: 0.22096,
    1672.0: 0.21210,
    1673.0: 0.21606,
    1674.0: 0.21907,
    1675.0: 0.21621,
    1676.0: 0.21319,
    1677.0: 0.21548,
    1678.0: 0.21585,
    1679.0: 0.21295,
    1680.0: 0.20641,
    1681.0: 0.19848,
    1682.0: 0.20341,
    1683.0: 0.20893,
    1684.0: 0.21198,
    1685.0: 0.21306,
    1686.0: 0.21026,
    1687.0: 0.21123,
    1688.0: 0.21081,
    1689.0: 0.20805,
    1690.0: 0.20845,
    1691.0: 0.21203,
    1692.0: 0.21211,
    1693.0: 0.21122,
    1694.0: 0.21068,
    1695.0: 0.21055,
    1696.0: 0.20962,
    1697.0: 0.20850,
    1698.0: 0.20879,
    1699.0: 0.20804,
    1700.0: 0.20539,
    1702.0: 0.20520,
    1705.0: 0.20428,
    1710.0: 0.19894,
    1715.0: 0.20189,
    1720.0: 0.19799,
    1725.0: 0.19688,
    1730.0: 0.19309,
    1735.0: 0.18200,
    1740.0: 0.19001,
    1745.0: 0.18624,
    1750.0: 0.18518,
    1755.0: 0.18509,
    1760.0: 0.18243,
    1765.0: 0.18143,
    1770.0: 0.17969,
    1775.0: 0.17600,
    1780.0: 0.17553,
    1785.0: 0.17411,
    1790.0: 0.17371,
    1795.0: 0.17102,
    1800.0: 0.16800,
    1805.0: 0.16884,
    1810.0: 0.16859,
    1815.0: 0.15500,
    1820.0: 0.16000,
    1825.0: 0.16257,
    1830.0: 0.15921,
    1835.0: 0.15760,
    1840.0: 0.15552,
    1845.0: 0.15288,
    1850.0: 0.15337,
    1855.0: 0.15126,
    1860.0: 0.14933,
    1865.0: 0.14710,
    1870.0: 0.14565,
    1875.0: 0.13231,
    1880.0: 0.14667,
    1885.0: 0.14573,
    1890.0: 0.14041,
    1895.0: 0.13842,
    1900.0: 0.14041,
    1905.0: 0.13904,
    1910.0: 0.13654,
    1915.0: 0.13611,
    1920.0: 0.13463,
    1925.0: 0.13447,
    1930.0: 0.13145,
    1935.0: 0.13243,
    1940.0: 0.12950,
    1945.0: 0.11950,
    1950.0: 0.12627,
    1955.0: 0.12833,
    1960.0: 0.12610,
    1965.0: 0.12291,
    1970.0: 0.12375,
    1975.0: 0.12150,
    1980.0: 0.11968,
    1985.0: 0.11902,
    1990.0: 0.11977,
    1995.0: 0.11684,
    2000.0: 0.11673,
    2005.0: 0.11501,
    2010.0: 0.11512,
    2015.0: 0.11399,
    2020.0: 0.11192,
    2025.0: 0.11176,
    2030.0: 0.10969,
    2035.0: 0.10915,
    2040.0: 0.10720,
    2045.0: 0.10756,
    2050.0: 0.10592,
    2055.0: 0.10471,
    2060.0: 0.10320,
    2065.0: 0.10182,
    2070.0: 0.10095,
    2075.0: 0.10095,
    2080.0: 0.09933,
    2085.0: 0.09830,
    2090.0: 0.09754,
    2095.0: 0.09599,
    2100.0: 0.09624,
    2105.0: 0.09575,
    2110.0: 0.09463,
    2115.0: 0.09385,
    2120.0: 0.09314,
    2125.0: 0.09218,
    2130.0: 0.09238,
    2135.0: 0.09112,
    2140.0: 0.09105,
    2145.0: 0.08994,
    2150.0: 0.08971,
    2155.0: 0.08888,
    2160.0: 0.08789,
    2165.0: 0.08200,
    2170.0: 0.08537,
    2175.0: 0.08576,
    2180.0: 0.08464,
    2185.0: 0.08468,
    2190.0: 0.08314,
    2195.0: 0.08347,
    2200.0: 0.08279,
    2205.0: 0.08090,
    2210.0: 0.08081,
    2215.0: 0.08041,
    2220.0: 0.07999,
    2225.0: 0.07884,
    2230.0: 0.07840,
    2235.0: 0.07793,
    2240.0: 0.07651,
    2245.0: 0.07625,
    2250.0: 0.07537,
    2255.0: 0.07434,
    2260.0: 0.07409,
    2265.0: 0.07328,
    2270.0: 0.07310,
    2275.0: 0.07263,
    2280.0: 0.07139,
    2285.0: 0.07138,
    2290.0: 0.07119,
    2295.0: 0.06930,
    2300.0: 0.06964,
    2305.0: 0.06940,
    2310.0: 0.06890,
    2315.0: 0.06815,
    2320.0: 0.06763,
    2325.0: 0.06556,
    2330.0: 0.06622,
    2335.0: 0.06572,
    2340.0: 0.06522,
    2345.0: 0.06510,
    2350.0: 0.06434,
    2355.0: 0.06260,
    2360.0: 0.06307,
    2365.0: 0.06301,
    2370.0: 0.06239,
    2375.0: 0.06141,
    2380.0: 0.06181,
    2385.0: 0.05913,
    2390.0: 0.06036,
    2395.0: 0.06007,
    2400.0: 0.05974,
    2405.0: 0.05944,
    2410.0: 0.05915,
    2415.0: 0.05698,
    2420.0: 0.05782,
    2425.0: 0.05733,
    2430.0: 0.05719,
    2435.0: 0.05647,
    2440.0: 0.05627,
    2445.0: 0.05556,
    2450.0: 0.05459,
    2455.0: 0.05416,
    2460.0: 0.05451,
    2465.0: 0.05430,
    2470.0: 0.05340,
    2475.0: 0.05337,
    2480.0: 0.05210,
    2485.0: 0.05177,
    2490.0: 0.05222,
    2495.0: 0.04889,
    2500.0: 0.05138,
    2505.0: 0.05082,
    2510.0: 0.04960,
    2515.0: 0.04876,
    2520.0: 0.04814,
    2525.0: 0.04765,
    2530.0: 0.04770,
    2535.0: 0.04712,
    2540.0: 0.04677,
    2545.0: 0.04661,
    2550.0: 0.04626,
    2555.0: 0.04593,
    2560.0: 0.04540,
    2565.0: 0.04516,
    2570.0: 0.04485,
    2575.0: 0.04448,
    2580.0: 0.04410,
    2585.0: 0.04327,
    2590.0: 0.04341,
    2595.0: 0.04304,
    2600.0: 0.04291,
    2605.0: 0.04280,
    2610.0: 0.04250,
    2615.0: 0.04228,
    2620.0: 0.04190,
    2625.0: 0.04058,
    2630.0: 0.04116,
    2635.0: 0.04096,
    2640.0: 0.04022,
    2645.0: 0.04018,
    2650.0: 0.03996,
    2655.0: 0.03959,
    2660.0: 0.03948,
    2665.0: 0.03895,
    2670.0: 0.03910,
    2675.0: 0.03861,
    2680.0: 0.03843,
    2685.0: 0.03816,
    2690.0: 0.03779,
    2695.0: 0.03746,
    2700.0: 0.03712,
    2705.0: 0.03693,
    2710.0: 0.03668,
    2715.0: 0.03656,
    2720.0: 0.03638,
    2725.0: 0.03613,
    2730.0: 0.03601,
    2735.0: 0.03574,
    2740.0: 0.03553,
    2745.0: 0.03538,
    2750.0: 0.03518,
    2755.0: 0.03486,
    2760.0: 0.03455,
    2765.0: 0.03440,
    2770.0: 0.03426,
    2775.0: 0.03393,
    2780.0: 0.03364,
    2785.0: 0.03342,
    2790.0: 0.03320,
    2795.0: 0.03293,
    2800.0: 0.03276,
    2805.0: 0.03253,
    2810.0: 0.03242,
    2815.0: 0.03226,
    2820.0: 0.03206,
    2825.0: 0.03184,
    2830.0: 0.03170,
    2835.0: 0.03145,
    2840.0: 0.03118,
    2845.0: 0.03095,
    2850.0: 0.03068,
    2855.0: 0.03043,
    2860.0: 0.03016,
    2865.0: 0.02996,
    2870.0: 0.02960,
    2875.0: 0.02940,
    2880.0: 0.02940,
    2885.0: 0.02922,
    2890.0: 0.02907,
    2895.0: 0.02896,
    2900.0: 0.02881,
    2905.0: 0.02867,
    2910.0: 0.02850,
    2915.0: 0.02822,
    2920.0: 0.02804,
    2925.0: 0.02778,
    2930.0: 0.02757,
    2935.0: 0.02733,
    2940.0: 0.02714,
    2945.0: 0.02684,
    2950.0: 0.02668,
    2955.0: 0.02656,
    2960.0: 0.02638,
    2965.0: 0.02626,
    2970.0: 0.02612,
    2975.0: 0.02601,
    2980.0: 0.02580,
    2985.0: 0.02569,
    2990.0: 0.02554,
    2995.0: 0.02540,
    3000.0: 0.02525,
    3005.0: 0.02512,
    3010.0: 0.02498,
    3015.0: 0.02486,
    3020.0: 0.02466,
    3025.0: 0.02454,
    3030.0: 0.02442,
    3035.0: 0.02421,
    3040.0: 0.02393,
    3045.0: 0.02402,
    3050.0: 0.02395,
    3055.0: 0.02386,
    3060.0: 0.02376,
    3065.0: 0.02354,
    3070.0: 0.02344,
    3075.0: 0.02335,
    3080.0: 0.02315,
    3085.0: 0.02295,
    3090.0: 0.02280,
    3095.0: 0.02263,
    3100.0: 0.02243,
    3105.0: 0.02228,
    3110.0: 0.02211,
    3115.0: 0.02177,
    3120.0: 0.02174,
    3125.0: 0.02158,
    3130.0: 0.02145,
    3135.0: 0.02136,
    3140.0: 0.02118,
    3145.0: 0.02107,
    3150.0: 0.02088,
    3155.0: 0.02083,
    3160.0: 0.02073,
    3165.0: 0.02066,
    3170.0: 0.02052,
    3175.0: 0.02047,
    3180.0: 0.02032,
    3185.0: 0.02024,
    3190.0: 0.02019,
    3195.0: 0.02008,
    3200.0: 0.01994,
    3205.0: 0.01986,
    3210.0: 0.01976,
    3215.0: 0.01962,
    3220.0: 0.01954,
    3225.0: 0.01942,
    3230.0: 0.01938,
    3235.0: 0.01926,
    3240.0: 0.01921,
    3245.0: 0.01910,
    3250.0: 0.01909,
    3255.0: 0.01890,
    3260.0: 0.01880,
    3265.0: 0.01870,
    3270.0: 0.01860,
    3275.0: 0.01848,
    3280.0: 0.01837,
    3285.0: 0.01832,
    3290.0: 0.01826,
    3295.0: 0.01803,
    3300.0: 0.01792,
    3305.0: 0.01786,
    3310.0: 0.01770,
    3315.0: 0.01760,
    3320.0: 0.01730,
    3325.0: 0.01728,
    3330.0: 0.01713,
    3335.0: 0.01703,
    3340.0: 0.01685,
    3345.0: 0.01678,
    3350.0: 0.01660,
    3355.0: 0.01650,
    3360.0: 0.01636,
    3365.0: 0.01629,
    3370.0: 0.01618,
    3375.0: 0.01611,
    3380.0: 0.01601,
    3385.0: 0.01598,
    3390.0: 0.01590,
    3395.0: 0.01580,
    3400.0: 0.01564,
    3405.0: 0.01565,
    3410.0: 0.01554,
    3415.0: 0.01547,
    3420.0: 0.01543,
    3425.0: 0.01532,
    3430.0: 0.01522,
    3435.0: 0.01515,
    3440.0: 0.01505,
    3445.0: 0.01502,
    3450.0: 0.01492,
    3455.0: 0.01488,
    3460.0: 0.01480,
    3465.0: 0.01476,
    3470.0: 0.01466,
    3475.0: 0.01456,
    3480.0: 0.01448,
    3485.0: 0.01445,
    3490.0: 0.01436,
    3495.0: 0.01432,
    3500.0: 0.01423,
    3505.0: 0.01415,
    3510.0: 0.01405,
    3515.0: 0.01402,
    3520.0: 0.01393,
    3525.0: 0.01389,
    3530.0: 0.01380,
    3535.0: 0.01376,
    3540.0: 0.01367,
    3545.0: 0.01366,
    3550.0: 0.01357,
    3555.0: 0.01345,
    3560.0: 0.01336,
    3565.0: 0.01336,
    3570.0: 0.01325,
    3575.0: 0.01318,
    3580.0: 0.01308,
    3585.0: 0.01305,
    3590.0: 0.01298,
    3595.0: 0.01294,
    3600.0: 0.01285,
    3605.0: 0.01283,
    3610.0: 0.01274,
    3615.0: 0.01272,
    3620.0: 0.01264,
    3625.0: 0.01261,
    3630.0: 0.01251,
    3635.0: 0.01249,
    3640.0: 0.01240,
    3645.0: 0.01236,
    3650.0: 0.01228,
    3655.0: 0.01222,
    3660.0: 0.01215,
    3665.0: 0.01215,
    3670.0: 0.01204,
    3675.0: 0.01203,
    3680.0: 0.01189,
    3685.0: 0.01183,
    3690.0: 0.01177,
    3695.0: 0.01172,
    3700.0: 0.01155,
    3705.0: 0.01160,
    3710.0: 0.01151,
    3715.0: 0.01150,
    3720.0: 0.01144,
    3725.0: 0.01143,
    3730.0: 0.01134,
    3735.0: 0.01128,
    3740.0: 0.01108,
    3745.0: 0.01108,
    3750.0: 0.01104,
    3755.0: 0.01100,
    3760.0: 0.01094,
    3765.0: 0.01093,
    3770.0: 0.01086,
    3775.0: 0.01080,
    3780.0: 0.01074,
    3785.0: 0.01073,
    3790.0: 0.01063,
    3795.0: 0.01062,
    3800.0: 0.01053,
    3805.0: 0.01053,
    3810.0: 0.01043,
    3815.0: 0.01041,
    3820.0: 0.01034,
    3825.0: 0.01033,
    3830.0: 0.01025,
    3835.0: 0.01021,
    3840.0: 0.01015,
    3845.0: 0.01013,
    3850.0: 0.01005,
    3855.0: 0.01003,
    3860.0: 0.00994,
    3865.0: 0.00991,
    3870.0: 0.00983,
    3875.0: 0.00983,
    3880.0: 0.00975,
    3885.0: 0.00970,
    3890.0: 0.00971,
    3895.0: 0.00962,
    3900.0: 0.00960,
    3905.0: 0.00959,
    3910.0: 0.00952,
    3915.0: 0.00950,
    3920.0: 0.00941,
    3925.0: 0.00940,
    3930.0: 0.00932,
    3935.0: 0.00930,
    3940.0: 0.00923,
    3945.0: 0.00920,
    3950.0: 0.00911,
    3955.0: 0.00908,
    3960.0: 0.00902,
    3965.0: 0.00901,
    3970.0: 0.00893,
    3975.0: 0.00891,
    3980.0: 0.00884,
    3985.0: 0.00880,
    3990.0: 0.00878,
    3995.0: 0.00870,
    4000.0: 0.00868}

ASTM_G_173_ETR = SpectralPowerDistribution('ASTM G-173 ETR',
                                           ASTM_G_173_ETR_DATA)

"""
Extraterrestrial Radiation (solar spectrum at top of atmosphere) at mean
Earth-Sun distance.

ASTM_G_173_ETR : SpectralPowerDistribution
"""
