#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
Defines unit tests for :mod:`colour.notation.munsell` module.
"""

from __future__ import division, unicode_literals

import numpy as np
import unittest

from colour.notation.munsell import (
    parse_munsell_colour,
    is_grey_munsell_colour,
    normalize_munsell_specification)
from colour.notation.munsell import (
    munsell_colour_to_munsell_specification,
    munsell_specification_to_munsell_colour)
from colour.notation.munsell import (
    xyY_from_renotation,
    is_specification_in_renotation)
from colour.notation.munsell import bounding_hues_from_renotation
from colour.notation.munsell import hue_to_hue_angle, hue_angle_to_hue
from colour.notation.munsell import hue_to_ASTM_hue
from colour.notation.munsell import (
    interpolation_method_from_renotation_ovoid,
    xy_from_renotation_ovoid)
from colour.notation.munsell import LCHab_to_munsell_specification
from colour.notation.munsell import maximum_chroma_from_renotation
from colour.notation.munsell import munsell_specification_to_xy
from colour.notation.munsell import (
    munsell_specification_to_xyY,
    xyY_to_munsell_specification)
from colour.notation import (
    munsell_value_Priest1920,
    munsell_value_Munsell1933,
    munsell_value_Moon1943,
    munsell_value_Saunderson1944,
    munsell_value_Ladd1955,
    munsell_value_McCamy1987,
    munsell_value_ASTMD153508)
from colour.utilities import ignore_numpy_errors

__author__ = 'Colour Developers'
__copyright__ = 'Copyright (C) 2013-2017 - Colour Developers'
__license__ = 'New BSD License - http://opensource.org/licenses/BSD-3-Clause'
__maintainer__ = 'Colour Developers'
__email__ = 'colour-science@googlegroups.com'
__status__ = 'Production'

__all__ = ['MUNSELL_SPECIFICATIONS',
           'MUNSELL_GREYS_SPECIFICATIONS',
           'MUNSELL_EVEN_SPECIFICATIONS',
           'MUNSELL_BOUNDING_HUES',
           'MUNSELL_HUE_TO_ANGLE',
           'MUNSELL_HUE_TO_ASTM_HUE',
           'MUNSELL_INTERPOLATION_METHODS',
           'MUNSELL_XY_FROM_RENOTATION_OVOID',
           'MUNSELL_SPECIFICATIONS_TO_XY',
           'MUNSELL_COLOURS_TO_XYY',
           'MUNSELL_GREYS_TO_XYY',
           'XYY_TO_MUNSELL_SPECIFICATIONS',
           'XYY_TO_MUNSELL_GREYS_SPECIFICATIONS',
           'NON_CONVERGING_XYY',
           'TestMunsellValuePriest1920',
           'TestMunsellValueMunsell1933',
           'TestMunsellValueMoon1943',
           'TestMunsellValueSaunderson1944',
           'TestMunsellValueLadd1955',
           'TestMunsellValueMcCamy1992',
           'TestMunsellValueASTMD153508',
           'TestMunsellSpecification_to_xyY',
           'TestMunsellColour_to_xyY',
           'TestxyY_to_munsell_specification',
           'TestxyY_to_munsell_colour',
           'TestParseMunsellColour',
           'TestIsGreyMunsellColour',
           'TestNormalizeMunsellSpecification',
           'TestMunsellColourToMunsellSpecification',
           'TestMunsellSpecificationToMunsellColour',
           'Test_xyY_fromRenotation',
           'TestIsSpecificationInRenotation',
           'TestBoundingHuesFromRenotation',
           'TestHueToHueAngle',
           'TestHueAngleToHue',
           'TestHueTo_ASTM_hue',
           'TestInterpolationMethodFromRenotationOvoid',
           'Test_xy_fromRenotationOvoid',
           'TestLCHabToMunsellSpecification',
           'TestMaximumChromaFromRenotation',
           'TestMunsellSpecification_to_xy']

# TODO: Investigate if tests can be simplified by using a common valid set of
# specifications.
MUNSELL_SPECIFICATIONS = np.array([
    [2.500000000000000, 7.965379847082716, 11.928546308350969, 4],
    [2.500000000000000, 6.197794822090879, 6.923610826208884, 4],
    [2.500000000000000, 5.311956978256753, 2.000000000000000, 4],
    [5.613007062442384, 8.402756538070792, 18.565908940443911, 4],
    [5.845640071004907, 8.062638664520136, 5.782325614552295, 4],
    [5.780794121059599, 3.174804081025836, 3.349208682559149, 4],
    [5.483684299639117, 3.899412099408013, 5.761459062506715, 4],
    [5.809580308813496, 5.816975143899512, 6.662613753958899, 4],
    [5.209252955662903, 2.977036448356911, 5.141472643810014, 4],
    [7.706105853911573, 2.789942201654241, 11.396648897274897, 4],
    [7.567594286746361, 9.569378264154928, 16.714918860774414, 4],
    [8.117640564564343, 2.748942965149203, 3.165356383264027, 4],
    [7.873120301231125, 2.643847262009281, 13.241107969297714, 4],
    [8.049833222142890, 2.463064987097342, 7.501924679081063, 4],
    [8.355307569391062, 2.703242274198649, 11.925441344336392, 4],
    [8.342795760577609, 1.062744669123403, 6.298818145909256, 4],
    [7.594724402006285, 1.575074512180332, 4.626613135331287, 4],
    [8.195177866085791, 8.732504313513864, 23.571122010181508, 4],
    [7.754763634912469, 8.437206137825585, 21.009449010610680, 4],
    [9.010231962978862, 6.131271188386640, 6.803370568930175, 4],
    [9.041566851651622, 6.454053198559397, 17.010037203566448, 4],
    [9.915652169827913, 8.564387976791460, 11.131082159884320, 4],
    [10.000000000000000, 8.651470349341308, 27.322046186799103, 4],
    [9.961336111598143, 8.039682739223524, 13.200098633440559, 4],
    [9.887406551063181, 8.321342653987184, 2.066096323559838, 4],
    [10.000000000000000, 3.400787121787084, 2.570093220097414, 4],
    [10.000000000000000, 3.063915609453643, 13.514066607169514, 4],
    [10.000000000000000, 5.461465491798149, 12.753899774963989, 4],
    [10.000000000000000, 5.900814094860590, 15.244598276849418, 4],
    [10.000000000000000, 5.422208705414755, 27.929001019877095, 4],
    [9.757039645743053, 5.653647411872443, 3.411287127078690, 4],
    [10.000000000000000, 5.790357134071424, 24.863601306584311, 4],
    [9.862075817629322, 4.487864213671867, 7.671968095000380, 4],
    [3.214093719801356, 9.345163595199718, 3.436793937608287, 3],
    [3.484005759599379, 9.572118958552942, 14.905079424139613, 3],
    [3.196703526060703, 9.059573376604588, 24.780031389053288, 3],
    [2.500000000000000, 9.479129956842218, 27.736581704977635, 3],
    [2.790876344933768, 8.166099921946278, 20.868304564027603, 3],
    [3.221499566897477, 5.507741920664265, 5.467726257137659, 3],
    [2.622512070432247, 5.989380652373817, 19.364472252973304, 3],
    [3.287306102484981, 5.439892524933965, 19.855724192587914, 3],
    [5.727612405003367, 3.013295327457818, 10.746642552166502, 3],
    [5.347955701149093, 3.003537709503816, 18.900471815194905, 3],
    [5.738575171320432, 3.987559993529851, 4.223160837759656, 3],
    [5.720824103581511, 1.804037523043165, 4.878068159363519, 3],
    [5.316780024484356, 1.030508013578952, 8.043957606541364, 3],
    [5.762323000831238, 1.654193495936313, 9.507411716255689, 3],
    [5.985579505387595, 2.210976567398028, 14.803434527189347, 3],
    [5.461619603420755, 2.805568235937479, 6.647154736097002, 3],
    [7.838277926195208, 2.805050016159560, 6.238528025218592, 3],
    [8.283061396817500, 2.716343821673611, 10.350825174769154, 3],
    [7.603155032355272, 6.139421295158034, 29.139541165198704, 3],
    [8.324115039527976, 6.971801555303874, 23.515778973195257, 3],
    [8.444242731246860, 6.657492305333222, 2.413084311304666, 3],
    [8.309061774521076, 6.371190719454564, 17.507252134514488, 3],
    [8.140371170680920, 2.686857386753684, 14.649933295354042, 3],
    [8.484903553213694, 2.205704517797600, 11.879562262633948, 3],
    [8.454109029623016, 2.363050628470814, 4.606317173304252, 3],
    [8.305262429168986, 5.460535517182709, 3.904507271901792, 3],
    [8.189730004579287, 5.069933398792441, 28.126992759236863, 3],
    [7.540287781074750, 5.779995612547662, 6.635319193935916, 3],
    [7.962999134236298, 5.233597701388516, 20.293354805626866, 3],
    [8.432959559038371, 5.797128354507666, 26.469970873757067, 3],
    [10.000000000000000, 9.005161484782885, 6.046995658143270, 3],
    [9.771353946056914, 9.383759836829901, 20.829752715478890, 3],
    [9.376380796522223, 9.460448204508941, 13.348522394106682, 3],
    [9.912704179532229, 4.057804958576875, 25.778231770351923, 3],
    [10.000000000000000, 4.853695964045051, 13.712247643370837, 3],
    [10.000000000000000, 4.221211292509457, 28.587923360931033, 3],
    [9.287535146732925, 4.404206868704275, 6.997389565284625, 3],
    [10.000000000000000, 5.717897422867529, 30.932435068478792, 3],
    [10.000000000000000, 5.121046242854478, 7.946854746461393, 3],
    [10.000000000000000, 5.631186501571907, 26.172410297895773, 3],
    [2.500000000000000, 6.822278767379375, 12.643410557057086, 2],
    [2.500000000000000, 3.343559643400603, 19.167537762557394, 2],
    [3.284581774573411, 3.745747765546542, 10.316761862277126, 2],
    [3.081407549428113, 3.302789020993419, 4.031683724514751, 2],
    [2.500000000000000, 9.595267222759654, 9.136435041220121, 2],
    [2.589916911553009, 9.550557855080539, 8.263133397233354, 2],
    [2.534263462549973, 9.494299074607266, 14.863663104253218, 2],
    [5.275920564662094, 9.022820187513741, 12.879135949769728, 2],
    [5.522856449128964, 9.387711396347438, 17.412586595686815, 2],
    [5.885914939777947, 9.191119089368966, 17.388086814072437, 2],
    [5.471740111697462, 9.868862187868638, 11.646848538821667, 2],
    [5.956560321967156, 4.186123335197883, 4.311690204814390, 2],
    [5.627911194894264, 4.547202429787774, 16.566819144431150, 2],
    [5.853454724533456, 4.592599799739227, 18.835069805085350, 2],
    [5.144720369630256, 5.318575486426688, 18.979172966805407, 2],
    [5.290707446388017, 6.000990946276877, 13.598520998056053, 2],
    [5.415844403197766, 6.398031110922737, 15.178617464461626, 2],
    [8.204144852288245, 5.902107978077237, 4.020177691372295, 2],
    [9.366069953403018, 3.372865386949827, 15.422766182794579, 2],
    [10.000000000000000, 3.949081763597084, 9.192387616705815, 2],
    [10.000000000000000, 3.187455579956449, 15.954247893607032, 2],
    [9.260586271537607, 3.454517733921040, 10.595175791701619, 2],
    [9.571675864670619, 3.149737124891618, 17.398847531397934, 2],
    [3.238739382175979, 4.827650915864795, 3.743510694098863, 1],
    [2.500000000000000, 4.302204354084260, 7.399343614420917, 1],
    [2.500000000000000, 4.329470943798639, 8.860840417367838, 1],
    [2.500000000000000, 7.620094327678255, 10.887265616829124, 1],
    [2.500000000000000, 7.144999653185772, 10.102335374185911, 1],
    [2.610434945585585, 7.700939489093993, 4.236171515065992, 1],
    [2.500000000000000, 8.524455647347406, 5.361363698027430, 1],
    [3.173101460658481, 8.133658146416419, 15.199536235308303, 1],
    [2.500000000000000, 7.129372162253073, 5.420260862573993, 1],
    [2.500000000000000, 7.708509850248770, 9.619364938403443, 1],
    [3.252581509053177, 7.081532543557421, 6.222406020434375, 1],
    [2.500000000000000, 7.675579449401560, 12.261808397585057, 1],
    [2.500000000000000, 3.482580786553791, 7.768505546917617, 1],
    [2.500000000000000, 3.020783157962588, 6.998840911724095, 1],
    [3.020562119690717, 3.122317490920135, 5.203087539105082, 1],
    [5.219091168761326, 3.007065595158593, 13.573887550967275, 1],
    [5.596250628047351, 2.157282552163390, 5.165106850365733, 1],
    [5.078574838897358, 2.963755264589505, 6.859942724404370, 1],
    [5.175617155844582, 2.772951703906637, 4.560800382141030, 1],
    [5.497353020782844, 5.410551418942688, 2.000000000000000, 1],
    [5.841773513544001, 5.686667624085427, 13.289365667818551, 1],
    [5.580549185463668, 6.964187735662777, 16.180320149263402, 1],
    [5.287772726922527, 6.865396694853934, 14.098946461580404, 1],
    [8.358221285614269, 4.591594256415192, 17.271563597297103, 1],
    [7.877244796359770, 4.744438140664897, 5.598934346859475, 1],
    [8.323336953587479, 4.566800376285041, 7.088152366811920, 1],
    [7.845486096299681, 4.586270737017715, 16.233795179282389, 1],
    [8.382569502344943, 4.562211644069123, 13.975124110876290, 1],
    [7.855593749782354, 3.238350356301548, 5.360435825061775, 1],
    [7.655501153733914, 3.903923881082662, 9.769593047963392, 1],
    [7.653019158008493, 6.348396270933699, 11.704589766625281, 1],
    [10.000000000000000, 2.717635329532909, 5.415846167802247, 1],
    [9.196648156004963, 8.150782934993490, 5.069223366759241, 1],
    [10.000000000000000, 6.040694822625091, 7.762802316406850, 1],
    [10.000000000000000, 6.719017792521678, 18.374375386402509, 1],
    [2.873950134580900, 3.510038900108437, 4.494521106912674, 10],
    [2.979763831715893, 8.564237486111701, 6.426710793964199, 10],
    [2.500000000000000, 8.924876646785982, 2.491252841450378, 10],
    [2.500000000000000, 8.121352187119456, 8.823379864036189, 10],
    [2.500000000000000, 4.643160393937538, 18.839339977864491, 10],
    [2.500000000000000, 4.925443059836121, 5.417711811598947, 10],
    [2.500000000000000, 8.509385882792433, 8.045356725346910, 10],
    [2.500000000000000, 2.709647356385667, 16.195810159806815, 10],
    [5.667887162619730, 1.844462206458549, 18.226010811743183, 10],
    [5.759673840199206, 1.960972599684376, 30.428731527415248, 10],
    [5.783634661463273, 1.536081970823734, 21.480194214511137, 10],
    [5.118173248862928, 1.540056335460298, 41.868473358578832, 10],
    [5.757349724389667, 1.638345335050530, 13.609604267804956, 10],
    [5.279304061296045, 4.900840641360432, 22.876127528048663, 10],
    [5.715709801059808, 4.570357108788123, 30.360213488022158, 10],
    [5.947947304520848, 4.273422536180247, 4.896643906619794, 10],
    [5.098993224817240, 4.947505227279317, 26.268750424752579, 10],
    [5.532229497629850, 4.629910893964432, 7.756449262721482, 10],
    [5.923584541768192, 4.593239396795306, 19.567605030849386, 10],
    [5.950156387030171, 2.424634993436330, 4.953666946161412, 10],
    [5.614158136535322, 2.481272758716141, 20.644953904366893, 10],
    [5.435908140730638, 2.788484759470275, 21.585064332200393, 10],
    [5.539908561343329, 2.986434402350627, 44.903699039953160, 10],
    [5.379251432099132, 2.137036038265424, 25.889074558828732, 10],
    [5.632909830682246, 5.934948211512451, 21.384042506861697, 10],
    [5.203326514932920, 5.825367195549048, 15.514467427422431, 10],
    [5.927793692134072, 5.448079050348612, 3.776639519741425, 10],
    [5.817322396187511, 5.292185862716667, 11.318041580907520, 10],
    [7.949960633591607, 2.873765731226449, 25.621368902089333, 10],
    [8.382592436810759, 2.461570417216745, 40.541271952926010, 10],
    [7.963797363322570, 2.200134671312228, 36.707318709966948, 10],
    [8.373924456610474, 2.306688315438474, 8.623846064990166, 10],
    [8.151990686473388, 2.262225123930558, 42.229127196458144, 10],
    [8.255027645326059, 9.609182815192318, 7.080986046028279, 10],
    [8.488384085232076, 8.098523111957578, 9.779628072315807, 10],
    [8.438357068876163, 2.689345228362070, 26.873452492074044, 10],
    [8.309434906530441, 2.462322901174240, 48.499663993444990, 10],
    [7.711579414965501, 2.724728645017314, 5.729859843354196, 10],
    [7.627374087940193, 2.225192393206842, 26.724973070776922, 10],
    [7.693923337226084, 2.657927412397889, 48.407897505690485, 10],
    [10.000000000000000, 6.197418391023862, 10.971953815910661, 10],
    [9.113097274740381, 6.270996638245157, 2.756464595173648, 10],
    [10.000000000000000, 9.235232580795238, 6.003388325186025, 10],
    [10.000000000000000, 5.050367446997329, 19.170756721559698, 10],
    [9.380110088755156, 5.535664930510515, 18.817507743754415, 10],
    [9.001795946577033, 7.786061808916703, 4.453854563212078, 10],
    [10.000000000000000, 7.692030956316567, 3.653159723688856, 10],
    [9.046182896421445, 3.043925987515629, 22.300946806849847, 10],
    [9.459420796383784, 3.037218855958692, 10.552556949414955, 10],
    [10.000000000000000, 3.322950626925242, 31.247622019812400, 10],
    [10.000000000000000, 3.100489343503265, 29.273434731152499, 10],
    [2.500000000000000, 7.990213836555715, 8.375074375178261, 9],
    [2.500000000000000, 7.298301069157875, 9.502846862649331, 9],
    [2.861900517122356, 7.275426002317967, 7.466126134628901, 9],
    [3.087422194135551, 8.485000561300847, 2.493857829360787, 9],
    [2.500000000000000, 3.690667859366627, 19.774716780756169, 9],
    [3.220553507003754, 3.281507210559706, 37.059380662726163, 9],
    [2.500000000000000, 3.898942841249920, 39.166418500944374, 9],
    [2.765403701684196, 3.116906918736094, 29.726535569137937, 9],
    [2.500000000000000, 3.703448940191029, 12.087654687250128, 9],
    [2.500000000000000, 3.433194385943258, 3.382852759577178, 9],
    [2.836612137080781, 3.992426583719960, 2.000000000000000, 9],
    [2.888854554705095, 3.247434603609590, 14.618307037832857, 9],
    [5.164399331990519, 6.234662742406393, 9.111465383743912, 9],
    [5.500356903003388, 6.736841239972426, 13.154128131968298, 9],
    [5.535810057742433, 6.970342536034459, 8.892716664134475, 9],
    [5.590040966343994, 3.566860968884717, 22.756612786898550, 9],
    [5.282620261743346, 3.236734032301957, 18.732823688754383, 9],
    [5.172895640160181, 3.004305123123196, 6.229254345814851, 9],
    [5.259721854731981, 3.300433342987486, 35.890872110681414, 9],
    [5.553646341595925, 3.494850834989339, 10.076683709549055, 9],
    [5.730003972159145, 2.488034141173207, 15.985698390269977, 9],
    [5.782381516990652, 2.481204541395183, 28.774618518379302, 9],
    [5.069379781665461, 6.741533325352479, 2.219484171420659, 9],
    [5.134679670979660, 6.103139133682482, 27.726398643923417, 9],
    [5.383260687864624, 5.560997841342890, 18.302295934127923, 9],
    [5.869792088464701, 5.233311379347905, 32.553432167966633, 9],
    [5.462451143540612, 5.746471808899983, 30.948864634440213, 9],
    [5.357445269639698, 5.688526671944410, 5.261434469006405, 9],
    [5.626373453003034, 5.771003693827525, 25.170846666445236, 9],
    [8.284200895164993, 2.466049819474928, 17.238899804160177, 9],
    [8.318102784124019, 2.265803572668724, 22.596147383535918, 9],
    [7.851936866242713, 7.452293353458780, 20.962374407911458, 9],
    [8.146081336032703, 7.714405906472637, 13.533962918469337, 9],
    [8.097208643162750, 7.247339841946607, 17.338991550524540, 9],
    [7.830256291991797, 6.872416994269415, 10.706822163825924, 9],
    [7.800658970688480, 6.330678323824742, 6.211375680877805, 9],
    [8.044863647118635, 6.808226317611471, 15.557155261544228, 9],
    [8.461774802909071, 4.745965252820717, 36.037296939777320, 9],
    [7.612382882207284, 4.372367470892327, 14.168690780706225, 9],
    [8.169633927695997, 4.488334738003570, 27.235846103864411, 9],
    [9.602031136015775, 5.527970638413552, 20.580635675818101, 9],
    [9.663686030178818, 5.516978463101205, 29.047658472982956, 9],
    [9.752928547364711, 5.461162553197844, 34.114931605281292, 9],
    [10.000000000000000, 5.650424904167431, 4.216215730437086, 9],
    [10.000000000000000, 5.736543677665970, 34.728526755839162, 9],
    [10.000000000000000, 5.436085484926386, 14.779627294882367, 9],
    [10.000000000000000, 5.795441557952790, 2.000000000000000, 9],
    [9.497050913948730, 5.914105479148815, 10.808854780098731, 9],
    [9.826635163465532, 1.975999288230087, 7.067114431849850, 9],
    [9.382502350301259, 4.709738717837755, 19.999476877446362, 9],
    [9.115530591819274, 4.986025386567032, 5.883436488694818, 9],
    [10.000000000000000, 4.813033015882831, 24.745870232952445, 9],
    [9.378359588580793, 4.574376802251692, 26.295787257422923, 9],
    [10.000000000000000, 2.170932245950155, 21.572576356602351, 9],
    [10.000000000000000, 2.571304614356922, 26.039872491235577, 9],
    [2.500000000000000, 2.635760551271271, 4.712138166253982, 8],
    [2.887457866682928, 2.033768154097059, 13.994896052145748, 8],
    [3.435419560439465, 2.229919086421125, 6.718989113532732, 8],
    [2.992533606273717, 1.928933557645075, 7.198014339866309, 8],
    [2.500000000000000, 1.372689060484596, 14.156726710024465, 8],
    [2.610457928897581, 1.213770481399764, 3.345815626895192, 8],
    [5.167065304553812, 7.761502840367845, 2.140948156850635, 8],
    [5.054434114346951, 7.011456904063963, 6.442157332603133, 8],
    [5.803735682450612, 8.512993454403910, 10.443841773523394, 8],
    [5.044877539779968, 6.342036003669621, 18.424428701407553, 8],
    [5.484832402621484, 6.739510598555563, 5.474777491295647, 8],
    [5.162300427200289, 6.576722169349890, 24.999056248525125, 8],
    [5.877256360743413, 6.789776791182118, 15.450444143259661, 8],
    [8.197449080109873, 2.209298497930928, 2.000000000000000, 8],
    [7.997237265754237, 2.060313094466323, 11.655829335806517, 8],
    [7.973192560907184, 8.671283074887089, 4.272886886879181, 8],
    [7.836498646186221, 8.168701526186094, 13.596658717999025, 8],
    [7.782186965908517, 9.202193528464585, 13.902105524067945, 8],
    [9.531795266771761, 5.037755377967032, 2.000000000000000, 8],
    [10.000000000000000, 5.416612103313970, 11.055624912778937, 8],
    [9.312270837393163, 7.466203120412419, 11.185222099189973, 8],
    [10.000000000000000, 7.097905887270363, 13.895455902446677, 8],
    [9.925669940032272, 4.692192166283825, 7.204078988766796, 8],
    [9.416740882402403, 4.697368796121149, 8.720116348180492, 8],
    [10.000000000000000, 4.338509514756336, 16.469698910991372, 8],
    [10.000000000000000, 6.402201264456283, 6.599237233947309, 8],
    [10.000000000000000, 5.182208073338139, 4.550269784467781, 8],
    [9.970332530519679, 5.903209540812212, 10.837022722087644, 8],
    [2.962707587174585, 9.251352163485700, 9.999116931630539, 7],
    [3.167205272899491, 9.141134617154027, 7.383624729892915, 7],
    [2.500000000000000, 5.049858089466979, 17.881593853007615, 7],
    [2.741501863896628, 5.680976628228491, 18.002908737801381, 7],
    [2.500000000000000, 5.848115418935317, 10.232668996271492, 7],
    [2.877902226185231, 5.567414385297515, 3.558203423120179, 7],
    [2.500000000000000, 5.853445073334600, 27.779995926916971, 7],
    [5.412821771284458, 2.521454920411534, 7.258040020605607, 7],
    [5.837547476050840, 2.530273181625722, 11.998261380615471, 7],
    [5.969397543974988, 4.348770633848800, 14.397906420283302, 7],
    [5.004079000563381, 4.657273345320005, 22.736677614468775, 7],
    [5.168438425945292, 4.246412717207690, 4.844860547907693, 7],
    [5.863284315202094, 4.359153796629064, 23.489710023246513, 7],
    [5.756333389411959, 8.952011225713635, 7.301135618422141, 7],
    [5.108337403014788, 8.311542024325179, 11.359771531491097, 7],
    [8.314898437378535, 9.185953513281046, 4.238233636005843, 7],
    [8.201460399608226, 4.230965415446139, 11.589840844520428, 7],
    [7.595604919273442, 4.884451138651340, 6.798265747221928, 7],
    [8.378186361828917, 9.484819582257831, 8.022357890675561, 7],
    [8.028236284464779, 9.757701617444052, 11.574198271062086, 7],
    [8.229270762113973, 8.691786353579515, 6.350022396927342, 7],
    [10.000000000000000, 3.305950965855861, 3.115225963548792, 7],
    [9.756267998308681, 3.186360651735488, 14.803384721914584, 7],
    [10.000000000000000, 3.504689167815543, 13.901609609717390, 7],
    [10.000000000000000, 8.784136629159212, 6.218490965882184, 7],
    [10.000000000000000, 8.374345283261381, 13.887493044276624, 7],
    [10.000000000000000, 4.614045878641700, 14.689071599466930, 7],
    [10.000000000000000, 8.033037300917030, 13.518172354943417, 7],
    [2.745564054714475, 1.652100185202669, 5.569110673549164, 6],
    [3.145288089149191, 5.155515834056653, 8.595832717291000, 6],
    [2.500000000000000, 4.389047661368727, 4.950679151608691, 6],
    [2.500000000000000, 4.394863837189541, 4.383231249423155, 6],
    [2.500000000000000, 1.558025251052636, 3.307282274836235, 6],
    [5.045583268005572, 8.635334543903529, 9.591945248602441, 6],
    [5.594284526041456, 8.632025269800300, 10.197201238166286, 6],
    [5.988802467213943, 8.132531816914582, 12.305951956169229, 6],
    [5.425850947396252, 5.185445600639579, 8.046156862703112, 6],
    [5.369364240119585, 5.088077743168478, 7.340573827339962, 6],
    [5.702045821590509, 5.271793984998375, 10.325652051724541, 6],
    [5.411096326958829, 5.545898372969883, 5.292034843095026, 6],
    [8.242968536635763, 9.082400742895011, 4.900205865328810, 6],
    [8.050426422258862, 9.780537958506372, 18.978339720751418, 6],
    [8.238754570485817, 8.602489911338367, 5.941330110378650, 6],
    [8.395684243897479, 4.506736427736353, 9.461515968715135, 6],
    [10.000000000000000, 5.138757136469953, 12.704963485646498, 6],
    [10.000000000000000, 5.159912610631281, 15.675370760759400, 6],
    [10.000000000000000, 5.549472965121217, 3.506573388368494, 6],
    [10.000000000000000, 5.795090421330749, 14.063922879568509, 6],
    [10.000000000000000, 6.983123234599715, 3.128443413944953, 6],
    [10.000000000000000, 6.680204754366847, 11.632405914314647, 6],
    [9.050263182466011, 6.721800647918977, 17.083676942759791, 6],
    [10.000000000000000, 6.063461620134571, 4.736966947326921, 6],
    [9.409402543801862, 6.944203630692490, 6.287660211686590, 6],
    [9.633394604006961, 7.505827554006868, 4.623044001702525, 6],
    [9.020770192275748, 7.313879416061702, 13.422245014577644, 6],
    [9.263176096861541, 7.357994930871833, 15.233295182477667, 6],
    [3.332782026387723, 7.225679089752617, 16.113419977677538, 5],
    [2.500000000000000, 5.428663116358418, 6.543649602836132, 5],
    [2.500000000000000, 2.829072524106358, 2.000000000000000, 5],
    [2.828559184243374, 8.730390823623916, 21.473258817290873, 5],
    [2.500000000000000, 8.170120100361350, 12.020108658634838, 5],
    [2.500000000000000, 8.743540456183981, 14.427904414153719, 5],
    [2.500000000000000, 4.638913962811717, 8.380243803410817, 5],
    [3.363079416671538, 4.670651645625486, 2.775509664209031, 5],
    [5.339079962653624, 8.064094823108675, 16.611574939424255, 5],
    [5.347356764781598, 8.436417621014639, 15.412165198232049, 5],
    [5.368950609634622, 7.371653807185894, 7.038165919924306, 5],
    [5.929552854535908, 6.895926920816455, 7.572813447048060, 5],
    [5.727946559508910, 6.581660847859535, 10.668172633934036, 5],
    [5.641782139668679, 6.458019104693064, 9.549016885745186, 5],
    [5.344359642058747, 2.871097758194079, 5.430489560972486, 5],
    [7.749909297802317, 4.328832721055091, 4.268933751175051, 5],
    [8.145409228909998, 4.865021714408405, 7.545633529064384, 5],
    [7.907253670159305, 5.688395096546548, 10.770986229289623, 5],
    [7.592508492261312, 5.098997604455221, 4.933568344499713, 5],
    [7.674872690410821, 5.441049019888879, 3.550245288479484, 5],
    [7.991979987062054, 6.616295483614106, 3.283701248747225, 5],
    [9.345599185286883, 7.224736586735167, 17.488521757881820, 5],
    [9.659595218511388, 7.899577776723924, 3.357217748484464, 5]])

MUNSELL_GREYS_SPECIFICATIONS = np.linspace(0, 10, 25)

MUNSELL_EVEN_SPECIFICATIONS = np.array([
    [2.5, 5.0, 12.0, 4],
    [2.5, 5.0, 32.0, 4],
    [2.5, 5.0, 22.0, 4],
    [2.5, 5.0, 32.0, 4],
    [2.5, 6.0, 18.0, 4],
    [2.5, 6.0, 32.0, 4],
    [2.5, 6.0, 6.0, 4],
    [2.5, 5.0, 42.0, 4],
    [2.5, 5.0, 26.0, 4],
    [2.5, 5.0, 48.0, 4],
    [2.5, 2.0, 14.0, 4],
    [2.5, 2.0, 14.0, 4],
    [2.5, 0.0, 14.0, 4],
    [2.5, 0.0, 2.0, 4],
    [5.0, 1.0, 46.0, 4],
    [5.0, 1.0, 38.0, 4],
    [5.0, 1.0, 12.0, 4],
    [5.0, 1.0, 10.0, 4],
    [5.0, 4.0, 16.0, 4],
    [5.0, 2.0, 44.0, 4],
    [5.0, 7.0, 2.0, 4],
    [5.0, 7.0, 8.0, 4],
    [5.0, 7.0, 32.0, 4],
    [7.5, 2.0, 28.0, 4],
    [7.5, 2.0, 12.0, 4],
    [7.5, 2.0, 34.0, 4],
    [7.5, 4.0, 24.0, 4],
    [7.5, 4.0, 10.0, 4],
    [7.5, 4.0, 18.0, 4],
    [7.5, 9.0, 44.0, 4],
    [7.5, 5.0, 12.0, 4],
    [7.5, 5.0, 40.0, 4],
    [7.5, 5.0, 30.0, 4],
    [7.5, 5.0, 12.0, 4],
    [10.0, 3.0, 38.0, 4],
    [10.0, 3.0, 16.0, 4],
    [10.0, 3.0, 32.0, 4],
    [10.0, 3.0, 44.0, 4],
    [10.0, 3.0, 42.0, 4],
    [10.0, 3.0, 34.0, 4],
    [10.0, 3.0, 18.0, 4],
    [10.0, 7.0, 10.0, 4],
    [10.0, 7.0, 40.0, 4],
    [10.0, 7.0, 12.0, 4],
    [10.0, 6.0, 42.0, 4],
    [10.0, 6.0, 6.0, 4],
    [10.0, 4.0, 40.0, 4],
    [2.5, 7.0, 28.0, 3],
    [2.5, 7.0, 26.0, 3],
    [2.5, 9.0, 44.0, 3],
    [2.5, 9.0, 26.0, 3],
    [2.5, 0.0, 32.0, 3],
    [2.5, 0.0, 26.0, 3],
    [2.5, 8.0, 30.0, 3],
    [2.5, 8.0, 30.0, 3],
    [2.5, 8.0, 6.0, 3],
    [2.5, 6.0, 32.0, 3],
    [2.5, 6.0, 12.0, 3],
    [5.0, 7.0, 28.0, 3],
    [5.0, 7.0, 26.0, 3],
    [5.0, 7.0, 46.0, 3],
    [5.0, 7.0, 10.0, 3],
    [5.0, 6.0, 10.0, 3],
    [5.0, 6.0, 44.0, 3],
    [5.0, 1.0, 2.0, 3],
    [5.0, 9.0, 34.0, 3],
    [5.0, 9.0, 30.0, 3],
    [7.5, 3.0, 12.0, 3],
    [7.5, 7.0, 26.0, 3],
    [7.5, 7.0, 18.0, 3],
    [7.5, 7.0, 42.0, 3],
    [7.5, 7.0, 20.0, 3],
    [7.5, 7.0, 16.0, 3],
    [7.5, 3.0, 36.0, 3],
    [7.5, 3.0, 38.0, 3],
    [7.5, 3.0, 14.0, 3],
    [7.5, 2.0, 30.0, 3],
    [7.5, 2.0, 12.0, 3],
    [7.5, 2.0, 8.0, 3],
    [7.5, 2.0, 6.0, 3],
    [7.5, 6.0, 34.0, 3],
    [7.5, 6.0, 12.0, 3],
    [10.0, 4.0, 14.0, 3],
    [10.0, 4.0, 40.0, 3],
    [10.0, 5.0, 2.0, 3],
    [10.0, 5.0, 26.0, 3],
    [10.0, 6.0, 40.0, 3],
    [10.0, 6.0, 46.0, 3],
    [10.0, 6.0, 18.0, 3],
    [10.0, 6.0, 38.0, 3],
    [10.0, 3.0, 16.0, 3],
    [10.0, 3.0, 32.0, 3],
    [10.0, 3.0, 26.0, 3],
    [10.0, 3.0, 22.0, 3],
    [10.0, 8.0, 2.0, 3],
    [10.0, 8.0, 10.0, 3],
    [10.0, 8.0, 12.0, 3],
    [10.0, 8.0, 18.0, 3],
    [10.0, 8.0, 44.0, 3],
    [2.5, 8.0, 2.0, 2],
    [2.5, 8.0, 42.0, 2],
    [2.5, 7.0, 34.0, 2],
    [2.5, 4.0, 36.0, 2],
    [2.5, 4.0, 34.0, 2],
    [2.5, 4.0, 22.0, 2],
    [2.5, 0.0, 42.0, 2],
    [2.5, 0.0, 32.0, 2],
    [2.5, 1.0, 28.0, 2],
    [2.5, 1.0, 2.0, 2],
    [2.5, 1.0, 24.0, 2],
    [2.5, 1.0, 12.0, 2],
    [5.0, 5.0, 22.0, 2],
    [5.0, 5.0, 46.0, 2],
    [5.0, 5.0, 24.0, 2],
    [5.0, 1.0, 48.0, 2],
    [5.0, 1.0, 12.0, 2],
    [5.0, 1.0, 16.0, 2],
    [5.0, 1.0, 2.0, 2],
    [5.0, 1.0, 18.0, 2],
    [5.0, 8.0, 28.0, 2],
    [5.0, 8.0, 32.0, 2],
    [5.0, 8.0, 24.0, 2],
    [5.0, 8.0, 38.0, 2],
    [5.0, 2.0, 24.0, 2],
    [5.0, 2.0, 4.0, 2],
    [5.0, 2.0, 32.0, 2],
    [5.0, 2.0, 38.0, 2],
    [5.0, 9.0, 36.0, 2],
    [5.0, 9.0, 34.0, 2],
    [5.0, 9.0, 4.0, 2],
    [7.5, 7.0, 28.0, 2],
    [7.5, 7.0, 10.0, 2],
    [7.5, 7.0, 48.0, 2],
    [7.5, 9.0, 48.0, 2],
    [7.5, 9.0, 48.0, 2],
    [7.5, 9.0, 30.0, 2],
    [7.5, 5.0, 42.0, 2],
    [7.5, 5.0, 46.0, 2],
    [7.5, 6.0, 26.0, 2],
    [7.5, 6.0, 28.0, 2],
    [7.5, 6.0, 22.0, 2],
    [7.5, 6.0, 10.0, 2],
    [7.5, 6.0, 32.0, 2],
    [7.5, 6.0, 32.0, 2],
    [10.0, 7.0, 10.0, 2],
    [10.0, 7.0, 30.0, 2],
    [10.0, 7.0, 30.0, 2],
    [10.0, 7.0, 14.0, 2],
    [10.0, 7.0, 10.0, 2],
    [10.0, 7.0, 12.0, 2],
    [10.0, 8.0, 12.0, 2],
    [10.0, 8.0, 28.0, 2],
    [10.0, 8.0, 42.0, 2],
    [10.0, 8.0, 4.0, 2],
    [10.0, 8.0, 10.0, 2],
    [10.0, 8.0, 22.0, 2],
    [10.0, 9.0, 6.0, 2],
    [10.0, 9.0, 38.0, 2],
    [2.5, 2.0, 18.0, 1],
    [2.5, 2.0, 24.0, 1],
    [2.5, 9.0, 18.0, 1],
    [2.5, 9.0, 28.0, 1],
    [2.5, 9.0, 20.0, 1],
    [2.5, 4.0, 14.0, 1],
    [2.5, 4.0, 36.0, 1],
    [2.5, 4.0, 26.0, 1],
    [2.5, 3.0, 22.0, 1],
    [2.5, 3.0, 42.0, 1],
    [2.5, 3.0, 32.0, 1],
    [2.5, 3.0, 16.0, 1],
    [2.5, 3.0, 38.0, 1],
    [2.5, 9.0, 2.0, 1],
    [2.5, 9.0, 8.0, 1],
    [2.5, 9.0, 26.0, 1],
    [2.5, 9.0, 42.0, 1],
    [5.0, 2.0, 2.0, 1],
    [5.0, 2.0, 14.0, 1],
    [5.0, 2.0, 8.0, 1],
    [5.0, 1.0, 20.0, 1],
    [5.0, 1.0, 32.0, 1],
    [5.0, 3.0, 48.0, 1],
    [5.0, 0.0, 42.0, 1],
    [7.5, 3.0, 2.0, 1],
    [7.5, 3.0, 36.0, 1],
    [7.5, 5.0, 32.0, 1],
    [7.5, 5.0, 20.0, 1],
    [7.5, 5.0, 34.0, 1],
    [7.5, 0.0, 6.0, 1],
    [7.5, 0.0, 12.0, 1],
    [7.5, 8.0, 48.0, 1],
    [7.5, 8.0, 32.0, 1],
    [7.5, 8.0, 4.0, 1],
    [10.0, 5.0, 22.0, 1],
    [10.0, 5.0, 18.0, 1],
    [10.0, 5.0, 46.0, 1],
    [10.0, 5.0, 12.0, 1],
    [10.0, 5.0, 30.0, 1],
    [10.0, 7.0, 36.0, 1],
    [10.0, 7.0, 30.0, 1],
    [10.0, 7.0, 20.0, 1],
    [10.0, 7.0, 38.0, 1],
    [10.0, 7.0, 20.0, 1],
    [10.0, 1.0, 18.0, 1],
    [10.0, 1.0, 10.0, 1],
    [10.0, 1.0, 18.0, 1],
    [10.0, 1.0, 20.0, 1],
    [10.0, 0.0, 12.0, 1],
    [10.0, 0.0, 46.0, 1],
    [10.0, 0.0, 38.0, 1],
    [2.5, 7.0, 40.0, 10],
    [2.5, 7.0, 22.0, 10],
    [2.5, 4.0, 12.0, 10],
    [2.5, 4.0, 32.0, 10],
    [2.5, 4.0, 36.0, 10],
    [2.5, 0.0, 20.0, 10],
    [2.5, 0.0, 30.0, 10],
    [2.5, 3.0, 40.0, 10],
    [2.5, 3.0, 10.0, 10],
    [2.5, 8.0, 42.0, 10],
    [2.5, 8.0, 4.0, 10],
    [2.5, 8.0, 44.0, 10],
    [2.5, 8.0, 32.0, 10],
    [2.5, 8.0, 24.0, 10],
    [5.0, 9.0, 42.0, 10],
    [5.0, 9.0, 18.0, 10],
    [5.0, 9.0, 2.0, 10],
    [5.0, 7.0, 46.0, 10],
    [5.0, 7.0, 42.0, 10],
    [5.0, 7.0, 34.0, 10],
    [5.0, 0.0, 46.0, 10],
    [5.0, 0.0, 8.0, 10],
    [5.0, 5.0, 28.0, 10],
    [5.0, 1.0, 4.0, 10],
    [5.0, 1.0, 10.0, 10],
    [5.0, 1.0, 26.0, 10],
    [7.5, 3.0, 26.0, 10],
    [7.5, 3.0, 42.0, 10],
    [7.5, 3.0, 36.0, 10],
    [7.5, 0.0, 16.0, 10],
    [7.5, 0.0, 40.0, 10],
    [7.5, 2.0, 4.0, 10],
    [7.5, 2.0, 14.0, 10],
    [7.5, 2.0, 46.0, 10],
    [7.5, 8.0, 38.0, 10],
    [7.5, 8.0, 6.0, 10],
    [7.5, 8.0, 24.0, 10],
    [7.5, 8.0, 20.0, 10],
    [7.5, 0.0, 48.0, 10],
    [7.5, 0.0, 20.0, 10],
    [7.5, 0.0, 46.0, 10],
    [7.5, 0.0, 38.0, 10],
    [10.0, 2.0, 32.0, 10],
    [10.0, 2.0, 10.0, 10],
    [10.0, 2.0, 30.0, 10],
    [10.0, 8.0, 14.0, 10],
    [10.0, 8.0, 24.0, 10],
    [10.0, 8.0, 44.0, 10],
    [10.0, 9.0, 28.0, 10],
    [10.0, 9.0, 36.0, 10],
    [10.0, 9.0, 12.0, 10],
    [10.0, 6.0, 20.0, 10],
    [10.0, 6.0, 46.0, 10],
    [10.0, 6.0, 20.0, 10],
    [10.0, 6.0, 28.0, 10],
    [10.0, 6.0, 16.0, 10],
    [10.0, 6.0, 44.0, 10],
    [10.0, 6.0, 28.0, 10],
    [2.5, 6.0, 6.0, 9],
    [2.5, 5.0, 24.0, 9],
    [2.5, 5.0, 6.0, 9],
    [2.5, 2.0, 42.0, 9],
    [2.5, 2.0, 24.0, 9],
    [2.5, 2.0, 36.0, 9],
    [2.5, 2.0, 42.0, 9],
    [2.5, 2.0, 16.0, 9],
    [2.5, 2.0, 22.0, 9],
    [2.5, 2.0, 26.0, 9],
    [2.5, 2.0, 36.0, 9],
    [2.5, 8.0, 30.0, 9],
    [2.5, 8.0, 6.0, 9],
    [5.0, 9.0, 6.0, 9],
    [5.0, 9.0, 22.0, 9],
    [5.0, 9.0, 42.0, 9],
    [5.0, 1.0, 10.0, 9],
    [5.0, 2.0, 32.0, 9],
    [5.0, 2.0, 28.0, 9],
    [5.0, 0.0, 34.0, 9],
    [5.0, 0.0, 22.0, 9],
    [5.0, 4.0, 2.0, 9],
    [5.0, 4.0, 2.0, 9],
    [5.0, 4.0, 4.0, 9],
    [7.5, 5.0, 6.0, 9],
    [7.5, 5.0, 28.0, 9],
    [7.5, 3.0, 2.0, 9],
    [7.5, 3.0, 34.0, 9],
    [7.5, 3.0, 8.0, 9],
    [7.5, 7.0, 46.0, 9],
    [7.5, 9.0, 34.0, 9],
    [7.5, 9.0, 44.0, 9],
    [7.5, 4.0, 10.0, 9],
    [7.5, 4.0, 10.0, 9],
    [10.0, 4.0, 16.0, 9],
    [10.0, 4.0, 4.0, 9],
    [10.0, 1.0, 44.0, 9],
    [10.0, 1.0, 16.0, 9],
    [10.0, 1.0, 30.0, 9],
    [10.0, 1.0, 44.0, 9],
    [10.0, 3.0, 4.0, 9],
    [10.0, 3.0, 46.0, 9],
    [10.0, 0.0, 14.0, 9],
    [2.5, 1.0, 4.0, 8],
    [2.5, 1.0, 18.0, 8],
    [2.5, 1.0, 8.0, 8],
    [2.5, 3.0, 32.0, 8],
    [2.5, 3.0, 28.0, 8],
    [2.5, 3.0, 46.0, 8],
    [2.5, 3.0, 12.0, 8],
    [2.5, 3.0, 18.0, 8],
    [2.5, 0.0, 34.0, 8],
    [2.5, 0.0, 22.0, 8],
    [2.5, 2.0, 22.0, 8],
    [2.5, 2.0, 14.0, 8],
    [2.5, 2.0, 42.0, 8],
    [2.5, 0.0, 16.0, 8],
    [5.0, 4.0, 24.0, 8],
    [5.0, 4.0, 26.0, 8],
    [5.0, 0.0, 26.0, 8],
    [5.0, 2.0, 44.0, 8],
    [5.0, 2.0, 38.0, 8],
    [5.0, 2.0, 48.0, 8],
    [5.0, 2.0, 26.0, 8],
    [5.0, 2.0, 6.0, 8],
    [5.0, 4.0, 12.0, 8],
    [5.0, 9.0, 36.0, 8],
    [5.0, 9.0, 48.0, 8],
    [5.0, 9.0, 16.0, 8],
    [5.0, 9.0, 6.0, 8],
    [7.5, 0.0, 34.0, 8],
    [7.5, 6.0, 10.0, 8],
    [7.5, 6.0, 12.0, 8],
    [7.5, 6.0, 10.0, 8],
    [7.5, 9.0, 26.0, 8],
    [7.5, 9.0, 6.0, 8],
    [7.5, 9.0, 40.0, 8],
    [7.5, 9.0, 18.0, 8],
    [7.5, 4.0, 20.0, 8],
    [7.5, 4.0, 42.0, 8],
    [7.5, 4.0, 16.0, 8],
    [10.0, 5.0, 24.0, 8],
    [10.0, 5.0, 44.0, 8],
    [10.0, 5.0, 20.0, 8],
    [10.0, 5.0, 22.0, 8],
    [10.0, 3.0, 16.0, 8],
    [10.0, 3.0, 14.0, 8],
    [10.0, 3.0, 6.0, 8],
    [10.0, 2.0, 24.0, 8],
    [10.0, 2.0, 36.0, 8],
    [10.0, 2.0, 46.0, 8],
    [10.0, 0.0, 42.0, 8],
    [10.0, 0.0, 26.0, 8],
    [10.0, 0.0, 42.0, 8],
    [10.0, 0.0, 10.0, 8],
    [10.0, 9.0, 12.0, 8],
    [10.0, 9.0, 8.0, 8],
    [2.5, 4.0, 42.0, 7],
    [2.5, 4.0, 14.0, 7],
    [2.5, 4.0, 46.0, 7],
    [2.5, 4.0, 18.0, 7],
    [2.5, 2.0, 2.0, 7],
    [2.5, 2.0, 38.0, 7],
    [2.5, 2.0, 14.0, 7],
    [2.5, 8.0, 26.0, 7],
    [2.5, 7.0, 12.0, 7],
    [2.5, 7.0, 46.0, 7],
    [2.5, 1.0, 44.0, 7],
    [5.0, 8.0, 26.0, 7],
    [5.0, 0.0, 46.0, 7],
    [5.0, 9.0, 44.0, 7],
    [5.0, 9.0, 16.0, 7],
    [5.0, 9.0, 40.0, 7],
    [5.0, 5.0, 14.0, 7],
    [5.0, 7.0, 6.0, 7],
    [5.0, 7.0, 30.0, 7],
    [7.5, 1.0, 16.0, 7],
    [7.5, 1.0, 18.0, 7],
    [7.5, 1.0, 30.0, 7],
    [7.5, 1.0, 4.0, 7],
    [7.5, 1.0, 10.0, 7],
    [7.5, 1.0, 40.0, 7],
    [7.5, 1.0, 18.0, 7],
    [7.5, 3.0, 14.0, 7],
    [7.5, 3.0, 48.0, 7],
    [7.5, 3.0, 48.0, 7],
    [7.5, 3.0, 6.0, 7],
    [7.5, 0.0, 36.0, 7],
    [7.5, 0.0, 42.0, 7],
    [7.5, 0.0, 22.0, 7],
    [10.0, 9.0, 8.0, 7],
    [10.0, 9.0, 8.0, 7],
    [10.0, 7.0, 36.0, 7],
    [10.0, 7.0, 46.0, 7],
    [10.0, 7.0, 20.0, 7],
    [10.0, 7.0, 14.0, 7],
    [10.0, 4.0, 32.0, 7],
    [10.0, 4.0, 16.0, 7],
    [10.0, 1.0, 24.0, 7],
    [10.0, 1.0, 40.0, 7],
    [10.0, 1.0, 18.0, 7],
    [10.0, 1.0, 24.0, 7],
    [10.0, 7.0, 30.0, 7],
    [10.0, 7.0, 30.0, 7],
    [2.5, 5.0, 24.0, 6],
    [2.5, 5.0, 42.0, 6],
    [2.5, 8.0, 34.0, 6],
    [2.5, 8.0, 34.0, 6],
    [2.5, 8.0, 48.0, 6],
    [2.5, 8.0, 26.0, 6],
    [2.5, 3.0, 28.0, 6],
    [5.0, 4.0, 44.0, 6],
    [5.0, 4.0, 46.0, 6],
    [5.0, 6.0, 28.0, 6],
    [5.0, 6.0, 46.0, 6],
    [5.0, 2.0, 10.0, 6],
    [5.0, 2.0, 4.0, 6],
    [5.0, 2.0, 34.0, 6],
    [5.0, 4.0, 46.0, 6],
    [5.0, 4.0, 20.0, 6],
    [7.5, 7.0, 2.0, 6],
    [7.5, 6.0, 46.0, 6],
    [7.5, 6.0, 48.0, 6],
    [7.5, 6.0, 36.0, 6],
    [7.5, 6.0, 42.0, 6],
    [10.0, 7.0, 8.0, 6],
    [10.0, 8.0, 18.0, 6],
    [10.0, 8.0, 42.0, 6],
    [10.0, 8.0, 32.0, 6],
    [10.0, 2.0, 22.0, 6],
    [10.0, 2.0, 28.0, 6],
    [10.0, 2.0, 6.0, 6],
    [10.0, 2.0, 36.0, 6],
    [10.0, 2.0, 48.0, 6],
    [10.0, 2.0, 28.0, 6],
    [10.0, 2.0, 36.0, 6],
    [10.0, 2.0, 6.0, 6],
    [10.0, 9.0, 24.0, 6],
    [2.5, 6.0, 42.0, 5],
    [2.5, 6.0, 8.0, 5],
    [2.5, 7.0, 20.0, 5],
    [2.5, 7.0, 4.0, 5],
    [2.5, 7.0, 16.0, 5],
    [2.5, 5.0, 34.0, 5],
    [2.5, 5.0, 22.0, 5],
    [2.5, 2.0, 26.0, 5],
    [2.5, 2.0, 38.0, 5],
    [2.5, 2.0, 30.0, 5],
    [5.0, 9.0, 42.0, 5],
    [5.0, 9.0, 20.0, 5],
    [5.0, 9.0, 32.0, 5],
    [5.0, 6.0, 46.0, 5],
    [5.0, 6.0, 14.0, 5],
    [5.0, 0.0, 38.0, 5],
    [5.0, 0.0, 16.0, 5],
    [5.0, 4.0, 16.0, 5],
    [5.0, 4.0, 42.0, 5],
    [5.0, 4.0, 8.0, 5],
    [5.0, 9.0, 2.0, 5],
    [5.0, 9.0, 26.0, 5],
    [7.5, 4.0, 20.0, 5],
    [7.5, 4.0, 6.0, 5],
    [7.5, 9.0, 6.0, 5],
    [7.5, 9.0, 48.0, 5],
    [7.5, 9.0, 46.0, 5],
    [7.5, 9.0, 36.0, 5],
    [7.5, 6.0, 40.0, 5],
    [7.5, 3.0, 28.0, 5],
    [7.5, 3.0, 24.0, 5],
    [7.5, 7.0, 14.0, 5],
    [7.5, 7.0, 26.0, 5],
    [7.5, 7.0, 48.0, 5],
    [7.5, 7.0, 16.0, 5],
    [10.0, 4.0, 42.0, 5],
    [10.0, 9.0, 42.0, 5],
    [10.0, 9.0, 6.0, 5],
    [10.0, 9.0, 12.0, 5],
    [10.0, 0.0, 16.0, 5],
    [10.0, 0.0, 14.0, 5],
    [10.0, 8.0, 28.0, 5],
    [10.0, 8.0, 12.0, 5],
    [10.0, 8.0, 34.0, 5],
    [10.0, 6.0, 22.0, 5],
    [10.0, 6.0, 44.0, 5],
    [10.0, 6.0, 32.0, 5]])

MUNSELL_BOUNDING_HUES = np.array([
    [(2.5, 4.0), (2.5, 4.0)],
    [(2.5, 4.0), (2.5, 4.0)],
    [(2.5, 4.0), (2.5, 4.0)],
    [(5.0, 4.0), (7.5, 4.0)],
    [(5.0, 4.0), (7.5, 4.0)],
    [(5.0, 4.0), (7.5, 4.0)],
    [(5.0, 4.0), (7.5, 4.0)],
    [(5.0, 4.0), (7.5, 4.0)],
    [(5.0, 4.0), (7.5, 4.0)],
    [(7.5, 4.0), (10.0, 4.0)],
    [(7.5, 4.0), (10.0, 4.0)],
    [(7.5, 4.0), (10.0, 4.0)],
    [(7.5, 4.0), (10.0, 4.0)],
    [(7.5, 4.0), (10.0, 4.0)],
    [(7.5, 4.0), (10.0, 4.0)],
    [(7.5, 4.0), (10.0, 4.0)],
    [(7.5, 4.0), (10.0, 4.0)],
    [(7.5, 4.0), (10.0, 4.0)],
    [(7.5, 4.0), (10.0, 4.0)],
    [(7.5, 4.0), (10.0, 4.0)],
    [(7.5, 4.0), (10.0, 4.0)],
    [(7.5, 4.0), (10.0, 4.0)],
    [(10.0, 4.0), (10.0, 4.0)],
    [(7.5, 4.0), (10.0, 4.0)],
    [(7.5, 4.0), (10.0, 4.0)],
    [(10.0, 4.0), (10.0, 4.0)],
    [(10.0, 4.0), (10.0, 4.0)],
    [(10.0, 4.0), (10.0, 4.0)],
    [(10.0, 4.0), (10.0, 4.0)],
    [(10.0, 4.0), (10.0, 4.0)],
    [(7.5, 4.0), (10.0, 4.0)],
    [(10.0, 4.0), (10.0, 4.0)],
    [(7.5, 4.0), (10.0, 4.0)],
    [(2.5, 3.0), (5.0, 3.0)],
    [(2.5, 3.0), (5.0, 3.0)],
    [(2.5, 3.0), (5.0, 3.0)],
    [(2.5, 3.0), (2.5, 3.0)],
    [(2.5, 3.0), (5.0, 3.0)],
    [(2.5, 3.0), (5.0, 3.0)],
    [(2.5, 3.0), (5.0, 3.0)],
    [(2.5, 3.0), (5.0, 3.0)],
    [(5.0, 3.0), (7.5, 3.0)],
    [(5.0, 3.0), (7.5, 3.0)],
    [(5.0, 3.0), (7.5, 3.0)],
    [(5.0, 3.0), (7.5, 3.0)],
    [(5.0, 3.0), (7.5, 3.0)],
    [(5.0, 3.0), (7.5, 3.0)],
    [(5.0, 3.0), (7.5, 3.0)],
    [(5.0, 3.0), (7.5, 3.0)],
    [(7.5, 3.0), (10.0, 3.0)],
    [(7.5, 3.0), (10.0, 3.0)],
    [(7.5, 3.0), (10.0, 3.0)],
    [(7.5, 3.0), (10.0, 3.0)],
    [(7.5, 3.0), (10.0, 3.0)],
    [(7.5, 3.0), (10.0, 3.0)],
    [(7.5, 3.0), (10.0, 3.0)],
    [(7.5, 3.0), (10.0, 3.0)],
    [(7.5, 3.0), (10.0, 3.0)],
    [(7.5, 3.0), (10.0, 3.0)],
    [(7.5, 3.0), (10.0, 3.0)],
    [(7.5, 3.0), (10.0, 3.0)],
    [(7.5, 3.0), (10.0, 3.0)],
    [(7.5, 3.0), (10.0, 3.0)],
    [(10.0, 3.0), (10.0, 3.0)],
    [(7.5, 3.0), (10.0, 3.0)],
    [(7.5, 3.0), (10.0, 3.0)],
    [(7.5, 3.0), (10.0, 3.0)],
    [(10.0, 3.0), (10.0, 3.0)],
    [(10.0, 3.0), (10.0, 3.0)],
    [(7.5, 3.0), (10.0, 3.0)],
    [(10.0, 3.0), (10.0, 3.0)],
    [(10.0, 3.0), (10.0, 3.0)],
    [(10.0, 3.0), (10.0, 3.0)],
    [(2.5, 2.0), (2.5, 2.0)],
    [(2.5, 2.0), (2.5, 2.0)],
    [(2.5, 2.0), (5.0, 2.0)],
    [(2.5, 2.0), (5.0, 2.0)],
    [(2.5, 2.0), (2.5, 2.0)],
    [(2.5, 2.0), (5.0, 2.0)],
    [(2.5, 2.0), (5.0, 2.0)],
    [(5.0, 2.0), (7.5, 2.0)],
    [(5.0, 2.0), (7.5, 2.0)],
    [(5.0, 2.0), (7.5, 2.0)],
    [(5.0, 2.0), (7.5, 2.0)],
    [(5.0, 2.0), (7.5, 2.0)],
    [(5.0, 2.0), (7.5, 2.0)],
    [(5.0, 2.0), (7.5, 2.0)],
    [(5.0, 2.0), (7.5, 2.0)],
    [(5.0, 2.0), (7.5, 2.0)],
    [(5.0, 2.0), (7.5, 2.0)],
    [(7.5, 2.0), (10.0, 2.0)],
    [(7.5, 2.0), (10.0, 2.0)],
    [(10.0, 2.0), (10.0, 2.0)],
    [(10.0, 2.0), (10.0, 2.0)],
    [(7.5, 2.0), (10.0, 2.0)],
    [(7.5, 2.0), (10.0, 2.0)],
    [(2.5, 1.0), (5.0, 1.0)],
    [(2.5, 1.0), (2.5, 1.0)],
    [(2.5, 1.0), (2.5, 1.0)],
    [(2.5, 1.0), (2.5, 1.0)],
    [(2.5, 1.0), (2.5, 1.0)],
    [(2.5, 1.0), (5.0, 1.0)],
    [(2.5, 1.0), (2.5, 1.0)],
    [(2.5, 1.0), (5.0, 1.0)],
    [(2.5, 1.0), (2.5, 1.0)],
    [(2.5, 1.0), (2.5, 1.0)],
    [(2.5, 1.0), (5.0, 1.0)],
    [(2.5, 1.0), (2.5, 1.0)],
    [(2.5, 1.0), (2.5, 1.0)],
    [(2.5, 1.0), (2.5, 1.0)],
    [(2.5, 1.0), (5.0, 1.0)],
    [(5.0, 1.0), (7.5, 1.0)],
    [(5.0, 1.0), (7.5, 1.0)],
    [(5.0, 1.0), (7.5, 1.0)],
    [(5.0, 1.0), (7.5, 1.0)],
    [(5.0, 1.0), (7.5, 1.0)],
    [(5.0, 1.0), (7.5, 1.0)],
    [(5.0, 1.0), (7.5, 1.0)],
    [(5.0, 1.0), (7.5, 1.0)],
    [(7.5, 1.0), (10.0, 1.0)],
    [(7.5, 1.0), (10.0, 1.0)],
    [(7.5, 1.0), (10.0, 1.0)],
    [(7.5, 1.0), (10.0, 1.0)],
    [(7.5, 1.0), (10.0, 1.0)],
    [(7.5, 1.0), (10.0, 1.0)],
    [(7.5, 1.0), (10.0, 1.0)],
    [(7.5, 1.0), (10.0, 1.0)],
    [(10.0, 1.0), (10.0, 1.0)],
    [(7.5, 1.0), (10.0, 1.0)],
    [(10.0, 1.0), (10.0, 1.0)],
    [(10.0, 1.0), (10.0, 1.0)],
    [(2.5, 10.0), (5.0, 10.0)],
    [(2.5, 10.0), (5.0, 10.0)],
    [(2.5, 10.0), (2.5, 10.0)],
    [(2.5, 10.0), (2.5, 10.0)],
    [(2.5, 10.0), (2.5, 10.0)],
    [(2.5, 10.0), (2.5, 10.0)],
    [(2.5, 10.0), (2.5, 10.0)],
    [(2.5, 10.0), (2.5, 10.0)],
    [(5.0, 10.0), (7.5, 10.0)],
    [(5.0, 10.0), (7.5, 10.0)],
    [(5.0, 10.0), (7.5, 10.0)],
    [(5.0, 10.0), (7.5, 10.0)],
    [(5.0, 10.0), (7.5, 10.0)],
    [(5.0, 10.0), (7.5, 10.0)],
    [(5.0, 10.0), (7.5, 10.0)],
    [(5.0, 10.0), (7.5, 10.0)],
    [(5.0, 10.0), (7.5, 10.0)],
    [(5.0, 10.0), (7.5, 10.0)],
    [(5.0, 10.0), (7.5, 10.0)],
    [(5.0, 10.0), (7.5, 10.0)],
    [(5.0, 10.0), (7.5, 10.0)],
    [(5.0, 10.0), (7.5, 10.0)],
    [(5.0, 10.0), (7.5, 10.0)],
    [(5.0, 10.0), (7.5, 10.0)],
    [(5.0, 10.0), (7.5, 10.0)],
    [(5.0, 10.0), (7.5, 10.0)],
    [(5.0, 10.0), (7.5, 10.0)],
    [(5.0, 10.0), (7.5, 10.0)],
    [(7.5, 10.0), (10.0, 10.0)],
    [(7.5, 10.0), (10.0, 10.0)],
    [(7.5, 10.0), (10.0, 10.0)],
    [(7.5, 10.0), (10.0, 10.0)],
    [(7.5, 10.0), (10.0, 10.0)],
    [(7.5, 10.0), (10.0, 10.0)],
    [(7.5, 10.0), (10.0, 10.0)],
    [(7.5, 10.0), (10.0, 10.0)],
    [(7.5, 10.0), (10.0, 10.0)],
    [(7.5, 10.0), (10.0, 10.0)],
    [(7.5, 10.0), (10.0, 10.0)],
    [(7.5, 10.0), (10.0, 10.0)],
    [(10.0, 10.0), (10.0, 10.0)],
    [(7.5, 10.0), (10.0, 10.0)],
    [(10.0, 10.0), (10.0, 10.0)],
    [(10.0, 10.0), (10.0, 10.0)],
    [(7.5, 10.0), (10.0, 10.0)],
    [(7.5, 10.0), (10.0, 10.0)],
    [(10.0, 10.0), (10.0, 10.0)],
    [(7.5, 10.0), (10.0, 10.0)],
    [(7.5, 10.0), (10.0, 10.0)],
    [(10.0, 10.0), (10.0, 10.0)],
    [(10.0, 10.0), (10.0, 10.0)],
    [(2.5, 9.0), (2.5, 9.0)],
    [(2.5, 9.0), (2.5, 9.0)],
    [(2.5, 9.0), (5.0, 9.0)],
    [(2.5, 9.0), (5.0, 9.0)],
    [(2.5, 9.0), (2.5, 9.0)],
    [(2.5, 9.0), (5.0, 9.0)],
    [(2.5, 9.0), (2.5, 9.0)],
    [(2.5, 9.0), (5.0, 9.0)],
    [(2.5, 9.0), (2.5, 9.0)],
    [(2.5, 9.0), (2.5, 9.0)],
    [(2.5, 9.0), (5.0, 9.0)],
    [(2.5, 9.0), (5.0, 9.0)],
    [(5.0, 9.0), (7.5, 9.0)],
    [(5.0, 9.0), (7.5, 9.0)],
    [(5.0, 9.0), (7.5, 9.0)],
    [(5.0, 9.0), (7.5, 9.0)],
    [(5.0, 9.0), (7.5, 9.0)],
    [(5.0, 9.0), (7.5, 9.0)],
    [(5.0, 9.0), (7.5, 9.0)],
    [(5.0, 9.0), (7.5, 9.0)],
    [(5.0, 9.0), (7.5, 9.0)],
    [(5.0, 9.0), (7.5, 9.0)],
    [(5.0, 9.0), (7.5, 9.0)],
    [(5.0, 9.0), (7.5, 9.0)],
    [(5.0, 9.0), (7.5, 9.0)],
    [(5.0, 9.0), (7.5, 9.0)],
    [(5.0, 9.0), (7.5, 9.0)],
    [(5.0, 9.0), (7.5, 9.0)],
    [(5.0, 9.0), (7.5, 9.0)],
    [(7.5, 9.0), (10.0, 9.0)],
    [(7.5, 9.0), (10.0, 9.0)],
    [(7.5, 9.0), (10.0, 9.0)],
    [(7.5, 9.0), (10.0, 9.0)],
    [(7.5, 9.0), (10.0, 9.0)],
    [(7.5, 9.0), (10.0, 9.0)],
    [(7.5, 9.0), (10.0, 9.0)],
    [(7.5, 9.0), (10.0, 9.0)],
    [(7.5, 9.0), (10.0, 9.0)],
    [(7.5, 9.0), (10.0, 9.0)],
    [(7.5, 9.0), (10.0, 9.0)],
    [(7.5, 9.0), (10.0, 9.0)],
    [(7.5, 9.0), (10.0, 9.0)],
    [(7.5, 9.0), (10.0, 9.0)],
    [(10.0, 9.0), (10.0, 9.0)],
    [(10.0, 9.0), (10.0, 9.0)],
    [(10.0, 9.0), (10.0, 9.0)],
    [(10.0, 9.0), (10.0, 9.0)],
    [(7.5, 9.0), (10.0, 9.0)],
    [(7.5, 9.0), (10.0, 9.0)],
    [(7.5, 9.0), (10.0, 9.0)],
    [(7.5, 9.0), (10.0, 9.0)],
    [(10.0, 9.0), (10.0, 9.0)],
    [(7.5, 9.0), (10.0, 9.0)],
    [(10.0, 9.0), (10.0, 9.0)],
    [(10.0, 9.0), (10.0, 9.0)],
    [(2.5, 8.0), (2.5, 8.0)],
    [(2.5, 8.0), (5.0, 8.0)],
    [(2.5, 8.0), (5.0, 8.0)],
    [(2.5, 8.0), (5.0, 8.0)],
    [(2.5, 8.0), (2.5, 8.0)],
    [(2.5, 8.0), (5.0, 8.0)],
    [(5.0, 8.0), (7.5, 8.0)],
    [(5.0, 8.0), (7.5, 8.0)],
    [(5.0, 8.0), (7.5, 8.0)],
    [(5.0, 8.0), (7.5, 8.0)],
    [(5.0, 8.0), (7.5, 8.0)],
    [(5.0, 8.0), (7.5, 8.0)],
    [(5.0, 8.0), (7.5, 8.0)],
    [(7.5, 8.0), (10.0, 8.0)],
    [(7.5, 8.0), (10.0, 8.0)],
    [(7.5, 8.0), (10.0, 8.0)],
    [(7.5, 8.0), (10.0, 8.0)],
    [(7.5, 8.0), (10.0, 8.0)],
    [(7.5, 8.0), (10.0, 8.0)],
    [(10.0, 8.0), (10.0, 8.0)],
    [(7.5, 8.0), (10.0, 8.0)],
    [(10.0, 8.0), (10.0, 8.0)],
    [(7.5, 8.0), (10.0, 8.0)],
    [(7.5, 8.0), (10.0, 8.0)],
    [(10.0, 8.0), (10.0, 8.0)],
    [(10.0, 8.0), (10.0, 8.0)],
    [(10.0, 8.0), (10.0, 8.0)],
    [(7.5, 8.0), (10.0, 8.0)],
    [(2.5, 7.0), (5.0, 7.0)],
    [(2.5, 7.0), (5.0, 7.0)],
    [(2.5, 7.0), (2.5, 7.0)],
    [(2.5, 7.0), (5.0, 7.0)],
    [(2.5, 7.0), (2.5, 7.0)],
    [(2.5, 7.0), (5.0, 7.0)],
    [(2.5, 7.0), (2.5, 7.0)],
    [(5.0, 7.0), (7.5, 7.0)],
    [(5.0, 7.0), (7.5, 7.0)],
    [(5.0, 7.0), (7.5, 7.0)],
    [(5.0, 7.0), (7.5, 7.0)],
    [(5.0, 7.0), (7.5, 7.0)],
    [(5.0, 7.0), (7.5, 7.0)],
    [(5.0, 7.0), (7.5, 7.0)],
    [(5.0, 7.0), (7.5, 7.0)],
    [(7.5, 7.0), (10.0, 7.0)],
    [(7.5, 7.0), (10.0, 7.0)],
    [(7.5, 7.0), (10.0, 7.0)],
    [(7.5, 7.0), (10.0, 7.0)],
    [(7.5, 7.0), (10.0, 7.0)],
    [(7.5, 7.0), (10.0, 7.0)],
    [(10.0, 7.0), (10.0, 7.0)],
    [(7.5, 7.0), (10.0, 7.0)],
    [(10.0, 7.0), (10.0, 7.0)],
    [(10.0, 7.0), (10.0, 7.0)],
    [(10.0, 7.0), (10.0, 7.0)],
    [(10.0, 7.0), (10.0, 7.0)],
    [(10.0, 7.0), (10.0, 7.0)],
    [(2.5, 6.0), (5.0, 6.0)],
    [(2.5, 6.0), (5.0, 6.0)],
    [(2.5, 6.0), (2.5, 6.0)],
    [(2.5, 6.0), (2.5, 6.0)],
    [(2.5, 6.0), (2.5, 6.0)],
    [(5.0, 6.0), (7.5, 6.0)],
    [(5.0, 6.0), (7.5, 6.0)],
    [(5.0, 6.0), (7.5, 6.0)],
    [(5.0, 6.0), (7.5, 6.0)],
    [(5.0, 6.0), (7.5, 6.0)],
    [(5.0, 6.0), (7.5, 6.0)],
    [(5.0, 6.0), (7.5, 6.0)],
    [(7.5, 6.0), (10.0, 6.0)],
    [(7.5, 6.0), (10.0, 6.0)],
    [(7.5, 6.0), (10.0, 6.0)],
    [(7.5, 6.0), (10.0, 6.0)],
    [(10.0, 6.0), (10.0, 6.0)],
    [(10.0, 6.0), (10.0, 6.0)],
    [(10.0, 6.0), (10.0, 6.0)],
    [(10.0, 6.0), (10.0, 6.0)],
    [(10.0, 6.0), (10.0, 6.0)],
    [(10.0, 6.0), (10.0, 6.0)],
    [(7.5, 6.0), (10.0, 6.0)],
    [(10.0, 6.0), (10.0, 6.0)],
    [(7.5, 6.0), (10.0, 6.0)],
    [(7.5, 6.0), (10.0, 6.0)],
    [(7.5, 6.0), (10.0, 6.0)],
    [(7.5, 6.0), (10.0, 6.0)],
    [(2.5, 5.0), (5.0, 5.0)],
    [(2.5, 5.0), (2.5, 5.0)],
    [(2.5, 5.0), (2.5, 5.0)],
    [(2.5, 5.0), (5.0, 5.0)],
    [(2.5, 5.0), (2.5, 5.0)],
    [(2.5, 5.0), (2.5, 5.0)],
    [(2.5, 5.0), (2.5, 5.0)],
    [(2.5, 5.0), (5.0, 5.0)],
    [(5.0, 5.0), (7.5, 5.0)],
    [(5.0, 5.0), (7.5, 5.0)],
    [(5.0, 5.0), (7.5, 5.0)],
    [(5.0, 5.0), (7.5, 5.0)],
    [(5.0, 5.0), (7.5, 5.0)],
    [(5.0, 5.0), (7.5, 5.0)],
    [(5.0, 5.0), (7.5, 5.0)],
    [(7.5, 5.0), (10.0, 5.0)],
    [(7.5, 5.0), (10.0, 5.0)],
    [(7.5, 5.0), (10.0, 5.0)],
    [(7.5, 5.0), (10.0, 5.0)],
    [(7.5, 5.0), (10.0, 5.0)],
    [(7.5, 5.0), (10.0, 5.0)],
    [(7.5, 5.0), (10.0, 5.0)],
    [(7.5, 5.0), (10.0, 5.0)]])

MUNSELL_HUE_TO_ANGLE = np.array([
    [2.5, 1, 208.750],
    [2.5, 2, 153.750],
    [2.5, 3, 118.750],
    [2.5, 4, 63.750],
    [2.5, 5, 39.375],
    [2.5, 6, 16.875],
    [2.5, 7, 348.750],
    [2.5, 8, 300.000],
    [2.5, 9, 251.250],
    [2.5, 10, 236.250],
    [5.0, 1, 225.000],
    [5.0, 2, 160.000],
    [5.0, 3, 135.000],
    [5.0, 4, 70.000],
    [5.0, 5, 45.000],
    [5.0, 6, 22.500],
    [5.0, 7, 0.000],
    [5.0, 8, 315.000],
    [5.0, 9, 255.000],
    [5.0, 10, 240.000],
    [7.5, 1, 228.750],
    [7.5, 2, 176.250],
    [7.5, 3, 141.250],
    [7.5, 4, 86.250],
    [7.5, 5, 51.250],
    [7.5, 6, 28.125],
    [7.5, 7, 5.625],
    [7.5, 8, 326.250],
    [7.5, 9, 270.000],
    [7.5, 10, 243.750],
    [10.0, 1, 232.500],
    [10.0, 2, 192.500],
    [10.0, 3, 147.500],
    [10.0, 4, 102.500],
    [10.0, 5, 57.500],
    [10.0, 6, 33.750],
    [10.0, 7, 11.250],
    [10.0, 8, 337.500],
    [10.0, 9, 285.000],
    [10.0, 10, 247.500]])

MUNSELL_HUE_TO_ASTM_HUE = np.array([
    [2.5, 0, 72.5],
    [2.5, 1, 62.5],
    [2.5, 2, 52.5],
    [2.5, 3, 42.5],
    [2.5, 4, 32.5],
    [2.5, 5, 22.5],
    [2.5, 6, 12.5],
    [2.5, 7, 2.5],
    [2.5, 8, 92.5],
    [2.5, 9, 82.5],
    [2.5, 10, 72.5],
    [5.0, 0, 75.0],
    [5.0, 1, 65.0],
    [5.0, 2, 55.0],
    [5.0, 3, 45.0],
    [5.0, 4, 35.0],
    [5.0, 5, 25.0],
    [5.0, 6, 15.0],
    [5.0, 7, 5.0],
    [5.0, 8, 95.0],
    [5.0, 9, 85.0],
    [5.0, 10, 75.0],
    [7.5, 0, 77.5],
    [7.5, 1, 67.5],
    [7.5, 2, 57.5],
    [7.5, 3, 47.5],
    [7.5, 4, 37.5],
    [7.5, 5, 27.5],
    [7.5, 6, 17.5],
    [7.5, 7, 7.5],
    [7.5, 8, 97.5],
    [7.5, 9, 87.5],
    [7.5, 10, 77.5],
    [10.0, 0, 80.0],
    [10.0, 1, 70.0],
    [10.0, 2, 60.0],
    [10.0, 3, 50.0],
    [10.0, 4, 40.0],
    [10.0, 5, 30.0],
    [10.0, 6, 20.0],
    [10.0, 7, 10.0],
    [10.0, 8, 100.0],
    [10.0, 9, 90.0],
    [10.0, 10, 80.0]])

MUNSELL_INTERPOLATION_METHODS = (
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Linear',
    'Linear',
    None,
    None,
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Radial',
    'Linear',
    'Radial',
    'Radial',
    'Radial',
    'Linear',
    'Linear',
    'Linear',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Linear',
    'Radial',
    'Radial',
    'Radial',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    None,
    None,
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    None,
    None,
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Radial',
    'Radial',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Radial',
    'Linear',
    'Linear',
    'Radial',
    None,
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    None,
    None,
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    None,
    None,
    None,
    'Linear',
    'Linear',
    'Radial',
    'Radial',
    'Radial',
    None,
    None,
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Linear',
    'Linear',
    'Radial',
    'Linear',
    'Linear',
    'Linear',
    None,
    None,
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    None,
    None,
    'Radial',
    'Linear',
    'Linear',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    None,
    None,
    None,
    None,
    'Linear',
    'Linear',
    'Linear',
    'Radial',
    'Radial',
    'Radial',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Radial',
    'Linear',
    'Radial',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Radial',
    'Radial',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    None,
    None,
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    None,
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    None,
    None,
    'Linear',
    'Linear',
    'Linear',
    None,
    'Linear',
    'Linear',
    None,
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    None,
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    None,
    None,
    None,
    None,
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    None,
    'Linear',
    'Linear',
    'Linear',
    'Radial',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    None,
    None,
    None,
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Linear',
    'Radial',
    'Linear',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Radial',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Radial',
    'Linear',
    'Radial',
    'Radial',
    'Linear',
    'Radial',
    'Linear',
    'Radial',
    'Radial',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Radial',
    'Radial',
    None,
    None,
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Linear',
    'Radial',
    'Radial',
    'Radial',
    'Linear',
    'Linear',
    'Linear',
    'Radial',
    'Radial',
    'Radial',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Radial',
    'Linear',
    'Radial',
    'Radial',
    None,
    None,
    'Linear',
    'Radial',
    'Linear',
    'Radial',
    'Radial',
    'Radial')

MUNSELL_XY_FROM_RENOTATION_OVOID = (
    (0.4333, 0.5602),
    None,
    None,
    None,
    None,
    None,
    (0.3799, 0.4470),
    None,
    None,
    None,
    None,
    None,
    None,
    None,
    None,
    None,
    None,
    None,
    None,
    None,
    (0.3284, 0.3559),
    (0.3722, 0.4669),
    None,
    None,
    (0.2740, 0.8790),
    None,
    None,
    (0.3395, 0.5913),
    (0.3030, 0.8090),
    None,
    (0.3450, 0.5949),
    None,
    None,
    (0.3450, 0.5949),
    None,
    (0.2020, 0.8070),
    None,
    None,
    None,
    None,
    (0.1680, 0.8800),
    (0.3123, 0.4732),
    None,
    (0.3092, 0.5095),
    None,
    (0.3128, 0.4175),
    None,
    (0.1490, 0.6810),
    (0.1689, 0.6549),
    None,
    (0.2060, 0.6190),
    None,
    None,
    (0.1630, 0.6700),
    (0.1630, 0.6700),
    (0.2952, 0.3851),
    (0.0690, 0.7640),
    (0.2574, 0.4814),
    (0.1230, 0.5460),
    (0.1397, 0.5312),
    None,
    (0.2554, 0.4087),
    (0.2466, 0.4181),
    None,
    (0.2833, 0.3564),
    None,
    None,
    (0.1516, 0.4505),
    (0.1303, 0.4858),
    (0.1841, 0.4448),
    None,
    (0.1688, 0.4570),
    (0.1982, 0.4330),
    None,
    None,
    (0.1262, 0.4667),
    None,
    (0.1022, 0.4759),
    (0.1842, 0.4244),
    (0.2200, 0.3983),
    (0.0340, 0.5460),
    (0.2171, 0.4138),
    (0.1398, 0.4168),
    None,
    (0.2910, 0.3310),
    (0.0690, 0.4542),
    None,
    None,
    (0.1551, 0.4208),
    None,
    (0.0925, 0.4275),
    None,
    None,
    (0.0333, 0.4444),
    (0.2957, 0.3293),
    (0.2430, 0.3710),
    (0.2282, 0.3811),
    (0.1866, 0.4086),
    None,
    (0.2940, 0.3268),
    None,
    None,
    None,
    None,
    (0.0636, 0.3788),
    None,
    None,
    None,
    (0.26, 0.3289),
    None,
    None,
    (0.0781, 0.3211),
    None,
    (0.0670, 0.3200),
    None,
    None,
    None,
    (0.2500, 0.3141),
    None,
    None,
    None,
    (0.1220, 0.3510),
    None,
    None,
    (0.2234, 0.3150),
    None,
    None,
    None,
    None,
    (0.2768, 0.3287),
    None,
    (0.2094, 0.3165),
    None,
    None,
    None,
    None,
    None,
    None,
    (0.0680, 0.2830),
    None,
    (0.0920, 0.2900),
    (0.1961, 0.3110),
    None,
    None,
    (0.2035, 0.2956),
    None,
    None,
    (0.1671, 0.2832),
    (0.2035, 0.2956),
    (0.1841, 0.2892),
    (0.1937, 0.2978),
    None,
    None,
    (0.2686, 0.3130),
    (0.2120, 0.3025),
    (0.1180, 0.2730),
    (0.2501, 0.3118),
    None,
    None,
    None,
    None,
    None,
    None,
    (0.1027, 0.2057),
    None,
    None,
    None,
    None,
    None,
    (0.0650, 0.1700),
    None,
    (0.2909, 0.3125),
    (0.2280, 0.2960),
    None,
    None,
    (0.2559, 0.2874),
    None,
    (0.1245, 0.1827),
    None,
    None,
    None,
    None,
    (0.2616, 0.2857),
    None,
    None,
    (0.0980, 0.1460),
    None,
    None,
    None,
    None,
    None,
    (0.2688, 0.2956),
    (0.0960, 0.1260),
    (0.1203, 0.1505),
    None,
    (0.1666, 0.1964),
    None,
    None,
    None,
    (0.1280, 0.1620),
    None,
    (0.1280, 0.1620),
    None,
    (0.0840, 0.0940),
    None,
    None,
    None,
    None,
    None,
    None,
    None,
    (0.1634, 0.1698),
    None,
    None,
    None,
    None,
    None,
    (0.1576, 0.1600),
    None,
    (0.2758, 0.2879),
    None,
    None,
    None,
    None,
    None,
    (0.2991, 0.3057),
    None,
    None,
    None,
    None,
    None,
    (0.1090, 0.0790),
    (0.2012, 0.1867),
    (0.1285, 0.0870),
    (0.0950, 0.0270),
    (0.1642, 0.0655),
    (0.1570, 0.0340),
    (0.1590, 0.0440),
    None,
    None,
    (0.2420, 0.2148),
    (0.1762, 0.0955),
    (0.1610, 0.0160),
    None,
    (0.2702, 0.2648),
    None,
    None,
    None,
    None,
    None,
    None,
    (0.1918, 0.0379),
    (0.2200, 0.1330),
    (0.1925, 0.0420),
    (0.2400, 0.1960),
    None,
    None,
    None,
    None,
    None,
    (0.2140, 0.1430),
    None,
    (0.2140, 0.1430),
    (0.1940, 0.1010),
    (0.2265, 0.1671),
    None,
    (0.1940, 0.1010),
    (0.2842, 0.2550),
    (0.2372, 0.1223),
    (0.2806, 0.2444),
    (0.2180, 0.0220),
    (0.2277, 0.0621),
    (0.2200, 0.0310),
    (0.2180, 0.0220),
    (0.2372, 0.0980),
    (0.2298, 0.0696),
    (0.2260, 0.0555),
    (0.2200, 0.0310),
    None,
    (0.2881, 0.2671),
    (0.2960, 0.2710),
    None,
    None,
    (0.2701, 0.1178),
    (0.2540, 0.0390),
    (0.2559, 0.0525),
    None,
    None,
    (0.3022, 0.2825),
    (0.3022, 0.2825),
    (0.2958, 0.2565),
    (0.3093, 0.2555),
    (0.3018, 0.1253),
    (0.3088, 0.2740),
    (0.2910, 0.0600),
    (0.3037, 0.1981),
    None,
    None,
    None,
    (0.3056, 0.2060),
    (0.3056, 0.2060),
    (0.3370, 0.1756),
    (0.3210, 0.2686),
    None,
    (0.3078, 0.0839),
    None,
    None,
    (0.3214, 0.2517),
    None,
    None,
    (0.3290, 0.2095),
    (0.3370, 0.0800),
    (0.3342, 0.1551),
    None,
    (0.4140, 0.1020),
    None,
    (0.3754, 0.1898),
    (0.3929, 0.1506),
    None,
    None,
    (0.3830, 0.0960),
    (0.3711, 0.1449),
    None,
    None,
    (0.4730, 0.1720),
    (0.4820, 0.1620),
    None,
    None,
    None,
    None,
    None,
    (0.3708, 0.2380),
    (0.4104, 0.2361),
    None,
    None,
    (0.4000, 0.2630),
    (0.3431, 0.2988),
    None,
    (0.3960, 0.2860),
    (0.4125, 0.2784),
    (0.3960, 0.2860),
    None,
    (0.3512, 0.3052),
    None,
    None,
    (0.5130, 0.2101),
    None,
    (0.4799, 0.2329),
    (0.5700, 0.2400),
    None,
    (0.5396, 0.2535),
    (0.5540, 0.2460),
    (0.5628, 0.2241),
    (0.5380, 0.2369),
    (0.4218, 0.2864),
    None,
    None,
    None,
    None,
    None,
    None,
    None,
    (0.4140, 0.3020),
    (0.3760, 0.3100),
    None,
    (0.5369, 0.2810),
    None,
    (0.5898, 0.2622),
    (0.3614, 0.3033),
    None,
    (0.5734, 0.2083),
    None,
    (0.4435, 0.3119),
    None,
    None,
    None,
    None,
    None,
    None,
    None,
    (0.5341, 0.3158),
    (0.3805, 0.3244),
    None,
    None,
    None,
    None,
    (0.4660, 0.2888),
    (0.6111, 0.2290),
    None,
    None,
    (0.6492, 0.3012),
    None,
    None,
    (0.4738, 0.3316),
    None,
    None,
    None,
    (0.4160, 0.350),
    (0.4160, 0.350),
    None,
    None,
    (0.5920, 0.3740),
    (0.5234, 0.3700),
    None,
    (0.6409, 0.3533),
    None,
    None,
    None,
    None,
    None,
    None,
    None,
    None,
    None,
    None,
    None,
    (0.6020, 0.4050),
    None,
    None,
    None,
    None,
    None,
    (0.6840, 0.4150),
    (0.4674, 0.3738),
    None,
    None,
    None,
    (0.3437, 0.3397),
    None,
    None,
    None,
    None,
    (0.4399, 0.4164),
    (0.5179, 0.4670),
    None,
    None,
    None,
    None,
    (0.5450, 0.4580),
    None,
    None,
    None,
    None,
    (0.5450, 0.4580),
    (0.5320, 0.4780),
    None,
    (0.4517, 0.4421),
    (0.5160, 0.4920),
    (0.3761, 0.3800),
    (0.5049, 0.4843),
    None,
    None,
    None,
    None,
    None,
    None,
    (0.4830, 0.5092),
    None,
    None,
    (0.4905, 0.5038),
    None,
    None,
    None,
    None,
    (0.4745, 0.4810),
    (0.3378, 0.3504),
    None,
    None,
    (0.4331, 0.4688),
    (0.3811, 0.4123),
    None,
    None,
    None,
    None,
    None,
    None,
    (0.4652, 0.5128),
    None,
    None,
    (0.4728, 0.5215),
    None,
    None,
    (0.3761, 0.4155),
    (0.4271, 0.4920),
    None,
    None,
    None,
    (0.4341, 0.5020),
    None,
    None,
    None,
    None)

MUNSELL_SPECIFICATIONS_TO_XY = (
    ((2.500000000000000, 8.000000000000000, 11.928546308350969, 4),
     (0.414924832950534, 0.512356811270233)),
    ((2.500000000000000, 6.000000000000000, 6.923610826208884, 4),
     (0.389459372051262, 0.466164924643834)),
    ((3.058810707301036, 6.000000000000000, 14.501966677704569, 4),
     (0.429714278322432, 0.566581210415527)),
    ((2.500000000000000, 5.000000000000000, 2.000000000000000, 4),
     (0.335200000000000, 0.363600000000000)),
    ((5.613007062442384, 8.000000000000000, 18.565908940443911, 4),
     (0.399091079423150, 0.588883924459270)),
    ((5.845640071004907, 8.000000000000000, 5.782325614552295, 4),
     (0.347654432084958, 0.415451228906664)),
    ((5.780794121059599, 3.000000000000000, 3.349208682559149, 4),
     (0.339741340813812, 0.416892589145093)),
    ((5.483684299639117, 4.000000000000000, 5.761459062506715, 4),
     (0.362428593444426, 0.476818283660711)),
    ((5.809580308813496, 6.000000000000000, 6.662613753958899, 4),
     (0.356925102949576, 0.455404367595562)),
    ((5.209252955662903, 3.000000000000000, 5.141472643810014, 4),
     (0.363317629070254, 0.480842514232725)),
    ((7.706105853911573, 3.000000000000000, 11.396648897274897, 4),
     (0.313161079560575, 0.693666965393307)),
    ((8.117640564564343, 3.000000000000000, 3.165356383264027, 4),
     (0.319171758010897, 0.398868757129186)),
    ((7.873120301231125, 3.000000000000000, 13.241107969297714, 4),
     (0.294543112263357, 0.758491189799204)),
    ((8.049833222142890, 2.000000000000000, 7.501924679081063, 4),
     (0.307469094366649, 0.609430676657304)),
    ((8.355307569391062, 3.000000000000000, 11.925441344336392, 4),
     (0.292586452038024, 0.701304329488722)),
    ((8.342795760577609, 1.000000000000000, 6.298818145909256, 4),
     (0.256331549626150, 0.719331994133773)),
    ((7.594724402006285, 2.000000000000000, 4.626613135331287, 4),
     (0.324009937928935, 0.473539106622130)),
    ((8.195177866085791, 9.000000000000000, 23.571122010181508, 4),
     (0.339373755412936, 0.653075913883924)),
    ((7.754763634912469, 8.000000000000000, 21.009449010610680, 4),
     (0.352452265849368, 0.635758225109271)),
    ((9.010231962978862, 6.000000000000000, 6.803370568930175, 4),
     (0.321588168876590, 0.440408462323315)),
    ((9.041566851651622, 6.000000000000000, 17.010037203566448, 4),
     (0.307708766948230, 0.641958268406179)),
    ((9.915652169827913, 9.000000000000000, 11.131082159884320, 4),
     (0.315639442075870, 0.471906496338189)),
    ((10.000000000000000, 9.000000000000000, 27.322046186799103, 4),
     (0.271050792159404, 0.698186600428388)),
    ((9.961336111598143, 8.000000000000000, 13.200098633440559, 4),
     (0.311005019058988, 0.512285067166623)),
    ((9.887406551063181, 8.000000000000000, 2.066096323559838, 4),
     (0.312455696303902, 0.347019046173704)),
    ((10.000000000000000, 3.000000000000000, 2.570093220097414, 4),
     (0.307802336864830, 0.373335040247655)),
    ((10.000000000000000, 3.000000000000000, 13.514066607169514, 4),
     (0.234325574071098, 0.724733507849178)),
    ((10.000000000000000, 5.000000000000000, 12.753899774963989, 4),
     (0.290155111147684, 0.592326609857927)),
    ((10.000000000000000, 6.000000000000000, 15.244598276849418, 4),
     (0.290599307754178, 0.604905275795461)),
    ((10.000000000000000, 5.000000000000000, 27.929001019877095, 4),
     (0.154887487251536, 0.947189526006866)),
    ((9.757039645743053, 6.000000000000000, 3.411287127078690, 4),
     (0.313314005372534, 0.373444230411303)),
    ((10.000000000000000, 6.000000000000000, 24.863601306584311, 4),
     (0.232386591507202, 0.798408424825102)),
    ((9.862075817629322, 4.000000000000000, 7.671968095000380, 4),
     (0.303621095439005, 0.502074414055485)),
    ((3.214093719801356, 9.000000000000000, 3.436793937608287, 3),
     (0.300852413560114, 0.352667851495112)),
    ((3.196703526060703, 9.000000000000000, 24.780031389053288, 3),
     (0.200274406860364, 0.577845271609722)),
    ((2.500000000000000, 9.000000000000000, 27.736581704977635, 3),
     (0.193843928065156, 0.639838980459732)),
    ((2.790876344933768, 8.000000000000000, 20.868304564027603, 3),
     (0.224234903920175, 0.557123014638552)),
    ((3.221499566897477, 6.000000000000000, 5.467726257137659, 3),
     (0.287385820777856, 0.384841342730020)),
    ((2.622512070432247, 6.000000000000000, 19.364472252973304, 3),
     (0.196379669815820, 0.589608573518457)),
    ((3.287306102484981, 5.000000000000000, 19.855724192587914, 3),
     (0.151072226632175, 0.603663263146906)),
    ((5.727612405003367, 3.000000000000000, 10.746642552166502, 3),
     (0.179206769971742, 0.466532035093420)),
    ((5.347955701149093, 3.000000000000000, 18.900471815194905, 3),
     (0.075435215618251, 0.559876493385234)),
    ((5.738575171320432, 4.000000000000000, 4.223160837759656, 3),
     (0.273419119343245, 0.370375972731338)),
    ((5.720824103581511, 2.000000000000000, 4.878068159363519, 3),
     (0.246751950450133, 0.396042965876929)),
    ((5.316780024484356, 1.000000000000000, 8.043957606541364, 3),
     (0.053872420440526, 0.562315540040553)),
    ((5.762323000831238, 2.000000000000000, 9.507411716255689, 3),
     (0.162578865656637, 0.474811764395130)),
    ((5.569555328848931, 2.000000000000000, 17.594491934810442, 3),
     (-0.005438870615513, 0.598907896308392)),
    ((5.985579505387595, 2.000000000000000, 14.803434527189347, 3),
     (0.050881997214090, 0.550823564092298)),
    ((5.461619603420755, 3.000000000000000, 6.647154736097002, 3),
     (0.236839350145529, 0.414907691361922)),
    ((8.479050960745253, 3.000000000000000, 19.932170607445244, 3),
     (0.055960429731629, 0.480882860057432)),
    ((7.838277926195208, 3.000000000000000, 6.238528025218592, 3),
     (0.230069324763842, 0.389658395640305)),
    ((8.283061396817500, 3.000000000000000, 10.350825174769154, 3),
     (0.171548627931731, 0.423489618156710)),
    ((7.603155032355272, 6.000000000000000, 29.139541165198704, 3),
     (0.075525853634839, 0.516780282986480)),
    ((8.324115039527976, 7.000000000000000, 23.515778973195257, 3),
     (0.141040887200172, 0.462472690727359)),
    ((8.444242731246860, 7.000000000000000, 2.413084311304666, 3),
     (0.293504382807958, 0.334775936434053)),
    ((8.309061774521076, 6.000000000000000, 17.507252134514488, 3),
     (0.166396308607052, 0.440959226883446)),
    ((7.698664797242625, 3.000000000000000, 18.828802660207376, 3),
     (0.069764765186394, 0.495578148736382)),
    ((8.140371170680920, 3.000000000000000, 14.649933295354042, 3),
     (0.115871169503681, 0.458873914115499)),
    ((8.484903553213694, 2.000000000000000, 11.879562262633948, 3),
     (0.101184043070763, 0.451986673713637)),
    ((8.454109029623016, 2.000000000000000, 4.606317173304252, 3),
     (0.239836802119732, 0.372221966303795)),
    ((8.305262429168986, 5.000000000000000, 3.904507271901792, 3),
     (0.276294816459756, 0.350763081373635)),
    ((8.189730004579287, 5.000000000000000, 28.126992759236863, 3),
     (0.057052927340563, 0.505472465155673)),
    ((7.540287781074750, 6.000000000000000, 6.635319193935916, 3),
     (0.261247430499608, 0.371985114558266)),
    ((7.962999134236298, 5.000000000000000, 20.293354805626866, 3),
     (0.117119989260433, 0.474735735919444)),
    ((8.432959559038371, 6.000000000000000, 26.469970873757067, 3),
     (0.095037407780540, 0.485347092303173)),
    ((10.000000000000000, 9.000000000000000, 6.046995658143270, 3),
     (0.269996878004976, 0.351546727205252)),
    ((9.771353946056914, 9.000000000000000, 20.829752715478890, 3),
     (0.170327175987403, 0.417633887427604)),
    ((9.376380796522223, 9.000000000000000, 13.348522394106682, 3),
     (0.224831061170618, 0.390499753199537)),
    ((9.912704179532229, 4.000000000000000, 25.778231770351923, 3),
     (0.041682531090742, 0.456381082796448)),
    ((10.000000000000000, 5.000000000000000, 13.712247643370837, 3),
     (0.169704158827494, 0.407504401070349)),
    ((10.000000000000000, 4.000000000000000, 28.587923360931033, 3),
     (0.018590574793017, 0.461469808402328)),
    ((9.287535146732925, 4.000000000000000, 6.997389565284625, 3),
     (0.227796181191180, 0.378236347701388)),
    ((10.000000000000000, 6.000000000000000, 30.932435068478792, 3),
     (0.059472954520648, 0.467797305205436)),
    ((10.000000000000000, 5.000000000000000, 7.946854746461393, 3),
     (0.230289912314279, 0.372620011437199)),
    ((10.000000000000000, 6.000000000000000, 26.172410297895773, 3),
     (0.092970712548783, 0.452517230893687)),
    ((2.500000000000000, 7.000000000000000, 12.643410557057086, 2),
     (0.204731010265015, 0.365465890615466)),
    ((2.500000000000000, 3.000000000000000, 19.167537762557394, 2),
     (0.055109436570774, 0.368958899545662)),
    ((3.284581774573411, 4.000000000000000, 10.316761862277126, 2),
     (0.166172630157802, 0.348471104500449)),
    ((3.081407549428113, 3.000000000000000, 4.031683724514751, 2),
     (0.241024099198915, 0.334369599328307)),
    ((2.534263462549973, 9.000000000000000, 14.863663104253218, 2),
     (0.201778056814616, 0.372238386401537)),
    ((5.275920564662094, 9.000000000000000, 12.879135949769728, 2),
     (0.203973598369059, 0.342464374531538)),
    ((5.522856449128964, 9.000000000000000, 17.412586595686815, 2),
     (0.166903313442593, 0.343182296801951)),
    ((5.885914939777947, 9.000000000000000, 17.388086814072437, 2),
     (0.165843536466827, 0.339343003270739)),
    ((5.956560321967156, 4.000000000000000, 4.311690204814390, 2),
     (0.241297098717691, 0.318198504267149)),
    ((5.627911194894264, 5.000000000000000, 16.566819144431150, 2),
     (0.116166230383999, 0.316118110564487)),
    ((5.853454724533456, 5.000000000000000, 18.835069805085350, 2),
     (0.095381757604379, 0.309528999343537)),
    ((5.445581146699364, 5.000000000000000, 25.690737024023207, 2),
     (0.058083951403331, 0.309570785504992)),
    ((5.144720369630256, 5.000000000000000, 18.979172966805407, 2),
     (0.097016920475013, 0.321351921464780)),
    ((5.290707446388017, 6.000000000000000, 13.598520998056053, 2),
     (0.168946408689271, 0.330877420678264)),
    ((5.415844403197766, 6.000000000000000, 15.178617464461626, 2),
     (0.154783205339871, 0.329214743000905)),
    ((8.204144852288245, 6.000000000000000, 4.020177691372295, 2),
     (0.259276270369745, 0.314414757640664)),
    ((9.366069953403018, 3.000000000000000, 15.422766182794579, 2),
     (0.069614357522415, 0.221514524590655)),
    ((10.000000000000000, 4.000000000000000, 9.192387616705815, 2),
     (0.159306573366119, 0.265249480491412)),
    ((10.000000000000000, 3.000000000000000, 15.954247893607032, 2),
     (0.065338565587308, 0.206208172084088)),
    ((9.260586271537607, 3.000000000000000, 10.595175791701619, 2),
     (0.119448379672809, 0.249172358477458)),
    ((9.571675864670619, 3.000000000000000, 17.398847531397934, 2),
     (0.056120220665006, 0.209874274591936)),
    ((3.238739382175979, 5.000000000000000, 3.743510694098863, 1),
     (0.252983143427223, 0.294845758875212)),
    ((2.500000000000000, 4.000000000000000, 7.399343614420917, 1),
     (0.183040206795755, 0.257926038747328)),
    ((2.500000000000000, 4.000000000000000, 8.860840417367838, 1),
     (0.161906486282061, 0.245082856452373)),
    ((2.500000000000000, 8.000000000000000, 10.887265616829124, 1),
     (0.198215339920965, 0.280040394566793)),
    ((2.500000000000000, 7.000000000000000, 10.102335374185911, 1),
     (0.198391996564269, 0.276972972822943)),
    ((2.610434945585585, 8.000000000000000, 4.236171515065992, 1),
     (0.264351548213181, 0.305587520209317)),
    ((2.500000000000000, 9.000000000000000, 5.361363698027430, 1),
     (0.253705681170712, 0.303692386200227)),
    ((3.173101460658481, 8.000000000000000, 15.199536235308303, 1),
     (0.158059048588620, 0.251474757981066)),
    ((2.500000000000000, 7.000000000000000, 5.420260862573993, 1),
     (0.247916247899844, 0.298260982635961)),
    ((2.500000000000000, 8.000000000000000, 9.619364938403443, 1),
     (0.210368287109806, 0.285498667258705)),
    ((3.252581509053177, 7.000000000000000, 6.222406020434375, 1),
     (0.239063093515316, 0.291770588126056)),
    ((2.500000000000000, 8.000000000000000, 12.261808397585057, 1),
     (0.185644804078957, 0.273995681371109)),
    ((2.500000000000000, 3.000000000000000, 7.768505546917617, 1),
     (0.154746037636048, 0.235472818144094)),
    ((2.500000000000000, 3.000000000000000, 6.998840911724095, 1),
     (0.166868255640345, 0.243361880654828)),
    ((3.020562119690717, 3.000000000000000, 5.203087539105082, 1),
     (0.196738485628185, 0.259114709372600)),
    ((5.219091168761326, 3.000000000000000, 13.573887550967275, 1),
     (0.090628282442604, 0.152612885829902)),
    ((5.596250628047351, 2.000000000000000, 5.165106850365733, 1),
     (0.180299870423273, 0.228005797566952)),
    ((5.078574838897358, 3.000000000000000, 6.859942724404370, 1),
     (0.170387704769325, 0.226102592073249)),
    ((5.175617155844582, 3.000000000000000, 4.560800382141030, 1),
     (0.208205729711216, 0.255277944336981)),
    ((5.497353020782844, 5.000000000000000, 2.000000000000000, 1),
     (0.279552235480715, 0.302560720761057)),
    ((5.841773513544001, 6.000000000000000, 13.289365667818551, 1),
     (0.158050958526590, 0.219578147164172)),
    ((5.580549185463668, 7.000000000000000, 16.180320149263402, 1),
     (0.144396157428003, 0.213838310682324)),
    ((5.287772726922527, 7.000000000000000, 14.098946461580404, 1),
     (0.161089420918960, 0.228415961482639)),
    ((8.358221285614269, 5.000000000000000, 17.271563597297103, 1),
     (0.117739490491995, 0.159653713920656)),
    ((7.877244796359770, 5.000000000000000, 5.598934346859475, 1),
     (0.230766781258932, 0.264177189934782)),
    ((8.323336953587479, 5.000000000000000, 7.088152366811920, 1),
     (0.213473486372212, 0.248271815336452)),
    ((7.845486096299681, 5.000000000000000, 16.233795179282389, 1),
     (0.122698773315744, 0.168346491930928)),
    ((8.020564429287921, 5.000000000000000, 18.390260797283936, 1),
     (0.109135701507796, 0.153342345928621)),
    ((8.382569502344943, 5.000000000000000, 13.975124110876290, 1),
     (0.143683213547791, 0.185080815727973)),
    ((7.855593749782354, 3.000000000000000, 5.360435825061775, 1),
     (0.198627621625029, 0.233533189340665)),
    ((7.655501153733914, 4.000000000000000, 9.769593047963392, 1),
     (0.163115313640817, 0.204675855143315)),
    ((7.653019158008493, 6.000000000000000, 11.704589766625281, 1),
     (0.176756875213577, 0.222279098127299)),
    ((9.843286146335094, 3.000000000000000, 15.473317400474043, 1),
     (0.097277064147663, 0.119028072912671)),
    ((10.000000000000000, 3.000000000000000, 5.415846167802247, 1),
     (0.202442007473895, 0.225887061333307)),
    ((9.474422358368296, 3.000000000000000, 15.178161395592507, 1),
     (0.097225235576713, 0.122129020073115)),
    ((9.196648156004963, 8.000000000000000, 5.069223366759241, 1),
     (0.259590110606016, 0.284703049494546)),
    ((9.596611284326340, 8.000000000000000, 11.180193797198104, 1),
     (0.197252729846718, 0.232737751406665)),
    ((10.000000000000000, 6.000000000000000, 7.762802316406850, 1),
     (0.221390575677728, 0.248958498920698)),
    ((10.000000000000000, 7.000000000000000, 18.374375386402509, 1),
     (0.138566559988384, 0.173379372295182)),
    ((2.873950134580900, 4.000000000000000, 4.494521106912674, 10),
     (0.243634634404475, 0.252786403616982)),
    ((2.979763831715893, 9.000000000000000, 6.426710793964199, 10),
     (0.253606279609641, 0.267991102833414)),
    ((2.500000000000000, 9.000000000000000, 2.491252841450378, 10),
     (0.292219031954409, 0.302050662921454)),
    ((2.500000000000000, 8.000000000000000, 8.823379864036189, 10),
     (0.229177891291656, 0.245589581223674)),
    ((2.500000000000000, 5.000000000000000, 18.839339977864491, 10),
     (0.131138059136133, 0.135544290143881)),
    ((2.500000000000000, 5.000000000000000, 5.417711811598947, 10),
     (0.243341886213712, 0.255554542985452)),
    ((2.500000000000000, 9.000000000000000, 8.045356725346910, 10),
     (0.238659824559898, 0.256682502922572)),
    ((2.500000000000000, 3.000000000000000, 16.195810159806815, 10),
     (0.111923044121063, 0.105629328881352)),
    ((5.667887162619730, 2.000000000000000, 18.226010811743183, 10),
     (0.127103952725580, 0.064818487134261)),
    ((5.759673840199206, 2.000000000000000, 30.428731527415248, 10),
     (0.117092216382896, 0.036741995906091)),
    ((5.783634661463273, 2.000000000000000, 21.480194214511137, 10),
     (0.125254682876167, 0.054567458290390)),
    ((5.118173248862928, 2.000000000000000, 41.868473358578832, 10),
     (0.090795701409267, 0.021796051692064)),
    ((5.757349724389667, 2.000000000000000, 13.609604267804956, 10),
     (0.143014206576537, 0.091995236934855)),
    ((5.279304061296045, 5.000000000000000, 22.876127528048663, 10),
     (0.134587585343470, 0.107132235234522)),
    ((5.715709801059808, 5.000000000000000, 30.360213488022158, 10),
     (0.118860494873022, 0.069816164636304)),
    ((5.947947304520848, 4.000000000000000, 4.896643906619794, 10),
     (0.250049023697660, 0.242887789725632)),
    ((5.098993224817240, 5.000000000000000, 26.268750424752579, 10),
     (0.117096132818362, 0.087690668485178)),
    ((5.532229497629850, 5.000000000000000, 7.756449262721482, 10),
     (0.231186250285971, 0.225513878836785)),
    ((5.923584541768192, 5.000000000000000, 19.567605030849386, 10),
     (0.157954180595996, 0.125471507021846)),
    ((5.950156387030171, 2.000000000000000, 4.953666946161412, 10),
     (0.218650721667177, 0.199406747402398)),
    ((5.614158136535322, 2.000000000000000, 20.644953904366893, 10),
     (0.122339902185713, 0.056899972429795)),
    ((5.435908140730638, 3.000000000000000, 21.585064332200393, 10),
     (0.123015777433131, 0.073970339895444)),
    ((5.539908561343329, 3.000000000000000, 44.903699039953160, 10),
     (0.103761643828449, 0.027945682306968)),
    ((5.379251432099132, 2.000000000000000, 25.889074558828732, 10),
     (0.108735458942973, 0.042362347609226)),
    ((5.632909830682246, 6.000000000000000, 21.384042506861697, 10),
     (0.154592529173952, 0.131579114651922)),
    ((5.203326514932920, 6.000000000000000, 15.514467427422431, 10),
     (0.177211705255285, 0.168150476585649)),
    ((5.927793692134072, 5.000000000000000, 3.776639519741425, 10),
     (0.271348054830933, 0.270457116809751)),
    ((5.817322396187511, 5.000000000000000, 11.318041580907520, 10),
     (0.204847251634824, 0.190800086861277)),
    ((7.887786042250045, 1.000000000000000, 12.574240714561657, 10),
     (0.180313036150869, 0.078042018359953)),
    ((7.949960633591607, 3.000000000000000, 25.621368902089333, 10),
     (0.170444913418066, 0.067254741226718)),
    ((8.382592436810759, 2.000000000000000, 40.541271952926010, 10),
     (0.171789490785898, 0.024188092070611)),
    ((7.963797363322570, 2.000000000000000, 36.707318709966948, 10),
     (0.167702090890729, 0.029939021935050)),
    ((8.373924456610474, 2.000000000000000, 8.623846064990166, 10),
     (0.207284448018209, 0.143883709711674)),
    ((8.151990686473388, 2.000000000000000, 42.229127196458144, 10),
     (0.169041499413913, 0.021656309205313)),
    ((8.488384085232076, 8.000000000000000, 9.779628072315807, 10),
     (0.248845036045543, 0.226277885179969)),
    ((8.438357068876163, 3.000000000000000, 26.873452492074044, 10),
     (0.175995592832801, 0.064839541407938)),
    ((8.309434906530441, 2.000000000000000, 48.499663993444990, 10),
     (0.169572954089372, 0.012250504009832)),
    ((7.711579414965501, 3.000000000000000, 5.729859843354196, 10),
     (0.235595172778513, 0.205220057746795)),
    ((7.627374087940193, 2.000000000000000, 26.724973070776922, 10),
     (0.166579536145938, 0.047841689338013)),
    ((7.693923337226084, 3.000000000000000, 48.407897505690485, 10),
     (0.158520077735065, 0.024265876491785)),
    ((10.000000000000000, 6.000000000000000, 10.971953815910661, 10),
     (0.249140230920447, 0.208949611038395)),
    ((9.113097274740381, 6.000000000000000, 2.756464595173648, 10),
     (0.292493248553354, 0.288040049211416)),
    ((10.000000000000000, 9.000000000000000, 6.003388325186025, 10),
     (0.277981364211477, 0.265964422585547)),
    ((10.000000000000000, 5.000000000000000, 19.170756721559698, 10),
     (0.214297494687867, 0.137668148851032)),
    ((9.380110088755156, 6.000000000000000, 18.817507743754415, 10),
     (0.211170413810277, 0.149143742818635)),
    ((9.001795946577033, 8.000000000000000, 4.453854563212078, 10),
     (0.285932161124230, 0.280096509875731)),
    ((10.000000000000000, 8.000000000000000, 3.653159723688856, 10),
     (0.293111673602605, 0.288042956583509)),
    ((9.046182896421445, 3.000000000000000, 22.300946806849847, 10),
     (0.187897071888804, 0.080632507523559)),
    ((9.459420796383784, 3.000000000000000, 10.552556949414955, 10),
     (0.219916070333410, 0.150998257710435)),
    ((10.000000000000000, 3.000000000000000, 31.247622019812400, 10),
     (0.193051426788112, 0.056344277243535)),
    ((10.000000000000000, 3.000000000000000, 29.273434731152499, 10),
     (0.194235939161308, 0.061752741435561)),
    ((2.500000000000000, 8.000000000000000, 8.375074375178261, 9),
     (0.278874776874465, 0.245461838060914)),
    ((2.500000000000000, 7.000000000000000, 9.502846862649331, 9),
     (0.274640035980727, 0.233125801667481)),
    ((2.861900517122356, 7.000000000000000, 7.466126134628901, 9),
     (0.283496386595293, 0.251113999968638)),
    ((3.087422194135551, 8.000000000000000, 2.493857829360787, 9),
     (0.303294460735900, 0.299503468262916)),
    ((2.500000000000000, 4.000000000000000, 19.774716780756169, 9),
     (0.239805509794639, 0.123350321866803)),
    ((2.500000000000000, 3.000000000000000, 45.299844868071496, 9),
     (0.218350077565964, 0.029400310263857)),
    ((3.220553507003754, 3.000000000000000, 37.059380662726163, 9),
     (0.230693394558285, 0.046593234792596)),
    ((2.500000000000000, 4.000000000000000, 39.166418500944374, 9),
     (0.218833581499056, 0.053500744497167)),
    ((2.765403701684196, 3.000000000000000, 29.726535569137937, 9),
     (0.228766216486755, 0.065172432992089)),
    ((2.500000000000000, 4.000000000000000, 12.087654687250128, 9),
     (0.255680863281875, 0.172364503517437)),
    ((2.500000000000000, 3.000000000000000, 3.382852759577178, 9),
     (0.283211457062748, 0.244629788363146)),
    ((2.836612137080781, 4.000000000000000, 2.000000000000000, 9),
     (0.297049647880986, 0.280882385640405)),
    ((2.888854554705095, 3.000000000000000, 14.618307037832857, 9),
     (0.247764983842671, 0.129703268687627)),
    ((5.164399331990519, 6.000000000000000, 9.111465383743912, 9),
     (0.289517649641162, 0.233718247846502)),
    ((5.954180129965368, 6.000000000000000, 34.844915916827865, 9),
     (0.271416151318253, 0.098572005081295)),
    ((5.500356903003388, 7.000000000000000, 13.154128131968298, 9),
     (0.287212199884520, 0.214729179031137)),
    ((5.777207914079591, 6.000000000000000, 29.943983535383389, 9),
     (0.272037051630125, 0.115924028863176)),
    ((5.535810057742433, 7.000000000000000, 8.892716664134475, 9),
     (0.294270834424265, 0.245109987134089)),
    ((5.590040966343994, 4.000000000000000, 22.756612786898550, 9),
     (0.272874208878486, 0.120668152471860)),
    ((5.282620261743346, 3.000000000000000, 18.732823688754383, 9),
     (0.268608981711224, 0.113902242397197)),
    ((5.172895640160181, 3.000000000000000, 6.229254345814851, 9),
     (0.287733235794235, 0.211424158395775)),
    ((5.259721854731981, 3.000000000000000, 35.890872110681414, 9),
     (0.257794558652800, 0.051590154528538)),
    ((5.553646341595925, 3.000000000000000, 10.076683709549055, 9),
     (0.282579229397556, 0.171975753203702)),
    ((5.730003972159145, 2.000000000000000, 15.985698390269977, 9),
     (0.273099534393889, 0.106367886040496)),
    ((5.782381516990652, 2.000000000000000, 28.774618518379302, 9),
     (0.264837964744767, 0.050195747909619)),
    ((5.069379781665461, 7.000000000000000, 2.219484171420659, 9),
     (0.305505886009227, 0.299117247046096)),
    ((5.903716333756614, 6.000000000000000, 35.505574291994968, 9),
     (0.270028537614373, 0.096267061660057)),
    ((5.134679670979660, 6.000000000000000, 27.726398643923417, 9),
     (0.262766024827111, 0.122215217258406)),
    ((5.383260687864624, 6.000000000000000, 18.302295934127923, 9),
     (0.277988632708452, 0.174036647913653)),
    ((5.869792088464701, 5.000000000000000, 32.553432167966633, 9),
     (0.271127898857235, 0.097961273352764)),
    ((5.462451143540612, 6.000000000000000, 30.948864634440213, 9),
     (0.265463016656343, 0.109979248323483)),
    ((5.357445269639698, 6.000000000000000, 5.261434469006405, 9),
     (0.299017357208753, 0.266251845277156)),
    ((5.626373453003034, 6.000000000000000, 25.170846666445236, 9),
     (0.274227581031220, 0.137726168353716)),
    ((8.284200895164993, 2.000000000000000, 17.238899804160177, 9),
     (0.301554192719056, 0.105940692285444)),
    ((8.318102784124019, 2.000000000000000, 22.596147383535918, 9),
     (0.299868230517303, 0.079339467240112)),
    ((7.851936866242713, 7.000000000000000, 20.962374407911458, 9),
     (0.312781044647713, 0.181683501761474)),
    ((8.146081336032703, 8.000000000000000, 13.533962918469337, 9),
     (0.316935935360315, 0.229791958040202)),
    ((8.097208643162750, 7.000000000000000, 17.338991550524540, 9),
     (0.316388068784989, 0.202848417285126)),
    ((7.830256291991797, 7.000000000000000, 10.706822163825924, 9),
     (0.313176540349729, 0.241129458290891)),
    ((7.800658970688480, 6.000000000000000, 6.211375680877805, 9),
     (0.311626643681603, 0.264249896282407)),
    ((8.044863647118635, 7.000000000000000, 15.557155261544228, 9),
     (0.315667069117247, 0.212574888771317)),
    ((8.461774802909071, 5.000000000000000, 36.037296939777320, 9),
     (0.319385325854545, 0.098292696546847)),
    ((7.612382882207284, 4.000000000000000, 14.168690780706225, 9),
     (0.304871335995044, 0.174914896990705)),
    ((8.169633927695997, 4.000000000000000, 27.235846103864411, 9),
     (0.310188645891036, 0.111966253205247)),
    ((9.602031136015775, 6.000000000000000, 20.580635675818101, 9),
     (0.335870247812519, 0.183156832613590)),
    ((9.663686030178818, 6.000000000000000, 29.047658472982956, 9),
     (0.342051652884019, 0.145103906217593)),
    ((9.752928547364711, 5.000000000000000, 34.114931605281292, 9),
     (0.346005849641413, 0.113063524126797)),
    ((10.000000000000000, 6.000000000000000, 4.216215730437086, 9),
     (0.318586485393483, 0.285424328089113)),
    ((10.000000000000000, 6.000000000000000, 34.728526755839162, 9),
     (0.351364263377919, 0.126814419732483)),
    ((10.000000000000000, 5.000000000000000, 14.779627294882367, 9),
     (0.336857590024371, 0.202117143054426)),
    ((10.000000000000000, 6.000000000000000, 2.000000000000000, 9),
     (0.314600000000000, 0.301800000000000)),
    ((9.497050913948730, 6.000000000000000, 10.808854780098731, 9),
     (0.326144598376847, 0.238037678448062)),
    ((9.826635163465532, 2.000000000000000, 7.067114431849850, 9),
     (0.319941272561048, 0.198254895237219)),
    ((9.382502350301259, 5.000000000000000, 19.999476877446362, 9),
     (0.332813686722605, 0.170316458520961)),
    ((9.115530591819274, 5.000000000000000, 5.883436488694818, 9),
     (0.318838136384733, 0.261415293720058)),
    ((10.000000000000000, 5.000000000000000, 24.745870232952445, 9),
     (0.345671283209657, 0.151957116393476)),
    ((9.378359588580793, 5.000000000000000, 26.295787257422923, 9),
     (0.335806344161522, 0.141705436552867)),
    ((10.000000000000000, 2.000000000000000, 21.572576356602351, 9),
     (0.323021371182170, 0.088258489399158)),
    ((10.000000000000000, 3.000000000000000, 26.039872491235577, 9),
     (0.334313955371932, 0.097604624792946)),
    ((2.500000000000000, 3.000000000000000, 4.712138166253982, 8),
     (0.343596297739583, 0.255314225245773)),
    ((2.887457866682928, 2.000000000000000, 13.994896052145748, 8),
     (0.378355481667273, 0.147252258590914)),
    ((3.435419560439465, 2.000000000000000, 6.718989113532732, 8),
     (0.359835221372602, 0.221439588093081)),
    ((2.992533606273717, 2.000000000000000, 7.198014339866309, 8),
     (0.357547561896294, 0.213138099047494)),
    ((2.500000000000000, 1.000000000000000, 14.156726710024465, 8),
     (0.336800000000000, 0.101075312410856)),
    ((2.610457928897581, 1.000000000000000, 3.345815626895192, 8),
     (0.328197250844990, 0.221826392076904)),
    ((5.167065304553812, 8.000000000000000, 2.140948156850635, 8),
     (0.319050790877408, 0.311639000884062)),
    ((5.054434114346951, 7.000000000000000, 6.442157332603133, 8),
     (0.350153660761721, 0.293300427029458)),
    ((5.803735682450612, 9.000000000000000, 10.443841773523394, 8),
     (0.375378009345310, 0.285903762496986)),
    ((5.044877539779968, 6.000000000000000, 18.424428701407553, 8),
     (0.427725199984177, 0.236404193142223)),
    ((5.484832402621484, 7.000000000000000, 5.474777491295647, 8),
     (0.344981896508023, 0.298362968683559)),
    ((5.162300427200289, 7.000000000000000, 24.999056248525125, 8),
     (0.449995149062994, 0.222672748688972)),
    ((5.877256360743413, 7.000000000000000, 15.450444143259661, 8),
     (0.413513802236461, 0.261501863268415)),
    ((8.197449080109873, 2.000000000000000, 2.000000000000000, 8),
     (0.347936551313921, 0.291013367608286)),
    ((7.997237265754237, 2.000000000000000, 11.655829335806517, 8),
     (0.453649259316601, 0.196496132094268)),
    ((7.973192560907184, 9.000000000000000, 4.272886886879181, 8),
     (0.338229080610646, 0.310099315638142)),
    ((7.510355740108461, 8.000000000000000, 14.320141317950995, 8),
     (0.416025731428905, 0.279111321198999)),
    ((7.836498646186221, 8.000000000000000, 13.596658717999025, 8),
     (0.414495068387035, 0.283399327774669)),
    ((7.782186965908517, 9.000000000000000, 13.902105524067945, 8),
     (0.411711956783959, 0.285819458289889)),
    ((9.531795266771761, 5.000000000000000, 2.000000000000000, 8),
     (0.332525785184151, 0.312481969752139)),
    ((10.000000000000000, 5.000000000000000, 11.055624912778937, 8),
     (0.446236967672820, 0.287735844085801)),
    ((9.312270837393163, 7.000000000000000, 11.185222099189973, 8),
     (0.411556979914053, 0.296267158912712)),
    ((10.000000000000000, 7.000000000000000, 13.895455902446677, 8),
     (0.444575467843977, 0.293356133039006)),
    ((9.925669940032272, 5.000000000000000, 7.204078988766796, 8),
     (0.399934233564596, 0.300002893590000)),
    ((9.416740882402403, 5.000000000000000, 8.720116348180492, 8),
     (0.414208752743750, 0.292638547733766)),
    ((10.000000000000000, 4.000000000000000, 16.469698910991372, 8),
     (0.528848507367500, 0.250510595771746)),
    ((10.000000000000000, 6.000000000000000, 6.599237233947309, 8),
     (0.379692753722499, 0.306321372978895)),
    ((10.000000000000000, 5.000000000000000, 4.550269784467781, 8),
     (0.366470966730411, 0.307596812049607)),
    ((9.970332530519679, 6.000000000000000, 10.837022722087644, 8),
     (0.423540916702414, 0.296517382764761)),
    ((2.962707587174585, 9.000000000000000, 9.999116931630539, 7),
     (0.408211200945887, 0.317851549976094)),
    ((3.167205272899491, 9.000000000000000, 7.383624729892915, 7),
     (0.381205195546474, 0.320169938327552)),
    ((2.500000000000000, 5.000000000000000, 17.881593853007615, 7),
     (0.552579126236091, 0.280849943358571)),
    ((2.741501863896628, 6.000000000000000, 18.002908737801381, 7),
     (0.529028904249654, 0.294821634686008)),
    ((2.500000000000000, 6.000000000000000, 10.232668996271492, 7),
     (0.434885095553766, 0.311381195806711)),
    ((2.877902226185231, 6.000000000000000, 3.558203423120179, 7),
     (0.351935372594286, 0.317129472632533)),
    ((2.500000000000000, 6.000000000000000, 27.779995926916971, 7),
     (0.596009981671126, 0.265660012219249)),
    ((5.412821771284458, 3.000000000000000, 7.258040020605607, 7),
     (0.491726487604509, 0.316210739247126)),
    ((5.837547476050840, 3.000000000000000, 11.998261380615471, 7),
     (0.597544857410210, 0.297948096889035)),
    ((5.969397543974988, 4.000000000000000, 14.397906420283302, 7),
     (0.588161792444571, 0.312364189993051)),
    ((5.004079000563381, 5.000000000000000, 22.736677614468775, 7),
     (0.639205493884730, 0.287830772244423)),
    ((5.168438425945292, 4.000000000000000, 4.844860547907693, 7),
     (0.408397195097130, 0.323050826599914)),
    ((5.863284315202094, 4.000000000000000, 23.489710023246513, 7),
     (0.702789215715017, 0.269545829472011)),
    ((5.756333389411959, 9.000000000000000, 7.301135618422141, 7),
     (0.390784756289814, 0.328937859095861)),
    ((5.108337403014788, 8.000000000000000, 11.359771531491097, 7),
     (0.445391823063023, 0.327012532862878)),
    ((8.314898437378535, 9.000000000000000, 4.238233636005843, 7),
     (0.359969741993689, 0.331173667756613)),
    ((7.729982986777109, 5.000000000000000, 24.923686571499648, 7),
     (0.688052040164930, 0.313362183522719)),
    ((8.201460399608226, 4.000000000000000, 11.589840844520428, 7),
     (0.558484086531362, 0.339597217358427)),
    ((7.595604919273442, 5.000000000000000, 6.798265747221928, 7),
     (0.433825115928638, 0.336935765914877)),
    ((8.378186361828917, 9.000000000000000, 8.022357890675561, 7),
     (0.410583875078034, 0.342047963494690)),
    ((8.300135000740797, 8.000000000000000, 14.433553547681656, 7),
     (0.501918669959518, 0.351724471284244)),
    ((8.229270762113973, 9.000000000000000, 6.350022396927342, 7),
     (0.387920006753518, 0.338015696654435)),
    ((10.000000000000000, 3.000000000000000, 3.115225963548792, 7),
     (0.405141552942915, 0.336864607221389)),
    ((9.756267998308681, 3.000000000000000, 14.803384721914584, 7),
     (0.681447628179619, 0.317961809965490)),
    ((10.000000000000000, 4.000000000000000, 13.901609609717390, 7),
     (0.613663409611512, 0.356898390390283)),
    ((10.000000000000000, 8.000000000000000, 19.365358380679876, 7),
     (0.563288225426119, 0.374000000000000)),
    ((10.000000000000000, 9.000000000000000, 6.218490965882184, 7),
     (0.391058873522351, 0.344566397445941)),
    ((10.000000000000000, 8.000000000000000, 13.887493044276624, 7),
     (0.505537409575596, 0.368774986088553)),
    ((10.000000000000000, 5.000000000000000, 14.689071599466930, 7),
     (0.586264652272910, 0.366158824940187)),
    ((10.000000000000000, 5.000000000000000, 24.263442351912005, 7),
     (0.704502702343164, 0.353473115296176)),
    ((10.000000000000000, 8.000000000000000, 13.518172354943417, 7),
     (0.500736240614264, 0.368036344709887)),
    ((2.745564054714475, 2.000000000000000, 5.569110673549164, 6),
     (0.514796592794881, 0.359474270790530)),
    ((3.145288089149191, 5.000000000000000, 8.595832717291000, 6),
     (0.491565597829998, 0.383613984252967)),
    ((2.500000000000000, 4.000000000000000, 4.950679151608691, 6),
     (0.436488494020385, 0.360317084273359)),
    ((2.500000000000000, 4.000000000000000, 4.383231249423155, 6),
     (0.423125095923915, 0.356486810933606)),
    ((2.500000000000000, 2.000000000000000, 3.307282274836235, 6),
     (0.433961628851392, 0.345847068265079)),
    ((5.045583268005572, 9.000000000000000, 9.591945248602441, 6),
     (0.442697682390147, 0.388049303050288)),
    ((5.594284526041456, 9.000000000000000, 10.197201238166286, 6),
     (0.450086516455153, 0.395001049978596)),
    ((5.988802467213943, 8.000000000000000, 12.305951956169229, 6),
     (0.487257498046796, 0.413699925815657)),
    ((5.425850947396252, 5.000000000000000, 8.046156862703112, 6),
     (0.483706012487679, 0.399301290856490)),
    ((5.405852543210212, 6.000000000000000, 16.635714109554605, 6),
     (0.561334046027989, 0.428929910349990)),
    ((5.369364240119585, 5.000000000000000, 7.340573827339962, 6),
     (0.469587365934968, 0.393454978115723)),
    ((5.702045821590509, 5.000000000000000, 10.325652051724541, 6),
     (0.518931169895089, 0.413739242504772)),
    ((5.411096326958829, 6.000000000000000, 5.292034843095026, 6),
     (0.409462568713660, 0.370225502550786)),
    ((8.242968536635763, 9.000000000000000, 4.900205865328810, 6),
     (0.380066730838685, 0.369356110185534)),
    ((8.238754570485817, 9.000000000000000, 5.941330110378650, 6),
     (0.394085190479792, 0.379080970224506)),
    ((8.395684243897479, 5.000000000000000, 9.461515968715135, 6),
     (0.500650818343943, 0.431478442460858)),
    ((10.000000000000000, 5.000000000000000, 12.704963485646498, 6),
     (0.527409423196536, 0.462819853942586)),
    ((10.000000000000000, 5.000000000000000, 15.675370760759400, 6),
     (0.549889909944936, 0.475539168423417)),
    ((10.000000000000000, 6.000000000000000, 3.506573388368494, 6),
     (0.376971607684817, 0.369693342114833)),
    ((10.000000000000000, 6.000000000000000, 14.063922879568509, 6),
     (0.520351575837627, 0.462514141646555)),
    ((10.000000000000000, 7.000000000000000, 3.128443413944953, 6),
     (0.363201427183578, 0.360351875234771)),
    ((10.000000000000000, 7.000000000000000, 11.632405914314647, 6),
     (0.485717528901766, 0.445334942878781)),
    ((9.050263182466011, 7.000000000000000, 17.083676942759791, 6),
     (0.528750641050191, 0.459929219239207)),
    ((10.000000000000000, 6.000000000000000, 4.736966947326921, 6),
     (0.400065523651845, 0.386391115357349)),
    ((9.409402543801862, 7.000000000000000, 6.287660211686590, 6),
     (0.414788350147883, 0.395484856370224)),
    ((9.633394604006961, 8.000000000000000, 4.623044001702525, 6),
     (0.379313164731457, 0.372825868614124)),
    ((9.020770192275748, 7.000000000000000, 13.422245014577644, 6),
     (0.506023004801611, 0.447380080680689)),
    ((9.263176096861541, 7.000000000000000, 15.233295182477667, 6),
     (0.517940485402123, 0.456442102727039)),
    ((3.332782026387723, 7.000000000000000, 16.113419977677538, 5),
     (0.499438433533997, 0.491316662788289)),
    ((2.500000000000000, 5.000000000000000, 6.543649602836132, 5),
     (0.446290656443251, 0.435506335392899)),
    ((2.500000000000000, 6.000000000000000, 15.572129740854304, 5),
     (0.513882042217229, 0.487694995709606)),
    ((2.500000000000000, 3.000000000000000, 2.000000000000000, 5),
     (0.370300000000000, 0.370000000000000)),
    ((2.828559184243374, 9.000000000000000, 21.473258817290873, 5),
     (0.504333788056536, 0.495186596225649)),
    ((2.500000000000000, 8.000000000000000, 12.020108658634838, 5),
     (0.467964891000806, 0.459023668250604)),
    ((2.500000000000000, 9.000000000000000, 14.427904414153719, 5),
     (0.475781378691999, 0.467353474277846)),
    ((2.500000000000000, 5.000000000000000, 8.380243803410817, 5),
     (0.472682681837519, 0.455422938237116)),
    ((3.363079416671538, 5.000000000000000, 2.775509664209031, 5),
     (0.368892605122633, 0.374353471894843)),
    ((5.927152426102055, 9.000000000000000, 20.603131952471927, 5),
     (0.480247225161527, 0.515977492813773)),
    ((5.339079962653624, 8.000000000000000, 16.611574939424255, 5),
     (0.478987767148309, 0.504943952840018)),
    ((5.347356764781598, 8.000000000000000, 15.412165198232049, 5),
     (0.474580692066424, 0.500508188346594)),
    ((5.368950609634622, 7.000000000000000, 7.038165919924306, 5),
     (0.413411547161925, 0.434813609675807)),
    ((5.063316239211655, 7.000000000000000, 16.013319334821031, 5),
     (0.487164109418344, 0.505154324096613)),
    ((5.929552854535908, 7.000000000000000, 7.572813447048060, 5),
     (0.418536531241438, 0.444243976557503)),
    ((5.727946559508910, 7.000000000000000, 10.668172633934036, 5),
     (0.453183571118484, 0.479048722680844)),
    ((5.641782139668679, 6.000000000000000, 9.549016885745186, 5),
     (0.456106761504047, 0.478327489299549)),
    ((5.344359642058747, 3.000000000000000, 5.430489560972486, 5),
     (0.451633359289691, 0.461109580193912)),
    ((7.749909297802317, 4.000000000000000, 4.268933751175051, 5),
     (0.401758834499508, 0.433410572084067)),
    ((8.145409228909998, 5.000000000000000, 7.545633529064384, 5),
     (0.435789245569801, 0.481062345229275)),
    ((7.907253670159305, 6.000000000000000, 10.770986229289623, 5),
     (0.453801635046687, 0.502116755437095)),
    ((7.592508492261312, 5.000000000000000, 4.933568344499713, 5),
     (0.400903332667102, 0.432330970614901)),
    ((7.674872690410821, 5.000000000000000, 3.550245288479484, 5),
     (0.375905962921118, 0.401447352486808)),
    ((7.991979987062054, 7.000000000000000, 3.283701248747225, 5),
     (0.356783033016474, 0.379785023517449)),
    ((9.345599185286883, 7.000000000000000, 17.488521757881820, 5),
     (0.464927819285986, 0.537405269620011)),
    ((9.659595218511388, 8.000000000000000, 3.357217748484464, 5),
     (0.351436092963224, 0.377417525766746)))

MUNSELL_COLOURS_TO_XYY = (
    (0.41515095, 0.51288165, 0.57024410),
    (0.38804358, 0.46299149, 0.31592072),
    (0.33491518, 0.36277402, 0.22128409),
    (0.39936353, 0.58547238, 0.64852094),
    (0.34767896, 0.41529220, 0.58706989),
    (0.33966055, 0.41527226, 0.07167165),
    (0.36265912, 0.47966922, 0.11068168),
    (0.35748002, 0.45915987, 0.27273590),
    (0.36348032, 0.48213512, 0.06293782),
    (0.30330033, 0.73038471, 0.05538644),
    (0.33159302, 0.43388935, 0.89380734),
    (0.31838794, 0.40167814, 0.05382145),
    (0.27202005, 0.83522048, 0.04995375),
    (0.31425413, 0.58372544, 0.04377268),
    (0.27634942, 0.75063178, 0.05211431),
    (0.25883700, 0.71096717, 0.01266934),
    (0.31405111, 0.53120144, 0.02111891),
    (0.33914454, 0.65636470, 0.71217401),
    (0.35328989, 0.63157007, 0.65497851),
    (0.32167873, 0.43862617, 0.30809910),
    (0.31168045, 0.62700640, 0.34717087),
    (0.31496017, 0.47530248, 0.67920304),
    (0.26882355, 0.70549119, 0.69614462),
    (0.31107787, 0.51188895, 0.58306925),
    (0.31254722, 0.34686238, 0.63343340),
    (0.30880402, 0.37157402, 0.08263161),
    (0.23582365, 0.72197618, 0.06667783),
    (0.29476305, 0.57521949, 0.23583791),
    (0.28891056, 0.61005165, 0.28191444),
    (0.17590584, 0.91365000, 0.23196178),
    (0.31292041, 0.37520740, 0.25538037),
    (0.22307972, 0.81536440, 0.26986020),
    (0.30648167, 0.48754769, 0.15098549),
    (0.30382174, 0.34089453, 0.84210967),
    (0.28517207, 0.38369148, 0.89445395),
    (0.20621151, 0.56369357, 0.77955867),
    (0.24658480, 0.49294784, 0.87271533),
    (0.22538285, 0.55646110, 0.60532773),
    (0.28500017, 0.38833563, 0.24045742),
    (0.19598037, 0.59002914, 0.29181101),
    (0.16437784, 0.59069112, 0.23370301),
    (0.17940333, 0.46639290, 0.06448045),
    (0.07553293, 0.55981543, 0.06406275),
    (0.27330162, 0.37048932, 0.11621278),
    (0.23251367, 0.40832841, 0.02585745),
    (0.05704598, 0.55990299, 0.01221862),
    (0.09405428, 0.51916421, 0.02268015),
    (0.06306305, 0.54336526, 0.03610370),
    (0.23250342, 0.41833342, 0.05599130),
    (0.22630523, 0.39163204, 0.05597116),
    (0.15858055, 0.42916814, 0.05259972),
    (0.07933408, 0.51474312, 0.30905098),
    (0.14028772, 0.46282023, 0.41589047),
    (0.29271668, 0.33531051, 0.37326792),
    (0.17253811, 0.43786778, 0.33686994),
    (0.09180367, 0.46823752, 0.05151176),
    (0.10903846, 0.44893518, 0.03595462),
    (0.24286930, 0.37094376, 0.04060119),
    (0.27771166, 0.34994832, 0.23574564),
    (0.05867972, 0.50502648, 0.19891229),
    (0.25930387, 0.37349411, 0.26874577),
    (0.12284826, 0.47211684, 0.21388094),
    (0.08906820, 0.48703791, 0.27058998),
    (0.27018357, 0.35138182, 0.76804186),
    (0.22062535, 0.38110738, 0.85084234),
    (0.26193025, 0.35814050, 0.86839733),
    (0.04310530, 0.45634623, 0.12074655),
    (0.16522669, 0.40881359, 0.18014875),
    (0.02517244, 0.46138968, 0.13173010),
    (0.23349872, 0.37536989, 0.14476492),
    (0.05119965, 0.46839242, 0.26212526),
    (0.23159950, 0.37207726, 0.20351563),
    (0.08301372, 0.45335265, 0.25304755),
    (0.20183026, 0.36561544, 0.39526058),
    (0.06340759, 0.37121187, 0.07975536),
    (0.16044634, 0.34707426, 0.10145605),
    (0.24416648, 0.33434737, 0.07774819),
    (0.28155768, 0.33248001, 0.89992977),
    (0.28105936, 0.33270880, 0.88937678),
    (0.25255297, 0.34594245, 0.87623351),
    (0.20616318, 0.34192146, 0.77176579),
    (0.21898553, 0.33335124, 0.85174026),
    (0.19119679, 0.33526743, 0.80792502),
    (0.29624596, 0.31950269, 0.96665647),
    (0.24328961, 0.31868567, 0.12931978),
    (0.10471116, 0.30938022, 0.15549815),
    (0.08624520, 0.30268915, 0.15900713),
    (0.10497041, 0.32451898, 0.22191645),
    (0.16894641, 0.33087742, 0.29312371),
    (0.16144965, 0.33133829, 0.34018592),
    (0.25864013, 0.31415379, 0.28205753),
    (0.07732853, 0.22846579, 0.08121964),
    (0.15795868, 0.26417318, 0.11377678),
    (0.06907834, 0.20994435, 0.07225730),
    (0.12862477, 0.25616557, 0.08539517),
    (0.05881481, 0.21256736, 0.07052095),
    (0.25058288, 0.29329096, 0.17796585),
    (0.18830894, 0.26192867, 0.13740285),
    (0.16840760, 0.25029878, 0.13934697),
    (0.19516480, 0.27716957, 0.51306785),
    (0.19935306, 0.27783329, 0.44060477),
    (0.26308512, 0.30462120, 0.52610451),
    (0.25324160, 0.30291555, 0.67153139),
    (0.15890128, 0.25325980, 0.59956247),
    (0.24841933, 0.29869620, 0.43833832),
    (0.20821330, 0.28356991, 0.52733609),
    (0.23939654, 0.29206110, 0.43144538),
    (0.18279859, 0.27122662, 0.52199238),
    (0.16449512, 0.24371038, 0.08686299),
    (0.16724393, 0.24366794, 0.06480227),
    (0.19881487, 0.26071106, 0.06927689),
    (0.09076654, 0.15277497, 0.06421355),
    (0.18253778, 0.23018215, 0.03460635),
    (0.16926303, 0.22496873, 0.06237928),
    (0.20398493, 0.25134710, 0.05473403),
    (0.28140041, 0.30378091, 0.23081828),
    (0.15231331, 0.21384066, 0.25883348),
    (0.14386953, 0.21327677, 0.41482428),
    (0.15935060, 0.22670722, 0.40114326),
    (0.10949743, 0.15034868, 0.15892888),
    (0.22674934, 0.26033997, 0.17110185),
    (0.20569472, 0.24048470, 0.15700695),
    (0.11359218, 0.15851929, 0.15851498),
    (0.13446868, 0.17456223, 0.15665285),
    (0.20295637, 0.23758918, 0.07464645),
    (0.16020908, 0.20160833, 0.11096053),
    (0.17946292, 0.22546056, 0.33406930),
    (0.19584886, 0.21874231, 0.05264774),
    (0.25950493, 0.28494406, 0.60260113),
    (0.22170777, 0.24928491, 0.29763974),
    (0.13564759, 0.16991066, 0.38138893),
    (0.23373145, 0.24171207, 0.08831548),
    (0.25339824, 0.26720506, 0.67917402),
    (0.29210338, 0.30192924, 0.75127547),
    (0.22958296, 0.24621680, 0.59738522),
    (0.12585350, 0.12764109, 0.16297312),
    (0.24227309, 0.25436998, 0.18624748),
    (0.23758242, 0.25457444, 0.66865194),
    (0.10476265, 0.09497701, 0.05235122),
    (0.12612865, 0.06066443, 0.02676646),
    (0.11705747, 0.03587748, 0.02951591),
    (0.12329050, 0.04415430, 0.02037758),
    (0.09139852, 0.01529466, 0.02045231),
    (0.13833192, 0.07953813, 0.02236117),
    (0.13361693, 0.10504399, 0.18414205),
    (0.12104740, 0.06862453, 0.15728175),
    (0.25249867, 0.24628189, 0.13536950),
    (0.11706407, 0.08706468, 0.18814811),
    (0.22549284, 0.21806210, 0.16192792),
    (0.15344950, 0.11674072, 0.15905692),
    (0.22358720, 0.20668864, 0.04253357),
    (0.12515256, 0.06568452, 0.04436879),
    (0.12125722, 0.06874820, 0.05533026),
    (0.10373316, 0.02774140, 0.06333516),
    (0.10925991, 0.04419045, 0.03405371),
    (0.15402461, 0.13042053, 0.28570417),
    (0.17573216, 0.16578146, 0.27364637),
    (0.27401103, 0.27401935, 0.23451177),
    (0.20759130, 0.19464274, 0.21940166),
    (0.17049737, 0.06465369, 0.05868583),
    (0.17064728, 0.02889150, 0.04372401),
    (0.16720380, 0.03196773, 0.03579761),
    (0.21031018, 0.15034168, 0.03888934),
    (0.16827351, 0.02413193, 0.03757647),
    (0.29178046, 0.29061931, 0.90323404),
    (0.24910224, 0.22648966, 0.59336016),
    (0.17601554, 0.05876060, 0.05160293),
    (0.16834537, 0.01686511, 0.04374851),
    (0.23182863, 0.19825806, 0.05291206),
    (0.16638758, 0.05075245, 0.03650792),
    (0.16028497, 0.01948654, 0.05046003),
    (0.24957235, 0.21006823, 0.31587613),
    (0.29306654, 0.28917618, 0.32466527),
    (0.28495343, 0.27687408, 0.81760638),
    (0.21441304, 0.13814375, 0.19716723),
    (0.20941829, 0.14321541, 0.24327119),
    (0.28541299, 0.27913907, 0.54006024),
    (0.29230469, 0.28656219, 0.52465762),
    (0.18804124, 0.08137467, 0.06580398),
    (0.22025958, 0.15180899, 0.06551257),
    (0.19309397, 0.06115047, 0.07873642),
    (0.19437258, 0.06326427, 0.06829742),
    (0.27887167, 0.24543217, 0.57450962),
    (0.27487624, 0.23376357, 0.46322748),
    (0.28356864, 0.25190050, 0.45980664),
    (0.30333596, 0.30005216, 0.66401066),
    (0.23835467, 0.11558036, 0.09827669),
    (0.23067198, 0.05028062, 0.07671426),
    (0.21902307, 0.05208443, 0.11065271),
    (0.22907253, 0.06719948, 0.06903321),
    (0.25361450, 0.16387485, 0.09900850),
    (0.28535713, 0.25114971, 0.08429109),
    (0.29701504, 0.28076672, 0.11652327),
    (0.24894294, 0.13513311, 0.07507850),
    (0.28976435, 0.23551078, 0.32030680),
    (0.28699217, 0.21227390, 0.38376156),
    (0.29423180, 0.24483482, 0.41568603),
    (0.27112866, 0.10892559, 0.09137276),
    (0.26932562, 0.11871922, 0.07456975),
    (0.28774446, 0.21149857, 0.06409553),
    (0.25815891, 0.05632389, 0.07763328),
    (0.28438514, 0.18361032, 0.08751006),
    (0.27466364, 0.11623324, 0.04459164),
    (0.26635689, 0.06032880, 0.04436654),
    (0.30526917, 0.29787617, 0.38438766),
    (0.26275899, 0.12295408, 0.30482710),
    (0.27733084, 0.16764806, 0.24584118),
    (0.27121622, 0.09967670, 0.21385417),
    (0.26547923, 0.10802713, 0.26515926),
    (0.29841781, 0.26325636, 0.25902873),
    (0.27412192, 0.13541072, 0.26778091),
    (0.30429530, 0.11611832, 0.04387000),
    (0.30157505, 0.08506396, 0.03768091),
    (0.31391169, 0.18564420, 0.48667459),
    (0.31670790, 0.22835511, 0.52829657),
    (0.31664956, 0.20454265, 0.45562827),
    (0.31300137, 0.23982828, 0.40210613),
    (0.31187872, 0.26667157, 0.33190218),
    (0.31537904, 0.21052765, 0.39335492),
    (0.31803143, 0.09273886, 0.17122630),
    (0.30594132, 0.18152717, 0.14244072),
    (0.31195968, 0.12089229, 0.15102095),
    (0.33618672, 0.17589268, 0.24249386),
    (0.34207627, 0.13875616, 0.24138597),
    (0.34605075, 0.11899797, 0.23580785),
    (0.31923003, 0.28291153, 0.25504488),
    (0.35136426, 0.12256902, 0.26410270),
    (0.33639641, 0.20777481, 0.23332748),
    (0.31464507, 0.30107880, 0.27040807),
    (0.32622786, 0.23679153, 0.28338647),
    (0.31964789, 0.19702337, 0.02988488),
    (0.33202416, 0.16293316, 0.16828902),
    (0.31883410, 0.26119414, 0.19149517),
    (0.34497302, 0.14740581, 0.17674791),
    (0.33396066, 0.13204228, 0.15759269),
    (0.32447663, 0.09207588, 0.03498261),
    (0.32823298, 0.08288658, 0.04740281),
    (0.34263192, 0.24928260, 0.04966462),
    (0.37863885, 0.14805570, 0.03133476),
    (0.36067287, 0.22508694, 0.03664306),
    (0.35583972, 0.20890369, 0.02874030),
    (0.34728299, 0.11402692, 0.01746108),
    (0.32940771, 0.22789278, 0.01489395),
    (0.31972567, 0.31122932, 0.53600948),
    (0.35012172, 0.29333067, 0.42147094),
    (0.37589661, 0.28507170, 0.66934047),
    (0.42549932, 0.23904177, 0.33329037),
    (0.34641765, 0.29725050, 0.38411768),
    (0.45441652, 0.21797623, 0.36276856),
    (0.41521602, 0.25989123, 0.39086156),
    (0.34780042, 0.29284040, 0.03605620),
    (0.45445510, 0.19822245, 0.03201793),
    (0.33858745, 0.30985450, 0.70004006),
    (0.41381262, 0.28393710, 0.60579167),
    (0.39278492, 0.29146870, 0.81034741),
    (0.33239612, 0.31251827, 0.19604738),
    (0.43846181, 0.29096381, 0.23141236),
    (0.40958022, 0.29719222, 0.48882871),
    (0.44399899, 0.29369509, 0.43379687),
    (0.40554919, 0.29723013, 0.16687769),
    (0.42007003, 0.28930815, 0.16729330),
    (0.52108329, 0.25574146, 0.13999526),
    (0.37638010, 0.30728007, 0.34070289),
    (0.36495307, 0.30801481, 0.20910915),
    (0.42566912, 0.29564012, 0.28217939),
    (0.38537971, 0.31745807, 0.82116554),
    (0.37201534, 0.31965197, 0.79705828),
    (0.55136347, 0.28138892, 0.19712193),
    (0.53899416, 0.29048788, 0.25823634),
    (0.43854811, 0.31033170, 0.27612362),
    (0.35589069, 0.31655370, 0.24649473),
    (0.60150190, 0.26287828, 0.27670596),
    (0.49631592, 0.30111191, 0.04570504),
    (0.60338354, 0.27468340, 0.04600213),
    (0.57619776, 0.31554717, 0.14073356),
    (0.65681487, 0.27970869, 0.16409107),
    (0.40414547, 0.32310724, 0.13347887),
    (0.68743116, 0.27762719, 0.14148314),
    (0.39097754, 0.32893313, 0.75691217),
    (0.44274163, 0.32668916, 0.63163346),
    (0.35143720, 0.32860694, 0.80679695),
    (0.55200335, 0.34090583, 0.13240521),
    (0.43719237, 0.33673056, 0.18274766),
    (0.36439573, 0.33015311, 0.87403366),
    (0.34753957, 0.32343836, 0.93909237),
    (0.38880059, 0.33783693, 0.70408740),
    (0.40006019, 0.33663147, 0.07790261),
    (0.67248369, 0.32330365, 0.07220649),
    (0.64354918, 0.33973639, 0.08803122),
    (0.39181364, 0.34469480, 0.72252254),
    (0.49958346, 0.36703778, 0.64322822),
    (0.60231065, 0.36168845, 0.16068191),
    (0.50023387, 0.36789369, 0.58191450),
    (0.54791150, 0.34892913, 0.02263783),
    (0.48794187, 0.38293202, 0.20665660),
    (0.42836733, 0.35891726, 0.14365536),
    (0.41518890, 0.35495306, 0.14408043),
    (0.46542405, 0.34082576, 0.02079253),
    (0.44637245, 0.38945422, 0.69298338),
    (0.45404134, 0.39647154, 0.69233629),
    (0.48546253, 0.41305920, 0.59936297),
    (0.47978428, 0.39825981, 0.20940783),
    (0.46785973, 0.39297126, 0.20053901),
    (0.51331905, 0.41265964, 0.21747039),
    (0.41770486, 0.37303747, 0.24430738),
    (0.37481816, 0.36536787, 0.78442772),
    (0.35992953, 0.34813048, 0.94470358),
    (0.39620152, 0.37951049, 0.68657974),
    (0.51299224, 0.43528921, 0.15241166),
    (0.52529122, 0.46205863, 0.20512581),
    (0.54693353, 0.47444863, 0.20705936),
    (0.38249101, 0.37271251, 0.24466996),
    (0.52465528, 0.46381752, 0.27037022),
    (0.36332827, 0.36042625, 0.41747900),
    (0.49103786, 0.44741782, 0.37625337),
    (0.53260672, 0.46085797, 0.38175879),
    (0.39947636, 0.38607101, 0.30024818),
    (0.41566751, 0.39590836, 0.41203391),
    (0.38379951, 0.37533829, 0.49499836),
    (0.50184813, 0.44587964, 0.46556628),
    (0.51381974, 0.45512612, 0.47222976),
    (0.49758967, 0.49049612, 0.45242214),
    (0.43935016, 0.43103452, 0.23259639),
    (0.37285524, 0.37178029, 0.05690988),
    (0.50597795, 0.49533702, 0.71175266),
    (0.46625888, 0.45805354, 0.60604474),
    (0.47877996, 0.46893263, 0.71437706),
    (0.48009865, 0.45955664, 0.16263768),
    (0.37334852, 0.37797455, 0.16515524),
    (0.47856178, 0.50475320, 0.58732427),
    (0.47119689, 0.49876821, 0.65483021),
    (0.40983957, 0.43186708, 0.47430500),
    (0.42018308, 0.44575131, 0.40534100),
    (0.45841191, 0.48304710, 0.36340530),
    (0.45040867, 0.47371048, 0.34766863),
    (0.45990386, 0.46807933, 0.05857873),
    (0.39793427, 0.42894800, 0.13930127),
    (0.43789903, 0.48337683, 0.18110313),
    (0.45785880, 0.50623924, 0.25901491),
    (0.39985008, 0.43116457, 0.20152195),
    (0.37226961, 0.39738120, 0.23381716),
    (0.35899980, 0.38250951, 0.36788985),
    (0.46379107, 0.53608140, 0.45228296),
    (0.35180708, 0.37798088, 0.55904288))

MUNSELL_GREYS_TO_XYY = (
    (0.31006, 0.31616, 0.00000000),
    (0.31006, 0.31616, 0.00473582),
    (0.31006, 0.31616, 0.00961944),
    (0.31006, 0.31616, 0.01545756),
    (0.31006, 0.31616, 0.02293343),
    (0.31006, 0.31616, 0.03261914),
    (0.31006, 0.31616, 0.04498800),
    (0.31006, 0.31616, 0.06042690),
    (0.31006, 0.31616, 0.07924864),
    (0.31006, 0.31616, 0.10170428),
    (0.31006, 0.31616, 0.12799549),
    (0.31006, 0.31616, 0.15828689),
    (0.31006, 0.31616, 0.19271844),
    (0.31006, 0.31616, 0.23141772),
    (0.31006, 0.31616, 0.27451233),
    (0.31006, 0.31616, 0.32214224),
    (0.31006, 0.31616, 0.37447210),
    (0.31006, 0.31616, 0.43170362),
    (0.31006, 0.31616, 0.49408790),
    (0.31006, 0.31616, 0.56193781),
    (0.31006, 0.31616, 0.63564030),
    (0.31006, 0.31616, 0.71566876),
    (0.31006, 0.31616, 0.80259539),
    (0.31006, 0.31616, 0.89710353),
    (0.31006, 0.31616, 1.00000000))

XYY_TO_MUNSELL_SPECIFICATIONS = (
    ((0.415150950000000, 0.512881650000000, 0.570244100000000),
     (2.497425498445040, 7.965379827810718, 11.928549858473941, 4)),
    ((0.388043580000000, 0.462991490000000, 0.315920720000000),
     (2.500643995489256, 6.197794793223818, 6.923610697067909, 4)),
    ((0.334915180000000, 0.362774020000000, 0.221284090000000),
     (2.490309162027092, 5.311956924522700, 1.998638034101546, 4)),
    ((0.399363530000000, 0.585472380000000, 0.648520940000000),
     (5.611784843667591, 8.402756525284460, 18.567138539062916, 4)),
    ((0.347678960000000, 0.415292200000000, 0.587069890000000),
     (5.850940225935846, 8.062638667064569, 5.784142504579139, 4)),
    ((0.339660550000000, 0.415272260000000, 0.071671650000000),
     (5.781668206923682, 3.174803944794176, 3.349385637554362, 4)),
    ((0.362659120000000, 0.479669220000000, 0.110681680000000),
     (5.488326650747126, 3.899412030844517, 5.762709554941629, 4)),
    ((0.357480020000000, 0.459159870000000, 0.272735900000000),
     (5.809160702731946, 5.816975152160862, 6.666016191540733, 4)),
    ((0.363480320000000, 0.482135120000000, 0.062937820000000),
     (5.210688984693519, 2.977036365566830, 5.141836619699955, 4)),
    ((0.318387940000000, 0.401678140000000, 0.053821450000000),
     (8.112670582764574, 2.748942921505224, 3.164465784963284, 4)),
    ((0.314254130000000, 0.583725440000000, 0.043772680000000),
     (8.045496002723631, 2.463064986124605, 7.500061207356352, 4)),
    ((0.314051110000000, 0.531201440000000, 0.021118910000000),
     (7.593016363160379, 1.575074388843772, 4.626174843109444, 4)),
    ((0.321678730000000, 0.438626170000000, 0.308099100000000),
     (9.015210356738553, 6.131271191284028, 6.805293664943519, 4)),
    ((0.311680450000000, 0.627006400000000, 0.347170870000000),
     (9.042981019275471, 6.454053179809056, 17.010229922329884, 4)),
    ((0.314960170000000, 0.475302480000000, 0.679203040000000),
     (9.914792278510397, 8.564387948380265, 11.130516599293573, 4)),
    ((0.311077870000000, 0.511888950000000, 0.583069250000000),
     (9.960002841731850, 8.039682743685185, 13.199185802941963, 4)),
    ((0.312547220000000, 0.346862380000000, 0.633433400000000),
     (9.885206994602566, 8.321342629562913, 2.070847452019443, 4)),
    ((0.308804020000000, 0.371574020000000, 0.082631610000000),
     (9.988091619718151, 3.400787111000257, 2.568453642887343, 4)),
    ((0.235823650000000, 0.721976180000000, 0.066677830000000),
     (9.998487416360815, 3.063915653085683, 13.513388270694880, 4)),
    ((0.294763050000000, 0.575219490000000, 0.235837910000000),
     (9.997637565846503, 5.461465412699126, 12.753028500364775, 4)),
    ((0.288910560000000, 0.610051650000000, 0.281914440000000),
     (9.997853767792066, 5.900814094339881, 15.244070244743728, 4)),
    ((0.312920410000000, 0.375207400000000, 0.255380370000000),
     (9.754827852478819, 5.653647397777864, 3.410784496514246, 4)),
    ((0.306481670000000, 0.487547690000000, 0.150985490000000),
     (9.861620267925183, 4.487864137008210, 7.675705535447796, 4)),
    ((0.303821740000000, 0.340894530000000, 0.842109670000000),
     (3.230390862780030, 9.345163602252851, 3.440344036721676, 3)),
    ((0.225382850000000, 0.556461100000000, 0.605327730000000),
     (2.789389601104997, 8.166099914244480, 20.865361708231859, 3)),
    ((0.285000170000000, 0.388335630000000, 0.240457420000000),
     (3.221789515970928, 5.507741932212069, 5.463072825387044, 3)),
    ((0.195980370000000, 0.590029140000000, 0.291811010000000),
     (2.621280587765934, 5.989380653020767, 19.362394859916822, 3)),
    ((0.164377840000000, 0.590691120000000, 0.233703010000000),
     (3.286631910333986, 5.439892532971944, 19.854410847201201, 3)),
    ((0.179403330000000, 0.466392900000000, 0.064480450000000),
     (5.728249856243197, 3.013295281777591, 10.742779986622125, 3)),
    ((0.075532930000000, 0.559815430000000, 0.064062750000000),
     (5.344712442461743, 3.003537623832734, 18.896804785539025, 3)),
    ((0.273301620000000, 0.370489320000000, 0.116212780000000),
     (5.745004084685066, 3.987559935789684, 4.223830433964055, 3)),
    ((0.232513670000000, 0.408328410000000, 0.025857450000000),
     (5.716748436340575, 1.804037706983459, 4.877534962939156, 3)),
    ((0.057045980000000, 0.559902990000000, 0.012218620000000),
     (5.313401400943851, 1.030507983615428, 8.042375156649795, 3)),
    ((0.094054280000000, 0.519164210000000, 0.022680150000000),
     (5.758224302599046, 1.654193650718819, 9.505587064089532, 3)),
    ((0.063063050000000, 0.543365260000000, 0.036103700000000),
     (5.982440185520268, 2.210976690548788, 14.800767502406950, 3)),
    ((0.232503420000000, 0.418333420000000, 0.055991300000000),
     (5.456131530261654, 2.805568287031161, 6.645891042225424, 3)),
    ((0.226305230000000, 0.391632040000000, 0.055971160000000),
     (7.835583297178577, 2.805049961496140, 6.238116104100503, 3)),
    ((0.158580550000000, 0.429168140000000, 0.052599720000000),
     (8.277360898674804, 2.716343806369254, 10.349070921928152, 3)),
    ((0.140287720000000, 0.462820230000000, 0.415890470000000),
     (8.319398373380178, 6.971801563527217, 23.511305166143138, 3)),
    ((0.292716680000000, 0.335310510000000, 0.373267920000000),
     (8.444607396784560, 6.657492273933433, 2.410001695449483, 3)),
    ((0.172538110000000, 0.437867780000000, 0.336869940000000),
     (8.306253661690928, 6.371190713569692, 17.505739300252280, 3)),
    ((0.091803670000000, 0.468237520000000, 0.051511760000000),
     (8.136676061916708, 2.686857271264772, 14.647241441839769, 3)),
    ((0.109038460000000, 0.448935180000000, 0.035954620000000),
     (8.480473141446918, 2.205704472173196, 11.877741549341193, 3)),
    ((0.242869300000000, 0.370943760000000, 0.040601190000000),
     (8.452813646934851, 2.363050544449938, 4.606241088392112, 3)),
    ((0.277711660000000, 0.349948320000000, 0.235745640000000),
     (8.305392474490336, 5.460535497258294, 3.902088380849585, 3)),
    ((0.058679720000000, 0.505026480000000, 0.198912290000000),
     (8.185433274836029, 5.069933372464482, 28.120377018384665, 3)),
    ((0.259303870000000, 0.373494110000000, 0.268745770000000),
     (7.539538441031937, 5.779995583306195, 6.641297576530984, 3)),
    ((0.122848260000000, 0.472116840000000, 0.213880940000000),
     (7.959727933812406, 5.233597701617976, 20.290421956393593, 3)),
    ((0.089068200000000, 0.487037910000000, 0.270589980000000),
     (8.429197168007434, 5.797128296716738, 26.464916551909961, 3)),
    ((0.270183570000000, 0.351381820000000, 0.768041860000000),
     (0.002911633501341, 9.005161480780865, 6.052147995993515, 2)),
    ((0.043105300000000, 0.456346230000000, 0.120746550000000),
     (9.909094065799948, 4.057805004330375, 25.772345770419661, 3)),
    ((0.165226690000000, 0.408813590000000, 0.180148750000000),
     (9.997628426551737, 4.853695951188125, 13.711432917514352, 3)),
    ((0.233498720000000, 0.375369890000000, 0.144764920000000),
     (9.280500053668934, 4.404206871605598, 6.996679713939470, 3)),
    ((0.231599500000000, 0.372077260000000, 0.203515630000000),
     (9.996848968829886, 5.121046204055429, 7.946601187910490, 3)),
    ((0.083013720000000, 0.453352650000000, 0.253047550000000),
     (9.999565269457635, 5.631186454479625, 26.172410781131642, 3)),
    ((0.201830260000000, 0.365615440000000, 0.395260580000000),
     (2.492755334964887, 6.822278726706118, 12.637144808932388, 2)),
    ((0.063407590000000, 0.371211870000000, 0.079755360000000),
     (2.498357789866246, 3.343559577006007, 19.166881554015059, 2)),
    ((0.160446340000000, 0.347074260000000, 0.101456050000000),
     (3.288950283217948, 3.745747769937435, 10.316221955381685, 2)),
    ((0.244166480000000, 0.334347370000000, 0.077748190000000),
     (3.083448369727106, 3.302788969299710, 4.031478930213133, 2)),
    ((0.281059360000000, 0.332708800000000, 0.889376780000000),
     (2.556904703831329, 9.550557838390485, 8.259294961927313, 2)),
    ((0.243289610000000, 0.318685670000000, 0.129319780000000),
     (5.961374986700250, 4.186123350016321, 4.311062434737238, 2)),
    ((0.104711160000000, 0.309380220000000, 0.155498150000000),
     (5.628355118072585, 4.547202442474988, 16.561283650587430, 2)),
    ((0.086245200000000, 0.302689150000000, 0.159007130000000),
     (5.853797676384849, 4.592599710279466, 18.829597868809777, 2)),
    ((0.104970410000000, 0.324518980000000, 0.221916450000000),
     (5.145089894686512, 5.318575501712839, 18.972215318595008, 2)),
    ((0.168946410000000, 0.330877420000000, 0.293123710000000),
     (5.290854458501917, 6.000000000000000, 13.596495394774379, 2)),
    ((0.161449650000000, 0.331338290000000, 0.340185920000000),
     (5.416047434705886, 6.398031131984855, 15.175835231610054, 2)),
    ((0.258640130000000, 0.314153790000000, 0.282057530000000),
     (8.202555916772454, 5.902107941302051, 4.026097765262806, 2)),
    ((0.157958680000000, 0.264173180000000, 0.113776780000000),
     (0.012536235321443, 3.949081770241679, 9.188897738692464, 1)),
    ((0.128624770000000, 0.256165570000000, 0.085395170000000),
     (9.261236783464240, 3.454517723556683, 10.589381636411930, 2)),
    ((0.250582880000000, 0.293290960000000, 0.177965850000000),
     (3.241079303134331, 4.827650836237364, 3.743365882969607, 1)),
    ((0.188308940000000, 0.261928670000000, 0.137402850000000),
     (2.515892094966077, 4.302204353374153, 7.397111178524947, 1)),
    ((0.168407600000000, 0.250298780000000, 0.139346970000000),
     (2.513142783736173, 4.329470861478244, 8.858704782948033, 1)),
    ((0.195164800000000, 0.277169570000000, 0.513067850000000),
     (2.501378936441103, 7.620094307256593, 10.879242856664603, 1)),
    ((0.199353060000000, 0.277833290000000, 0.440604770000000),
     (2.501347396657643, 7.144999662322620, 10.096343120964025, 1)),
    ((0.263085120000000, 0.304621200000000, 0.526104510000000),
     (2.608670482131163, 7.700939484497191, 4.240922869043813, 1)),
    ((0.253241600000000, 0.302915550000000, 0.671531390000000),
     (2.497079955512378, 8.524455640257985, 5.361811270572150, 1)),
    ((0.248419330000000, 0.298696200000000, 0.438338320000000),
     (2.504562981475109, 7.129372137361416, 5.419670697065572, 1)),
    ((0.208213300000000, 0.283569910000000, 0.527336090000000),
     (2.513177157741247, 7.708509847460905, 9.614914133684164, 1)),
    ((0.239396540000000, 0.292061100000000, 0.431445380000000),
     (3.265189619019777, 7.081532506587583, 6.220404225264666, 1)),
    ((0.182798590000000, 0.271226620000000, 0.521992380000000),
     (2.503597855026785, 7.675579434243279, 12.259534701163053, 1)),
    ((0.164495120000000, 0.243710380000000, 0.086862990000000),
     (2.513424745241100, 3.482580761459573, 7.766187653700966, 1)),
    ((0.167243930000000, 0.243667940000000, 0.064802270000000),
     (2.501110830370967, 3.020783164448117, 6.995204172520695, 1)),
    ((0.198814870000000, 0.260711060000000, 0.069276890000000),
     (3.040289315642877, 3.122317474257397, 5.200639560347585, 1)),
    ((0.182537780000000, 0.230182150000000, 0.034606350000000),
     (5.597657625685617, 2.157282377122426, 5.162764461216634, 1)),
    ((0.169263030000000, 0.224968730000000, 0.062379280000000),
     (5.096342794277389, 2.963755163552476, 6.858793778789698, 1)),
    ((0.203984930000000, 0.251347100000000, 0.054734030000000),
     (5.196969271609682, 2.772951672996174, 4.559422294586083, 1)),
    ((0.281400410000000, 0.303780910000000, 0.230818280000000),
     (5.496561269157576, 5.410551350846415, 2.000813675440119, 1)),
    ((0.152313310000000, 0.213840660000000, 0.258833480000000),
     (5.856496115186554, 5.686667585056314, 13.288114571635580, 1)),
    ((0.159350600000000, 0.226707220000000, 0.401143260000000),
     (5.273849130045392, 6.865396659362913, 14.101257686332218, 1)),
    ((0.226749340000000, 0.260339970000000, 0.171101850000000),
     (7.873484311933980, 4.744438161887368, 5.598894447763097, 1)),
    ((0.205694720000000, 0.240484700000000, 0.157006950000000),
     (8.315104802352762, 4.566800408984840, 7.088011481510175, 1)),
    ((0.113592180000000, 0.158519290000000, 0.158514980000000),
     (7.847260539813377, 4.586270676721280, 16.224610174484769, 1)),
    ((0.134468680000000, 0.174562230000000, 0.156652850000000),
     (8.383036867929183, 4.562211614435363, 13.972942153711159, 1)),
    ((0.202956370000000, 0.237589180000000, 0.074646450000000),
     (7.856435533177164, 3.238350319314909, 5.358491628406157, 1)),
    ((0.160209080000000, 0.201608330000000, 0.110960530000000),
     (7.634707868173862, 3.903923806923521, 9.768736791650276, 1)),
    ((0.179462920000000, 0.225460560000000, 0.334069300000000),
     (7.647721287507236, 6.348396276976587, 11.704570140652267, 1)),
    ((0.195848860000000, 0.218742310000000, 0.052647740000000),
     (9.989835273170078, 2.717635221781741, 5.416015499804008, 1)),
    ((0.259504930000000, 0.284944060000000, 0.602601130000000),
     (9.195888193447592, 8.150782937003752, 5.072514995394412, 1)),
    ((0.221707770000000, 0.249284910000000, 0.297639740000000),
     (9.999339091088224, 6.040694769916541, 7.762161515915440, 1)),
    ((0.233731450000000, 0.241712070000000, 0.088315480000000),
     (2.872500827433520, 3.510038916922015, 4.494437475944106, 10)),
    ((0.292103380000000, 0.301929240000000, 0.751275470000000),
     (2.557356680932417, 8.924876653502857, 2.493240458692857, 10)),
    ((0.125853500000000, 0.127641090000000, 0.162973120000000),
     (2.500736083893322, 4.643160321584332, 18.831399277234545, 10)),
    ((0.242273090000000, 0.254369980000000, 0.186247480000000),
     (2.493358398116623, 4.925443018249129, 5.417488749765833, 10)),
    ((0.252498670000000, 0.246281890000000, 0.135369500000000),
     (5.947831282193707, 4.273422498949389, 4.896630871834125, 10)),
    ((0.225492840000000, 0.218062100000000, 0.161927920000000),
     (5.532284159657728, 4.629910934986653, 7.752252929326820, 10)),
    ((0.153449500000000, 0.116740720000000, 0.159056920000000),
     (5.922075417311818, 4.593239333657982, 19.565654953046173, 10)),
    ((0.223587200000000, 0.206688640000000, 0.042533570000000),
     (5.951775900304748, 2.424635039644159, 4.953911358694802, 10)),
    ((0.175732160000000, 0.165781460000000, 0.273646370000000),
     (5.200821696085480, 5.825367168245284, 15.513073935981227, 10)),
    ((0.274011030000000, 0.274019350000000, 0.234511770000000),
     (5.890663845819670, 5.448079002981345, 3.774035430896981, 10)),
    ((0.207591300000000, 0.194642740000000, 0.219401660000000),
     (5.817623930263078, 5.292185853604325, 11.318160969256796, 10)),
    ((0.210310180000000, 0.150341680000000, 0.038889340000000),
     (8.373369647938560, 2.306688368985876, 8.631037795394968, 10)),
    ((0.231828630000000, 0.198258060000000, 0.052912060000000),
     (7.712159021952001, 2.724728704051856, 5.730016018596034, 10)),
    ((0.249572350000000, 0.210068230000000, 0.315876130000000),
     (9.994821305584765, 6.197418387200809, 10.969746297783991, 10)),
    ((0.293066540000000, 0.289176180000000, 0.324665270000000),
     (9.112675897084408, 6.270996610193586, 2.750829451410893, 10)),
    ((0.214413040000000, 0.138143750000000, 0.197167230000000),
     (9.998713332858582, 5.050367418553515, 19.168834467026883, 10)),
    ((0.209418290000000, 0.143215410000000, 0.243271190000000),
     (9.382928745299282, 5.535664936769356, 18.821027597932278, 10)),
    ((0.285412990000000, 0.279139070000000, 0.540060240000000),
     (9.002080407348938, 7.786061808645494, 4.453888691443739, 10)),
    ((0.292304690000000, 0.286562190000000, 0.524657620000000),
     (9.991235613354732, 7.692030973969491, 3.652324068875179, 10)),
    ((0.188041240000000, 0.081374670000000, 0.065803980000000),
     (9.042565422303745, 3.043925936875429, 22.292781603620266, 10)),
    ((0.220259580000000, 0.151808990000000, 0.065512570000000),
     (9.461821985933510, 3.037218805732587, 10.554387963579108, 10)),
    ((0.278871670000000, 0.245432170000000, 0.574509620000000),
     (2.501572839035511, 7.990213833073329, 8.383060885731012, 9)),
    ((0.274876240000000, 0.233763570000000, 0.463227480000000),
     (2.499665174466825, 7.298301026831824, 9.502846729929050, 9)),
    ((0.283568640000000, 0.251900500000000, 0.459806640000000),
     (2.861869979585210, 7.275426003257595, 7.466115678753070, 9)),
    ((0.303335960000000, 0.300052160000000, 0.664010660000000),
     (3.082129094210853, 8.485000568140805, 2.493500289871143, 9)),
    ((0.238354670000000, 0.115580360000000, 0.098276690000000),
     (2.503749921088607, 3.690667757119861, 19.779266412450269, 9)),
    ((0.253614500000000, 0.163874850000000, 0.099008500000000),
     (2.499285756311949, 3.703448918491747, 12.087654980113633, 9)),
    ((0.285357130000000, 0.251149710000000, 0.084291090000000),
     (2.489031809529632, 3.433194334498058, 3.381492920102487, 9)),
    ((0.297015040000000, 0.280766720000000, 0.116523270000000),
     (2.836526130092274, 3.992426600758435, 2.000804943904234, 9)),
    ((0.248942940000000, 0.135133110000000, 0.075078500000000),
     (2.880116623996170, 3.247434448470748, 14.610390101910370, 9)),
    ((0.289764350000000, 0.235510780000000, 0.320306800000000),
     (5.164415919630265, 6.234662720232918, 9.113326052349766, 9)),
    ((0.286992170000000, 0.212273900000000, 0.383761560000000),
     (5.500136738127175, 6.736841234298917, 13.153893654633693, 9)),
    ((0.294231800000000, 0.244834820000000, 0.415686030000000),
     (5.535582092968188, 6.970342546444610, 8.892567135407102, 9)),
    ((0.271128660000000, 0.108925590000000, 0.091372760000000),
     (5.589639732551746, 3.566860887599992, 22.756075736806582, 9)),
    ((0.269325620000000, 0.118719220000000, 0.074569750000000),
     (5.281680439564447, 3.236733901648726, 18.731686204369552, 9)),
    ((0.287744460000000, 0.211498570000000, 0.064095530000000),
     (5.173297653271298, 3.004304986875510, 6.229409296844157, 9)),
    ((0.284385140000000, 0.183610320000000, 0.087510060000000),
     (5.552784726302882, 3.494850862760521, 10.076128044498652, 9)),
    ((0.274663640000000, 0.116233240000000, 0.044591640000000),
     (5.729543881773527, 2.488034275716341, 15.986078477134842, 9)),
    ((0.305269170000000, 0.297876170000000, 0.384387660000000),
     (5.069713488890670, 6.741533285380119, 2.219542488732086, 9)),
    ((0.277330840000000, 0.167648060000000, 0.245841180000000),
     (5.383068359312304, 5.560997886025167, 18.302040299981690, 9)),
    ((0.298417810000000, 0.263256360000000, 0.259028730000000),
     (5.357348971068703, 5.688526651960111, 5.255330957728729, 9)),
    ((0.304295300000000, 0.116118320000000, 0.043870000000000),
     (8.284882756207175, 2.466049771912841, 17.239324057538755, 9)),
    ((0.316707900000000, 0.228355110000000, 0.528296570000000),
     (8.143701921833522, 7.714405879305765, 13.532589823660204, 9)),
    ((0.316649560000000, 0.204542650000000, 0.455628270000000),
     (8.095067300354142, 7.247339823928828, 17.337282390950395, 9)),
    ((0.313001370000000, 0.239828280000000, 0.402106130000000),
     (7.828669695023791, 6.872416979415612, 10.705903656469590, 9)),
    ((0.311878720000000, 0.266671570000000, 0.331902180000000),
     (7.799718001735325, 6.330678302366303, 6.211065875095611, 9)),
    ((0.315379040000000, 0.210527650000000, 0.393354920000000),
     (8.044182832558544, 6.808226296488199, 15.556673432533850, 9)),
    ((0.305941320000000, 0.181527170000000, 0.142440720000000),
     (7.612535237679090, 4.372367444809608, 14.168811791437310, 9)),
    ((0.311959680000000, 0.120892290000000, 0.151020950000000),
     (8.169231340583814, 4.488334714008468, 27.235344441294184, 9)),
    ((0.336186720000000, 0.175892680000000, 0.242493860000000),
     (9.602007767814236, 5.527970670885737, 20.580420190036015, 9)),
    ((0.342076270000000, 0.138756160000000, 0.241385970000000),
     (9.663339737858312, 5.516978443905920, 29.047422373609325, 9)),
    ((0.319230030000000, 0.282911530000000, 0.255044880000000),
     (9.997637328587867, 5.650424898482105, 4.215952139435246, 9)),
    ((0.336396410000000, 0.207774810000000, 0.233327480000000),
     (9.997329060311806, 5.436085468960591, 14.778819186976173, 9)),
    ((0.314645070000000, 0.301078800000000, 0.270408070000000),
     (9.998181289548320, 5.795441484549336, 1.989415064944841, 9)),
    ((0.326227860000000, 0.236791530000000, 0.283386470000000),
     (9.495786346209876, 5.914105474642816, 10.808393081380504, 9)),
    ((0.319647890000000, 0.197023370000000, 0.029884880000000),
     (9.825528611679406, 1.975999308579267, 7.066915765354878, 9)),
    ((0.332024160000000, 0.162933160000000, 0.168289020000000),
     (9.383343036273200, 4.709738690312933, 20.001567691188572, 9)),
    ((0.318834100000000, 0.261194140000000, 0.191495170000000),
     (9.115136549174583, 4.986025423238965, 5.883351646224200, 9)),
    ((0.344973020000000, 0.147405810000000, 0.176747910000000),
     (9.999366179917253, 4.813032963700893, 24.745869339266416, 9)),
    ((0.333960660000000, 0.132042280000000, 0.157592690000000),
     (9.378821700655990, 4.574376792767962, 26.296103195549775, 9)),
    ((0.342631920000000, 0.249282600000000, 0.049664620000000),
     (2.500037811120279, 2.635760366412213, 4.711601146262771, 8)),
    ((0.378638850000000, 0.148055700000000, 0.031334760000000),
     (2.889989048496027, 2.033768033611565, 13.996029826620704, 8)),
    ((0.360672870000000, 0.225086940000000, 0.036643060000000),
     (3.435642502603535, 2.229919149923648, 6.721902205502408, 8)),
    ((0.355839720000000, 0.208903690000000, 0.028740300000000),
     (2.992392870634681, 1.928933382936380, 7.195747540270221, 8)),
    ((0.347282990000000, 0.114026920000000, 0.017461080000000),
     (2.500584368220391, 1.372688833884682, 14.170766991414973, 8)),
    ((0.329407710000000, 0.227892780000000, 0.014893950000000),
     (2.610719016988927, 1.213770391629024, 3.348897655453758, 8)),
    ((0.319725670000000, 0.311229320000000, 0.536009480000000),
     (5.165696331210352, 7.761502798227757, 2.141042339555734, 8)),
    ((0.350121720000000, 0.293330670000000, 0.421470940000000),
     (5.047603361893334, 7.011456896127714, 6.451158237816637, 8)),
    ((0.375896610000000, 0.285071700000000, 0.669340470000000),
     (5.807823248298298, 8.512993428798092, 10.441995222524335, 8)),
    ((0.425499320000000, 0.239041770000000, 0.333290370000000),
     (5.044106185145303, 6.342035967589280, 18.425239270239359, 8)),
    ((0.346417650000000, 0.297250500000000, 0.384117680000000),
     (5.485508416723803, 6.739510600333609, 5.474628963370974, 8)),
    ((0.415216020000000, 0.259891230000000, 0.390861560000000),
     (5.876649266935097, 6.789776772017164, 15.440071996140499, 8)),
    ((0.347800420000000, 0.292840400000000, 0.036056200000000),
     (8.198063479678463, 2.209298544324237, 1.999968831556617, 8)),
    ((0.454455100000000, 0.198222450000000, 0.032017930000000),
     (7.996459595708689, 2.060313217212614, 11.656390753035632, 8)),
    ((0.338587450000000, 0.309854500000000, 0.700040060000000),
     (7.984928295826705, 8.671283058190554, 4.270518902402659, 8)),
    ((0.332396120000000, 0.312518270000000, 0.196047380000000),
     (9.535474055690418, 5.037755329068913, 1.999288540673044, 8)),
    ((0.438461810000000, 0.290963810000000, 0.231412360000000),
     (9.995880363112217, 5.416612027549052, 11.058404052106437, 8)),
    ((0.409580220000000, 0.297192220000000, 0.488828710000000),
     (9.309553338445067, 7.466203072891926, 11.186885768515551, 8)),
    ((0.443998990000000, 0.293695090000000, 0.433796870000000),
     (9.994974549532749, 7.097905906793407, 13.900038696652488, 8)),
    ((0.405549190000000, 0.297230130000000, 0.166877690000000),
     (9.924764794317937, 4.692192114726735, 7.197057639852876, 8)),
    ((0.420070030000000, 0.289308150000000, 0.167293300000000),
     (9.414977576869390, 4.697368817320752, 8.721128174425862, 8)),
    ((0.521083290000000, 0.255741460000000, 0.139995260000000),
     (0.001922018136362, 4.338509437960340, 16.467729969190529, 7)),
    ((0.376380100000000, 0.307280070000000, 0.340702890000000),
     (0.013622323425029, 6.402201258375587, 6.594463952286430, 7)),
    ((0.364953070000000, 0.308014810000000, 0.209109150000000),
     (0.010859256215312, 5.182208076594741, 4.548054950595445, 7)),
    ((0.425669120000000, 0.295640120000000, 0.282179390000000),
     (9.966442261794679, 5.903209495564671, 10.839641908347603, 8)),
    ((0.551363470000000, 0.281388920000000, 0.197121930000000),
     (2.497658061438166, 5.049858107371041, 17.884020184861189, 7)),
    ((0.538994160000000, 0.290487880000000, 0.258236340000000),
     (2.748554662960458, 5.680976668543921, 17.995724599746377, 7)),
    ((0.438548110000000, 0.310331700000000, 0.276123620000000),
     (2.494307870920611, 5.848115423309936, 10.235754640176427, 7)),
    ((0.355890690000000, 0.316553700000000, 0.246494730000000),
     (2.898397913137316, 5.567414364302070, 3.554523683911656, 7)),
    ((0.496315920000000, 0.301111910000000, 0.045705040000000),
     (5.410368617452352, 2.521454883551788, 7.258854844117816, 7)),
    ((0.603383540000000, 0.274683400000000, 0.046002130000000),
     (5.834202790114457, 2.530273170283877, 12.000615206905717, 7)),
    ((0.576197760000000, 0.315547170000000, 0.140733560000000),
     (5.978828021210861, 4.348770661536209, 14.392102528163205, 7)),
    ((0.404145470000000, 0.323107240000000, 0.133478870000000),
     (5.178376304944726, 4.246412666988667, 4.842813896150843, 7)),
    ((0.351437200000000, 0.328606940000000, 0.806796950000000),
     (8.323808896102669, 9.185953487188891, 4.236444337774098, 7)),
    ((0.552003350000000, 0.340905830000000, 0.132405210000000),
     (8.195743765912791, 4.230965436896463, 11.592666113548890, 7)),
    ((0.437192370000000, 0.336730560000000, 0.182747660000000),
     (7.609549840587148, 4.884451070500727, 6.793632746717810, 7)),
    ((0.388800590000000, 0.337836930000000, 0.704087400000000),
     (8.254754870479271, 8.691786357762131, 6.342564639006561, 7)),
    ((0.400060190000000, 0.336631470000000, 0.077902610000000),
     (9.998274746120362, 3.305950898246991, 3.115406955944987, 7)),
    ((0.672483690000000, 0.323303650000000, 0.072206490000000),
     (9.762707402102009, 3.186360682992453, 14.800047578330392, 7)),
    ((0.643549180000000, 0.339736390000000, 0.088031220000000),
     (9.998281866790068, 3.504689156556842, 13.900125103004391, 7)),
    ((0.391813640000000, 0.344694800000000, 0.722522540000000),
     (0.012146453319250, 8.784136610032485, 6.215547341236711, 6)),
    ((0.602310650000000, 0.361688450000000, 0.160681910000000),
     (9.998726208990446, 4.614045920487802, 14.686472574413189, 7)),
    ((0.547911500000000, 0.348929130000000, 0.022637830000000),
     (2.748446567605200, 1.652100043800458, 5.568003028926484, 6)),
    ((0.487941870000000, 0.382932020000000, 0.206656600000000),
     (3.141420665213015, 5.155515837617080, 8.596656843523602, 6)),
    ((0.428367330000000, 0.358917260000000, 0.143655360000000),
     (2.497847430895263, 4.389047620589019, 4.951047519073632, 6)),
    ((0.415188900000000, 0.354953060000000, 0.144080430000000),
     (2.497765367996308, 4.394863818043045, 4.383585180612334, 6)),
    ((0.465424050000000, 0.340825760000000, 0.020792530000000),
     (2.491556944128257, 1.558025122442278, 3.308961030538007, 6)),
    ((0.446372450000000, 0.389454220000000, 0.692983380000000),
     (5.039232375718989, 8.635334526629093, 9.593295926760944, 6)),
    ((0.454041340000000, 0.396471540000000, 0.692336290000000),
     (5.589825180605786, 8.632025264001571, 10.198248046129890, 6)),
    ((0.485462530000000, 0.413059200000000, 0.599362970000000),
     (5.986348538882265, 8.132531770555124, 12.306429102395787, 6)),
    ((0.479784280000000, 0.398259810000000, 0.209407830000000),
     (5.423024282295728, 5.185445616155039, 8.046653489436829, 6)),
    ((0.467859730000000, 0.392971260000000, 0.200539010000000),
     (5.362801438508162, 5.088077699814325, 7.341529902605595, 6)),
    ((0.513319050000000, 0.412659640000000, 0.217470390000000),
     (5.698466023468301, 5.271793925817517, 10.326189160215149, 6)),
    ((0.417704860000000, 0.373037470000000, 0.244307380000000),
     (5.407738597396095, 5.545898351021886, 5.292471393326970, 6)),
    ((0.374818160000000, 0.365367870000000, 0.784427720000000),
     (8.240476434727421, 9.082400712707463, 4.900546438317654, 6)),
    ((0.396201520000000, 0.379510490000000, 0.686579740000000),
     (8.233550289285581, 8.602489889558802, 5.942135876130584, 6)),
    ((0.512992240000000, 0.435289210000000, 0.152411660000000),
     (8.394173477975302, 4.506736390569333, 9.461660902200202, 6)),
    ((0.525291220000000, 0.462058630000000, 0.205125810000000),
     (9.997678885733523, 5.138757061322305, 12.705101930365435, 6)),
    ((0.382491010000000, 0.372712510000000, 0.244669960000000),
     (9.998425310708878, 5.549472940108251, 3.506732159833540, 6)),
    ((0.524655280000000, 0.463817520000000, 0.270370220000000),
     (9.998293613015804, 5.795090446536200, 14.064036258689498, 6)),
    ((0.363328270000000, 0.360426250000000, 0.417479000000000),
     (9.998229887302589, 6.983123200777477, 3.128628040783353, 6)),
    ((0.491037860000000, 0.447417820000000, 0.376253370000000),
     (9.999639780116620, 6.680204704210012, 11.629600879949830, 6)),
    ((0.532606720000000, 0.460857970000000, 0.381758790000000),
     (9.050495861543142, 6.721800644191588, 17.087646825588862, 6)),
    ((0.399476360000000, 0.386071010000000, 0.300248180000000),
     (9.993793266201919, 6.063461553492133, 4.737775043523384, 6)),
    ((0.415667510000000, 0.395908360000000, 0.412033910000000),
     (9.403400268682468, 6.944203609450778, 6.288715586566071, 6)),
    ((0.383799510000000, 0.375338290000000, 0.494998360000000),
     (9.628530025690907, 7.505827534557014, 4.623713090688181, 6)),
    ((0.501848130000000, 0.445879640000000, 0.465566280000000),
     (9.018170538969292, 7.313879376334315, 13.422606463167975, 6)),
    ((0.513819740000000, 0.455126120000000, 0.472229760000000),
     (9.262112680928013, 7.357994962704260, 15.233390464742534, 6)),
    ((0.497589670000000, 0.490496120000000, 0.452422140000000),
     (3.334519035091363, 7.225679043279789, 16.113161767033556, 5)),
    ((0.439350160000000, 0.431034520000000, 0.232596390000000),
     (2.497697946529520, 5.428663062814679, 6.543946978037446, 5)),
    ((0.372855240000000, 0.371780290000000, 0.056909880000000),
     (2.501498725356315, 2.829072412172277, 2.002239254370286, 5)),
    ((0.466258880000000, 0.458053540000000, 0.606044740000000),
     (2.500339283326589, 8.170120105651444, 12.020108503258743, 5)),
    ((0.478779960000000, 0.468932630000000, 0.714377060000000),
     (2.498862691127872, 8.743540432230319, 14.428234223724066, 5)),
    ((0.480098650000000, 0.459556640000000, 0.162637680000000),
     (2.496552079019970, 4.638913954949600, 8.380657446927768, 5)),
    ((0.373348520000000, 0.377974550000000, 0.165155240000000),
     (3.362019033713876, 4.670651625286859, 2.775534906359987, 5)),
    ((0.478561780000000, 0.504753200000000, 0.587324270000000),
     (5.338487615233340, 8.064094809657833, 16.611694509643459, 5)),
    ((0.471196890000000, 0.498768210000000, 0.654830210000000),
     (5.346059938632446, 8.436417586671546, 15.412331133566334, 5)),
    ((0.409839570000000, 0.431867080000000, 0.474305000000000),
     (5.366812938492127, 7.371653756211234, 7.038251111170072, 5)),
    ((0.420183080000000, 0.445751310000000, 0.405341000000000),
     (5.928030773207911, 6.895926932708628, 7.572887338365732, 5)),
    ((0.458411910000000, 0.483047100000000, 0.363405300000000),
     (5.727938903185166, 6.581660843240257, 10.668259627347156, 5)),
    ((0.450408670000000, 0.473710480000000, 0.347668630000000),
     (5.643292598616427, 6.458019138270375, 9.548898590442330, 5)),
    ((0.459903860000000, 0.468079330000000, 0.058578730000000),
     (5.343623409410245, 2.871097706288225, 5.430515721489820, 5)),
    ((0.397934270000000, 0.428948000000000, 0.139301270000000),
     (7.752957755299303, 4.328832715073768, 4.269028839349022, 5)),
    ((0.437899030000000, 0.483376830000000, 0.181103130000000),
     (8.147774342384855, 4.865021683103260, 7.545599812926251, 5)),
    ((0.457858800000000, 0.506239240000000, 0.259014910000000),
     (7.907631298810656, 5.688395094215606, 10.778720186943521, 5)),
    ((0.399850080000000, 0.431164570000000, 0.201521950000000),
     (7.592789164109943, 5.098997586200677, 4.937003269893032, 5)),
    ((0.372269610000000, 0.397381200000000, 0.233817160000000),
     (7.675084918755988, 5.441049037053221, 3.553157956788445, 5)),
    ((0.358999800000000, 0.382509510000000, 0.367889850000000),
     (7.992180735900773, 6.616295461168346, 3.285836544709042, 5)),
    ((0.351807080000000, 0.377980880000000, 0.559042880000000),
     (9.711479428331863, 7.899577772446339, 3.357708715547692, 5)))

XYY_TO_MUNSELL_GREYS_SPECIFICATIONS = (
    ((0.31006, 0.31616, 0.00000000), 0.000000000000000),
    ((0.31006, 0.31616, 0.00473582), 0.416666548555144),
    ((0.31006, 0.31616, 0.00961944), 0.833333381418545),
    ((0.31006, 0.31616, 0.01545756), 1.249999720639959),
    ((0.31006, 0.31616, 0.02293343), 1.666666607627812),
    ((0.31006, 0.31616, 0.03261914), 2.083333449681369),
    ((0.31006, 0.31616, 0.04498800), 2.500000058759632),
    ((0.31006, 0.31616, 0.06042690), 2.916666583317045),
    ((0.31006, 0.31616, 0.07924864), 3.333333246296839),
    ((0.31006, 0.31616, 0.10170428), 3.749999998453559),
    ((0.31006, 0.31616, 0.12799549), 4.166666677349198),
    ((0.31006, 0.31616, 0.15828689), 4.583333263574459),
    ((0.31006, 0.31616, 0.19271844), 5.000000000000000),
    ((0.31006, 0.31616, 0.23141772), 5.416666667938872),
    ((0.31006, 0.31616, 0.27451233), 5.833333278917798),
    ((0.31006, 0.31616, 0.32214224), 6.249999992443541),
    ((0.31006, 0.31616, 0.37447210), 6.666666652206092),
    ((0.31006, 0.31616, 0.43170362), 7.083333340369354),
    ((0.31006, 0.31616, 0.49408790), 7.499999997498019),
    ((0.31006, 0.31616, 0.56193781), 7.916666646517515),
    ((0.31006, 0.31616, 0.63564030), 8.333333327302425),
    ((0.31006, 0.31616, 0.71566876), 8.749999993544938),
    ((0.31006, 0.31616, 0.80259539), 9.166666622097262),
    ((0.31006, 0.31616, 0.89710353), 9.583333299971065),
    ((0.31006, 0.31616, 1.00000000), 10.000000000000000))

NON_CONVERGING_XYY = (
    (0.33159302, 0.43388935, 0.89380734),
    (0.25883700, 0.71096717, 0.01266934),
    (0.28517207, 0.38369148, 0.89445395),
    (0.28155768, 0.33248001, 0.89992977),
    (0.29624596, 0.31950269, 0.96665647),
    (0.13833192, 0.07953813, 0.02236117),
    (0.22907253, 0.06719948, 0.06903321))


class TestMunsellValuePriest1920(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.munsell_value_Priest1920` definition
    unit tests methods.
    """

    def test_munsell_value_Priest1920(self):
        """
        Tests :func:`colour.notation.munsell.munsell_value_Priest1920`
        definition.
        """

        self.assertAlmostEqual(
            munsell_value_Priest1920(10.08),
            3.17490157,
            places=7)

        self.assertAlmostEqual(
            munsell_value_Priest1920(56.76),
            7.53392328,
            places=7)

        self.assertAlmostEqual(
            munsell_value_Priest1920(98.32),
            9.91564420,
            places=7)

    def test_n_dimensional_munsell_value_Priest1920(self):
        """
        Tests :func:`colour.notation.munsell.munsell_value_Priest1920`
        definition n-dimensional arrays support.
        """

        Y = 10.08
        V = 3.17490157
        np.testing.assert_almost_equal(
            munsell_value_Priest1920(Y),
            V,
            decimal=7)

        V = np.tile(V, 6)
        Y = np.tile(Y, 6)
        np.testing.assert_almost_equal(
            munsell_value_Priest1920(Y),
            V,
            decimal=7)

        V = np.reshape(V, (2, 3))
        Y = np.reshape(Y, (2, 3))
        np.testing.assert_almost_equal(
            munsell_value_Priest1920(Y),
            V,
            decimal=7)

        V = np.reshape(V, (2, 3, 1))
        Y = np.reshape(Y, (2, 3, 1))
        np.testing.assert_almost_equal(
            munsell_value_Priest1920(Y),
            V,
            decimal=7)

    @ignore_numpy_errors
    def test_nan_munsell_value_Priest1920(self):
        """
        Tests :func:`colour.notation.munsell.munsell_value_Priest1920`
        definition nan support.
        """

        munsell_value_Priest1920(
            np.array([-1.0, 0.0, 1.0, -np.inf, np.inf, np.nan]))


class TestMunsellValueMunsell1933(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.munsell_value_Munsell1933`
    definition unit tests methods.
    """

    def test_munsell_value_Munsell1933(self):
        """
        Tests :func:`colour.notation.munsell.munsell_value_Munsell1933`
        definition.
        """

        self.assertAlmostEqual(
            munsell_value_Munsell1933(10.08),
            3.79183555,
            places=7)

        self.assertAlmostEqual(
            munsell_value_Munsell1933(56.76),
            8.27013182,
            places=7)

        self.assertAlmostEqual(
            munsell_value_Munsell1933(98.32),
            9.95457711,
            places=7)

    def test_n_dimensional_munsell_value_Munsell1933(self):
        """
        Tests :func:`colour.notation.munsell.munsell_value_Munsell1933`
        definition n-dimensional arrays support.
        """

        Y = 10.08
        V = 3.79183555
        np.testing.assert_almost_equal(
            munsell_value_Munsell1933(Y),
            V,
            decimal=7)

        V = np.tile(V, 6)
        Y = np.tile(Y, 6)
        np.testing.assert_almost_equal(
            munsell_value_Munsell1933(Y),
            V,
            decimal=7)

        V = np.reshape(V, (2, 3))
        Y = np.reshape(Y, (2, 3))
        np.testing.assert_almost_equal(
            munsell_value_Munsell1933(Y),
            V,
            decimal=7)

        V = np.reshape(V, (2, 3, 1))
        Y = np.reshape(Y, (2, 3, 1))
        np.testing.assert_almost_equal(
            munsell_value_Munsell1933(Y),
            V,
            decimal=7)

    @ignore_numpy_errors
    def test_nan_munsell_value_Munsell1933(self):
        """
        Tests :func:`colour.notation.munsell.munsell_value_Munsell1933`
        definition nan support.
        """

        munsell_value_Munsell1933(
            np.array([-1.0, 0.0, 1.0, -np.inf, np.inf, np.nan]))


class TestMunsellValueMoon1943(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.munsell_value_Moon1943` definition
    unit tests methods.
    """

    def test_munsell_value_Moon1943(self):
        """
        Tests :func:`colour.notation.munsell.munsell_value_Moon1943`
        definition.
        """

        self.assertAlmostEqual(
            munsell_value_Moon1943(10.08),
            3.74629715,
            places=7)

        self.assertAlmostEqual(
            munsell_value_Moon1943(56.76),
            7.82258143,
            places=7)

        self.assertAlmostEqual(
            munsell_value_Moon1943(98.32),
            9.88538236,
            places=7)

    def test_n_dimensional_munsell_value_Moon1943(self):
        """
        Tests :func:`colour.notation.munsell.munsell_value_Moon1943`
        definition n-dimensional arrays support.
        """

        Y = 10.08
        V = 3.74629715
        np.testing.assert_almost_equal(
            munsell_value_Moon1943(Y),
            V,
            decimal=7)

        V = np.tile(V, 6)
        Y = np.tile(Y, 6)
        np.testing.assert_almost_equal(
            munsell_value_Moon1943(Y),
            V,
            decimal=7)

        V = np.reshape(V, (2, 3))
        Y = np.reshape(Y, (2, 3))
        np.testing.assert_almost_equal(
            munsell_value_Moon1943(Y),
            V,
            decimal=7)

        V = np.reshape(V, (2, 3, 1))
        Y = np.reshape(Y, (2, 3, 1))
        np.testing.assert_almost_equal(
            munsell_value_Moon1943(Y),
            V,
            decimal=7)

    @ignore_numpy_errors
    def test_nan_munsell_value_Moon1943(self):
        """
        Tests :func:`colour.notation.munsell.munsell_value_Moon1943`
        definition nan support.
        """

        munsell_value_Moon1943(
            np.array([-1.0, 0.0, 1.0, -np.inf, np.inf, np.nan]))


class TestMunsellValueSaunderson1944(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.munsell_value_Saunderson1944`
    definition unit tests methods.
    """

    def test_munsell_value_Saunderson1944(self):
        """
        Tests :func:`colour.notation.munsell.munsell_value_Saunderson1944`
        definition.
        """

        self.assertAlmostEqual(
            munsell_value_Saunderson1944(10.08),
            3.68650806,
            places=7)

        self.assertAlmostEqual(
            munsell_value_Saunderson1944(56.76),
            7.89881184,
            places=7)

        self.assertAlmostEqual(
            munsell_value_Saunderson1944(98.32),
            9.85197101,
            places=7)

    def test_n_dimensional_munsell_value_Saunderson1944(self):
        """
        Tests :func:`colour.notation.munsell.munsell_value_Saunderson1944`
        definition n-dimensional arrays support.
        """

        Y = 10.08
        V = 3.68650806
        np.testing.assert_almost_equal(
            munsell_value_Saunderson1944(Y),
            V,
            decimal=7)

        V = np.tile(V, 6)
        Y = np.tile(Y, 6)
        np.testing.assert_almost_equal(
            munsell_value_Saunderson1944(Y),
            V,
            decimal=7)

        V = np.reshape(V, (2, 3))
        Y = np.reshape(Y, (2, 3))
        np.testing.assert_almost_equal(
            munsell_value_Saunderson1944(Y),
            V,
            decimal=7)

        V = np.reshape(V, (2, 3, 1))
        Y = np.reshape(Y, (2, 3, 1))
        np.testing.assert_almost_equal(
            munsell_value_Saunderson1944(Y),
            V,
            decimal=7)

    @ignore_numpy_errors
    def test_nan_munsell_value_Saunderson1944(self):
        """
        Tests :func:`colour.notation.munsell.munsell_value_Saunderson1944`
        definition nan support.
        """

        munsell_value_Saunderson1944(
            np.array([-1.0, 0.0, 1.0, -np.inf, np.inf, np.nan]))


class TestMunsellValueLadd1955(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.munsell_value_Ladd1955` definition
    unit tests methods.
    """

    def test_munsell_value_Ladd1955(self):
        """
        Tests :func:`colour.notation.munsell.munsell_value_Ladd1955`
        definition.
        """

        self.assertAlmostEqual(
            munsell_value_Ladd1955(10.08),
            3.69528622,
            places=7)

        self.assertAlmostEqual(
            munsell_value_Ladd1955(56.76),
            7.84875137,
            places=7)

        self.assertAlmostEqual(
            munsell_value_Ladd1955(98.32),
            9.75492814,
            places=7)

    def test_n_dimensional_munsell_value_Ladd1955(self):
        """
        Tests :func:`colour.notation.munsell.munsell_value_Ladd1955`
        definition n-dimensional arrays support.
        """

        Y = 10.08
        V = 3.69528622
        np.testing.assert_almost_equal(
            munsell_value_Ladd1955(Y),
            V,
            decimal=7)

        V = np.tile(V, 6)
        Y = np.tile(Y, 6)
        np.testing.assert_almost_equal(
            munsell_value_Ladd1955(Y),
            V,
            decimal=7)

        V = np.reshape(V, (2, 3))
        Y = np.reshape(Y, (2, 3))
        np.testing.assert_almost_equal(
            munsell_value_Ladd1955(Y),
            V,
            decimal=7)

        V = np.reshape(V, (2, 3, 1))
        Y = np.reshape(Y, (2, 3, 1))
        np.testing.assert_almost_equal(
            munsell_value_Ladd1955(Y),
            V,
            decimal=7)

    @ignore_numpy_errors
    def test_nan_munsell_value_Ladd1955(self):
        """
        Tests :func:`colour.notation.munsell.munsell_value_Ladd1955`
        definition nan support.
        """

        munsell_value_Ladd1955(
            np.array([-1.0, 0.0, 1.0, -np.inf, np.inf, np.nan]))


class TestMunsellValueMcCamy1992(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.munsell_value_McCamy1987` definition
    unit tests methods.
    """

    def test_munsell_value_McCamy1987(self):
        """
        Tests :func:`colour.notation.munsell.munsell_value_McCamy1987`
        definition.
        """

        self.assertAlmostEqual(
            munsell_value_McCamy1987(10.08),
            3.734723525854632,
            places=7)

        self.assertAlmostEqual(
            munsell_value_McCamy1987(56.76),
            7.951693123402992,
            places=7)

        self.assertAlmostEqual(
            munsell_value_McCamy1987(98.32),
            9.935265378231314,
            places=7)

    def test_n_dimensional_munsell_value_McCamy1987(self):
        """
        Tests :func:`colour.notation.munsell.munsell_value_McCamy1987`
        definition n-dimensional arrays support.
        """

        Y = 10.08
        V = 3.734723525854632
        np.testing.assert_almost_equal(
            munsell_value_McCamy1987(Y),
            V,
            decimal=7)

        V = np.tile(V, 6)
        Y = np.tile(Y, 6)
        np.testing.assert_almost_equal(
            munsell_value_McCamy1987(Y),
            V,
            decimal=7)

        V = np.reshape(V, (2, 3))
        Y = np.reshape(Y, (2, 3))
        np.testing.assert_almost_equal(
            munsell_value_McCamy1987(Y),
            V,
            decimal=7)

        V = np.reshape(V, (2, 3, 1))
        Y = np.reshape(Y, (2, 3, 1))
        np.testing.assert_almost_equal(
            munsell_value_McCamy1987(Y),
            V,
            decimal=7)

    @ignore_numpy_errors
    def test_nan_munsell_value_McCamy1987(self):
        """
        Tests :func:`colour.notation.munsell.munsell_value_McCamy1987`
        definition nan support.
        """

        munsell_value_McCamy1987(
            np.array([-1.0, 0.0, 1.0, -np.inf, np.inf, np.nan]))


class TestMunsellValueASTMD153508(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.munsell_value_ASTMD153508`
    definition unit tests methods.
    """

    def test_munsell_value_ASTMD153508(self):
        """
        Tests :func:`colour.notation.munsell.munsell_value_ASTMD153508`
        definition.
        """

        self.assertAlmostEqual(
            munsell_value_ASTMD153508(10.08),
            3.734476476931135,
            places=7)

        self.assertAlmostEqual(
            munsell_value_ASTMD153508(56.76),
            7.94992421,
            places=7)

        self.assertAlmostEqual(
            munsell_value_ASTMD153508(98.32),
            9.934372853115544,
            places=7)

    def test_n_dimensional_munsell_value_ASTMD153508(self):
        """
        Tests :func:`colour.notation.munsell.munsell_value_ASTMD153508`
        definition n-dimensional arrays support.
        """

        Y = 10.08
        V = 3.734476476931135
        np.testing.assert_almost_equal(
            munsell_value_ASTMD153508(Y),
            V,
            decimal=7)

        V = np.tile(V, 6)
        Y = np.tile(Y, 6)
        np.testing.assert_almost_equal(
            munsell_value_ASTMD153508(Y),
            V,
            decimal=7)

        V = np.reshape(V, (2, 3))
        Y = np.reshape(Y, (2, 3))
        np.testing.assert_almost_equal(
            munsell_value_ASTMD153508(Y),
            V,
            decimal=7)

        V = np.reshape(V, (2, 3, 1))
        Y = np.reshape(Y, (2, 3, 1))
        np.testing.assert_almost_equal(
            munsell_value_ASTMD153508(Y),
            V,
            decimal=7)

    @ignore_numpy_errors
    def test_nan_munsell_value_ASTMD153508(self):
        """
        Tests :func:`colour.notation.munsell.munsell_value_ASTMD153508`
        definition nan support.
        """

        munsell_value_ASTMD153508(
            np.array([-1.0, 0.0, 1.0, -np.inf, np.inf, np.nan]))


class TestMunsellSpecification_to_xyY(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.munsell_specification_to_xyY`
    definition unit tests methods.
    """

    def test_munsell_specification_to_xyY(self):
        """
        Tests :func:`colour.notation.munsell.munsell_specification_to_xyY`
        definition.
        """

        for i, specification in enumerate(MUNSELL_SPECIFICATIONS):
            np.testing.assert_almost_equal(
                munsell_specification_to_xyY(specification),
                np.array(MUNSELL_COLOURS_TO_XYY[i]),
                decimal=7)

        for i, specification in enumerate(MUNSELL_GREYS_SPECIFICATIONS):
            np.testing.assert_almost_equal(
                munsell_specification_to_xyY(specification),
                np.array(MUNSELL_GREYS_TO_XYY[i]),
                decimal=7)


class TestMunsellColour_to_xyY(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.munsell_colour_to_xyY` definition
    unit tests methods.
    """

    def test_munsell_colour_to_xyY(self):
        """
        Tests :func:`colour.notation.munsell.munsell_colour_to_xyY` definition.
        """

        # TODO: This test is covered by the previous class,
        # do we need a dedicated one?

        pass


class TestxyY_to_munsell_specification(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.xyY_to_munsell_specification`
    definition unit tests methods.
    """

    def test_xyY_to_munsell_specification(self):
        """
        Tests :func:`colour.notation.munsell.xyY_to_munsell_specification`
        definition.
        """

        for xyY, specification in XYY_TO_MUNSELL_SPECIFICATIONS:
            np.testing.assert_almost_equal(
                xyY_to_munsell_specification(xyY),
                specification,
                decimal=7)

        for xyY in NON_CONVERGING_XYY:
            self.assertRaises((RuntimeError, ValueError),
                              xyY_to_munsell_specification, xyY)

        for xyY, specification in XYY_TO_MUNSELL_GREYS_SPECIFICATIONS:
            np.testing.assert_almost_equal(
                xyY_to_munsell_specification(xyY),
                specification,
                decimal=7)


class TestxyY_to_munsell_colour(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.xyY_to_munsell_colour` definition
    unit tests methods.
    """

    def test_xyY_to_munsell_colour(self):
        """
        Tests :func:`colour.notation.munsell.xyY_to_munsell_colour` definition.
        """

        # TODO: This test is covered by the previous class,
        # do we need a dedicated one?

        pass


class TestParseMunsellColour(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.parse_munsell_colour` definition
    unit tests methods.
    """

    def test_parse_munsell_colour(self):
        """
        Tests :func:`colour.notation.munsell.is_grey_munsell_colour`
        definition.
        """

        self.assertEqual(parse_munsell_colour('N5.2'), 5.2)

        self.assertTupleEqual(parse_munsell_colour('0YR 2.0/4.0'),
                              (0.0, 2.0, 4.0, 6))

        self.assertTupleEqual(parse_munsell_colour('4.2YR 8.1/5.3'),
                              (4.2, 8.1, 5.3, 6))


class TestIsGreyMunsellColour(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.is_grey_munsell_colour` definition
    unit tests methods.
    """

    def test_is_grey_munsell_colour(self):
        """
        Tests :func:`colour.notation.munsell.is_grey_munsell_colour`
        definition.
        """

        self.assertTrue(is_grey_munsell_colour(5.2))

        self.assertFalse(is_grey_munsell_colour((0.0, 2.0, 4.0, 6)))

        self.assertFalse(is_grey_munsell_colour((4.2, 8.1, 5.3, 6)))


class TestNormalizeMunsellSpecification(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.normalize_munsell_specification`
    definition unit tests methods.
    """

    def test_normalize_munsell_specification(self):
        """
        Tests :func:`colour.notation.munsell.normalize_munsell_specification`
        definition.
        """

        self.assertTupleEqual(
            normalize_munsell_specification((0.0, 2.0, 4.0, 6)),
            (10.0, 2.0, 4.0, 7))

        self.assertTupleEqual(
            normalize_munsell_specification((0.0, 2.0, 4.0, 8)),
            (10.0, 2.0, 4.0, 9))

        self.assertTupleEqual(
            normalize_munsell_specification((0, 2.0, 4.0, 10)),
            (10.0, 2.0, 4.0, 1))

        self.assertEqual(
            normalize_munsell_specification((0, 2.0, 0, 10)),
            2)


class TestMunsellColourToMunsellSpecification(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.\
munsell_colour_to_munsell_specification` definition unit tests methods.
    """

    def test_munsell_colour_to_munsell_specification(self):
        """
        Tests :func:`colour.notation.munsell.\
munsell_colour_to_munsell_specification` definition.
        """

        self.assertTupleEqual(
            munsell_colour_to_munsell_specification('0.0YR 2.0/4.0'),
            (10.0, 2.0, 4.0, 7))

        self.assertTupleEqual(
            munsell_colour_to_munsell_specification('0.0RP 2.0/4.0'),
            (10.0, 2.0, 4.0, 9))

        self.assertTupleEqual(
            munsell_colour_to_munsell_specification('10.0B 2.0/4.0'),
            (10.0, 2.0, 4.0, 1))

        self.assertEqual(munsell_colour_to_munsell_specification('N5.2'),
                         5.2)
        self.assertEqual(
            munsell_colour_to_munsell_specification('0.0YR 2.0/0.0'),
            2)


class TestMunsellSpecificationToMunsellColour(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.\
munsell_specification_to_munsell_colour` definition unit tests methods.
    """

    def test_munsell_specification_to_munsell_colour(self):
        """
        Tests :func:`colour.notation.munsell.\
munsell_specification_to_munsell_colour` definition.
        """

        self.assertEqual(
            munsell_specification_to_munsell_colour((10, 2.0, 4.0, 7)),
            '10.0R 2.0/4.0')

        self.assertEqual(
            munsell_specification_to_munsell_colour((10.0, 2.0, 4.0, 9)),
            '10.0P 2.0/4.0')

        self.assertEqual(
            munsell_specification_to_munsell_colour((10.0, 2.0, 4.0, 1)),
            '10.0B 2.0/4.0')

        self.assertEqual(munsell_specification_to_munsell_colour(5.2),
                         'N5.2')


class Test_xyY_fromRenotation(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.xyY_from_renotation` definition
    unit tests methods.
    """

    def test_xyY_from_renotation(self):
        """
        Tests :func:`colour.notation.munsell.xyY_from_renotation`
        definition.
        """

        np.testing.assert_array_equal(
            xyY_from_renotation((2.5, 0.2, 2.0, 4)),
            np.array([0.713, 1.414, 0.237]))

        np.testing.assert_array_equal(
            xyY_from_renotation((5.0, 0.2, 2.0, 4)),
            np.array([0.449, 1.145, 0.237]))

        np.testing.assert_array_equal(
            xyY_from_renotation((7.5, 0.2, 2.0, 4)),
            np.array([0.262, 0.837, 0.237]))


class TestIsSpecificationInRenotation(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.is_specification_in_renotation`
    definition unit tests methods.
    """

    def test_is_specification_in_renotation(self):
        """
        Tests :func:`colour.notation.munsell.is_specification_in_renotation`
        definition.
        """

        self.assertTrue(is_specification_in_renotation((2.5, 0.2, 2.0, 4)))

        self.assertTrue(is_specification_in_renotation((5.0, 0.2, 2.0, 4)))

        self.assertFalse(is_specification_in_renotation((25.0, 0.2, 2.0, 4)))


class TestBoundingHuesFromRenotation(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.bounding_hues_from_renotation`
    definition unit tests methods.
    """

    def test_bounding_hues_from_renotation(self):
        """
        Tests :func:`colour.notation.munsell.bounding_hues_from_renotation`
        definition.
        """

        for i, specification in enumerate(MUNSELL_SPECIFICATIONS):
            hue, _value, _chroma, code = specification
            np.testing.assert_array_equal(
                bounding_hues_from_renotation(hue, code),
                MUNSELL_BOUNDING_HUES[i])


class TestHueToHueAngle(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.hue_to_hue_angle` definition units
    tests methods.
    """

    def test_hue_to_hue_angle(self):
        """
        Tests :func:`colour.notation.munsell.hue_to_hue_angle` definition.
        """

        for hue, code, angle in MUNSELL_HUE_TO_ANGLE:
            self.assertEqual(hue_to_hue_angle(hue, code), angle)


class TestHueAngleToHue(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.hue_angle_to_hue` definition units
    tests methods.
    """

    def test_hue_angle_to_hue(self):
        """
        Tests :func:`colour.notation.munsell.hue_angle_to_hue` definition.
        """

        for hue, code, angle in MUNSELL_HUE_TO_ANGLE:
            self.assertEqual(hue_angle_to_hue(angle), (hue, code))


class TestHueTo_ASTM_hue(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.hue_to_ASTM_hue` definition units
    tests methods.
    """

    def test_hue_to_ASTM_hue(self):
        """
        Tests :func:`colour.notation.munsell.hue_to_ASTM_hue` definition.
        """

        for hue, code, angle in MUNSELL_HUE_TO_ASTM_HUE:
            self.assertEqual(hue_to_ASTM_hue(hue, code), angle)


class TestInterpolationMethodFromRenotationOvoid(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.\
interpolation_method_from_renotation_ovoid` definition unit tests methods.
    """

    def test_interpolation_method_from_renotation_ovoid(self):
        """
        Tests :func:`colour.notation.munsell.\
interpolation_method_from_renotation_ovoid` definition.
        """

        for i, specification in enumerate(MUNSELL_EVEN_SPECIFICATIONS):
            self.assertEqual(
                interpolation_method_from_renotation_ovoid(specification),
                MUNSELL_INTERPOLATION_METHODS[i])


class Test_xy_fromRenotationOvoid(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.xy_from_renotation_ovoid` definition
    unit tests methods.
    """

    def test_xy_from_renotation_ovoid(self):
        """
        Tests :func:`colour.notation.munsell.xy_from_renotation_ovoid`
        definition.
        """

        for i, specification in enumerate(MUNSELL_EVEN_SPECIFICATIONS):
            if is_specification_in_renotation(specification):
                self.assertEqual(xy_from_renotation_ovoid(specification),
                                 MUNSELL_XY_FROM_RENOTATION_OVOID[i])


class TestLCHabToMunsellSpecification(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.LCHab_to_munsell_specification`
    definition unit tests methods.
    """

    def test_LCHab_to_munsell_specification(self):
        """
        Tests :func:`colour.notation.munsell.LCHab_to_munsell_specification`
        definition.
        """

        np.testing.assert_almost_equal(
            LCHab_to_munsell_specification(
                np.array([100.00000000, 21.57210357, 272.22819350])),
            (5.618942638888882, 10.0, 4.314420714000000, 10),
            decimal=7)

        np.testing.assert_almost_equal(
            LCHab_to_munsell_specification(
                np.array([100.00000000, 426.67945353, 72.39590835])),
            (0.109974541666666, 10.0, 85.335890706000001, 5),
            decimal=7)

        np.testing.assert_almost_equal(
            LCHab_to_munsell_specification(
                np.array([100.00000000, 74.05216981, 276.45318193])),
            (6.792550536111119, 10.0, 14.810433961999999, 10),
            decimal=7)


class TestMaximumChromaFromRenotation(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.maximum_chroma_from_renotation`
    definition unit tests methods.
    """

    def test_maximum_chroma_from_renotation(self):
        """
        Tests :func:`colour.notation.munsell.maximum_chroma_from_renotation`
        definition.
        """

        self.assertEqual(maximum_chroma_from_renotation(2.5, 5, 5),
                         14.0)

        self.assertEqual(maximum_chroma_from_renotation(8.675, 1.225, 10),
                         48.0)

        self.assertEqual(maximum_chroma_from_renotation(6.875, 3.425, 1),
                         16.0)


class TestMunsellSpecification_to_xy(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.munsell_specification_to_xy`
    definition unit tests methods.
    """

    def test_munsell_specification_to_xy(self):
        """
        Tests :func:`colour.notation.munsell.munsell_specification_to_xy`
        definition.
        """

        for specification, xy in MUNSELL_SPECIFICATIONS_TO_XY:
            np.testing.assert_almost_equal(
                munsell_specification_to_xy(specification),
                xy,
                decimal=7)

        for specification in MUNSELL_GREYS_SPECIFICATIONS:
            np.testing.assert_array_equal(
                munsell_specification_to_xy(specification),
                np.array([0.31006, 0.31616]))


if __name__ == '__main__':
    unittest.main()
