#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
Illuminants Relative Spectral Power Distributions
=================================================

Defines *CIE* illuminants relative spectral power distributions for the
*CIE 1931 2 Degree Standard Observer*.

The *CIE* illuminants data is in the form of a *dict* of
:class:`colour.colorimetry.spectrum.SpectralPowerDistribution` classes as
follows::

    {'name': SpectralPowerDistribution, ..., 'name': SpectralPowerDistribution}

The following *CIE* illuminants are available:

-   CIE Standard Illuminant A
-   CIE Illuminant B
-   CIE Illuminant C
-   CIE Illuminant D Series (D50, D55, D60, D65, D75)
-   CIE Illuminant E
-   Illuminants F Series (F1, F10, F11, F12, F2, F3, F4, F5, F6, F7, F8, F9,
    FL3.1, FL3.10, FL3.11, FL3.12, FL3.13, FL3.14, FL3.15, FL3.2, FL3.3, FL3.4,
    FL3.5, FL3.6, FL3.7, FL3.8, FL3.9)
-   High Pressure Discharge Lamps (HP1, HP2, HP3, HP4, HP5)

See Also
--------
`Illuminants Jupyter Notebook
<http://nbviewer.jupyter.org/github/colour-science/colour-notebooks/\
blob/master/notebooks/colorimetry/illuminants.ipynb>`_

Notes
-----
-   The relative spectral power distributions are provided at 5 nm interval.
-   *CIE Illuminant D Series* *D60* relative spectral power
    distribution is calculated using
    :func:`colour.colorimetry.illuminants.D_illuminant_relative_spd`
    definition.

References
----------
.. [1]  CIE. (n.d.). CIE Spectral Data. Retrieved from
        http://files.cie.co.at/204.xls
.. [2]  CIE. (n.d.). CIE 15:2004 Tables Data. Retrieved from
        https://law.resource.org/pub/us/cfr/ibr/003/cie.15.2004.tables.xls
"""

from __future__ import division, unicode_literals

from colour.colorimetry.spectrum import SpectralPowerDistribution
from colour.utilities import CaseInsensitiveMapping

__author__ = 'Colour Developers'
__copyright__ = 'Copyright (C) 2013-2017 - Colour Developers'
__license__ = 'New BSD License - http://opensource.org/licenses/BSD-3-Clause'
__maintainer__ = 'Colour Developers'
__email__ = 'colour-science@googlegroups.com'
__status__ = 'Production'

__all__ = ['ILLUMINANTS_RELATIVE_SPDS_DATA', 'ILLUMINANTS_RELATIVE_SPDS']

# *CIE Illuminant D Series* *D60* relative spectral power distribution
# has been calculated as follows:
# xy = colour.ILLUMINANTS['CIE 1931 2 Degree Standard Observer']['D60']
# D65_illuminant_relative_spd = colour.D_illuminant_relative_spd(xy)
# D65_illuminant_relative_spd.name = 'D60'
ILLUMINANTS_RELATIVE_SPDS_DATA = {
    'A': {
        300: 0.930483,
        305: 1.128210,
        310: 1.357690,
        315: 1.622190,
        320: 1.925080,
        325: 2.269800,
        330: 2.659810,
        335: 3.098610,
        340: 3.589680,
        345: 4.136480,
        350: 4.742380,
        355: 5.410700,
        360: 6.144620,
        365: 6.947200,
        370: 7.821350,
        375: 8.769800,
        380: 9.795100,
        385: 10.899600,
        390: 12.085300,
        395: 13.354300,
        400: 14.708000,
        405: 16.148000,
        410: 17.675300,
        415: 19.290700,
        420: 20.995000,
        425: 22.788300,
        430: 24.670900,
        435: 26.642500,
        440: 28.702700,
        445: 30.850800,
        450: 33.085900,
        455: 35.406800,
        460: 37.812100,
        465: 40.300200,
        470: 42.869300,
        475: 45.517400,
        480: 48.242300,
        485: 51.041800,
        490: 53.913200,
        495: 56.853900,
        500: 59.861100,
        505: 62.932000,
        510: 66.063500,
        515: 69.252500,
        520: 72.495900,
        525: 75.790300,
        530: 79.132600,
        535: 82.519300,
        540: 85.947000,
        545: 89.412400,
        550: 92.912000,
        555: 96.442300,
        560: 100.000000,
        565: 103.582000,
        570: 107.184000,
        575: 110.803000,
        580: 114.436000,
        585: 118.080000,
        590: 121.731000,
        595: 125.386000,
        600: 129.043000,
        605: 132.697000,
        610: 136.346000,
        615: 139.988000,
        620: 143.618000,
        625: 147.235000,
        630: 150.836000,
        635: 154.418000,
        640: 157.979000,
        645: 161.516000,
        650: 165.028000,
        655: 168.510000,
        660: 171.963000,
        665: 175.383000,
        670: 178.769000,
        675: 182.118000,
        680: 185.429000,
        685: 188.701000,
        690: 191.931000,
        695: 195.118000,
        700: 198.261000,
        705: 201.359000,
        710: 204.409000,
        715: 207.411000,
        720: 210.365000,
        725: 213.268000,
        730: 216.120000,
        735: 218.920000,
        740: 221.667000,
        745: 224.361000,
        750: 227.000000,
        755: 229.585000,
        760: 232.115000,
        765: 234.589000,
        770: 237.008000,
        775: 239.370000,
        780: 241.675000
    },
    'B': {
        320: 0.02,
        325: 0.26,
        330: 0.50,
        335: 1.45,
        340: 2.40,
        345: 4.00,
        350: 5.60,
        355: 7.60,
        360: 9.60,
        365: 12.40,
        370: 15.20,
        375: 18.80,
        380: 22.40,
        385: 26.85,
        390: 31.30,
        395: 36.18,
        400: 41.30,
        405: 46.62,
        410: 52.10,
        415: 57.70,
        420: 63.20,
        425: 68.37,
        430: 73.10,
        435: 77.31,
        440: 80.80,
        445: 83.44,
        450: 85.40,
        455: 86.88,
        460: 88.30,
        465: 90.08,
        470: 92.00,
        475: 93.75,
        480: 95.20,
        485: 96.23,
        490: 96.50,
        495: 95.71,
        500: 94.20,
        505: 92.37,
        510: 90.70,
        515: 89.95,
        520: 89.50,
        525: 90.43,
        530: 92.20,
        535: 94.46,
        540: 96.90,
        545: 99.16,
        550: 101.00,
        555: 102.20,
        560: 102.80,
        565: 102.92,
        570: 102.60,
        575: 101.90,
        580: 101.00,
        585: 100.07,
        590: 99.20,
        595: 98.44,
        600: 98.00,
        605: 98.08,
        610: 98.50,
        615: 99.06,
        620: 99.70,
        625: 100.36,
        630: 101.00,
        635: 101.56,
        640: 102.20,
        645: 103.05,
        650: 103.90,
        655: 104.59,
        660: 105.00,
        665: 105.08,
        670: 104.90,
        675: 104.55,
        680: 103.90,
        685: 102.84,
        690: 101.60,
        695: 100.38,
        700: 99.10,
        705: 97.70,
        710: 96.20,
        715: 94.60,
        720: 92.90,
        725: 91.10,
        730: 89.40,
        735: 88.00,
        740: 86.90,
        745: 85.90,
        750: 85.20,
        755: 84.80,
        760: 84.70,
        765: 84.90,
        770: 85.40,
        775: 86.10,
        780: 87.00
    },
    'C': {
        300: 0.00,
        305: 0.00,
        310: 0.00,
        315: 0.00,
        320: 0.01,
        325: 0.20,
        330: 0.40,
        335: 1.55,
        340: 2.70,
        345: 4.85,
        350: 7.00,
        355: 9.95,
        360: 12.90,
        365: 17.20,
        370: 21.40,
        375: 27.50,
        380: 33.00,
        385: 39.92,
        390: 47.40,
        395: 55.17,
        400: 63.30,
        405: 71.81,
        410: 80.60,
        415: 89.53,
        420: 98.10,
        425: 105.80,
        430: 112.40,
        435: 117.75,
        440: 121.50,
        445: 123.45,
        450: 124.00,
        455: 123.60,
        460: 123.10,
        465: 123.30,
        470: 123.80,
        475: 124.09,
        480: 123.90,
        485: 122.92,
        490: 120.70,
        495: 116.90,
        500: 112.10,
        505: 106.98,
        510: 102.30,
        515: 98.81,
        520: 96.90,
        525: 96.78,
        530: 98.00,
        535: 99.94,
        540: 102.10,
        545: 103.95,
        550: 105.20,
        555: 105.67,
        560: 105.30,
        565: 104.11,
        570: 102.30,
        575: 100.15,
        580: 97.80,
        585: 95.43,
        590: 93.20,
        595: 91.22,
        600: 89.70,
        605: 88.83,
        610: 88.40,
        615: 88.19,
        620: 88.10,
        625: 88.06,
        630: 88.00,
        635: 87.86,
        640: 87.80,
        645: 87.99,
        650: 88.20,
        655: 88.20,
        660: 87.90,
        665: 87.22,
        670: 86.30,
        675: 85.30,
        680: 84.00,
        685: 82.21,
        690: 80.20,
        695: 78.24,
        700: 76.30,
        705: 74.36,
        710: 72.40,
        715: 70.40,
        720: 68.30,
        725: 66.30,
        730: 64.40,
        735: 62.80,
        740: 61.50,
        745: 60.20,
        750: 59.20,
        755: 58.50,
        760: 58.10,
        765: 58.00,
        770: 58.20,
        775: 58.50,
        780: 59.10
    },
    'D50': {
        300: 0.019,
        305: 1.035,
        310: 2.051,
        315: 4.914,
        320: 7.778,
        325: 11.263,
        330: 14.748,
        335: 16.348,
        340: 17.948,
        345: 19.479,
        350: 21.010,
        355: 22.476,
        360: 23.942,
        365: 25.451,
        370: 26.961,
        375: 25.724,
        380: 24.488,
        385: 27.179,
        390: 29.871,
        395: 39.589,
        400: 49.308,
        405: 52.910,
        410: 56.513,
        415: 58.273,
        420: 60.034,
        425: 58.926,
        430: 57.818,
        435: 66.321,
        440: 74.825,
        445: 81.036,
        450: 87.247,
        455: 88.930,
        460: 90.612,
        465: 90.990,
        470: 91.368,
        475: 93.238,
        480: 95.109,
        485: 93.536,
        490: 91.963,
        495: 93.843,
        500: 95.724,
        505: 96.169,
        510: 96.613,
        515: 96.871,
        520: 97.129,
        525: 99.614,
        530: 102.099,
        535: 101.427,
        540: 100.755,
        545: 101.536,
        550: 102.317,
        555: 101.159,
        560: 100.000,
        565: 98.868,
        570: 97.735,
        575: 98.327,
        580: 98.918,
        585: 96.208,
        590: 93.499,
        595: 95.593,
        600: 97.688,
        605: 98.478,
        610: 99.269,
        615: 99.155,
        620: 99.042,
        625: 97.382,
        630: 95.722,
        635: 97.290,
        640: 98.857,
        645: 97.262,
        650: 95.667,
        655: 96.929,
        660: 98.190,
        665: 100.597,
        670: 103.003,
        675: 101.068,
        680: 99.133,
        685: 93.257,
        690: 87.381,
        695: 89.492,
        700: 91.604,
        705: 92.246,
        710: 92.889,
        715: 84.872,
        720: 76.854,
        725: 81.683,
        730: 86.511,
        735: 89.546,
        740: 92.580,
        745: 85.405,
        750: 78.230,
        755: 67.961,
        760: 57.692,
        765: 70.307,
        770: 82.923,
        775: 80.599,
        780: 78.274
    },
    'D55': {
        300: 0.024,
        305: 1.048,
        310: 2.072,
        315: 6.648,
        320: 11.224,
        325: 15.936,
        330: 20.647,
        335: 22.266,
        340: 23.885,
        345: 25.851,
        350: 27.817,
        355: 29.219,
        360: 30.621,
        365: 32.464,
        370: 34.308,
        375: 33.446,
        380: 32.584,
        385: 35.335,
        390: 38.087,
        395: 49.518,
        400: 60.949,
        405: 64.751,
        410: 68.554,
        415: 70.065,
        420: 71.577,
        425: 69.746,
        430: 67.914,
        435: 76.760,
        440: 85.605,
        445: 91.799,
        450: 97.993,
        455: 99.228,
        460: 100.463,
        465: 100.188,
        470: 99.913,
        475: 101.326,
        480: 102.739,
        485: 100.409,
        490: 98.078,
        495: 99.379,
        500: 100.680,
        505: 100.688,
        510: 100.695,
        515: 100.341,
        520: 99.987,
        525: 102.098,
        530: 104.210,
        535: 103.156,
        540: 102.102,
        545: 102.535,
        550: 102.968,
        555: 101.484,
        560: 100.000,
        565: 98.608,
        570: 97.216,
        575: 97.482,
        580: 97.749,
        585: 94.590,
        590: 91.432,
        595: 92.926,
        600: 94.419,
        605: 94.780,
        610: 95.140,
        615: 94.680,
        620: 94.220,
        625: 92.334,
        630: 90.448,
        635: 91.389,
        640: 92.330,
        645: 90.592,
        650: 88.854,
        655: 89.586,
        660: 90.317,
        665: 92.133,
        670: 93.950,
        675: 91.953,
        680: 89.956,
        685: 84.817,
        690: 79.677,
        695: 81.258,
        700: 82.840,
        705: 83.842,
        710: 84.844,
        715: 77.539,
        720: 70.235,
        725: 74.768,
        730: 79.301,
        735: 82.147,
        740: 84.993,
        745: 78.437,
        750: 71.880,
        755: 62.337,
        760: 52.793,
        765: 64.360,
        770: 75.927,
        775: 73.872,
        780: 71.818
    },
    'D60': {
        300: 0.029370758174923,
        310: 2.619241317964963,
        320: 15.716890613128260,
        330: 28.774580263919134,
        340: 31.864839936661980,
        350: 36.377426444674100,
        360: 38.683115463162864,
        370: 42.717548461834966,
        380: 41.454940579637523,
        390: 46.605319243279432,
        400: 72.278593838848636,
        410: 80.440599992794645,
        420: 82.915026938943186,
        430: 77.676263977317561,
        440: 95.681274303793984,
        450: 107.954820867505958,
        460: 109.559186805074063,
        470: 107.758140706827916,
        480: 109.671404235341797,
        490: 103.707873310050914,
        500: 105.232198575232047,
        510: 104.427666921854353,
        520: 102.522933578052459,
        530: 106.052670879047824,
        540: 103.315154034581980,
        550: 103.538598917326581,
        560: 100.000000000000000,
        570: 96.751421418866897,
        580: 96.712822501540316,
        590: 89.921479084426167,
        600: 91.999793295044071,
        610: 92.098709550672751,
        620: 90.646002697010346,
        630: 86.526482724860287,
        640: 87.579186235501524,
        650: 83.976140035832955,
        660: 84.724074228057717,
        670: 87.493490847729831,
        680: 83.483070156949736,
        690: 74.172451118766631,
        700: 76.620385310991381,
        710: 79.051849073755832,
        720: 65.471370717416463,
        730: 74.106079027252520,
        740: 79.527120427726302,
        750: 67.307162771623837,
        760: 49.273538206159095,
        770: 70.892412117890245,
        780: 67.163996226304974,
        790: 68.171370717416465,
        800: 62.989808616705801,
        810: 54.990892361077115,
        820: 60.825600670913168,
        830: 63.893495862261560
    },
    'D65': {
        300: 0.034100,
        305: 1.664300,
        310: 3.294500,
        315: 11.765200,
        320: 20.236000,
        325: 28.644700,
        330: 37.053500,
        335: 38.501100,
        340: 39.948800,
        345: 42.430200,
        350: 44.911700,
        355: 45.775000,
        360: 46.638300,
        365: 49.363700,
        370: 52.089100,
        375: 51.032300,
        380: 49.975500,
        385: 52.311800,
        390: 54.648200,
        395: 68.701500,
        400: 82.754900,
        405: 87.120400,
        410: 91.486000,
        415: 92.458900,
        420: 93.431800,
        425: 90.057000,
        430: 86.682300,
        435: 95.773600,
        440: 104.865000,
        445: 110.936000,
        450: 117.008000,
        455: 117.410000,
        460: 117.812000,
        465: 116.336000,
        470: 114.861000,
        475: 115.392000,
        480: 115.923000,
        485: 112.367000,
        490: 108.811000,
        495: 109.082000,
        500: 109.354000,
        505: 108.578000,
        510: 107.802000,
        515: 106.296000,
        520: 104.790000,
        525: 106.239000,
        530: 107.689000,
        535: 106.047000,
        540: 104.405000,
        545: 104.225000,
        550: 104.046000,
        555: 102.023000,
        560: 100.000000,
        565: 98.167100,
        570: 96.334200,
        575: 96.061100,
        580: 95.788000,
        585: 92.236800,
        590: 88.685600,
        595: 89.345900,
        600: 90.006200,
        605: 89.802600,
        610: 89.599100,
        615: 88.648900,
        620: 87.698700,
        625: 85.493600,
        630: 83.288600,
        635: 83.493900,
        640: 83.699200,
        645: 81.863000,
        650: 80.026800,
        655: 80.120700,
        660: 80.214600,
        665: 81.246200,
        670: 82.277800,
        675: 80.281000,
        680: 78.284200,
        685: 74.002700,
        690: 69.721300,
        695: 70.665200,
        700: 71.609100,
        705: 72.979000,
        710: 74.349000,
        715: 67.976500,
        720: 61.604000,
        725: 65.744800,
        730: 69.885600,
        735: 72.486300,
        740: 75.087000,
        745: 69.339800,
        750: 63.592700,
        755: 55.005400,
        760: 46.418200,
        765: 56.611800,
        770: 66.805400,
        775: 65.094100,
        780: 63.382800
    },
    'D75': {
        300: 0.043,
        305: 2.588,
        310: 5.133,
        315: 17.470,
        320: 29.808,
        325: 42.369,
        330: 54.930,
        335: 56.095,
        340: 57.259,
        345: 60.000,
        350: 62.740,
        355: 62.861,
        360: 62.982,
        365: 66.647,
        370: 70.312,
        375: 68.507,
        380: 66.703,
        385: 68.333,
        390: 69.963,
        395: 85.946,
        400: 101.929,
        405: 106.911,
        410: 111.894,
        415: 112.346,
        420: 112.798,
        425: 107.945,
        430: 103.092,
        435: 112.145,
        440: 121.198,
        445: 127.104,
        450: 133.010,
        455: 132.682,
        460: 132.355,
        465: 129.838,
        470: 127.322,
        475: 127.061,
        480: 126.800,
        485: 122.291,
        490: 117.783,
        495: 117.186,
        500: 116.589,
        505: 115.146,
        510: 113.702,
        515: 111.181,
        520: 108.659,
        525: 109.552,
        530: 110.445,
        535: 108.367,
        540: 106.289,
        545: 105.596,
        550: 104.904,
        555: 102.452,
        560: 100.000,
        565: 97.808,
        570: 95.616,
        575: 94.914,
        580: 94.213,
        585: 90.605,
        590: 86.997,
        595: 87.112,
        600: 87.227,
        605: 86.684,
        610: 86.140,
        615: 84.861,
        620: 83.581,
        625: 81.164,
        630: 78.747,
        635: 78.587,
        640: 78.428,
        645: 76.614,
        650: 74.801,
        655: 74.562,
        660: 74.324,
        665: 74.873,
        670: 75.422,
        675: 73.499,
        680: 71.576,
        685: 67.714,
        690: 63.852,
        695: 64.464,
        700: 65.076,
        705: 66.573,
        710: 68.070,
        715: 62.256,
        720: 56.443,
        725: 60.343,
        730: 64.242,
        735: 66.697,
        740: 69.151,
        745: 63.890,
        750: 58.629,
        755: 50.623,
        760: 42.617,
        765: 51.985,
        770: 61.352,
        775: 59.838,
        780: 58.324
    },
    'E': {
        300: 100.000,
        305: 100.000,
        310: 100.000,
        315: 100.000,
        320: 100.000,
        325: 100.000,
        330: 100.000,
        335: 100.000,
        340: 100.000,
        345: 100.000,
        350: 100.000,
        355: 100.000,
        360: 100.000,
        365: 100.000,
        370: 100.000,
        375: 100.000,
        380: 100.000,
        385: 100.000,
        390: 100.000,
        395: 100.000,
        400: 100.000,
        405: 100.000,
        410: 100.000,
        415: 100.000,
        420: 100.000,
        425: 100.000,
        430: 100.000,
        435: 100.000,
        440: 100.000,
        445: 100.000,
        450: 100.000,
        455: 100.000,
        460: 100.000,
        465: 100.000,
        470: 100.000,
        475: 100.000,
        480: 100.000,
        485: 100.000,
        490: 100.000,
        495: 100.000,
        500: 100.000,
        505: 100.000,
        510: 100.000,
        515: 100.000,
        520: 100.000,
        525: 100.000,
        530: 100.000,
        535: 100.000,
        540: 100.000,
        545: 100.000,
        550: 100.000,
        555: 100.000,
        560: 100.000,
        565: 100.000,
        570: 100.000,
        575: 100.000,
        580: 100.000,
        585: 100.000,
        590: 100.000,
        595: 100.000,
        600: 100.000,
        605: 100.000,
        610: 100.000,
        615: 100.000,
        620: 100.000,
        625: 100.000,
        630: 100.000,
        635: 100.000,
        640: 100.000,
        645: 100.000,
        650: 100.000,
        655: 100.000,
        660: 100.000,
        665: 100.000,
        670: 100.000,
        675: 100.000,
        680: 100.000,
        685: 100.000,
        690: 100.000,
        695: 100.000,
        700: 100.000,
        705: 100.000,
        710: 100.000,
        715: 100.000,
        720: 100.000,
        725: 100.000,
        730: 100.000,
        735: 100.000,
        740: 100.000,
        745: 100.000,
        750: 100.000,
        755: 100.000,
        760: 100.000,
        765: 100.000,
        770: 100.000,
        775: 100.000,
        780: 100.000
    },
    'F1': {
        380: 1.87,
        385: 2.36,
        390: 2.94,
        395: 3.47,
        400: 5.17,
        405: 19.49,
        410: 6.13,
        415: 6.24,
        420: 7.01,
        425: 7.79,
        430: 8.56,
        435: 43.67,
        440: 16.94,
        445: 10.72,
        450: 11.35,
        455: 11.89,
        460: 12.37,
        465: 12.75,
        470: 13.00,
        475: 13.15,
        480: 13.23,
        485: 13.17,
        490: 13.13,
        495: 12.85,
        500: 12.52,
        505: 12.20,
        510: 11.83,
        515: 11.50,
        520: 11.22,
        525: 11.05,
        530: 11.03,
        535: 11.18,
        540: 11.53,
        545: 27.74,
        550: 17.05,
        555: 13.55,
        560: 14.33,
        565: 15.01,
        570: 15.52,
        575: 18.29,
        580: 19.55,
        585: 15.48,
        590: 14.91,
        595: 14.15,
        600: 13.22,
        605: 12.19,
        610: 11.12,
        615: 10.03,
        620: 8.95,
        625: 7.96,
        630: 7.02,
        635: 6.20,
        640: 5.42,
        645: 4.73,
        650: 4.15,
        655: 3.64,
        660: 3.20,
        665: 2.81,
        670: 2.47,
        675: 2.18,
        680: 1.93,
        685: 1.72,
        690: 1.67,
        695: 1.43,
        700: 1.29,
        705: 1.19,
        710: 1.08,
        715: 0.96,
        720: 0.88,
        725: 0.81,
        730: 0.77,
        735: 0.75,
        740: 0.73,
        745: 0.68,
        750: 0.69,
        755: 0.64,
        760: 0.68,
        765: 0.69,
        770: 0.61,
        775: 0.52,
        780: 0.43
    },
    'F2': {
        380: 1.18,
        385: 1.48,
        390: 1.84,
        395: 2.15,
        400: 3.44,
        405: 15.69,
        410: 3.85,
        415: 3.74,
        420: 4.19,
        425: 4.62,
        430: 5.06,
        435: 34.98,
        440: 11.81,
        445: 6.27,
        450: 6.63,
        455: 6.93,
        460: 7.19,
        465: 7.40,
        470: 7.54,
        475: 7.62,
        480: 7.65,
        485: 7.62,
        490: 7.62,
        495: 7.45,
        500: 7.28,
        505: 7.15,
        510: 7.05,
        515: 7.04,
        520: 7.16,
        525: 7.47,
        530: 8.04,
        535: 8.88,
        540: 10.01,
        545: 24.88,
        550: 16.64,
        555: 14.59,
        560: 16.16,
        565: 17.56,
        570: 18.62,
        575: 21.47,
        580: 22.79,
        585: 19.29,
        590: 18.66,
        595: 17.73,
        600: 16.54,
        605: 15.21,
        610: 13.80,
        615: 12.36,
        620: 10.95,
        625: 9.65,
        630: 8.40,
        635: 7.32,
        640: 6.31,
        645: 5.43,
        650: 4.68,
        655: 4.02,
        660: 3.45,
        665: 2.96,
        670: 2.55,
        675: 2.19,
        680: 1.89,
        685: 1.64,
        690: 1.53,
        695: 1.27,
        700: 1.10,
        705: 0.99,
        710: 0.88,
        715: 0.76,
        720: 0.68,
        725: 0.61,
        730: 0.56,
        735: 0.54,
        740: 0.51,
        745: 0.47,
        750: 0.47,
        755: 0.43,
        760: 0.46,
        765: 0.47,
        770: 0.40,
        775: 0.33,
        780: 0.27
    },
    'F3': {
        380: 0.82,
        385: 1.02,
        390: 1.26,
        395: 1.44,
        400: 2.57,
        405: 14.36,
        410: 2.70,
        415: 2.45,
        420: 2.73,
        425: 3.00,
        430: 3.28,
        435: 31.85,
        440: 9.47,
        445: 4.02,
        450: 4.25,
        455: 4.44,
        460: 4.59,
        465: 4.72,
        470: 4.80,
        475: 4.86,
        480: 4.87,
        485: 4.85,
        490: 4.88,
        495: 4.77,
        500: 4.67,
        505: 4.62,
        510: 4.62,
        515: 4.73,
        520: 4.99,
        525: 5.48,
        530: 6.25,
        535: 7.34,
        540: 8.78,
        545: 23.82,
        550: 16.14,
        555: 14.59,
        560: 16.63,
        565: 18.49,
        570: 19.95,
        575: 23.11,
        580: 24.69,
        585: 21.41,
        590: 20.85,
        595: 19.93,
        600: 18.67,
        605: 17.22,
        610: 15.65,
        615: 14.04,
        620: 12.45,
        625: 10.95,
        630: 9.51,
        635: 8.27,
        640: 7.11,
        645: 6.09,
        650: 5.22,
        655: 4.45,
        660: 3.80,
        665: 3.23,
        670: 2.75,
        675: 2.33,
        680: 1.99,
        685: 1.70,
        690: 1.55,
        695: 1.27,
        700: 1.09,
        705: 0.96,
        710: 0.83,
        715: 0.71,
        720: 0.62,
        725: 0.54,
        730: 0.49,
        735: 0.46,
        740: 0.43,
        745: 0.39,
        750: 0.39,
        755: 0.35,
        760: 0.38,
        765: 0.39,
        770: 0.33,
        775: 0.28,
        780: 0.21
    },
    'F4': {
        380: 0.57,
        385: 0.70,
        390: 0.87,
        395: 0.98,
        400: 2.01,
        405: 13.75,
        410: 1.95,
        415: 1.59,
        420: 1.76,
        425: 1.93,
        430: 2.10,
        435: 30.28,
        440: 8.03,
        445: 2.55,
        450: 2.70,
        455: 2.82,
        460: 2.91,
        465: 2.99,
        470: 3.04,
        475: 3.08,
        480: 3.09,
        485: 3.09,
        490: 3.14,
        495: 3.06,
        500: 3.00,
        505: 2.98,
        510: 3.01,
        515: 3.14,
        520: 3.41,
        525: 3.90,
        530: 4.69,
        535: 5.81,
        540: 7.32,
        545: 22.59,
        550: 15.11,
        555: 13.88,
        560: 16.33,
        565: 18.68,
        570: 20.64,
        575: 24.28,
        580: 26.26,
        585: 23.28,
        590: 22.94,
        595: 22.14,
        600: 20.91,
        605: 19.43,
        610: 17.74,
        615: 16.00,
        620: 14.42,
        625: 12.56,
        630: 10.93,
        635: 9.52,
        640: 8.18,
        645: 7.01,
        650: 6.00,
        655: 5.11,
        660: 4.36,
        665: 3.69,
        670: 3.13,
        675: 2.64,
        680: 2.24,
        685: 1.91,
        690: 1.70,
        695: 1.39,
        700: 1.18,
        705: 1.03,
        710: 0.88,
        715: 0.74,
        720: 0.64,
        725: 0.54,
        730: 0.49,
        735: 0.46,
        740: 0.42,
        745: 0.37,
        750: 0.37,
        755: 0.33,
        760: 0.35,
        765: 0.36,
        770: 0.31,
        775: 0.26,
        780: 0.19
    },
    'F5': {
        380: 1.87,
        385: 2.35,
        390: 2.92,
        395: 3.45,
        400: 5.10,
        405: 18.91,
        410: 6.00,
        415: 6.11,
        420: 6.85,
        425: 7.58,
        430: 8.31,
        435: 40.76,
        440: 16.06,
        445: 10.32,
        450: 10.91,
        455: 11.40,
        460: 11.83,
        465: 12.17,
        470: 12.40,
        475: 12.54,
        480: 12.58,
        485: 12.52,
        490: 12.47,
        495: 12.20,
        500: 11.89,
        505: 11.61,
        510: 11.33,
        515: 11.10,
        520: 10.96,
        525: 10.97,
        530: 11.16,
        535: 11.54,
        540: 12.12,
        545: 27.78,
        550: 17.73,
        555: 14.47,
        560: 15.20,
        565: 15.77,
        570: 16.10,
        575: 18.54,
        580: 19.50,
        585: 15.39,
        590: 14.64,
        595: 13.72,
        600: 12.69,
        605: 11.57,
        610: 10.45,
        615: 9.35,
        620: 8.29,
        625: 7.32,
        630: 6.41,
        635: 5.63,
        640: 4.90,
        645: 4.26,
        650: 3.72,
        655: 3.25,
        660: 2.83,
        665: 2.49,
        670: 2.19,
        675: 1.93,
        680: 1.71,
        685: 1.52,
        690: 1.48,
        695: 1.26,
        700: 1.13,
        705: 1.05,
        710: 0.96,
        715: 0.85,
        720: 0.78,
        725: 0.72,
        730: 0.68,
        735: 0.67,
        740: 0.65,
        745: 0.61,
        750: 0.62,
        755: 0.59,
        760: 0.62,
        765: 0.64,
        770: 0.55,
        775: 0.47,
        780: 0.40
    },
    'F6': {
        380: 1.05,
        385: 1.31,
        390: 1.63,
        395: 1.90,
        400: 3.11,
        405: 14.80,
        410: 3.43,
        415: 3.30,
        420: 3.68,
        425: 4.07,
        430: 4.45,
        435: 32.61,
        440: 10.74,
        445: 5.48,
        450: 5.78,
        455: 6.03,
        460: 6.25,
        465: 6.41,
        470: 6.52,
        475: 6.58,
        480: 6.59,
        485: 6.56,
        490: 6.56,
        495: 6.42,
        500: 6.28,
        505: 6.20,
        510: 6.19,
        515: 6.30,
        520: 6.60,
        525: 7.12,
        530: 7.94,
        535: 9.07,
        540: 10.49,
        545: 25.22,
        550: 17.46,
        555: 15.63,
        560: 17.22,
        565: 18.53,
        570: 19.43,
        575: 21.97,
        580: 23.01,
        585: 19.41,
        590: 18.56,
        595: 17.42,
        600: 16.09,
        605: 14.64,
        610: 13.15,
        615: 11.68,
        620: 10.25,
        625: 8.95,
        630: 7.74,
        635: 6.69,
        640: 5.71,
        645: 4.87,
        650: 4.16,
        655: 3.55,
        660: 3.02,
        665: 2.57,
        670: 2.20,
        675: 1.87,
        680: 1.60,
        685: 1.37,
        690: 1.29,
        695: 1.05,
        700: 0.91,
        705: 0.81,
        710: 0.71,
        715: 0.61,
        720: 0.54,
        725: 0.48,
        730: 0.44,
        735: 0.43,
        740: 0.40,
        745: 0.37,
        750: 0.38,
        755: 0.35,
        760: 0.39,
        765: 0.41,
        770: 0.33,
        775: 0.26,
        780: 0.21
    },
    'F7': {
        380: 2.56,
        385: 3.18,
        390: 3.84,
        395: 4.53,
        400: 6.15,
        405: 19.37,
        410: 7.37,
        415: 7.05,
        420: 7.71,
        425: 8.41,
        430: 9.15,
        435: 44.14,
        440: 17.52,
        445: 11.35,
        450: 12.00,
        455: 12.58,
        460: 13.08,
        465: 13.45,
        470: 13.71,
        475: 13.88,
        480: 13.95,
        485: 13.93,
        490: 13.82,
        495: 13.64,
        500: 13.43,
        505: 13.25,
        510: 13.08,
        515: 12.93,
        520: 12.78,
        525: 12.60,
        530: 12.44,
        535: 12.33,
        540: 12.26,
        545: 29.52,
        550: 17.05,
        555: 12.44,
        560: 12.58,
        565: 12.72,
        570: 12.83,
        575: 15.46,
        580: 16.75,
        585: 12.83,
        590: 12.67,
        595: 12.45,
        600: 12.19,
        605: 11.89,
        610: 11.60,
        615: 11.35,
        620: 11.12,
        625: 10.95,
        630: 10.76,
        635: 10.42,
        640: 10.11,
        645: 10.04,
        650: 10.02,
        655: 10.11,
        660: 9.87,
        665: 8.65,
        670: 7.27,
        675: 6.44,
        680: 5.83,
        685: 5.41,
        690: 5.04,
        695: 4.57,
        700: 4.12,
        705: 3.77,
        710: 3.46,
        715: 3.08,
        720: 2.73,
        725: 2.47,
        730: 2.25,
        735: 2.06,
        740: 1.90,
        745: 1.75,
        750: 1.62,
        755: 1.54,
        760: 1.45,
        765: 1.32,
        770: 1.17,
        775: 0.99,
        780: 0.81
    },
    'F8': {
        380: 1.21,
        385: 1.50,
        390: 1.81,
        395: 2.13,
        400: 3.17,
        405: 13.08,
        410: 3.83,
        415: 3.45,
        420: 3.86,
        425: 4.42,
        430: 5.09,
        435: 34.10,
        440: 12.42,
        445: 7.68,
        450: 8.60,
        455: 9.46,
        460: 10.24,
        465: 10.84,
        470: 11.33,
        475: 11.71,
        480: 11.98,
        485: 12.17,
        490: 12.28,
        495: 12.32,
        500: 12.35,
        505: 12.44,
        510: 12.55,
        515: 12.68,
        520: 12.77,
        525: 12.72,
        530: 12.60,
        535: 12.43,
        540: 12.22,
        545: 28.96,
        550: 16.51,
        555: 11.79,
        560: 11.76,
        565: 11.77,
        570: 11.84,
        575: 14.61,
        580: 16.11,
        585: 12.34,
        590: 12.53,
        595: 12.72,
        600: 12.92,
        605: 13.12,
        610: 13.34,
        615: 13.61,
        620: 13.87,
        625: 14.07,
        630: 14.20,
        635: 14.16,
        640: 14.13,
        645: 14.34,
        650: 14.50,
        655: 14.46,
        660: 14.00,
        665: 12.58,
        670: 10.99,
        675: 9.98,
        680: 9.22,
        685: 8.62,
        690: 8.07,
        695: 7.39,
        700: 6.71,
        705: 6.16,
        710: 5.63,
        715: 5.03,
        720: 4.46,
        725: 4.02,
        730: 3.66,
        735: 3.36,
        740: 3.09,
        745: 2.85,
        750: 2.65,
        755: 2.51,
        760: 2.37,
        765: 2.15,
        770: 1.89,
        775: 1.61,
        780: 1.32
    },
    'F9': {
        380: 0.90,
        385: 1.12,
        390: 1.36,
        395: 1.60,
        400: 2.59,
        405: 12.80,
        410: 3.05,
        415: 2.56,
        420: 2.86,
        425: 3.30,
        430: 3.82,
        435: 32.62,
        440: 10.77,
        445: 5.84,
        450: 6.57,
        455: 7.25,
        460: 7.86,
        465: 8.35,
        470: 8.75,
        475: 9.06,
        480: 9.31,
        485: 9.48,
        490: 9.61,
        495: 9.68,
        500: 9.74,
        505: 9.88,
        510: 10.04,
        515: 10.26,
        520: 10.48,
        525: 10.63,
        530: 10.78,
        535: 10.96,
        540: 11.18,
        545: 27.71,
        550: 16.29,
        555: 12.28,
        560: 12.74,
        565: 13.21,
        570: 13.65,
        575: 16.57,
        580: 18.14,
        585: 14.55,
        590: 14.65,
        595: 14.66,
        600: 14.61,
        605: 14.50,
        610: 14.39,
        615: 14.40,
        620: 14.47,
        625: 14.62,
        630: 14.72,
        635: 14.55,
        640: 14.40,
        645: 14.58,
        650: 14.88,
        655: 15.51,
        660: 15.47,
        665: 13.20,
        670: 10.57,
        675: 9.18,
        680: 8.25,
        685: 7.57,
        690: 7.03,
        695: 6.35,
        700: 5.72,
        705: 5.25,
        710: 4.80,
        715: 4.29,
        720: 3.80,
        725: 3.43,
        730: 3.12,
        735: 2.86,
        740: 2.64,
        745: 2.43,
        750: 2.26,
        755: 2.14,
        760: 2.02,
        765: 1.83,
        770: 1.61,
        775: 1.38,
        780: 1.12
    },
    'F10': {
        380: 1.11,
        385: 0.80,
        390: 0.62,
        395: 0.57,
        400: 1.48,
        405: 12.16,
        410: 2.12,
        415: 2.70,
        420: 3.74,
        425: 5.14,
        430: 6.75,
        435: 34.39,
        440: 14.86,
        445: 10.40,
        450: 10.76,
        455: 10.67,
        460: 10.11,
        465: 9.27,
        470: 8.29,
        475: 7.29,
        480: 7.91,
        485: 16.64,
        490: 16.73,
        495: 10.44,
        500: 5.94,
        505: 3.34,
        510: 2.35,
        515: 1.88,
        520: 1.59,
        525: 1.47,
        530: 1.80,
        535: 5.71,
        540: 40.98,
        545: 73.69,
        550: 33.61,
        555: 8.24,
        560: 3.38,
        565: 2.47,
        570: 2.14,
        575: 4.86,
        580: 11.45,
        585: 14.79,
        590: 12.16,
        595: 8.97,
        600: 6.52,
        605: 8.31,
        610: 44.12,
        615: 34.55,
        620: 12.09,
        625: 12.15,
        630: 10.52,
        635: 4.43,
        640: 1.95,
        645: 2.19,
        650: 3.19,
        655: 2.77,
        660: 2.29,
        665: 2.00,
        670: 1.52,
        675: 1.35,
        680: 1.47,
        685: 1.79,
        690: 1.74,
        695: 1.02,
        700: 1.14,
        705: 3.32,
        710: 4.49,
        715: 2.05,
        720: 0.49,
        725: 0.24,
        730: 0.21,
        735: 0.21,
        740: 0.24,
        745: 0.24,
        750: 0.21,
        755: 0.17,
        760: 0.21,
        765: 0.22,
        770: 0.17,
        775: 0.12,
        780: 0.09
    },
    'F11': {
        380: 0.91,
        385: 0.63,
        390: 0.46,
        395: 0.37,
        400: 1.29,
        405: 12.68,
        410: 1.59,
        415: 1.79,
        420: 2.46,
        425: 3.33,
        430: 4.49,
        435: 33.94,
        440: 12.13,
        445: 6.95,
        450: 7.19,
        455: 7.12,
        460: 6.72,
        465: 6.13,
        470: 5.46,
        475: 4.79,
        480: 5.66,
        485: 14.29,
        490: 14.96,
        495: 8.97,
        500: 4.72,
        505: 2.33,
        510: 1.47,
        515: 1.10,
        520: 0.89,
        525: 0.83,
        530: 1.18,
        535: 4.90,
        540: 39.59,
        545: 72.84,
        550: 32.61,
        555: 7.52,
        560: 2.83,
        565: 1.96,
        570: 1.67,
        575: 4.43,
        580: 11.28,
        585: 14.76,
        590: 12.73,
        595: 9.74,
        600: 7.33,
        605: 9.72,
        610: 55.27,
        615: 42.58,
        620: 13.18,
        625: 13.16,
        630: 12.26,
        635: 5.11,
        640: 2.07,
        645: 2.34,
        650: 3.58,
        655: 3.01,
        660: 2.48,
        665: 2.14,
        670: 1.54,
        675: 1.33,
        680: 1.46,
        685: 1.94,
        690: 2.00,
        695: 1.20,
        700: 1.35,
        705: 4.10,
        710: 5.58,
        715: 2.51,
        720: 0.57,
        725: 0.27,
        730: 0.23,
        735: 0.21,
        740: 0.24,
        745: 0.24,
        750: 0.20,
        755: 0.24,
        760: 0.32,
        765: 0.26,
        770: 0.16,
        775: 0.12,
        780: 0.09
    },
    'F12': {
        380: 0.96,
        385: 0.64,
        390: 0.40,
        395: 0.33,
        400: 1.19,
        405: 12.48,
        410: 1.12,
        415: 0.94,
        420: 1.08,
        425: 1.37,
        430: 1.78,
        435: 29.05,
        440: 7.90,
        445: 2.65,
        450: 2.71,
        455: 2.65,
        460: 2.49,
        465: 2.33,
        470: 2.10,
        475: 1.91,
        480: 3.01,
        485: 10.83,
        490: 11.88,
        495: 6.88,
        500: 3.43,
        505: 1.49,
        510: 0.92,
        515: 0.71,
        520: 0.60,
        525: 0.63,
        530: 1.10,
        535: 4.56,
        540: 34.40,
        545: 65.40,
        550: 29.48,
        555: 7.16,
        560: 3.08,
        565: 2.47,
        570: 2.27,
        575: 5.09,
        580: 11.96,
        585: 15.32,
        590: 14.27,
        595: 11.86,
        600: 9.28,
        605: 12.31,
        610: 68.53,
        615: 53.02,
        620: 14.67,
        625: 14.38,
        630: 14.71,
        635: 6.46,
        640: 2.57,
        645: 2.75,
        650: 4.18,
        655: 3.44,
        660: 2.81,
        665: 2.42,
        670: 1.64,
        675: 1.36,
        680: 1.49,
        685: 2.14,
        690: 2.34,
        695: 1.42,
        700: 1.61,
        705: 5.04,
        710: 6.98,
        715: 3.19,
        720: 0.71,
        725: 0.30,
        730: 0.26,
        735: 0.23,
        740: 0.28,
        745: 0.28,
        750: 0.21,
        755: 0.17,
        760: 0.21,
        765: 0.19,
        770: 0.15,
        775: 0.10,
        780: 0.05
    },
    'FL3.1': {
        380: 2.39,
        385: 2.93,
        390: 3.82,
        395: 4.23,
        400: 4.97,
        405: 86.30,
        410: 11.65,
        415: 7.09,
        420: 7.84,
        425: 8.59,
        430: 9.44,
        435: 196.54,
        440: 10.94,
        445: 11.38,
        450: 11.89,
        455: 12.37,
        460: 12.81,
        465: 13.15,
        470: 13.39,
        475: 13.56,
        480: 13.59,
        485: 13.56,
        490: 14.07,
        495: 13.39,
        500: 13.29,
        505: 13.25,
        510: 13.53,
        515: 14.24,
        520: 15.74,
        525: 18.26,
        530: 22.28,
        535: 27.97,
        540: 35.70,
        545: 148.98,
        550: 56.55,
        555: 68.68,
        560: 79.99,
        565: 91.47,
        570: 101.32,
        575: 123.16,
        580: 129.53,
        585: 115.05,
        590: 113.48,
        595: 110.08,
        600: 104.28,
        605: 97.98,
        610: 89.60,
        615: 80.74,
        620: 71.92,
        625: 63.50,
        630: 55.46,
        635: 47.97,
        640: 41.39,
        645: 35.50,
        650: 30.32,
        655: 25.79,
        660: 21.84,
        665: 18.53,
        670: 15.67,
        675: 13.22,
        680: 11.14,
        685: 9.40,
        690: 8.65,
        695: 6.75,
        700: 5.69,
        705: 4.87,
        710: 4.29,
        715: 3.54,
        720: 3.03,
        725: 2.62,
        730: 2.28,
        735: 1.94,
        740: 1.70,
        745: 1.50,
        750: 1.36,
        755: 1.16,
        760: 4.91,
        765: 0.95,
        770: 1.50,
        775: 0.89,
        780: 0.68
    },
    'FL3.2': {
        380: 5.80,
        385: 6.99,
        390: 8.70,
        395: 9.89,
        400: 11.59,
        405: 94.53,
        410: 20.80,
        415: 16.52,
        420: 18.30,
        425: 20.33,
        430: 22.00,
        435: 231.90,
        440: 25.81,
        445: 27.63,
        450: 29.10,
        455: 30.61,
        460: 31.92,
        465: 33.11,
        470: 33.83,
        475: 34.70,
        480: 35.02,
        485: 35.22,
        490: 35.81,
        495: 35.14,
        500: 35.14,
        505: 34.90,
        510: 34.70,
        515: 35.02,
        520: 36.13,
        525: 37.92,
        530: 40.62,
        535: 44.70,
        540: 49.63,
        545: 154.16,
        550: 62.21,
        555: 68.92,
        560: 75.83,
        565: 81.95,
        570: 86.95,
        575: 103.54,
        580: 109.94,
        585: 91.95,
        590: 89.85,
        595: 87.15,
        600: 83.26,
        605: 78.93,
        610: 73.93,
        615: 68.84,
        620: 63.44,
        625: 58.84,
        630: 53.84,
        635: 49.43,
        640: 45.54,
        645: 41.53,
        650: 38.31,
        655: 34.62,
        660: 31.80,
        665: 29.02,
        670: 26.72,
        675: 24.22,
        680: 22.19,
        685: 20.41,
        690: 19.10,
        695: 16.79,
        700: 15.13,
        705: 13.82,
        710: 12.63,
        715: 11.39,
        720: 10.32,
        725: 9.21,
        730: 8.89,
        735: 7.50,
        740: 6.71,
        745: 6.11,
        750: 5.40,
        755: 4.80,
        760: 8.70,
        765: 4.01,
        770: 4.09,
        775: 3.30,
        780: 2.82
    },
    'FL3.3': {
        380: 8.94,
        385: 11.21,
        390: 14.08,
        395: 16.48,
        400: 19.63,
        405: 116.33,
        410: 32.07,
        415: 29.72,
        420: 33.39,
        425: 36.94,
        430: 40.33,
        435: 262.66,
        440: 46.87,
        445: 49.79,
        450: 52.46,
        455: 54.81,
        460: 56.81,
        465: 58.44,
        470: 59.52,
        475: 60.12,
        480: 60.24,
        485: 59.88,
        490: 59.88,
        495: 58.60,
        500: 57.85,
        505: 56.29,
        510: 54.81,
        515: 53.42,
        520: 52.70,
        525: 52.50,
        530: 53.30,
        535: 54.89,
        540: 57.61,
        545: 182.75,
        550: 65.27,
        555: 69.41,
        560: 73.28,
        565: 76.56,
        570: 78.67,
        575: 95.74,
        580: 97.22,
        585: 76.79,
        590: 73.36,
        595: 69.33,
        600: 64.23,
        605: 58.92,
        610: 53.38,
        615: 47.91,
        620: 42.61,
        625: 37.74,
        630: 33.11,
        635: 29.04,
        640: 25.29,
        645: 22.10,
        650: 19.31,
        655: 16.84,
        660: 14.68,
        665: 12.89,
        670: 11.37,
        675: 9.97,
        680: 8.82,
        685: 7.86,
        690: 7.78,
        695: 6.30,
        700: 5.67,
        705: 5.15,
        710: 4.91,
        715: 4.31,
        720: 3.99,
        725: 3.67,
        730: 3.43,
        735: 3.19,
        740: 2.95,
        745: 2.75,
        750: 2.63,
        755: 2.43,
        760: 7.14,
        765: 2.19,
        770: 2.71,
        775: 2.00,
        780: 1.80
    },
    'FL3.4': {
        380: 3.46,
        385: 3.86,
        390: 4.41,
        395: 4.51,
        400: 4.86,
        405: 71.22,
        410: 8.72,
        415: 5.36,
        420: 5.61,
        425: 5.91,
        430: 6.42,
        435: 192.77,
        440: 7.77,
        445: 8.37,
        450: 9.22,
        455: 10.18,
        460: 11.18,
        465: 12.28,
        470: 13.38,
        475: 14.54,
        480: 15.74,
        485: 17.09,
        490: 19.60,
        495: 21.05,
        500: 23.96,
        505: 27.77,
        510: 32.68,
        515: 38.29,
        520: 43.76,
        525: 47.72,
        530: 50.27,
        535: 51.78,
        540: 52.68,
        545: 167.36,
        550: 55.29,
        555: 56.94,
        560: 59.30,
        565: 62.15,
        570: 65.26,
        575: 84.26,
        580: 89.22,
        585: 75.79,
        590: 79.19,
        595: 82.80,
        600: 85.76,
        605: 88.62,
        610: 91.12,
        615: 93.43,
        620: 96.89,
        625: 101.45,
        630: 103.65,
        635: 100.30,
        640: 97.89,
        645: 96.59,
        650: 106.21,
        655: 109.97,
        660: 117.49,
        665: 96.04,
        670: 80.15,
        675: 70.42,
        680: 65.01,
        685: 60.15,
        690: 56.04,
        695: 50.92,
        700: 46.26,
        705: 42.60,
        710: 38.85,
        715: 35.09,
        720: 31.73,
        725: 28.77,
        730: 25.76,
        735: 23.16,
        740: 21.30,
        745: 18.55,
        750: 17.74,
        755: 14.74,
        760: 12.93,
        765: 13.63,
        770: 10.43,
        775: 9.67,
        780: 8.07
    },
    'FL3.5': {
        380: 4.72,
        385: 5.82,
        390: 7.18,
        395: 8.39,
        400: 9.96,
        405: 58.86,
        410: 15.78,
        415: 15.10,
        420: 17.30,
        425: 19.66,
        430: 22.43,
        435: 176.00,
        440: 28.67,
        445: 31.92,
        450: 35.38,
        455: 38.73,
        460: 41.98,
        465: 44.92,
        470: 47.49,
        475: 49.58,
        480: 51.21,
        485: 52.36,
        490: 53.99,
        495: 53.78,
        500: 54.04,
        505: 53.88,
        510: 53.62,
        515: 53.25,
        520: 53.09,
        525: 52.88,
        530: 52.99,
        535: 53.15,
        540: 53.67,
        545: 167.93,
        550: 55.61,
        555: 56.82,
        560: 58.39,
        565: 60.22,
        570: 62.21,
        575: 81.45,
        580: 84.96,
        585: 68.71,
        590: 70.70,
        595: 73.01,
        600: 74.69,
        605: 76.26,
        610: 77.68,
        615: 78.67,
        620: 80.14,
        625: 81.71,
        630: 82.08,
        635: 79.98,
        640: 78.15,
        645: 76.52,
        650: 79.20,
        655: 79.51,
        660: 81.08,
        665: 70.76,
        670: 62.58,
        675: 56.87,
        680: 52.83,
        685: 49.11,
        690: 46.28,
        695: 42.24,
        700: 38.58,
        705: 35.59,
        710: 32.76,
        715: 29.61,
        720: 26.89,
        725: 24.53,
        730: 22.17,
        735: 20.02,
        740: 18.45,
        745: 16.09,
        750: 15.62,
        755: 13.10,
        760: 11.69,
        765: 12.42,
        770: 9.43,
        775: 8.96,
        780: 7.39
    },
    'FL3.6': {
        380: 5.53,
        385: 6.63,
        390: 8.07,
        395: 9.45,
        400: 11.28,
        405: 61.47,
        410: 17.80,
        415: 17.47,
        420: 20.12,
        425: 23.05,
        430: 26.37,
        435: 186.01,
        440: 33.94,
        445: 37.98,
        450: 42.12,
        455: 46.38,
        460: 50.30,
        465: 53.95,
        470: 56.94,
        475: 59.48,
        480: 61.36,
        485: 62.68,
        490: 64.34,
        495: 63.90,
        500: 63.85,
        505: 63.24,
        510: 62.46,
        515: 61.41,
        520: 60.47,
        525: 59.48,
        530: 58.65,
        535: 57.93,
        540: 57.49,
        545: 175.17,
        550: 57.27,
        555: 57.49,
        560: 57.99,
        565: 58.76,
        570: 59.64,
        575: 78.77,
        580: 81.26,
        585: 63.18,
        590: 64.29,
        595: 65.78,
        600: 66.77,
        605: 67.77,
        610: 68.60,
        615: 69.10,
        620: 70.15,
        625: 71.69,
        630: 71.97,
        635: 69.81,
        640: 68.05,
        645: 66.66,
        650: 69.70,
        655: 70.37,
        660: 72.47,
        665: 62.30,
        670: 54.45,
        675: 49.20,
        680: 45.60,
        685: 42.40,
        690: 40.02,
        695: 36.48,
        700: 33.28,
        705: 30.84,
        710: 28.30,
        715: 25.65,
        720: 23.33,
        725: 21.23,
        730: 19.29,
        735: 17.41,
        740: 16.31,
        745: 14.21,
        750: 14.04,
        755: 11.55,
        760: 10.39,
        765: 11.28,
        770: 8.51,
        775: 8.24,
        780: 7.02
    },
    'FL3.7': {
        380: 3.79,
        385: 2.56,
        390: 1.91,
        395: 1.42,
        400: 1.51,
        405: 73.64,
        410: 7.37,
        415: 4.69,
        420: 5.33,
        425: 6.75,
        430: 8.51,
        435: 181.81,
        440: 11.71,
        445: 11.96,
        450: 12.18,
        455: 11.90,
        460: 11.16,
        465: 11.22,
        470: 9.83,
        475: 8.94,
        480: 12.08,
        485: 52.56,
        490: 55.42,
        495: 31.69,
        500: 16.03,
        505: 6.72,
        510: 4.59,
        515: 3.67,
        520: 3.02,
        525: 3.21,
        530: 4.90,
        535: 19.05,
        540: 177.64,
        545: 347.34,
        550: 116.80,
        555: 31.87,
        560: 16.37,
        565: 14.92,
        570: 14.12,
        575: 29.50,
        580: 61.40,
        585: 85.05,
        590: 64.86,
        595: 65.01,
        600: 53.17,
        605: 34.22,
        610: 427.27,
        615: 201.10,
        620: 58.63,
        625: 72.01,
        630: 88.19,
        635: 20.07,
        640: 13.10,
        645: 12.92,
        650: 24.54,
        655: 15.94,
        660: 13.56,
        665: 13.38,
        670: 8.42,
        675: 6.57,
        680: 7.18,
        685: 9.90,
        690: 11.47,
        695: 8.88,
        700: 3.05,
        705: 22.04,
        710: 42.79,
        715: 14.40,
        720: 1.88,
        725: 1.60,
        730: 1.42,
        735: 1.05,
        740: 1.23,
        745: 1.76,
        750: 0.74,
        755: 0.52,
        760: 4.10,
        765: 0.46,
        770: 0.99,
        775: 0.43,
        780: 0.00
    },
    'FL3.8': {
        380: 4.18,
        385: 2.93,
        390: 2.29,
        395: 1.98,
        400: 2.44,
        405: 70.70,
        410: 10.19,
        415: 9.79,
        420: 13.21,
        425: 17.79,
        430: 22.98,
        435: 191.43,
        440: 31.76,
        445: 33.35,
        450: 33.87,
        455: 32.89,
        460: 30.60,
        465: 28.28,
        470: 24.81,
        475: 21.60,
        480: 23.40,
        485: 68.99,
        490: 70.85,
        495: 42.29,
        500: 22.67,
        505: 11.08,
        510: 7.66,
        515: 6.07,
        520: 5.07,
        525: 4.88,
        530: 6.26,
        535: 20.29,
        540: 204.67,
        545: 390.25,
        550: 135.69,
        555: 34.57,
        560: 15.71,
        565: 12.60,
        570: 11.05,
        575: 25.05,
        580: 54.98,
        585: 82.84,
        590: 58.22,
        595: 53.06,
        600: 41.44,
        605: 25.26,
        610: 329.89,
        615: 161.29,
        620: 54.19,
        625: 66.30,
        630: 71.43,
        635: 15.74,
        640: 10.22,
        645: 10.68,
        650: 20.32,
        655: 14.13,
        660: 11.72,
        665: 11.75,
        670: 7.87,
        675: 6.38,
        680: 7.23,
        685: 8.94,
        690: 9.79,
        695: 7.26,
        700: 2.59,
        705: 17.03,
        710: 33.69,
        715: 12.02,
        720: 1.68,
        725: 1.50,
        730: 1.31,
        735: 1.01,
        740: 1.16,
        745: 1.59,
        750: 0.79,
        755: 0.67,
        760: 4.82,
        765: 0.61,
        770: 1.25,
        775: 0.79,
        780: 0.58
    },
    'FL3.9': {
        380: 3.77,
        385: 2.64,
        390: 2.06,
        395: 1.87,
        400: 2.55,
        405: 71.68,
        410: 12.05,
        415: 13.57,
        420: 19.60,
        425: 27.33,
        430: 35.39,
        435: 211.82,
        440: 49.02,
        445: 51.83,
        450: 52.50,
        455: 50.73,
        460: 46.93,
        465: 42.42,
        470: 37.16,
        475: 31.84,
        480: 31.94,
        485: 77.74,
        490: 79.45,
        495: 47.93,
        500: 26.24,
        505: 13.15,
        510: 8.80,
        515: 6.70,
        520: 5.38,
        525: 4.93,
        530: 6.06,
        535: 19.76,
        540: 215.94,
        545: 412.13,
        550: 142.39,
        555: 34.74,
        560: 14.76,
        565: 10.99,
        570: 9.25,
        575: 23.50,
        580: 53.05,
        585: 81.90,
        590: 54.92,
        595: 47.80,
        600: 36.65,
        605: 21.82,
        610: 285.69,
        615: 139.94,
        620: 53.37,
        625: 64.30,
        630: 64.04,
        635: 13.79,
        640: 9.06,
        645: 9.83,
        650: 18.60,
        655: 13.38,
        660: 10.99,
        665: 10.77,
        670: 7.57,
        675: 6.19,
        680: 7.09,
        685: 8.54,
        690: 8.77,
        695: 6.41,
        700: 2.26,
        705: 15.02,
        710: 29.39,
        715: 10.22,
        720: 1.42,
        725: 1.23,
        730: 1.10,
        735: 0.84,
        740: 0.97,
        745: 1.35,
        750: 0.65,
        755: 0.13,
        760: 4.22,
        765: 0.10,
        770: 0.68,
        775: 0.16,
        780: 0.00
    },
    'FL3.10': {
        380: 0.25,
        385: 0.00,
        390: 0.00,
        395: 0.00,
        400: 0.69,
        405: 21.24,
        410: 2.18,
        415: 1.86,
        420: 3.10,
        425: 5.00,
        430: 7.03,
        435: 45.08,
        440: 16.78,
        445: 12.28,
        450: 13.31,
        455: 13.66,
        460: 13.69,
        465: 13.13,
        470: 12.28,
        475: 11.42,
        480: 11.66,
        485: 22.04,
        490: 26.17,
        495: 18.57,
        500: 11.36,
        505: 6.83,
        510: 5.58,
        515: 4.88,
        520: 4.31,
        525: 3.76,
        530: 3.61,
        535: 5.62,
        540: 38.59,
        545: 100.00,
        550: 36.54,
        555: 10.57,
        560: 2.98,
        565: 2.05,
        570: 1.84,
        575: 6.09,
        580: 17.27,
        585: 21.77,
        590: 18.72,
        595: 10.15,
        600: 7.26,
        605: 5.17,
        610: 56.66,
        615: 49.39,
        620: 18.57,
        625: 14.21,
        630: 14.01,
        635: 5.99,
        640: 2.68,
        645: 3.14,
        650: 6.25,
        655: 5.78,
        660: 6.75,
        665: 5.16,
        670: 3.03,
        675: 1.57,
        680: 1.72,
        685: 1.54,
        690: 1.71,
        695: 1.10,
        700: 0.28,
        705: 3.65,
        710: 7.54,
        715: 2.34,
        720: 0.05,
        725: 0.04,
        730: 0.04,
        735: 0.03,
        740: 0.03,
        745: 0.02,
        750: 0.02,
        755: 0.01,
        760: 0.01,
        765: 0.00,
        770: 0.00,
        775: 0.00,
        780: 0.00
    },
    'FL3.11': {
        380: 3.85,
        385: 2.91,
        390: 2.56,
        395: 2.59,
        400: 3.63,
        405: 74.54,
        410: 14.69,
        415: 17.22,
        420: 24.99,
        425: 34.40,
        430: 44.57,
        435: 228.08,
        440: 61.53,
        445: 65.31,
        450: 66.35,
        455: 64.37,
        460: 59.81,
        465: 54.24,
        470: 47.42,
        475: 41.10,
        480: 40.04,
        485: 85.54,
        490: 86.55,
        495: 53.47,
        500: 30.91,
        505: 17.41,
        510: 12.56,
        515: 10.10,
        520: 8.48,
        525: 7.74,
        530: 8.58,
        535: 21.39,
        540: 220.12,
        545: 417.35,
        550: 146.13,
        555: 36.67,
        560: 16.51,
        565: 12.56,
        570: 10.81,
        575: 25.31,
        580: 53.31,
        585: 80.75,
        590: 53.56,
        595: 44.02,
        600: 33.05,
        605: 20.26,
        610: 233.61,
        615: 118.20,
        620: 51.66,
        625: 61.27,
        630: 55.15,
        635: 12.95,
        640: 8.93,
        645: 9.77,
        650: 17.12,
        655: 13.01,
        660: 10.45,
        665: 10.33,
        670: 7.70,
        675: 6.34,
        680: 7.35,
        685: 8.22,
        690: 7.93,
        695: 5.70,
        700: 2.23,
        705: 12.43,
        710: 24.24,
        715: 8.74,
        720: 1.39,
        725: 1.23,
        730: 1.10,
        735: 0.84,
        740: 0.94,
        745: 1.23,
        750: 0.68,
        755: 0.52,
        760: 4.60,
        765: 0.45,
        770: 1.04,
        775: 0.45,
        780: 0.00
    },
    'FL3.12': {
        380: 1.62,
        385: 2.06,
        390: 2.71,
        395: 3.11,
        400: 3.67,
        405: 74.60,
        410: 8.88,
        415: 4.77,
        420: 4.72,
        425: 4.72,
        430: 4.94,
        435: 150.29,
        440: 6.08,
        445: 7.13,
        450: 9.10,
        455: 11.76,
        460: 14.96,
        465: 18.54,
        470: 22.48,
        475: 26.76,
        480: 31.66,
        485: 40.93,
        490: 45.83,
        495: 46.00,
        500: 45.26,
        505: 43.16,
        510: 41.63,
        515: 39.75,
        520: 37.83,
        525: 36.16,
        530: 35.25,
        535: 37.04,
        540: 59.86,
        545: 183.53,
        550: 59.03,
        555: 47.93,
        560: 48.67,
        565: 52.69,
        570: 57.24,
        575: 77.75,
        580: 87.81,
        585: 80.55,
        590: 84.83,
        595: 86.84,
        600: 91.44,
        605: 96.51,
        610: 105.25,
        615: 106.74,
        620: 108.53,
        625: 106.92,
        630: 101.54,
        635: 95.20,
        640: 89.34,
        645: 82.95,
        650: 75.78,
        655: 68.65,
        660: 61.70,
        665: 55.23,
        670: 48.58,
        675: 42.90,
        680: 37.74,
        685: 32.93,
        690: 29.65,
        695: 25.19,
        700: 21.69,
        705: 19.28,
        710: 17.36,
        715: 14.74,
        720: 12.86,
        725: 11.28,
        730: 9.97,
        735: 8.88,
        740: 7.78,
        745: 7.04,
        750: 6.30,
        755: 5.55,
        760: 10.15,
        765: 4.50,
        770: 4.81,
        775: 3.72,
        780: 3.28
    },
    'FL3.13': {
        380: 2.23,
        385: 2.92,
        390: 3.91,
        395: 4.55,
        400: 5.46,
        405: 77.40,
        410: 11.25,
        415: 7.69,
        420: 8.29,
        425: 8.98,
        430: 10.01,
        435: 204.45,
        440: 13.75,
        445: 16.88,
        450: 21.73,
        455: 27.96,
        460: 34.92,
        465: 41.96,
        470: 48.62,
        475: 54.33,
        480: 59.49,
        485: 67.91,
        490: 70.01,
        495: 66.40,
        500: 62.07,
        505: 56.95,
        510: 52.70,
        515: 48.54,
        520: 44.80,
        525: 41.75,
        530: 39.77,
        535: 40.50,
        540: 59.27,
        545: 184.09,
        550: 59.06,
        555: 49.95,
        560: 50.90,
        565: 54.51,
        570: 58.33,
        575: 77.49,
        580: 85.78,
        585: 76.20,
        590: 78.73,
        595: 78.95,
        600: 81.48,
        605: 84.57,
        610: 87.75,
        615: 89.56,
        620: 91.36,
        625: 89.00,
        630: 83.67,
        635: 78.26,
        640: 73.19,
        645: 67.61,
        650: 61.42,
        655: 55.49,
        660: 49.78,
        665: 44.46,
        670: 39.13,
        675: 34.45,
        680: 30.28,
        685: 26.37,
        690: 23.88,
        695: 20.10,
        700: 17.40,
        705: 15.29,
        710: 13.62,
        715: 11.68,
        720: 10.31,
        725: 9.11,
        730: 8.03,
        735: 7.13,
        740: 6.31,
        745: 5.67,
        750: 5.11,
        755: 4.55,
        760: 9.06,
        765: 3.74,
        770: 4.04,
        775: 3.14,
        780: 2.75
    },
    'FL3.14': {
        380: 2.87,
        385: 3.69,
        390: 4.87,
        395: 5.82,
        400: 7.17,
        405: 72.21,
        410: 13.69,
        415: 11.12,
        420: 12.43,
        425: 13.90,
        430: 15.82,
        435: 200.99,
        440: 21.72,
        445: 26.33,
        450: 32.85,
        455: 40.80,
        460: 49.23,
        465: 57.39,
        470: 65.26,
        475: 71.99,
        480: 78.25,
        485: 88.85,
        490: 91.67,
        495: 86.81,
        500: 80.42,
        505: 73.82,
        510: 69.12,
        515: 63.69,
        520: 58.44,
        525: 53.57,
        530: 49.66,
        535: 48.44,
        540: 72.56,
        545: 200.42,
        550: 65.00,
        555: 47.49,
        560: 44.14,
        565: 44.71,
        570: 46.01,
        575: 63.52,
        580: 71.73,
        585: 63.52,
        590: 64.13,
        595: 63.74,
        600: 66.82,
        605: 70.65,
        610: 79.29,
        615: 80.77,
        620: 83.59,
        625: 82.59,
        630: 77.60,
        635: 72.47,
        640: 68.34,
        645: 63.82,
        650: 58.57,
        655: 53.18,
        660: 47.97,
        665: 43.14,
        670: 38.19,
        675: 33.85,
        680: 29.94,
        685: 26.24,
        690: 23.90,
        695: 20.33,
        700: 17.42,
        705: 15.64,
        710: 14.34,
        715: 12.21,
        720: 10.65,
        725: 9.43,
        730: 8.34,
        735: 7.52,
        740: 6.73,
        745: 6.08,
        750: 5.52,
        755: 5.00,
        760: 9.47,
        765: 4.08,
        770: 4.43,
        775: 3.39,
        780: 3.17
    },
    'FL3.15': {
        380: 300,
        385: 286,
        390: 268,
        395: 244,
        400: 304,
        405: 581,
        410: 225,
        415: 155,
        420: 152,
        425: 170,
        430: 295,
        435: 1417,
        440: 607,
        445: 343,
        450: 386,
        455: 430,
        460: 469,
        465: 502,
        470: 531,
        475: 552,
        480: 567,
        485: 572,
        490: 575,
        495: 561,
        500: 548,
        505: 527,
        510: 507,
        515: 482,
        520: 461,
        525: 438,
        530: 418,
        535: 404,
        540: 429,
        545: 1016,
        550: 581,
        555: 370,
        560: 368,
        565: 371,
        570: 377,
        575: 490,
        580: 525,
        585: 402,
        590: 404,
        595: 412,
        600: 418,
        605: 425,
        610: 428,
        615: 432,
        620: 433,
        625: 431,
        630: 427,
        635: 420,
        640: 410,
        645: 399,
        650: 385,
        655: 370,
        660: 352,
        665: 336,
        670: 317,
        675: 298,
        680: 277,
        685: 260,
        690: 242,
        695: 223,
        700: 202,
        705: 187,
        710: 167,
        715: 152,
        720: 136,
        725: 125,
        730: 113,
        735: 103,
        740: 93,
        745: 84,
        750: 75,
        755: 66,
        760: 58,
        765: 51,
        770: 46,
        775: 41,
        780: 37
    },
    'HP1': {
        380: 1.90,
        385: 2.20,
        390: 2.50,
        395: 2.70,
        400: 3.10,
        405: 4.30,
        410: 3.80,
        415: 4.20,
        420: 4.80,
        425: 5.19,
        430: 5.89,
        435: 7.39,
        440: 7.89,
        445: 5.69,
        450: 12.89,
        455: 6.69,
        460: 4.30,
        465: 20.78,
        470: 12.99,
        475: 6.69,
        480: 1.40,
        485: 1.50,
        490: 3.20,
        495: 18.18,
        500: 56.24,
        505: 2.90,
        510: 2.10,
        515: 13.39,
        520: 2.10,
        525: 2.00,
        530: 2.20,
        535: 2.30,
        540: 2.60,
        545: 5.10,
        550: 11.39,
        555: 15.48,
        560: 20.78,
        565: 55.64,
        570: 254.03,
        575: 56.14,
        580: 111.78,
        585: 297.98,
        590: 142.55,
        595: 334.84,
        600: 189.40,
        605: 117.78,
        610: 79.92,
        615: 108.09,
        620: 46.85,
        625: 38.16,
        630: 32.47,
        635: 28.37,
        640: 25.37,
        645: 22.98,
        650: 20.38,
        655: 19.78,
        660: 17.78,
        665: 16.78,
        670: 19.18,
        675: 17.98,
        680: 13.69,
        685: 9.99,
        690: 8.19,
        695: 7.59,
        700: 6.99,
        705: 6.79,
        710: 6.49,
        715: 6.39,
        720: 6.09,
        725: 5.99,
        730: 5.79,
        735: 5.79,
        740: 5.79,
        745: 5.79,
        750: 6.39,
        755: 5.99,
        760: 5.59,
        765: 31.97,
        770: 27.87,
        775: 5.89,
        780: 6.69
    },
    'HP2': {
        380: 2.64,
        385: 2.77,
        390: 3.42,
        395: 3.68,
        400: 4.33,
        405: 5.50,
        410: 5.94,
        415: 7.20,
        420: 9.02,
        425: 10.27,
        430: 12.48,
        435: 16.82,
        440: 16.04,
        445: 15.26,
        450: 22.58,
        455: 20.07,
        460: 15.13,
        465: 25.27,
        470: 28.04,
        475: 15.99,
        480: 10.40,
        485: 11.10,
        490: 13.44,
        495: 22.62,
        500: 49.71,
        505: 17.21,
        510: 17.12,
        515: 27.26,
        520: 20.02,
        525: 21.54,
        530: 23.36,
        535: 25.66,
        540: 29.69,
        545: 43.12,
        550: 98.30,
        555: 125.60,
        560: 134.57,
        565: 149.70,
        570: 166.12,
        575: 98.77,
        580: 30.47,
        585: 1.17,
        590: 0.39,
        595: 1.65,
        600: 21.41,
        605: 76.11,
        610: 126.16,
        615: 161.96,
        620: 160.06,
        625: 158.19,
        630: 153.69,
        635: 147.40,
        640: 140.60,
        645: 134.92,
        650: 127.59,
        655: 124.65,
        660: 118.02,
        665: 113.94,
        670: 118.10,
        675: 115.16,
        680: 102.85,
        685: 90.54,
        690: 83.34,
        695: 79.44,
        700: 76.97,
        705: 74.85,
        710: 73.12,
        715: 71.51,
        720: 70.13,
        725: 69.04,
        730: 67.48,
        735: 66.70,
        740: 66.31,
        745: 65.14,
        750: 65.70,
        755: 64.79,
        760: 64.10,
        765: 83.04,
        770: 86.25,
        775: 63.93,
        780: 64.92
    },
    'HP3': {
        380: 3.15,
        385: 7.49,
        390: 10.87,
        395: 12.57,
        400: 12.97,
        405: 21.29,
        410: 26.29,
        415: 30.18,
        420: 43.06,
        425: 29.58,
        430: 23.18,
        435: 35.28,
        440: 26.29,
        445: 24.29,
        450: 22.91,
        455: 26.20,
        460: 29.31,
        465: 25.30,
        470: 28.14,
        475: 24.05,
        480: 21.82,
        485: 20.51,
        490: 23.05,
        495: 26.98,
        500: 30.96,
        505: 30.72,
        510: 27.13,
        515: 29.55,
        520: 34.22,
        525: 29.98,
        530: 41.21,
        535: 173.14,
        540: 141.37,
        545: 64.98,
        550: 33.83,
        555: 34.26,
        560: 33.32,
        565: 52.80,
        570: 74.29,
        575: 47.97,
        580: 49.20,
        585: 96.07,
        590: 85.41,
        595: 175.18,
        600: 153.73,
        605: 120.22,
        610: 98.90,
        615: 90.22,
        620: 70.07,
        625: 66.84,
        630: 57.61,
        635: 53.03,
        640: 49.85,
        645: 48.16,
        650: 42.76,
        655: 50.64,
        660: 48.42,
        665: 41.27,
        670: 43.44,
        675: 40.48,
        680: 35.16,
        685: 34.94,
        690: 24.68,
        695: 24.70,
        700: 21.49,
        705: 19.49,
        710: 18.48,
        715: 17.55,
        720: 17.36,
        725: 17.09,
        730: 16.32,
        735: 16.07,
        740: 16.58,
        745: 15.78,
        750: 17.66,
        755: 20.46,
        760: 16.59,
        765: 17.81,
        770: 16.07,
        775: 14.83,
        780: 14.61
    },
    'HP4': {
        380: 9.80,
        385: 13.30,
        390: 19.97,
        395: 25.81,
        400: 24.69,
        405: 47.66,
        410: 54.44,
        415: 63.82,
        420: 85.52,
        425: 60.54,
        430: 38.37,
        435: 88.20,
        440: 44.94,
        445: 35.64,
        450: 30.75,
        455: 33.77,
        460: 40.81,
        465: 33.77,
        470: 35.28,
        475: 32.55,
        480: 29.44,
        485: 26.16,
        490: 29.96,
        495: 32.83,
        500: 33.58,
        505: 41.16,
        510: 32.93,
        515: 32.13,
        520: 34.45,
        525: 30.12,
        530: 41.13,
        535: 187.10,
        540: 101.37,
        545: 123.96,
        550: 42.47,
        555: 34.73,
        560: 31.82,
        565: 54.67,
        570: 57.45,
        575: 70.43,
        580: 69.50,
        585: 49.37,
        590: 183.35,
        595: 162.15,
        600: 109.35,
        605: 72.38,
        610: 70.60,
        615: 58.08,
        620: 44.13,
        625: 50.20,
        630: 40.80,
        635: 37.91,
        640: 36.71,
        645: 38.30,
        650: 31.24,
        655: 35.31,
        660: 45.62,
        665: 35.82,
        670: 89.91,
        675: 36.01,
        680: 32.57,
        685: 39.26,
        690: 23.27,
        695: 25.30,
        700: 20.02,
        705: 17.54,
        710: 16.25,
        715: 15.20,
        720: 15.15,
        725: 15.22,
        730: 14.26,
        735: 12.63,
        740: 14.75,
        745: 13.19,
        750: 17.63,
        755: 23.38,
        760: 16.02,
        765: 24.46,
        770: 22.05,
        775: 16.11,
        780: 12.91
    },
    'HP5': {
        380: 0.34,
        385: 7.11,
        390: 11.49,
        395: 14.97,
        400: 14.95,
        405: 29.14,
        410: 38.08,
        415: 51.56,
        420: 62.56,
        425: 55.61,
        430: 41.98,
        435: 50.02,
        440: 42.14,
        445: 39.04,
        450: 40.52,
        455: 45.29,
        460: 51.01,
        465: 49.18,
        470: 49.05,
        475: 46.12,
        480: 45.73,
        485: 39.46,
        490: 44.39,
        495: 46.14,
        500: 49.54,
        505: 59.76,
        510: 48.47,
        515: 48.38,
        520: 48.70,
        525: 44.25,
        530: 54.42,
        535: 128.93,
        540: 81.26,
        545: 67.36,
        550: 48.48,
        555: 51.41,
        560: 48.88,
        565: 68.52,
        570: 80.85,
        575: 65.96,
        580: 59.43,
        585: 67.57,
        590: 128.34,
        595: 131.85,
        600: 101.70,
        605: 77.05,
        610: 66.27,
        615: 77.09,
        620: 60.51,
        625: 65.23,
        630: 57.86,
        635: 56.20,
        640: 54.32,
        645: 56.34,
        650: 45.74,
        655: 50.79,
        660: 56.66,
        665: 51.99,
        670: 84.31,
        675: 47.48,
        680: 47.46,
        685: 61.78,
        690: 34.51,
        695: 38.74,
        700: 30.98,
        705: 25.45,
        710: 22.88,
        715: 20.82,
        720: 21.05,
        725: 20.81,
        730: 18.69,
        735: 17.54,
        740: 19.58,
        745: 16.42,
        750: 23.77,
        755: 35.39,
        760: 21.37,
        765: 34.58,
        770: 30.21,
        775: 19.71,
        780: 15.61
    }
}

ILLUMINANTS_RELATIVE_SPDS = CaseInsensitiveMapping({
    'A': SpectralPowerDistribution(
        'A', ILLUMINANTS_RELATIVE_SPDS_DATA['A']),
    'B': SpectralPowerDistribution(
        'B', ILLUMINANTS_RELATIVE_SPDS_DATA['B']),
    'C': SpectralPowerDistribution(
        'C', ILLUMINANTS_RELATIVE_SPDS_DATA['C']),
    'D50': SpectralPowerDistribution(
        'D50', ILLUMINANTS_RELATIVE_SPDS_DATA['D50']),
    'D55': SpectralPowerDistribution(
        'D55', ILLUMINANTS_RELATIVE_SPDS_DATA['D55']),
    'D60': SpectralPowerDistribution(
        'D60', ILLUMINANTS_RELATIVE_SPDS_DATA['D60']),
    'D65': SpectralPowerDistribution(
        'D65', ILLUMINANTS_RELATIVE_SPDS_DATA['D65']),
    'D75': SpectralPowerDistribution(
        'D75', ILLUMINANTS_RELATIVE_SPDS_DATA['D75']),
    'E': SpectralPowerDistribution(
        'E', ILLUMINANTS_RELATIVE_SPDS_DATA['E']),
    'F1': SpectralPowerDistribution(
        'F1', ILLUMINANTS_RELATIVE_SPDS_DATA['F1']),
    'F2': SpectralPowerDistribution(
        'F2', ILLUMINANTS_RELATIVE_SPDS_DATA['F2']),
    'F3': SpectralPowerDistribution(
        'F3', ILLUMINANTS_RELATIVE_SPDS_DATA['F3']),
    'F4': SpectralPowerDistribution(
        'F4', ILLUMINANTS_RELATIVE_SPDS_DATA['F4']),
    'F5': SpectralPowerDistribution(
        'F5', ILLUMINANTS_RELATIVE_SPDS_DATA['F5']),
    'F6': SpectralPowerDistribution(
        'F6', ILLUMINANTS_RELATIVE_SPDS_DATA['F6']),
    'F7': SpectralPowerDistribution(
        'F7', ILLUMINANTS_RELATIVE_SPDS_DATA['F7']),
    'F8': SpectralPowerDistribution(
        'F8', ILLUMINANTS_RELATIVE_SPDS_DATA['F8']),
    'F9': SpectralPowerDistribution(
        'F9', ILLUMINANTS_RELATIVE_SPDS_DATA['F9']),
    'F10': SpectralPowerDistribution(
        'F10', ILLUMINANTS_RELATIVE_SPDS_DATA['F10']),
    'F11': SpectralPowerDistribution(
        'F11', ILLUMINANTS_RELATIVE_SPDS_DATA['F11']),
    'F12': SpectralPowerDistribution(
        'F12', ILLUMINANTS_RELATIVE_SPDS_DATA['F12']),
    'FL3.1': SpectralPowerDistribution(
        'FL3.1', ILLUMINANTS_RELATIVE_SPDS_DATA['FL3.1']),
    'FL3.2': SpectralPowerDistribution(
        'FL3.2', ILLUMINANTS_RELATIVE_SPDS_DATA['FL3.2']),
    'FL3.3': SpectralPowerDistribution(
        'FL3.3', ILLUMINANTS_RELATIVE_SPDS_DATA['FL3.3']),
    'FL3.4': SpectralPowerDistribution(
        'FL3.4', ILLUMINANTS_RELATIVE_SPDS_DATA['FL3.4']),
    'FL3.5': SpectralPowerDistribution(
        'FL3.5', ILLUMINANTS_RELATIVE_SPDS_DATA['FL3.5']),
    'FL3.6': SpectralPowerDistribution(
        'FL3.6', ILLUMINANTS_RELATIVE_SPDS_DATA['FL3.6']),
    'FL3.7': SpectralPowerDistribution(
        'FL3.7', ILLUMINANTS_RELATIVE_SPDS_DATA['FL3.7']),
    'FL3.8': SpectralPowerDistribution(
        'FL3.8', ILLUMINANTS_RELATIVE_SPDS_DATA['FL3.8']),
    'FL3.9': SpectralPowerDistribution(
        'FL3.9', ILLUMINANTS_RELATIVE_SPDS_DATA['FL3.9']),
    'FL3.10': SpectralPowerDistribution(
        'FL3.10', ILLUMINANTS_RELATIVE_SPDS_DATA['FL3.10']),
    'FL3.11': SpectralPowerDistribution(
        'FL3.11', ILLUMINANTS_RELATIVE_SPDS_DATA['FL3.11']),
    'FL3.12': SpectralPowerDistribution(
        'FL3.12', ILLUMINANTS_RELATIVE_SPDS_DATA['FL3.12']),
    'FL3.13': SpectralPowerDistribution(
        'FL3.13', ILLUMINANTS_RELATIVE_SPDS_DATA['FL3.13']),
    'FL3.14': SpectralPowerDistribution(
        'FL3.14', ILLUMINANTS_RELATIVE_SPDS_DATA['FL3.14']),
    'FL3.15': SpectralPowerDistribution(
        'FL3.15', ILLUMINANTS_RELATIVE_SPDS_DATA['FL3.15']),
    'HP1': SpectralPowerDistribution(
        'HP1', ILLUMINANTS_RELATIVE_SPDS_DATA['HP1']),
    'HP2': SpectralPowerDistribution(
        'HP2', ILLUMINANTS_RELATIVE_SPDS_DATA['HP2']),
    'HP3': SpectralPowerDistribution(
        'HP3', ILLUMINANTS_RELATIVE_SPDS_DATA['HP3']),
    'HP4': SpectralPowerDistribution(
        'HP4', ILLUMINANTS_RELATIVE_SPDS_DATA['HP4']),
    'HP5': SpectralPowerDistribution(
        'HP5', ILLUMINANTS_RELATIVE_SPDS_DATA['HP5'])
})  # yapf: disable
"""
*CIE* illuminants relative spectral power distributions.

ILLUMINANTS_RELATIVE_SPDS : CaseInsensitiveMapping
"""
