#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
ColourCheckers Spectral Power Distributions
===========================================

Defines *ColourCheckers* spectral power distributions.

Each *ColourChecker* data is in the form of a *dict* of
:class:`colour.colorimetry.spectrum.SpectralPowerDistribution` classes as
follows::

    {'name': SpectralPowerDistribution, ..., 'name': SpectralPowerDistribution}

The following *ColourCheckers* data is available:

-   :attr:`COLORCHECKER_N_OHTA_SPDS`: Measured by Ohta (1997).
-   :attr:`BABELCOLOR_AVERAGE_SPDS`: Average data derived from measurements of
    30 *ColourChecker* charts.

See Also
--------
`Colour Fitting Jupyter Notebook
<http://nbviewer.jupyter.org/github/colour-science/colour-notebooks/\
blob/master/notebooks/characterisation/fitting.ipynb>`_

References
----------
.. [1]  Ohta, N. (1997). The basis of color reproduction engineering.
.. [2]  Munsell Color Science. (n.d.). Macbeth Colorchecker. Retrieved from
        http://www.rit-mcsl.org/UsefulData/MacbethColorChecker.xls
.. [3]  BabelColor. (2012). ColorChecker RGB and spectra. Retrieved from
        http://www.babelcolor.com/download/ColorChecker_RGB_and_spectra.xls
.. [4]  BabelColor. (2012). The ColorChecker (since 1976!). Retrieved
        September 26, 2014, from
        http://www.babelcolor.com/main_level/ColorChecker.htm
"""

from __future__ import division, unicode_literals

from colour.colorimetry import SpectralPowerDistribution
from colour.utilities import CaseInsensitiveMapping

__author__ = 'Colour Developers'
__copyright__ = 'Copyright (C) 2013-2017 - Colour Developers'
__license__ = 'New BSD License - http://opensource.org/licenses/BSD-3-Clause'
__maintainer__ = 'Colour Developers'
__email__ = 'colour-science@googlegroups.com'
__status__ = 'Production'

__all__ = [
    'COLOURCHECKER_INDEXES_TO_NAMES_MAPPING', 'COLORCHECKER_N_OHTA_SPDS_DATA',
    'COLORCHECKER_N_OHTA_SPDS', 'BABELCOLOR_AVERAGE_SPDS_DATA',
    'BABELCOLOR_AVERAGE_SPDS', 'COLOURCHECKERS_SPDS'
]

COLOURCHECKER_INDEXES_TO_NAMES_MAPPING = {
    1: 'dark skin',
    2: 'light skin',
    3: 'blue sky',
    4: 'foliage',
    5: 'blue flower',
    6: 'bluish green',
    7: 'orange',
    8: 'purplish blue',
    9: 'moderate red',
    10: 'purple',
    11: 'yellow green',
    12: 'orange yellow',
    13: 'blue',
    14: 'green',
    15: 'red',
    16: 'yellow',
    17: 'magenta',
    18: 'cyan',
    19: 'white 9.5 (.05 D)',
    20: 'neutral 8 (.23 D)',
    21: 'neutral 6.5 (.44 D)',
    22: 'neutral 5 (.70 D)',
    23: 'neutral 3.5 (1.05 D)',
    24: 'black 2 (1.5 D)'
}

COLORCHECKER_N_OHTA_SPDS_DATA = {
    'dark skin': {
        380: 0.048,
        385: 0.051,
        390: 0.055,
        395: 0.060,
        400: 0.065,
        405: 0.068,
        410: 0.068,
        415: 0.067,
        420: 0.064,
        425: 0.062,
        430: 0.059,
        435: 0.057,
        440: 0.055,
        445: 0.054,
        450: 0.053,
        455: 0.053,
        460: 0.052,
        465: 0.052,
        470: 0.052,
        475: 0.053,
        480: 0.054,
        485: 0.055,
        490: 0.057,
        495: 0.059,
        500: 0.061,
        505: 0.062,
        510: 0.065,
        515: 0.067,
        520: 0.070,
        525: 0.072,
        530: 0.074,
        535: 0.075,
        540: 0.076,
        545: 0.078,
        550: 0.079,
        555: 0.082,
        560: 0.087,
        565: 0.092,
        570: 0.100,
        575: 0.107,
        580: 0.115,
        585: 0.122,
        590: 0.129,
        595: 0.134,
        600: 0.138,
        605: 0.142,
        610: 0.146,
        615: 0.150,
        620: 0.154,
        625: 0.158,
        630: 0.163,
        635: 0.167,
        640: 0.173,
        645: 0.180,
        650: 0.188,
        655: 0.196,
        660: 0.204,
        665: 0.213,
        670: 0.222,
        675: 0.231,
        680: 0.242,
        685: 0.251,
        690: 0.261,
        695: 0.271,
        700: 0.282,
        705: 0.294,
        710: 0.305,
        715: 0.318,
        720: 0.334,
        725: 0.354,
        730: 0.372,
        735: 0.392,
        740: 0.409,
        745: 0.420,
        750: 0.436,
        755: 0.450,
        760: 0.462,
        765: 0.465,
        770: 0.448,
        775: 0.432,
        780: 0.421
    },
    'light skin': {
        380: 0.103,
        385: 0.120,
        390: 0.141,
        395: 0.163,
        400: 0.182,
        405: 0.192,
        410: 0.197,
        415: 0.199,
        420: 0.201,
        425: 0.203,
        430: 0.205,
        435: 0.208,
        440: 0.212,
        445: 0.217,
        450: 0.224,
        455: 0.231,
        460: 0.240,
        465: 0.251,
        470: 0.262,
        475: 0.273,
        480: 0.282,
        485: 0.289,
        490: 0.293,
        495: 0.296,
        500: 0.301,
        505: 0.310,
        510: 0.321,
        515: 0.326,
        520: 0.322,
        525: 0.310,
        530: 0.298,
        535: 0.291,
        540: 0.292,
        545: 0.297,
        550: 0.300,
        555: 0.298,
        560: 0.295,
        565: 0.295,
        570: 0.305,
        575: 0.326,
        580: 0.358,
        585: 0.397,
        590: 0.435,
        595: 0.468,
        600: 0.494,
        605: 0.514,
        610: 0.530,
        615: 0.541,
        620: 0.550,
        625: 0.557,
        630: 0.564,
        635: 0.569,
        640: 0.574,
        645: 0.582,
        650: 0.590,
        655: 0.597,
        660: 0.605,
        665: 0.614,
        670: 0.624,
        675: 0.637,
        680: 0.652,
        685: 0.668,
        690: 0.682,
        695: 0.697,
        700: 0.713,
        705: 0.728,
        710: 0.745,
        715: 0.753,
        720: 0.762,
        725: 0.774,
        730: 0.783,
        735: 0.788,
        740: 0.791,
        745: 0.787,
        750: 0.789,
        755: 0.794,
        760: 0.801,
        765: 0.799,
        770: 0.771,
        775: 0.747,
        780: 0.734
    },
    'blue sky': {
        380: 0.113,
        385: 0.138,
        390: 0.174,
        395: 0.219,
        400: 0.266,
        405: 0.300,
        410: 0.320,
        415: 0.330,
        420: 0.336,
        425: 0.337,
        430: 0.337,
        435: 0.337,
        440: 0.335,
        445: 0.334,
        450: 0.331,
        455: 0.327,
        460: 0.322,
        465: 0.316,
        470: 0.310,
        475: 0.302,
        480: 0.293,
        485: 0.285,
        490: 0.276,
        495: 0.268,
        500: 0.260,
        505: 0.251,
        510: 0.243,
        515: 0.234,
        520: 0.225,
        525: 0.215,
        530: 0.208,
        535: 0.203,
        540: 0.198,
        545: 0.195,
        550: 0.191,
        555: 0.188,
        560: 0.183,
        565: 0.177,
        570: 0.172,
        575: 0.167,
        580: 0.163,
        585: 0.160,
        590: 0.157,
        595: 0.153,
        600: 0.150,
        605: 0.147,
        610: 0.144,
        615: 0.141,
        620: 0.137,
        625: 0.133,
        630: 0.130,
        635: 0.126,
        640: 0.123,
        645: 0.120,
        650: 0.118,
        655: 0.115,
        660: 0.112,
        665: 0.110,
        670: 0.108,
        675: 0.106,
        680: 0.105,
        685: 0.104,
        690: 0.104,
        695: 0.103,
        700: 0.103,
        705: 0.102,
        710: 0.102,
        715: 0.102,
        720: 0.102,
        725: 0.102,
        730: 0.102,
        735: 0.104,
        740: 0.104,
        745: 0.104,
        750: 0.104,
        755: 0.106,
        760: 0.106,
        765: 0.107,
        770: 0.110,
        775: 0.115,
        780: 0.12
    },
    'foliage': {
        380: 0.048,
        385: 0.049,
        390: 0.049,
        395: 0.049,
        400: 0.050,
        405: 0.049,
        410: 0.049,
        415: 0.050,
        420: 0.050,
        425: 0.051,
        430: 0.052,
        435: 0.053,
        440: 0.054,
        445: 0.056,
        450: 0.058,
        455: 0.060,
        460: 0.061,
        465: 0.063,
        470: 0.064,
        475: 0.065,
        480: 0.067,
        485: 0.068,
        490: 0.070,
        495: 0.072,
        500: 0.078,
        505: 0.088,
        510: 0.106,
        515: 0.130,
        520: 0.155,
        525: 0.173,
        530: 0.181,
        535: 0.182,
        540: 0.177,
        545: 0.168,
        550: 0.157,
        555: 0.147,
        560: 0.137,
        565: 0.129,
        570: 0.126,
        575: 0.125,
        580: 0.122,
        585: 0.119,
        590: 0.115,
        595: 0.109,
        600: 0.104,
        605: 0.100,
        610: 0.098,
        615: 0.097,
        620: 0.098,
        625: 0.100,
        630: 0.100,
        635: 0.099,
        640: 0.097,
        645: 0.096,
        650: 0.095,
        655: 0.095,
        660: 0.095,
        665: 0.097,
        670: 0.101,
        675: 0.110,
        680: 0.125,
        685: 0.147,
        690: 0.174,
        695: 0.210,
        700: 0.247,
        705: 0.283,
        710: 0.311,
        715: 0.329,
        720: 0.343,
        725: 0.353,
        730: 0.358,
        735: 0.362,
        740: 0.364,
        745: 0.360,
        750: 0.362,
        755: 0.364,
        760: 0.368,
        765: 0.368,
        770: 0.355,
        775: 0.346,
        780: 0.341
    },
    'blue flower': {
        380: 0.123,
        385: 0.152,
        390: 0.197,
        395: 0.258,
        400: 0.328,
        405: 0.385,
        410: 0.418,
        415: 0.437,
        420: 0.446,
        425: 0.448,
        430: 0.448,
        435: 0.447,
        440: 0.444,
        445: 0.440,
        450: 0.434,
        455: 0.428,
        460: 0.421,
        465: 0.413,
        470: 0.405,
        475: 0.394,
        480: 0.381,
        485: 0.372,
        490: 0.362,
        495: 0.352,
        500: 0.342,
        505: 0.330,
        510: 0.314,
        515: 0.294,
        520: 0.271,
        525: 0.249,
        530: 0.231,
        535: 0.219,
        540: 0.211,
        545: 0.209,
        550: 0.209,
        555: 0.207,
        560: 0.201,
        565: 0.196,
        570: 0.196,
        575: 0.199,
        580: 0.206,
        585: 0.215,
        590: 0.223,
        595: 0.229,
        600: 0.235,
        605: 0.241,
        610: 0.245,
        615: 0.245,
        620: 0.243,
        625: 0.243,
        630: 0.247,
        635: 0.254,
        640: 0.269,
        645: 0.291,
        650: 0.318,
        655: 0.351,
        660: 0.384,
        665: 0.417,
        670: 0.446,
        675: 0.470,
        680: 0.490,
        685: 0.504,
        690: 0.511,
        695: 0.517,
        700: 0.520,
        705: 0.522,
        710: 0.523,
        715: 0.522,
        720: 0.521,
        725: 0.521,
        730: 0.522,
        735: 0.521,
        740: 0.521,
        745: 0.516,
        750: 0.514,
        755: 0.514,
        760: 0.517,
        765: 0.515,
        770: 0.500,
        775: 0.491,
        780: 0.487
    },
    'bluish green': {
        380: 0.110,
        385: 0.133,
        390: 0.167,
        395: 0.208,
        400: 0.252,
        405: 0.284,
        410: 0.303,
        415: 0.314,
        420: 0.322,
        425: 0.329,
        430: 0.336,
        435: 0.344,
        440: 0.353,
        445: 0.363,
        450: 0.375,
        455: 0.390,
        460: 0.408,
        465: 0.433,
        470: 0.460,
        475: 0.492,
        480: 0.523,
        485: 0.548,
        490: 0.566,
        495: 0.577,
        500: 0.582,
        505: 0.583,
        510: 0.580,
        515: 0.576,
        520: 0.569,
        525: 0.560,
        530: 0.549,
        535: 0.535,
        540: 0.519,
        545: 0.501,
        550: 0.480,
        555: 0.458,
        560: 0.436,
        565: 0.414,
        570: 0.392,
        575: 0.369,
        580: 0.346,
        585: 0.324,
        590: 0.302,
        595: 0.279,
        600: 0.260,
        605: 0.245,
        610: 0.234,
        615: 0.226,
        620: 0.221,
        625: 0.217,
        630: 0.215,
        635: 0.212,
        640: 0.210,
        645: 0.209,
        650: 0.208,
        655: 0.209,
        660: 0.211,
        665: 0.215,
        670: 0.220,
        675: 0.227,
        680: 0.233,
        685: 0.239,
        690: 0.244,
        695: 0.249,
        700: 0.252,
        705: 0.252,
        710: 0.250,
        715: 0.248,
        720: 0.244,
        725: 0.245,
        730: 0.245,
        735: 0.251,
        740: 0.260,
        745: 0.269,
        750: 0.278,
        755: 0.288,
        760: 0.297,
        765: 0.301,
        770: 0.297,
        775: 0.296,
        780: 0.296
    },
    'orange': {
        380: 0.053,
        385: 0.054,
        390: 0.054,
        395: 0.054,
        400: 0.054,
        405: 0.054,
        410: 0.053,
        415: 0.053,
        420: 0.052,
        425: 0.052,
        430: 0.052,
        435: 0.052,
        440: 0.052,
        445: 0.052,
        450: 0.052,
        455: 0.052,
        460: 0.052,
        465: 0.052,
        470: 0.053,
        475: 0.054,
        480: 0.055,
        485: 0.056,
        490: 0.057,
        495: 0.059,
        500: 0.061,
        505: 0.064,
        510: 0.068,
        515: 0.076,
        520: 0.086,
        525: 0.101,
        530: 0.120,
        535: 0.143,
        540: 0.170,
        545: 0.198,
        550: 0.228,
        555: 0.260,
        560: 0.297,
        565: 0.338,
        570: 0.380,
        575: 0.418,
        580: 0.452,
        585: 0.481,
        590: 0.503,
        595: 0.520,
        600: 0.532,
        605: 0.543,
        610: 0.552,
        615: 0.560,
        620: 0.566,
        625: 0.572,
        630: 0.578,
        635: 0.583,
        640: 0.587,
        645: 0.593,
        650: 0.599,
        655: 0.602,
        660: 0.604,
        665: 0.606,
        670: 0.608,
        675: 0.611,
        680: 0.615,
        685: 0.619,
        690: 0.622,
        695: 0.625,
        700: 0.628,
        705: 0.630,
        710: 0.633,
        715: 0.633,
        720: 0.633,
        725: 0.636,
        730: 0.637,
        735: 0.639,
        740: 0.638,
        745: 0.633,
        750: 0.633,
        755: 0.636,
        760: 0.641,
        765: 0.639,
        770: 0.616,
        775: 0.598,
        780: 0.582
    },
    'purplish blue': {
        380: 0.099,
        385: 0.120,
        390: 0.150,
        395: 0.189,
        400: 0.231,
        405: 0.268,
        410: 0.293,
        415: 0.311,
        420: 0.324,
        425: 0.335,
        430: 0.348,
        435: 0.361,
        440: 0.373,
        445: 0.383,
        450: 0.387,
        455: 0.383,
        460: 0.374,
        465: 0.361,
        470: 0.345,
        475: 0.325,
        480: 0.301,
        485: 0.275,
        490: 0.247,
        495: 0.223,
        500: 0.202,
        505: 0.184,
        510: 0.167,
        515: 0.152,
        520: 0.137,
        525: 0.125,
        530: 0.116,
        535: 0.110,
        540: 0.106,
        545: 0.103,
        550: 0.099,
        555: 0.094,
        560: 0.090,
        565: 0.086,
        570: 0.083,
        575: 0.083,
        580: 0.083,
        585: 0.085,
        590: 0.086,
        595: 0.087,
        600: 0.087,
        605: 0.086,
        610: 0.085,
        615: 0.084,
        620: 0.084,
        625: 0.085,
        630: 0.088,
        635: 0.092,
        640: 0.098,
        645: 0.105,
        650: 0.111,
        655: 0.118,
        660: 0.123,
        665: 0.126,
        670: 0.126,
        675: 0.124,
        680: 0.120,
        685: 0.117,
        690: 0.115,
        695: 0.115,
        700: 0.116,
        705: 0.118,
        710: 0.120,
        715: 0.124,
        720: 0.128,
        725: 0.133,
        730: 0.139,
        735: 0.149,
        740: 0.162,
        745: 0.178,
        750: 0.197,
        755: 0.219,
        760: 0.242,
        765: 0.259,
        770: 0.275,
        775: 0.294,
        780: 0.316
    },
    'moderate red': {
        380: 0.096,
        385: 0.108,
        390: 0.123,
        395: 0.135,
        400: 0.144,
        405: 0.145,
        410: 0.144,
        415: 0.141,
        420: 0.138,
        425: 0.134,
        430: 0.132,
        435: 0.132,
        440: 0.131,
        445: 0.131,
        450: 0.129,
        455: 0.128,
        460: 0.126,
        465: 0.126,
        470: 0.125,
        475: 0.123,
        480: 0.119,
        485: 0.114,
        490: 0.109,
        495: 0.105,
        500: 0.103,
        505: 0.102,
        510: 0.100,
        515: 0.097,
        520: 0.094,
        525: 0.091,
        530: 0.089,
        535: 0.090,
        540: 0.092,
        545: 0.096,
        550: 0.102,
        555: 0.106,
        560: 0.108,
        565: 0.109,
        570: 0.112,
        575: 0.126,
        580: 0.157,
        585: 0.208,
        590: 0.274,
        595: 0.346,
        600: 0.415,
        605: 0.473,
        610: 0.517,
        615: 0.547,
        620: 0.567,
        625: 0.582,
        630: 0.591,
        635: 0.597,
        640: 0.601,
        645: 0.604,
        650: 0.607,
        655: 0.608,
        660: 0.607,
        665: 0.606,
        670: 0.605,
        675: 0.605,
        680: 0.605,
        685: 0.604,
        690: 0.605,
        695: 0.606,
        700: 0.606,
        705: 0.604,
        710: 0.602,
        715: 0.601,
        720: 0.599,
        725: 0.598,
        730: 0.596,
        735: 0.595,
        740: 0.593,
        745: 0.587,
        750: 0.584,
        755: 0.584,
        760: 0.586,
        765: 0.584,
        770: 0.566,
        775: 0.551,
        780: 0.54
    },
    'purple': {
        380: 0.101,
        385: 0.115,
        390: 0.135,
        395: 0.157,
        400: 0.177,
        405: 0.191,
        410: 0.199,
        415: 0.203,
        420: 0.206,
        425: 0.198,
        430: 0.190,
        435: 0.179,
        440: 0.168,
        445: 0.156,
        450: 0.144,
        455: 0.132,
        460: 0.120,
        465: 0.110,
        470: 0.101,
        475: 0.093,
        480: 0.086,
        485: 0.080,
        490: 0.075,
        495: 0.070,
        500: 0.067,
        505: 0.063,
        510: 0.061,
        515: 0.059,
        520: 0.058,
        525: 0.056,
        530: 0.054,
        535: 0.053,
        540: 0.052,
        545: 0.052,
        550: 0.053,
        555: 0.054,
        560: 0.055,
        565: 0.055,
        570: 0.054,
        575: 0.053,
        580: 0.052,
        585: 0.052,
        590: 0.053,
        595: 0.055,
        600: 0.059,
        605: 0.065,
        610: 0.074,
        615: 0.086,
        620: 0.099,
        625: 0.113,
        630: 0.126,
        635: 0.138,
        640: 0.149,
        645: 0.161,
        650: 0.172,
        655: 0.182,
        660: 0.193,
        665: 0.205,
        670: 0.217,
        675: 0.232,
        680: 0.248,
        685: 0.266,
        690: 0.282,
        695: 0.301,
        700: 0.319,
        705: 0.338,
        710: 0.355,
        715: 0.371,
        720: 0.388,
        725: 0.406,
        730: 0.422,
        735: 0.436,
        740: 0.451,
        745: 0.460,
        750: 0.471,
        755: 0.481,
        760: 0.492,
        765: 0.495,
        770: 0.482,
        775: 0.471,
        780: 0.467
    },
    'yellow green': {
        380: 0.056,
        385: 0.058,
        390: 0.059,
        395: 0.059,
        400: 0.060,
        405: 0.061,
        410: 0.061,
        415: 0.061,
        420: 0.062,
        425: 0.063,
        430: 0.064,
        435: 0.066,
        440: 0.068,
        445: 0.071,
        450: 0.075,
        455: 0.079,
        460: 0.085,
        465: 0.093,
        470: 0.104,
        475: 0.118,
        480: 0.135,
        485: 0.157,
        490: 0.185,
        495: 0.221,
        500: 0.269,
        505: 0.326,
        510: 0.384,
        515: 0.440,
        520: 0.484,
        525: 0.516,
        530: 0.534,
        535: 0.542,
        540: 0.545,
        545: 0.541,
        550: 0.533,
        555: 0.524,
        560: 0.513,
        565: 0.501,
        570: 0.487,
        575: 0.472,
        580: 0.454,
        585: 0.436,
        590: 0.416,
        595: 0.394,
        600: 0.374,
        605: 0.358,
        610: 0.346,
        615: 0.337,
        620: 0.331,
        625: 0.328,
        630: 0.325,
        635: 0.322,
        640: 0.320,
        645: 0.319,
        650: 0.319,
        655: 0.320,
        660: 0.324,
        665: 0.330,
        670: 0.337,
        675: 0.345,
        680: 0.354,
        685: 0.362,
        690: 0.368,
        695: 0.375,
        700: 0.379,
        705: 0.381,
        710: 0.379,
        715: 0.376,
        720: 0.373,
        725: 0.372,
        730: 0.375,
        735: 0.382,
        740: 0.392,
        745: 0.401,
        750: 0.412,
        755: 0.422,
        760: 0.433,
        765: 0.436,
        770: 0.426,
        775: 0.413,
        780: 0.404
    },
    'orange yellow': {
        380: 0.060,
        385: 0.061,
        390: 0.063,
        395: 0.064,
        400: 0.065,
        405: 0.065,
        410: 0.064,
        415: 0.064,
        420: 0.064,
        425: 0.064,
        430: 0.064,
        435: 0.065,
        440: 0.065,
        445: 0.066,
        450: 0.067,
        455: 0.068,
        460: 0.069,
        465: 0.073,
        470: 0.077,
        475: 0.084,
        480: 0.092,
        485: 0.100,
        490: 0.107,
        495: 0.115,
        500: 0.123,
        505: 0.133,
        510: 0.146,
        515: 0.166,
        520: 0.193,
        525: 0.229,
        530: 0.273,
        535: 0.323,
        540: 0.374,
        545: 0.418,
        550: 0.456,
        555: 0.487,
        560: 0.512,
        565: 0.534,
        570: 0.554,
        575: 0.570,
        580: 0.584,
        585: 0.598,
        590: 0.609,
        595: 0.617,
        600: 0.624,
        605: 0.630,
        610: 0.635,
        615: 0.640,
        620: 0.645,
        625: 0.650,
        630: 0.654,
        635: 0.658,
        640: 0.662,
        645: 0.667,
        650: 0.672,
        655: 0.675,
        660: 0.676,
        665: 0.677,
        670: 0.678,
        675: 0.681,
        680: 0.685,
        685: 0.688,
        690: 0.690,
        695: 0.693,
        700: 0.696,
        705: 0.698,
        710: 0.698,
        715: 0.698,
        720: 0.698,
        725: 0.700,
        730: 0.701,
        735: 0.701,
        740: 0.701,
        745: 0.695,
        750: 0.694,
        755: 0.696,
        760: 0.700,
        765: 0.698,
        770: 0.673,
        775: 0.653,
        780: 0.639
    },
    'blue': {
        380: 0.069,
        385: 0.081,
        390: 0.096,
        395: 0.114,
        400: 0.136,
        405: 0.156,
        410: 0.175,
        415: 0.193,
        420: 0.208,
        425: 0.224,
        430: 0.244,
        435: 0.265,
        440: 0.290,
        445: 0.316,
        450: 0.335,
        455: 0.342,
        460: 0.338,
        465: 0.324,
        470: 0.302,
        475: 0.273,
        480: 0.239,
        485: 0.205,
        490: 0.172,
        495: 0.144,
        500: 0.120,
        505: 0.101,
        510: 0.086,
        515: 0.074,
        520: 0.066,
        525: 0.059,
        530: 0.054,
        535: 0.051,
        540: 0.048,
        545: 0.046,
        550: 0.045,
        555: 0.044,
        560: 0.043,
        565: 0.042,
        570: 0.041,
        575: 0.041,
        580: 0.040,
        585: 0.040,
        590: 0.040,
        595: 0.040,
        600: 0.039,
        605: 0.039,
        610: 0.040,
        615: 0.040,
        620: 0.040,
        625: 0.040,
        630: 0.041,
        635: 0.041,
        640: 0.042,
        645: 0.042,
        650: 0.042,
        655: 0.043,
        660: 0.043,
        665: 0.043,
        670: 0.044,
        675: 0.044,
        680: 0.044,
        685: 0.044,
        690: 0.045,
        695: 0.046,
        700: 0.048,
        705: 0.050,
        710: 0.051,
        715: 0.053,
        720: 0.056,
        725: 0.060,
        730: 0.064,
        735: 0.070,
        740: 0.079,
        745: 0.091,
        750: 0.104,
        755: 0.120,
        760: 0.138,
        765: 0.154,
        770: 0.168,
        775: 0.186,
        780: 0.204
    },
    'green': {
        380: 0.055,
        385: 0.056,
        390: 0.057,
        395: 0.058,
        400: 0.058,
        405: 0.058,
        410: 0.059,
        415: 0.059,
        420: 0.059,
        425: 0.060,
        430: 0.062,
        435: 0.063,
        440: 0.065,
        445: 0.067,
        450: 0.070,
        455: 0.074,
        460: 0.078,
        465: 0.084,
        470: 0.091,
        475: 0.101,
        480: 0.113,
        485: 0.125,
        490: 0.140,
        495: 0.157,
        500: 0.180,
        505: 0.208,
        510: 0.244,
        515: 0.286,
        520: 0.324,
        525: 0.351,
        530: 0.363,
        535: 0.363,
        540: 0.355,
        545: 0.342,
        550: 0.323,
        555: 0.303,
        560: 0.281,
        565: 0.260,
        570: 0.238,
        575: 0.217,
        580: 0.196,
        585: 0.177,
        590: 0.158,
        595: 0.140,
        600: 0.124,
        605: 0.111,
        610: 0.101,
        615: 0.094,
        620: 0.089,
        625: 0.086,
        630: 0.084,
        635: 0.082,
        640: 0.080,
        645: 0.078,
        650: 0.077,
        655: 0.076,
        660: 0.075,
        665: 0.075,
        670: 0.075,
        675: 0.077,
        680: 0.078,
        685: 0.080,
        690: 0.082,
        695: 0.085,
        700: 0.088,
        705: 0.089,
        710: 0.089,
        715: 0.090,
        720: 0.090,
        725: 0.090,
        730: 0.089,
        735: 0.092,
        740: 0.094,
        745: 0.097,
        750: 0.102,
        755: 0.106,
        760: 0.110,
        765: 0.111,
        770: 0.112,
        775: 0.112,
        780: 0.112
    },
    'red': {
        380: 0.052,
        385: 0.052,
        390: 0.052,
        395: 0.052,
        400: 0.051,
        405: 0.051,
        410: 0.050,
        415: 0.050,
        420: 0.049,
        425: 0.049,
        430: 0.049,
        435: 0.049,
        440: 0.049,
        445: 0.049,
        450: 0.049,
        455: 0.048,
        460: 0.048,
        465: 0.047,
        470: 0.047,
        475: 0.046,
        480: 0.045,
        485: 0.045,
        490: 0.044,
        495: 0.044,
        500: 0.044,
        505: 0.044,
        510: 0.044,
        515: 0.044,
        520: 0.044,
        525: 0.044,
        530: 0.044,
        535: 0.044,
        540: 0.045,
        545: 0.046,
        550: 0.047,
        555: 0.048,
        560: 0.050,
        565: 0.053,
        570: 0.057,
        575: 0.063,
        580: 0.072,
        585: 0.086,
        590: 0.109,
        595: 0.143,
        600: 0.192,
        605: 0.256,
        610: 0.332,
        615: 0.413,
        620: 0.486,
        625: 0.550,
        630: 0.598,
        635: 0.631,
        640: 0.654,
        645: 0.672,
        650: 0.686,
        655: 0.694,
        660: 0.700,
        665: 0.704,
        670: 0.707,
        675: 0.712,
        680: 0.718,
        685: 0.721,
        690: 0.724,
        695: 0.727,
        700: 0.729,
        705: 0.730,
        710: 0.730,
        715: 0.729,
        720: 0.727,
        725: 0.728,
        730: 0.729,
        735: 0.729,
        740: 0.727,
        745: 0.723,
        750: 0.721,
        755: 0.724,
        760: 0.728,
        765: 0.727,
        770: 0.702,
        775: 0.680,
        780: 0.664
    },
    'yellow': {
        380: 0.054,
        385: 0.053,
        390: 0.054,
        395: 0.053,
        400: 0.053,
        405: 0.053,
        410: 0.053,
        415: 0.052,
        420: 0.052,
        425: 0.052,
        430: 0.053,
        435: 0.053,
        440: 0.053,
        445: 0.054,
        450: 0.055,
        455: 0.056,
        460: 0.059,
        465: 0.065,
        470: 0.075,
        475: 0.093,
        480: 0.121,
        485: 0.157,
        490: 0.202,
        495: 0.252,
        500: 0.303,
        505: 0.351,
        510: 0.394,
        515: 0.436,
        520: 0.475,
        525: 0.512,
        530: 0.544,
        535: 0.572,
        540: 0.597,
        545: 0.615,
        550: 0.630,
        555: 0.645,
        560: 0.660,
        565: 0.673,
        570: 0.686,
        575: 0.698,
        580: 0.708,
        585: 0.718,
        590: 0.726,
        595: 0.732,
        600: 0.737,
        605: 0.742,
        610: 0.746,
        615: 0.749,
        620: 0.753,
        625: 0.757,
        630: 0.761,
        635: 0.765,
        640: 0.768,
        645: 0.772,
        650: 0.777,
        655: 0.779,
        660: 0.780,
        665: 0.780,
        670: 0.781,
        675: 0.782,
        680: 0.785,
        685: 0.785,
        690: 0.787,
        695: 0.789,
        700: 0.792,
        705: 0.792,
        710: 0.793,
        715: 0.792,
        720: 0.790,
        725: 0.792,
        730: 0.792,
        735: 0.790,
        740: 0.787,
        745: 0.782,
        750: 0.778,
        755: 0.780,
        760: 0.782,
        765: 0.781,
        770: 0.752,
        775: 0.728,
        780: 0.71
    },
    'magenta': {
        380: 0.118,
        385: 0.142,
        390: 0.179,
        395: 0.228,
        400: 0.283,
        405: 0.322,
        410: 0.343,
        415: 0.354,
        420: 0.359,
        425: 0.357,
        430: 0.350,
        435: 0.339,
        440: 0.327,
        445: 0.313,
        450: 0.298,
        455: 0.282,
        460: 0.267,
        465: 0.253,
        470: 0.239,
        475: 0.225,
        480: 0.209,
        485: 0.195,
        490: 0.182,
        495: 0.172,
        500: 0.163,
        505: 0.155,
        510: 0.146,
        515: 0.135,
        520: 0.124,
        525: 0.113,
        530: 0.106,
        535: 0.102,
        540: 0.102,
        545: 0.105,
        550: 0.107,
        555: 0.107,
        560: 0.106,
        565: 0.107,
        570: 0.112,
        575: 0.123,
        580: 0.141,
        585: 0.166,
        590: 0.198,
        595: 0.235,
        600: 0.279,
        605: 0.333,
        610: 0.394,
        615: 0.460,
        620: 0.522,
        625: 0.580,
        630: 0.628,
        635: 0.666,
        640: 0.696,
        645: 0.722,
        650: 0.742,
        655: 0.756,
        660: 0.766,
        665: 0.774,
        670: 0.780,
        675: 0.785,
        680: 0.791,
        685: 0.794,
        690: 0.798,
        695: 0.801,
        700: 0.804,
        705: 0.806,
        710: 0.807,
        715: 0.807,
        720: 0.807,
        725: 0.810,
        730: 0.813,
        735: 0.814,
        740: 0.813,
        745: 0.810,
        750: 0.808,
        755: 0.811,
        760: 0.814,
        765: 0.813,
        770: 0.785,
        775: 0.765,
        780: 0.752
    },
    'cyan': {
        380: 0.093,
        385: 0.110,
        390: 0.134,
        395: 0.164,
        400: 0.195,
        405: 0.220,
        410: 0.238,
        415: 0.249,
        420: 0.258,
        425: 0.270,
        430: 0.281,
        435: 0.296,
        440: 0.315,
        445: 0.334,
        450: 0.352,
        455: 0.370,
        460: 0.391,
        465: 0.414,
        470: 0.434,
        475: 0.449,
        480: 0.458,
        485: 0.461,
        490: 0.457,
        495: 0.447,
        500: 0.433,
        505: 0.414,
        510: 0.392,
        515: 0.366,
        520: 0.339,
        525: 0.310,
        530: 0.282,
        535: 0.255,
        540: 0.228,
        545: 0.204,
        550: 0.180,
        555: 0.159,
        560: 0.141,
        565: 0.126,
        570: 0.114,
        575: 0.104,
        580: 0.097,
        585: 0.092,
        590: 0.088,
        595: 0.083,
        600: 0.080,
        605: 0.077,
        610: 0.075,
        615: 0.074,
        620: 0.073,
        625: 0.073,
        630: 0.073,
        635: 0.073,
        640: 0.073,
        645: 0.073,
        650: 0.074,
        655: 0.075,
        660: 0.076,
        665: 0.076,
        670: 0.077,
        675: 0.076,
        680: 0.075,
        685: 0.074,
        690: 0.074,
        695: 0.073,
        700: 0.072,
        705: 0.072,
        710: 0.071,
        715: 0.073,
        720: 0.075,
        725: 0.078,
        730: 0.082,
        735: 0.090,
        740: 0.100,
        745: 0.116,
        750: 0.133,
        755: 0.154,
        760: 0.176,
        765: 0.191,
        770: 0.200,
        775: 0.208,
        780: 0.214
    },
    'white 9.5 (.05 D)': {
        380: 0.153,
        385: 0.189,
        390: 0.245,
        395: 0.319,
        400: 0.409,
        405: 0.536,
        410: 0.671,
        415: 0.772,
        420: 0.840,
        425: 0.868,
        430: 0.878,
        435: 0.882,
        440: 0.883,
        445: 0.885,
        450: 0.886,
        455: 0.886,
        460: 0.887,
        465: 0.888,
        470: 0.888,
        475: 0.888,
        480: 0.888,
        485: 0.888,
        490: 0.888,
        495: 0.888,
        500: 0.887,
        505: 0.887,
        510: 0.887,
        515: 0.887,
        520: 0.887,
        525: 0.887,
        530: 0.887,
        535: 0.887,
        540: 0.887,
        545: 0.886,
        550: 0.886,
        555: 0.887,
        560: 0.887,
        565: 0.887,
        570: 0.888,
        575: 0.888,
        580: 0.887,
        585: 0.886,
        590: 0.886,
        595: 0.886,
        600: 0.887,
        605: 0.888,
        610: 0.889,
        615: 0.890,
        620: 0.891,
        625: 0.891,
        630: 0.891,
        635: 0.891,
        640: 0.890,
        645: 0.889,
        650: 0.889,
        655: 0.889,
        660: 0.889,
        665: 0.889,
        670: 0.888,
        675: 0.888,
        680: 0.888,
        685: 0.888,
        690: 0.888,
        695: 0.888,
        700: 0.888,
        705: 0.887,
        710: 0.886,
        715: 0.886,
        720: 0.886,
        725: 0.885,
        730: 0.885,
        735: 0.885,
        740: 0.884,
        745: 0.884,
        750: 0.883,
        755: 0.882,
        760: 0.882,
        765: 0.881,
        770: 0.880,
        775: 0.880,
        780: 0.879
    },
    'neutral 8 (.23 D)': {
        380: 0.150,
        385: 0.184,
        390: 0.235,
        395: 0.299,
        400: 0.372,
        405: 0.459,
        410: 0.529,
        415: 0.564,
        420: 0.580,
        425: 0.584,
        430: 0.585,
        435: 0.587,
        440: 0.587,
        445: 0.588,
        450: 0.588,
        455: 0.587,
        460: 0.586,
        465: 0.585,
        470: 0.583,
        475: 0.582,
        480: 0.581,
        485: 0.580,
        490: 0.580,
        495: 0.580,
        500: 0.580,
        505: 0.580,
        510: 0.580,
        515: 0.581,
        520: 0.581,
        525: 0.582,
        530: 0.582,
        535: 0.582,
        540: 0.583,
        545: 0.583,
        550: 0.583,
        555: 0.584,
        560: 0.584,
        565: 0.585,
        570: 0.586,
        575: 0.587,
        580: 0.588,
        585: 0.588,
        590: 0.588,
        595: 0.588,
        600: 0.588,
        605: 0.587,
        610: 0.586,
        615: 0.586,
        620: 0.585,
        625: 0.584,
        630: 0.583,
        635: 0.581,
        640: 0.580,
        645: 0.579,
        650: 0.578,
        655: 0.577,
        660: 0.576,
        665: 0.575,
        670: 0.574,
        675: 0.573,
        680: 0.572,
        685: 0.571,
        690: 0.570,
        695: 0.569,
        700: 0.568,
        705: 0.567,
        710: 0.566,
        715: 0.565,
        720: 0.564,
        725: 0.562,
        730: 0.562,
        735: 0.560,
        740: 0.560,
        745: 0.558,
        750: 0.557,
        755: 0.556,
        760: 0.555,
        765: 0.554,
        770: 0.553,
        775: 0.551,
        780: 0.55
    },
    'neutral 6.5 (.44 D)': {
        380: 0.138,
        385: 0.167,
        390: 0.206,
        395: 0.249,
        400: 0.289,
        405: 0.324,
        410: 0.346,
        415: 0.354,
        420: 0.357,
        425: 0.358,
        430: 0.359,
        435: 0.360,
        440: 0.361,
        445: 0.362,
        450: 0.362,
        455: 0.361,
        460: 0.361,
        465: 0.359,
        470: 0.358,
        475: 0.358,
        480: 0.357,
        485: 0.356,
        490: 0.356,
        495: 0.356,
        500: 0.356,
        505: 0.356,
        510: 0.356,
        515: 0.356,
        520: 0.357,
        525: 0.357,
        530: 0.357,
        535: 0.358,
        540: 0.358,
        545: 0.358,
        550: 0.358,
        555: 0.358,
        560: 0.359,
        565: 0.359,
        570: 0.360,
        575: 0.361,
        580: 0.361,
        585: 0.361,
        590: 0.361,
        595: 0.361,
        600: 0.360,
        605: 0.360,
        610: 0.359,
        615: 0.358,
        620: 0.357,
        625: 0.356,
        630: 0.355,
        635: 0.354,
        640: 0.353,
        645: 0.352,
        650: 0.351,
        655: 0.350,
        660: 0.349,
        665: 0.348,
        670: 0.346,
        675: 0.346,
        680: 0.345,
        685: 0.344,
        690: 0.343,
        695: 0.342,
        700: 0.341,
        705: 0.340,
        710: 0.339,
        715: 0.338,
        720: 0.337,
        725: 0.336,
        730: 0.335,
        735: 0.334,
        740: 0.333,
        745: 0.332,
        750: 0.331,
        755: 0.330,
        760: 0.329,
        765: 0.328,
        770: 0.327,
        775: 0.326,
        780: 0.325
    },
    'neutral 5 (.70 D)': {
        380: 0.113,
        385: 0.131,
        390: 0.150,
        395: 0.169,
        400: 0.183,
        405: 0.193,
        410: 0.199,
        415: 0.201,
        420: 0.202,
        425: 0.203,
        430: 0.203,
        435: 0.204,
        440: 0.205,
        445: 0.205,
        450: 0.205,
        455: 0.205,
        460: 0.204,
        465: 0.204,
        470: 0.203,
        475: 0.203,
        480: 0.202,
        485: 0.202,
        490: 0.202,
        495: 0.202,
        500: 0.202,
        505: 0.202,
        510: 0.202,
        515: 0.202,
        520: 0.202,
        525: 0.202,
        530: 0.203,
        535: 0.203,
        540: 0.203,
        545: 0.203,
        550: 0.203,
        555: 0.203,
        560: 0.203,
        565: 0.203,
        570: 0.204,
        575: 0.204,
        580: 0.205,
        585: 0.205,
        590: 0.205,
        595: 0.205,
        600: 0.204,
        605: 0.204,
        610: 0.204,
        615: 0.203,
        620: 0.203,
        625: 0.202,
        630: 0.201,
        635: 0.201,
        640: 0.200,
        645: 0.199,
        650: 0.198,
        655: 0.198,
        660: 0.197,
        665: 0.197,
        670: 0.196,
        675: 0.195,
        680: 0.195,
        685: 0.194,
        690: 0.194,
        695: 0.193,
        700: 0.192,
        705: 0.192,
        710: 0.191,
        715: 0.191,
        720: 0.190,
        725: 0.189,
        730: 0.189,
        735: 0.188,
        740: 0.188,
        745: 0.187,
        750: 0.187,
        755: 0.186,
        760: 0.185,
        765: 0.185,
        770: 0.184,
        775: 0.184,
        780: 0.183
    },
    'neutral 3.5 (1.05 D)': {
        380: 0.074,
        385: 0.079,
        390: 0.084,
        395: 0.088,
        400: 0.091,
        405: 0.093,
        410: 0.094,
        415: 0.094,
        420: 0.094,
        425: 0.094,
        430: 0.094,
        435: 0.095,
        440: 0.095,
        445: 0.095,
        450: 0.095,
        455: 0.094,
        460: 0.094,
        465: 0.094,
        470: 0.094,
        475: 0.093,
        480: 0.093,
        485: 0.093,
        490: 0.093,
        495: 0.092,
        500: 0.092,
        505: 0.093,
        510: 0.093,
        515: 0.093,
        520: 0.093,
        525: 0.093,
        530: 0.093,
        535: 0.093,
        540: 0.093,
        545: 0.093,
        550: 0.093,
        555: 0.092,
        560: 0.093,
        565: 0.093,
        570: 0.093,
        575: 0.093,
        580: 0.093,
        585: 0.093,
        590: 0.093,
        595: 0.092,
        600: 0.092,
        605: 0.092,
        610: 0.092,
        615: 0.091,
        620: 0.091,
        625: 0.091,
        630: 0.090,
        635: 0.090,
        640: 0.090,
        645: 0.090,
        650: 0.089,
        655: 0.089,
        660: 0.089,
        665: 0.088,
        670: 0.088,
        675: 0.088,
        680: 0.087,
        685: 0.087,
        690: 0.087,
        695: 0.087,
        700: 0.086,
        705: 0.086,
        710: 0.086,
        715: 0.086,
        720: 0.085,
        725: 0.085,
        730: 0.085,
        735: 0.085,
        740: 0.085,
        745: 0.084,
        750: 0.084,
        755: 0.084,
        760: 0.084,
        765: 0.084,
        770: 0.083,
        775: 0.083,
        780: 0.083
    },
    'black 2 (1.5 D)': {
        380: 0.032,
        385: 0.033,
        390: 0.033,
        395: 0.034,
        400: 0.035,
        405: 0.035,
        410: 0.036,
        415: 0.036,
        420: 0.036,
        425: 0.036,
        430: 0.036,
        435: 0.036,
        440: 0.035,
        445: 0.035,
        450: 0.035,
        455: 0.035,
        460: 0.035,
        465: 0.035,
        470: 0.035,
        475: 0.035,
        480: 0.034,
        485: 0.034,
        490: 0.034,
        495: 0.034,
        500: 0.034,
        505: 0.034,
        510: 0.034,
        515: 0.034,
        520: 0.034,
        525: 0.034,
        530: 0.034,
        535: 0.034,
        540: 0.034,
        545: 0.034,
        550: 0.034,
        555: 0.034,
        560: 0.033,
        565: 0.033,
        570: 0.033,
        575: 0.033,
        580: 0.033,
        585: 0.033,
        590: 0.033,
        595: 0.033,
        600: 0.033,
        605: 0.033,
        610: 0.033,
        615: 0.033,
        620: 0.033,
        625: 0.033,
        630: 0.033,
        635: 0.033,
        640: 0.033,
        645: 0.033,
        650: 0.033,
        655: 0.033,
        660: 0.033,
        665: 0.033,
        670: 0.033,
        675: 0.033,
        680: 0.033,
        685: 0.033,
        690: 0.032,
        695: 0.032,
        700: 0.032,
        705: 0.032,
        710: 0.032,
        715: 0.032,
        720: 0.032,
        725: 0.032,
        730: 0.032,
        735: 0.032,
        740: 0.032,
        745: 0.032,
        750: 0.032,
        755: 0.032,
        760: 0.032,
        765: 0.032,
        770: 0.032,
        775: 0.032,
        780: 0.032
    }
}

COLORCHECKER_N_OHTA_SPDS = dict(
    (key, SpectralPowerDistribution(key, value))
    for key, value in COLORCHECKER_N_OHTA_SPDS_DATA.items())
"""
Measured by *Ohta (1997)*.

COLORCHECKER_N_OHTA_SPDS : dict
"""

BABELCOLOR_AVERAGE_SPDS_DATA = {
    'dark skin': {
        380: 0.055,
        390: 0.058,
        400: 0.061,
        410: 0.062,
        420: 0.062,
        430: 0.062,
        440: 0.062,
        450: 0.062,
        460: 0.062,
        470: 0.062,
        480: 0.062,
        490: 0.063,
        500: 0.065,
        510: 0.070,
        520: 0.076,
        530: 0.079,
        540: 0.081,
        550: 0.084,
        560: 0.091,
        570: 0.103,
        580: 0.119,
        590: 0.134,
        600: 0.143,
        610: 0.147,
        620: 0.151,
        630: 0.158,
        640: 0.168,
        650: 0.179,
        660: 0.188,
        670: 0.190,
        680: 0.186,
        690: 0.181,
        700: 0.182,
        710: 0.187,
        720: 0.196,
        730: 0.209
    },
    'light skin': {
        380: 0.117,
        390: 0.143,
        400: 0.175,
        410: 0.191,
        420: 0.196,
        430: 0.199,
        440: 0.204,
        450: 0.213,
        460: 0.228,
        470: 0.251,
        480: 0.280,
        490: 0.309,
        500: 0.329,
        510: 0.333,
        520: 0.315,
        530: 0.286,
        540: 0.273,
        550: 0.276,
        560: 0.277,
        570: 0.289,
        580: 0.339,
        590: 0.420,
        600: 0.488,
        610: 0.525,
        620: 0.546,
        630: 0.562,
        640: 0.578,
        650: 0.595,
        660: 0.612,
        670: 0.625,
        680: 0.638,
        690: 0.656,
        700: 0.678,
        710: 0.700,
        720: 0.717,
        730: 0.734
    },
    'blue sky': {
        380: 0.130,
        390: 0.177,
        400: 0.251,
        410: 0.306,
        420: 0.324,
        430: 0.330,
        440: 0.333,
        450: 0.331,
        460: 0.323,
        470: 0.311,
        480: 0.298,
        490: 0.285,
        500: 0.269,
        510: 0.250,
        520: 0.231,
        530: 0.214,
        540: 0.199,
        550: 0.185,
        560: 0.169,
        570: 0.157,
        580: 0.149,
        590: 0.145,
        600: 0.142,
        610: 0.141,
        620: 0.141,
        630: 0.141,
        640: 0.143,
        650: 0.147,
        660: 0.152,
        670: 0.154,
        680: 0.150,
        690: 0.144,
        700: 0.136,
        710: 0.132,
        720: 0.135,
        730: 0.147
    },
    'foliage': {
        380: 0.051,
        390: 0.054,
        400: 0.056,
        410: 0.057,
        420: 0.058,
        430: 0.059,
        440: 0.060,
        450: 0.061,
        460: 0.062,
        470: 0.063,
        480: 0.065,
        490: 0.067,
        500: 0.075,
        510: 0.101,
        520: 0.145,
        530: 0.178,
        540: 0.184,
        550: 0.170,
        560: 0.149,
        570: 0.133,
        580: 0.122,
        590: 0.115,
        600: 0.109,
        610: 0.105,
        620: 0.104,
        630: 0.106,
        640: 0.109,
        650: 0.112,
        660: 0.114,
        670: 0.114,
        680: 0.112,
        690: 0.112,
        700: 0.115,
        710: 0.120,
        720: 0.125,
        730: 0.13
    },
    'blue flower': {
        380: 0.144,
        390: 0.198,
        400: 0.294,
        410: 0.375,
        420: 0.408,
        430: 0.421,
        440: 0.426,
        450: 0.426,
        460: 0.419,
        470: 0.403,
        480: 0.379,
        490: 0.346,
        500: 0.311,
        510: 0.281,
        520: 0.254,
        530: 0.229,
        540: 0.214,
        550: 0.208,
        560: 0.202,
        570: 0.194,
        580: 0.193,
        590: 0.200,
        600: 0.214,
        610: 0.230,
        620: 0.241,
        630: 0.254,
        640: 0.279,
        650: 0.313,
        660: 0.348,
        670: 0.366,
        680: 0.366,
        690: 0.359,
        700: 0.358,
        710: 0.365,
        720: 0.377,
        730: 0.398
    },
    'bluish green': {
        380: 0.136,
        390: 0.179,
        400: 0.247,
        410: 0.297,
        420: 0.320,
        430: 0.337,
        440: 0.355,
        450: 0.381,
        460: 0.419,
        470: 0.466,
        480: 0.510,
        490: 0.546,
        500: 0.567,
        510: 0.574,
        520: 0.569,
        530: 0.551,
        540: 0.524,
        550: 0.488,
        560: 0.445,
        570: 0.400,
        580: 0.350,
        590: 0.299,
        600: 0.252,
        610: 0.221,
        620: 0.204,
        630: 0.196,
        640: 0.191,
        650: 0.188,
        660: 0.191,
        670: 0.199,
        680: 0.212,
        690: 0.223,
        700: 0.232,
        710: 0.233,
        720: 0.229,
        730: 0.229
    },
    'orange': {
        380: 0.054,
        390: 0.054,
        400: 0.053,
        410: 0.054,
        420: 0.054,
        430: 0.055,
        440: 0.055,
        450: 0.055,
        460: 0.056,
        470: 0.057,
        480: 0.058,
        490: 0.061,
        500: 0.068,
        510: 0.089,
        520: 0.125,
        530: 0.154,
        540: 0.174,
        550: 0.199,
        560: 0.248,
        570: 0.335,
        580: 0.444,
        590: 0.538,
        600: 0.587,
        610: 0.595,
        620: 0.591,
        630: 0.587,
        640: 0.584,
        650: 0.584,
        660: 0.590,
        670: 0.603,
        680: 0.620,
        690: 0.639,
        700: 0.655,
        710: 0.663,
        720: 0.663,
        730: 0.667
    },
    'purplish blue': {
        380: 0.122,
        390: 0.164,
        400: 0.229,
        410: 0.286,
        420: 0.327,
        430: 0.361,
        440: 0.388,
        450: 0.400,
        460: 0.392,
        470: 0.362,
        480: 0.316,
        490: 0.260,
        500: 0.209,
        510: 0.168,
        520: 0.138,
        530: 0.117,
        540: 0.104,
        550: 0.096,
        560: 0.090,
        570: 0.086,
        580: 0.084,
        590: 0.084,
        600: 0.084,
        610: 0.084,
        620: 0.084,
        630: 0.085,
        640: 0.090,
        650: 0.098,
        660: 0.109,
        670: 0.123,
        680: 0.143,
        690: 0.169,
        700: 0.205,
        710: 0.244,
        720: 0.287,
        730: 0.332
    },
    'moderate red': {
        380: 0.096,
        390: 0.115,
        400: 0.131,
        410: 0.135,
        420: 0.133,
        430: 0.132,
        440: 0.130,
        450: 0.128,
        460: 0.125,
        470: 0.120,
        480: 0.115,
        490: 0.110,
        500: 0.105,
        510: 0.100,
        520: 0.095,
        530: 0.093,
        540: 0.092,
        550: 0.093,
        560: 0.096,
        570: 0.108,
        580: 0.156,
        590: 0.265,
        600: 0.399,
        610: 0.500,
        620: 0.556,
        630: 0.579,
        640: 0.588,
        650: 0.591,
        660: 0.593,
        670: 0.594,
        680: 0.598,
        690: 0.602,
        700: 0.607,
        710: 0.609,
        720: 0.609,
        730: 0.61
    },
    'purple': {
        380: 0.092,
        390: 0.116,
        400: 0.146,
        410: 0.169,
        420: 0.178,
        430: 0.173,
        440: 0.158,
        450: 0.139,
        460: 0.119,
        470: 0.101,
        480: 0.087,
        490: 0.075,
        500: 0.066,
        510: 0.060,
        520: 0.056,
        530: 0.053,
        540: 0.051,
        550: 0.051,
        560: 0.052,
        570: 0.052,
        580: 0.051,
        590: 0.052,
        600: 0.058,
        610: 0.073,
        620: 0.096,
        630: 0.119,
        640: 0.141,
        650: 0.166,
        660: 0.194,
        670: 0.227,
        680: 0.265,
        690: 0.309,
        700: 0.355,
        710: 0.396,
        720: 0.436,
        730: 0.478
    },
    'yellow green': {
        380: 0.061,
        390: 0.061,
        400: 0.062,
        410: 0.063,
        420: 0.064,
        430: 0.066,
        440: 0.069,
        450: 0.075,
        460: 0.085,
        470: 0.105,
        480: 0.139,
        490: 0.192,
        500: 0.271,
        510: 0.376,
        520: 0.476,
        530: 0.531,
        540: 0.549,
        550: 0.546,
        560: 0.528,
        570: 0.504,
        580: 0.471,
        590: 0.428,
        600: 0.381,
        610: 0.347,
        620: 0.327,
        630: 0.318,
        640: 0.312,
        650: 0.310,
        660: 0.314,
        670: 0.327,
        680: 0.345,
        690: 0.363,
        700: 0.376,
        710: 0.381,
        720: 0.378,
        730: 0.379
    },
    'orange yellow': {
        380: 0.063,
        390: 0.063,
        400: 0.063,
        410: 0.064,
        420: 0.064,
        430: 0.064,
        440: 0.065,
        450: 0.066,
        460: 0.067,
        470: 0.068,
        480: 0.071,
        490: 0.076,
        500: 0.087,
        510: 0.125,
        520: 0.206,
        530: 0.305,
        540: 0.383,
        550: 0.431,
        560: 0.469,
        570: 0.518,
        580: 0.568,
        590: 0.607,
        600: 0.628,
        610: 0.637,
        620: 0.640,
        630: 0.642,
        640: 0.645,
        650: 0.648,
        660: 0.651,
        670: 0.653,
        680: 0.657,
        690: 0.664,
        700: 0.673,
        710: 0.680,
        720: 0.684,
        730: 0.688
    },
    'blue': {
        380: 0.066,
        390: 0.079,
        400: 0.102,
        410: 0.146,
        420: 0.200,
        430: 0.244,
        440: 0.282,
        450: 0.309,
        460: 0.308,
        470: 0.278,
        480: 0.231,
        490: 0.178,
        500: 0.130,
        510: 0.094,
        520: 0.070,
        530: 0.054,
        540: 0.046,
        550: 0.042,
        560: 0.039,
        570: 0.038,
        580: 0.038,
        590: 0.038,
        600: 0.038,
        610: 0.039,
        620: 0.039,
        630: 0.040,
        640: 0.041,
        650: 0.042,
        660: 0.044,
        670: 0.045,
        680: 0.046,
        690: 0.046,
        700: 0.048,
        710: 0.052,
        720: 0.057,
        730: 0.065
    },
    'green': {
        380: 0.052,
        390: 0.053,
        400: 0.054,
        410: 0.055,
        420: 0.057,
        430: 0.059,
        440: 0.061,
        450: 0.066,
        460: 0.075,
        470: 0.093,
        480: 0.125,
        490: 0.178,
        500: 0.246,
        510: 0.307,
        520: 0.337,
        530: 0.334,
        540: 0.317,
        550: 0.293,
        560: 0.262,
        570: 0.230,
        580: 0.198,
        590: 0.165,
        600: 0.135,
        610: 0.115,
        620: 0.104,
        630: 0.098,
        640: 0.094,
        650: 0.092,
        660: 0.093,
        670: 0.097,
        680: 0.102,
        690: 0.108,
        700: 0.113,
        710: 0.115,
        720: 0.114,
        730: 0.114
    },
    'red': {
        380: 0.050,
        390: 0.049,
        400: 0.048,
        410: 0.047,
        420: 0.047,
        430: 0.047,
        440: 0.047,
        450: 0.047,
        460: 0.046,
        470: 0.045,
        480: 0.044,
        490: 0.044,
        500: 0.045,
        510: 0.046,
        520: 0.047,
        530: 0.048,
        540: 0.049,
        550: 0.050,
        560: 0.054,
        570: 0.060,
        580: 0.072,
        590: 0.104,
        600: 0.178,
        610: 0.312,
        620: 0.467,
        630: 0.581,
        640: 0.644,
        650: 0.675,
        660: 0.690,
        670: 0.698,
        680: 0.706,
        690: 0.715,
        700: 0.724,
        710: 0.730,
        720: 0.734,
        730: 0.738
    },
    'yellow': {
        380: 0.058,
        390: 0.054,
        400: 0.052,
        410: 0.052,
        420: 0.053,
        430: 0.054,
        440: 0.056,
        450: 0.059,
        460: 0.067,
        470: 0.081,
        480: 0.107,
        490: 0.152,
        500: 0.225,
        510: 0.336,
        520: 0.462,
        530: 0.559,
        540: 0.616,
        550: 0.650,
        560: 0.672,
        570: 0.694,
        580: 0.710,
        590: 0.723,
        600: 0.731,
        610: 0.739,
        620: 0.746,
        630: 0.752,
        640: 0.758,
        650: 0.764,
        660: 0.769,
        670: 0.771,
        680: 0.776,
        690: 0.782,
        700: 0.790,
        710: 0.796,
        720: 0.799,
        730: 0.804
    },
    'magenta': {
        380: 0.145,
        390: 0.195,
        400: 0.283,
        410: 0.346,
        420: 0.362,
        430: 0.354,
        440: 0.334,
        450: 0.306,
        460: 0.276,
        470: 0.248,
        480: 0.218,
        490: 0.190,
        500: 0.168,
        510: 0.149,
        520: 0.127,
        530: 0.107,
        540: 0.100,
        550: 0.102,
        560: 0.104,
        570: 0.109,
        580: 0.137,
        590: 0.200,
        600: 0.290,
        610: 0.400,
        620: 0.516,
        630: 0.615,
        640: 0.687,
        650: 0.732,
        660: 0.760,
        670: 0.774,
        680: 0.783,
        690: 0.793,
        700: 0.803,
        710: 0.812,
        720: 0.817,
        730: 0.825
    },
    'cyan': {
        380: 0.108,
        390: 0.141,
        400: 0.192,
        410: 0.236,
        420: 0.261,
        430: 0.286,
        440: 0.317,
        450: 0.353,
        460: 0.390,
        470: 0.426,
        480: 0.446,
        490: 0.444,
        500: 0.423,
        510: 0.385,
        520: 0.337,
        530: 0.283,
        540: 0.231,
        550: 0.185,
        560: 0.146,
        570: 0.118,
        580: 0.101,
        590: 0.090,
        600: 0.082,
        610: 0.076,
        620: 0.074,
        630: 0.073,
        640: 0.073,
        650: 0.074,
        660: 0.076,
        670: 0.077,
        680: 0.076,
        690: 0.075,
        700: 0.073,
        710: 0.072,
        720: 0.074,
        730: 0.079
    },
    'white 9.5 (.05 D)': {
        380: 0.189,
        390: 0.255,
        400: 0.423,
        410: 0.660,
        420: 0.811,
        430: 0.862,
        440: 0.877,
        450: 0.884,
        460: 0.891,
        470: 0.896,
        480: 0.899,
        490: 0.904,
        500: 0.907,
        510: 0.909,
        520: 0.911,
        530: 0.910,
        540: 0.911,
        550: 0.914,
        560: 0.913,
        570: 0.916,
        580: 0.915,
        590: 0.916,
        600: 0.914,
        610: 0.915,
        620: 0.918,
        630: 0.919,
        640: 0.921,
        650: 0.923,
        660: 0.924,
        670: 0.922,
        680: 0.922,
        690: 0.925,
        700: 0.927,
        710: 0.930,
        720: 0.930,
        730: 0.933
    },
    'neutral 8 (.23 D)': {
        380: 0.171,
        390: 0.232,
        400: 0.365,
        410: 0.507,
        420: 0.567,
        430: 0.583,
        440: 0.588,
        450: 0.590,
        460: 0.591,
        470: 0.590,
        480: 0.588,
        490: 0.588,
        500: 0.589,
        510: 0.589,
        520: 0.591,
        530: 0.590,
        540: 0.590,
        550: 0.590,
        560: 0.589,
        570: 0.591,
        580: 0.590,
        590: 0.590,
        600: 0.587,
        610: 0.585,
        620: 0.583,
        630: 0.580,
        640: 0.578,
        650: 0.576,
        660: 0.574,
        670: 0.572,
        680: 0.571,
        690: 0.569,
        700: 0.568,
        710: 0.568,
        720: 0.566,
        730: 0.566
    },
    'neutral 6.5 (.44 D)': {
        380: 0.144,
        390: 0.192,
        400: 0.272,
        410: 0.331,
        420: 0.350,
        430: 0.357,
        440: 0.361,
        450: 0.363,
        460: 0.363,
        470: 0.361,
        480: 0.359,
        490: 0.358,
        500: 0.358,
        510: 0.359,
        520: 0.360,
        530: 0.360,
        540: 0.361,
        550: 0.361,
        560: 0.360,
        570: 0.362,
        580: 0.362,
        590: 0.361,
        600: 0.359,
        610: 0.358,
        620: 0.355,
        630: 0.352,
        640: 0.350,
        650: 0.348,
        660: 0.345,
        670: 0.343,
        680: 0.340,
        690: 0.338,
        700: 0.335,
        710: 0.334,
        720: 0.332,
        730: 0.331
    },
    'neutral 5 (.70 D)': {
        380: 0.105,
        390: 0.131,
        400: 0.163,
        410: 0.180,
        420: 0.186,
        430: 0.190,
        440: 0.193,
        450: 0.194,
        460: 0.194,
        470: 0.192,
        480: 0.191,
        490: 0.191,
        500: 0.191,
        510: 0.192,
        520: 0.192,
        530: 0.192,
        540: 0.192,
        550: 0.192,
        560: 0.192,
        570: 0.193,
        580: 0.192,
        590: 0.192,
        600: 0.191,
        610: 0.189,
        620: 0.188,
        630: 0.186,
        640: 0.184,
        650: 0.182,
        660: 0.181,
        670: 0.179,
        680: 0.178,
        690: 0.176,
        700: 0.174,
        710: 0.173,
        720: 0.172,
        730: 0.171
    },
    'neutral 3.5 (1.05 D)': {
        380: 0.068,
        390: 0.077,
        400: 0.084,
        410: 0.087,
        420: 0.089,
        430: 0.090,
        440: 0.092,
        450: 0.092,
        460: 0.091,
        470: 0.090,
        480: 0.090,
        490: 0.090,
        500: 0.090,
        510: 0.090,
        520: 0.090,
        530: 0.090,
        540: 0.090,
        550: 0.090,
        560: 0.090,
        570: 0.090,
        580: 0.090,
        590: 0.089,
        600: 0.089,
        610: 0.088,
        620: 0.087,
        630: 0.086,
        640: 0.086,
        650: 0.085,
        660: 0.084,
        670: 0.084,
        680: 0.083,
        690: 0.083,
        700: 0.082,
        710: 0.081,
        720: 0.081,
        730: 0.081
    },
    'black 2 (1.5 D)': {
        380: 0.031,
        390: 0.032,
        400: 0.032,
        410: 0.033,
        420: 0.033,
        430: 0.033,
        440: 0.033,
        450: 0.033,
        460: 0.032,
        470: 0.032,
        480: 0.032,
        490: 0.032,
        500: 0.032,
        510: 0.032,
        520: 0.032,
        530: 0.032,
        540: 0.032,
        550: 0.032,
        560: 0.032,
        570: 0.032,
        580: 0.032,
        590: 0.032,
        600: 0.032,
        610: 0.032,
        620: 0.032,
        630: 0.032,
        640: 0.032,
        650: 0.032,
        660: 0.032,
        670: 0.032,
        680: 0.032,
        690: 0.032,
        700: 0.032,
        710: 0.032,
        720: 0.032,
        730: 0.033
    }
}

BABELCOLOR_AVERAGE_SPDS = dict(
    (key, SpectralPowerDistribution(key, value))
    for key, value in BABELCOLOR_AVERAGE_SPDS_DATA.items())
"""
Average data derived from measurements of 30 *ColourChecker* charts.

BABELCOLOR_AVERAGE_SPDS : dict
"""

COLOURCHECKERS_SPDS = CaseInsensitiveMapping({
    'BabelColor Average': BABELCOLOR_AVERAGE_SPDS,
    'ColorChecker N Ohta': COLORCHECKER_N_OHTA_SPDS
})
"""
Aggregated *ColourCheckers* spectral power distributions.

COLOURCHECKERS : CaseInsensitiveMapping
    **{'BabelColor Average', 'ColorChecker N Ohta'}**

Aliases:

-   'babel_average': 'BabelColor Average'
-   'cc_ohta': 'ColorChecker N Ohta'
"""
COLOURCHECKERS_SPDS['babel_average'] = COLOURCHECKERS_SPDS[
    'BabelColor Average']
COLOURCHECKERS_SPDS['cc_ohta'] = COLOURCHECKERS_SPDS['ColorChecker N Ohta']
