from cis_interface import backwards  # , platform
from cis_interface.serialize import register_serializer, _default_newline
from cis_interface.serialize.DefaultSerialize import DefaultSerialize
from cis_interface.metaschema.datatypes.PlyMetaschemaType import PlyDict


@register_serializer
class PlySerialize(DefaultSerialize):
    r"""Class for serializing/deserializing .ply file formats.

    Args:
        write_header (bool, optional): If True, headers will be added to
            serialized output. Defaults to True.
        newline (str, optional): String that should be used for new lines.
            Defaults to '\n'.

    Attributes:
        write_header (bool): If True, headers will be added to serialized
            output.
        newline (str): String that should be used for new lines.
        default_rgb (list): Default color in RGB that should be used for
            missing colors.

    """
    
    _seritype = 'ply'
    _schema_properties = dict(
        newline={'type': 'string',
                 'default': backwards.as_str(_default_newline)})
    _default_type = {'type': 'ply'}

    def __init__(self, *args, **kwargs):
        r"""Initialize immediately as default is only type."""
        super(PlySerialize, self).__init__(*args, **kwargs)
        self._initialized = True

    def func_serialize(self, args):
        r"""Serialize a message.

        Args:
            args: List of arguments to be formatted or numpy array to be
                serialized.

        Returns:
            bytes, str: Serialized message.

        """
        return backwards.as_bytes(self.datatype.encode_data(args, self.typedef))

    def func_deserialize(self, msg):
        r"""Deserialize a message.

        Args:
            msg: Message to be deserialized.

        Returns:
            obj: Deserialized message.

        """
        return PlyDict(self.datatype.decode_data(backwards.as_str(msg),
                                                 self.typedef))

    @classmethod
    def get_testing_options(cls):
        r"""Method to return a dictionary of testing options for this class.

        Returns:
            dict: Dictionary of variables to use for testing.

        """
        out = super(PlySerialize, cls).get_testing_options()
        obj = PlyDict({'vertices': [{'x': float(0), 'y': float(0), 'z': float(0)},
                                    {'x': float(0), 'y': float(0), 'z': float(1)},
                                    {'x': float(0), 'y': float(1), 'z': float(1)}],
                       'faces': [{'vertex_index': [int(0), int(1), int(2)]}]})
        out.update(objects=[obj, obj],
                   empty=dict(vertices=[], faces=[]),
                   contents=(b'ply\n'
                             + b'format ascii 1.0\n'
                             + b'comment author cis_auto\n'
                             + b'comment File generated by cis_interface\n'
                             + b'element vertex 6\n'
                             + b'property double x\n'
                             + b'property double y\n'
                             + b'property double z\n'
                             + b'element face 2\nproperty list uchar int vertex_index\n'
                             + b'end_header\n'
                             + b'0.0000 0.0000 0.0000\n'
                             + b'0.0000 0.0000 1.0000\n'
                             + b'0.0000 1.0000 1.0000\n'
                             + b'0.0000 0.0000 0.0000\n'
                             + b'0.0000 0.0000 1.0000\n'
                             + b'0.0000 1.0000 1.0000\n'
                             + b'3 0 1 2\n'
                             + b'3 3 4 5\n'))
        # out['contents'] = out['contents'].replace(b'\n', platform._newline)
        return out
