/*
 * Copyright 2016 The Cartographer Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef CARTOGRAPHER_COMMON_PORT_H_
#define CARTOGRAPHER_COMMON_PORT_H_

#include <cinttypes>
#include <cmath>
#include <string>

#include <boost/iostreams/device/back_inserter.hpp>
#include <boost/iostreams/filter/gzip.hpp>
#include <boost/iostreams/filtering_stream.hpp>

// This logic was borrowed (then namespaced) from the examples on the gcc wiki:
//     https://gcc.gnu.org/wiki/Visibility

#if defined _WIN32 || defined __CYGWIN__
  #ifdef __GNUC__
    #define CARTOGRAPHER_API_EXPORT __attribute__ ((dllexport))
    #define CARTOGRAPHER_API_IMPORT __attribute__ ((dllimport))
  #else
    #define CARTOGRAPHER_API_EXPORT __declspec(dllexport)
    #define CARTOGRAPHER_API_IMPORT __declspec(dllimport)
  #endif
  #ifdef CARTOGRAPHER_API_BUILDING_DLL
    #define CARTOGRAPHER_API_PUBLIC CARTOGRAPHER_API_EXPORT
  #else
    #define CARTOGRAPHER_API_PUBLIC CARTOGRAPHER_API_IMPORT
  #endif
  #define CARTOGRAPHER_API_PUBLIC_TYPE CARTOGRAPHER_API_PUBLIC
  #define CARTOGRAPHER_API_LOCAL
#else
  #define CARTOGRAPHER_API_EXPORT __attribute__ ((visibility("default")))
  #define CARTOGRAPHER_API_IMPORT
  #if __GNUC__ >= 4
    #define CARTOGRAPHER_API_PUBLIC __attribute__ ((visibility("default")))
    #define CARTOGRAPHER_API_LOCAL  __attribute__ ((visibility("hidden")))
  #else
    #define CARTOGRAPHER_API_PUBLIC
    #define CARTOGRAPHER_API_LOCAL
  #endif
  #define CARTOGRAPHER_API_PUBLIC_TYPE
#endif

namespace cartographer {

using int8 = int8_t;
using int16 = int16_t;
using int32 = int32_t;
using int64 = int64_t;
using uint8 = uint8_t;
using uint16 = uint16_t;
using uint32 = uint32_t;
using uint64 = uint64_t;

namespace common {

inline int RoundToInt(const float x) { return std::lround(x); }

inline int RoundToInt(const double x) { return std::lround(x); }

inline int64 RoundToInt64(const float x) { return std::lround(x); }

inline int64 RoundToInt64(const double x) { return std::lround(x); }

inline void FastGzipString(const std::string& uncompressed,
                           std::string* compressed) {
  boost::iostreams::filtering_ostream out;
  out.push(
      boost::iostreams::gzip_compressor(boost::iostreams::zlib::best_speed));
  out.push(boost::iostreams::back_inserter(*compressed));
  boost::iostreams::write(out,
                          reinterpret_cast<const char*>(uncompressed.data()),
                          uncompressed.size());
}

inline void FastGunzipString(const std::string& compressed,
                             std::string* decompressed) {
  boost::iostreams::filtering_ostream out;
  out.push(boost::iostreams::gzip_decompressor());
  out.push(boost::iostreams::back_inserter(*decompressed));
  boost::iostreams::write(out, reinterpret_cast<const char*>(compressed.data()),
                          compressed.size());
}

}  // namespace common
}  // namespace cartographer

#endif  // CARTOGRAPHER_COMMON_PORT_H_
