# -*- coding: utf-8 -*- 

import btk
import unittest
import _TDDConfigure 

class C3DFileWriterTest(unittest.TestCase):

  def assertAlmostEqual(self, first, second, places=7, msg=None, delta=None):
      try:
          super().assertAlmostEqual(first,second,places,msg, delta)
      except TypeError:
          first = float(first)
          second = float(second)
          super().assertAlmostEqual(first,second,places,msg, delta)

  def test_NoFileNoInput(self):
    writer = btk.btkAcquisitionFileWriter()
    self.assertRaises(RuntimeError, writer.Update)
    # self.assertRaisesRegexp(RuntimeError, 'Filename must be specified', writer.Update) # Python 2.7+

  def test_NoFile(self):
    reader = btk.btkAcquisitionFileReader()
    reader.SetFilename(_TDDConfigure.C3DFilePathIN + 'sample01/Eb015pi.c3d')
    writer = btk.btkAcquisitionFileWriter()
    writer.SetInput(reader.GetOutput())
    self.assertRaises(RuntimeError, writer.Update)
    # self.assertRaisesRegexp(RuntimeError, 'Filename must be specified', writer.Update) # Python 2.7+

  def test_NoFileWithIO(self):
    reader = btk.btkAcquisitionFileReader()
    reader.SetFilename(_TDDConfigure.C3DFilePathIN + 'sample01/Eb015pi.c3d')
    writer = btk.btkAcquisitionFileWriter()
    io = btk.btkC3DFileIO()
    writer.SetAcquisitionIO(io)
    self.assertTrue(io.HasInternalsUpdateOption(btk.btkC3DFileIO.DataBasedUpdate))
    self.assertTrue(io.HasInternalsUpdateOption(btk.btkC3DFileIO.CompatibleVicon))
    self.assertRaises(RuntimeError, writer.Update)
    # self.assertRaisesRegexp(RuntimeError, 'Filename must be specified', writer.Update) # Python 2.7+

  def test_sample01_Eb015pi_rewrited(self):
    reader = btk.btkAcquisitionFileReader()
    reader.SetFilename(_TDDConfigure.C3DFilePathIN + 'sample01/Eb015pi.c3d')
    writer = btk.btkAcquisitionFileWriter()
    reader.Update()
    io = btk.btkC3DFileIO(reader.GetAcquisitionIO())
    writer.SetAcquisitionIO(io)
    writer.SetInput(reader.GetOutput())
    writer.SetFilename(_TDDConfigure.C3DFilePathOUT + 'sample01_Eb015pi.c3d')
    writer.Update()

    reader2 = btk.btkAcquisitionFileReader()
    reader2.SetFilename(_TDDConfigure.C3DFilePathOUT + 'sample01_Eb015pi.c3d')
    reader2.Update()
    acq = reader2.GetOutput()
    self.assertEqual(acq.GetPointFrequency(), 50)
    self.assertEqual(acq.GetPointNumber(), 26)
    self.assertEqual(acq.GetAnalogFrequency(), 200)
    self.assertEqual(acq.GetAnalogNumber(), 16)
    self.assertEqual(acq.GetPoint(0).GetLabel(), 'RFT1')
    self.assertEqual(acq.GetPoint(9).GetDescription(), 'DISTAL FOOT')
    self.assertEqual(acq.GetAnalog(15).GetLabel(), 'CH16')
    self.assertEqual(acq.GetAnalog(2).GetDescription(), 'FORCE Z-COMP 1')
    self.assertAlmostEqual(acq.GetPoint(0).GetValues()[8,0], 250.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,0], 285.0, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,1], 222.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,2], 564.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(0).GetResiduals()[0], 1.3333, 3)
    self.assertAlmostEqual(acq.GetPoint(14).GetResiduals()[20], 1.91667, 3)
    self.assertAlmostEqual(acq.GetAnalog(0).GetValues()[0], -26.6599, 3)
    self.assertAlmostEqual(acq.GetAnalog(0).GetValues()[1], -25.8, 3)
    
    acqRef = reader.GetOutput()
    for i in range(0,50):
      self.assertAlmostEqual(acq.GetAnalog(0).GetValues()[i], acqRef.GetAnalog(0).GetValues()[i], 4)
      self.assertAlmostEqual(acq.GetAnalog(1).GetValues()[i], acqRef.GetAnalog(1).GetValues()[i], 4)
      self.assertAlmostEqual(acq.GetAnalog(0).GetValues()[i], acqRef.GetAnalog(0).GetValues()[i], 4)
      self.assertAlmostEqual(acq.GetAnalog(1).GetValues()[i], acqRef.GetAnalog(1).GetValues()[i], 4)

    self.assertAlmostEqual(io.GetPointScale(), 0.08333, 3)

    self.assertEqual(acq.GetEventNumber(), 3)
    self.assertEqual(acq.GetEvent(0).GetLabel(), 'RIC')
    self.assertAlmostEqual(acq.GetEvent(0).GetTime(), 2.72, 3)
    self.assertEqual(acq.GetEvent(1).GetLabel(), 'RHS')
    self.assertAlmostEqual(acq.GetEvent(1).GetTime(), 5.40, 3)
    self.assertEqual(acq.GetEvent(2).GetLabel(), 'RTO')
    self.assertAlmostEqual(acq.GetEvent(2).GetTime(), 7.32, 3)
    
    self.assertEqual(len(acq.GetMetaData().GetChild('EVENT').GetChild('TIMES').GetInfo().GetDimensions()), 2)
    self.assertEqual(acq.GetMetaData().GetChild('EVENT').GetChild('TIMES').GetInfo().GetDimensions()[0], 2)
    self.assertEqual(acq.GetMetaData().GetChild('EVENT').GetChild('TIMES').GetInfo().GetDimensions()[1], 3)

  def test_sample01_Eb015pr_rewrited(self):
    reader = btk.btkAcquisitionFileReader()
    reader.SetFilename(_TDDConfigure.C3DFilePathIN + 'sample01/Eb015pr.c3d')
    writer = btk.btkAcquisitionFileWriter()
    reader.Update()
    io = btk.btkC3DFileIO(reader.GetAcquisitionIO())
    io.SetInternalsUpdateOptions(btk.btkC3DFileIO.NoUpdate)
    writer.SetAcquisitionIO(io)
    writer.SetInput(reader.GetOutput())
    writer.SetFilename(_TDDConfigure.C3DFilePathOUT + 'sample01_Eb015pr.c3d')
    writer.Update()

    reader2 = btk.btkAcquisitionFileReader()
    reader2.SetFilename(_TDDConfigure.C3DFilePathOUT + 'sample01_Eb015pi.c3d')
    reader2.Update()
    acq = reader2.GetOutput()
    self.assertEqual(acq.GetPointFrequency(), 50)
    self.assertEqual(acq.GetPointNumber(), 26)
    self.assertEqual(acq.GetAnalogFrequency(), 200)
    self.assertEqual(acq.GetAnalogNumber(), 16)
    self.assertEqual(acq.GetPoint(0).GetLabel(), 'RFT1')
    self.assertEqual(acq.GetPoint(9).GetDescription(), 'DISTAL FOOT')
    self.assertEqual(acq.GetAnalog(15).GetLabel(), 'CH16')
    self.assertEqual(acq.GetAnalog(2).GetDescription(), 'FORCE Z-COMP 1')
    self.assertAlmostEqual(acq.GetPoint(0).GetValues()[8,0], 250.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,0], 285.0, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,1], 222.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,2], 564.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(0).GetResiduals()[0], 1.3333, 3)
    self.assertAlmostEqual(acq.GetPoint(14).GetResiduals()[20], 1.91667, 3)
    self.assertAlmostEqual(acq.GetAnalog(0).GetValues()[0], -26.6599, 3)
    self.assertAlmostEqual(acq.GetAnalog(0).GetValues()[1], -25.8, 3)

    acqRef = reader.GetOutput()
    for i in range(0,50):
      self.assertAlmostEqual(acq.GetPoint(0).GetResiduals()[i], acqRef.GetPoint(0).GetResiduals()[i], 3)
      self.assertAlmostEqual(acq.GetPoint(14).GetResiduals()[i], acqRef.GetPoint(14).GetResiduals()[i], 3)
      self.assertAlmostEqual(acq.GetAnalog(0).GetValues()[i], acqRef.GetAnalog(0).GetValues()[i], 4)
      self.assertAlmostEqual(acq.GetAnalog(1).GetValues()[i], acqRef.GetAnalog(1).GetValues()[i], 4)

    self.assertAlmostEqual(io.GetPointScale(), 0.08333, 3)

  def test_sample01_Eb015si_rewrited(self):
    reader = btk.btkAcquisitionFileReader()
    reader.SetFilename(_TDDConfigure.C3DFilePathIN + 'sample01/Eb015si.c3d')
    writer = btk.btkAcquisitionFileWriter()
    reader.Update()
    io = btk.btkC3DFileIO(reader.GetAcquisitionIO())
    io.SetInternalsUpdateOptions(btk.btkC3DFileIO.NoUpdate)
    writer.SetAcquisitionIO(io)
    writer.SetInput(reader.GetOutput())
    writer.SetFilename(_TDDConfigure.C3DFilePathOUT + 'sample01_Eb015si.c3d')
    writer.Update()

    reader2 = btk.btkAcquisitionFileReader()
    reader2.SetFilename(_TDDConfigure.C3DFilePathOUT + 'sample01_Eb015si.c3d')
    reader2.Update()
    acq = reader2.GetOutput()
    self.assertEqual(acq.GetPointFrequency(), 50)
    self.assertEqual(acq.GetPointNumber(), 26)
    self.assertEqual(acq.GetAnalogFrequency(), 200)
    self.assertEqual(acq.GetAnalogNumber(), 16)
    self.assertEqual(acq.GetPoint(0).GetLabel(), 'RFT1')
    self.assertEqual(acq.GetPoint(9).GetDescription(), 'DISTAL FOOT')
    self.assertEqual(acq.GetAnalog(15).GetLabel(), 'CH16')
    self.assertEqual(acq.GetAnalog(2).GetDescription(), 'FORCE Z-COMP 1')
    self.assertAlmostEqual(acq.GetPoint(0).GetValues()[8,0], 250.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,0], 285.0, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,1], 222.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,2], 564.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(0).GetResiduals()[0], 1.3333, 3)
    self.assertAlmostEqual(acq.GetPoint(14).GetResiduals()[20], 1.91667, 3)
    self.assertAlmostEqual(acq.GetAnalog(0).GetValues()[0], -26.6599, 3)
    self.assertAlmostEqual(acq.GetAnalog(0).GetValues()[1], -25.8, 3)
    
    acqRef = reader.GetOutput()
    for i in range(0,50):
      self.assertAlmostEqual(acq.GetPoint(0).GetResiduals()[i], acqRef.GetPoint(0).GetResiduals()[i], 3)
      self.assertAlmostEqual(acq.GetPoint(14).GetResiduals()[i], acqRef.GetPoint(14).GetResiduals()[i], 3)
      self.assertAlmostEqual(acq.GetAnalog(0).GetValues()[i], acqRef.GetAnalog(0).GetValues()[i], 4)
      self.assertAlmostEqual(acq.GetAnalog(1).GetValues()[i], acqRef.GetAnalog(1).GetValues()[i], 4)
    
    self.assertAlmostEqual(io.GetPointScale(), 0.08333, 3)

  def test_sample01_Eb015sr_rewrited(self):
    reader = btk.btkAcquisitionFileReader()
    reader.SetFilename(_TDDConfigure.C3DFilePathIN + 'sample01/Eb015sr.c3d')
    writer = btk.btkAcquisitionFileWriter()
    reader.Update()
    io = btk.btkC3DFileIO(reader.GetAcquisitionIO())
    io.SetInternalsUpdateOptions(btk.btkC3DFileIO.NoUpdate)
    writer.SetAcquisitionIO(io)
    writer.SetInput(reader.GetOutput())
    writer.SetFilename(_TDDConfigure.C3DFilePathOUT + 'sample01_Eb015sr.c3d')
    writer.Update()

    reader2 = btk.btkAcquisitionFileReader()
    reader2.SetFilename(_TDDConfigure.C3DFilePathOUT + 'sample01_Eb015sr.c3d')
    reader2.Update()
    acq = reader2.GetOutput()
    self.assertEqual(acq.GetPointFrequency(), 50)
    self.assertEqual(acq.GetPointNumber(), 26)
    self.assertEqual(acq.GetAnalogFrequency(), 200)
    self.assertEqual(acq.GetAnalogNumber(), 16)
    self.assertEqual(acq.GetPoint(0).GetLabel(), 'RFT1')
    self.assertEqual(acq.GetPoint(9).GetDescription(), 'DISTAL FOOT')
    self.assertEqual(acq.GetAnalog(15).GetLabel(), 'CH16')
    self.assertEqual(acq.GetAnalog(2).GetDescription(), 'FORCE Z-COMP 1')
    self.assertAlmostEqual(acq.GetPoint(0).GetValues()[8,0], 250.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,0], 285.0, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,1], 222.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,2], 564.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(0).GetResiduals()[0], 1.3333, 3)
    self.assertAlmostEqual(acq.GetPoint(14).GetResiduals()[20], 1.91667, 3)
    self.assertAlmostEqual(acq.GetAnalog(0).GetValues()[0], -26.6599, 3)
    self.assertAlmostEqual(acq.GetAnalog(0).GetValues()[1], -25.8, 3)
    
    acqRef = reader.GetOutput()
    for i in range(0,50):
      self.assertAlmostEqual(acq.GetPoint(0).GetResiduals()[i], acqRef.GetPoint(0).GetResiduals()[i], 3)
      self.assertAlmostEqual(acq.GetPoint(14).GetResiduals()[i], acqRef.GetPoint(14).GetResiduals()[i], 3)
      self.assertAlmostEqual(acq.GetAnalog(0).GetValues()[i], acqRef.GetAnalog(0).GetValues()[i], 4)
      self.assertAlmostEqual(acq.GetAnalog(1).GetValues()[i], acqRef.GetAnalog(1).GetValues()[i], 4)

    self.assertAlmostEqual(io.GetPointScale(), 0.08333, 3)

  def test_sample01_Eb015vi_rewrited(self):
    reader = btk.btkAcquisitionFileReader()
    reader.SetFilename(_TDDConfigure.C3DFilePathIN + 'sample01/Eb015vi.c3d')
    writer = btk.btkAcquisitionFileWriter()
    reader.Update()
    io = btk.btkC3DFileIO(reader.GetAcquisitionIO())
    io.SetInternalsUpdateOptions(btk.btkC3DFileIO.NoUpdate)
    writer.SetAcquisitionIO(io)
    writer.SetInput(reader.GetOutput())
    writer.SetFilename(_TDDConfigure.C3DFilePathOUT + 'sample01_Eb015vi.c3d')
    writer.Update()

    reader2 = btk.btkAcquisitionFileReader()
    reader2.SetFilename(_TDDConfigure.C3DFilePathOUT + 'sample01_Eb015vi.c3d')
    reader2.Update()
    acq = reader2.GetOutput()
    self.assertEqual(acq.GetPointFrequency(), 50)
    self.assertEqual(acq.GetPointNumber(), 26)
    self.assertEqual(acq.GetAnalogFrequency(), 200)
    self.assertEqual(acq.GetAnalogNumber(), 16)
    self.assertEqual(acq.GetPoint(0).GetLabel(), 'RFT1')
    self.assertEqual(acq.GetPoint(9).GetDescription(), 'DISTAL FOOT')
    self.assertEqual(acq.GetAnalog(15).GetLabel(), 'CH16')
    self.assertEqual(acq.GetAnalog(2).GetDescription(), 'FORCE Z-COMP 1')
    self.assertAlmostEqual(acq.GetPoint(0).GetValues()[8,0], 250.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,0], 285.0, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,1], 222.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,2], 564.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(0).GetResiduals()[0], 1.3333, 3)
    self.assertAlmostEqual(acq.GetPoint(14).GetResiduals()[20], 1.91667, 3)
    self.assertAlmostEqual(acq.GetAnalog(0).GetValues()[0], -26.6599, 3)
    self.assertAlmostEqual(acq.GetAnalog(0).GetValues()[1], -25.8, 3)
    
    acqRef = reader.GetOutput()
    for i in range(0,50):
      self.assertAlmostEqual(acq.GetPoint(0).GetResiduals()[i], acqRef.GetPoint(0).GetResiduals()[i], 3)
      self.assertAlmostEqual(acq.GetPoint(14).GetResiduals()[i], acqRef.GetPoint(14).GetResiduals()[i], 3)
      self.assertAlmostEqual(acq.GetAnalog(0).GetValues()[i], acqRef.GetAnalog(0).GetValues()[i], 4)
      self.assertAlmostEqual(acq.GetAnalog(1).GetValues()[i], acqRef.GetAnalog(1).GetValues()[i], 4)

    self.assertAlmostEqual(io.GetPointScale(), 0.08333, 3)

  def test_sample01_Eb015vr_rewrited(self):
    reader = btk.btkAcquisitionFileReader()
    reader.SetFilename(_TDDConfigure.C3DFilePathIN + 'sample01/Eb015vr.c3d')
    writer = btk.btkAcquisitionFileWriter()
    reader.Update()
    io = btk.btkC3DFileIO(reader.GetAcquisitionIO())
    io.SetInternalsUpdateOptions(btk.btkC3DFileIO.NoUpdate)
    writer.SetAcquisitionIO(io)
    writer.SetInput(reader.GetOutput())
    writer.SetFilename(_TDDConfigure.C3DFilePathOUT + 'sample01_Eb015vr.c3d')
    writer.Update()

    reader2 = btk.btkAcquisitionFileReader()
    reader2.SetFilename(_TDDConfigure.C3DFilePathOUT + 'sample01_Eb015vr.c3d')
    reader2.Update()
    acq = reader2.GetOutput()
    self.assertEqual(acq.GetPointFrequency(), 50)
    self.assertEqual(acq.GetPointNumber(), 26)
    self.assertEqual(acq.GetAnalogFrequency(), 200)
    self.assertEqual(acq.GetAnalogNumber(), 16)
    self.assertEqual(acq.GetPoint(0).GetLabel(), 'RFT1')
    self.assertEqual(acq.GetPoint(9).GetDescription(), 'DISTAL FOOT')
    self.assertEqual(acq.GetAnalog(15).GetLabel(), 'CH16')
    self.assertEqual(acq.GetAnalog(2).GetDescription(), 'FORCE Z-COMP 1')
    self.assertAlmostEqual(acq.GetPoint(0).GetValues()[8,0], 250.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,0], 285.0, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,1], 222.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,2], 564.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(0).GetResiduals()[0], 1.3333, 3)
    self.assertAlmostEqual(acq.GetPoint(14).GetResiduals()[20], 1.91667, 3)
    self.assertAlmostEqual(acq.GetAnalog(0).GetValues()[0], -26.6599, 3)
    self.assertAlmostEqual(acq.GetAnalog(0).GetValues()[1], -25.8, 3)

    acqRef = reader.GetOutput()
    for i in range(0,50):
      self.assertAlmostEqual(acq.GetPoint(0).GetResiduals()[i], acqRef.GetPoint(0).GetResiduals()[i], 3)
      self.assertAlmostEqual(acq.GetPoint(14).GetResiduals()[i], acqRef.GetPoint(14).GetResiduals()[i], 3)
      self.assertAlmostEqual(acq.GetAnalog(0).GetValues()[i], acqRef.GetAnalog(0).GetValues()[i], 4)
      self.assertAlmostEqual(acq.GetAnalog(1).GetValues()[i], acqRef.GetAnalog(1).GetValues()[i], 4)
    
    self.assertAlmostEqual(io.GetPointScale(), 0.08333, 3)

  def test_sample01_Eb015vr_clearAnalog(self):
    reader = btk.btkAcquisitionFileReader()
    reader.SetFilename(_TDDConfigure.C3DFilePathIN + 'sample01/Eb015vr.c3d')
    writer = btk.btkAcquisitionFileWriter()
    reader.Update()
    acq = reader.GetOutput()
    acq.ClearAnalogs()
    acq.RemovePoint(25)
    acq.RemovePoint(24)
    acq.RemovePoint(23)
    acq.RemovePoint(22)
    writer.SetAcquisitionIO(reader.GetAcquisitionIO())
    writer.SetInput(reader.GetOutput())
    writer.SetFilename(_TDDConfigure.C3DFilePathOUT + 'sample01_Eb015vr_clearAnalog.c3d')
    writer.Update()

    reader.SetFilename(_TDDConfigure.C3DFilePathOUT + 'sample01_Eb015vr_clearAnalog.c3d')
    reader.Update()
    acq = reader.GetOutput()
    self.assertEqual(acq.GetPointFrequency(), 50)
    self.assertEqual(acq.GetPointNumber(), 22)
    self.assertEqual(acq.GetAnalogFrequency(), 200)
    self.assertEqual(acq.GetAnalogNumber(), 0)
    self.assertEqual(acq.GetPoint(0).GetLabel(), 'RFT1')
    self.assertEqual(acq.GetPoint(9).GetDescription(), 'DISTAL FOOT')
    self.assertAlmostEqual(acq.GetPoint(0).GetValues()[8,0], 250.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,0], 285.0, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,1], 222.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,2], 564.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(0).GetResiduals()[0], 1.3333, 3)
    self.assertAlmostEqual(acq.GetPoint(14).GetResiduals()[20], 1.91667, 3)

    io = btk.btkC3DFileIO(writer.GetAcquisitionIO())
    self.assertAlmostEqual(io.GetPointScale(), 0.08333, 3)

  def test_sample01_Eb015vr_clearPoint(self):
    reader = btk.btkAcquisitionFileReader()
    reader.SetFilename(_TDDConfigure.C3DFilePathIN + 'sample01/Eb015vr.c3d')
    writer = btk.btkAcquisitionFileWriter()
    reader.Update()
    acq = reader.GetOutput()
    acq.ClearPoints()
    acq.ClearEvents()
    acq.SetMaxInterpolationGap(50)
    writer.SetInput(reader.GetOutput())
    writer.SetFilename(_TDDConfigure.C3DFilePathOUT + 'sample01_Eb015vr_clearPoint.c3d')
    writer.Update()

    reader.SetFilename(_TDDConfigure.C3DFilePathOUT + 'sample01_Eb015vr_clearPoint.c3d')
    reader.Update()
    acq = reader.GetOutput()
    self.assertEqual(acq.GetMaxInterpolationGap(), 50)
    self.assertEqual(acq.GetPointFrequency(), 50)
    self.assertEqual(acq.GetPointNumber(), 0)
    self.assertEqual(acq.GetAnalogFrequency(), 200)
    self.assertEqual(acq.GetAnalogNumber(), 16)
    self.assertEqual(acq.GetEventNumber(), 0)
    self.assertEqual(acq.GetAnalog(15).GetLabel(), 'CH16')
    self.assertEqual(acq.GetAnalog(2).GetDescription(), 'FORCE Z-COMP 1')
    self.assertAlmostEqual(acq.GetAnalog(0).GetValues()[0], -26.6599, 3)
    self.assertAlmostEqual(acq.GetAnalog(0).GetValues()[1], -25.8, 3)

    io = btk.btkC3DFileIO(writer.GetAcquisitionIO())
    self.assertAlmostEqual(io.GetPointScale(), 0.1, 3)

  def test_sample01_Eb015vr_clearAll(self):
    reader = btk.btkAcquisitionFileReader()
    reader.SetFilename(_TDDConfigure.C3DFilePathIN + 'sample01/Eb015vr.c3d')
    writer = btk.btkAcquisitionFileWriter()
    reader.Update()
    acq = reader.GetOutput()
    acq.ClearPoints()
    acq.ClearAnalogs()
    acq.ClearEvents()
    writer.SetAcquisitionIO(reader.GetAcquisitionIO())
    writer.SetInput(reader.GetOutput())
    writer.SetFilename(_TDDConfigure.C3DFilePathOUT + 'sample01_Eb015vr_clearAll.c3d')
    writer.Update()

    reader.SetFilename(_TDDConfigure.C3DFilePathOUT + 'sample01_Eb015vr_clearAll.c3d')
    reader.Update()
    acq = reader.GetOutput()
    self.assertEqual(acq.GetPointFrequency(), 50)
    self.assertEqual(acq.GetPointNumber(), 0)
    self.assertEqual(acq.GetAnalogFrequency(), 200)
    self.assertEqual(acq.GetAnalogNumber(), 0)
    self.assertEqual(acq.GetEventNumber(), 0)

    io = btk.btkC3DFileIO(reader.GetAcquisitionIO())
    self.assertAlmostEqual(io.GetPointScale(), 0.08333, 3)

  def test_sample01_Eb015si_from_Eb015vr(self):
    reader = btk.btkAcquisitionFileReader()
    reader.SetFilename(_TDDConfigure.C3DFilePathIN + 'sample01/Eb015vr.c3d')
    writer = btk.btkAcquisitionFileWriter()
    reader.Update()
    acq = reader.GetOutput()
    acq.ClearAnalogs()
    acq.ClearEvents()
    acq.RemovePoint(25)
    acq.RemovePoint(24)
    acq.RemovePoint(23)
    acq.RemovePoint(22)
    io = btk.btkC3DFileIO(reader.GetAcquisitionIO())
    self.assertEqual(io.GetStorageFormat(), btk.btkAcquisitionFileIO.Float)
    self.assertEqual(io.GetByteOrder(), btk.btkAcquisitionFileIO.VAX_LittleEndian)
    io.SetStorageFormat(btk.btkAcquisitionFileIO.Integer)
    io.SetByteOrder(btk.btkAcquisitionFileIO.IEEE_BigEndian)
    writer.SetAcquisitionIO(io)
    writer.SetInput(reader.GetOutput())
    writer.SetFilename(_TDDConfigure.C3DFilePathOUT + 'sample01_Eb015si_from_Eb015vr.c3d')
    writer.Update()

    reader.SetFilename(_TDDConfigure.C3DFilePathOUT + 'sample01_Eb015si_from_Eb015vr.c3d')
    reader.Update()
    acq = reader.GetOutput()
    io = btk.btkC3DFileIO(reader.GetAcquisitionIO())
    self.assertEqual(io.GetStorageFormat(), btk.btkAcquisitionFileIO.Integer)
    self.assertEqual(io.GetByteOrder(), btk.btkAcquisitionFileIO.IEEE_BigEndian)
    self.assertAlmostEqual(io.GetPointScale(), 0.08333, 3)
    self.assertEqual(acq.GetPointFrequency(), 50)
    self.assertEqual(acq.GetPointNumber(), 22)
    self.assertEqual(acq.GetAnalogFrequency(), 200)
    self.assertEqual(acq.GetAnalogNumber(), 0)
    self.assertEqual(acq.GetEventNumber(), 0)
    self.assertEqual(acq.GetPoint(0).GetLabel(), 'RFT1')
    self.assertEqual(acq.GetPoint(9).GetDescription(), 'DISTAL FOOT')
    self.assertAlmostEqual(acq.GetPoint(0).GetValues()[8,0], 250.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,0], 285.0, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,1], 222.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,2], 564.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(0).GetResiduals()[0], 1.3333, 3)
    self.assertAlmostEqual(acq.GetPoint(14).GetResiduals()[20], 1.91667, 3)

  def test_sample01_Eb015pi_from_Eb015vr(self):
    reader = btk.btkAcquisitionFileReader()
    reader.SetFilename(_TDDConfigure.C3DFilePathIN + 'sample01/Eb015vr.c3d')
    writer = btk.btkAcquisitionFileWriter()
    reader.Update()
    acq = reader.GetOutput()
    acq.ClearAnalogs()
    acq.ClearEvents()
    acq.RemovePoint(25)
    acq.RemovePoint(24)
    acq.RemovePoint(23)
    acq.RemovePoint(22)
    io = btk.btkC3DFileIO(reader.GetAcquisitionIO())
    self.assertEqual(io.GetStorageFormat(), btk.btkAcquisitionFileIO.Float)
    self.assertEqual(io.GetByteOrder(), btk.btkAcquisitionFileIO.VAX_LittleEndian)
    io.SetStorageFormat(btk.btkAcquisitionFileIO.Integer)
    io.SetByteOrder(btk.btkAcquisitionFileIO.IEEE_LittleEndian)
    writer.SetAcquisitionIO(io)
    writer.SetInput(reader.GetOutput())
    writer.SetFilename(_TDDConfigure.C3DFilePathOUT + 'sample01_Eb015pi_from_Eb015vr.c3d')
    writer.Update()

    reader.SetFilename(_TDDConfigure.C3DFilePathOUT + 'sample01_Eb015pi_from_Eb015vr.c3d')
    reader.Update()
    acq = reader.GetOutput()
    io = btk.btkC3DFileIO(reader.GetAcquisitionIO())
    self.assertEqual(io.GetStorageFormat(), btk.btkAcquisitionFileIO.Integer)
    self.assertEqual(io.GetByteOrder(), btk.btkAcquisitionFileIO.IEEE_LittleEndian)
    self.assertAlmostEqual(io.GetPointScale(), 0.08333, 3)
    self.assertEqual(acq.GetPointFrequency(), 50)
    self.assertEqual(acq.GetPointNumber(), 22)
    self.assertEqual(acq.GetAnalogFrequency(), 200)
    self.assertEqual(acq.GetAnalogNumber(), 0)
    self.assertEqual(acq.GetEventNumber(), 0)
    self.assertEqual(acq.GetPoint(0).GetLabel(), 'RFT1')
    self.assertEqual(acq.GetPoint(9).GetDescription(), 'DISTAL FOOT')
    self.assertAlmostEqual(acq.GetPoint(0).GetValues()[8,0], 250.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,0], 285.0, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,1], 222.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,2], 564.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(0).GetResiduals()[0], 1.3333, 3)
    self.assertAlmostEqual(acq.GetPoint(14).GetResiduals()[20], 1.91667, 3)

  def test_sample01_Eb015vi_from_Eb015vr(self):
    reader = btk.btkAcquisitionFileReader()
    reader.SetFilename(_TDDConfigure.C3DFilePathIN + 'sample01/Eb015vr.c3d')
    writer = btk.btkAcquisitionFileWriter()
    reader.Update()
    acq = reader.GetOutput()
    acq.ClearAnalogs()
    acq.ClearEvents()
    acq.RemovePoint(25)
    acq.RemovePoint(24)
    acq.RemovePoint(23)
    acq.RemovePoint(22)
    io = btk.btkC3DFileIO(reader.GetAcquisitionIO())
    self.assertEqual(io.GetStorageFormat(), btk.btkAcquisitionFileIO.Float)
    self.assertEqual(io.GetByteOrder(), btk.btkAcquisitionFileIO.VAX_LittleEndian)
    io.SetStorageFormat(btk.btkAcquisitionFileIO.Integer)
    io.SetByteOrder(btk.btkAcquisitionFileIO.VAX_LittleEndian)
    writer.SetAcquisitionIO(io)
    writer.SetInput(reader.GetOutput())
    writer.SetFilename(_TDDConfigure.C3DFilePathOUT + 'sample01_Eb015vi_from_Eb015vr.c3d')
    writer.Update()

    reader.SetFilename(_TDDConfigure.C3DFilePathOUT + 'sample01_Eb015vi_from_Eb015vr.c3d')
    reader.Update()
    acq = reader.GetOutput()
    io = btk.btkC3DFileIO(reader.GetAcquisitionIO())
    self.assertEqual(io.GetStorageFormat(), btk.btkAcquisitionFileIO.Integer)
    self.assertEqual(io.GetByteOrder(), btk.btkAcquisitionFileIO.VAX_LittleEndian)
    self.assertAlmostEqual(io.GetPointScale(), 0.08333, 3)
    self.assertEqual(acq.GetPointFrequency(), 50)
    self.assertEqual(acq.GetPointNumber(), 22)
    self.assertEqual(acq.GetAnalogFrequency(), 200)
    self.assertEqual(acq.GetAnalogNumber(), 0)
    self.assertEqual(acq.GetEventNumber(), 0)
    self.assertEqual(acq.GetPoint(0).GetLabel(), 'RFT1')
    self.assertEqual(acq.GetPoint(9).GetDescription(), 'DISTAL FOOT')
    self.assertAlmostEqual(acq.GetPoint(0).GetValues()[8,0], 250.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,0], 285.0, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,1], 222.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,2], 564.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(0).GetResiduals()[0], 1.3333, 3)
    self.assertAlmostEqual(acq.GetPoint(14).GetResiduals()[20], 1.91667, 3)

  def test_sample01_Eb015si_from_Eb015pr(self):
    reader = btk.btkAcquisitionFileReader()
    reader.SetFilename(_TDDConfigure.C3DFilePathIN + 'sample01/Eb015pr.c3d')
    writer = btk.btkAcquisitionFileWriter()
    reader.Update()
    acq = reader.GetOutput()
    acq.ClearAnalogs()
    acq.ClearEvents()
    acq.RemovePoint(25)
    acq.RemovePoint(24)
    acq.RemovePoint(23)
    acq.RemovePoint(22)
    io = btk.btkC3DFileIO(reader.GetAcquisitionIO())
    self.assertEqual(io.GetStorageFormat(), btk.btkAcquisitionFileIO.Float)
    self.assertEqual(io.GetByteOrder(), btk.btkAcquisitionFileIO.IEEE_LittleEndian)
    io.SetStorageFormat(btk.btkAcquisitionFileIO.Integer)
    io.SetByteOrder(btk.btkAcquisitionFileIO.IEEE_BigEndian)
    writer.SetAcquisitionIO(io)
    writer.SetInput(reader.GetOutput())
    writer.SetFilename(_TDDConfigure.C3DFilePathOUT + 'sample01_Eb015si_from_Eb015pr.c3d')
    writer.Update()

    reader.SetFilename(_TDDConfigure.C3DFilePathOUT + 'sample01_Eb015si_from_Eb015pr.c3d')
    reader.Update()
    acq = reader.GetOutput()
    io = btk.btkC3DFileIO(reader.GetAcquisitionIO())
    self.assertEqual(io.GetStorageFormat(), btk.btkAcquisitionFileIO.Integer)
    self.assertEqual(io.GetByteOrder(), btk.btkAcquisitionFileIO.IEEE_BigEndian)
    self.assertAlmostEqual(io.GetPointScale(), 0.08333, 3)
    self.assertEqual(acq.GetPointFrequency(), 50)
    self.assertEqual(acq.GetPointNumber(), 22)
    self.assertEqual(acq.GetAnalogFrequency(), 200)
    self.assertEqual(acq.GetAnalogNumber(), 0)
    self.assertEqual(acq.GetEventNumber(), 0)
    self.assertEqual(acq.GetPoint(0).GetLabel(), 'RFT1')
    self.assertEqual(acq.GetPoint(9).GetDescription(), 'DISTAL FOOT')
    self.assertAlmostEqual(acq.GetPoint(0).GetValues()[8,0], 250.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,0], 285.0, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,1], 222.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,2], 564.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(0).GetResiduals()[0], 1.3333, 3)
    self.assertAlmostEqual(acq.GetPoint(14).GetResiduals()[20], 1.91667, 3)

  def test_sample01_Eb015pi_from_Eb015pr(self):
    reader = btk.btkAcquisitionFileReader()
    reader.SetFilename(_TDDConfigure.C3DFilePathIN + 'sample01/Eb015pr.c3d')
    writer = btk.btkAcquisitionFileWriter()
    reader.Update()
    acq = reader.GetOutput()
    acq.ClearAnalogs()
    acq.ClearEvents()
    acq.RemovePoint(25)
    acq.RemovePoint(24)
    acq.RemovePoint(23)
    acq.RemovePoint(22)
    io = btk.btkC3DFileIO(reader.GetAcquisitionIO())
    self.assertEqual(io.GetStorageFormat(), btk.btkAcquisitionFileIO.Float)
    self.assertEqual(io.GetByteOrder(), btk.btkAcquisitionFileIO.IEEE_LittleEndian)
    io.SetStorageFormat(btk.btkAcquisitionFileIO.Integer)
    io.SetByteOrder(btk.btkAcquisitionFileIO.IEEE_LittleEndian)
    writer.SetAcquisitionIO(io)
    writer.SetInput(reader.GetOutput())
    writer.SetFilename(_TDDConfigure.C3DFilePathOUT + 'sample01_Eb015pi_from_Eb015pr.c3d')
    writer.Update()

    reader.SetFilename(_TDDConfigure.C3DFilePathOUT + 'sample01_Eb015pi_from_Eb015pr.c3d')
    reader.Update()
    acq = reader.GetOutput()
    io = btk.btkC3DFileIO(reader.GetAcquisitionIO())
    self.assertEqual(io.GetStorageFormat(), btk.btkAcquisitionFileIO.Integer)
    self.assertEqual(io.GetByteOrder(), btk.btkAcquisitionFileIO.IEEE_LittleEndian)
    self.assertAlmostEqual(io.GetPointScale(), 0.08333, 3)
    self.assertEqual(acq.GetPointFrequency(), 50)
    self.assertEqual(acq.GetPointNumber(), 22)
    self.assertEqual(acq.GetAnalogFrequency(), 200)
    self.assertEqual(acq.GetAnalogNumber(), 0)
    self.assertEqual(acq.GetEventNumber(), 0)
    self.assertEqual(acq.GetPoint(0).GetLabel(), 'RFT1')
    self.assertEqual(acq.GetPoint(9).GetDescription(), 'DISTAL FOOT')
    self.assertAlmostEqual(acq.GetPoint(0).GetValues()[8,0], 250.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,0], 285.0, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,1], 222.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,2], 564.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(0).GetResiduals()[0], 1.3333, 3)
    self.assertAlmostEqual(acq.GetPoint(14).GetResiduals()[20], 1.91667, 3)

  def test_sample01_Eb015vi_from_Eb015pr(self):
    reader = btk.btkAcquisitionFileReader()
    reader.SetFilename(_TDDConfigure.C3DFilePathIN + 'sample01/Eb015pr.c3d')
    writer = btk.btkAcquisitionFileWriter()
    reader.Update()
    acq = reader.GetOutput()
    acq.ClearAnalogs()
    acq.ClearEvents()
    acq.RemovePoint(25)
    acq.RemovePoint(24)
    acq.RemovePoint(23)
    acq.RemovePoint(22)
    io = btk.btkC3DFileIO(reader.GetAcquisitionIO())
    self.assertEqual(io.GetStorageFormat(), btk.btkAcquisitionFileIO.Float)
    self.assertEqual(io.GetByteOrder(), btk.btkAcquisitionFileIO.IEEE_LittleEndian)
    io.SetStorageFormat(btk.btkAcquisitionFileIO.Integer)
    io.SetByteOrder(btk.btkAcquisitionFileIO.VAX_LittleEndian)
    writer.SetAcquisitionIO(io)
    writer.SetInput(reader.GetOutput())
    writer.SetFilename(_TDDConfigure.C3DFilePathOUT + 'sample01_Eb015vi_from_Eb015pr.c3d')
    writer.Update()

    reader.SetFilename(_TDDConfigure.C3DFilePathOUT + 'sample01_Eb015vi_from_Eb015pr.c3d')
    reader.Update()
    acq = reader.GetOutput()
    io = btk.btkC3DFileIO(reader.GetAcquisitionIO())
    self.assertEqual(io.GetStorageFormat(), btk.btkAcquisitionFileIO.Integer)
    self.assertEqual(io.GetByteOrder(), btk.btkAcquisitionFileIO.VAX_LittleEndian)
    self.assertAlmostEqual(io.GetPointScale(), 0.08333, 3)
    self.assertEqual(acq.GetPointFrequency(), 50)
    self.assertEqual(acq.GetPointNumber(), 22)
    self.assertEqual(acq.GetAnalogFrequency(), 200)
    self.assertEqual(acq.GetAnalogNumber(), 0)
    self.assertEqual(acq.GetEventNumber(), 0)
    self.assertEqual(acq.GetPoint(0).GetLabel(), 'RFT1')
    self.assertEqual(acq.GetPoint(9).GetDescription(), 'DISTAL FOOT')
    self.assertAlmostEqual(acq.GetPoint(0).GetValues()[8,0], 250.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,0], 285.0, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,1], 222.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,2], 564.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(0).GetResiduals()[0], 1.3333, 3)
    self.assertAlmostEqual(acq.GetPoint(14).GetResiduals()[20], 1.91667, 3)

  def test_sample01_Eb015si_from_Eb015pi(self):
    reader = btk.btkAcquisitionFileReader()
    reader.SetFilename(_TDDConfigure.C3DFilePathIN + 'sample01/Eb015pi.c3d')
    writer = btk.btkAcquisitionFileWriter()
    reader.Update()
    acq = reader.GetOutput()
    acq.ClearAnalogs()
    acq.ClearEvents()
    acq.RemovePoint(25)
    acq.RemovePoint(24)
    acq.RemovePoint(23)
    acq.RemovePoint(22)
    io = btk.btkC3DFileIO(reader.GetAcquisitionIO())
    self.assertEqual(io.GetStorageFormat(), btk.btkAcquisitionFileIO.Integer)
    self.assertEqual(io.GetByteOrder(), btk.btkAcquisitionFileIO.IEEE_LittleEndian)
    io.SetStorageFormat(btk.btkAcquisitionFileIO.Float)
    io.SetByteOrder(btk.btkAcquisitionFileIO.IEEE_BigEndian)
    writer.SetAcquisitionIO(io)
    writer.SetInput(reader.GetOutput())
    writer.SetFilename(_TDDConfigure.C3DFilePathOUT + 'sample01_Eb015si_from_Eb015pi.c3d')
    writer.Update()

    reader.SetFilename(_TDDConfigure.C3DFilePathOUT + 'sample01_Eb015si_from_Eb015pi.c3d')
    reader.Update()
    acq = reader.GetOutput()
    io = btk.btkC3DFileIO(reader.GetAcquisitionIO())
    self.assertEqual(io.GetStorageFormat(), btk.btkAcquisitionFileIO.Float)
    self.assertEqual(io.GetByteOrder(), btk.btkAcquisitionFileIO.IEEE_BigEndian)
    self.assertAlmostEqual(io.GetPointScale(), 0.08333, 3)
    self.assertEqual(acq.GetPointFrequency(), 50)
    self.assertEqual(acq.GetPointNumber(), 22)
    self.assertEqual(acq.GetAnalogFrequency(), 200)
    self.assertEqual(acq.GetAnalogNumber(), 0)
    self.assertEqual(acq.GetEventNumber(), 0)
    self.assertEqual(acq.GetPoint(0).GetLabel(), 'RFT1')
    self.assertEqual(acq.GetPoint(9).GetDescription(), 'DISTAL FOOT')
    self.assertAlmostEqual(acq.GetPoint(0).GetValues()[8,0], 250.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,0], 285.0, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,1], 222.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,2], 564.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(0).GetResiduals()[0], 1.3333, 3)
    self.assertAlmostEqual(acq.GetPoint(14).GetResiduals()[20], 1.91667, 3)

  def test_sample01_Eb015vr_from_Eb015pi(self):
    reader = btk.btkAcquisitionFileReader()
    reader.SetFilename(_TDDConfigure.C3DFilePathIN + 'sample01/Eb015pi.c3d')
    writer = btk.btkAcquisitionFileWriter()
    reader.Update()
    acq = reader.GetOutput()
    acq.ClearAnalogs()
    acq.ClearEvents()
    acq.RemovePoint(25)
    acq.RemovePoint(24)
    acq.RemovePoint(23)
    acq.RemovePoint(22)
    io = btk.btkC3DFileIO(reader.GetAcquisitionIO())
    self.assertEqual(io.GetStorageFormat(), btk.btkAcquisitionFileIO.Integer)
    self.assertEqual(io.GetByteOrder(), btk.btkAcquisitionFileIO.IEEE_LittleEndian)
    io.SetStorageFormat(btk.btkAcquisitionFileIO.Float)
    io.SetByteOrder(btk.btkAcquisitionFileIO.VAX_LittleEndian)
    writer.SetAcquisitionIO(io)
    writer.SetInput(reader.GetOutput())
    writer.SetFilename(_TDDConfigure.C3DFilePathOUT + 'sample01_Eb015vr_from_Eb015pi.c3d')
    writer.Update()

    reader.SetFilename(_TDDConfigure.C3DFilePathOUT + 'sample01_Eb015vr_from_Eb015pi.c3d')
    reader.Update()
    acq = reader.GetOutput()
    io = btk.btkC3DFileIO(reader.GetAcquisitionIO())
    self.assertEqual(io.GetStorageFormat(), btk.btkAcquisitionFileIO.Float)
    self.assertEqual(io.GetByteOrder(), btk.btkAcquisitionFileIO.VAX_LittleEndian)
    self.assertAlmostEqual(io.GetPointScale(), 0.08333, 3)
    self.assertEqual(acq.GetPointFrequency(), 50)
    self.assertEqual(acq.GetPointNumber(), 22)
    self.assertEqual(acq.GetAnalogFrequency(), 200)
    self.assertEqual(acq.GetAnalogNumber(), 0)
    self.assertEqual(acq.GetEventNumber(), 0)
    self.assertEqual(acq.GetPoint(0).GetLabel(), 'RFT1')
    self.assertEqual(acq.GetPoint(9).GetDescription(), 'DISTAL FOOT')
    self.assertAlmostEqual(acq.GetPoint(0).GetValues()[8,0], 250.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,0], 285.0, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,1], 222.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,2], 564.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(0).GetResiduals()[0], 1.3333, 3)
    self.assertAlmostEqual(acq.GetPoint(14).GetResiduals()[20], 1.91667, 3)

  def test_sample01_Eb015vr_from_Eb015sr(self):
    reader = btk.btkAcquisitionFileReader()
    reader.SetFilename(_TDDConfigure.C3DFilePathIN + 'sample01/Eb015sr.c3d')
    writer = btk.btkAcquisitionFileWriter()
    reader.Update()
    acq = reader.GetOutput()
    acq.ClearAnalogs()
    acq.ClearEvents()
    acq.RemovePoint(25)
    acq.RemovePoint(24)
    acq.RemovePoint(23)
    acq.RemovePoint(22)
    io = btk.btkC3DFileIO(reader.GetAcquisitionIO())
    self.assertEqual(io.GetStorageFormat(), btk.btkAcquisitionFileIO.Float)
    self.assertEqual(io.GetByteOrder(), btk.btkAcquisitionFileIO.IEEE_BigEndian)
    io.SetStorageFormat(btk.btkAcquisitionFileIO.Float)
    io.SetByteOrder(btk.btkAcquisitionFileIO.VAX_LittleEndian)
    writer.SetAcquisitionIO(io)
    writer.SetInput(reader.GetOutput())
    writer.SetFilename(_TDDConfigure.C3DFilePathOUT + 'sample01_Eb015vr_from_Eb015sr.c3d')
    writer.Update()

    reader.SetFilename(_TDDConfigure.C3DFilePathOUT + 'sample01_Eb015vr_from_Eb015sr.c3d')
    reader.Update()
    acq = reader.GetOutput()
    io = btk.btkC3DFileIO(reader.GetAcquisitionIO())
    self.assertEqual(io.GetStorageFormat(), btk.btkAcquisitionFileIO.Float)
    self.assertEqual(io.GetByteOrder(), btk.btkAcquisitionFileIO.VAX_LittleEndian)
    self.assertAlmostEqual(io.GetPointScale(), 0.08333, 3)
    self.assertEqual(acq.GetPointFrequency(), 50)
    self.assertEqual(acq.GetPointNumber(), 22)
    self.assertEqual(acq.GetAnalogFrequency(), 200)
    self.assertEqual(acq.GetAnalogNumber(), 0)
    self.assertEqual(acq.GetEventNumber(), 0)
    self.assertEqual(acq.GetPoint(0).GetLabel(), 'RFT1')
    self.assertEqual(acq.GetPoint(9).GetDescription(), 'DISTAL FOOT')
    self.assertAlmostEqual(acq.GetPoint(0).GetValues()[8,0], 250.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,0], 285.0, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,1], 222.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,2], 564.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(0).GetResiduals()[0], 1.3333, 3)
    self.assertAlmostEqual(acq.GetPoint(14).GetResiduals()[20], 1.91667, 3)

  def test_sample01_Eb015sr_from_Eb015pr(self):
    reader = btk.btkAcquisitionFileReader()
    reader.SetFilename(_TDDConfigure.C3DFilePathIN + 'sample01/Eb015pr.c3d')
    writer = btk.btkAcquisitionFileWriter()
    reader.Update()
    acq = reader.GetOutput()
    acq.ClearAnalogs()
    acq.ClearEvents()
    acq.RemovePoint(25)
    acq.RemovePoint(24)
    acq.RemovePoint(23)
    acq.RemovePoint(22)
    io = btk.btkC3DFileIO(reader.GetAcquisitionIO())
    self.assertEqual(io.GetStorageFormat(), btk.btkAcquisitionFileIO.Float)
    self.assertEqual(io.GetByteOrder(), btk.btkAcquisitionFileIO.IEEE_LittleEndian)
    io.SetStorageFormat(btk.btkAcquisitionFileIO.Float)
    io.SetByteOrder(btk.btkAcquisitionFileIO.IEEE_BigEndian)
    writer.SetAcquisitionIO(io)
    writer.SetInput(reader.GetOutput())
    writer.SetFilename(_TDDConfigure.C3DFilePathOUT + 'sample01_Eb015sr_from_Eb015pr.c3d')
    writer.Update()

    reader.SetFilename(_TDDConfigure.C3DFilePathOUT + 'sample01_Eb015sr_from_Eb015pr.c3d')
    reader.Update()
    acq = reader.GetOutput()
    io = btk.btkC3DFileIO(reader.GetAcquisitionIO())
    self.assertEqual(io.GetStorageFormat(), btk.btkAcquisitionFileIO.Float)
    self.assertEqual(io.GetByteOrder(), btk.btkAcquisitionFileIO.IEEE_BigEndian)
    self.assertAlmostEqual(io.GetPointScale(), 0.08333, 3)
    self.assertEqual(acq.GetPointFrequency(), 50)
    self.assertEqual(acq.GetPointNumber(), 22)
    self.assertEqual(acq.GetAnalogFrequency(), 200)
    self.assertEqual(acq.GetAnalogNumber(), 0)
    self.assertEqual(acq.GetEventNumber(), 0)
    self.assertEqual(acq.GetPoint(0).GetLabel(), 'RFT1')
    self.assertEqual(acq.GetPoint(9).GetDescription(), 'DISTAL FOOT')
    self.assertAlmostEqual(acq.GetPoint(0).GetValues()[8,0], 250.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,0], 285.0, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,1], 222.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,2], 564.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(0).GetResiduals()[0], 1.3333, 3)
    self.assertAlmostEqual(acq.GetPoint(14).GetResiduals()[20], 1.91667, 3)

  def test_sample01_Eb015pr_from_Eb015vr(self):
    reader = btk.btkAcquisitionFileReader()
    reader.SetFilename(_TDDConfigure.C3DFilePathIN + 'sample01/Eb015vr.c3d')
    writer = btk.btkAcquisitionFileWriter()
    reader.Update()
    acq = reader.GetOutput()
    acq.ClearAnalogs()
    acq.ClearEvents()
    acq.RemovePoint(25)
    acq.RemovePoint(24)
    acq.RemovePoint(23)
    acq.RemovePoint(22)
    io = btk.btkC3DFileIO(reader.GetAcquisitionIO())
    self.assertEqual(io.GetStorageFormat(), btk.btkAcquisitionFileIO.Float)
    self.assertEqual(io.GetByteOrder(), btk.btkAcquisitionFileIO.VAX_LittleEndian)
    io.SetStorageFormat(btk.btkAcquisitionFileIO.Float)
    io.SetByteOrder(btk.btkAcquisitionFileIO.IEEE_LittleEndian)
    writer.SetAcquisitionIO(io)
    writer.SetInput(reader.GetOutput())
    writer.SetFilename(_TDDConfigure.C3DFilePathOUT + 'sample01_Eb015pr_from_Eb015vr.c3d')
    writer.Update()

    reader.SetFilename(_TDDConfigure.C3DFilePathOUT + 'sample01_Eb015pr_from_Eb015vr.c3d')
    reader.Update()
    acq = reader.GetOutput()
    io = btk.btkC3DFileIO(reader.GetAcquisitionIO())
    self.assertEqual(io.GetStorageFormat(), btk.btkAcquisitionFileIO.Float)
    self.assertEqual(io.GetByteOrder(), btk.btkAcquisitionFileIO.IEEE_LittleEndian)
    self.assertAlmostEqual(io.GetPointScale(), 0.08333, 3)
    self.assertEqual(acq.GetPointFrequency(), 50)
    self.assertEqual(acq.GetPointNumber(), 22)
    self.assertEqual(acq.GetAnalogFrequency(), 200)
    self.assertEqual(acq.GetAnalogNumber(), 0)
    self.assertEqual(acq.GetEventNumber(), 0)
    self.assertEqual(acq.GetPoint(0).GetLabel(), 'RFT1')
    self.assertEqual(acq.GetPoint(9).GetDescription(), 'DISTAL FOOT')
    self.assertAlmostEqual(acq.GetPoint(0).GetValues()[8,0], 250.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,0], 285.0, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,1], 222.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(17).GetValues()[16,2], 564.4166, 3)
    self.assertAlmostEqual(acq.GetPoint(0).GetResiduals()[0], 1.3333, 3)
    self.assertAlmostEqual(acq.GetPoint(14).GetResiduals()[20], 1.91667, 3)

  def test_sample09_PlugInC3D_rewrited(self):
    reader = btk.btkAcquisitionFileReader()
    reader.SetFilename(_TDDConfigure.C3DFilePathIN + 'sample09/PlugInC3D.c3d')
    writer = btk.btkAcquisitionFileWriter()
    reader.Update()
    writer.SetAcquisitionIO(reader.GetAcquisitionIO())
    writer.SetInput(reader.GetOutput())
    writer.SetFilename(_TDDConfigure.C3DFilePathOUT + 'sample09_PlugInC3D.c3d')
    writer.Update()
    reader2 = btk.btkAcquisitionFileReader()
    reader2.SetFilename(_TDDConfigure.C3DFilePathOUT + 'sample09_PlugInC3D.c3d')
    reader2.Update()

    acq = reader.GetOutput()
    acq2 = reader2.GetOutput()
    self.assertEqual(acq.GetFirstFrame(), acq2.GetFirstFrame())
    self.assertEqual(acq.GetPointFrequency(), acq2.GetPointFrequency())
    self.assertEqual(acq.GetPointNumber(), acq2.GetPointNumber())
    self.assertEqual(acq.GetPointFrameNumber(), acq2.GetPointFrameNumber())
    self.assertEqual(acq.GetAnalogFrequency(), acq2.GetAnalogFrequency())
    self.assertEqual(acq.GetAnalogNumber(), acq2.GetAnalogNumber())
    self.assertEqual(acq.GetPointUnit(), acq2.GetPointUnit())
    
    self.assertAlmostEqual(acq2.GetMetaData().GetChild('ANALOG').GetChild('RATE').GetInfo().ToDouble(0), acq.GetAnalogFrequency(), 5)
    
    for i in range(0,50):
      for j in range(0,acq.GetPointNumber()):
        self.assertAlmostEqual(acq.GetPoint(j).GetValues()[i,0], acq2.GetPoint(j).GetValues()[i,0], 4)
        self.assertAlmostEqual(acq.GetPoint(j).GetValues()[i,1], acq2.GetPoint(j).GetValues()[i,1], 4)
        self.assertAlmostEqual(acq.GetPoint(j).GetValues()[i,2], acq2.GetPoint(j).GetValues()[i,2], 4)
        self.assertAlmostEqual(acq.GetPoint(j).GetResiduals()[i], acq2.GetPoint(j).GetResiduals()[i], 4)
      for j in range(0,acq.GetAnalogNumber()):
        self.assertAlmostEqual(acq.GetAnalog(j).GetValues()[i], acq2.GetAnalog(j).GetValues()[i], 4)
    
    self.assertEqual(acq2.GetAnalog(0).GetGain(), btk.btkAnalog.PlusMinus5)
    self.assertEqual(acq2.GetAnalog(1).GetGain(), btk.btkAnalog.PlusMinus5)
    self.assertEqual(acq2.GetAnalog(2).GetGain(), btk.btkAnalog.PlusMinus5)
    self.assertEqual(acq2.GetAnalog(3).GetGain(), btk.btkAnalog.PlusMinus5)
    self.assertEqual(acq2.GetAnalog(4).GetGain(), btk.btkAnalog.PlusMinus5)
    self.assertEqual(acq2.GetAnalog(5).GetGain(), btk.btkAnalog.PlusMinus5)
    self.assertEqual(acq2.GetAnalog(6).GetGain(), btk.btkAnalog.PlusMinus5)
    self.assertEqual(acq2.GetAnalog(7).GetGain(), btk.btkAnalog.PlusMinus5)
    self.assertEqual(acq2.GetAnalog(8).GetGain(), btk.btkAnalog.PlusMinus5)
    self.assertEqual(acq2.GetAnalog(9).GetGain(), btk.btkAnalog.PlusMinus5)
    self.assertEqual(acq2.GetAnalog(10).GetGain(), btk.btkAnalog.PlusMinus5)
    self.assertEqual(acq2.GetAnalog(11).GetGain(), btk.btkAnalog.PlusMinus5)
    self.assertEqual(acq2.GetAnalog(12).GetGain(), btk.btkAnalog.PlusMinus10)
    self.assertEqual(acq2.GetAnalog(13).GetGain(), btk.btkAnalog.PlusMinus10)
    self.assertEqual(acq2.GetAnalog(14).GetGain(), btk.btkAnalog.PlusMinus10)
    self.assertEqual(acq2.GetAnalog(15).GetGain(), btk.btkAnalog.PlusMinus10)
    self.assertEqual(acq2.GetAnalog(16).GetGain(), btk.btkAnalog.PlusMinus10)
    self.assertEqual(acq2.GetAnalog(17).GetGain(), btk.btkAnalog.PlusMinus10)
    self.assertEqual(acq2.GetAnalog(18).GetGain(), btk.btkAnalog.PlusMinus10)
    self.assertEqual(acq2.GetAnalog(19).GetGain(), btk.btkAnalog.PlusMinus10)
    self.assertEqual(acq2.GetAnalog(20).GetGain(), btk.btkAnalog.PlusMinus5)
    self.assertEqual(acq2.GetAnalog(21).GetGain(), btk.btkAnalog.PlusMinus5)
    self.assertEqual(acq2.GetAnalog(22).GetGain(), btk.btkAnalog.PlusMinus5)
    self.assertEqual(acq2.GetAnalog(23).GetGain(), btk.btkAnalog.PlusMinus5)
    self.assertEqual(acq2.GetAnalog(24).GetGain(), btk.btkAnalog.PlusMinus5)
    self.assertEqual(acq2.GetAnalog(25).GetGain(), btk.btkAnalog.PlusMinus5)
    self.assertEqual(acq2.GetAnalog(26).GetGain(), btk.btkAnalog.PlusMinus5)
    self.assertEqual(acq2.GetAnalog(27).GetGain(), btk.btkAnalog.PlusMinus5)
    self.assertEqual(acq2.GetAnalog(28).GetGain(), btk.btkAnalog.PlusMinus5)
    self.assertEqual(acq2.GetAnalog(29).GetGain(), btk.btkAnalog.PlusMinus5)
    self.assertEqual(acq2.GetAnalog(30).GetGain(), btk.btkAnalog.PlusMinus5)
    self.assertEqual(acq2.GetAnalog(31).GetGain(), btk.btkAnalog.PlusMinus5)
    self.assertEqual(acq2.GetAnalog(32).GetGain(), btk.btkAnalog.PlusMinus5)
    self.assertEqual(acq2.GetAnalog(33).GetGain(), btk.btkAnalog.PlusMinus5)
    self.assertEqual(acq2.GetAnalog(34).GetGain(), btk.btkAnalog.PlusMinus5)
    self.assertEqual(acq2.GetAnalog(35).GetGain(), btk.btkAnalog.PlusMinus5)

  def test_sample19_sample19_rewrited(self):
    reader = btk.btkAcquisitionFileReader()
    reader.SetFilename(_TDDConfigure.C3DFilePathIN + 'sample19/sample19.c3d')
    writer = btk.btkAcquisitionFileWriter()
    reader.Update()
    writer.SetInput(reader.GetOutput())
    writer.SetFilename(_TDDConfigure.C3DFilePathOUT + 'sample19_sample19.c3d')
    writer.Update()
    reader2 = btk.btkAcquisitionFileReader()
    reader2.SetFilename(_TDDConfigure.C3DFilePathOUT + 'sample19_sample19.c3d')
    reader2.Update()
    acq = reader.GetOutput()
    acq2 = reader.GetOutput()

    self.assertEqual(acq.GetFirstFrame(), acq2.GetFirstFrame())
    self.assertEqual(acq.GetPointFrequency(), acq2.GetPointFrequency())
    self.assertEqual(acq.GetPointNumber(), acq2.GetPointNumber())
    self.assertEqual(acq.GetPointFrameNumber(), acq2.GetPointFrameNumber())
    self.assertEqual(acq.GetAnalogFrequency(), acq2.GetAnalogFrequency())
    self.assertEqual(acq.GetAnalogNumber(), acq2.GetAnalogNumber())
    self.assertEqual(acq.GetAnalogFrameNumber(), acq2.GetAnalogFrameNumber())
    self.assertEqual(acq.GetPointUnit(), acq2.GetPointUnit())
    for i in range(0,50, 10):
      self.assertAlmostEqual(acq.GetAnalog(0).GetValues()[i], acq2.GetAnalog(0).GetValues()[i], 4)
      self.assertAlmostEqual(acq.GetAnalog(1).GetValues()[i], acq2.GetAnalog(1).GetValues()[i], 4)
    
  def test_emptyAcquisition_Template(self):
    acq = btk.btkAcquisition()
    writer = btk.btkAcquisitionFileWriter()
    writer.SetInput(acq)
    writer.SetFilename(_TDDConfigure.C3DFilePathOUT + 'EmptyAcquisition.c3d')
    writer.Update()
    
    reader = btk.btkAcquisitionFileReader()
    reader.SetFilename(_TDDConfigure.C3DFilePathOUT + 'EmptyAcquisition.c3d')
    reader.Update()
    acq = reader.GetOutput()
    metadata = acq.GetMetaData()
    self.assertEqual(acq.GetFirstFrame(), 1)
    self.assertEqual(acq.GetPointFrequency(), 0.0)
    self.assertEqual(acq.GetPointNumber(), 0)
    self.assertEqual(acq.GetPointFrameNumber(), 0)
    self.assertEqual(acq.GetAnalogFrequency(), 0.0)
    self.assertEqual(acq.GetAnalogNumber(), 0)
    self.assertEqual(acq.GetPointUnit(), 'mm')
    itPoint = metadata.FindChild('POINT')
    itAnalog = metadata.FindChild('ANALOG')
    itFP = metadata.FindChild('FORCE_PLATFORM')
    self.assertTrue(itPoint != metadata.End())
    self.assertTrue(itAnalog != metadata.End())
    self.assertTrue(itFP != metadata.End())
    itUsed = itFP.value().FindChild('USED')
    itType = itFP.value().FindChild('TYPE')
    itZero = itFP.value().FindChild('ZERO')
    itCorners = itFP.value().FindChild('CORNERS')
    itOrigin = itFP.value().FindChild('ORIGIN')
    itChannel = itFP.value().FindChild('CHANNEL')
    self.assertTrue(itUsed != itFP.value().End())
    self.assertTrue(itType != itFP.value().End())
    self.assertTrue(itZero != itFP.value().End())
    self.assertTrue(itCorners != itFP.value().End())
    self.assertTrue(itOrigin != itFP.value().End())
    self.assertTrue(itChannel != itFP.value().End())
    self.assertTrue(itUsed.value().HasInfo())
    self.assertTrue(itType.value().HasInfo())
    self.assertTrue(itZero.value().HasInfo())
    self.assertTrue(itCorners.value().HasInfo())
    self.assertTrue(itOrigin.value().HasInfo())
    self.assertTrue(itChannel.value().HasInfo())
    self.assertEqual(itUsed.value().GetInfo().GetDimensionsProduct(), 1)
    self.assertEqual(itType.value().GetInfo().GetDimensionsProduct(), 0)
    self.assertEqual(itZero.value().GetInfo().GetDimensionsProduct(), 2)
    self.assertEqual(itCorners.value().GetInfo().GetDimensionsProduct(), 0)
    self.assertEqual(itOrigin.value().GetInfo().GetDimensionsProduct(), 0)
    self.assertEqual(itChannel.value().GetInfo().GetDimensionsProduct(), 0)
    self.assertEqual(len(itUsed.value().GetInfo().GetDimensions()), 0)
    self.assertEqual(len(itType.value().GetInfo().GetDimensions()), 1)
    self.assertEqual(len(itZero.value().GetInfo().GetDimensions()), 1)
    self.assertEqual(len(itCorners.value().GetInfo().GetDimensions()), 3)
    self.assertEqual(len(itOrigin.value().GetInfo().GetDimensions()), 2)
    self.assertEqual(len(itChannel.value().GetInfo().GetDimensions()), 2)

  def test_convertTRC2C3D(self):
    reader = btk.btkAcquisitionFileReader()
    reader.SetFilename(_TDDConfigure.TRCFilePathIN + 'Knee.trc')
    writer = btk.btkAcquisitionFileWriter()
    writer.SetInput(reader.GetOutput())
    writer.SetFilename(_TDDConfigure.C3DFilePathOUT + 'Knee_rewritedInC3D.c3d')
    writer.Update()
    
    reader2 = btk.btkAcquisitionFileReader()
    reader2.SetFilename(_TDDConfigure.C3DFilePathOUT + 'Knee_rewritedInC3D.c3d')
    reader2.Update()
    acq = reader.GetOutput()
    acq2 = reader2.GetOutput()

    self.assertAlmostEqual(acq.GetPoint(70).GetResiduals()[82], -1.0, 5)

    self.assertEqual(acq.GetFirstFrame(), acq2.GetFirstFrame())
    self.assertEqual(acq.GetPointFrequency(), acq2.GetPointFrequency())
    self.assertEqual(acq.GetPointNumber(), acq2.GetPointNumber())
    self.assertEqual(acq.GetPointFrameNumber(), acq2.GetPointFrameNumber())
    self.assertEqual(acq.GetAnalogFrequency(), acq2.GetAnalogFrequency())
    self.assertEqual(acq.GetAnalogNumber(), acq2.GetAnalogNumber())
    self.assertEqual(acq.GetPointUnit(), acq2.GetPointUnit())
    for i in range(0,50):
      self.assertAlmostEqual(acq.GetPoint(1).GetValues()[i,0], acq2.GetPoint(1).GetValues()[i,0], 4)
      self.assertAlmostEqual(acq.GetPoint(2).GetValues()[i,1], acq2.GetPoint(2).GetValues()[i,1], 4)
      self.assertAlmostEqual(acq.GetPoint(3).GetValues()[i,2], acq2.GetPoint(3).GetValues()[i,2], 4)
      self.assertAlmostEqual(acq.GetPoint(4).GetResiduals()[i], acq2.GetPoint(4).GetResiduals()[i], 4)

    self.assertAlmostEqual(acq.GetPoint(70).GetResiduals()[82], acq2.GetPoint(70).GetResiduals()[82], 5)
  
  def test_noInputModification(self):
    reader = btk.btkAcquisitionFileReader()
    reader.SetFilename(_TDDConfigure.C3DFilePathIN + 'sample01/Eb015pr.c3d')
    reader.Update()
    output = reader.GetOutput()
    writer = btk.btkAcquisitionFileWriter()
    writer.SetInput(reader.GetOutput())
    writer.SetFilename(_TDDConfigure.C3DFilePathOUT + 'sample01_Eb015pr.c3d')
    writer.Update()
  
  def test_acq100000(self):
    acq = btk.btkAcquisition()
    acq.Init(3,100000,1,2)
    writer = btk.btkAcquisitionFileWriter()
    writer.SetInput(acq)
    writer.SetFilename(_TDDConfigure.C3DFilePathOUT + 'acq100000.c3d')
    writer.Update()
    
    reader = btk.btkAcquisitionFileReader()
    reader.SetFilename(_TDDConfigure.C3DFilePathOUT + 'acq100000.c3d')
    reader.Update()
    output = reader.GetOutput()
    
    self.assertEqual(output.GetPointFrameNumber(),100000)
    self.assertEqual(output.GetAnalogFrameNumber(),200000)
    self.assertEqual(output.GetPointNumber(),3)
    self.assertEqual(output.GetAnalogNumber(),1)
    # WARNING 65535 replaced by -1 due to the missing of possible cast with Python!
    self.assertEqual(output.GetMetaData().GetChild('POINT').GetChild('FRAMES').GetInfo().ToInt(0), -1) 
    self.assertEqual(output.GetLastFrame(), 100000)
    
  def test_UTF8(self):
    reader = btk.btkAcquisitionFileReader()
    reader.SetFilename(_TDDConfigure.C3DFilePathIN + 'sample19/sample19.c3d')
    writer = btk.btkAcquisitionFileWriter()
    reader.Update()
    writer.SetInput(reader.GetOutput())
    writer.SetFilename(_TDDConfigure.C3DFilePathOUT + 'Я могу есть стекло, оно мне не вредит - python.c3d')
    writer.Update()
    reader2 = btk.btkAcquisitionFileReader()
    reader2.SetFilename(_TDDConfigure.C3DFilePathOUT + 'Я могу есть стекло, оно мне не вредит - python.c3d')
    reader2.Update()
    acq = reader.GetOutput()
    acq2 = reader.GetOutput()

    self.assertEqual(acq.GetFirstFrame(), acq2.GetFirstFrame())
    self.assertEqual(acq.GetPointFrequency(), acq2.GetPointFrequency())
    self.assertEqual(acq.GetPointNumber(), acq2.GetPointNumber())
    self.assertEqual(acq.GetPointFrameNumber(), acq2.GetPointFrameNumber())
    self.assertEqual(acq.GetAnalogFrequency(), acq2.GetAnalogFrequency())
    self.assertEqual(acq.GetAnalogNumber(), acq2.GetAnalogNumber())
    self.assertEqual(acq.GetAnalogFrameNumber(), acq2.GetAnalogFrameNumber())
    self.assertEqual(acq.GetPointUnit(), acq2.GetPointUnit())
      
  def test_InternalsUpdateUpdateMetaDataBased(self):
    reader = btk.btkAcquisitionFileReader()
    reader.SetFilename(_TDDConfigure.C3DFilePathIN + 'others/PiGMotion-FlatFoot-Full.c3d')
    reader.Update()
    acq = reader.GetOutput()
    reader._reset()
    # To test the *non use* of the CompatibleVicon option
    for it in btk.Iterate(acq.GetPoints()):
        it.SetDescription('')
    for it in btk.Iterate(acq.GetAnalogs()):
        it.SetDescription('')
    metadata = acq.GetMetaData().GetChild('ANALYSIS')
    num = metadata.GetChild('USED').GetInfo().ToInt(0)
    blank = btk.btkStringArray(num)
    dims = btk.btkIntArray(2,0)
    dims[1] = num
    metadata.GetChild('DESCRIPTIONS').GetInfo().SetValues(dims, blank)
    metadata.GetChild('SUBJECTS').GetInfo().SetValues(dims, blank)
    # To test the *non use* of the DataBasedUpdate option
    inc = 1
    for it in btk.Iterate(acq.GetPoints()):
        it.SetLabel('MyPoint#' + str(inc))
        it.SetDescription('test')
        inc += 1 
    inc = 1
    for it in btk.Iterate(acq.GetAnalogs()):
        it.SetLabel('MyAnalog#' + str(inc))
        it.SetDescription('test')
        it.SetScale(1.0)
        it.SetOffset(0)
        inc += 1
    acq.GetMetaData().GetChild('POINT').GetChild('RATE').GetInfo().SetValues(400.0)
      
    io = btk.btkC3DFileIO()
    io.SetInternalsUpdateOptions(btk.btkC3DFileIO.MetaDataBasedUpdate)
    
    writer = btk.btkAcquisitionFileWriter()
    writer.SetInput(acq)
    writer.SetAcquisitionIO(io)
    writer.SetFilename(_TDDConfigure.C3DFilePathOUT + 'InternalsUpdateUpdateMetaDataBased.c3d')
    writer.Update()
    
    reader = btk.btkAcquisitionFileReader()
    reader.SetFilename(_TDDConfigure.C3DFilePathIN + 'others/PiGMotion-FlatFoot-Full.c3d')
    reader.Update()
    acq = reader.GetOutput()
    
    reader2 = btk.btkAcquisitionFileReader()
    reader2.SetFilename(_TDDConfigure.C3DFilePathOUT + 'InternalsUpdateUpdateMetaDataBased.c3d')
    reader2.Update()
    acq2 = reader2.GetOutput()

    self.assertEqual(acq.GetFirstFrame(), acq2.GetFirstFrame())
    self.assertEqual(400.0, acq2.GetPointFrequency())
    self.assertEqual(acq.GetPointNumber(), acq2.GetPointNumber())
    self.assertEqual(acq.GetPointFrameNumber(), acq2.GetPointFrameNumber())
    self.assertEqual(4000.0, acq2.GetAnalogFrequency())
    self.assertEqual(acq.GetAnalogNumber(), acq2.GetAnalogNumber())
    self.assertEqual(acq.GetAnalogFrameNumber(), acq2.GetAnalogFrameNumber())
    self.assertEqual(acq.GetPointUnit(), acq2.GetPointUnit())
    
    inc = 0
    for it in btk.Iterate(acq.GetPoints()):
        self.assertEqual(it.GetLabel(), acq.GetPoint(inc).GetLabel())
        self.assertEqual(it.GetDescription(), acq.GetPoint(inc).GetDescription())
        inc += 1
    inc = 0
    for it in btk.Iterate(acq.GetAnalogs()):
        self.assertEqual(it.GetLabel(), acq.GetAnalog(inc).GetLabel())
        self.assertEqual(it.GetDescription(), acq.GetAnalog(inc).GetDescription())
        self.assertEqual(it.GetScale(), acq.GetAnalog(inc).GetScale())
        self.assertEqual(it.GetOffset(), acq.GetAnalog(inc).GetOffset())
        inc += 1

    metadata = acq2.GetMetaData().GetChild('ANALYSIS')
    dims = metadata.GetChild('DESCRIPTIONS').GetInfo().GetDimensions()
    blank = metadata.GetChild('DESCRIPTIONS').GetInfo().ToString()
    self.assertEqual(dims[0], 0)
    for i in range(0,num):
          self.assertEqual(blank[i], '')
    dims = metadata.GetChild('SUBJECTS').GetInfo().GetDimensions()
    blank = metadata.GetChild('SUBJECTS').GetInfo().ToString()
    self.assertEqual(dims[0], 0)
    for i in range(0,num):
          self.assertEqual(blank[i], '')
    