# utils.py

"""Internal utils."""

#pylint: disable=too-few-public-methods, useless-object-inheritance
class Iterator(object):
    """Universal iterator for many of BLPAPI objects.

    It can be used to iterate any sub-items in an item which has
    the following methods:
        * method returning the number of sub-items
        * method returning the 'index'ed sub-item

    For example, it is currently used as an iterator for Service's event
    definition in the following way:

        class Service(object):
            ...
            # Return an iterator over this Service's event definitions.
            def eventDefinitions(self):
                return utils.Iterator(
                    self,
                    Service.numEventDefinitions,
                    Service.getEventDefinitionAt)

            ...

    """

    def __init__(self, objToIterate, numFunc, getFunc):
        self.__obj = objToIterate
        self.__index = 0
        self.__num = numFunc(objToIterate)
        self.__getter = getFunc

    def __iter__(self):
        return self

    def __next__(self):
        if self.__index == self.__num:
            raise StopIteration()
        else:
            res = self.__getter(self.__obj, self.__index)
            self.__index += 1
            return res

    next = __next__


class MetaClassForClassesWithEnums(type):
    """This meta class protects enums from changes.

    This meta class does not let change values of class members with names in
    uppercase (a typical naming convention for enums).

    """

    class EnumError(TypeError):
        """Raise this on attempt to change value of an enumeration constant.
        """
        pass

    def __setattr__(cls, name, value):
        """Change the value of an attribute if it is not an enum.

        Raise EnumError exception otherwise.
        """
        if name.isupper() and name in cls.__dict__:
            raise cls.EnumError("Can't change value of enum %s" % name)
        else:
            type.__setattr__(cls, name, value)

    def __delattr__(cls, name):
        """Unbind the attribute if it is not an enum.

        Raise EnumError exception otherwise.
        """
        if name.isupper() and name in cls.__dict__:
            raise cls.EnumError("Can't unbind enum %s" % name)
        else:
            type.__delattr__(cls, name)

def get_handle(thing):
    """Returns the result of thing._handle() or None if thing is None"""
    return None if thing is None else thing._handle() #pylint: disable=protected-access

def invoke_if_valid(cb, value):
    """Returns the result of cb(value) if cb is callable, else -- just value"""
    if cb is None or not callable(cb):
        return value
    return cb(value)

__copyright__ = """
Copyright 2012. Bloomberg Finance L.P.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:  The above
copyright notice and this permission notice shall be included in all copies
or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
IN THE SOFTWARE.
"""
