# session.py

"""Provide consumer session to get Bloomberg Service.

This component implements a consumer session for getting services.
"""

from __future__ import print_function
from __future__ import absolute_import
import weakref
import sys
import traceback
import os
import functools
from .abstractsession import AbstractSession
from .event import Event
from . import exception
from .exception import _ExceptionUtil
from . import internals
from .internals import CorrelationId
from .sessionoptions import SessionOptions
from .requesttemplate import RequestTemplate


class Session(AbstractSession):
    """Consumer session for making requests for Bloomberg services.

    This class provides a consumer session for making requests for Bloomberg
    services.

    Sessions manage access to services either by requests and responses or
    subscriptions. A Session can dispatch events and replies in either
    a synchronous or asynchronous mode. The mode of a Session is determined
    when it is constructed and cannot be changed subsequently.

    A Session is asynchronous if an eventHandler argument is supplied when it
    is constructed. The nextEvent() method may not be called. All incoming
    events are delivered to the event handler supplied on construction.

    A Session is synchronous if an eventHandler argument is not supplied when
    it is constructed. The nextEvent() method must be called to read incoming
    events.

    Several methods in Session take a CorrelationId parameter. The application
    may choose to supply its own CorrelationId values or allow the Session to
    create values. If the application supplys its own CorrelationId values it
    must manage their lifetime such that the same value is not reused for more
    than one operation at a time. The lifetime of a CorrelationId begins when
    it is supplied in a method invoked on a Session and ends either when it is
    explicitly cancelled using cancel() or unsubscribe(), when a RESPONSE Event
    (not a PARTIAL_RESPONSE) containing it is received or when a
    SUBSCRIPTION_STATUS Event which indicates that the subscription it refers
    to has been terminated is received.

    When using an asynchronous Session the application must be aware that
    because the callbacks are generated from another thread they may be
    processed before the call which generates them has returned. For example,
    the SESSION_STATUS Event generated by a startAsync() may be processed
    before startAsync() has returned (even though startAsync() itself will not
    block).

    This becomes more significant when Session generated CorrelationIds are in
    use. For example, if a call to subscribe() which returns a Session
    generated CorrelationId has not completed before the first Events which
    contain that CorrelationId arrive the application may not be able to
    interpret those events correctly. For this reason, it is preferable to use
    user generated CorrelationIds when using asynchronous Sessions. This issue
    does not arise when using a synchronous Session as long as the calls to
    subscribe() etc are made on the same thread as the calls to nextEvent().

    The possible statuses a subscription may be in.

    UNSUBSCRIBED          No longer active, terminated by API.
    SUBSCRIBING           Initiated but no updates received.
    SUBSCRIBED            Updates are flowing.
    CANCELLED             No longer active, terminated by Application.
    PENDING_CANCELLATION  Pending cancellation.
    """

    UNSUBSCRIBED = internals.SUBSCRIPTIONSTATUS_UNSUBSCRIBED
    """No longer active, terminated by API."""
    SUBSCRIBING = internals.SUBSCRIPTIONSTATUS_SUBSCRIBING
    """Initiated but no updates received."""
    SUBSCRIBED = internals.SUBSCRIPTIONSTATUS_SUBSCRIBED
    """Updates are flowing."""
    CANCELLED = internals.SUBSCRIPTIONSTATUS_CANCELLED
    """No longer active, terminated by Application."""
    PENDING_CANCELLATION = \
        internals.SUBSCRIPTIONSTATUS_PENDING_CANCELLATION
    """No longer active, terminated by Application."""

    __handle = None
    __handlerProxy = None

    @staticmethod
    def __dispatchEvent(sessionRef, eventHandle):
        try:
            session = sessionRef()
            if session is not None:
                event = Event(eventHandle, session)
                session.__handler(event, session)
        except:
            print("Exception in event handler:", file=sys.stderr)
            traceback.print_exc(file=sys.stderr)
            os._exit(1)

    def __init__(self, options=None, eventHandler=None, eventDispatcher=None):
        """Constructor.

        Session([options, [eventHandler, [eventDispatcher]]]) constructs a
        Session object using the optionally specified 'options', the
        optionally specified 'eventHandler' and the optionally specified
        'eventDispatcher'.

        See the SessionOptions documentation for details on what can be
        specified in the 'options'.

        'eventHandler' can be None or a callable object that takes two
        arguments: received event and related session.

        Note that in case of unhandled exception in 'eventHandler', the
        exception traceback will be printed to sys.stderr and application
        will be terminated with nonzero exit code.

        If 'eventHandler' is not None then this Session will operate in
        asynchronous mode, otherwise the Session will operate in synchronous
        mode.

        If 'eventDispatcher' is None then the Session will create a default
        EventDispatcher for this Session which will use a single thread for
        dispatching events. For more control over event dispatching a specific
        instance of EventDispatcher can be supplied. This can be used to share
        a single EventDispatcher amongst multiple Session objects.

        If an 'eventDispatcher' is supplied which uses more than one thread the
        Session will ensure that events which should be ordered are passed to
        callbacks in a correct order. For example, partial response to
        a request or updates to a single subscription.

        If 'eventHandler' is None and and the 'eventDispatcher' is not None
        an exception is raised.

        Each 'eventDispatcher' uses it's own thread or pool of threads so if
        you want to ensure that a session which receives very large messages
        and takes a long time to process them does not delay a session that
        receives small messages and processes each one very quickly then give
        each one a separate 'eventDispatcher'.
        """
        if (eventHandler is None) and (eventDispatcher is not None):
            raise exception.InvalidArgumentException(
                "eventDispatcher is specified but eventHandler is None", 0)
        if options is None:
            options = SessionOptions()
        if eventHandler is not None:
            self.__handler = eventHandler
            self.__handlerProxy = functools.partial(Session.__dispatchEvent,
                                                    weakref.ref(self))
        self.__handle = internals.Session_createHelper(
            options._handle(),
            self.__handlerProxy,
            None if eventDispatcher is None else eventDispatcher._handle())
        AbstractSession.__init__(
            self,
            internals.blpapi_Session_getAbstractSession(self.__handle))

    def __del__(self):
        try:
            self.destroy()
        except (NameError, AttributeError):
            pass

    def destroy(self):
        if self.__handle:
            internals.Session_destroyHelper(self.__handle, self.__handlerProxy)
            self.__handle = None

    def start(self):
        """Start this session in synchronous mode.

        Attempt to start this Session and blocks until the Session has started
        or failed to start. If the Session is started successfully iTrue is
        returned, otherwise False is returned. Before start() returns a
        SESSION_STATUS Event is generated. If this is an asynchronous Session
        then the SESSION_STATUS may be processed by the registered EventHandler
        before start() has returned. A Session may only be started once.
        """
        return internals.blpapi_Session_start(self.__handle) == 0

    def startAsync(self):
        """Start this session in asynchronous mode.

        Attempt to begin the process to start this Session and return True if
        successful, otherwise return False. The application must monitor events
        for a SESSION_STATUS Event which will be generated once the Session has
        started or if it fails to start. If this is an asynchronous Session
        then the SESSION_STATUS Event may be processed by the registered
        EventHandler before startAsync() has returned. A Session may only be
        started once.
        """
        return internals.blpapi_Session_startAsync(self.__handle) == 0

    def stop(self):
        """Stop operation of this session and wait until it stops.

        Stop operation of this session and block until all callbacks to
        EventHandler objects relating to this Session which are currently in
        progress have completed (including the callback to handle
        the SESSION_STATUS Event this call generates). Once this returns no
        further callbacks to EventHandlers will occur. If stop() is called from
        within an EventHandler callback it is silently converted to
        a stopAsync() in order to prevent deadlock. Once a Session has been
        stopped it can only be destroyed.
        """
        return internals.blpapi_Session_stop(self.__handle) == 0

    def stopAsync(self):
        """Begin the process to stop this Session and return immediately.

        The application must monitor events for a
        SESSION_STATUS Event which will be generated once the
        Session has been stopped. After this SESSION_STATUS Event
        no further callbacks to EventHandlers will occur(). Once a
        Session has been stopped it can only be destroyed.
        """
        return internals.blpapi_Session_stop(self.__handle) == 0

    def nextEvent(self, timeout=0):
        """Return the next available Event for this session.

        If there is no event available this will block for up to the specified
        'timeoutMillis' milliseconds for an Event to arrive. A value of 0 for
        'timeoutMillis' (the default) indicates nextEvent() should not timeout
        and will not return until the next Event is available.

        If nextEvent() returns due to a timeout it will return an event of type
        'EventType.TIMEOUT'.

        If this is invoked on a Session which was created in asynchronous mode
        an InvalidStateException is raised.
        """
        retCode, event = internals.blpapi_Session_nextEvent(self.__handle,
                                                            timeout)

        _ExceptionUtil.raiseOnError(retCode)

        return Event(event, self)

    def tryNextEvent(self):
        """Return the next Event for this session if it is available.

        If there are Events available for the session, return the next Event
        If there is no event available for the session, return None. This
        method never blocks.
        """
        retCode, event = internals.blpapi_Session_tryNextEvent(self.__handle)
        if retCode:
            return None
        else:
            return Event(event, self)

    def subscribe(self, subscriptionList, identity=None, requestLabel=""):
        """Begin subscriptions for each entry in the specified list.

        Begin subscriptions for each entry in the specified 'subscriptionList',
        which must be an object of type 'SubscriptionList', optionally using
        the specified 'identity' for authorization. If no 'identity' is
        specified, the default authorization information is used.  If the
        optional 'requestLabel' is provided it defines a string which will be
        recorded along with any diagnostics for this operation.

        A SUBSCRIPTION_STATUS Event will be generated for each
        entry in the 'subscriptionList'.
        """
        _ExceptionUtil.raiseOnError(internals.blpapi_Session_subscribe(
            self.__handle,
            subscriptionList._handle(),
            None if identity is None else identity._handle(),
            requestLabel,
            len(requestLabel)))

    def unsubscribe(self, subscriptionList):
        """Cancel subscriptions from the specified 'subscriptionList'.

        Cancel each of the current subscriptions identified by the specified
        'subscriptionList', which must be an object of type 'SubscriptionList'.
        If the correlation ID of any entry in the 'subscriptionList' does not
        identify a current subscription then that entry is ignored. All entries
        which have valid correlation IDs will be cancelled.

        Once this call returns the correlation ids in the 'subscriptionList'
        will not be seen in any subsequent Message obtained from
        a MessageIterator by calling next(). However, any Message currently
        pointed to by a MessageIterator when unsubscribe() is called is not
        affected even if it has one of the correlation IDs in the
        'subscriptionList'. Also any Message where a reference has been
        retained by the application may still contain a correlation ID from
        the 'subscriptionList'. For these reasons, although technically
        an application is free to re-use the correlation IDs as soon as this
        method returns it is preferable not to aggressively re-use correlation
        IDs, particularly with an asynchronous Session.
        """
        _ExceptionUtil.raiseOnError(internals.blpapi_Session_unsubscribe(
            self.__handle,
            subscriptionList._handle(),
            None,
            0))

    def resubscribe(self, subscriptionList, requestLabel="", resubscriptionId=None):
        """Modify subscriptions in 'subscriptionList'.

        Modify each subscription in the specified 'subscriptionList', which
        must be an object of type 'SubscriptionList', to reflect the modified
        options specified for it. If the optional 'requestLabel' is provided it
        defines a string which will be recorded along with any diagnostics for
        this operation.

        For each entry in the 'subscriptionList' which has a correlation ID
        which identifies a current subscription the modified options replace
        the current options for the subscription and a SUBSCRIPTION_STATUS
        event will be generated in the event stream before the first update
        based on the new options. If the correlation ID of an entry in the
        'subscriptionList' does not identify a current subscription then that
        entry is ignored.
        """
        error = None
        if resubscriptionId is None:
            error = internals.blpapi_Session_resubscribe(
                    self.__handle,
                    subscriptionList._handle(),
                    requestLabel,
                    len(requestLabel))
        else:
            error = internals.blpapi_Session_resubscribeWithId(
                    self.__handle,
                    subscriptionList._handle(),
                    resubscriptionId,
                    requestLabel,
                    len(requestLabel))

        _ExceptionUtil.raiseOnError(error)

    def setStatusCorrelationId(self, service, correlationId, identity=None):
        """Set the CorrelationID on which service status messages.

        Set the CorrelationID on which service status messages will be
        received.
        Note: No service status messages are received prior to this call
        """
        _ExceptionUtil.raiseOnError(
            internals.blpapi_Session_setStatusCorrelationId(
                self.__handle,
                service._handle(),
                None if identity is None else identity._handle(),
                correlationId._handle()))

    def sendRequest(self,
                    request,
                    identity=None,
                    correlationId=None,
                    eventQueue=None,
                    requestLabel=""):
        """Send the specified 'request'.

        Send the specified 'request' using the specified 'identity' for
        authorization. If the optionally specified 'correlationId' is supplied
        use it, otherwise create a CorrelationId. The actual CorrelationId used
        is returned. If the optionally specified 'eventQueue' is supplied all
        events relating to this Request will arrive on that EventQueue. If
        the optional 'requestLabel' is provided it defines a string which will
        be recorded along with any diagnostics for this operation.

        A successful request will generate zero or more PARTIAL_RESPONSE
        Messages followed by exactly one RESPONSE Message. Once the final
        RESPONSE Message has been received the CorrelationId associated with
        this request may be re-used. If the request fails at any stage
        a REQUEST_STATUS will be generated after which the CorrelationId
        associated with the request may be re-used.
        """
        if correlationId is None:
            correlationId = CorrelationId()
        res = internals.blpapi_Session_sendRequest(
            self.__handle,
            request._handle(),
            correlationId._handle(),
            None if identity is None else identity._handle(),
            None if eventQueue is None else eventQueue._handle(),
            requestLabel,
            len(requestLabel))
        _ExceptionUtil.raiseOnError(res)
        if eventQueue is not None:
            eventQueue._registerSession(self)
        return correlationId

    def sendRequestTemplate(self, requestTemplate, correlationId=None):
        """Send a request defined by the specified 'requestTemplate'. If the
        optionally specified 'correlationId' is supplied, use it otherwise
        create a new 'CorrelationId'. The actual 'CorrelationId' used is
        returned.
       
        A successful request will generate zero or more 'PARTIAL_RESPONSE'
        events followed by exactly one 'RESPONSE' event. Once the final
        'RESPONSE' event has been received the 'CorrelationId' associated
        with  this request may be re-used. If the request fails at any stage
        a 'REQUEST_STATUS' will be generated after which the 'CorrelationId'
        associated with the request may be re-used.
        """
        if correlationId is None:
            correlationId = CorrelationId()
        res = internals.blpapi_Session_sendRequestTemplate(
                                                self.__handle,
                                                requestTemplate._handle(),
                                                correlationId._handle())
        _ExceptionUtil.raiseOnError(res)
        return correlationId

    def createSnapshotRequestTemplate(self,
                                      subscriptionString,
                                      identity,
                                      correlationId):
        """Create a snapshot request template for getting subscription data
        specified by the 'subscriptionString' using the specified 'identity' 
        if all the following conditions are met: the session is
        established, 'subscriptionString' is a valid subscription string and
        'correlationId' is not used in this session. If one or more conditions
        are not met, an exception is thrown. The provided 'correlationId' will
        be used for status updates about the created request template state
        and an implied subscription associated with it delivered by
        'SUBSCRIPTION_STATUS' events.

        The benefit of the snapshot request templates is that these requests
        may be serviced from a cache and the user may expect to see
        significantly lower response time.

        There are 3 possible states for a created request template:
        'Pending', 'Available', and 'Terminated'. Right after creation a
        request template is in the 'Pending' state.

        If a state is 'Pending', the user may send a request using this
        template but there are no guarantees about response time since cache
        is not available yet. Request template may transition into 'Pending'
        state only from the 'Available' state. In this case the
        'RequestTemplatePending' message is generated.

        If state is 'Available', all requests will be serviced from a cache
        and the user may expect to see significantly reduced latency. Note,
        that a snapshot request template can transition out of the
        'Available' state concurrently with requests being sent, so no
        guarantee of service from the cache can be provided. Request
        template may transition into 'Available' state only from the
        'Pending' state. In this case the 'RequestTemplateAvailable' message
        is generated. This message will also contain information about
        currently used connection in the 'boundTo' field. Note that it is
        possible to get the 'RequestTemplateAvailable' message with a new
        connection information, even if a request template is already in the
        'Available' state.

        If state is 'Terminated', sending request will always result in a
        failure response. Request template may transition into this state
        from any other state. This is a final state and it is guaranteed
        that the last message associated with the provided 'correlationId' will
        be the 'RequestTemplateTerminated' message which is generated when a
        request template transitions into this state. If a request template
        transitions into this state, all outstanding requests will be failed
        and appropriate messages will be generated for each request. After
        receiving the 'RequestTemplateTerminated' message, 'correlationId' may
        be reused.

        Note that resources used by a snapshot request template are released
        only when request template transitions into the 'Terminated' state
        or when session is destroyed. In order to release resources when
        request template is not needed anymore, user should call the
        'Session::cancel(correlationId)' unless the 'RequestTemplateTerminated'
        message was already received due to some problems. When the last
        copy of a 'RequestTemplate' object goes out of scope and there are
        no outstanding requests left, the snapshot request template will be
        destroyed automatically. If the last copy of a 'RequestTemplate'
        object goes out of scope while there are still some outstanding
        requests left, snapshot service request template will be destroyed
        automatically when the last request gets a final response.
        """
        rc, template = internals.blpapi_Session_createSnapshotRequestTemplate(
                                                        self.__handle,
                                                        subscriptionString,
                                                        identity._handle(),
                                                        correlationId._handle())
        _ExceptionUtil.raiseOnError(rc)
        reqTemplate = RequestTemplate(template)
        return reqTemplate

__copyright__ = """
Copyright 2012. Bloomberg Finance L.P.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:  The above
copyright notice and this permission notice shall be included in all copies
or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
IN THE SOFTWARE.
"""
