#!/usr/bin/env python
#
# Test modules import

import os
import sys
from importlib import import_module
try:
    from pathlib import Path
except ImportError as exc:
    import warnings
    warnings.warn(str(exc))
    sys.exit(77)

import pytest

try:
    FileNotFoundError
except NameError:  # python < 3
    FileNotFoundError = IOError

HERE = Path(__file__).parent.absolute()
SRCDIR = Path(os.getenv("LAL_TEST_SRCDIR", HERE.parent.parent))
BUILDDIR = Path(os.getenv("LAL_TEST_BUILDDIR", SRCDIR))
EXCLUDEFILE = SRCDIR / "exclude-modules.txt"

SUBPACKAGE = "lalinference"
PYTHONDIR = BUILDDIR.parent.parent / "python" / SUBPACKAGE


def read_excludes(source):
    """Read all excluded file paths from the given source file
    """
    excludes = set()
    with open(str(source), "r") as fobj:
        for line in fobj:
            if isinstance(line, bytes):
                line = line.decode("utf-8")
            content = line.strip().split("#", 1)[0].strip()
            if content:
                excludes.add(content)
    return excludes


def find_modules(path):
    """Returns the paths to all python module files
    """
    return {str(x.relative_to(path.parent)) for x in path.rglob("*.py")}


def path_to_name(filepath):
    name = filepath.replace(".py", "").replace(r"/", ".")
    if name.endswith(".__init__"):
        name = name[:-9]
    return name


FILENAMES = find_modules(PYTHONDIR)
try:
    EXCLUDE = read_excludes(EXCLUDEFILE)
except FileNotFoundError:  # no exclusion file
    EXCLUDE = set()
MODULES = [path_to_name(x) for x in sorted(FILENAMES)]
EXCLUDE = set(map(path_to_name, EXCLUDE))


@pytest.mark.parametrize("module", MODULES)
def test_import(module):
    if module in EXCLUDE:
        pytest.skip("excluded {}".format(str(module)))
    import_module(module)


# run from command-line
if __name__ == "__main__":
    if "-v" not in " ".join(sys.argv[1:]):  # default to verbose
        sys.argv.append("-v")
    sys.argv.append("-rs")
    sys.exit(pytest.main(args=[__file__] + sys.argv[1:]))
